/***************************************************************************
 									    
 SDL XMAME display driver, based on
 Linux SVGALib adaptation by Phillip Ezolt.
  
***************************************************************************/
#define __SDL_C

/* #define SDL_DEBUG */
/* #define DIRECT_HERMES */

#include <signal.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <SDL/SDL.h>
#include "xmame.h"
#include "devices.h"
#include "keyboard.h"
#include "SDL-keytable.h"
#ifdef DIRECT_HERMES 
#include <Hermes/Hermes.h>
#endif /* DIRECT_HERMES */

static int Vid_width;
static int Vid_height;
static int Vid_depth = 8;
static SDL_Surface* Surface;
static SDL_Surface* Offscreen_surface;
SDL_Color *Colors;

#ifdef DIRECT_HERMES
HermesHandle  H_PaletteHandle;
HermesHandle H_ConverterHandle;
int32_t* H_Palette;
static int H_Palette_modified = 0;
#endif

static unsigned char kstate[128];

int sysdep_init(void)
{
   if (SDL_Init(SDL_INIT_VIDEO) < 0) {
		fprintf (stderr, "SDL: Error: Unable to initialize SDL\n");
		return OSD_NOT_OK;
   } 
#ifdef DIRECT_HERMES
   Hermes_Init(0);
#endif /* DIRECT_HERMES */
   fprintf (stderr, "SDL: Info: SDL initialized\n");
   atexit (SDL_Quit);
   return OSD_OK;
}

void sysdep_close(void)
{
  fprintf (stderr, "SDL: Info: SDL Closed\n");
  SDL_Quit();
}


int sysdep_create_display(void)
{
  SDL_Rect** vid_modes;
  SDL_VideoInfo* video_info;
  SDL_Event event;

#ifdef DIRECT_HERMES 
  HermesFormat* H_src_format;
  HermesFormat* H_dst_format;
#endif /* DIRECT_HERMES */
  
  int vid_modes_i;
  int vid_mode_found = 0;
  
  video_info = SDL_GetVideoInfo();
  
  fprintf (stderr,"SDL: Info: Best matching mode: \n");
  fprintf (stderr,"SDL: Info: HW blits %d\n"
	   "SDL: Info: SW blits %d\n"
	   "SDL: Info: Vid mem %d\n"
	   "SDL: Info: Best supported depth %d\n",
	   video_info->blit_hw,
	   video_info->blit_sw,
	   video_info->video_mem,
	   video_info->vfmt->BitsPerPixel);

  Vid_depth = video_info->vfmt->BitsPerPixel;

  vid_modes = SDL_ListModes(NULL,SDL_HWSURFACE);
  vid_modes_i  =0;
  
  if ( (! vid_modes) || ((long)vid_modes == -1)) {
    fprintf (stderr_file, "SDL: Info: Possible all video modes available\n");
    Vid_height = visual_height;
    Vid_width = visual_width;
  } else {

    while ( (vid_modes + vid_modes_i) && (! vid_mode_found)) {
    
#ifdef SDL_DEBUG
      fprintf (stderr_file, "SDL: Info: Found mode %d x %d\n",
	    (*(vid_modes+vid_modes_i))->w,
	       (*(vid_modes+vid_modes_i))->h);
#endif /* SDL_DEBUG */
      
    /* Agree to minimal satisfactional Vmode geometry */
      if ( ((*(vid_modes + vid_modes_i))->w > visual_width) && 
	   ((*(vid_modes + vid_modes_i))->h > visual_height)) {
	vid_mode_found++;
      } else
	vid_modes_i++;
    }
        
    Vid_width = (*(vid_modes + vid_modes_i))->w;
    Vid_height = (*(vid_modes + vid_modes_i))->h;
  }
  
  if (! (Surface = SDL_SetVideoMode(Vid_width, Vid_height,Vid_depth, SDL_HWSURFACE))) 
    {
      fprintf (stderr_file, "SDL: Error: Setting video mode failed\n");
      exit (OSD_NOT_OK);
    } else {
      fprintf (stderr_file, "SDL: Info: Video mode set as %d x %d, depth %d\n", Vid_width, Vid_height, Vid_depth);
    }

#ifndef DIRECT_HERMES   
  Offscreen_surface = SDL_AllocSurface(SDL_SWSURFACE,visual_width,visual_height,8,0,0,0,0); 
#else /* DIRECT_HERMES */
  /* No offscreen surface when using hermes directly */
  H_ConverterHandle = Hermes_ConverterInstance(0);
  H_src_format = Hermes_FormatNew (8,0,0,0,0,HERMES_INDEXED);
  /* TODO: More general destination choosing - uptil
     now only 16 bit */
  H_dst_format = Hermes_FormatNew (16,Surface->format->Rmask,Surface->format->Gmask,Surface->format->Bmask,0,0);
  //   H_dst_format = Hermes_FormatNew (16,5,5,5,0,0);
  if ( ! (Hermes_ConverterRequest(H_ConverterHandle,H_src_format , H_dst_format)) ) {
    fprintf (stderr_file, "Hermes: Info: Converter request failed\n");
    exit (OSD_NOT_OK);
  }
  
#endif
  
  sysdep_mark_palette_dirty();
  osd_mark_dirty (0,0,bitmap->width-1,bitmap->height-1,1);

  /* Creating event mask */
  SDL_EventState(SDL_KEYUP, SDL_ENABLE);
  SDL_EventState(SDL_KEYDOWN, SDL_ENABLE);
  local_key = kstate;
  
  /* TODO - moving to external procedure */
  memset(kstate, 0, sizeof(kstate));
   
  return OSD_OK;
}

void
my_toggle_video_mode (void) 
{
  if (! Surface)
    return;
  Surface = SDL_SetVideoMode(Vid_width, Vid_height, Vid_depth,
			     SDL_FULLSCREEN ^ Surface->flags );
}



/* shut up the display */
void osd_close_display(void)
{
  
   osd_dirty_close();
   osd_free_bitmap(bitmap);
}

/*
 * In 8 bpp we should alloc pallete - some ancient people  
 * are still using 8bpp displays
 */
void sysdep_alloc_palette(void)
{
  int ncolors;
  int i;
  ncolors = totalcolors;

#ifndef DIRECT_HERMES
  Colors = (SDL_Color*) malloc (totalcolors * sizeof(SDL_Color));
  for (i=0;i<totalcolors;i++) {
    (Colors + i)->r = 0xFF;
    (Colors + i)->g = 0x00;
    (Colors + i)->b = 0x00;
  }
  SDL_SetColors (Offscreen_surface,Colors,0,totalcolors-1);
#else /* DIRECT_HERMES */
  H_PaletteHandle = Hermes_PaletteInstance();
  if ( !(H_Palette = Hermes_PaletteGet(H_PaletteHandle)) ) {
    fprintf (stderr_file, "Hermes: Info: PaletteHandle invalid");
    exit(OSD_NOT_OK);
  }
#endif /* DIRECT_HERMES */

  fprintf (stderr_file, "SDL: Info: Palette with %d colors allocated\n", totalcolors);
}

void sysdep_alloc_palette_16bpp(unsigned short *pens)
{
  /* Assuming, that we already have 16bpp display, when
     needed - don't need to allocate */
  fprintf (stderr_file, "STD: Warn: Trying to allocate 16 bit palette - %d totalcolors\n", totalcolors);
}

void sysdep_modify_pen(int pen,unsigned char red, unsigned char green, unsigned char blue)
{
  static int warned = 0;

#ifndef DIRECT_HERMES

  (Colors + pen)->r = red;
  (Colors + pen)->g = green;
  (Colors + pen)->b = blue;
  if ( (! SDL_SetColors(Offscreen_surface, Colors + pen, pen,1)) && (! warned)) {
    printf ("Color allocation failed, or > 8 bit display\n");
    warned = 1;
  }
#else /* DIRECT_HERMES */

  /* LOW ENDIAN */
  
  *(H_Palette + pen) = (red<<16) | ((green) <<8) | (blue );
  H_Palette_modified = 1; 
#endif 

#ifdef SDL_DEBUG
  fprintf(stderr_file, "STD: Debug: Pen %d modification: r %d, g %d, b, %d\n",
	  pen, red,green,blue);
#endif /* SDL_DEBUG */
}

void sysdep_get_pen_16bpp(int pen, unsigned char* red, unsigned char *green,
   unsigned char *blue)
{
}

/* Update the display. */

void sysdep_update_display(void)
{
  int i,j,x,y,w,h;
  int locked =0 ;
 
  static int first_run = 1;

#ifdef DIRECT_HERMES
  int line_amount;
#else /* DIRECT_HERMES */
  SDL_Surface* converted;
  SDL_Rect srect = { 0,0,Vid_width,Vid_height };
  SDL_Rect drect = { 0,0,Vid_width,Vid_height };
#endif /* DIRECT_HERMES */

#ifdef SDL_DEBUG
  static int update_count = 0;
  static char* bak_bitmap;
  int corrected = 0;
  int debug = 0;
#endif /* SDL_DEBUG */

#ifdef DIRECT_HERMES
  if (H_Palette_modified) {
    Hermes_PaletteInvalidateCache(H_PaletteHandle);
    Hermes_ConverterPalette(H_ConverterHandle,H_PaletteHandle,0);
    H_Palette_modified = 0;
  }
#endif /* DIRECT_HERMES */
  

#ifdef PANANOIC 
    memset(Offscreen_surface->pixels,'\0' ,Vid_height * Vid_width);
#endif 

  switch  (use_dirty) {
    long line_min;
    long line_max;
    long col_min;
    long col_amount;

    
#ifdef SDL_DEBUG
    int my_off;
    
#endif     
    


  case 0:
    /* Not using dirty */
#ifndef DIRECT_HERMES
    for (i=0;i<bitmap->height;i++)
      memcpy (Offscreen_surface->pixels + (i * Vid_width), bitmap->line[i],bitmap->width);
#else /* DIRECT_HERMES */
    if (SDL_MUSTLOCK(Surface))
      SDL_LockSurface(Surface);
    
    Hermes_ConverterCopy (H_ConverterHandle, 
			  bitmap->line[0] ,
			  0, 0 , 
			  Vid_width,Vid_height, bitmap->line[1] - bitmap->line[0],
			  Surface->pixels, 
			  0,0,
			  Vid_width, Vid_height, Vid_width <<1 );
    
    SDL_UnlockSurface(Surface);
    SDL_UpdateRect(Surface,0,0,Vid_width,Vid_height);
#endif /* DIRECT_HERMES */

    break;
  case 1:
    
    /* Algorithm:
       search through dirty & find max maximal polygon, which 
       we can get to clipping (don't know if 8x8 is enought)
    */
     
    osd_dirty_merge();
    
  case 2:
    h = (bitmap->height+7) >> 3; /* Divide by 8, up rounding */
    w = (bitmap->width +7) >> 3; /* Divide by 8, up rounding */
    
#ifdef PARANOIC
    /* Rechecking dirty correctness ...*/
    if ( (! first_run) && debug) {
      
      for (y=0;y<h;y++ ) {
	for (i=0;i<8;i++) {
	  int line_off = ((y<<3) + i);
	  for (x=0;x<w;x++) {
	    for (j=0;j<8;j++) {
	      int col_off = ((x<<3) + j);
	      if ( *(bak_bitmap + (line_off * (bitmap->line[1]- bitmap->line[0])) + col_off ) != *(*(bitmap->line + line_off) + col_off)) {
		if (! dirty_blocks[y][x] ) {
		  printf ("Warn!!! Block should be dirty %d, %d, %d - correcting \n",y,x,i);
		  dirty_blocks[y][x] = 1;
		  dirty_lines[y]=1;
		  corrected = 1;
		  
		}
	      } 
	      
	    }
	  }
	}
      }
    } else {
      bak_bitmap = (void*)malloc(w<<3 * h<<3);
    }
    
    if (! corrected) {
      printf ("dirty ok\n");
    }
    
    
    first_run = 0;
    for (i = 0;i< bitmap->height;i++)
      memcpy(bak_bitmap + (bitmap->line[1] - bitmap->line[0])*i,*(bitmap->line+i),w<<3);
    
#endif /* PARANOIC */

    // #define dirty_lines old_dirty_lines    
    // #define dirty_blocks old_dirty_blocks
    
    for (y=0;y<h;y++) {
      if (dirty_lines[y]) {
	line_min = y<<3;
	line_max = line_min + 8;
	//	old_dirty_lines[y]=1;
	for (x=0;x<w;x++) {
	  if (dirty_blocks[y][x]) {
	    col_min = x<<3;
	    col_amount = 0;
	    do { 
 	      col_amount++; 
 	      dirty_blocks[y][x] = 0;
	      x++; 
 	    } while (dirty_blocks[y][x]); 
	    dirty_blocks[y][x] = 0;
	    col_amount <<= 3;

#ifdef DIRECT_HERMES
	    line_amount = line_max - line_min;
	    /* Trying to use direct hermes library for fast blitting */
	    if (SDL_MUSTLOCK(Surface))
	      SDL_LockSurface(Surface);
	    
	    Hermes_ConverterCopy (H_ConverterHandle, 
				  bitmap->line[0] ,
				  col_min, line_min , 
				  col_amount,line_amount, bitmap->line[1] - bitmap->line[0],
				  Surface->pixels, 
				  col_min, line_min, 
				  col_amount ,line_amount, Vid_width <<1 );
	    
	    SDL_UnlockSurface(Surface);
	    SDL_UpdateRect(Surface,col_min,line_min,col_amount,line_amount);
#else 	    
			    for (i=line_min;i<line_max;i++)
				  memcpy(Offscreen_surface->pixels + (i * Vid_width) + col_min, bitmap->line[i] + col_min, col_amount );
				  
				  //	    old_dirty_blocks[y][x] = 1;
#endif
	  }
	  
	
	}
	dirty_lines[y] = 0;
      }    
    }
    
    /* Vector game .... */
    break;
    return ;
  }
  
  /* TODO - It's the real evil - better to use would be original 
     hermes routines */

#ifndef DIRECT_HERMES
  converted = SDL_DisplayFormat(Offscreen_surface);
  if(! converted)
    fprintf (stderr_file, "SDL: Warn: Surface conversion failed\n");
  
  else {
    if (SDL_MUSTLOCK(Surface))
      SDL_LockSurface(Surface);


    if (SDL_BlitSurface (converted, &srect, Surface, &drect)<0) 
      fprintf (stderr_file,"SDL: Warn: Unsuccessful blitting\n");
    SDL_UnlockSurface(Surface);
    SDL_FreeSurface(converted);
    SDL_UpdateRects (Surface,1, &drect);
  }
#endif /* DIRECT_HERMES */
  update_count++;
}

int sysdep_mapkey(char *arg) 
{
  return OSD_OK;
}

void sysdep_mouse_poll (void)
{
}

/* Keyboard procs */
/* Lighting keyboard leds */
void  osd_led_w (int led, int on) 
{
}

void sysdep_update_keyboard() 
{
  SDL_Event event;
  
  if (Surface) {
    while(SDL_PollEvent(&event)) {
      unsigned char key_sym;
      switch (event.type) {
      case SDL_KEYDOWN:
	key_sym  = klookup[event.key.keysym.sym];
	switch  (key_sym) {
	case 0:
	  fprintf (stderr_file,"Unknown symbol 0x%x\n",event.key.keysym.sym);
	  break;
	default:
	  kstate[key_sym] = 1;
	}
	

#ifdef SDL_DEBUG
	fprintf (stderr_file, "Key %s pressed\n",SDL_GetKeyName(event.key.keysym.sym));
#endif /* SDL_DEBUG */
	break;
      case SDL_KEYUP:
	key_sym  = klookup[event.key.keysym.sym];
	switch  (key_sym) {
	case 0:
	  fprintf (stderr_file, "Unknown symbol 0x%x\n",event.key.keysym.sym);
	  break;
	default:
	  kstate[key_sym] = 0;
	}
      
	
#ifdef SDL_DEBUG
	fprintf (stderr_file, "Key %s released\n",SDL_GetKeyName(event.key.keysym.sym));
#endif /* SDL_DEBUG */
	break;
      case SDL_QUIT:
	/* Shoult leave this to application */
	exit(OSD_OK);
	break;
      default:
#ifdef SDL_DEBUG
	fprintf(stderr_file, "SDL: Debug: Other event\n");
#endif /* SDL_DEBUG */
    }
    }
  }
}
