/* move_points.c */
/************************************************************
 *                                                          *
 *  Permission is hereby granted  to  any  individual   or  *
 *  institution   for  use,  copying, or redistribution of  *
 *  the xgobi code and associated documentation,  provided  *
 *  that   such  code  and documentation are not sold  for  *
 *  profit and the  following copyright notice is retained  *
 *  in the code and documentation:                          *
 *     Copyright (c)  AT&T Labs (1998).                     *
 *  All Rights Reserved.                                    *
 *                                                          *
 *  We welcome your questions and comments, and request     *
 *  that you share any modifications with us.               *
 *                                                          *
 *      Deborah Swayne             Andreas Buja             *
 *    dfs@research.att.com     andreas@research.att.com     *
 *                                                          *
 ************************************************************/

#include <stdio.h>
#include "xincludes.h"
#include "xgobitypes.h"
#include "xgobivars.h"
#include "xgobiexterns.h"

int moving_point = -1;

/* choose a cursor at some point */

#define UP 0
#define DOWN 1
static Boolean buttonpos = UP;
static int last_moved = -1;
static int last_x = -1, last_y = -1;
lcoords eps;
icoords prev_planar;

static Widget mp_panel;
static Widget reset_all_cmd, reset_one_cmd;

static Boolean use_brush_groups = False;

static Widget mpdir_menu_label, mpdir_menu_cmd, mpdir_menu, mpdir_menu_btn[3];
static char *mpdir_menu_btn_label[] = {"Both", "Vert", "Horiz"};
static enum {both, vert, horiz} mpdir_type = both;

/******************** Reverse pipeline ********************/

void
screen_to_plane(xgobidata *xg, int pt)
{
  find_plot_center(xg);

  if (mpdir_type == horiz || mpdir_type == both) {
    xg->screen[pt].x -= xg->cntr.x;
    prev_planar.x = xg->planar[pt].x;
    xg->planar[pt].x = (float) xg->screen[pt].x * PRECISION1 / xg->is.x ;
    eps.x = xg->planar[pt].x - prev_planar.x;
  }

  if (mpdir_type == vert || mpdir_type == both) {
    xg->screen[pt].y -= xg->cntr.y;
    prev_planar.y = xg->planar[pt].y;
    xg->planar[pt].y = (float) xg->screen[pt].y * PRECISION1 / xg->is.y ;
    eps.y = xg->planar[pt].y - prev_planar.y;
  }
}

void
plane_to_world(xgobidata *xg, int pt)
{
  int j, var; 

  if (xg->is_xyplotting) {

    xg->world_data[pt][xg->xy_vars.x] = xg->planar[pt].x;
    xg->world_data[pt][xg->xy_vars.y] = xg->planar[pt].y;

  } else if (xg->is_spinning) {

    long nx, ny, nz;
    long lx, ly, lz;
    int ix = xg->spin_vars.x;
    int iy = xg->spin_vars.y;
    int iz = xg->spin_vars.z;

    nx = xg->planar[pt].x;
    ny = xg->planar[pt].y;

    if (xg->is_spin_type.oblique) {
      extern long lRmat[3][3];

      /*
       * Use the new values of x and y, but reconstruct
       * what the old value of nz would have been, had we
       * calculated it in ob_rot_reproject.
      */
      nz =
        lRmat[2][0] * xg->world_data[pt][ix] +
        lRmat[2][1] * xg->world_data[pt][iy] +
        lRmat[2][2] * xg->world_data[pt][iz];
      nz /= PRECISION2;

      /*
       * Calculate the inner product of t(lRmat) and (nx, ny, nz)
       * (This works because the transpose of lRmat is its inverse.)
      */ 
      lx = lRmat[0][0] * nx + lRmat[1][0] * ny + lRmat[2][0] * nz;
      ly = lRmat[0][1] * nx + lRmat[1][1] * ny + lRmat[2][1] * nz;
      lz = lRmat[0][2] * nx + lRmat[1][2] * ny + lRmat[2][2] * nz;

      xg->world_data[pt][ix] = lx / PRECISION2;
      xg->world_data[pt][iy] = ly / PRECISION2;
      xg->world_data[pt][iz] = lz / PRECISION2;
    }
    else if (xg->is_spin_type.yaxis) {

      /* nz isn't normally used or calculated; do it here */
      nz = -xg->isint.y * xg->world_data[pt][ix] +
        xg->icost.y * xg->world_data[pt][iz];
      nz /= PRECISION2;

      /*
       * Calculate the inner product of the transpose of the
       * rotation matrix and (nx, ny, nz)
      */ 
      lx = xg->icost.y * nx + -xg->isint.y * nz;
      ly = ny;
      lz = xg->isint.y * nx + xg->icost.y * nz;

      xg->world_data[pt][ix] = lx / PRECISION2;
      xg->world_data[pt][iy] = ly;
      xg->world_data[pt][iz] = lz / PRECISION2;
    }
    else if (xg->is_spin_type.xaxis) {

      /* nz isn't normally used or calculated; do it here */
      nz = -xg->isint.x * xg->world_data[pt][iy] +
        xg->icost.x * xg->world_data[pt][iz];
      nz /= PRECISION2;

      /*
       * Calculate the inner product of the transpose of the
       * rotation matrix and (nx, ny, nz)
      */ 
      lx = nx;
      ly = xg->icost.x * ny - xg->isint.x * nz;
      lz = xg->isint.x * ny + xg->icost.x * nz;

      xg->world_data[pt][ix] = lx;
      xg->world_data[pt][iy] = ly / PRECISION2;
      xg->world_data[pt][iz] = lz / PRECISION2;
    }
  } 
  else if (xg->is_touring && !xg->is_pp) {
    for (j=0; j<xg->numvars_t; j++) {
      var = xg->tour_vars[j];
      xg->world_data[pt][var] += 
       (eps.x * xg->u[0][var] + eps.y * xg->u[1][var]);
    }
  }
  else if (xg->is_corr_touring && !xg->is_corr_pursuit) {
    for (j=0; j<xg->ncorr_xvars; j++) {
      var = xg->corr_xvars[j];
      xg->world_data[pt][var] += 
       (eps.x * xg->cu[0][var] + eps.y * xg->cu[1][var]);
    }
    for (j=0; j<xg->ncorr_yvars; j++) {
      var = xg->corr_yvars[j];
      xg->world_data[pt][var] += 
       (eps.x * xg->cu[0][var] + eps.y * xg->cu[1][var]);
    }
  }
}

/*
 * world_data includes jitter_data
*/

void
world_to_raw_by_var(xgobidata *xg, int pt, int var)
{
  float precis = PRECISION1;
  float ftmp, rdiff;

  rdiff = xg->lim0[var].max - xg->lim0[var].min;

  /*
   * The new world_data value is taken to include the
   * value of jitter_data
  */
  ftmp = (xg->world_data[pt][var] - xg->jitter_data[pt][var]) / precis;

  xg->tform_data[pt][var] = (ftmp + 1.0) * .5 * rdiff;
  xg->tform_data[pt][var] += xg->lim0[var].min;

/*
 * Do a reverse transform of the tform data point and send
 * that back to the raw data array.
*/
  xg->raw_data[pt][var] = inv_transform(pt, var, xg);
}

void
world_to_raw(xgobidata *xg, int pt)
{
  int i;

  if (xg->is_xyplotting) {
    world_to_raw_by_var(xg, pt, xg->xy_vars.x);
    world_to_raw_by_var(xg, pt, xg->xy_vars.y);
  } else if (xg->is_spinning) {
    world_to_raw_by_var(xg, pt, xg->spin_vars.x);
    world_to_raw_by_var(xg, pt, xg->spin_vars.y);
    world_to_raw_by_var(xg, pt, xg->spin_vars.z);
  } else if (xg->is_touring) {
    for (i=0; i<xg->numvars_t; i++)
      world_to_raw_by_var(xg, pt, xg->tour_vars[i]);
  } else if (xg->is_corr_touring) {
    /* something like this, I guess */
    for (i=0; i<xg->ncorr_xvars; i++)
      world_to_raw_by_var(xg, pt, xg->corr_xvars[i]);
    for (i=0; i<xg->ncorr_yvars; i++)
      world_to_raw_by_var(xg, pt, xg->corr_yvars[i]);
  }
}

/******************* End of reverse pipeline *******************/


void
init_point_moving(xgobidata *xg)
{
  xg->is_point_moving = False;
}

/* ARGSUSED */
XtEventHandler
mp_button(Widget w, xgobidata *xg, XEvent *evnt, Boolean *cont)
{
  XButtonEvent *xbutton = (XButtonEvent *) evnt;

  if (xbutton->button == 1) {
    if (xbutton->type == ButtonPress) {
      buttonpos = DOWN;
      last_moved = xg->nearest_point;
      last_x = xg->screen[xg->nearest_point].x;
      last_y = xg->screen[xg->nearest_point].y;
    }
    else if (xbutton->type == ButtonRelease) {
      buttonpos = UP;

      moving_point = -1;
    }
  }
}

void
move_pt_pipeline(int id, xgobidata *xg)
{
  /* run the pipeline backwards */
  /* screen_to_plane(xg, id); */
  plane_to_world(xg, id);
  world_to_raw(xg, id);
}

void
move_pt(int id, int x, int y, xgobidata *xg) {
  int i, k;

  if (mpdir_type == horiz || mpdir_type == both)
    xg->screen[id].x = x;
  if (mpdir_type == vert || mpdir_type == both)
    xg->screen[id].y = y;

/*
 * I'm going to import screen_to_plane in here to get better
 * control over the motion of clusters.  Perfect.
*/
  screen_to_plane(xg, id);

  /* Move the selected point */
  move_pt_pipeline(id, xg);

  if (use_brush_groups) {
    if (xg->nclust > 1) {
      float cur_clust = xg->raw_data[id][xg->ncols_used-1];

      /*
       * Move all points which belong to the same cluster
       * as the selected point.
      */
      for (i=0; i<xg->nrows_in_plot; i++) {
        k = xg->rows_in_plot[i];
        if (k == id)
          ;
        else {
          if (xg->raw_data[k][xg->ncols_used-1] == cur_clust) {
            if (!xg->erased[k]) {   /* ignore erased values altogether */
              if (mpdir_type == horiz || mpdir_type == both)
                xg->planar[k].x += eps.x;
              if (mpdir_type == vert || mpdir_type == both)
                xg->planar[k].y += eps.y;
              move_pt_pipeline(k, xg);
            }
          }
        }
      }
    }
  }

  /* and now forward again, all the way ... */
  update_world(xg);
  world_to_plane(xg);
  plane_to_screen(xg);
  if (xg->is_brushing) {
    assign_points_to_bins(xg);
    if (xg->brush_mode == transient)
      reinit_transient_brushing(xg);
  }
  plot_once(xg);
  if (xg->is_cprof_plotting && xg->link_cprof_plotting)
    update_cprof_plot(xg);
}

void
move_points_proc(xgobidata *xg)
{
  int root_x, root_y;
  unsigned int kb;
  Window root, child;
  static int ocpos_x = 0, ocpos_y = 0;
  icoords cpos;
  static int inwindow = 1;
  int wasinwindow;
  Boolean pointer_moved = False;

  wasinwindow = inwindow;

/*
 * Find the nearest point, just as in identification, and highlight it
*/

/*
 * Get the current pointer position.
*/
  if (XQueryPointer(display, xg->plot_window, &root, &child,
            &root_x, &root_y, &cpos.x, &cpos.y, &kb))
  {
    inwindow = (0 < cpos.x && cpos.x < xg->max.x &&
                0 < cpos.y && cpos.y < xg->max.y) ;
    /*
     * If the pointer is inside the plotting region ...
    */
    if (inwindow)
    {
      /*
       * If the pointer has moved ...
      */
      if ((cpos.x != ocpos_x) || (cpos.y != ocpos_y)) {
        pointer_moved = True;
        ocpos_x = cpos.x;
        ocpos_y = cpos.y;
      }
    }
  }

  if (pointer_moved) {
    
    if (buttonpos == UP) {
      moving_point = -1;
      if ( (xg->nearest_point = find_nearest_point(&cpos, xg)) != -1) {
        quickplot_once(xg);
      }
    }
    else {
      /*
       * If the left button is down, move the point: compute the
       * data pipeline in reverse, (then run it forward again?) and
       * draw the plot.
      */
      if (buttonpos == DOWN && xg->nearest_point != -1) {
        moving_point = xg->nearest_point;
        move_pt(xg->nearest_point, cpos.x, cpos.y, xg);
      }
    }
  }

  if (!inwindow && wasinwindow) {
    /*
     * Don't draw the diamond if the pointer leaves the plot window.
    */
    xg->nearest_point = -1;
    quickplot_once(xg);
  }
}

static void
map_move_points(xgobidata *xg, Boolean on) {
  if (on) {
    XtMapWidget(mp_panel);
  }
  else {
    XtUnmapWidget(mp_panel);
  }
}

static XtCallbackProc
reset_all_cback(Widget w, xgobidata *xg, XtPointer cb_data) {
/*
 * read the raw data back in here instead of having to go
 * to the file menu to accomplish it.
*/
  Boolean reset = False;

  if (xg->datafilename != "") {
    if (reread_dat(xg->datafilename, xg)) {
      plot_once(xg);
      reset = True;
    }
  }

  if (!reset)
    show_message(
     "Sorry, I can\'t reset because I can\'t reread the data file\n", xg);
}

static XtCallbackProc
reset_one_cback(Widget w, xgobidata *xg, XtPointer cb_data) {
/*
 * move the most recently moved point back to its position
 * when the mouse went down
*/
  if (last_moved >=0 && last_moved < xg->nrows_in_plot) {
    if ((last_x > -1 && last_x < xg->plotsize.width) &&
        (last_y > -1 && last_y < xg->plotsize.height))
    {
      move_pt(last_moved, last_x, last_y, xg);
    }
  }
}

/* ARGSUSED */
static XtCallbackProc
use_groups_cback(Widget w, xgobidata *xg, XtPointer callback_data)
{
  use_brush_groups = !use_brush_groups;

  if (use_brush_groups) {
    if (xg->ncols_used < xg->ncols) {
      save_brush_groups(xg);
    }
  }
  setToggleBitmap(w, use_brush_groups);

}

XtCallbackProc
set_mpdir_cback(Widget w, xgobidata *xg, XtPointer cb)
{
  int k;
  int id;

  for (k=0; k<3; k++) {
    if (w == mpdir_menu_btn[k]) {
      id = k;
      break;
    }
  }

  XtVaSetValues(mpdir_menu_cmd,
    XtNlabel, mpdir_menu_btn_label[id],
    NULL);

  if (id == 0)
    mpdir_type = both;
  else if (id == 1)
    mpdir_type = vert;
  else if (id == 2)
    mpdir_type = horiz;
}

void
make_move_points(xgobidata *xg) {
  Widget use_groups_cmd;
  Widget mpdir_box;
  int k;

  mp_panel = XtVaCreateManagedWidget("MvPtsPanel",
    boxWidgetClass, xg->box0,
    XtNleft, (XtEdgeType) XtChainLeft,
    XtNright, (XtEdgeType) XtChainLeft,
    XtNtop, (XtEdgeType) XtChainTop,
    XtNbottom, (XtEdgeType) XtChainTop,
    XtNmappedWhenManaged, (Boolean) False,
    NULL);
  if (mono) set_mono(mp_panel);

  build_labelled_menu(&mpdir_box, &mpdir_menu_label, "MotionDir:",
    &mpdir_menu_cmd, &mpdir_menu, mpdir_menu_btn,
    mpdir_menu_btn_label, mpdir_menu_btn_label,  /* no nicknames */
    3, 0, mp_panel, mpdir_box,
    XtorientHorizontal, appdata.font, "MovePts", xg);
  for (k=0; k<3; k++)
    XtAddCallback(mpdir_menu_btn[k],  XtNcallback,
      (XtCallbackProc) set_mpdir_cback, (XtPointer) xg);

  use_groups_cmd = CreateToggle(xg, "Use 'group' var",
    True, (Widget) NULL, (Widget) NULL, (Widget) NULL, use_brush_groups,
    ANY_OF_MANY, mp_panel, "MP_Groups");
  XtManageChild(use_groups_cmd);
  XtAddCallback(use_groups_cmd, XtNcallback,
    (XtCallbackProc) use_groups_cback, (XtPointer) xg);

  reset_all_cmd = CreateCommand(xg, "Reset all",
    True, (Widget) NULL, (Widget) NULL,
    mp_panel, "MP_Reset");
  XtManageChild(reset_all_cmd);
  if (xg->progname == NULL)
    XtSetSensitive(reset_all_cmd, False);
  XtAddCallback(reset_all_cmd, XtNcallback,
    (XtCallbackProc) reset_all_cback, (XtPointer) xg);

  reset_one_cmd = CreateCommand(xg, "Undo last",
    True, (Widget) NULL, (Widget) NULL,
    mp_panel, "MP_Reset");
  XtManageChild(reset_one_cmd);
  XtAddCallback(reset_one_cmd, XtNcallback,
    (XtCallbackProc) reset_one_cback, (XtPointer) xg);
}

void move_points_on (xgobidata *xg)
{
/*
 *  If this mode is currently selected, turn it off.
*/
  if (xg->prev_plot_mode == MOVEPTS_MODE && xg->plot_mode != MOVEPTS_MODE)
  {
    xg->is_point_moving = False;
    XtRemoveEventHandler(xg->workspace,
      XtAllEvents, TRUE,
      (XtEventHandler) mp_button, (XtPointer) xg);
    map_move_points(xg, False);
    xg->nearest_point = -1;
    plot_once(xg);
  }
  /* Else -- if xyplotting -- turn it on */
  else if (xg->plot_mode == MOVEPTS_MODE &&
           xg->prev_plot_mode != MOVEPTS_MODE)
  {
    if (xg->is_xyplotting || xg->is_spinning ||
      (xg->is_touring && !xg->is_pp) ||
      (xg->is_corr_touring && !xg->is_corr_pursuit))
    {
      xg->is_point_moving = True;
      map_move_points(xg, True);
      XtAddEventHandler(xg->workspace,
        ButtonPressMask | ButtonReleaseMask, FALSE,
        (XtEventHandler) mp_button, (XtPointer) xg);
      (void) XtAppAddWorkProc(app_con, RunWorkProcs, (XtPointer) NULL);
      /* Should use a different cursor ... */

    } else {
      show_message("Moving points won\'t work in this mode; sorry\n", xg);

      /*
       * It may be necessary to disable point moving if xg->std_type != 0
      */
    }
  }
}
