/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		Socket.h

	Contains:	Provides a simple, object oriented socket abstraction, also
				hides the details of socket event handling. Sockets can post
				events (such as S_DATA, S_CONNECTIONCLOSED) to Tasks.
					
	
	$Log: Socket.h,v $
	Revision 1.2  1999/02/19 23:13:19  ds
	Created
	
	
*/

#ifndef __SOCKET_H__
#define __SOCKET_H__

#include "OSThread.h"
#include "Task.h"
#include "OSRef.h"
#include "QTSS.h"

#ifndef __MW_
#include <netinet/in.h>
#include "ev.h"
#endif

class SocketEventQueueThread;

class Socket
{
public:

	//What type of socket do you want this to be?
	enum
	{
		kNonBlocking = 0x0001,
		kWantsEvents = 0x0002
	};

	//call initialize before using any socket functions
	static void Initialize();

	//call start thread before expecting any event activity
	static void StartThread();
	
	//Binds the socket to the following address.
	//Returns: QTSS_FileNotOpen, QTSS_NoErr, or POSIX errorcode.
	QTSS_ErrorCode 	Bind(UInt32 addr, UInt16 port);
	//The same. but in reverse
	void			Unbind();	
	
	//Sets the SO_REUSEADDR socket option
	void			ReuseAddr();
	
	//Send
	//Returns: QTSS_FileNotOpen, QTSS_NoErr, or POSIX errorcode.
	QTSS_ErrorCode 	Send(const char* inData, const UInt32 inLength, UInt32* outLengthSent = NULL);

	//Read
	//Reads some data.
	//Returns: QTSS_FileNotOpen, QTSS_NoErr, or POSIX errorcode.
	QTSS_ErrorCode 	Read(void *buffer, const UInt32 length, UInt32 *rcvLen);
	
	//WriteV: same as send, but takes an iovec
	//Returns: QTSS_FileNotOpen, QTSS_NoErr, or POSIX errorcode.
	QTSS_ErrorCode		WriteV(const struct iovec* iov, const UInt32 numIOvecs);
	
	//You can query for the socket's state
	bool	IsConnected() 	{ return fState & kConnected; }
	bool	IsBound() 		{ return fState & kBound; }
	
	//If the socket is bound, you may find out to which addr it is bound
	UInt32	GetLocalAddr()	{ return ntohl(fLocalAddr.sin_addr.s_addr); }
	UInt16	GetLocalPort()	{ return ntohs(fLocalAddr.sin_port); }
	
	StrPtrLen*	GetLocalAddrStr();
	StrPtrLen*	GetLocalPortStr();
	StrPtrLen* GetLocalDNSStr();

	enum
	{
		kMaxNumSockets = 4096	//UInt32
	};

protected:

	//TCPSocket takes an optional task object which will get notified when
	//certain events happen on this socket. Those events are:
	//
	//S_DATA:				Data is currently available on the socket.
	//S_CONNECTIONCLOSING:	Client is closing the connection. No longer necessary
	//						to call Close or Disconnect, Snd & Rcv will fail.
	
	Socket(UInt32 inSocketType, Task *notifytask = NULL);
	virtual ~Socket();

	//returns QTSS_NoErr, or appropriate posix error
	QTSS_ErrorCode 	Open(int theType);
	void 			InitNonblocking();
	void			Modwatch(int theMask = EV_RE);
	
	//this function can be overridden by specific socket types.
	//default is to just signal the associated task object (if it exists)
	virtual void ProcessEvent(Task::EventFlags theEvent)
		{ if (fTask != NULL) fTask->Signal(theEvent); }
	
	int				fSocket;
	Task*			fTask;
	bool			fNonblocking;
	OSRef			fRef;
	unsigned int	fUniqueID;
	StrPtrLen 		fUniqueIDStr;
	UInt32			fState;
	struct eventreq fEventReq;
	
	enum
	{
		kInvalidSocket = -1,	//int
		kPortBufSizeInBytes = 8,	//UInt32
		kMaxIPAddrSizeInBytes = 20	//UInt32
	};
	
	//address information (available if bound)
	//these are always stored in network order. Conver
	struct sockaddr_in	fLocalAddr;
	struct sockaddr_in	fDestAddr;
	
	char fLocalAddrStr[kMaxIPAddrSizeInBytes];
	StrPtrLen fLocalAddrStrPtr;
	StrPtrLen* fLocalDNSStrPtr;
	char fPortBuffer[kPortBufSizeInBytes];
	StrPtrLen fPortStr;
	
#if DEBUG
	bool	fModwatched;
#endif
	
	//State flags. Be careful when changing these values, as subclasses add their own
	enum
	{
		kBound		= 0x0004,
		kConnected 	= 0x0008
	};
	
	static SocketEventQueueThread* sThread;
	static unsigned int				sUniqueID;
	static OSRefTable*				sRefTable;
	
	friend class SocketEventQueueThread;
};

class SocketEventQueueThread : private OSThread
{
	//This class handles the asynchronicity of these sockets. ie, all sockets
	//that are in nonblocking mode get woken up when there is data or interesting stuff
	private:
	
		SocketEventQueueThread() : OSThread("SocketEventQueueThread") {}
		virtual ~SocketEventQueueThread() {}
		
		virtual void Entry();
		
		friend class Socket;
};
#endif // __SOCKET_H__

