
/*
 * SERVER.C	- /news/dserver.hosts management
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype void CheckServerConfig(time_t t, int force);
Prototype void NNArticleRetrieveByMessageId(Connection *conn, const char *msgid);
Prototype void NNServerIdle(Connection *conn);
Prototype void NNServerTerminate(Connection *conn);
Prototype void NNFinishSReq(Connection *conn, const char *ctl, int requeue);
Prototype void NNServerRequest(Connection *conn, const char *grp, const char *msgid, FILE *cache, int req);

Prototype int   ServersTerminated;
Prototype int	NReadServers;
Prototype int	NReadServAct;
Prototype int	NWriteServers;
Prototype int	NWriteServAct;

void NNServerStart(Connection *conn);
void QueueServerRequests(void);
int queueServerRequest(ServReq *sreq, int type, int maxq);
void queueDesc(ServReq *sreq, ForkDesc *desc);
int requeueServerRequest(Connection *conn, ServReq *sreq, int type, int maxq);
int AddServer(const char *host, int type, int port, int flags, int pri);
void NNServerPrime1(Connection *conn);
void NNServerPrime2(Connection *conn);
void NNServerPrime3(Connection *conn);
void NNServerConnect(Connection *conn);

ServReq	*SReadBase;
ServReq **PSRead = &SReadBase;
ServReq *SWriteBase;
ServReq **PSWrite = &SWriteBase;
int	NReadServers;
int	NReadServAct;
int	NWriteServers;
int	NWriteServAct;
int	ServersTerminated;

/*
 * CheckServerConfig() - determine if configuration file has changed and
 *			 resynchronize with servers list.
 *
 *			 NOTE: a server cannot be destroyed until its current
 *			 pending request completes, but no new requests will
 *			 be queued to it during that time.
 */

void
setServerMaybeCloseFlag(ForkDesc *desc)
{
    Connection *conn = desc->d_Data;
    conn->co_Flags |= COF_MAYCLOSESRV;
    if (conn->co_Func == NNServerIdle)		/* wakeup idle server */
	FD_SET(desc->d_Fd, &RFds);
}

void
setServerClosedRemovals(ForkDesc *desc)
{
    Connection *conn;

    if ((conn = desc->d_Data) != NULL) {
	if (conn->co_Flags & COF_MAYCLOSESRV) {
	    NNServerTerminate(conn);
	}
    }
}

void 
CheckServerConfig(time_t t, int force)
{
    static struct stat St;
    struct stat st;
    const char *path = PatLibExpand(DServerHostsPat);

    printf("CheckServerConfig %d %d\n", force, ServersTerminated);

    /*
     * Assuming we can stat the file, if we haven't checked the server config
     * before or the server config time is different from when we last checked
     * AND the server config time is not now (so we don't read the file while
     * someone is writing it), then read the configuration file (again).
     * I've also got some reverse-time-index protection in there.
     *
     * NOTE: path only valid until next Pat*Expand() call, so be careful
     * with it's use.
     */

    if (force || (stat(path, &st) == 0 && 
	(St.st_mode == 0 || st.st_mtime != St.st_mtime) && 
	((long)(t - st.st_mtime) > 2 || (long)(t - st.st_mtime) < 0))
    ) {
	FILE *fi;
	if ((fi = fopen(path, "r")) != NULL) {
	    char buf[256];

	    St = st;

	    ScanThreads(THREAD_SPOOL, setServerMaybeCloseFlag);
	    ScanThreads(THREAD_POST, setServerMaybeCloseFlag);

	    while (fgets(buf, sizeof(buf), fi) != NULL) {
		char *host = strtok(buf, " \t\n");
		char *flags = "";
		char c;
		int addAsServer = 0;
		int addAsPoster = 0;
		int port = 119;
		int nflags = 0;
		int npri = 0;

		if (host == NULL || host[0] == '#')
		    continue;
		if ((flags = strtok(NULL, " \t\n")) == NULL)
		    flags = "";
		while ((c = *flags) != 0) {
		    ForkDesc *desc = NULL;

		    switch(c) {
		    case 'p':	/* port			*/
			port = strtol(flags + 1, NULL, 0);
			if (port == 0)
			    port = 119;
			break;
		    case 's':	/* spool		*/
			npri = strtol(flags + 1, NULL, 10);
			if ((desc = FindThreadId(THREAD_SPOOL, host)) != NULL) {
			    Connection *conn = desc->d_Data;
			    conn->co_Flags &= ~COF_MAYCLOSESRV;
			} else {
			    addAsServer = 1;
			}
			break;
		    case 'M':
			nflags = COF_MODEREADER;
			break;
		    case 'o':	/* outgoing (post)	*/
			npri = strtol(flags + 1, NULL, 10);
			if ((desc = FindThreadId(THREAD_POST, host)) != NULL) {
			    Connection *conn = desc->d_Data;
			    conn->co_Flags &= ~COF_MAYCLOSESRV;
			} else {
			    addAsPoster = 1;
			}
			break;
		    default:
			/*
			 * ignore unknown flag or number
			 */
			break;
		    }
		    ++flags;
		}
		if (addAsServer) {
		    if (AddServer(host, THREAD_SPOOL, port, nflags, npri) == 0) {
			++NReadServers;
			++NReadServAct;
		    } else {
			++ServersTerminated;
		    }
		}
		if (addAsPoster) {
		    if (AddServer(host, THREAD_POST, port, nflags, npri) == 0) {
			++NWriteServers;
			++NWriteServAct;
		    } else {
			++ServersTerminated;
		    }
		}
	    }
	    fclose(fi);
	    ScanThreads(THREAD_SPOOL, setServerClosedRemovals);
	    ScanThreads(THREAD_POST, setServerClosedRemovals);
	}
    }
}

int
AddServer(const char *host, int type, int port, int flags, int pri)
{
    ForkDesc *desc;
    int fd;
    struct sockaddr_in sin;
    Connection *conn;

    /*
     * connect() to the host (use asynchronous connect())
     */

    bzero(&sin, sizeof(sin));
    {
	struct hostent *he = gethostbyname(host);
        if (he != NULL) {
	    sin.sin_family = he->h_addrtype;
	    memmove(&sin.sin_addr, he->h_addr, he->h_length);
	} else if (strtol(host, NULL, 0) != 0) {
	    sin.sin_family = AF_INET;
	    sin.sin_addr.s_addr = inet_addr(host);
	} else {
	    logit(LOG_ERR, "hostname lookup failure: %s\n", host);
	    return(-1);
	}
    }
    sin.sin_port = htons(port);

    if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	logit(LOG_ERR, "socket() call failed on host %s\n", host);
	return(-1);
    }
    {
	int on;
	setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, (void *)&on, sizeof(on));
	/* 
	 * XXX set TxBufSize, RxBufSize, but different from bufsize options
	 * for client connections.. make these bigger.
	 */
    }
    fcntl(fd, F_SETFL, O_NONBLOCK);	/* asynchronous connect() */
    errno = 0;
    if (connect(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	if (errno != EINPROGRESS) {
	    close(fd);
	    logit(LOG_ERR, "connect() call failed on host %s: %s\n", host, strerror(errno));
	    return(-1);
	}
    }

    /*
     * add thread.  Preset d_Count to LIMSIZE to prevent the server
     * from being allocated for client requests until we know
     * we have a good connection.
     */

    desc = AddThread(host, fd, -1, type, -1, pri);
    FD_SET(desc->d_Fd, &RFds);
    conn = InitConnection(desc, NULL);
    conn->co_Flags |= flags;
    conn->co_Flags |= COF_INPROGRESS | COF_ININIT;
    desc->d_Count = THREAD_LIMSIZE;
    NNServerConnect(conn);
    return(0);
}

/*
 * QUEUESERVERREADREQUEST() - move requests to the appropriate server.
 *
 *	New article fetch and post requests are placed on the SRead/SWrite
 *	lists.  This function moves as many of those requests as possible to
 *	actual spool & post server queues.
 *
 *	Even though a server can only handle one request at a time, we 
 *	allow up to N requests (THREAD_QSIZE/PSIZE) to be queued to an
 *	actual server in order to judge load.  If the load exceeds the
 *	queue limit, FindLeastUsedThread will start queueing to higher priority
 *	servers.
 */

void
QueueServerRequests(void)
{
    ServReq	*sreq;

    while ((sreq = SReadBase) != NULL) {
	if ((SReadBase = sreq->sr_Next) == NULL)
	    PSRead = &SReadBase;
	sreq->sr_Next = NULL;
	if (queueServerRequest(sreq, THREAD_SPOOL, THREAD_QSIZE) < 0) {
	    *PSRead = sreq;
	    PSRead = &sreq->sr_Next;
	    break;
	}
    }
    while ((sreq = SWriteBase) != NULL) {
	if ((SWriteBase = sreq->sr_Next) == NULL)
	    PSWrite = &SWriteBase;
	sreq->sr_Next = NULL;
	if (queueServerRequest(sreq, THREAD_POST, THREAD_PSIZE) < 0) {
	    *PSWrite = sreq;
	    PSWrite = &sreq->sr_Next;
	    break;
	}
    }
}

/*
 * queueServerRequest() - find best server to queue request to, return 0
 *			  on success, -1 if the request could not be queued.
 *
 *			  On success, the request will have been properly
 *			  queued.
 */

int
queueServerRequest(ServReq *sreq, int type, int maxq)
{
    ForkDesc *desc;
    static int Randex1;

    desc = FindLeastUsedThread(type, maxq, 0, &Randex1, -1);

    if (desc != NULL) {
	sreq->sr_NoPass = desc->d_Fd;
	queueDesc(sreq, desc);
	return(0);
    } 
    if (DebugOpt)
	printf("Unable to find any servers for request %s\n", sreq->sr_MsgId);
    return(-1);
}

void
queueDesc(ServReq *sreq, ForkDesc *desc)
{
    Connection *conn = desc->d_Data;
    ServReq **psreq = &conn->co_SReq;

    while (*psreq)
	psreq = &(*psreq)->sr_Next;
    *psreq = sreq;

    sreq->sr_SConn = conn;

    ++desc->d_Count;
    if (desc->d_Type == THREAD_SPOOL)
	++NReadServAct;
    else
	++NWriteServAct;

    sreq->sr_Rolodex = desc->d_Fd;

    /*
     * If server was idle, kick it.
     */

    if (psreq == &conn->co_SReq)
	NNServerIdle(conn);
}

/*
 * requeueServerRequest() - if request failed with previous server,
 *			    requeue for next server.
 *
 *	This function requeues a request to another server given the previous
 *	server.  It will attempt to send the request to all servers at the
 *	current priority and then will go to the next priority level, and
 *	so on, return -1 if the request could not be queued.
 *
 *	When requeueing a request, queue limits are relaxed in order to
 *	ensure that the request does not skip to higher priority queues
 *	due to high load.
 */

int
requeueServerRequest(Connection *conn, ServReq *sreq, int type, int maxq)
{
    static int Randex2;
    ForkDesc *desc;

    /*
     * Find the next server
     */

    desc = FindLeastUsedThread(
	type,
	maxq * 2,
	conn->co_Desc->d_Pri,
	&sreq->sr_Rolodex, 
	sreq->sr_NoPass
    );

    if (desc == NULL) {
	desc = FindLeastUsedThread(
	    type, 
	    maxq * 2, 
	    conn->co_Desc->d_Pri + 1, 
	    &Randex2, 
	    -1
	);
	if (desc) {
	    sreq->sr_NoPass = desc->d_Fd;
	    if (DebugOpt)
		printf("Requeue %s to nextpri %d\n", sreq->sr_MsgId, desc->d_Pri);
	} else {
	    if (DebugOpt)
		printf("Requeue %s failed\n", sreq->sr_MsgId);
	}
    } else {
	if (DebugOpt)
	    printf("Requeue %s to priority %d\n", sreq->sr_MsgId, desc->d_Pri);
    }
    if (desc) {
	queueDesc(sreq, desc);
	return(0);
    } 
    return(-1);
}

/*
 * FreeSReq()
 */

void
FreeSReq(ServReq *sreq)
{

    /*
     * If cache file write in progress, abort it.  allow the
     * fclose() to release the lock after the truncation.
     *
     * We NULL the FILE * out even though we free the structure
     * so potential memory corruption doesn't mess with random 
     * (future) files.
     */
    if (sreq->sr_Cache != NULL) {
	fflush(sreq->sr_Cache);
	AbortCache(fileno(sreq->sr_Cache), sreq->sr_MsgId, 0);
	fclose(sreq->sr_Cache);
	sreq->sr_Cache = NULL;
    }

    zfreeStr(&SysMemPool, &sreq->sr_Group);
    zfreeStr(&SysMemPool, &sreq->sr_MsgId);

    zfree(&SysMemPool, sreq, sizeof(ServReq));
}

/*
 * NNServerRequest()
 *
 *	NOTE: don't get confused by co_SReq, it serves two functions.  It
 *	placeholds a single request from a client in client Connection 
 *	structures, which this call handles, and placeholds MULTIPLE client
 *	requests in server Connection structures.
 */

void
NNServerRequest(Connection *conn, const char *grp, const char *msgid, FILE *cache, int req)
{
    ServReq *sreq = zalloc(&SysMemPool, sizeof(ServReq));

    sreq->sr_CConn = conn;
    sreq->sr_SConn = NULL;	/* for clarity: not assigned to server yet */
    sreq->sr_Time = time(NULL);
    sreq->sr_Group = grp ? zallocStr(&SysMemPool, grp) : NULL;
    sreq->sr_MsgId = msgid ? zallocStr(&SysMemPool, msgid) : NULL;
    sreq->sr_Cache = cache;	/* may be NULL 			*/

    conn->co_SReq = sreq;	/* client has active sreq		*/

    FD_CLR(conn->co_Desc->d_Fd, &RFds);

    if (req == SREQ_RETRIEVE) {
	*PSRead = sreq;
	PSRead = &sreq->sr_Next;
    } else if (req == SREQ_POST) {
	*PSWrite = sreq;
	PSWrite = &sreq->sr_Next;
    }
    QueueServerRequests();
}

/*
 * NNFinishSReq() - finish up an SReq, but requeue to a new server if requested
 *		    (i.e. article not found on old server).
 */

void
NNFinishSReq(Connection *conn, const char *ctl, int requeue)
{
    ServReq *sreq;

    if ((sreq = conn->co_SReq)) {
	/*
	 * dequeue request from server side
	 */
	conn->co_SReq = sreq->sr_Next;
	sreq->sr_Next = NULL;

	if (conn->co_Desc) {
	    --conn->co_Desc->d_Count;
	    if (conn->co_Desc->d_Type == THREAD_POST)
		--NWriteServAct;
	    else
		--NReadServAct;
	}

	conn->co_Flags &= ~COF_INPROGRESS;

	/*
	 * if requeue requested, attempt to requeue.  requeueServerRequest
	 * returns 0 on success, -1 on failure, so we have to reverse the 
	 * sense of the return value.
	 */

	if (requeue) {
	    if (conn->co_Desc->d_Type == THREAD_POST)
		requeue= requeueServerRequest(conn, sreq, THREAD_POST, THREAD_PSIZE);
	    else
		requeue= requeueServerRequest(conn, sreq, THREAD_SPOOL,THREAD_QSIZE);
	    requeue = !requeue;
	}

	/*
	 * If no requeue, closeout the request.
	 */

	if (requeue == 0) {
	    /*
	     * can be NULL if client was terminated
	     */
	    if (sreq->sr_CConn) {
		if (ctl)
		    MBWrite(&sreq->sr_CConn->co_TMBuf, ctl, strlen(ctl));
		/*
		 * set FCounter to 1 to prevent further recursion, which might
		 * feed back and screw up the state machine for this
		 * connection.
		 */
		sreq->sr_CConn->co_FCounter = 1;
		sreq->sr_CConn->co_SReq = NULL;
		NNCommand(sreq->sr_CConn);
	    }
	    FreeSReq(sreq);
	}
    }

    /*
     * XXX what if NNCommand does something which hits a server which
     * kills the server ?  boom, conn will bad illegal. XXX
     */

    NNServerIdle(conn);
}

/*
 * we have to send a garbage command to prevent INN's nnrpd from timing
 * out in 60 seconds upon initial connect.
 */

void
NNServerPrime1(Connection *conn)
{
    if (conn->co_Flags & COF_MODEREADER) {
	MBPrintf(&conn->co_TMBuf, "mode reader\r\n");
	NNServerPrime2(conn);
    } else {
	MBPrintf(&conn->co_TMBuf, "mode thisbetterfail\r\n");
	NNServerPrime3(conn);
    }
}

void
NNServerPrime2(Connection *conn)
{
    int len;
    char *ptr;

    conn->co_Func = NNServerPrime2;
    conn->co_State = "prime2";

    if ((len = MBReadLine(&conn->co_RMBuf, &ptr)) != 0) {
	int code = strtol(ptr, NULL, 10);
	if (code >= 200 && code <= 299) {
	    logit(LOG_ERR, "mode-reader(%s) %s", conn->co_Desc->d_Id, ptr);
	    NNServerStart(conn);
	} else {
	    logit(LOG_ERR, "mode-reader(%s) failed: %s", conn->co_Desc->d_Id, ptr);
	    NNServerTerminate(conn);
	}
    }
}

void
NNServerPrime3(Connection *conn)
{
    int len;
    char *buf;

    conn->co_Func = NNServerPrime3;
    conn->co_State = "prime2";

    /*
     * any return code or EOF.  0 means we haven't gotten the
     * return code yet.
     */

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) != 0) {
	NNServerStart(conn);
    }
}

/*
 * NNSERVERSTART() - start normal server operations.  Clean up from connect
 *		     code, make server available to clients.
 */

void
NNServerStart(Connection *conn)
{
    conn->co_Desc->d_Count = 0;
    conn->co_Flags &= ~(COF_INPROGRESS|COF_ININIT);

    if (conn->co_Desc->d_Type ==  THREAD_SPOOL)
	--NReadServAct;
    else
	--NWriteServAct;

    NNServerIdle(conn);
}

/*
 * NNSERVERIDLE() - server idle, wait for EOF or start next queued request.
 *		    If no more queued requests, try to requeue from unqueued
 *		    requests.
 */

void
NNServerIdle(Connection *conn)
{
    conn->co_Func  = NNServerIdle;
    conn->co_State = "sidle";

    /*
     * Ooops, not really idle.  This can happen due to the recursive
     * nature of much of the code.
     */

    if ((conn->co_Flags & (COF_INPROGRESS|COF_CLOSESERVER)) == COF_INPROGRESS)
	return;

    /*
     * Check for an unexpected condition on server, i.e. data or
     * EOF on the input where we didn't expect any.
     */

    if ((conn->co_Flags & COF_CLOSESERVER) == 0) {
	int len;
	char *buf;

	if ((len = MBReadLine(&conn->co_RMBuf, &buf)) != 0) {
	    if (len > 0)
		logit(LOG_ERR, "Server closed connection: %s", buf);
	    else
		logit(LOG_ERR, "Server closed connection");
	    conn->co_Flags |= COF_CLOSESERVER;
	}
    }
    if (conn->co_Flags & COF_CLOSESERVER) {
	conn->co_Desc->d_Count = THREAD_LIMSIZE;
	NNServerTerminate(conn);
    } else if (conn->co_SReq) {
	conn->co_Flags |= COF_INPROGRESS;

	if (conn->co_Desc->d_Type == THREAD_POST)
	    NNPostCommand1(conn);
	else
	    NNSpoolCommand1(conn);
    } else {
	QueueServerRequests();
    }
}

/*
 * NNServerTerminate() - terminate a server connection.  Usually occurs
 *			 if the server goes down or the related process
 *			 on the server is killed.   Any client request
 *			 queued to the server is requeued to another
 *			 server.
 */

void
NNServerTerminate(Connection *conn)
{
    ServReq *sreq;

    while ((sreq = conn->co_SReq) != NULL) {
	conn->co_SReq = sreq->sr_Next;
	--conn->co_Desc->d_Count;
	sreq->sr_Next = NULL;
	sreq->sr_SConn = NULL;
	sreq->sr_Time = time(NULL);
	if (conn->co_Desc->d_Type == THREAD_POST) {
	    *PSWrite = sreq;
	    PSWrite = &sreq->sr_Next;
	} else {
	    *PSRead = sreq;
	    PSRead = &sreq->sr_Next;
	}
	if (conn->co_Desc->d_Type == THREAD_POST)
	    --NWriteServAct;
	else
	    --NReadServAct;
    }

    /*
     * If we bumped the active count for the duration of the startup and
     * terminated prior to the startup completing, we have to fix it here.
     */

    if (conn->co_Flags & COF_ININIT) {
	conn->co_Flags &= ~COF_ININIT;
	if (conn->co_Desc->d_Type == THREAD_POST)
	    --NWriteServAct;
	else
	    --NReadServAct;
    }

    /*
     * closeup the server.  If d_Count is non-zero we have
     * to cleanup our reference counts, but then we set d_Count
     * to ensure nothing else gets queued to the server
     * between calling NNTerminate() and the actual termination.
     */

    conn->co_Flags |= COF_CLOSESERVER;
    if (conn->co_Desc->d_Type == THREAD_POST)
	--NWriteServers;
    else
	--NReadServers;

    ++ServersTerminated;	/* flag for rescan/reopen  */

    conn->co_Desc->d_Count = THREAD_LIMSIZE;
    conn->co_Flags |= COF_INPROGRESS;

    QueueServerRequests();	/* try to requeue requests */
    NNTerminate(conn);
}

/*
 * NNSERVERCONNECT() - initial connection, get startup message (co_UCounter
 *			starts out 1, we do not clear it and make the server
 *			available until we get the startup message)
 */

void
NNServerConnect(Connection *conn)
{
    char *buf;
    char *ptr;
    int len;
    int code;

    conn->co_Func  = NNServerConnect;
    conn->co_State = "sconn";

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) <= 0) {
	if (len < 0) {
	    logit(LOG_ERR, "connect() failed");
	    NNServerTerminate(conn);
	}
	return;
    }

    ptr = buf;
    code = strtol(ptr, NULL, 10);
    if (code == 200) {
	logit(LOG_INFO, "connect(%s) %s", conn->co_Desc->d_Id, buf);
    } else if (code == 201) {
	if (conn->co_Desc->d_Type == THREAD_POST)
	    logit(LOG_INFO, "connect(%s) %s", conn->co_Desc->d_Id, buf);
	else
	    logit(LOG_INFO, "connect(%s) %s", conn->co_Desc->d_Id, buf);
    } else {
	logit(LOG_ERR, "connect(%s) unrecognized banner: %s", conn->co_Desc->d_Id, buf);
	NNServerTerminate(conn);
	return;
    }
    NNServerPrime1(conn);
}

/*
 *  NNARTICLERETRIEVEBYMESSAGEID() - retrieve article by message-id
 *
 *	Retrieve an article by its message id and write the article
 *	to the specified connection.
 *
 *	(a) attempt to fetch the article from cache (positive or negative hit)
 *
 *	(b) initiate the state machine to attempt to fetch the article from a
 *	    remote server.
 *
 *	(c) on remote server fetch completion, cache the article locally
 *
 *	(d) place article in transmission buffer for connection, transmitting
 *	    it, then return to the command state.
 *
 *	COM_ARTICLEWVF	retrieve article from remote by message-id but retrieve
 *			headers by co_ArtNo.
 *
 *	COM_ARTICLE	retrieve entire article from remote by message-id
 *
 *	COM_...
 */

void
NNArticleRetrieveByMessageId(Connection *conn, const char *msgid)
{   
    FILE *cache = NULL;

    /*
     * (a) retrieve from cache if caching is enabled
     */

    if (DiabloReaderCacheMode) {
	int valid;
	int size;
	int cfd;

	valid = OpenCache(msgid, &cfd, &size);

	if (valid > 0) {
	    /*
	     * good cache
	     */
	    const char *map;
	    printf("good cache\n");
	    if ((map = xmap(NULL, size, PROT_READ, MAP_SHARED, cfd, 0)) != NULL) {
		xadvise(map, ((size < 65536) ? size : 65536), XADV_WILLNEED);
		if (conn->co_ArtMode != COM_BODYNOSTAT) {
		    MBPrintf(&conn->co_TMBuf, "%03d %d %s %s\r\n", 
			GoodRC(conn),
			((conn->co_ArtMode==COM_ARTICLEWVF)?conn->co_ArtNo:0),
			msgid,
			GoodResId(conn)
		    );
		}
		if (conn->co_ArtMode != COM_STAT) {
		    DumpArticleFromCache(conn, map, size);
		    MBPrintf(&conn->co_TMBuf, ".\r\n");
		}
		xunmap((void *)map, size);
	    } else {
		if (conn->co_ArtMode == COM_BODYNOSTAT)
		    MBPrintf(&conn->co_TMBuf, "(article not available)\r\n.\r\n");
		else
		    MBPrintf(&conn->co_TMBuf, "430 No such article\r\n");
	    }
	    close(cfd);
	    NNCommand(conn);
	    return;
	} else if (valid == 0) {
	    /*
	     * not in cache, cache file created if cfd >= 0
	     */
	    printf("bad cache\n");
	    if (cfd >= 0) {
		cache = fdopen(cfd, "w");
	    }
	    /* fall through */
	} else if (valid < 0) {
	    /*
	     * negatively cached
	     */
	    printf("neg cache\n");
	    if (conn->co_ArtMode == COM_BODYNOSTAT)
		MBPrintf(&conn->co_TMBuf, "(article not available)\r\n.\r\n");
	    else
		MBPrintf(&conn->co_TMBuf, "430 No such article\r\n");
	    NNCommand(conn);
	    return;
	}
    }

    /*
     * (b)(c)(d)
     */
    NNServerRequest(conn, conn->co_GroupName, msgid, cache, SREQ_RETRIEVE);
    NNWaitThread(conn);
}

