
/*
 * LIB/HISTORY.C
 *
 * (c)Copyright 1997, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 *
 * Generally speaking, the history mechanism can be pictured from the
 * point of view of a reader or from the point of view of a writer.  From
 * the point of view of a reader, A base table lookup begins a hash chain
 * of history records that runs backwards through the history file.  More
 * recent entries are encountered first, resulting in a certain degree of
 * locality of reference based on age.
 *
 * A writer must scan the chain to ensure that the message-id does not
 * already exist.  The new history record is appended to the history file
 * and inserted at the base of the hash chain.  The file append requires
 * an exclusive lock to ensure atomic operation (O_APPEND is often not atomic
 * on heavily loaded systems, the exclusive lock is required).
 *
 * In a heavily loaded system, the exclusive lock and append may become a
 * bottleneck.
 *
 * WARNING!  offsets stored in history records / hash table index are signed
 * 32 bits but cast to unsigned in any lseek() operations.  The history file
 * is thus currently limited to 4GB even with 64 bit capable filesystems.
 * 2GB on linux, 4GB under FreeBSD (which is 64 bit capable).
 */

#include "defs.h"

Prototype void HistoryOpen(const char *fileName, int fastMode);
Prototype int  HistoryClose(void);
Prototype int HistoryLookup(const char *msgid, char **pfi, int32 *rsize, int *pmart, int *pheadOnly);
Prototype int HistoryLookupByHash(hash_t hv, History *h);
Prototype int HistoryAdd(const char *msgid, History *h);

Prototype int NewHSize;

#define HBLKINCR	16
#define HBLKSIZE	256

HistHead	HHead;
uint32 		*HAry;
int		HFd = -1;
int		LoggedDHistCorrupt;
int		HFlags;
int		HSize;
int		HMask;
off_t		HScanOff;
int		NewHSize = HSIZE;
int		HBlkIncr = HBLKINCR;
int		HBlkGood;

void
HistoryOpen(const char *fileName, int hflags)
{
    int fd;
    struct stat st;

    HFlags = hflags;

    if (NewHSize == 0)
	NewHSize = DiabloHashSize;	/* which may also be 0 */

    /*
     * open the history file
     */

    if (fileName)
	fd = open(fileName, O_RDWR|O_CREAT, 0644);
    else
	fd = open(PatDbExpand(DHistoryPat), O_RDWR|O_CREAT, 0644);

    if (fd < 0) {
	syslog(LOG_ERR, "open %s failed", PatDbExpand(DHistoryPat));
	exit(1);
    }

    if (fstat(fd, &st) < 0) {
	syslog(LOG_ERR, "fstat %s failed", PatDbExpand(DHistoryPat));
	exit(1);
    }

    /*
     * initial history file creation, if necessary
     */

    bzero(&HHead, sizeof(HHead));

    if (st.st_size == 0 || 
	read(fd, &HHead, sizeof(HHead)) != sizeof(HHead) ||
	HHead.hmagic != HMAGIC
    ) {
	/*
	 * lock after finding the history file to be invalid and recheck.
	 */

	hflock(fd, 0, XLOCK_EX);

	fstat(fd, &st);
	lseek(fd, 0L, 0);

	if (st.st_size == 0 || 
	    read(fd, &HHead, sizeof(HHead)) != sizeof(HHead) ||
	    HHead.hmagic != HMAGIC
	) {
	    int n;
	    int b;
	    char *z = calloc(4096, 1);

	    /*
	     * check for old version of history file
	     */

	    if (st.st_size) {
		syslog(LOG_ERR, "Incompatible history file version or corrupted history file\n");
		exit(1);
	    }

	    /*
	     * create new history file
	     */

	    lseek(fd, 0L, 0);
	    ftruncate(fd, 0);
	    bzero(&HHead, sizeof(HHead));

	    HHead.hashSize = NewHSize;
	    HHead.version  = HVERSION;
	    HHead.henSize  = sizeof(History);
	    HHead.headSize = sizeof(HHead);

	    write(fd, &HHead, sizeof(HHead));

	    /*
	     * write out the hash table
	     */

	    n = 0;
	    b = HHead.hashSize * sizeof(int32);

	    while (n < b) {
		int r = (b - n > 4096) ? 4096 : b - n;

		write(fd, z, r);
		n += r;
	    }
	    fsync(fd);

	    /*
	     * rewrite header with magic number
	     */

	    lseek(fd, 0L, 0);
	    HHead.hmagic = HMAGIC;
	    write(fd, &HHead, sizeof(HHead));

	    free(z);
	}

	hflock(fd, 0, XLOCK_UN);
    }

    if (HHead.version < HVERSION) {
	syslog(LOG_ERR, "DHistory version %d, expected %d, use the biweekly adm script to regenerate the history file",
	    HHead.version,
	    HVERSION
	);
	exit(1);
    }
    if (HHead.version > HVERSION) {
	syslog(LOG_ERR, "DHistory version %d, expected %d, YOU ARE RUNNING AN OLD DIABLO ON A NEW HISTORY FILE!",
	    HHead.version,
	    HVERSION
	);
	exit(1);
    }

    /*
     * Map history file
     */

    HSize = HHead.hashSize;
    HMask = HSize - 1;

    /*
     * In FAST mode we leave the history file locked in order to
     * cache the hash table array at the beginning, which in turn
     * allows us to essentially append new entries to the end of
     * the file without having to seek back and forth updating
     * the hash table.
     *
     * When we aren't in FAST mode, we memory-map the hash table
     * portion of the history file.
     */

    if (HFlags & HGF_FAST)
	hflock(fd, 0, XLOCK_EX);

    if (HFlags & HGF_FAST) {
	HAry = calloc(HSize, sizeof(int32));
	if (HAry == NULL) {
	    perror("calloc");
	    exit(1);
	}
	lseek(fd, HHead.headSize, 0);
	if (read(fd, HAry, HSize * sizeof(int32)) != HSize * sizeof(int32)) {
	    perror("read");
	    exit(1);
	}
    } else {
	HAry = xmap(NULL, HSize * sizeof(int32), PROT_READ, MAP_SHARED, fd, HHead.headSize);
    }

    if (HAry == NULL || HAry == (uint32 *)-1) {
	if (fd >= 0)
	    close(fd);
	perror("mmap");
	exit(1);
    }
    HFd = fd;
    HScanOff = HHead.headSize + HHead.hashSize * sizeof(int32);
}

/*
 * On close, we have to commit the hash table if we were in
 * FAST mode, otherwise we need only unmap the file before
 * closing it.
 */

int
HistoryClose(void)
{
    int r = RCOK;

    if (HFd >= 0) {
	if (HFlags & HGF_FAST) {
	    lseek(HFd, HHead.headSize, 0);
	    if (write(HFd, HAry, HSize * sizeof(int32)) != HSize * sizeof(int32))
		r = RCTRYAGAIN;
	    else
		free(HAry);
	} else {
	    if (HAry && HAry != (uint32 *)-1)
		xunmap((void *)HAry, HSize * sizeof(int32));
	}
    }
    if (r == RCOK) {
	if (HFd >= 0)
	    close(HFd);
	HFd = -1;
	HAry = NULL;
    }
    return(r);
}

int
HistoryLookup(const char *msgid, char **pfi, int32 *rsize, int *pmart, int *pheadOnly)
{
    hash_t hv = hhash(msgid);
    int32 poff = HHead.headSize + ((hv.h1 ^ hv.h2) & HMask) * sizeof(int32);
    int32 off = HAry[(hv.h1 ^ hv.h2) & HMask];
    History h = { 0 };
    static int HLAlt;
    int r = -1;

    while (off) {
	lseek(HFd, (off_t)(uint32)off, 0);
	if (read(HFd, &h, sizeof(h)) != sizeof(h)) {
	    if ((LoggedDHistCorrupt & 1) == 0 || DebugOpt) {
		LoggedDHistCorrupt |= 1;
		syslog(LOG_ERR, "dhistory file corrupted on lookup @ %d->%d chain %d", (int)poff, (int)off, (int)((hv.h1 ^ hv.h2) & HMask));
		sleep(1);
	    }
	    break;
	}
	if (h.hv.h1 == hv.h1 && h.hv.h2 == hv.h2)
	    break;
	poff = off;
	off = h.next;
    }
    if (off != 0) {
	if (pheadOnly)
	    *pheadOnly = (int)(h.exp & EXPF_HEADONLY);

	if (pfi) {
	    char path[128];
	    int fd;

	    ArticleFileName(path, sizeof(path), &h, 1);

	    /*
	     * multi-article file ?  If so, articles are zero-terminated
	     */

	    *pmart = 0;
	    if (h.boffset || h.bsize)
		*pmart = 1;

	    /*
	     * get the file
	     */

	    *rsize = 0;
	    if ((fd = cdopen(path, O_RDONLY, 0)) >= 0) {
		struct stat st;

		*pfi = NULL;
		if (fstat(fd, &st) == 0 && h.bsize + h.boffset < st.st_size) {
		    *rsize = h.bsize;
		    *pfi = xmap(NULL, h.bsize + *pmart, PROT_READ, MAP_SHARED, fd, h.boffset);
		    if (*pfi == (char *)-1)
			*pfi = NULL;
		}
		/*
		 * Sanity check.  Look for 0x00 guard character, make sure
		 * first char isn't 0x00, and make sure last char isn't 0x00
		 * (the last character actually must be an LF or the NNTP
		 * protocol wouldn't have worked).
		 */
		if (*pfi == NULL ||
		    h.bsize == 0 ||
		    (*pfi)[0] == 0 ||
		    (*pfi)[h.bsize-1] == 0 ||
		    (*pfi)[h.bsize] != 0
		) {
		    syslog(LOG_ERR, "corrupt spool entry for %s@%d,%d %s",
			path,
			(int)h.boffset,
			(int)h.bsize,
			msgid
		    );
		    if (*pfi) {
			xunmap(*pfi, h.bsize + *pmart);
			*pfi = NULL;
		    }
		}
	    }
	    close(fd);
	}
	r = 0;
    }

    /*
     * On failure, try alternate hash method (for lookup only)
     */

    if (r < 0 && DiabloCompatHashMethod >= 0 && HLAlt == 0) {
	int save = DiabloHashMethod;

	DiabloHashMethod = DiabloCompatHashMethod;
	HLAlt = 1;
	r = HistoryLookup(msgid, pfi, rsize, pmart, pheadOnly);
	DiabloHashMethod = save;
	HLAlt = 0;
    }
    return(r);
}

int
HistoryLookupByHash(hash_t hv, History *h)
{
    int32 poff = HHead.headSize + ((hv.h1 ^ hv.h2) & HMask) * sizeof(int32);
    int32 off = HAry[(hv.h1 ^ hv.h2) & HMask];

    while (off) {
	lseek(HFd, (off_t)(uint32)off, 0);
	if (read(HFd, h, sizeof(*h)) != sizeof(*h)) {
	    if ((LoggedDHistCorrupt & 1) == 0 || DebugOpt) {
		LoggedDHistCorrupt |= 1;
		syslog(LOG_ERR, "dhistory file corrupted on lookup");
		sleep(1);
	    }
	    break;
	}
	if (h->hv.h1 == hv.h1 && h->hv.h2 == hv.h2)
	    break;
	poff = off;
	off = h->next;
    }
    if (off != 0) {
	return(0);
    }
    return(-1);
}

int
HistoryAdd(const char *msgid, History *h)
{
    int32 hi = (h->hv.h1 ^ h->hv.h2) & HMask;
    int32 poff = HHead.headSize + hi * sizeof(int32);
    int32 boff = poff;
    int r = RCOK;

    /*
     * record lock, search for message id
     *
     */

    if ((HFlags & HGF_FAST) == 0)	/* lock hash chain */
	hflock(HFd, boff, XLOCK_EX);

    /*
     * make sure message is not already in hash table
     */

    {
	int32 off;

	off = HAry[hi];

	if ((HFlags & HGF_NOSEARCH) == 0) {
	    while (off) {
		History ht;

		lseek(HFd, (off_t)(uint32)off, 0);
		if (read(HFd, &ht, sizeof(ht)) != sizeof(ht)) {
		    if ((LoggedDHistCorrupt & 2) == 0 || DebugOpt) {
			LoggedDHistCorrupt |= 2;
			syslog(LOG_ERR, "dhistory file corrupted @ %d", off);
			sleep(1);
		    }
		    break;
		}
		if (ht.hv.h1 == h->hv.h1 && ht.hv.h2 == h->hv.h2) {
		    r = RCALREADY;
		    break;
		}
		poff = off;
		off = ht.next;
	    }
	}
    }

    /*
     * If we are ok, reserve space in the history file for the record
     * and insert the record.  This may loop if intermediate scans determine
     * that we must append a new block to the file.
     */

    while (r == RCOK) {
	off_t offBeg;
	off_t offEnd;
	History hary[HBLKSIZE];

	/*
	 * append/scan lock
	 */

	if ((HFlags & HGF_FAST) == 0)	/* append/scan lock */
	    hflock(HFd, 4, XLOCK_EX);

	/*
	 * Figure out scan start & end point.  We only look at the last
	 * HBLKSIZE*2 records worth of entries in the worst case.
	 */

	offEnd = lseek(HFd, 0L, 2);
	{
	    off_t n;

	    n = offEnd - HHead.headSize - HHead.hashSize * sizeof(int32);
	    n = n % HHead.henSize;
	    offEnd -= n;
	}

	offBeg = HScanOff;

	if (offBeg > offEnd)
	    offBeg = offEnd;

	if (offBeg < offEnd - HBLKSIZE * sizeof(History)) {
	    off_t n;

	    offBeg = offEnd - HBLKSIZE * sizeof(History);
	    n = offBeg - HHead.headSize - HHead.hashSize * sizeof(int32);
	    n = n % HHead.henSize;
	    offBeg -= n;
	}

	/*
	 * Look for a blank record (gmt value is 0)
	 */

	lseek(HFd, offBeg, 0);

	while (offBeg < offEnd) {
	    int q = (offEnd - offBeg) / sizeof(History);
	    int i;
	    int n;

	    if (q > arysize(hary))
		q = arysize(hary);
	    if (q > HBlkIncr)
		q = HBlkIncr;

	    n = read(HFd, hary, q * sizeof(History));

	    if (n != q * sizeof(History)) {
		r = RCTRYAGAIN;
		break;
	    }

	    for (i = 0; r == RCOK && i < q; ++i) {
		if (hary[i].gmt == 0 && 
		    ((HFlags & HGF_FAST) ||
		    hflock(HFd, offBeg, XLOCK_NB|XLOCK_EX) >= 0)
		) {
		    break;
		}
		offBeg += sizeof(History);
	    }

	    /*
	     * Found, break, or not found, continue.  Adjust HBlkIncr
	     * appropriately.
	     */
	    if (i != q) {
		if (++HBlkGood == 4) {
		    HBlkGood = 0;
		    if ((HBlkIncr -= HBLKINCR) < HBLKINCR)
			HBlkIncr = HBLKINCR;
		}
		break;
	    }
	    if (offBeg != offEnd) {
		HBlkGood = 0;
		if ((HBlkIncr += HBLKINCR) > HBLKSIZE) {
		    HBlkIncr = HBLKSIZE;
		}
	    }
	}

	HScanOff = offBeg;

	/*
	 * break if something went wrong or if we found the record.  With
	 * the proper record explicitly locked, we can release the append
	 * lock.
	 */

	if (r != RCOK || offBeg != offEnd) {
	    if ((HFlags & HGF_FAST) == 0)	/* append/scan unlock */
		hflock(HFd, 4, XLOCK_UN);
	    break;
	}

	/*
	 * Otherwise append a new block and retry.  XXX put on filesystem
	 * block boundry.
	 */

	lseek(HFd, 0L, 2);
	bzero(hary, sizeof(hary));
	write(HFd, hary, sizeof(hary));
	if ((HFlags & HGF_FAST) == 0)
	    hflock(HFd, 4, XLOCK_UN);
    } /* WHILE */

    /*
     * If we did find a blank record, HScanOff is set to it.  The record
     * is locked at this point.
     *
     * We insert the record into the chain at the beginning of the
     * chain, even though we are appending the record.  This is for
     * time-locality of reference and also reduces 'append-to-history'
     * file overhead by localizing disk writes a bit better.
     *
     * XXX next/linkages are 32 bits.  At the moment we can go to 4G even
     * though we are using a signed 32 bit int because we cast seeks to
     * unsigned 32 bits.
     */

    if (r == RCOK) {
	int n;

	h->next = HAry[hi];

	lseek(HFd, HScanOff, 0);

	n = write(HFd, h, sizeof(History));

	if ((HFlags & HGF_FAST) == 0)
	    hflock(HFd, HScanOff, XLOCK_UN);

	if (n == sizeof(History)) {
	    if ((HFlags & HGF_FAST) == 0) {
		int32 off = (int32)HScanOff;	/* for write XXX */

		lseek(HFd, HHead.headSize + hi * sizeof(int32), 0);
		if (write(HFd, &off, sizeof(int32)) != sizeof(int32)) {
		    r = RCTRYAGAIN;
		}
	    } else {
		HAry[hi] = (int)HScanOff;
	    }
	} else {
	    r = RCTRYAGAIN;
	}
    }

    if ((HFlags & HGF_FAST) == 0)	/* unlock hash chain */
	hflock(HFd, boff, XLOCK_UN);

    return(r);
}

