/* ;-*-C-*-; 
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** ctbase.i
**
** NOTE: This is not a stand-alone source file, but is included in ctype.c.
**       (This is necessary becuase there is no other way in C to have a
**       hidden scope, besides at the file level.)
*/

/*@access cprim*/

typedef /*@null@*/ struct __ctbase *ctbase;

typedef struct _ctentry {
  ctkind kind;
  ctbase ctbase;
  ctype base;     /* type I point to (or element of array) */
  ctype ptr;     /* type of pointer to me */
  ctype array;    /* type of array of me */
  cstring unparse; /* unparse me, if memoized */
} *ctentry ;

typedef /*@only@*/ ctentry o_ctentry;
          
typedef struct _cttable {
  int size;
  int nspace;
  /*@relnull@*/ /*@only@*/ o_ctentry *entries;
  /* memoize matches...maybe in context? */
} cttable ;

extern bool ctentry_isBogus (/*@sef@*/ ctentry c) /*@*/;
# define ctentry_isBogus(c) \
   ((c)->kind == CTK_INVALID || (c)->kind == CTK_DNE)

static cttable cttab = { 0, 0, NULL };

static /*@notnull@*/ /*@only@*/ ctbase ctbase_createAbstract (typeId u);
static /*@observer@*/ cstring ctentry_doUnparse (ctentry c) /*@modifies c@*/;
static /*@only@*/ ctentry
  ctentry_make (ctkind ctk, /*@keep@*/ ctbase c, ctype base, 
		ctype ptr, ctype array, /*@keep@*/ cstring unparse);
static /*@only@*/ ctentry ctentry_makeNew (ctkind ctk, /*@only@*/ ctbase c);
static /*@only@*/ cstring ctentry_unparse (ctentry c) /*@*/ ;

static void cttable_grow (void);
static ctype cttable_addDerived (ctkind ctk, /*@keep@*/ ctbase new, ctype base);
static ctype cttable_addFull (/*@keep@*/ ctentry new);
static bool ctentry_isInteresting (ctentry c) /*@*/;

/* 
** These are file-static macros (used in ctype.c).  No way to
** declare them as static in C.
*/

/*@-allmacros@*/ /*@-macrospec@*/ /*@-namechecks@*/
# define ctentry_getBase(c)       ((c)->base)
# define ctentry_getKind(c)       ((c)->kind)
# define ctentry_getArray(c)      ((c)->array)
# define ctentry_getPtr(c)        ((c)->ptr)
# define ctentry_isArray(c)       ((c)->kind == CTK_ARRAY)
# define ctentry_isComplex(c)     ((c)->kind == CTK_COMPLEX)
# define ctentry_isPlain(c)       ((c)->kind == CTK_PLAIN)
# define ctentry_isPointer(c)     ((c)->kind == CTK_PTR)
# define ctentry_setArray(c,b)    ((c)->array = (b))
# define ctentry_setPtr(c,b)      ((c)->ptr = (b))

# define ctbase_fixUser(c)        (c = ctbase_realType(c))
/*@=allmacros@*/ /*@=macrospec@*/ /*@=namechecks@*/

static ctype cttable_addComplex (/*@notnull@*/ /*@only@*/ ctbase new);
static /*@observer@*/ ctbase ctype_getCtbase (ctype c) /*@*/ ;
static ctype ctype_makeConjAux (ctype c1, ctype c2, bool isExplicit) /*@*/ ;
static /*@notnull@*/ /*@observer@*/ ctbase ctype_getCtbaseSafe (ctype c) /*@*/ ;
static /*@observer@*/ ctentry ctype_getCtentry (ctype c)  /*@*/ ;
static /*@observer@*/ /*@notnull@*/ ctbase 
  ctbase_realType (/*@notnull@*/ ctbase c) /*@*/ ;
static bool ctbase_isPointer (/*@notnull@*/ /*@dependent@*/ ctbase c) /*@*/ ;
static bool ctbase_isArray (/*@notnull@*/ /*@dependent@*/ ctbase c) /*@*/ ;
static /*@observer@*/ enumNameList ctbase_elist (ctbase c) /*@*/ ;
static /*@only@*/ cstring ctbase_unparse (ctbase v) /*@*/ ;
static /*@only@*/ cstring ctbase_unparseDeep (ctbase v) /*@*/ ;
static /*@only@*/ /*@notnull@*/ ctbase ctbase_copy (/*@notnull@*/ ctbase c) /*@*/ ;
static void ctbase_free (/*@only@*/ ctbase c);
static /*@notnull@*/ /*@only@*/ ctbase ctbase_createPrim (cprim p) /*@*/ ;
static /*@notnull@*/ /*@only@*/ ctbase ctbase_createBool (void) /*@*/ ;
static /*@notnull@*/ /*@observer@*/ ctbase ctbase_getBool (void) /*@*/ ;
static /*@notnull@*/ /*@only@*/ ctbase ctbase_createUser (typeId u) /*@*/ ;

static /*@notnull@*/ /*@only@*/ ctbase
  ctbase_createStruct (/*@only@*/ cstring n, /*@only@*/ uentryList f);

static /*@notnull@*/ /*@only@*/ ctbase 
  ctbase_createUnion  (/*@keep@*/ cstring n, /*@only@*/ uentryList f);
static /*@notnull@*/ /*@only@*/ ctbase ctbase_createEnum (/*@keep@*/ cstring etag, /*@keep@*/ enumNameList emembers);
static /*@notnull@*/ /*@only@*/ ctbase ctbase_createUnknown (void);
static bool ctbase_match (ctbase c1, ctbase c2) /*@modifies nothing@*/;
static bool ctbase_matchDef (ctbase c1, ctbase c2) /*@modifies nothing@*/;
static bool ctbase_genMatch (ctbase c1, ctbase c2, bool force, bool arg, bool def);
static bool ctbase_isAbstract (/*@notnull@*/ ctbase c) /*@*/ ;
static /*@notnull@*/ /*@only@*/ ctbase ctbase_makePointer (ctype c) /*@*/ ;
static /*@notnull@*/ /*@only@*/ ctbase ctbase_makeArray (ctype c) /*@*/ ;
static /*@notnull@*/ ctype 
  ctbase_makeFunction (ctype c, /*@only@*/ uentryList p) /*@*/ ;
static /*@notnull@*/ /*@only@*/ ctbase 
  ctbase_makeRealFunction (ctype c, /*@dependent@*/ uentryList p) /*@*/ ;
static /*@notnull@*/ /*@observer@*/ ctbase 
  ctbase_realFunction (/*@notnull@*/ /*@dependent@*/ ctbase c) /*@*/ ;
static ctype ctbase_baseArrayPtr (/*@notnull@*/ ctbase c) /*@*/ ;
static ctype ctbase_baseFunction (/*@notnull@*/ ctbase c) /*@*/ ;
static /*@observer@*/ uentryList ctbase_argsFunction (/*@notnull@*/ ctbase c) /*@*/ ;
static /*@observer@*/ uentryList ctbase_getuentryList (/*@notnull@*/ ctbase c) /*@*/ ;
static ctype ctbase_newBase (ctype c, ctype p) /*@*/ ;
static ctype ctbase_newBaseExpFcn (ctype c, ctype p) /*@*/ ;

/*@-macroundef@*/
/*@constant int CT_LAST;@*/          
# define CT_LAST  (cttab.size - 1)
/*@=macroundef@*/

typedef struct _cfcn
{
  ctype rval;
  /*@dependent@*/ uentryList params;  /* params are owned if liveparams is TRUE */
  bool liveparams;
} cfcn ; 

typedef struct _tsu
{
  cstring   name;
  uentryList fields;
} tsu;

typedef struct _tconj
{
  ctype a;
  ctype b;
  bool  isExplicit;
} tconj ;

typedef struct _tenum
{
  cstring      tag;
  enumNameList members;
} tenum;

/*
** convert to pointers for space efficiency
** nah...later
*/
 
typedef union _uconts
{
  cprim         prim;      /* primitive */
  typeId       tid;       /* abstract, user */
  ctype         base;      /* ptr, array */
  cfcn          fcn;       /* function     */
  tsu           su;        /* struct union */
  tenum         cenum;     /* enum */
  tconj         conj;      /* conj */
} uconts;

struct __ctbase
{
  ctuid    type;
  uconts   contents;
} ;

static /*@falsenull@*/ bool ctbase_isUA (ctbase c) /*@*/ ;
static bool ctbase_isBaseUA(ctbase c) /*@*/ ;
static typeId  ctbase_typeBaseUid(ctbase c) /*@*/ ;
static bool ctbase_isKind (/*@notnull@*/ ctbase c, ctuid kind) /*@*/ ;
static bool ctbase_isKind2 (/*@notnull@*/ ctbase c, ctuid kind1, ctuid kind2) /*@*/ ;
static /*@only@*/ /*@notnull@*/ ctbase 
  ctbase_getBaseType (/*@notnull@*/ ctbase c) /*@*/ ;

static /*@falsenull@*/ bool ctbase_isFunction(ctbase c) /*@*/ ;

/*@constant null ctbase ctbase_undefined; @*/
# define ctbase_undefined        ((ctbase)0)

static /*@owned@*/ ctbase ctbase_bool = ctbase_undefined;
static /*@owned@*/ ctbase ctbase_unknown = ctbase_undefined;

static /*@falsenull@*/ bool ctbase_isDefined (ctbase c) /*@*/
{
  return ((c) != ctbase_undefined);
}

static /*@truenull@*/ bool ctbase_isUndefined (ctbase c)
{
  return ((c) == ctbase_undefined);
}

static ctkind ctype_getCtKind (ctype c)
{
  ctentry ce = ctype_getCtentry (c);

  return ctentry_getKind (ce);
}

static bool ctbase_isUser (ctbase c)
{
  if (ctbase_isDefined (c))
    {
      return (ctbase_isKind (c, CT_USER));
    }
  else
    {
      return FALSE;
    }
}

static bool ctbase_isEnum (ctbase c)
{
  if (ctbase_isDefined (c))
    {
      return (ctbase_isKind (c, CT_ENUM));
    }
  else
    {
      return FALSE;
    }
}

static bool ctbase_isExpFcn (ctbase c)
{
  if (ctbase_isDefined (c))
    {
      return (c->type == CT_EXPFCN);
    }
  else
    {
      return FALSE;
    }
}

static /*@falsenull@*/ bool ctbase_isConj (ctbase c)
{
  if (ctbase_isDefined (c))
    {
      return (c->type == CT_CONJ);
    }
  else
    {
      return FALSE;
    }
}

static bool ctuid_isAP (ctuid c)
{
  return (c == CT_ARRAY || c == CT_PTR);
}

static typeId ctbase_typeId (ctbase c);
static /*@only@*/ cstring ctbase_dump (ctbase c);
static /*@only@*/ ctbase ctbase_undump (char **c);
static int ctbase_compare (ctbase c1, ctbase c2, bool strict);
static void ctbase_convertTypeTable (ctbase c, indxMap map);

static bool ctbase_matchArg (ctbase c1, ctbase c2);
static /*@notnull@*/ /*@only@*/ ctbase 
  ctbase_makeConj (ctype c1, ctype c2, bool isExplicit) /*@*/ ;
static ctype ctbase_getConjA (/*@notnull@*/ ctbase c) /*@*/ ;
static ctype ctbase_getConjB (/*@notnull@*/ ctbase c) /*@*/ ;
static bool ctbase_forceMatch (ctbase c1, ctbase c2);
static /*@notnull@*/ /*@only@*/ ctbase ctbase_expectFunction (ctype c);
static bool ctbase_isVoidPointer(/*@notnull@*/ /*@dependent@*/ ctbase c) /*@*/ ;
static bool ctbase_isUnion (/*@notnull@*/ /*@dependent@*/ ctbase c) /*@*/ ;
static bool ctbase_isStruct (/*@notnull@*/ /*@dependent@*/ ctbase c) /*@*/ ;
static /*@observer@*/ cstring ctbase_enumTag (/*@notnull@*/ ctbase ct) /*@*/ ;
static /*@only@*/ cstring ctbase_unparseNotypes (ctbase c) /*@*/ ;

static /*@out@*/ /*@notnull@*/ /*@only@*/ ctbase ctbase_new (void) /*@*/ ;
static int nctbases = 0;

static /*@notnull@*/ /*@only@*/ 
  ctbase ctbase_makeLiveFunction (ctype b, /*@owned@*/ uentryList p);

static /*@observer@*/ ctbase ctbase_realType (ctbase c)
{
  if (ctbase_isUA (c))
    {
      typeId uid = ctbase_typeId (c);

      if (usymtab_isBoolType (uid))
	{
	  DPRINTF (("bool!"));
	  return ctbase_getBool ();
	}
      else
	{
	  ctbase ret = ctype_getCtbase 
	    (uentry_getRealType (usymtab_getTypeEntry (ctbase_typeId (c))));
	  
	  llassert (ret != ctbase_undefined);
	  return ret;
	}
    }
  else
    {
      return c;
    }
}

static bool
ctbase_isVoidPointer (/*@dependent@*/ /*@notnull@*/ ctbase c)
{
  ctbase r = ctbase_realType (c);

  return (ctbase_isKind (r, CT_PTR) &&
	  ctype_isVoid (r->contents.base));
}

static bool
ctbase_isPointer (/*@notnull@*/ /*@dependent@*/ ctbase c)
{
  ctbase r = ctbase_realType (c);

  return (ctbase_isKind (r, CT_PTR));
}

static bool
ctbase_isArray (/*@notnull@*/ /*@dependent@*/ ctbase c)
{
  ctbase r = ctbase_realType (c);

  return (ctbase_isKind (r, CT_ARRAY));
}

static bool
ctbase_isStruct (/*@notnull@*/ /*@dependent@*/ ctbase c)
{
  ctbase r = ctbase_realType (c);

  return (ctbase_isKind (r, CT_STRUCT));
}

static bool
ctbase_isUnion (/*@notnull@*/ /*@dependent@*/ ctbase c)
{
  ctbase r = ctbase_realType (c);

  return (ctbase_isKind (r, CT_UNION));
}

/*
** clean this up -> typeTable should store ctype
*/

static typeId
ctbase_typeBaseUid (ctbase c)
{
  ctuid ct;

  if (ctbase_isDefined (c))
    {
      ct = c->type;

      if (ctuid_isAP (ct))
	return ctbase_typeBaseUid (ctype_getCtbase (c->contents.base));
      else if (ct == CT_USER || ct == CT_ABST)
	return c->contents.tid;
      else
	{
	  llcontbuglit ("ctbase_typeBaseUid: bad call");
	  return typeId_invalid;
	}
    }
  return typeId_invalid;
}

static bool
ctbase_isBaseUA (ctbase c)
{
  ctuid ct;

  if (ctbase_isDefined (c))
    {
      ct = c->type;

      if (ctuid_isAP (ct))
	return ctbase_isBaseUA (ctype_getCtbase (c->contents.base));
      else
	return (ct == CT_USER || ct == CT_ABST);
    }
  return FALSE;
}

static typeId
ctbase_typeId (ctbase c)
{
  if (ctbase_isUA (c))
    {
      return c->contents.tid;
    }
  else
    {
      llcontbug (message ("ctbase_typeId: bad call: %q", ctbase_unparse (c)));
      return typeId_invalid;
    }
}

/*
** all typeId's mentioned in c, are maped using indxMap.
**
*/

static void
ctbase_convertTypeTable (ctbase c, indxMap map)
{
  llassert (ctbase_isDefined (c));

  switch (c->type)
    {
    case CT_UNKNOWN:
    case CT_BOOL:
    case CT_PRIM:
    case CT_ENUMLIST:
      break;

    case CT_ENUM:
    case CT_USER:
    case CT_ABST:
      c->contents.tid = typeId_fromInt (indxMap_map (map, usymId_toInt (c->contents.tid)));
      break;
      
    case CT_EXPFCN:
    case CT_PTR:
    case CT_ARRAY:
      ctype_convertTypeTable (c->contents.base, map);
      break;

    case CT_FCN:
      ctype_convertTypeTable (c->contents.fcn.rval, map);
      break;

    case CT_STRUCT:
    case CT_UNION:
      uentryList_convertTypeTable (c->contents.su.fields, map);
      break;

    case CT_CONJ:
      ctype_convertTypeTable (c->contents.conj.a, map);
      ctype_convertTypeTable (c->contents.conj.b, map);
      break;

    BADDEFAULT;
    }
}

static /*@only@*/ cstring
ctbase_unparse (ctbase c)
{
  if (ctbase_isUndefined (c))
    return cstring_makeLiteral ("<<undef>>");
  
  switch (c->type)
    {
    case CT_UNKNOWN:
      return cstring_makeLiteral ("?");
    case CT_BOOL:
      return cstring_copy (context_getBoolName ());
    case CT_PRIM:
      return (cprim_unparse (c->contents.prim));
    case CT_USER:
      return (usymtab_getTypeEntryName (c->contents.tid));
    case CT_ABST:
      return (usymtab_getTypeEntryName (c->contents.tid));
    case CT_EXPFCN:
      return (message ("<expf: %t>", c->contents.base));
    case CT_PTR:
      /* no spaces for multiple pointers */

      if (ctype_isPointer (c->contents.base))
	{
	  return (cstring_appendChar (cstring_copy (ctype_unparse (c->contents.base)), '*'));
	}
      else
	{
	  return (message ("%t *", c->contents.base));
	}

    case CT_ARRAY:
      return (message ("%t []", c->contents.base));
    case CT_FCN:
      return (message ("[function (%q) returns %t]",
		       uentryList_unparseParams (c->contents.fcn.params),
		       c->contents.fcn.rval));
    case CT_STRUCT:
      if (cstring_isDefined (c->contents.su.name) &&
	  !cstring_isEmpty (c->contents.su.name) &&
	  !isFakeTag (c->contents.su.name))
	{
	  return (message ("struct %s", c->contents.su.name));
	}
      else
	{
	  return (message ("struct { %q }", uentryList_unparseAbbrev (c->contents.su.fields)));
	}
    case CT_UNION:
      if (cstring_isDefined (c->contents.su.name) &&
	  !cstring_isEmpty (c->contents.su.name) &&
	  !isFakeTag (c->contents.su.name))
	{
	  return (message ("union %s", c->contents.su.name));
	}
      else
	{
	  return (message ("union { %q }", 
			   uentryList_unparseAbbrev (c->contents.su.fields)));
	}
    case CT_ENUM:
      if (isFakeTag (c->contents.cenum.tag))
	{
	  return (message ("enum { %q }", 
			   enumNameList_unparseBrief (c->contents.cenum.members)));
	}
      else
	{
	  return (message ("enum %s { %q }", 
			   c->contents.cenum.tag,
			   enumNameList_unparseBrief (c->contents.cenum.members)));
	}
    case CT_CONJ:
      if (c->contents.conj.isExplicit || context_getFlag (FLG_SHOWALLCONJS))
	{
	  if (!ctype_isSimple (c->contents.conj.a) ||
	      !ctype_isSimple (c->contents.conj.b))
	    {
	      return (message ("<%t> | <%t>", c->contents.conj.a, c->contents.conj.b));
	    }
	  else
	    {
	      return (message ("%t | %t", c->contents.conj.a, c->contents.conj.b));
	    }
	}
      else
	{
	  return (cstring_copy (ctype_unparse (c->contents.conj.a)));
	}
    BADDEFAULT;
    }
  BADEXIT;
}

static /*@only@*/ cstring
  ctbase_unparseDeep (ctbase c)
{
  if (ctbase_isUndefined (c))
    {
      return cstring_makeLiteral ("<<undef>>");
    }

  switch (c->type)
    {
    case CT_UNKNOWN:
      return cstring_makeLiteral ("?");
    case CT_BOOL:
      return cstring_copy (context_getBoolName ());
    case CT_PRIM:
      return (cprim_unparse (c->contents.prim));
    case CT_ENUM:
      if (cstring_isNonEmpty (c->contents.cenum.tag))
	{
	  return (message ("enum %s { %q }",
			   c->contents.cenum.tag,
			   enumNameList_unparse (c->contents.cenum.members)));
	}
      else
	{
	  return (message ("enum { %q }",
			   enumNameList_unparse (c->contents.cenum.members)));
	}
    case CT_USER:
      return (usymtab_getTypeEntryName (c->contents.tid));
    case CT_ABST:
      return (usymtab_getTypeEntryName (c->contents.tid));
    case CT_EXPFCN:
      return (message ("<expf: %t>", c->contents.base));
    case CT_PTR:
      return (message ("%t *", c->contents.base));
    case CT_ARRAY:
      return (message ("%t []", c->contents.base));
    case CT_FCN:
      return (message ("[function (%q) returns %t]",
		       uentryList_unparse (c->contents.fcn.params),
		       c->contents.fcn.rval));
      /*
      ** return (message ("%t (%q)", c->contents.fcn.rval,
      **          uentryList_unparse (c->contents.fcn.params)));
      */
    case CT_STRUCT:
      return (message ("struct %s { ... } ", c->contents.su.name));
    case CT_UNION:
      return (message ("union %s { ... }", c->contents.su.name));
    case CT_CONJ:
      return (message ("%t", c->contents.conj.a));
    BADDEFAULT;
    }
  BADEXIT;
}

static /*@only@*/ cstring
ctbase_unparseNotypes (ctbase c)
{
  llassert (ctbase_isDefined (c));

  switch (c->type)
    {
    case CT_UNKNOWN:
      return cstring_makeLiteral ("?");
    case CT_BOOL:
      return cstring_copy (context_getBoolName ());
    case CT_PRIM:
      return (cprim_unparse (c->contents.prim));
    case CT_ENUM:
      if (typeId_isInvalid (c->contents.tid))
	{
	  return cstring_makeLiteral ("enum");
	}
      else
	{
	  return (message ("T#%d", c->contents.tid));
	}
    case CT_USER:
      return (message ("uT#%d", c->contents.tid));
    case CT_ABST:
      return (message ("aT#%d", c->contents.tid));
    case CT_EXPFCN:
      return (message ("<expf: %q >", ctbase_unparseNotypes (ctype_getCtbase (c->contents.base))));
    case CT_PTR:
      return (message ("%q  *", ctbase_unparseNotypes (ctype_getCtbase (c->contents.base))));
    case CT_ARRAY:
      return (message ("%q []", ctbase_unparseNotypes (ctype_getCtbase (c->contents.base))));
    case CT_FCN:
      return (message ("[function (%d) returns %q]", uentryList_size (c->contents.fcn.params),
		       ctbase_unparseNotypes (ctype_getCtbase (c->contents.fcn.rval))));
    case CT_STRUCT:
      return (message ("struct %s", c->contents.su.name));
    case CT_UNION:
      return (message ("union %s", c->contents.su.name));
    case CT_ENUMLIST:
      return (message ("[enumlist]"));
    case CT_CONJ:
      return (message ("%q/%q", 
		       ctbase_unparseNotypes (ctype_getCtbase (c->contents.conj.a)),
		       ctbase_unparseNotypes (ctype_getCtbase (c->contents.conj.b))));
    BADDEFAULT;
    }
  BADEXIT;
}

static /*@only@*/ cstring
ctbase_unparseDeclaration (ctbase c, /*@only@*/ cstring name) /*@*/ 
{
  if (ctbase_isUndefined (c))
    {
      return name;
    }
  
  switch (c->type)
    {
    case CT_UNKNOWN:
      return (message ("? %q", name));
    case CT_BOOL:
      return (message ("%s %q", context_getBoolName (), name));
    case CT_PRIM:
      return (message ("%q %q", cprim_unparse (c->contents.prim), name));
    case CT_USER:
    case CT_ABST:
      return (message ("%q %q", usymtab_getTypeEntryName (c->contents.tid), name));
    case CT_EXPFCN:
      llcontbuglit ("ctbase_unparseDeclaration: expfcn");
      return name;
    case CT_PTR:
      if (ctype_isFunction (c->contents.base))
	{
	  return ctbase_unparseDeclaration (ctype_getCtbase (c->contents.base), name);
	}
      else
	{
	  cstring s = cstring_prependChar ('*', name);
	  cstring ret = ctbase_unparseDeclaration (ctype_getCtbase (c->contents.base), s);
	  cstring_free (name);
	  return (ret);
	}
    case CT_ARRAY:
      return (message ("%q[]", 
		       ctbase_unparseDeclaration (ctype_getCtbase (c->contents.base), name)));
    case CT_FCN:
	{
	  cstring s = message ("%q(%q)", name, uentryList_unparse (c->contents.fcn.params));

	  return (ctbase_unparseDeclaration (ctype_getCtbase (c->contents.fcn.rval), s));
	}
    case CT_STRUCT:
      if (cstring_isDefined (c->contents.su.name) &&
	  !cstring_isEmpty (c->contents.su.name) &&
	  !isFakeTag (c->contents.su.name))
	{
	  return (message ("struct %s %q", c->contents.su.name, name));
	}
      else
	{
	  return (message ("struct { %q } %q", 
			   uentryList_unparseAbbrev (c->contents.su.fields),
			   name));
	}
    case CT_UNION:
      if (cstring_isDefined (c->contents.su.name) &&
	  !cstring_isEmpty (c->contents.su.name) &&
	  !isFakeTag (c->contents.su.name))
	{
	  return (message ("union %s %q", c->contents.su.name, name));
	}
      else
	{
	  return (message ("union { %q } %q", 
			   uentryList_unparseAbbrev (c->contents.su.fields),
			   name));
	}
    case CT_ENUM:
      if (isFakeTag (c->contents.cenum.tag))
	{
	  return (message ("enum { %q } %q", 
			   enumNameList_unparseBrief (c->contents.cenum.members),
			   name));
	}
      else
	{
	  return (message ("enum %s { %q } %q", 
			   c->contents.cenum.tag,
			   enumNameList_unparseBrief (c->contents.cenum.members),
			   name));
	}
    case CT_CONJ:      
      if (c->contents.conj.isExplicit || context_getFlag (FLG_SHOWALLCONJS))
	{
	  if (!ctype_isSimple (c->contents.conj.a) ||
	      !ctype_isSimple (c->contents.conj.b))
	    {
	      cstring name1 = cstring_copy (name);

	      return 
		(message 
		 ("<%q> | <%q>", 
		  ctbase_unparseDeclaration 
		  (ctype_getCtbase (c->contents.conj.a), name1),
		  ctbase_unparseDeclaration 
		  (ctype_getCtbase (c->contents.conj.b), name)));
	    }
	  else
	    {
	      cstring s1 = ctbase_unparseDeclaration (ctype_getCtbase (c->contents.conj.a), 
						      cstring_copy (name));
	      return
		(message ("%q | %q", s1, 
			  ctbase_unparseDeclaration (ctype_getCtbase (c->contents.conj.b),
						     name)));
	    }
	}
      else
	{
	  return (cstring_copy (ctype_unparse (c->contents.conj.a)));
	}
      BADDEFAULT;
    }
  BADEXIT;
}

static ctbase ctbase_undump (char **c)
{
  ctbase res;
  char p = **c;

  (*c)++;

  switch (p)
    {
    case '?':
      return (ctbase_undefined);
    case 'u':
      return (ctbase_createUnknown ());
    case 'b':
      return (ctbase_createBool ());
    case 'p':
      res = ctbase_createPrim (cprim_fromInt (getInt (c)));
      checkChar (c, '|');
      return res;
    case 's':
      res = ctbase_createUser (typeId_fromInt (getInt (c)));
      checkChar (c, '|');
      return res;
    case 'a':
      res = ctbase_createAbstract (typeId_fromInt (getInt (c)));
      checkChar (c, '|');
      return res;
    case 't':
      res = ctbase_makePointer (ctype_undump (c));
      checkChar (c, '|');
      return res;
    case 'y':
      res = ctbase_makeArray (ctype_undump (c));
      checkChar (c, '|');
      return res;
    case 'f':
      {
	ctype ct;
	char *lp = strchr (*c, '(');

	llassertfatal (lp != NULL);

	*lp = '\0';
	ct = ctype_undump (c);
	*c = lp + 1;

	return (ctbase_makeLiveFunction (ct, uentryList_undump (c)));
      }
    case 'S':
      {
	uentryList fields;
	ctbase ctb;
	char *sname;
	char *lc = strchr (*c, '{');

	llassert (lc != NULL);
	*lc = '\0';

	sname = mstring_copy (*c);

	*c = lc + 1;

	if (*sname == '!')
	  {
	    unsigned int i;

	    i = (unsigned) atoi (sname + 1);

	    setTagNo (i);
	  }

	fields = uentryList_undumpFields (c, currentloc);

	DPRINTF (("fields: %s / %s", sname, uentryList_unparseFull (fields)));

	ctb = ctbase_createStruct (cstring_fromCharsO (sname), fields);

	DPRINTF (("returns: %s", ctbase_unparse (ctb)));
	DPRINTF (("s -> %s", *c));
	return ctb;
      }
    case 'U':
      {
	char *sname;
	char *lc = strchr (*c, '{');

	llassert (lc != NULL);

	*lc = '\0';
	sname = mstring_copy (*c);

	*c = lc + 1;

	if (*sname == '!')
	  {
	    unsigned int i;

	    i = (unsigned) atoi (sname + 1);
	    setTagNo (i);
	  }

	return (ctbase_createUnion (cstring_fromCharsO (sname),
				    uentryList_undumpFields (c, currentloc)));
      }
    case 'e':
      {
	ctbase ret;
	char *sname;
	char *lc = strchr (*c, '{');

	llassert (lc != NULL);

	*lc = '\0';
	sname = mstring_copy (*c);
	*c = lc + 1;

	if (*sname == '!')
	  {
	    unsigned int i;

	    i = (unsigned) atoi (sname + 1);
	    setTagNo (i);
	    DPRINTF (("!!! load tag: %d", i));
	  }

	ret = ctbase_createEnum (cstring_fromCharsO (sname),
				 enumNameList_undump (c));
	DPRINTF (("ctbase enum %s: %s", sname, ctbase_unparse (ret)));
	return ret;
      }
    case 'C':
      {
	bool isExplicit;
	ctype c1, c2;

	isExplicit = bool_fromInt (getInt (c));
	checkChar (c, '.');
	c1 = ctype_undump (c);
	checkChar (c, '/');
	c2 = ctype_undump (c);
	checkChar (c, '|');

	return (ctbase_makeConj (c1, c2, isExplicit));
      }

    default:
      (*c)--;
      llerror (FLG_SYNTAX, 
	       message ("Bad Library line (type): %s", cstring_fromChars (*c)));

      while (**c != '\0')
	{
	  (*c)++;
	}

      return ctbase_createUnknown ();
    }
}

/* first letter of c encodes type:     */
/*    u  unknown                       */
/*    b  bool                          */
/*    p  prim                          */
/*    e  enum                          */
/*    l  enumList                      */
/*    s  uSer                          */
/*    a  abstract                      */
/*    t  poinTer                       */
/*    y  arraY                         */
/*    f  function                      */
/*    S  structure                     */
/*    U  union                         */
/*    C  conj                          */

static /*@only@*/ cstring
ctbase_dump (ctbase c)
{
  if (!ctbase_isDefined (c))
    {
      return cstring_makeLiteral ("?");
    }

  switch (c->type)
    {
    case CT_UNKNOWN:
      return cstring_makeLiteral ("u");
    case CT_BOOL:
      return cstring_makeLiteral ("b");
    case CT_PRIM:
      return (message ("p%d|", c->contents.prim));
    case CT_USER:
      return (message ("s%d|", 
		       usymtab_convertId (c->contents.tid)));
    case CT_ABST:
      return (message ("a%d|", usymtab_convertId (c->contents.tid)));
    case CT_PTR:
      return (message ("t%q|", ctype_dump (c->contents.base)));
    case CT_ARRAY:
      return (message ("y%q|", ctype_dump (c->contents.base)));
    case CT_FCN:
      return (message ("f%q (%q)", ctype_dump (c->contents.fcn.rval),
		       uentryList_dumpParams (c->contents.fcn.params)));
    case CT_STRUCT:
      return (message ("S%s{%q}", c->contents.su.name,
		       uentryList_dumpFields (c->contents.su.fields)));
    case CT_UNION:
      return (message ("U%s{%q}", c->contents.su.name,
		       uentryList_dumpFields (c->contents.su.fields)));
    case CT_ENUM:
      {
	cstring s;
	
	if (cstring_isNonEmpty (c->contents.cenum.tag))
	  {
	    s = message ("e%s{%q}", 
			 c->contents.cenum.tag,
			 enumNameList_dump (c->contents.cenum.members));
	  }
	else
	  {
	    s = message ("e{%q}", 
			 enumNameList_dump (c->contents.cenum.members));
	  }
	return (s);
      }
    case CT_CONJ:
      return (message ("C%d.%q/%q|", 
		       bool_toInt (c->contents.conj.isExplicit),
		       ctype_dump (c->contents.conj.a),
		       ctype_dump (c->contents.conj.b)));
    case CT_EXPFCN:
      /* should clean them up! */
      return (cstring_makeLiteral ("?"));
    case CT_ENUMLIST:
      llcontbug (message ("Cannot dump: %q", ctbase_unparse (c))); 
      return (message ("u"));
    BADDEFAULT;
    }

  BADEXIT;
}

static /*@only@*/ ctbase
ctbase_copy (/*@notnull@*/ ctbase c)
{
  switch (c->type)
    {
    case CT_UNKNOWN:
      return (ctbase_createUnknown ());
    case CT_BOOL:
      return (ctbase_createBool ());
    case CT_ENUM:
      return (ctbase_createEnum (cstring_copy (c->contents.cenum.tag), 
				 enumNameList_copy (c->contents.cenum.members)));
    case CT_PRIM:
      return (ctbase_createPrim (c->contents.prim));
    case CT_USER:
      return (ctbase_createUser (c->contents.tid));
    case CT_ABST:
      return (ctbase_createAbstract (c->contents.tid));
    case CT_EXPFCN:
      return (ctbase_expectFunction (c->contents.base));
    case CT_PTR:
      return (ctbase_makePointer (c->contents.base));
    case CT_ARRAY:
      return (ctbase_makeArray (c->contents.base));
    case CT_FCN:
      return (ctbase_makeLiveFunction (c->contents.fcn.rval,
				       uentryList_copy (c->contents.fcn.params)));	
    case CT_STRUCT:
      DPRINTF (("copy struct: %s", uentryList_unparse (c->contents.su.fields)));
      return (ctbase_createStruct (cstring_copy (c->contents.su.name),
				   uentryList_copy (c->contents.su.fields)));
    case CT_UNION:
      return (ctbase_createUnion (cstring_copy (c->contents.su.name),
				  uentryList_copy (c->contents.su.fields)));
    case CT_CONJ:
      /*@i@*/ return (c); /* not a real copy for conj's */
    default:
      llbug (("ctbase_copy: %s", ctbase_unparse (c)));
    }

  BADEXIT;
}
				
static enumNameList
ctbase_elist (ctbase c)
{
  llassert (ctbase_isDefined (c));
  llassert (c->type == CT_ENUM);

  return (c->contents.cenum.members);
}

static void
ctbase_free (/*@only@*/ ctbase c)
{
  if (c == ctbase_bool || c == ctbase_unknown)
    {
      /*@-mustfree@*/ return; /*@=mustfree@*/
    }

  --nctbases;
  
  DPRINTF (("free ctbase: [%ld]", (long) c));
  
  if (ctbase_isDefined (c))
    {
      switch (c->type)
	{
	case CT_UNKNOWN:
	  sfree (c);
	  break;
	case CT_PRIM:
	  sfree (c);
	  break;
	case CT_ENUM:
	  sfree (c);
	  break;
	case CT_ENUMLIST:
	  /* sfree list? */
	  sfree (c);
	  break;
	case CT_USER:
	  sfree (c);
	  break;
	case CT_ABST:
	  sfree (c);
	  break;
	case CT_PTR:
	case CT_ARRAY:
	  sfree (c);
	  break;
	case CT_FCN:

	  if (c->contents.fcn.liveparams)
	    {
	      /* Because of liveparams, we know this can be released. */

	      /*@-dependenttrans@*/
	      uentryList_free (c->contents.fcn.params); 
	      /*@=dependenttrans@*/
	    }

	  sfree (c);
	  break;
	case CT_STRUCT:
	case CT_UNION:
	  cstring_free (c->contents.su.name);
	  uentryList_free (c->contents.su.fields); 
	  sfree (c);
	  break;
	case CT_CONJ:
	  /* Don't bree conj's, */
	  break;
	default:
	  sfree (c);
	  break;
	}
    }
}

/*
** c should be * <unknown>
*/

static /*@only@*/ ctbase
ctbase_expectFunction (ctype c)
{
  ctbase f = ctbase_new ();

  f->type = CT_EXPFCN;
  f->contents.base = c;

  DPRINTF (("expect function: %s", ctype_unparse (c)));
  return (f);
}

static bool
ctbase_genMatch (ctbase c1, ctbase c2, bool force, bool arg, bool def)
{
  ctuid c1tid, c2tid;
  
  /* undefined types never match */
  
  if (ctbase_isUndefined (c1) || ctbase_isUndefined (c2))
    return FALSE;
  
  /* abstract types match user types of same name */

  DPRINTF (("before 1: %s / 2: %s",
	    ctbase_unparse (c1), ctbase_unparse (c2)));
  
  c1 = ctbase_realType (c1);
  c2 = ctbase_realType (c2);
  
  c1tid = c1->type;
  c2tid = c2->type;
  
  if (c1tid == CT_CONJ)
    {
      return (ctbase_genMatch (ctype_getCtbase (c1->contents.conj.a), c2,
			       force, arg, def)
	      || ctbase_genMatch (ctype_getCtbase (c1->contents.conj.b), c2,
				  force, arg, def));
    }
  
  if (c2tid == CT_CONJ)
    {
      return (ctbase_genMatch (c1, ctype_getCtbase (c2->contents.conj.a),
			       force, arg, def)
	      || ctbase_genMatch (c1, ctype_getCtbase (c2->contents.conj.b),
				  force, arg, def));
    }

  /*
  ** if the types don't match, there are some special cases...
  */

  if (c1tid != c2tid)
    {
      /* unknowns match anything */
      
      DPRINTF (("c1tid/c2tid: %d / %d",
		c1tid, c2tid));

      DPRINTF (("1: %s / 2: %s",
		ctbase_unparse (c1), ctbase_unparse (c2)));
      
      if (c1tid == CT_UNKNOWN || c2tid == CT_UNKNOWN)
	{
	  return TRUE;
	}
      
      /* in most cases, bool and int are matched if FLG_BOOLINT */
      
      if ((c1tid == CT_BOOL 
	   && (c2tid == CT_PRIM && cprim_isInt (c2->contents.prim))) ||
	  (c2tid == CT_BOOL 
	   && (c1tid == CT_PRIM && cprim_isInt (c1->contents.prim))))
	{
	  return (context_msgBoolInt ());
	}

      if ((c1tid == CT_ENUM && (c2tid == CT_PRIM && cprim_isInt (c2->contents.prim))) ||
	  (c2tid == CT_ENUM && (c1tid == CT_PRIM && cprim_isInt (c1->contents.prim))))
	{
	  return (context_msgEnumInt ());
	}

      /*
      ** arrays and pointers...yuk!
      **
      ** Considered equivalent except in definitions.  (e.g., function parameters 
      ** are equivalent)
      **
      */
      
      if (!def)
	{
	  if (ctuid_isAP (c1tid) && ctuid_isAP (c2tid))
	    {
	      c2tid = c1tid;
	    }
	}

      /*
      ** we allow forward declarations to structures like,
      **
      **          typedef struct _t *t;
      ** 
      ** to allow,
      **           struct _t * to match t
      */
  
      if (context_getFlag (FLG_FORWARDDECL))
	{
	  if (c1tid == CT_ABST || c1tid == CT_USER)
	    {
	      if (ctuid_isAP (c2tid))
		{
		  ctype ts = c2->contents.base;
		  
		  if (ctype_isUA (ts))
		    {
		      typeId ttid = ctype_typeId (ts);
		      typeId ctid = c1->contents.tid ;
		      
		      if (usymtab_matchForwardStruct (ctid, ttid))
			{
			  return TRUE;
			}
		    }
		}
	    }
	  
	  if (c2tid == CT_ABST || c2tid == CT_USER)
	    {
	      if (ctuid_isAP (c1tid))
		{
		  ctype ts = c1->contents.base;
		  
		  if (ctype_isUA (ts))
		    {
		      typeId ttid = ctype_typeId (ts);
		      typeId ctid = c2->contents.tid ;
		      
		      if (usymtab_matchForwardStruct (ctid, ttid))
			{
			  return TRUE;
			}
		    }
		}
	    }
	}
    }

  if (c1tid != c2tid)
    return FALSE;

  switch (c1tid)
    {
    case CT_UNKNOWN:
      return (TRUE);
    case CT_PRIM:
      return (cprim_closeEnough (c1->contents.prim, c2->contents.prim));
    case CT_BOOL:
      return (TRUE);
    case CT_ABST:
      return (typeId_equal (c1->contents.tid, c2->contents.tid));
    case CT_USER:
      return (typeId_equal (c1->contents.tid, c2->contents.tid));
    case CT_ENUM:	
      return (cstring_equal (c1->contents.cenum.tag, c2->contents.cenum.tag));
    case CT_PTR:
      if (ctype_isVoid (c1->contents.base) || (ctype_isVoid (c2->contents.base)))
	{
	  if (ctype_isFunction (ctype_realType (c1->contents.base))
	      || ctype_isFunction (ctype_realType (c2->contents.base)))
	    {
	      return (!context_getFlag (FLG_CASTFCNPTR));
	    }
	  else
	    {
	      return (context_getFlag (FLG_ABSTVOIDP) ||
		      (!(ctype_isRealAbstract (c1->contents.base)) &&
		       !(ctype_isRealAbstract (c2->contents.base))));
	    }
	}
      else
	{
	  return (ctype_genMatch (c1->contents.base, c2->contents.base, force, arg, def));
	}
    case CT_ARRAY:
      if (ctype_isVoid (c1->contents.base) || ctype_isVoid (c2->contents.base))
	return TRUE;
      return (ctype_genMatch (c1->contents.base, c2->contents.base, force, arg, def));
    case CT_FCN:
      return (ctype_genMatch (c1->contents.fcn.rval, 
			      c2->contents.fcn.rval, 
			      force, arg, def) 
	      && uentryList_matchParams (c1->contents.fcn.params, 
					 c2->contents.fcn.params, 
					 force, TRUE));
    case CT_STRUCT:
    case CT_UNION:
      if (!cstring_isEmpty (c1->contents.su.name))
	{
	  return (cstring_equal (c1->contents.su.name, c2->contents.su.name));
	}
      else
	{
	  if (!cstring_isEmpty (c2->contents.su.name))
	    {
	      return FALSE;
	    }

	  llcontbuglit ("ctbase_genMatch: match fields");
	  return (FALSE);
	}
    default:
      llcontbug (message ("ctbase_genMatch: unknown type: %d\n", (int)c1tid));
      return (FALSE);
    }
}

/*
** like ctbase_match, except for conjuncts:
**   modifies conjuncts to match only
*/

static bool
ctbase_forceMatch (ctbase c1, ctbase c2)
{
 /*
 ** modifies c1, c2
 */

  return (ctbase_genMatch (c1, c2, TRUE, FALSE, FALSE));
}

static bool
ctbase_match (ctbase c1, ctbase c2) /*@modifies nothing@*/
{
  return (ctbase_genMatch (c1, c2, FALSE, FALSE, FALSE));
}

static bool
ctbase_matchDef (ctbase c1, ctbase c2) /*@modifies nothing@*/
{
  return (ctbase_genMatch (c1, c2, FALSE, FALSE, TRUE));
}

static bool
ctbase_matchArg (ctbase c1, ctbase c2)
{
  return (ctbase_genMatch (c1, c2, FALSE, TRUE, FALSE));
}

static /*@out@*/ /*@only@*/ /*@notnull@*/ ctbase 
ctbase_new ()
{
  ctbase c = (ctbase) dmalloc (sizeof (*c));

  nctbases++;
 /*
   if (nctbases % 100 == 0 && nctbases > lastnc)
     {
       llmsg (message ("ctbases: %d", nctbases));
       lastnc = nctbases;
     }
 */
  return (c);
}

static /*@only@*/ ctbase
ctbase_createPrim (cprim p)
{
  ctbase c = ctbase_new ();

  c->type = CT_PRIM;
  c->contents.prim = p;

  return (c);
}

static /*@observer@*/ ctbase
ctbase_getBool (void)
{
  /*@i@*/ return ctbase_createBool ();
}

static ctbase
ctbase_createBool ()
{
  if (!ctbase_isDefined (ctbase_bool))
    {
      ctbase_bool = ctbase_new ();
      ctbase_bool->type = CT_BOOL;
      ctbase_bool->contents.prim = CTX_BOOL;
    }

  /*@-retalias@*/ /*@-globstate@*/
  return ctbase_bool;
  /*@=retalias@*/ /*@=globstate@*/
}

static /*@only@*/ ctbase
ctbase_createUser (typeId u)
{
  ctbase c = ctbase_new ();

  c->type = CT_USER;
  c->contents.tid = u;

  llassert (typeId_isValid (u));

  return (c);
}

static /*@only@*/ ctbase
ctbase_createEnum (/*@keep@*/ cstring etag, /*@keep@*/ enumNameList emembers)
{
  ctbase c = ctbase_new ();

  c->type = CT_ENUM;

  if (cstring_isUndefined (etag))
    {
      llcontbuglit ("Undefined enum tag!");
      etag = fakeTag ();
    }

  c->contents.cenum.tag = etag;
  c->contents.cenum.members = emembers;

  return (c);
}

static /*@observer@*/ cstring
ctbase_enumTag (/*@notnull@*/ ctbase ct)
{
  return (ct->contents.cenum.tag);
}

static /*@only@*/ ctbase
ctbase_createAbstract (typeId u)
{
  ctbase c = ctbase_new ();

  c->type = CT_ABST;
  c->contents.tid = u;

 /* also check its abstract? */

  llassert (typeId_isValid (c->contents.tid));

  return (c);
}

static /*@only@*/ ctbase
ctbase_createUnknown (void)
{
  if (!ctbase_isDefined (ctbase_unknown))
    {
      ctbase_unknown = ctbase_new ();
      ctbase_unknown->type = CT_UNKNOWN;
      ctbase_unknown->contents.prim = CTX_UNKNOWN;
    }

  /*@-retalias@*/ /*@-globstate@*/
  return ctbase_unknown;
  /*@=retalias@*/ /*@=globstate@*/
}

/*
** requires: result is not assigned to b
**           (should copy, but no way to reclaim storage)
*/

static /*@only@*/ ctbase
ctbase_makePointer (ctype b)
{
  ctbase c = ctbase_new ();

  c->type = CT_PTR;
  c->contents.base = b;

  return (c);
}

static /*@only@*/ ctbase
ctbase_makeArray (ctype b)
{
  ctbase c = ctbase_new ();

  c->type = CT_ARRAY;
  c->contents.base = b;

  return (c);
}

static ctype
ctbase_makeFunction (ctype b, /*@only@*/ uentryList p)
{
  ctbase c = ctbase_new ();
  ctype ct;
  
  DPRINTF (("makeFunction: %s", ctype_unparse (b)));

  c->type = CT_FCN;

  if (ctype_isFunction (b) && ctype_isPointer (b))
    {
      ctbase ctb = ctype_getCtbase (ctype_baseArrayPtr (b));
      ctype rval;
      
      llassert (ctbase_isDefined (ctb));
      llassert (ctb->type == CT_FCN);
      
      rval = ctype_makeFunction (ctb->contents.fcn.rval, p);
      c->contents.fcn.rval = rval;
      c->contents.fcn.params = ctb->contents.fcn.params;
      c->contents.fcn.liveparams = FALSE;
    }
  else
    {
      c->contents.fcn.rval = b;
      /*@i@*/ c->contents.fcn.params = p;
      c->contents.fcn.liveparams = TRUE;
      /*@-branchstate@*/ /* p is really released on this branch */
    } 
  /*@=branchstate@*/

  ct = cttable_addComplex (c);  
  return (ctype_makePointer (ct));
}

static ctype
ctbase_makeNFFunction (ctype b, /*@only@*/ uentryList p)
{
  ctbase c = ctbase_new ();
  ctype ct;

  c->type = CT_FCN;

  if (ctype_isFunction (b) && ctype_isPointer (b))
    {
      ctbase ctb  = ctype_getCtbase (ctype_baseArrayPtr (b));
      ctype rval;
      
      llassert (ctbase_isDefined (ctb));
      llassert (ctb->type == CT_FCN);
      
      rval = ctype_makeNFParamsFunction (ctb->contents.fcn.rval, p);
      c->contents.fcn.rval = rval;
      c->contents.fcn.params = ctb->contents.fcn.params;
      c->contents.fcn.liveparams = FALSE;
    }
  else
    {
      c->contents.fcn.rval = b;
      /*@i@*/ c->contents.fcn.params = p;
      c->contents.fcn.liveparams = TRUE;

      /*@-branchstate@*/ 
    }
  /*@=branchstate@*/

  ct = cttable_addComplex (c);
  return (ctype_makePointer (ct));
}

static /*@only@*/ ctbase
  ctbase_makeLiveFunction (ctype b, /*@owned@*/ uentryList p)
{
  ctbase c = ctbase_new ();

  c->type = CT_FCN;
  c->contents.fcn.rval = b;
  c->contents.fcn.params = p;
  c->contents.fcn.liveparams = TRUE;

  /*@-mustfree@*/ return (c); /*@=mustfree@*/
}

static /*@only@*/ ctbase
  ctbase_makeRealFunction (ctype b, /*@dependent@*/ uentryList p)
{
  ctbase c = ctbase_new ();

  c->type = CT_FCN;
  c->contents.fcn.rval = b;
  c->contents.fcn.params = p;
  c->contents.fcn.liveparams = FALSE;

  return (c);
}

static /*@observer@*/ /*@notnull@*/ ctbase
ctbase_realFunction (/*@dependent@*/ /*@notnull@*/ ctbase c)
{
  ctbase res;

  if (c->type == CT_FCN)
    {
      return c;
    }

  llassert (ctbase_isFunction (c));

  res = ctype_getCtbase (c->contents.base);
  
  llassert (ctbase_isDefined (res));

  return (res);
}

static bool
ctbase_isFunction (ctbase c)
{
  llassert (c != ctbase_undefined);

  if (c->type == CT_FCN)
    {
      return TRUE;
    }
  else
    {
      if (c->type == CT_PTR)
	{
	  ctbase fcn = ctype_getCtbase (ctbase_baseArrayPtr (c));

	  return (ctbase_isDefined (fcn) && fcn->type == CT_FCN);
	}

      return FALSE;
    }
}

/* doesn't copy c1 and c2 */

static /*@only@*/ ctbase
  ctbase_makeConj (ctype c1, ctype c2, bool isExplicit)
{
  ctbase c = ctbase_new ();

  c->type = CT_CONJ;
  c->contents.conj.a = c1;
  c->contents.conj.b = c2;
  c->contents.conj.isExplicit = isExplicit;

  return (c);
}

static ctype
ctbase_getConjA (/*@notnull@*/ ctbase c)
{
  llassert (c->type == CT_CONJ);
  return (c->contents.conj.a);
}

static ctype
ctbase_getConjB (/*@notnull@*/ ctbase c)
{
  llassert (c->type == CT_CONJ);
  return (c->contents.conj.b);
}

static /*@only@*/ ctbase
ctbase_createStruct (/*@only@*/ cstring n, /*@only@*/ uentryList f)
{
  ctbase c = ctbase_new ();

  c->type = CT_STRUCT;
  c->contents.su.name = n;
  c->contents.su.fields = f;

  DPRINTF (("struct: %s (%s)", n, uentryList_unparse (f))); 
  return (c);
}

static /*@observer@*/ uentryList
ctbase_getuentryList (/*@notnull@*/ ctbase c)
{
  c = ctbase_realType (c);

  if (!(c->type == CT_STRUCT || c->type == CT_UNION))
    llfatalbug (message ("ctbase_getuentryList: bad invocation: %q", ctbase_unparse (c)));

  return (c->contents.su.fields);
}

static ctbase
ctbase_createUnion (/*@keep@*/ cstring n, /*@only@*/ uentryList f)
{
  ctbase c = ctbase_new ();

  c->type = CT_UNION;
  c->contents.su.name = n;
  c->contents.su.fields = f;

  return (c);
}

static ctype
ctbase_baseArrayPtr (/*@notnull@*/ ctbase c)
{
  ctuid ct;
  c = ctbase_realType (c);
  ct = c->type;

  if (!(ctuid_isAP (ct)))
    llfatalbug (message ("ctbase_baseArrayPtr: bad call: %q", ctbase_unparse (c)));
  
  return c->contents.base;
}

static ctype
ctbase_baseFunction (/*@notnull@*/ ctbase c)
{
  ctbase_fixUser (c);
  c = ctbase_realFunction (c);

  if (c->type != CT_FCN)
    {
      llfatalbug (message ("ctbase_baseFunction: bad call: %q", ctbase_unparse (c)));
    }

  return (c->contents.fcn.rval);
}

static uentryList
ctbase_argsFunction (/*@notnull@*/ ctbase c)
{
  ctbase_fixUser (c);
  c = ctbase_realFunction (c);

  if (c->type != CT_FCN)
    {
      llfatalbug (message ("ctbase_argsFunction: bad call: %q", 
			   ctbase_unparse (c)));
    }
  return (c->contents.fcn.params);
}

static bool
ctbase_baseisExpFcn (ctype c)
{
  ctbase cb;
  c = ctype_removePointers (c);

  cb = ctype_getCtbase (c);
  llassert (ctbase_isDefined (cb));

  if (cb->type == CT_FCN)
    {
      c = ctype_removePointers (ctype_returnValue (c));

      cb = ctype_getCtbase (c);
      llassert (ctbase_isDefined (cb));

      return (cb->type == CT_EXPFCN);
    }
  return FALSE;
}

/*
** ctbase_newBase behaves specially when p is a CONJ:
**
**    c -> conj (newBase (c, p.a), p.b)
*/

static ctype
ctbase_newBase (ctype c, ctype p)
{
  ctbase cb;

  DPRINTF (("newBase: %s / %s", ctype_unparse (c), ctype_unparse (p)));

  cb = ctype_getCtbase (c);

  DPRINTF (("cb: %s", ctbase_unparse (cb)));

  if (ctype_isUndefined (c))
    {
      return p;
    }

  if (ctype_isConj (p))
    {
      ctbase pb = ctype_getCtbase (p);

      llassert (ctbase_isDefined (pb));

      if (pb->contents.conj.isExplicit)
	{
	  return (ctype_makeExplicitConj (pb->contents.conj.b, 
					  ctype_newBase (c, pb->contents.conj.a)));
	}
      else
	{
	  return (ctype_makeConj (pb->contents.conj.b, 
				  ctype_newBase (c, pb->contents.conj.a)));
	}
    }

  if (ctbase_baseisExpFcn (c))
    {
      return (ctbase_newBaseExpFcn (c, p));
    }

  llassert (ctbase_isDefined (cb));

  switch (cb->type)
    {
    case CT_UNKNOWN:
    case CT_PRIM:
    case CT_USER:
    case CT_ENUM:
    case CT_ABST:
    case CT_STRUCT:
    case CT_UNION:
    case CT_EXPFCN:
      DPRINTF (("return %s", ctype_unparse (p)));
      return (p);

    case CT_PTR:
      {
	ctype ret;
	ctype cbn;
	DPRINTF (("pointer"));
	cbn = ctbase_newBase (cb->contents.base, p);
	DPRINTF (("cbn: %s [%d]", ctype_unparse (cbn), cbn));
	ret = ctype_makePointer (cbn);
	DPRINTF (("pointer -> %s [%d]", ctype_unparse (ret), ret));
	return ret;
      }
    case CT_ARRAY:
      return (ctype_makeArray (ctbase_newBase (cb->contents.base, p)));
    case CT_FCN:
      DPRINTF (("new function base: %s", ctype_unparse (p)));
      return (ctype_makeRealFunction (ctbase_newBase (cb->contents.fcn.rval, p),
				      cb->contents.fcn.params));
    case CT_CONJ:
      return (ctype_makeConjAux (ctbase_newBase (cb->contents.conj.a, p),
				 ctbase_newBase (cb->contents.conj.b, p),
				 cb->contents.conj.isExplicit));
    default:
      llcontbug (message ("ctbase_newBase: bad ctbase: %q", ctbase_unparse (cb))); 
      return (p); 
    }
  BADEXIT;
}

static ctype
ctbase_newBaseExpFcn (ctype c, ctype p)
{
  ctbase cb = ctype_getCtbase (c);
  ctbase tcb;
  ctype ret, tmpct;
  ctype fp = ctype_unknown;
  uentryList ctargs = ctype_argsFunction (c);

  DPRINTF (("newBaseExpFcn: %s / %s",
	    ctype_unparse (c), ctype_unparse (p)));

  /*
  ** okay, this is really ugly...
  **
  ** pointers inside <expf> mean pointers to the function;
  ** pointers outside <expf> are pointers to the return value;
  ** because its a function there is one superfluous pointer.
  */
  
  /*
  ** bf is a ctype, used to derived structure of cb
  */
  
  if (!ctbase_isFunction (cb))
    llbuglit ("ctbase_newBaseExpFcn: expFcn -> not a function");
  
  tmpct = ctype_getBaseType (ctype_returnValue (c));
  
  /*
  ** pointers before expfcn -> p are pointers to function, not result
  **
  */
  
  tcb = ctype_getCtbase (tmpct);

  llassert (ctbase_isDefined (tcb));
  tmpct = tcb->contents.base;
  
  /*
  ** record pointers to base in fp
  */
  
  while (!ctype_isUnknown (tmpct))
    {
      switch (ctype_getCtKind (tmpct))
	{
	case CTK_PTR:
	  fp = ctype_makePointer (fp);
	  /*@switchbreak@*/ break;
	case CTK_ARRAY:
	  DPRINTF (("Make array: %s", ctype_unparse (fp)));
	  fp = ctype_makeArray (fp);
	  /*@switchbreak@*/ break;
	case CTK_COMPLEX:
	  {
	    ctbase fbase = ctype_getCtbase (tmpct);
	  
	    if (ctbase_isFunction (fbase))
	      {
		fp = ctype_makeFunction (fp, uentryList_copy (ctargs));
		ctargs = ctbase_argsFunction (fbase);
	      }
	    else
	      {
		llbug 
		  (message
		   ("ctbase_newBaseExpFcn: fixing expfunction: bad complex type: %s",
		    ctype_unparse (tmpct)));
	      }
	    goto exitLoop;
	  }
	default:
	  {
	    llcontbug 
	      (message ("ctbase_newBaseExpFcn: fixing expfunction: bad type: %s",
			ctype_unparse (tmpct)));
	    goto exitLoop; 
	  }
	}
      tmpct = ctype_baseArrayPtr (tmpct);
    }

 exitLoop:
  tmpct = ctype_returnValue (c);

  /*
  ** pointers to expf are pointers to return value
  */
  
  while (!ctype_isExpFcn (tmpct))
    {
      switch (ctype_getCtKind (tmpct))
	{
	case CTK_PTR:
	  p = ctype_makePointer (p);
	  /*@switchbreak@*/ break;
	case CTK_ARRAY:
	  p = ctype_makeArray (p);
	  /*@switchbreak@*/ break;
	case CTK_COMPLEX:
	  {
	    ctbase fbase = ctype_getCtbase (tmpct);
	  
	    if (ctbase_isFunction (fbase))
	      {
		p = ctype_makeFunction (p, uentryList_copy (ctbase_argsFunction (fbase)));
	      }
	    else
	      {
		llbug 
		  (message
		   ("ctbase_newBaseExpFcn: fixing expfunction: bad complex type: %s",
		    ctype_unparse (tmpct)));
	      }
	    goto exitLoop2;
	  }

	default:
	  {
	    llcontbug 
	      (message ("ctbase_newBaseExpFcn: fixing expfunction2: bad type: %t", 
			tmpct));
	    goto exitLoop2; 
	  }
	}
      tmpct = ctype_baseArrayPtr (tmpct);
    }
  
 exitLoop2:
  
  /*
  ** pointers to fp are pointers to function type
  */

  ret = ctype_makeRealFunction (p, ctargs);
  
  while (ctype_getCtKind (fp) > CTK_PLAIN)
    {
      switch (ctype_getCtKind (fp))
	{
	case CTK_PTR:
	  ret = ctype_makePointer (ret);
	  /*@switchbreak@*/ break;
	case CTK_ARRAY:
	  ret = ctype_makeArray (ret);
	  /*@switchbreak@*/ break;
	case CTK_COMPLEX:
	  {
	    ctbase fbase = ctype_getCtbase (fp);
	    
	    if (ctbase_isFunction (fbase))
	      {
		ret = 
		  ctype_makeFunction (ret,
				      uentryList_copy (ctbase_argsFunction (fbase)));
	      }
	    else
	      {
		BADBRANCH;
	      }
	    goto exitLoop3;
	  }

	default:
	  {
	    llcontbug (message ("post-fixing expfunction: bad type: %t", fp));
	    goto exitLoop3;
	  }
	}
      fp = ctype_baseArrayPtr (fp);
    }
  
 exitLoop3:
  return (ret);
}

/*
** returns lowest level base of c: plain type
*/

static /*@notnull@*/ /*@only@*/ ctbase
ctbase_getBaseType (/*@notnull@*/ ctbase c)
{
  switch (c->type)
    {
    case CT_UNKNOWN:
    case CT_PRIM:
    case CT_USER:
    case CT_ENUM:
    case CT_ENUMLIST:
    case CT_BOOL:
    case CT_ABST:
    case CT_FCN:
    case CT_STRUCT:
    case CT_UNION:
      return (ctbase_copy (c));

    case CT_PTR:
    case CT_ARRAY:
      return (ctbase_getBaseType (ctype_getCtbaseSafe (c->contents.base)));

    case CT_CONJ:		/* base type of A conj branch? */
      return (ctbase_getBaseType (ctype_getCtbaseSafe (c->contents.conj.a)));
    case CT_EXPFCN:
      return (ctbase_copy (c));

    default:
      llfatalbug (message ("ctbase_getBaseType: bad ctbase: %q", ctbase_unparse (c)));
    }

  BADEXIT;
}

static int
ctbase_compare (ctbase c1, ctbase c2, bool strict)
{
  ctuid c1tid, c2tid;

  if (ctbase_isUndefined (c1) || ctbase_isUndefined (c2))
    {
      llcontbuglit ("ctbase_compare: undefined ctbase");
      return -1;
    }

  c1tid = c1->type;
  c2tid = c2->type;

  if (c1tid < c2tid)
    return -1;
  if (c1tid > c2tid)
    return 1;

  switch (c1tid)
    {
    case CT_UNKNOWN:
      return 0;
    case CT_PRIM:
      return (int_compare (c1->contents.prim, c2->contents.prim));
    case CT_BOOL:
      return 0;
    case CT_USER:
      return (int_compare (c1->contents.tid, c2->contents.tid));
    case CT_ENUMLIST:	     
      return 1;
    case CT_ENUM:		/* for now, keep like abstract */
    case CT_ABST:
      return (int_compare (c1->contents.tid, c2->contents.tid));
    case CT_PTR:
      return (ctype_compare (c1->contents.base, c2->contents.base));
    case CT_ARRAY:
      return (ctype_compare (c1->contents.base, c2->contents.base));
    case CT_FCN:
      {
	COMPARERETURN (ctype_compare (c1->contents.fcn.rval, c2->contents.fcn.rval));

	if (strict)
	  {
	    return (uentryList_compareStrict (c1->contents.fcn.params, 
					      c2->contents.fcn.params));
	  }
	else
	  {
	    return (uentryList_compareParams (c1->contents.fcn.params, 
					      c2->contents.fcn.params));
	  }
      }
    case CT_EXPFCN:
      return (ctype_compare (c1->contents.base, c2->contents.base));
    case CT_STRUCT:
    case CT_UNION:
      return (uentryList_compareFields (c1->contents.su.fields,
					c2->contents.su.fields));
    case CT_CONJ:
      {
	COMPARERETURN (ctype_compare (c1->contents.conj.a,
				      c2->contents.conj.a));
	COMPARERETURN (ctype_compare (c1->contents.conj.b,
				      c2->contents.conj.b));
	return (bool_compare (c1->contents.conj.isExplicit,
			      c2->contents.conj.isExplicit));
      }
    }
  BADEXIT;
}

static int
ctbase_compareStrict (/*@notnull@*/ ctbase c1, /*@notnull@*/ ctbase c2)
{
 return (ctbase_compare (c1, c2, TRUE));
}

static bool ctbase_equivStrict (/*@notnull@*/ ctbase c1, /*@notnull@*/ ctbase c2)
{
  return (ctbase_compareStrict (c1,c2) == 0);
}

static bool ctbase_equiv (/*@notnull@*/ ctbase c1, /*@notnull@*/ ctbase c2)
{
  return (ctbase_compare (c1, c2, FALSE) == 0);
}

static bool
ctbase_isKind (/*@notnull@*/ ctbase c, ctuid kind)
{
  ctuid ck = c->type;

  if (ck == kind)
    return TRUE;

  if (ck == CT_CONJ)
    return (ctbase_isKind (ctype_getCtbaseSafe (c->contents.conj.a), kind) ||
	    ctbase_isKind (ctype_getCtbaseSafe (c->contents.conj.b), kind));

  return FALSE;
}

static bool
ctbase_isKind2 (/*@notnull@*/ ctbase c, ctuid kind1, ctuid kind2)
{
  ctuid ck = c->type;

  if (ck == kind1 || ck == kind2)
    return TRUE;

  if (ck == CT_CONJ)
    return (ctbase_isKind2 (ctype_getCtbaseSafe (c->contents.conj.a), kind1, kind2) ||
       ctbase_isKind2 (ctype_getCtbaseSafe (c->contents.conj.b), kind1, kind2));

  return FALSE;
}

static bool
ctbase_isAbstract (/*@notnull@*/ ctbase c)
{
  return (c->type == CT_ABST);
}

static bool ctbase_isUA (ctbase c) 
{
  return (ctbase_isDefined (c) && ((c)->type == CT_USER || (c)->type == CT_ABST));
}

