/* IluServiceThread.java */
/* Chris Jacobi, September 11, 1997 1:54 pm PDT */

/*
 * Copyright (c) 1997 Xerox Corporation.  All Rights Reserved.  
 * Unlimited use, reproduction, and distribution of this software is
 * permitted.  Any copy of this software must include both the above
 * copyright notice of Xerox Corporation and this paragraph.  Any
 * distribution of this software must comply with all applicable United
 * States export control laws.  This software is made available AS IS,
 * and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 * PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 * THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 * CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 * XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
 
/* $Id: IluServiceThread.java,v 1.4 1997/09/11 22:06:25 jacobi Exp $ */
 
package xerox.ilu;

/**
 * Subclass for certain ILU threads.<p>
 * This class is used to be able to transmit extra environment or thread
 * specific information between the ILU kernel and its application.<p>
 * 
 * Threads forked by ILU for serving skeleton methods do use this thread
 * class.
 * Applications optionally may use this thread class when requiring
 * service from ILU.  Application use is optional.
 *
 * @see java.lang.Thread
 */
 
public abstract class IluServiceThread extends java.lang.Thread {
    
    public IluServiceThread() {
        super("IluServiceThread");
    }
    
    public IluServiceThread(java.lang.String s) {
        super(s);
    } 
    
    
    /** 
     * Skeleton-features are set by ILU and queried 
     * by the implementation of skeleton methods (the callee). 
     */

    /*friendly*/ IluCall skeletonCall = null;

    /** 
     * Sets the call for a skeleton. 
     * Also cleans all stub features. 
     */
    /*friendly*/ void setSkeleton(IluCall sc) {
        stubPassport = null;
        stubPipeline = null;
        stubSerializationCtx = null;
        skeletonCall = sc;
    }


    /** 
     * Stub-features are set by the application (the caller) and 
     * queried by ILU. 
     */

    /*friendly*/ IluPipeline stubPipeline = null;
    /*friendly*/ IluSerializationContext stubSerializationCtx = null;
    private IluPassport stubPassport = null;

    /** 
     * Friendly: Thou shall not get the passport of a
     * thread which isn't yours. 
     */
    /*friendly*/ IluPassport getStubPassport() {
        return stubPassport;
    }

    /** 
     * Set the passport used when doing ILU calls 
     * queried by ILU. This method has been made static
     * for security so that a thread can only set its own 
     * stubPassport
     */
    public static void setCurrentPassport(IluPassport p) {
        java.lang.Thread thread = java.lang.Thread.currentThread();
        if (thread instanceof IluServiceThread) {
            ((IluServiceThread)thread).stubPassport = p;
        } else if (p != null) {
            throw new IluSomeSystemException("Not an ILU thread");
        }
    } //setCurrentPassport

    
    /** 
     * Sets the IluPipeline used when doing ILU calls
     * from this thread. <p> 
     */
    public static void setCurrentPipeline(IluPipeline p) {
        java.lang.Thread thread = java.lang.Thread.currentThread();
        if (thread instanceof IluServiceThread) {
            ((IluServiceThread)thread).stubPipeline = p;
        } else if (p != null) {
            throw new IluSomeSystemException("Not an ILU thread");
        }
    } //setCurrentPipeline


    /** 
     * Sets the IluSerializationContext used when doing ILU calls
     * from this thread. <p> 
     */
    public static void setCurrentSerializationContext(IluSerializationContext s) {
        java.lang.Thread thread = java.lang.Thread.currentThread();
        if (thread instanceof IluServiceThread) {
            ((IluServiceThread)thread).stubSerializationCtx = s;
        } else if (s != null) {
            throw new IluSomeSystemException("Not an ILU thread");
        }
    } //setCurrentSerializationContext


    /** 
     * Set the passport of an explicit thread.
     * Don't increase priviledges of another thread.
     * Not worried about simple denial of service.
     */
    public void setStubPassport(IluPassport p) {
        //Everybody may reduce priviledges; 
        //test on increases only
        if (p != null) {
            if (java.lang.Thread.currentThread() != this) {
                throw new java.lang.SecurityException();
            }
        }
    } //setStubPassport


    /** 
     * Reset all stub features. <p>
     * 
     * Clears Passport, Pipeline and SerializationContext, <p>
     * 
     * Security concern: This can be used from any thread,
     * as the risk is only a denial of service attack.<p>
     *
     * @see IluPassport#setCurrentPassport
     * @see IluPipeline#setCurrentPipeline
     * @see IluSerializationContext#setCurrentSerializationContext
     */
    final public void cleanAllStubFeatures() {
        stubPassport = null;
        stubPipeline = null;
        stubSerializationCtx = null;
    } //cleanAllStubFeatures
    
    
} //IluServiceThread


