// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _Renderer_h_
#define _Renderer_h_

#include <qobject.h>

#include "Url.h"

class Canvas;
class QPainter;

class Rect {
    int _x1;
    int _y1;
    int _x2;
    int _y2;
public:
    Rect();

    Rect( int x, int y, int w, int h );

    int x() const { return _x1; }

    int y() const { return _y1; }

    int width() const { return _x2 - _x1; }

    int height() const { return _y2 - _y1; }

    void moveBy( int dx, int dy ) { _x1 += dx; _y1 += dy; _x2 += dx; _y2 += dy; }

    void moveTopLeft( int x, int y ) { _x2 -= _x1 - x; _y2 -= _y1 - y; _x1 = x; _y1 = y; }

    void setSize( int w, int h ) { _x2 = _x1 + w; _y2 = _y1 + h; }

    void setRect( int x, int y, int w, int h ) { moveTopLeft( x, y ); setSize( w, h ); }

    bool intersects( const Rect& r ) const;

    Rect intersect( const Rect& r );

    bool contains( int x, int y ) const;
};

//: The Renderer class is the base class for anything that displays content in a browser window.
//. The Renderer specifies a minimal common interface for any class that displays
//. content (text, images, etc.) in a browser.  A Renderer keeps track of which
//. browser it is associated with, its preferred width, and its minimum and
//. maximum widths.
class Renderer : public QObject,
                 public Rect
{
    Q_OBJECT
    Canvas* _canvas;
protected:
    int     _clipWidth;
    int     _minimumWidth;
    int     _maximumWidth;
    bool    _needRedraw;
public:
    //. Initialize a new Renderer.  Just store the given information for later.
    Renderer( Canvas*     canvas,
              int         clipWidth=0,
              QObject*    parent=0,
              const char* name=0 );

    //. Virtual destructor.  This function does nothing.
    virtual ~Renderer();

    //. This function does the actual work of repositioning the content of the
    //. renderer.  This function is not called in response to paintEvent()s.
    virtual bool redraw();

    //. This function is called whenever the renderer may need to recompute the
    //. layout of its content.  Possible reasons for this are receiving new
    //. content, and processing a change in renderer's allocated screen width.
    //. This function is <B><EM>never</EM></B> called in response to a paintEvent().
    void needRedraw();

    //. Return the minimum width that the renderer needs to display its content.
    //. This value is calculated by a subclass, usually in the redraw() function.
    int minimumWidth() { return _minimumWidth; }

    //. Return the maximum width that the renderer needs to display its content.
    //. This value is calculated by a subclass, usually in the redraw() function.
    int maximumWidth() { return _maximumWidth; }

    //. Return a pointer to the browser that contains this renderer.
    Canvas* canvas() { return _canvas; }

    //. The current preferred width, as specified by our parent.
    int clipWidth() { return _clipWidth; }

    void resize( int w, int h );

    void move( int x, int y );

    virtual void repaint( QPainter& p, const Rect& r );
public slots:
    //. This slot informs the renderer that the preferred width has been
    //. changed.  This usually provokes a call to redraw().
    virtual void widthChanged( int w );

    //. This slot is used to notify a renderer that one of its children has
    //. resized itself.  Since the parent does not directly size its children,
    //. it must redraw() itself to cope with the new child size.
    virtual void childSizeChanged();
signals:
    //. This signal is emitted whenever a renderer resizes itself.  This signal
    //. is usually connected to the renderer's parent.
    void resized();
};

#endif
