/* cfengine for GNU
 
        Copyright (C) 1995
        Free Software Foundation, Inc.
 
   This file is part of GNU cfengine - written and maintained 
   by Mark Burgess, Dept of Computing and Engineering, Oslo College,
   Dept. of Theoretical physics, University of Oslo
 
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

*/
 

#include "cf.defs.h"
#include "cf.extern.h"


/*********************************************************************/

IsHomeDir(name)

      /* This assumes that the dir lies under mountpattern */

char *name;

{ char *sp;
  struct Item *ip;
  int slashes;
  
if (VMOUNTLIST == NULL)
   {
   return (false);
   } 

for (ip = VHOMEPATLIST; ip != NULL; ip=ip->next)
   {
   slashes = 0;

   for (sp = ip->name; *sp != '\0'; sp++)
      {
      if (*sp == '/')
	 {
	 slashes++;
	 }
      }
   
   for (sp = name+strlen(name); (*(sp-1) != '/') && (sp >= name) && (slashes >= 0); sp--)
      {
      if (*sp == '/')
	 {
	 slashes--;
	 }
      }

   /* Full comparison */
   
   if (WildMatch(ip->name,sp))
      {
      Debug("IsHomeDir(true)\n");
      return(true);
      }
   }

Debug("IsHomeDir(false)\n");
return(false);
}


/*********************************************************************/

EmptyDir(path)

char *path;

{ DIR *dirh;
  struct dirent *dirp;
  int count = 0;
  
Debug2("cfengine: EmptyDir(%s)\n",path);

if ((dirh = opendir(path)) == NULL)
   {
   sprintf(OUTPUT,"Can't open directory %s\n",path);
   CfLog(cferror,OUTPUT,"opendir");
   return true;
   }

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (!SensibleFile(dirp->d_name,path))
      {
      continue;
      }

   count++;
   }

closedir(dirh);

return (!count);
}

/*********************************************************************/

RecursiveCheck(name,plus,minus,action,uidlist,gidlist,recurse,rlevel,ptr)

char *name;
mode_t plus,minus;
struct UidList *uidlist;
struct GidList *gidlist;
enum fileactions action;
int recurse;
int rlevel;
struct File *ptr;

{ DIR *dirh;
  struct dirent *dirp;
  char pcwd[bufsize];
  struct stat statbuf;

if (recurse == -1)
   {
   return;
   }

Debug("RecursiveCheck(%s,+%o,-%o)\n",name,plus,minus);

if ((dirh = opendir(name)) == NULL)
   {
   if (lstat(name,&statbuf) != -1)
      {
      CheckExistingFile(name,plus,minus,action,uidlist,gidlist,&statbuf,ptr,ptr->acl_aliases);
      }
   return;
   }

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (!SensibleFile(dirp->d_name,name))
      {
      continue;
      }

   if (IgnoreFile(name,dirp->d_name))
      {
      continue;
      }

   strcpy(pcwd,name);                                   /* Assemble pathname */
   AddSlash(pcwd);

   if (BufferOverflow(pcwd,dirp->d_name))
      {
      return;
      }

   strcat(pcwd,dirp->d_name);

   
   if (TRAVLINKS)
      {
      if (stat(pcwd,&statbuf) == -1)
         {
         sprintf(OUTPUT,"RecursiveCheck was working on %s when this happened:\n",pcwd);
	 CfLog(cferror,OUTPUT,"stat");
         continue;
         }
      }
   else
      {
      if (lstat(pcwd,&statbuf) == -1)
         {
         sprintf(OUTPUT,"RecursiveCheck was working in %s when this happened:\n",pcwd);
         CfLog(cferror,OUTPUT,"lstat");
         continue;
         }
      }

   if (S_ISLNK(statbuf.st_mode))            /* should we ignore links? */
      {
      CheckExistingFile(pcwd,plus,minus,action,uidlist,gidlist,&statbuf,ptr,ptr->acl_aliases);
      continue;
      }

   if (S_ISDIR(statbuf.st_mode))
      {
      Debug("File is a directory\n");
      
      if (IsMountedFileSystem(&statbuf,pcwd,rlevel))
         {
         continue;
         }
      else
         {
         if (ptr->recurse > 1)  /* avoid infinite regression */
            {
            CheckExistingFile(pcwd,plus,minus,action,uidlist,gidlist,&statbuf,ptr,ptr->acl_aliases);
            RecursiveCheck(pcwd,plus,minus,action,uidlist,gidlist,recurse-1,rlevel+1,ptr);
            }
         else
            {
            CheckExistingFile(pcwd,plus,minus,action,uidlist,gidlist,&statbuf,ptr,ptr->acl_aliases);
            }
         }
      }
   else
      {
      CheckExistingFile(pcwd,plus,minus,action,uidlist,gidlist,&statbuf,ptr,ptr->acl_aliases);
      }
   }

closedir(dirh);
}


/*********************************************************************/

CheckExistingFile(file,plus,minus,action,uidlist,gidlist,dstat,ptr,acl_aliases)

char *file;
mode_t plus,minus;
struct UidList *uidlist;
struct GidList *gidlist;
enum fileactions action;
struct stat *dstat;
struct File *ptr;
struct Item *acl_aliases;

{ mode_t newperm = dstat->st_mode;
  int amroot = true;

Verbose("%s: Checking fs-object %s\n",VPREFIX,file);
Debug("CheckExistingFile(+%o,-%o)\n",plus,minus); 

if (ptr != NULL)
   {
   if (IgnoredOrExcluded(files,file,ptr->inclusions,ptr->exclusions))
      {
      Verbose("Skipping excluded file %s\n",file);
      return;
      }
   }

if (getuid() != 0)                            
   {
   amroot = false;
   }

 /* directories must have x set if r set, regardless  */


newperm = (dstat->st_mode & 07777) ;
newperm |= plus;
newperm &= ~minus;

if (S_ISREG(dstat->st_mode) && (action == fixdirs || action == warndirs)) 
   {
   Debug("Regular file, returning...\n");
   return;
   }

if (S_ISDIR(dstat->st_mode))  
   {
   if (action == fixplain || action == warnplain)
      {
      return;
      }

   Debug("Directory...fixing x bits\n");

   if (newperm & S_IRUSR)
      {
      newperm  |= S_IXUSR;
      }

   if (newperm & S_IRGRP)
      {
      newperm |= S_IXGRP;
      }

   if (newperm & S_IROTH)
      {
      newperm |= S_IXOTH;
      }
   }

if (dstat->st_uid == 0 && (dstat->st_mode & S_ISUID))
   {
   if (newperm & S_ISUID)
      {
      if (! IsItemIn(VSETUIDLIST,file))
         {
         if (amroot)
	     {
	     sprintf(OUTPUT,"NEW SETUID root PROGRAM %s\n",file);
	     CfLog(cfinform,OUTPUT,"");
	     }
         PrependItem(&VSETUIDLIST,file,NULL);
         }
      }
   else
      {
      switch (action)
         {
         case fixall:
         case fixdirs:
         case fixplain:  sprintf(OUTPUT,"Removing setuid (root) flag from %s...\n\n",file);
	                 CfLog(cfinform,OUTPUT,"");
			 
	                 if (ptr != NULL)
                            {
                            AddMultipleClasses(ptr->defines);
                            }
                         break;
         case warnall:
         case warndirs:
         case warnplain: if (amroot)
	                    {
      	                    sprintf(OUTPUT,"WARNING setuid (root) flag on %s...\n\n",file);
			    CfLog(cferror,OUTPUT,"");
	                    }
                         break;

         default:        break;
         }
      }
   }

if (dstat->st_uid == 0 && (dstat->st_mode & S_ISGID))
   {
   if (newperm & S_ISGID)
      {
      if (! IsItemIn(VSETUIDLIST,file))
         {
         if (S_ISDIR(dstat->st_mode))
            {
            /* setgid directory */
            }
         else
            {
            if (amroot)
		{
		sprintf(OUTPUT,"NEW SETGID root PROGRAM %s\n",file);
		CfLog(cferror,OUTPUT,"");
		}
            PrependItem(&VSETUIDLIST,file,NULL);
            }
         }
      }
   else
      {
      switch (action)
         {
         case fixall:
         case fixdirs:
         case fixplain: sprintf(OUTPUT,"Removing setgid (root) flag from %s...\n\n",file);
	                CfLog(cfinform,OUTPUT,"");
	                if (ptr != NULL)
			   {
                           AddMultipleClasses(ptr->defines);
			   }
	                break;
         case warnall:
         case warndirs:
         case warnplain: sprintf(OUTPUT,"WARNING setgid (root) flag on %s...\n\n",file);
	                 CfLog(cferror,OUTPUT,"");
                         break;

         default:        break;
         }
      }
   }

if (CheckOwner(file,action,uidlist,gidlist,dstat))
   {
   if (ptr != NULL)
      {
      AddMultipleClasses(ptr->defines);
      }
   }

if (S_ISLNK(dstat->st_mode))             /* No point in checking permission on a link */
   {
   KillOldLink(file);

   if (ptr != NULL)
      {
      AddMultipleClasses(ptr->defines);
      }
   return;
   }

if (stat(file,dstat) == -1)
   {
   sprintf(OUTPUT,"Can't stat file %s while looking at permissions (was not copied?)\n",file);
   CfLog(cfverbose,OUTPUT,"stat");
   return;
   }

if (CheckACLs(file,action,acl_aliases))
   {
   if (ptr != NULL)
      {
      AddMultipleClasses(ptr->defines);
      }
   }
 
if (((newperm & 07777) == (dstat->st_mode & 07777)) && (action != touch))    /* file okay */
   {
   Debug("File okay, newperm = %o, stat = %o\n",(newperm & 07777),(dstat->st_mode & 07777));
   return;
   }

Debug("Trying to fix...\n"); 

switch (action)
   {
   case linkchildren:

   case warnplain:
                if (S_ISREG(dstat->st_mode))
                   {
                   sprintf(OUTPUT,"%s has permission %o\n",file,dstat->st_mode & 07777);
		   CfLog(cferror,OUTPUT,"");
                   sprintf(OUTPUT,"[should be %o]\n",newperm & 07777);
		   CfLog(cferror,OUTPUT,"");
                   }
                break;
   case warndirs:
                if (S_ISDIR(dstat->st_mode))
                   {
                   sprintf(OUTPUT,"%s has permission %o\n",file,dstat->st_mode & 07777);
		   CfLog(cferror,OUTPUT,"");
                   sprintf(OUTPUT,"[should be %o]\n",newperm & 07777);
		   CfLog(cferror,OUTPUT,"");
                   }
                break;
   case warnall:   
                sprintf(OUTPUT,"%s has permission %o\n",file,dstat->st_mode & 07777);
		CfLog(cferror,OUTPUT,"");
                sprintf(OUTPUT,"[should be %o]\n",newperm & 07777);
		CfLog(cferror,OUTPUT,"");
                break;

   case fixplain:

                if (S_ISREG(dstat->st_mode))
                   {
                   if (! DONTDO)
                      {
                      if (chmod (file,newperm & 07777) == -1)
			 {
                         sprintf(OUTPUT,"chmod failed on %s\n",file);
                         CfLog(cferror,OUTPUT,"chmod");
                         break;
                         }
                      }

                   sprintf(OUTPUT,"%s had permission %o, changed it to %o\n",
			   file,dstat->st_mode & 07777,newperm & 07777);
		   CfLog(cfinform,OUTPUT,"");
		   
                   if (ptr != NULL)
		      {
		      AddMultipleClasses(ptr->defines);
		      }
                   }
                break;

   case fixdirs:
                if (S_ISDIR(dstat->st_mode))
                   {
                   if (! DONTDO)
                      {
                      if (chmod (file,newperm & 07777) == -1)
                         {
                         sprintf(OUTPUT,"chmod failed on %s\n",file);
			 CfLog(cferror,OUTPUT,"chmod");
                         break;
                         }
                      }

                   sprintf(OUTPUT,"%s had permission %o, changed it to %o\n",
			   file,dstat->st_mode & 07777,newperm & 07777);
		   CfLog(cfinform,OUTPUT,"");
		   
                   if (ptr != NULL)
		      {
		      AddMultipleClasses(ptr->defines);
		      }
                   }
                break;

   case fixall: if (! DONTDO)
                   {
                   if (chmod (file,newperm & 07777) == -1)
		      {
                      sprintf(OUTPUT,"chmod failed on %s\n",file);
                      CfLog(cferror,OUTPUT,"chmod");
                      break;
                      }
                   }

                   sprintf(OUTPUT,"%s had permission %o, changed it to %o\n",
			   file,dstat->st_mode & 07777,newperm & 07777);
		   CfLog(cfinform,OUTPUT,"");

                   if (ptr != NULL)
		      {
                      AddMultipleClasses(ptr->defines);
		      }
                break;

   case touch:  if (! DONTDO)
                   {
                   if (chmod (file,newperm & 07777) == -1)
		      {
                      sprintf(OUTPUT,"chmod failed on %s\n",file);
                      CfLog(cferror,OUTPUT,"chmod");
                      break;
		      }
                   utime (file,NULL);
                   }
                if (ptr != NULL)
		   {
                   AddMultipleClasses(ptr->defines);
		   }
                break;

   default:     FatalError("cfengine: internal error CheckExistingFile(): illegal file action\n");
   }

Debug("CheckExistingFile(Done)\n"); 
}

/*********************************************************************/

CheckCopiedFile(file,plus,minus,action,uidlist,gidlist,dstat,sstat,ptr,acl_aliases)

char *file;
mode_t plus,minus;
struct UidList *uidlist;
struct GidList *gidlist;
enum fileactions action;
struct stat *dstat;
struct stat *sstat;
struct File *ptr;
struct Item *acl_aliases;

{ mode_t newplus,newminus;

 /* plus/minus must be relative to source file, not to
    perms of newly created file! */

Debug("CheckCopiedFile(%s)\n",file); 

if ((plus == 0) && (minus == 0))
    {
    newplus = sstat->st_mode & 07777 | plus;
    newminus = ~(sstat->st_mode & 07777 & ~minus) & 07777;
    CheckExistingFile(file,newplus,newminus,fixall,uidlist,gidlist,dstat,NULL,acl_aliases);
    }
 else
    {
    CheckExistingFile(file,plus,minus,fixall,uidlist,gidlist,dstat,NULL,acl_aliases);
    }
}

/*********************************************************************/

CheckOwner(file,action,uidlist,gidlist,statbuf)

char *file;
enum fileactions action;
struct UidList *uidlist;
struct GidList *gidlist;
struct stat *statbuf;

{ struct passwd *pw;
  struct group *gp;
  struct UidList *ulp;
  struct GidList *glp;
  short uidmatch = false, gidmatch = false;
  int uid = -1; 
  int gid = -1;

Debug("CheckOwner: %d\n",statbuf->st_uid);
  
for (ulp = uidlist; ulp != NULL; ulp=ulp->next)
   {
   Debug(" uid %d\n",ulp->uid);

   if (ulp->uid == -1 || statbuf->st_uid == ulp->uid)   /* -1 matches anything */
      {
      uid = ulp->uid;
      uidmatch = true;
      break;
      }
   }

for (glp = gidlist; glp != NULL; glp=glp->next)
   {
   if (glp->gid == -1 || statbuf->st_gid == glp->gid)  /* -1 matches anything */
      {
      gid = glp->gid;
      gidmatch = true;
      break;
      }
   }


if (uidmatch && gidmatch)
   {
   return false;
   }
else
   {
   if ((! uidmatch) && (uidlist != NULL))
      {
      uid = uidlist->uid;    /* default is first item in list */
      }

   if ((! gidmatch) && (gidlist != NULL))
      {
      gid = gidlist->gid;
      }

   if (S_ISLNK(statbuf->st_mode) && (action == fixdirs || action == fixplain))
      {
      Debug2("File %s incorrect type (link), skipping...\n",file);
      return false;
      }

   if ((S_ISREG(statbuf->st_mode) && action == fixdirs) || (S_ISDIR(statbuf->st_mode) && action == fixplain))
      {
      Debug2("File %s incorrect type, skipping...\n",file);
      return false;
      }

   switch (action)
      {
      case fixplain:
      case fixdirs:
      case fixall: 
      case touch:
                  if (VERBOSE || DEBUG || D2)
                     {
                     if (uid == -1 && gid == -1)
                        {
                        printf("%s:   touching %s\n",VPREFIX,file);
                        }
                     else
                        {
                        if (uid != -1)
                           {
                           sprintf(OUTPUT,"  (Changing owner to uid %d)\n",uid);
                           CfLog(cfinform,OUTPUT,"");
                           }

                        if (gid != -1)
                           {
                           sprintf(OUTPUT,"  (Changing group to gid %d)\n",gid);
                           CfLog(cfinform,OUTPUT,"");
                           }
                        }
                     }

                  if (! DONTDO && S_ISLNK(statbuf->st_mode))
                     {
#ifdef HAVE_LCHOWN
		     Debug("Using LCHOWN function\n");
                     if (lchown(file,uid,gid) == -1)
                        {
                        sprintf(OUTPUT,"Cannot set ownership on link %s!\n",file);
                        CfLog(cferror,OUTPUT,"lchown");
                        }
		     else
			{
			return true;
			}
#endif
                     }
                  else if (! DONTDO)
                     {
                     if (chown(file,uid,gid) == -1)
                        {
                        sprintf(OUTPUT,"Cannot set ownership on file %s!\n",file);
                        CfLog(cferror,OUTPUT,"chown");
                        }
		     else
			{
			return true;
			}
                     }
                  break;

      case linkchildren:
      case warnall: 
      case warndirs:
      case warnplain:
                  if ((pw = getpwuid(statbuf->st_uid)) == NULL)
                     {
                     sprintf(OUTPUT,"File %s is not owned by anybody in the passwd database\n",file);
		     CfLog(cferror,OUTPUT,"");
                     sprintf(OUTPUT,"(uid = %d,gid = %d)\n",statbuf->st_uid,statbuf->st_gid);
		     CfLog(cferror,OUTPUT,"");
                     break;
                     }

                  if ((gp = getgrgid(statbuf->st_gid)) == NULL)
                     {
                     sprintf(OUTPUT,"File %s is not owned by any group in group database\n",file);
		     CfLog(cferror,OUTPUT,"");
                     break;
                     }

                  sprintf(OUTPUT,"File %s is owned by [%s], group [%s]\n",file,pw->pw_name,gp->gr_name);
		  CfLog(cferror,OUTPUT,"");
                  break;
      }
   }
return false; 
}


/*********************************************************************/

CheckHomeSubDir(testpath,tidypath,recurse)

char *testpath, *tidypath;
int recurse;

{ char *sp, *subdirstart, *sp1, *sp2;
  char buffer[bufsize];
  int homelen;

if (strncmp(tidypath,"home/",5) == 0)
   {
   strcpy(buffer,testpath);

   for (ChopLastNode(buffer); strlen(buffer) != 0; ChopLastNode(buffer))
     {
     if (IsHomeDir(buffer))
        {
        break;
        }
     }

   homelen = strlen(buffer);

   if (homelen == 0)   /* No homedir */
      {
      return false;
      }

   Debug2("CheckHomeSubDir(%s,%s)\n",testpath,tidypath);

   subdirstart = tidypath + 4;                                   /* Ptr to start of subdir */

   strcpy(buffer,testpath);

   ChopLastNode(buffer);                                         /* Filename only */

   for (sp1 = buffer + homelen +1; *sp1 != '/' && *sp1 != '\0'; sp1++) /* skip user name dir */
      {
      }

   sp2 = subdirstart;

   if (strncmp(sp1,sp2,strlen(sp2)) != 0)
      {
      return false;
      } 

   Debug2("CheckHomeSubDir(true)\n");
   return(true);
   }

return true;
}

/**************************************************************/

FileIsNewer(file1,file2)

/* True if file2 is newer than file 1 */

char *file1, *file2;

{ struct stat statbuf1, statbuf2;

if (stat(file2,&statbuf2) == -1)
   {
   CfLog(cferror,"","stat");
   return false;
   }

if (stat(file1,&statbuf1) == -1)
   {
   CfLog(cferror,"","stat");
   return true;
   }

if (statbuf1.st_mtime < statbuf2.st_mtime)
   {
   return true;
   }
else
   {
   return false;
   }
}


/*********************************************************************/
/* Ignore is used by files and tidy modules                          */
/*********************************************************************/

IgnoreFile (pathto,name)

char *pathto, *name;

{ struct Item *ip;

if (name == NULL || strlen(name) == 0)
   {
   return false;
   }
 
strcpy(VBUFF,pathto);
AddSlash(VBUFF);
strcat(VBUFF,name); 

for (ip = VIGNORE; ip != NULL; ip=ip->next)
   {
   if (IsExcluded(ip->classes))
      {
      continue;
      }

   if (*(ip->name) == '/')
      {
      if (strcmp(VBUFF,ip->name) == 0)
         {
         Debug("cfengine: Ignoring abs path [%s][%s]\n",pathto,name);
         return true;
         }
      }
   else
      {
      if (WildMatch(ip->name,name))
         {
         Debug("cfengine: Ignoring pattern [%s][%s]\n",pathto,name);
         return true;
         }
      }
   }

return false;
}



