#include <config.h>
#include <stdio.h>
#include <string.h>
#include <pwd.h>

#include <glibtop.h>
#include <glibtop/xmalloc.h>
#include <glibtop/union.h>

#ifdef GLIBTOP_INODEDB
#include <glibtop/inodedb.h>
#endif

#include <gnome.h>

#include "details.h"
#include "procview.h"
#include "graph.h"

#define NUM_INFO 15
#define MAP_COLS 7

GtkWidget *dwin = NULL;

static GtkWidget *nb;

static GtkWidget *icl = NULL;

static GtkWidget *gswin = NULL;
static Graph *gg;

static GtkWidget *mcl = NULL;

static ProcMapRow **rows = NULL;

static void display_details (gint pid);

static void add_info (gint pid);
static void add_mem_map (gint pid);
static void add_mem_graph (gint pid);

static int  delete_event_handler (GtkWidget *widget, gpointer gp);

void
procview_details (gint pid)
{
	GtkWidget *vb;
	gchar *title;
	gchar *form;

	form = _("Process %d details");
	title = g_new (char, strlen (form)+64);
	sprintf (title, form, pid);

	if (!dwin) {
		dwin = gnome_dialog_new(title,
					GNOME_STOCK_BUTTON_CLOSE,
					NULL);;
		gtk_widget_set_name (dwin, "GTopDetails");
		
		vb = GNOME_DIALOG(dwin)->vbox;
		nb = gtk_notebook_new ();

		gtk_widget_set_usize (dwin, 780, 360);
		gtk_window_set_policy(GTK_WINDOW(dwin), TRUE, TRUE, FALSE);

		gtk_box_pack_start (GTK_BOX (vb), nb, TRUE, TRUE, GNOME_PAD_SMALL);

		gtk_signal_connect (GTK_OBJECT (dwin), "delete_event",
				    GTK_SIGNAL_FUNC (delete_event_handler), NULL);
		gnome_dialog_button_connect (GNOME_DIALOG (dwin), 0,
					     GTK_SIGNAL_FUNC (delete_event_handler), NULL);

		gtk_widget_show (nb);
	}
	else {
	        gtk_window_set_title (GTK_WINDOW (dwin), title);
	}
	g_free (title);

	display_details (pid);

	/* gdk_window_raise (dwin->window); */
}

static int
delete_event_handler (GtkWidget *widget, gpointer gp)
{
	/* printf ("details: delete_event\n"); */

        gtk_widget_destroy (dwin);

	dwin = icl = mcl = gswin = NULL;

        return TRUE;
}

static void
display_details (gint pid)
{
	add_info (pid);
	add_mem_map (pid);
	add_mem_graph (pid);
	if (dwin) {
	  /* This has no effect if we've already shown. */
	  gtk_widget_show(dwin);
        }
}

static void
set_info (gint pid)
{
	ProcProcData d;
	proc_t *t = NULL;
	proc_field *f = p_fields;
	gint i;
	gchar *v;
	float old_adj_hval = GTK_SCROLLBAR (GTK_CLIST (icl)->hscrollbar)
		->range.adjustment->value;
	float old_adj_vval = GTK_SCROLLBAR (GTK_CLIST (icl)->vscrollbar)
		->range.adjustment->value;
	gint old_hoffset = GTK_CLIST (icl)->hoffset;
	gint old_voffset = GTK_CLIST (icl)->voffset;
	glibtop_proc_state procstate;
	glibtop_proc_time proctime;
	glibtop_proc_mem procmem;
	glibtop_proc_uid procuid;
	struct passwd *pwd;

	memset (&d._p, 0, sizeof (d._p));
	t = d.p = &d._p;
	
	t->pid = pid;
	
	glibtop_get_proc_state (&procstate, t->pid);
	
	t->cmd = glibtop_strdup (procstate.cmd);
	
	t->state = procstate.state;
	
	pwd = getpwuid (procstate.uid);
	if (pwd)
		t->user = glibtop_strdup (pwd->pw_name);
	else
		t->user = glibtop_strdup (_("<unknown>"));
	
	glibtop_get_proc_uid (&procuid, t->pid);
	
	t->nice     = procuid.nice;
	t->priority = procuid.priority;
	
	glibtop_get_proc_mem (&procmem, t->pid);

	t->size     = (unsigned long) procmem.size >> 10;
	t->rss      = (unsigned long) procmem.rss >> 10;
	t->resident = (unsigned long) procmem.resident >> 10;
	t->share    = (unsigned long) procmem.share >> 10;
	
	glibtop_get_proc_time (&proctime, t->pid);

	t->hz = proctime.frequency ? proctime.frequency : 1000000;
	
	t->utime  = ((unsigned long) (proctime.utime * 100) / t->hz);
	t->stime  = ((unsigned long) (proctime.stime * 100) / t->hz);
	t->cutime = ((unsigned long) (proctime.cutime * 100) / t->hz);
	t->cstime = ((unsigned long) (proctime.cstime * 100) / t->hz);
	
	t->start_time =
		((unsigned long) (proctime.start_time * 100) / t->hz);

	/* FIXME! */
	d.pcpu = d.pmem = 0;

	gtk_clist_freeze (GTK_CLIST (icl));
	for (i = 0; i < NUM_INFO; i++) {
		if (f->label) {
			v = (t) ? sprint_fmt (&d, f->fmt) : "";
			gtk_clist_set_text (GTK_CLIST (icl),
					    i, 1, v);
			/* printf ("clist set %s\n", v); */
			if (t)
				g_free (v);
			
			f++;
		}
	}

	/* keep old view position - somewhat ughly, but works for now */
	GTK_SCROLLBAR (GTK_CLIST (icl)->hscrollbar)
		->range.adjustment->value = old_adj_hval;
	GTK_SCROLLBAR (GTK_CLIST (icl)->vscrollbar)
		->range.adjustment->value = old_adj_vval;

	GTK_CLIST (icl)->hoffset = old_hoffset;
	GTK_CLIST (icl)->voffset = old_voffset;

	gtk_clist_thaw (GTK_CLIST (icl));
}

static void
add_info (gint pid)
{
	if (!icl) {
		GtkWidget *l;
		gint i;
		gchar *t [3];

		l = gtk_label_new (_("Process info"));

		t [0] = _("Attribute");
		t [1] = _("Value");
		t [2] = "";
		icl = gtk_clist_new_with_titles (3, t);

		gtk_clist_column_titles_show (GTK_CLIST (icl));
		gtk_clist_set_column_justification (GTK_CLIST (icl),
						    0, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (icl),
						    1, GTK_JUSTIFY_RIGHT);

		for (i = 0; i < NUM_INFO; i++) {
			t [0] = _(p_fields [i].long_info);
			t [1] = t [2] = "";
			
			gtk_clist_append (GTK_CLIST (icl), t);
			/* printf ("clist set %s\n", p_fields [i].long_info); */
		}

		gtk_clist_set_policy (GTK_CLIST (icl),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);

		gtk_clist_set_column_width (GTK_CLIST (icl), 0, 320);
		gtk_clist_set_column_width (GTK_CLIST (icl), 1, 140);

		gtk_container_border_width (GTK_CONTAINER (icl), GNOME_PAD_SMALL);

		gtk_widget_show (icl);
		gtk_widget_show (l);

		gtk_notebook_append_page (GTK_NOTEBOOK (nb), icl, l);
	}

	set_info (pid);
}

static ProcMapRow **
get_map_rows (pid_t pid, ProcMapRow **rs)
{
	ProcMapRow *row;
	ProcMapRow **rows = NULL;
#ifdef GLIBTOP_INODEDB
	static glibtop_inodedb *inodedb = NULL;
#endif
	GList *c, *list = NULL;
	gchar fn [4096];
	gint i;

	glibtop_proc_map procmap;
	glibtop_map_entry *maps;

#ifdef GLIBTOP_INODEDB
	if (!inodedb)
		inodedb = glibtop_inodedb_open (0, 0);
#endif

	/* free old rows */

	if (rs) {
		ProcMapRow **c = rs;
		/* i = 0; */
		while (*c) {
			if ((*c)->filename)
				g_free ((*c)->filename);
			g_free (*c);
			c++;
			/* i++; */
		}

		/* printf ("freed: %d\n", i); */
		g_free (rs);
	}

	/* now we sample new ones */

	maps = glibtop_get_proc_map (&procmap, pid);

	if (maps) {
		for (i = 0; i < procmap.number; i++) {
			unsigned perm = maps [i].perm;
			const char *filename = NULL;

			row = g_new (ProcMapRow, 1);
			list = g_list_append (list, row);

			memset (row, 0, sizeof (ProcMapRow));

			row->VMstart = maps [i].start;
			row->VMend = maps [i].end;
			row->VMoffset = maps [i].offset;
			row->dev_minor = maps [i].device & 255;
			row->dev_major = (maps [i].device >> 8) & 255;
			row->inode = maps [i].inode;

			row->flags [0] =
				(perm & GLIBTOP_MAP_PERM_READ) ? 'r' : '-';
			row->flags [1] =
				(perm & GLIBTOP_MAP_PERM_WRITE) ? 'w' : '-';
			row->flags [2] =
				(perm & GLIBTOP_MAP_PERM_EXECUTE) ? 'x' : '-';
			row->flags [3] =
				(perm & GLIBTOP_MAP_PERM_SHARED) ? 's' : '-';
			if (perm & GLIBTOP_MAP_PERM_PRIVATE)
				row->flags [3] = 'p';

			row->flags [4] = 0;

			if (maps [i].flags & (1 << GLIBTOP_MAP_ENTRY_FILENAME))
				filename = glibtop_strdup (maps [i].filename);
			
#ifdef GLIBTOP_INODEDB
			if (inodedb && !filename)
				filename = glibtop_inodedb_lookup
					(inodedb, maps [i].device,
					 maps [i].inode);
#endif

			row->filename = filename;
		}

		rows = g_new (ProcMapRow *, procmap.number+1);
		rows [procmap.number] = NULL;

		for (i = 0, c = list; i < procmap.number; i++) {
			rows [i] = c->data;
			c = c->next;

			/* row = rows [i];
			   printf ("%08lx-%08lx %s\n",
			   row->VMstart, row->VMend,
			   row->filename); */
		}
		g_list_free (list);
	}

	glibtop_free (maps);

	return rows;
}

static void
set_mem_map (ProcMapRow **rows)
{
	gchar *t [MAP_COLS];
	gchar buf [6][64];
	float old_adj_hval = GTK_SCROLLBAR (GTK_CLIST (mcl)->hscrollbar)
		->range.adjustment->value;
	float old_adj_vval = GTK_SCROLLBAR (GTK_CLIST (mcl)->vscrollbar)
		->range.adjustment->value;
	gint old_hoffset = GTK_CLIST (mcl)->hoffset;
	gint old_voffset = GTK_CLIST (mcl)->voffset;

	/* printf ("set mem map\n"); */

	gtk_clist_freeze (GTK_CLIST (mcl));
	gtk_clist_clear (GTK_CLIST (mcl));

	if (rows)
		for (; *rows; rows++) {
			sprintf (buf [0], "%08lx", (*rows)->VMstart);
			t [0] = buf [0];
			sprintf (buf [1], "%08lx", (*rows)->VMend);
			t [1] = buf [1];
			t [2] = (*rows)->flags;
			sprintf (buf [2], "%08lx", (*rows)->VMoffset);
			t [3] = buf [2];
			sprintf (buf [3], "%02hx:%02hx",
				 (*rows)->dev_major,
				 (*rows)->dev_minor);
			t [4] = buf [3];
			sprintf (buf [4], "%ld", (*rows)->inode);
			t [5] = buf [4];
			t [6] = (*rows)->filename;
			
			gtk_clist_append (GTK_CLIST (mcl), t);
		}
	
	/* keep old view position - somewhat ughly, but works for now */
	GTK_SCROLLBAR (GTK_CLIST (mcl)->hscrollbar)
		->range.adjustment->value = old_adj_hval;
	GTK_SCROLLBAR (GTK_CLIST (mcl)->vscrollbar)
		->range.adjustment->value = old_adj_vval;

	GTK_CLIST (mcl)->hoffset = old_hoffset;
	GTK_CLIST (mcl)->voffset = old_voffset;

	gtk_clist_thaw (GTK_CLIST (mcl));
}

static void
add_mem_map (gint pid)
{
	if (!mcl) {
		GtkWidget *l;
		gchar *t [MAP_COLS];

		l = gtk_label_new (_("Raw memory map"));

		t [0] = _("VM start");
		t [1] = _("VM end");
		t [2] = _("Flags");
		t [3] = _("VM offset");
		t [4] = _("Device");
		t [5] = _("Inode");
		t [6] = _("Filename");
		mcl = gtk_clist_new_with_titles (MAP_COLS, t);

		gtk_clist_set_policy (GTK_CLIST (mcl),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);

		gtk_clist_set_column_width (GTK_CLIST (mcl), 0, 90);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 1, 90);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 2, 50);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 3, 90);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 4, 60);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 5, 90);
		gtk_clist_set_column_width (GTK_CLIST (mcl), 6, 200);

		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    0, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    1, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    2, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    3, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    4, GTK_JUSTIFY_RIGHT);
		gtk_clist_set_column_justification (GTK_CLIST (mcl),
						    5, GTK_JUSTIFY_RIGHT);

		gtk_container_border_width (GTK_CONTAINER (mcl), GNOME_PAD_SMALL);

		gtk_widget_show (mcl);
		gtk_widget_show (l);

		gtk_notebook_append_page (GTK_NOTEBOOK (nb), mcl, l);		
	}

	rows = get_map_rows (pid, rows);

	set_mem_map (rows);
}

static gchar *
mem_seg (ProcMapRow **rs)
{
	ProcMapRow *r = *rs;
	gchar *f = r->flags;

	if (r->filename) {
		if (!strcmp (f, "rw-p"))
			return _("Data");
		else if (!strcmp (f, "r-xp"))
			return _("Text");
		else if (f[0] == 'r')
			return (f[1]=='w') ? "RW" : "RO";

	} else {
		if (!strcmp (f, "rwxp") && !*(rs+1))
			return _("Stack");
		if (!strcmp (f, "---p"))
			return _("Hole");
		else if (f[0]=='r')
			return (f[1]=='w') ? "RW" : "RO";
	}

	return "?????";
}

gpointer
mem_graph_data_fn (GraphCmd cmd, gpointer data)
{
	ProcMapRow **rs = data;
	static gchar buf [256];
	char *fname;

	switch (cmd) {
	case GRAPH_FIRST:
		return (rows) ? (*rows) ? rows : NULL : NULL;
	case GRAPH_NEXT:
		return (*(rs+1)) ? rs+1 : NULL;
	case GRAPH_VALUE:
		return (gpointer)((*rs)->VMend - (*rs)->VMstart);
	case GRAPH_LABEL:
		if ((*rs)->filename) {
			fname = strrchr ((*rs)->filename, '/');
			if (fname)
				fname++;
			else
				fname = (*rs)->filename;
		} else
			fname = "";

		sprintf (buf, "(%5s)\t: %8ldk  %s",
			 mem_seg (rs),
			 ((*rs)->VMend - (*rs)->VMstart) >> 10,
			 fname);
		
		return buf;
	default:
		return NULL;
	}
}

static void
add_mem_graph (gint pid)
{
	if (!gswin) {
		GtkWidget *l;
		GtkWidget *da;
		GtkWidget *f;

		l = gtk_label_new (_("Graphical memory map"));
		gswin = gtk_scrolled_window_new (NULL, NULL);
		gtk_widget_set_name (gswin, "GraphMap");
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (gswin),
						GTK_POLICY_AUTOMATIC,
						GTK_POLICY_AUTOMATIC);

		f = gtk_frame_new (NULL);
		gtk_frame_set_shadow_type (GTK_FRAME (f), GTK_SHADOW_NONE);
		gg = graph_new (mem_graph_data_fn);
		da = graph_widget (gg);
		graph_colors_set (gg, graph_default_colors, GRAPH_DEFAULT_COLORS);

		gtk_container_border_width (GTK_CONTAINER (gswin), GNOME_PAD_SMALL);
		gtk_container_border_width (GTK_CONTAINER (f), GNOME_PAD_SMALL << 1);

		gtk_container_add (GTK_CONTAINER (f), da);
		gtk_container_add (GTK_CONTAINER (gswin), f);

		gtk_widget_show (l);
		gtk_widget_show (da);
		gtk_widget_show (f);
		gtk_widget_show (gswin);

		gtk_notebook_append_page (GTK_NOTEBOOK (nb), gswin, l);		
	}

	graph_update (gg);
	gtk_widget_queue_resize (gswin);
}
