/*
 *	cook - file construction tool
 *	Copyright (C) 1997 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to print dependency graphs as a shell script
 */

#include <ac/stdio.h>

#include <dir_part.h>
#include <error_intl.h>
#include <graph/file.h>
#include <graph/file_list.h>
#include <graph/recipe.h>
#include <graph/script.h>
#include <id.h>
#include <id/variable.h>
#include <match.h>
#include <opcode/context.h>
#include <option.h>
#include <os.h>
#include <quote.h>
#include <recipe.h>
#include <star.h>
#include <stmt.h>
#include <str_list.h>
#include <trace.h>


/*
 * NAME
 *	graph_recipe_script
 *
 * SYNOPSIS
 *	graph_walk_status_ty graph_recipe_script(graph_recipe_ty *);
 *
 * DESCRIPTION
 *	The graph_recipe_script function is used to print a shell script
 *	fragment on the standard output which approximates this recipe
 *	instance.
 *
 * RETURNS
 *	graph_walk_status_ty
 *		error		something went wrong
 *		uptodate	sucecss
 */

graph_walk_status_ty
graph_recipe_script(grp, gp)
	graph_recipe_ty	*grp;
	struct graph_ty	*gp;
{
	graph_walk_status_ty status;
	size_t		j;
	string_list_ty	wl;
	int		forced;

	trace(("graph_recipe_script(grp = %08lX)\n{\n"/*}*/, (long)grp));
	status = graph_walk_status_done;

	/*
	 * construct the ``target'' variable
	 */
	string_list_constructor(&wl);
	assert(grp->output);
	assert(grp->output->nfiles > 0);
	if (grp->output->nfiles > 0)
		string_list_append(&wl, grp->output->file[0]->filename);
	id_assign_push(id_target, id_variable_new(&wl));
	string_list_destructor(&wl);

	/*
	 * construct the ``targets'' variable
	 */
	string_list_constructor(&wl);
	assert(grp->input);
	for (j = 0; j < grp->output->nfiles; ++j)
		string_list_append(&wl, grp->output->file[j]->filename);
	id_assign_push(id_targets, id_variable_new(&wl));
	string_list_destructor(&wl);

	/*
	 * construct the ``need'' variable
	 * (and ``younger'' will be identical)
	 */
	string_list_constructor(&wl);
	assert(grp->input);
	for (j = 0; j < grp->input->nfiles; ++j)
		string_list_append(&wl, grp->input->file[j]->filename);
	id_assign_push(id_need, id_variable_new(&wl));
	id_assign_push(id_younger, id_variable_new(&wl));
	string_list_destructor(&wl);

	/*
	 * Flags apply to the precondition and to the ingredients
	 * evaluation.  That is why the grammar puts them first.
	 */
	recipe_flags_set(grp->rp);

	/*
	 * see of the recipe is forced to activate
	 */
	forced = option_test(OPTION_FORCE);

	/*
	 * Print the original position, so the user can tell where it
	 * came from.
	 */
	if (grp->rp->pos.pos_line)
	{
		string_ty	*tmp;

		assert(grp->rp->pos.pos_name);
		tmp = quoted(grp->rp->pos.pos_name);
		printf
		(
			"\n#line %d %s\n",
			grp->rp->pos.pos_line,
			tmp->str_text
		);
		str_free(tmp);
	}

	/*
	 * print the test to see if this recipe should be run
	 */
	if (!forced)
	{
		printf("if test");
		for (j = 0; j < grp->output->nfiles; ++j)
		{ 
			string_ty	*js;
			size_t		k;

			js = quote(grp->output->file[j]->filename);
			if (j)
				printf(" \\\n\t-o");
			/* target does not exist */
			printf(" ! -e %s", js->str_text);
			for (k = 0; k < grp->input->nfiles; ++k)
			{
				string_ty	*ks;

				/* ingredient is newer than target */
				ks = quote(grp->input->file[k]->filename);
				printf(" \\\n\t-o %s -nt %s", ks->str_text, js->str_text);
				str_free(ks);
			}
			str_free(js);
		}
		printf("\nthen\n");
	}

	/*
	 * See if we need to perform the actions attached to this recipe.
	 */
	if (grp->rp->out_of_date)
	{
		int		echo;

		trace(("do recipe body\n"));
		echo = !option_test(OPTION_SILENT);
		if (option_test(OPTION_MKDIR))
		{
			for (j = 0; j < grp->output->nfiles; ++j)
			{
				graph_file_ty	*gfp;
				string_ty	*s;
				string_ty	*tmp;

				gfp = grp->output->file[j];
				s = dir_part(gfp->filename);
				if (!s)
					continue;
				tmp = quote(s);
				str_free(s);
				printf("if test ! -d %s; then\n", tmp->str_text);
				if (echo)
					printf("echo mkdir -p %s\n", tmp->str_text);
				printf("mkdir -p %s", tmp->str_text);
				if (!option_test(OPTION_ERROK))
					printf(" || exit 1");
				printf("\nfi\n");
				str_free(tmp);
			}
		}
		if (option_test(OPTION_UNLINK))
		{
			for (j = 0; j < grp->output->nfiles; ++j)
			{
				graph_file_ty	*gfp;
				string_ty	*tmp;

				gfp = grp->output->file[j];
				tmp = quote(gfp->filename);
				if (echo)
					printf("echo rm %s\n", tmp->str_text);
				printf("rm %s", tmp->str_text);
				if (!option_test(OPTION_ERROK))
					printf(" || exit 1");
				printf("\n");
				str_free(tmp);
			}
		}
		if (option_test(OPTION_TOUCH))
		{
			for (j = 0; j < grp->output->nfiles; ++j)
			{
				graph_file_ty	*gfp;
				string_ty	*tmp;

				gfp = grp->output->file[j];
				tmp = quote(gfp->filename);
				if (echo)
					printf("echo touch %s\n", tmp->str_text);
				printf("touch %s", tmp->str_text);
				if (!option_test(OPTION_ERROK))
					printf(" || exit 1");
				printf("\n");
				str_free(tmp);
			}
		}
		else
		{
			opcode_context_ty *ocp;

			trace(("doing it now\n"));
			ocp = opcode_context_new(grp->rp->out_of_date, grp->mp);
			status = opcode_context_script(ocp);
			if (status != opcode_status_success)
				status = graph_walk_status_error;
			opcode_context_delete(ocp);
		}
	}

	/*
	 * This recipe is being used, so
	 * perform its 'use' action.
	 *
	 * Ignore the 'touch' option,
	 * ignore the 'errok' option,
	 * don't delete files on errors.
	 */
	if (grp->rp->up_to_date)
	{
		opcode_context_ty *ocp;

		printf("else\n");
		trace(("perform ``use'' clause\n"));
		ocp = opcode_context_new(grp->rp->up_to_date, grp->mp);
		status = opcode_context_script(ocp);
		opcode_context_delete(ocp);
		if (status != opcode_status_success)
			status = graph_walk_status_error;
	}

	/*
	 * Delete the variables unique to recipe body execution.
	 */
	id_unassign(id_target);
	id_unassign(id_targets);
	id_unassign(id_need);
	id_unassign(id_younger);

	/*
	 * finish the conditional around this recipe
	 */
	if (!forced)
		printf("fi\n");

	/*
	 * cancel the recipe flags
	 */
	option_undo_level(OPTION_LEVEL_RECIPE);

	star_as_specified('*');
	trace(("return done;\n"));
	trace((/*{*/"}\n"));
	return graph_walk_status_done;
}
