------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                 S Y S T E M . T I M E _ O P E R A T I O N S              --
--                                                                          --
--                                  B o d y                                 --
--                         (Version for new GNARL)                          --
--                                                                          --
--                             $Revision: 1.6 $                            --
--                                                                          --
--             Copyright (C) 1991-1997, Florida State University            --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is a Linux (LinuxThreads) version of this package.

with System.Error_Reporting;
--  used for Shutdown

with System.OS_Interface;

with Interfaces.C;

with System.Task_Primitives.Operations;
--  used for Clock
--           Task_ID
--           Self

package body System.Time_Operations is

   use System.Error_Reporting;
   use System.OS_Interface;
   use Interfaces.C;
   use System.Task_Primitives.Operations;

   --  For abortion safety, we need to
   --  implement these using cond_timedwait. Use the service provided by
   --  System.Task_Primitives.Operations.

   --  -----------------
   --  -- Delay_Until --
   --  -----------------

   procedure Delay_Until (Abs_Time : Duration) is
      Self_ID   : Task_ID;
      Timed_Out : Boolean;
      Result    : Interfaces.C.int;
   begin
      if Abs_Time <= Clock then
         Result := sched_yield;
         return;
      end if;
      Self_ID := Self;
      Write_Lock (Self_ID);
      Sleep_Until (Self_ID, Abs_Time, Timed_Out);
      Unlock (Self_ID);
   end Delay_Until;

   ---------------
   -- Delay_For --
   ---------------

   procedure Delay_For (Rel_Time : Duration) is
   begin
      Delay_Until (Clock + Rel_Time);
   end Delay_For;

   --  For the delay implementation, we need to make sure we achieve
   --  following criterias:
   --  1) We have to delay at least for the amount requested.
   --  2) We have to give up CPU even though the actual delay does not
   --     result in blocking.
   --  3) The implementation has to be efficient so that the delay overhead
   --     is relatively cheap.
   --  1) and 2) are Ada requirements. Even though 2) is an Annex-D
   --     requirement we still want to provide the effect in all cases.
   --     The reason is that users may want to use short delays to implement
   --     their own scheduling effect in the absence of language provided
   --     scheduling policies.

   --  Implementation of Delay_for can be simplfied for the systems on which
   --  the nanosleep() function does not return early. For the reason, we
   --  provide two different implementations. Test both versions and use
   --  the simpler one if possible.

   --  ---------------
   --  -- Delay_For --
   --  ---------------
   --  procedure Delay_For (Rel_Time : Duration) is
   --     Request     : aliased timespec;
   --     New_Request : aliased timespec;
   --     Self_ID     : constant Task_ID := Self;
   --     Result      : Interfaces.C.int;
   --
   --  begin
   --     --  If the request is zero or negative, we need to add it to the
   --     --  tail of the ready queue for its priority.
   --
   --     if Rel_Time <= 0.0 then
   --        Result := sched_yield;
   --        return;
   --     end if;
   --
   --     Request := To_Timespec (Rel_Time);
   --
   --     --  Perform delays until one of the following conditions is true:
   --     --  1) nanosleep wakes up due to time expiration.
   --     --  2) We were interrupted by an abort signal (abortion is pending).
   --     --  3) An error has occurred in the OS-provided delay primitive.
   --     --  Conditions (1) and (2) are normal.
   --     --  Condition (3) should never happen unless the OS is broken,
   --     --  or there is an error in our own runtime system code.
   --
   --     loop
   --        if nanosleep (Request'Access, New_Request'Access) = 0 then
   --           exit;
   --        end if;
   --
   --        if (Self_ID.Pending_Action and then
   --          Self_ID.Pending_ATC_Level < Self_ID.ATC_Nesting_Level)
   --        then
   --           return;
   --        end if;
   --
   --        --  Other than a spurious wakeup is considered an erorr.
   --        pragma Assert (errno = EINTR
   --          or else Shutdown ("GNULLI failure---Delay_For (nanosleep)"));
   --
   --        Request := New_Request;
   --     end loop;
   --
   --     Result := sched_yield;
   --
   --  end Delay_For;
   --

end System.Time_Operations;
