/* vi:set ts=8 sts=0 sw=8:
 * $Id: main.c,v 1.10 1998/10/18 03:55:31 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include "main.h"
#include "prefs.h"
#include "win.h"
#include "doc.h"
#include "msgbar.h"
#include "recent.h"
#include "misc.h"


/*** external declarations ***/
extern char *build_date;	/* these are all in appinfo.c */
extern char *gtkver;
extern char *compiler;
extern char *gtk_cflags;
extern char *cflags;
extern char *gtk_ldflags;
extern char *ldflags;
extern char *unamenfo;


/*** local declarations ***/
#define PRINT_VERSION()							\
	do {								\
		printf("%s %s (%s, compiled %s)\n",			\
			APP_NAME, APP_VERSION, APP_DATE, build_date);	\
	} while(0)

typedef struct {
	GtkWidget *toplev;	/* top level init status window */
	GtkWidget *fnum;	/* label for num files remaining */
	GtkWidget *fname;	/* label for filename */
	int timeout_id;
	win_t *w;
	char **files;
	int num;
} init_data_t;


/*** local variables ***/
static int autosave_id;
static bool_t do_fork = TRUE;
static char *usage_msg[] = {
	"Options:",
	"  --help       shows this screen",
	"  --version    prints app name and version",
	"  --info       prints compile/link options used",
	"",
	"  --nofork     starts app in foreground",
	NULL
};


/*** local function prototypes ***/
static void app_autosave_init(void);
static void app_init(int num, char **files);
static void app_init_open_files(gpointer cbdata);
static void app_init_status_win_create(win_t *w, char **files, int num);
static void app_init_status_win_destroy(init_data_t *idp);
#ifndef GTK_HAVE_FEATURES_1_1_0
static void app_init_status_win_update(init_data_t *idp);
#endif
static void app_parse_args(int *argc, char ***argv);
static void app_print_appinfo(void);
static void app_show_help(char **argv);
static void app_show_info(void);
static void app_show_version(void);


/*** global function definitions ***/

int
main(int argc, char **argv)
{
	app_parse_args(&argc, &argv);

	if (do_fork) {
		pid_t pid;

		if ((pid = fork()) == -1)
			perror("could not fork");

		if (pid != 0)
			exit(0);
	}

	/*
	 * gdk is so braindead in that it catches these "fatal" signals,
	 * but does absolutely nothing with them unless you have compiled
	 * it with a debugging flag (e.g., ./configure --enable-debug=yes).
	 * we'll simply re-establish the signals to their default behavior.
	 * this should produce a core dump in the normal case, which is a
	 * lot more useful than what gdk does.
	 *
	 * if you've compiled GDK with debugging or simply don't care,
	 * then you don't need this.
	 */
#ifdef GDK_IS_BRAINDEAD
	signal(SIGHUP, SIG_DFL);
	signal(SIGINT, SIG_DFL);
	signal(SIGQUIT, SIG_DFL);
	signal(SIGBUS, SIG_DFL);
	signal(SIGSEGV, SIG_DFL);
	signal(SIGPIPE, SIG_DFL);
	signal(SIGTERM, SIG_DFL);
#endif
#ifdef WANT_GNOME
	gnome_init("gnotepad+", NULL, argc, argv, 0, NULL);
#else
	gtk_init(&argc, &argv);
#endif
	prefs_init();
	app_init(argc - 1, argv + 1);
	gtk_main();

	return 0;
} /* main */


/*** local function definitions ***/
/*
 * PRIVATE: app_autosave_init
 */
static void
app_autosave_init(void)
{
	if (autosave <= 0)
		return;

	autosave_id = gtk_timeout_add(
				autosave * (60000),
				(GtkFunction)win_autosave, NULL);
} /* app_autosave_init */


/*
 * PRIVATE: app_init
 *
 * performs all the initial creates of the first window.  creates "Untitled"
 * document if none were specified on the command line.  otherwise, creates
 * a popup showing the number of files remaining to be opened along with the
 * filename to be opened.
 */
static void
app_init(int num, char **files)
{
	win_t *w;

	GNPDBG_GENERIC(("app_init: files to open = %d\n", num));

	/* initialize the window list */
	w = win_new();

	/* add "Untitled" doc if no files specified */
	if (*files == NULL) {
		doc_new(w, UNTITLED);
		gtk_widget_show(w->toplev);
		msgbar_printf(w, "%s%s %s", WELCOME_MSG, APP_NAME, APP_VERSION);
		app_autosave_init();
	} else {
		app_init_status_win_create(w, files, num);
	}

} /* app_init */


/*
 * PRIVATE: app_init_open_files
 *
 * callback routine for app_init_status_win_create() which is attached to a
 * timeout during the creation of the status popup window.
 */
static void
app_init_open_files(gpointer cbdata)
{
	init_data_t *idp = (init_data_t *)cbdata;
	win_t *w;

	g_assert(idp != NULL);
	w = idp->w;
	/*
	 * for each file specified, create a document and add it to
	 * the current window.  since we are inserting into the doc
	 * notebook at the head of the list, we'll do the filenames
	 * backwards to preserve the order.
	 */
	while (idp->num > 0) {
#ifndef GTK_HAVE_FEATURES_1_1_0
		app_init_status_win_update(idp);
#endif
		doc_new(idp->w, idp->files[idp->num - 1]);
		recent_list_add(idp->w, idp->files[idp->num - 1]);

		g_free(idp->files[idp->num - 1]);
		idp->files[idp->num - 1] = NULL;
		idp->num--;
	}
	app_init_status_win_destroy(idp);
	gtk_widget_show(w->toplev);
	msgbar_printf(w, "%s%s %s", WELCOME_MSG, APP_NAME, APP_VERSION);
	app_autosave_init();
} /* app_init_open_files */


/*
 * PRIVATE: app_init_status_win_create
 *
 * creates all the widgets necessary for the popup window to show the
 * initialization status.  the files are actually opened through a callback
 * routine which is attached to a gtk_timeout near the end of this function.
 */
#define FILES_LEFT	"Files left to open: "
static void
app_init_status_win_create(win_t *w, char **files, int num)
{
	GtkWidget *vbox;
	init_data_t *idp;
	int i;
#ifdef DETERMINE_WINDOW_DIMENSIONS
	/* use this to get exact window dimensions */
	int wheight, wwidth, sheight, swidth;
#endif

	idp = (init_data_t *)g_malloc0(sizeof(init_data_t));
	idp->w = w;
	idp->num = num;
	idp->files = (char **)g_malloc(sizeof(char *) * num);
	for (i = 0; i < num; i++)
		idp->files[i] = g_strdup(files[i]);

	idp->toplev = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(idp->toplev), "gnotepad+ Init...");
	gtk_signal_connect(GTK_OBJECT(idp->toplev), "delete_event",
		GTK_SIGNAL_FUNC(gtk_true), NULL);
	gtk_container_border_width(GTK_CONTAINER(idp->toplev), 10);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add(GTK_CONTAINER(idp->toplev), vbox);

#ifdef GTK_HAVE_FEATURES_1_1_0	/* XXX - this doesn't work correctly */
	idp->fnum = gtk_label_new(" Opening files... ");
	gtk_box_pack_start(GTK_BOX(vbox), idp->fnum, FALSE, FALSE, 0);
#else
	/* for some reason, the labels below are not centered correctly
	 * when this is compiled with gtk-1.1 */
	idp->fnum = gtk_label_new(FILES_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), idp->fnum, FALSE, TRUE, 10);

	idp->fname = gtk_label_new(" ");
	gtk_box_pack_start(GTK_BOX(vbox), idp->fname, FALSE, TRUE, 10);
#endif
#ifdef DETERMINE_WINDOW_DIMENSIONS
	/* XXX - can we get window dimensions without showing the widget?? */
	gtk_widget_show(idp->toplev);
	/* get window width & height, then set it to be larger */
	gdk_window_get_size(idp->toplev->window, &wwidth, &wheight);
	wwidth = 3 * gdk_string_width(idp->toplev->style->font, FILES_LEFT);
	gtk_widget_set_usize(idp->toplev, wwidth, wheight);

	/* get screen width & height, so we can center the window */
	swidth = gdk_screen_width();
	sheight = gdk_screen_height();

	gtk_widget_set_uposition(idp->toplev,
		(swidth/2) - (wwidth/2), (sheight/2) - (wheight/2));
#else
	gtk_widget_set_usize(idp->toplev, 200, 96);
	gtk_window_position(GTK_WINDOW(idp->toplev), GTK_WIN_POS_CENTER);
#endif

	gtk_widget_show_all(idp->toplev);

	/* this will kick-off opening all the files */
	idp->timeout_id =
		gtk_timeout_add(1, (GtkFunction)app_init_open_files, idp);
} /* app_init_status_win_create */


/*
 * PRIVATE: app_init_status_win_destroy
 *
 * cleanup
 */
static void
app_init_status_win_destroy(init_data_t *idp)
{
	g_assert(idp->files[0] == NULL);
	g_free(idp->files);
	gtk_timeout_remove(idp->timeout_id);
	if (idp->toplev)
		gtk_widget_destroy(idp->toplev);
	g_free(idp);
} /* app_init_status_win_destroy */


#ifndef GTK_HAVE_FEATURES_1_1_0
/*
 * PRIVATE: app_init_status_win_update
 *
 * update the two labels in the status window.
 */
static void
app_init_status_win_update(init_data_t *idp)
{
	char *buf, *bfname;
	int idle_tag, len;
	GdkRectangle area = { 0, 0, (guint16)(-1), (guint16)(-1) };

	bfname = (char *)my_basename(idp->files[idp->num - 1]);
	len = MAX(strlen(bfname) + 1, strlen(FILES_LEFT) + 10);
	buf = (char *)g_malloc(len);
	g_snprintf(buf, len, " %s%d ", FILES_LEFT, idp->num);
	gtk_label_set(GTK_LABEL(idp->fnum), buf);
	g_snprintf(buf, len, " %s ", bfname);
	gtk_label_set(GTK_LABEL(idp->fname), buf);
	g_free(buf);

	/* this code hack borrowed from GIMP */
	gtk_widget_draw(idp->fnum, &area);
	gtk_widget_draw(idp->fname, &area);
	idle_tag = gtk_idle_add((GtkFunction)gtk_true, NULL);
	gtk_main_iteration();
	gtk_idle_remove(idle_tag);
} /* app_init_status_win_update */
#endif


/*
 * PRIVATE: app_parse_args
 *
 * parse for application specific arguments.  return argc and argv with any
 * app specific arguments omitted/removed.
 */
#define	SHIFT_ARGS(zz_argc, zz_argp)					\
	do {								\
		char **aptmp;						\
		(zz_argc)--;						\
		aptmp = zz_argp;					\
		while (*aptmp) {					\
			if (*(aptmp + 1) == NULL) {			\
				*aptmp = NULL;				\
			} else {					\
				*aptmp = *(aptmp + 1);			\
				aptmp++;				\
			}						\
		}							\
	} while (0)

static void
app_parse_args(int *argcp, char ***argvp)
{
	char **ap = *argvp;

	while (*ap) {
		GNPDBG_GENERIC(("app_parse_args: *ap = '%s'\n", *ap));
		if (strcmp(*ap, "--help") == 0)
			app_show_help(*argvp);	/* does not return */
		else if (strcmp(*ap, "--version") == 0)
			app_show_version();	/* does not return */
		else if (strcmp(*ap, "--info") == 0)
			app_show_info();	/* does not return */
		else if (strcmp(*ap, "--nofork") == 0) {
			do_fork = FALSE;
			SHIFT_ARGS(*argcp, ap);
		}
#ifdef GNP_DEBUG
		else if (strcmp(*ap, "--debug") == 0) {
			SHIFT_ARGS(*argcp, ap);
			if (*ap[0] == '0')
				sscanf(*ap, "%lx", &dbg_flags);
			else
				sscanf(*ap, "%ld", &dbg_flags);
			GNPDBG_ALWAYS(("dbg_flags = 0x%lx (%ld)\n",
					dbg_flags, dbg_flags));
			SHIFT_ARGS(*argcp, ap);
		}
#endif
		else
			ap++;
	}
} /* app_parse_args */


/*
 * PRIVATE: app_print_appinfo
 *
 * duh
 */
static void
app_print_appinfo(void)
{
	printf("Compile/Link info:\n");
	printf("\tGTK version = %s\n", gtkver);
	printf("\tCC          = %s\n", compiler);
	printf("\tGTK_CFLAGS  = %s\n", gtk_cflags);
	printf("\tCFLAGS      = %s\n", cflags);
	printf("\tGTK_LDFLAGS = %s\n", gtk_ldflags);
	printf("\tLDFLAGS     = %s\n", ldflags);
	printf("\tUNAME       = %s\n", unamenfo);
} /* app_print_appinfo */


/*
 * PRIVATE: app_show_help
 *
 * duh
 */
static void
app_show_help(char **argv)
{
	char **up = usage_msg;

	printf("Usage: %s [options] [files...]\n", argv[0]);
	while (*up != NULL) {
		printf("%s\n", *up);
		up++;
	}
	exit(0);
} /* app_show_help */


/*
 * PRIVATE: app_show_info
 *
 * duh
 */
static void
app_show_info(void)
{
	PRINT_VERSION();
	app_print_appinfo();
	exit(0);
} /* app_show_help */


/*
 * PRIVATE: app_show_version
 *
 * duh
 */
static void
app_show_version(void)
{
	PRINT_VERSION();
	exit(0);
} /* app_show_version */


/* the end */
