/* The main simulation-running code in Xconq.
   Copyright (C) 1986-1989, 1991-1998 Stanley T. Shebs.

Xconq is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.  See the file COPYING.  */

/* This is the main simulation-running code. */

#include "conq.h"
#include "kernel.h"
#include "kpublic.h"

/* Functions in run2.c, only called from this file. */

extern void run_turn_start PARAMS ((void));
extern void run_restored_turn_start PARAMS ((void));
extern void run_turn_end PARAMS ((void));

static void compose_actionvectors PARAMS ((void));
static void init_movement PARAMS ((void));
static void init_actionvectors PARAMS ((void));
static int move_some_units PARAMS ((int lim));
static int side_move_some_units PARAMS ((Side *side, int lim));
static int unit_still_acting PARAMS ((Unit *unit));
static int unit_still_acting_no_plan PARAMS ((Unit *unit));
static int move_one_unit_multiple PARAMS ((Unit *unit, int lim));
static void finish_movement PARAMS ((void));

/* Priority of sides that are now moving. */

int curpriority;

/* Priority of units that are now moving. */

int cur_unit_priority;

/* Highest and lowest possible priority of unit (smallest and largest
   number). */

int highest_unit_priority;
int lowest_unit_priority;

int maintimeout = -1;

int paused = FALSE;

/* State variables. */
/* (I don't think all of these are strictly necessary) */

/* This becomes TRUE the first time run_game is executed. */

int gameinited = FALSE;

/* This is true only before the game actually starts. */

int beforestart = TRUE;

/* This is true only at the beginning of a turn. */

int at_turn_start = FALSE;

/* This is true after the game is over. */

int endofgame = FALSE;

/* This is true when the program may actually exit. */

int ok_to_exit;

/* This is set FALSE whenever the game state changes, and TRUE whenever
   the game has been saved. */

int gamestatesafe = TRUE;

/* This is TRUE after the designer has been mucking around, or if
   networked versions are found to be inconsistent. */

int compromised = FALSE;

int in_run_game = FALSE;

/* The time at which the game actually starts. */

time_t game_start_in_real_time;

/* The point in the turn at which players can actually do things. */

time_t turn_play_start_in_real_time;

int planexecs;

int taskexecs;

/* The rate at which AIs play when acting more slowly (so as
   not to overwhelm human players), expressed as plan executions
   per minute.  0 means "as fast as possible". */

int slow_play_rate = 240;

/* The rate at which AIs play when acting more quickly (such
   as when all humans are done with their moves). */

int fast_play_rate = 0;

/* The current rate at which AIs are playing. */

int current_play_rate;

/* Debugging counts for when run_game does nothing many times. */

static int nothing_count;

static int nothing_timeout;

/* True when an event occurs and we need to check scorekeepers. */

int need_post_event_scores;

/* Flags that other code uses to signal the AI code that it ought
   to consider running something. */

int need_ai_init_turn;

int need_ai_planning;

int need_ai_task_reaction;

int need_ai_for_new_side;

int need_ai_finish_movement;

int debugging_state_sync;

void
init_run()
{
    int u;
    Unit *unit;
    Side *side;

    highest_unit_priority = 9999;
    lowest_unit_priority = -1;
    for_all_unit_types(u) {
        highest_unit_priority = min(highest_unit_priority, u_action_priority(u));
        lowest_unit_priority = max(lowest_unit_priority, u_action_priority(u));
        for_all_sides_plus_indep(side) {
	    if (side->action_priorities) {
		highest_unit_priority = min(highest_unit_priority, side->action_priorities[u]);
		lowest_unit_priority = max(lowest_unit_priority, side->action_priorities[u]);
	    }
        }
    }
    for_all_units(unit) {
        if (unit->extras) {
	    highest_unit_priority = min(highest_unit_priority, unit_extra_priority(unit));
	    lowest_unit_priority = max(lowest_unit_priority, unit_extra_priority(unit));
        }
    }
}

/* This function does a (small, usually) amount of simulation, then returns.
   It can be run multiple times at any time, will not go "too far".
   It returns the number of actions that were actually performed. Other
   important state changes (such a side finishing its turn or the turn
   ending) are also counted as actions, so that this function's callers
   will know that something was done. */

int
run_game(maxactions)
int maxactions;
{
    int numacted, numother, runtime, numdone, bump;
    long saved_randstate;
    time_t rungamestart, rungameend;
    Side *side;
    extern long randstate;

    in_run_game = TRUE;
    if (Debug && numremotes > 0 && 0) {
	sprintf(spbuf, ":states:state%06da", g_run_serial_number());
	debugging_state_sync = TRUE;
	write_entire_game_state(spbuf);
	debugging_state_sync = FALSE;
    }
    gameinited = TRUE;
    saved_randstate = randstate;
    time(&rungamestart);
    numacted = numother = planexecs = taskexecs = 0;
    need_ai_planning = FALSE;
    if (beforestart) {
	/* If we haven't started yet, see if it's time. */
	test_for_game_start();
	Dprintf("run_game: tested for game start.\n");
    }
    if (endofgame) {
	/* Nothing to do except wait for users to do exit commands. */
    	Dprintf("run_game: at end of game.\n");
    } else if (paused) {
	/* Don't do anything if we're paused. */
    	Dprintf("run_game: paused.\n");
    } else if (!beforestart) {
	if (at_turn_start) {
	    if (midturnrestore)
	      run_restored_turn_start();
	    else
	      run_turn_start();
	    check_all_units();
	    init_movement();
	    cur_unit_priority = highest_unit_priority;
	    compose_actionvectors();
	    update_all_progress_displays("", -1);
	    /* Game might have been ended by new turn init. */
	    test_for_game_end();
	    if (endofgame) {
	    	Dprintf("run_game: game ended by new turn init.\n");
	    	goto rungamereturn;
	    }
	    /* (should adjust this by recorded elapsed turn time) */
	    time(&turn_play_start_in_real_time);
	    at_turn_start = FALSE;
	    ++numother;
	    /* Might have local AIs that need to run, so give them a chance
	       now rather than after units start moving. */
	    need_ai_init_turn = TRUE;
	    need_ai_planning = TRUE;
	    goto rungamereturn;
	}
	/* If this game is running in realtime, update all clock displays. */
	if (realtime_game()) {
	    for_all_sides(side) {
		if (side->ingame && side_has_display(side)) {
		    update_clock_display(side, TRUE);
		}
	    }
	}
	/* If all sides are done acting, end the turn.  This will never be true
	   right at the start of a turn. */
	if (all_sides_finished() || exceeded_rt_per_turn()) {
	    run_turn_end();
	    Dprintf("run_game: at turn end.\n");
	    at_turn_start = TRUE;
	    ++numother;
	    need_ai_finish_movement = TRUE;
	} else {
	    /* Move some units around. */
	    numacted += move_some_units(maxactions);
	    if (cur_unit_priority < lowest_unit_priority) {
		/* Handle prioritized movement. */
		bump = TRUE;
		for_all_sides(side) {
		    if (!side->finishedturn
			&& units_still_acting(side)) {
			bump = FALSE;
		    }
		}
		if (bump) {
		    Dprintf("run_game: increment unit priority to %d\n",
			    cur_unit_priority + 1);
		    ++cur_unit_priority;
		    compose_actionvectors();
		    ++numother;
		}
	    } else {
		/* Possibly finish some sides' turns. */
		for_all_sides(side) {
		    if (!side->finishedturn
			&& !units_still_acting(side)
			&& side->autofinish
			&& !is_designer(side)) {
			Dprintf("run_game: %s auto-finishes.\n",
				side_desig(side));
			finish_turn(side);
			++numother;
		    }
		}
	    }
	}
	check_realtime();
	test_for_game_end();
    }
rungamereturn:
    if (need_post_event_scores)
      check_post_event_scores();
    numdone = numacted + planexecs + taskexecs + numother;
    if (Debug) {
	if (numdone > 0) {
	    if (nothing_count > 0) {
		Dprintf("run_game: Did nothing %d times\n", nothing_count);
		nothing_count = 0;
	    }
	    Dprintf("run_game #%d: %d/%d actions", g_run_serial_number(),
		    numacted, maxactions);
	    if (planexecs > 0)
	      Dprintf(", %d plan execs", planexecs);
	    if (taskexecs > 0)
	      Dprintf(", %d task execs", taskexecs);
	    if (numother > 0)
	      Dprintf(", %d other", numother);
	    /* (also number of units considered?) */
	    Dprintf("\n");
	    if (numremotes > 0) {
		Dprintf("run_game: Randstate started at %ld, is now %ld\n",
			saved_randstate, randstate);
#if 0 /* use this for high-powered networking debugging */
		if (1 /* record each run_game states */) {
		    sprintf(spbuf, ":states:state%06d", g_run_serial_number());
		    debugging_state_sync = TRUE;
		    write_entire_game_state(spbuf);
		    debugging_state_sync = FALSE;
		}
#endif
	    }
	} else {
	    if (nothing_count >= 1000) {
		Dprintf("run_game: Did nothing %d times\n", nothing_count);
		nothing_count = 0;
	    } else {
		++nothing_count;
	    }
	}
    }
    time(&rungameend);
    runtime = idifftime(rungameend, rungamestart);
    if (runtime > 0)
      Dprintf("run_game: runtime seconds = %d\n", runtime);
    if (Debug && numremotes > 0 && 0) {
	sprintf(spbuf, ":states:state%06db", g_run_serial_number());
	debugging_state_sync = TRUE;
	write_entire_game_state(spbuf);
	debugging_state_sync = FALSE;
    }
    in_run_game = FALSE;
    set_g_run_serial_number(g_run_serial_number() + 1);
    return numdone;
}

/* See if game is ready to get underway for real.  Note that displays will
   work before the game has started, but game time doesn't move. */

void
test_for_game_start()
{
    int anydisplays = FALSE;
    Side *side;

    /* We must have at least one unit on a side that is being displayed
       before the game can start for real. */
    for_all_sides(side) {
	if (side_has_display(side)) {
	    anydisplays = TRUE;
	}
	if (side_has_units(side) && side_has_display(side)) {
	    /* Now we're really ready to roll. */
	    beforestart = FALSE;
	    at_turn_start = TRUE;
	    if (midturnrestore) {
		record_event(H_GAME_RESTARTED, ALLSIDES);
	    } else {
		record_event(H_GAME_STARTED, ALLSIDES);
		set_g_elapsed_time(0);
	    }
	    /* Record the game as starting NOW in real time. */
	    time(&game_start_in_real_time);
	    /* Adjust by any recorded elapsed time. */
	    game_start_in_real_time -= g_elapsed_time();
	    /* No need to look at any more sides, just get on with the game. */
	    return;
	}
    }
    if (!anydisplays) {
	init_warning("No sides have a display");
    }
}

/* This routine looks to see if the game is completely over. */

void
test_for_game_end()
{
    Side *side;

    /* Declare a draw if everybody is amenable. */
    if (all_others_willing_to_quit(NULL)) {
    	notify_all("All sides have agreed to declare a draw.");
    	all_sides_draw();
	end_the_game();
	return;
    }
    for_all_sides(side) {
    	/* If we have an active side being displayed, we're not done yet. */
	if (side->ingame && side_has_display(side))
	  return;
	/* (If no displayed sides have units, turns will whiz by) */
    }
    notify_all("All sides with displays are out of the game.");
    end_the_game();
}

/* This is true when all participating sides have finished their turn. */

int
all_sides_finished()
{
    Side *side;

    for_all_sides(side) {
	if (side->ingame
	    && !side->finishedturn) {
	    return FALSE;
	}
    }
    return TRUE;
}

/* This is true when AIs should move more slowly. */

int
all_human_only_sides_finished()
{
    Side *side;

    for_all_sides(side) {
	if (side->ingame
	    && side_has_display(side)
	    && !side_has_ai(side)
	    && !side->finishedturn) {
	    return FALSE;
	}
    }
    return TRUE;
}

/* Call this from interfaces to check on realtime details without actually
   going into run_game.  Will call back to interface if necessary. */

void
check_realtime()
{
    Side *side;

    if (!realtime_game())
      return;
    if (exceeded_rt_for_game()) {
	notify_all("Time has run out!");
	end_the_game();
    }
    if (g_rt_per_side() > 0) {
	for_all_sides(side) {
	    if (side->ingame && side->totaltimeused > g_rt_per_side()) {
		remove_side_from_game(side);
	    }
	}
    }
}

int
exceeded_rt_for_game()
{
    time_t now;

    if (g_rt_for_game() <= 0)
      return FALSE;
    time(&now);
    /* Note that the game start time is already adjusted for any
       elapsed time recorded when the game was last saved. */
    return (idifftime(now, game_start_in_real_time) > g_rt_for_game());
}

int
exceeded_rt_per_turn()
{
    time_t now;

    if (g_rt_per_turn() <= 0)
      return FALSE;
    time(&now);
    return (idifftime(now, turn_play_start_in_real_time) > g_rt_per_turn());
}

/* This returns true if the given side is still wanting to do stuff. */

int
units_still_acting(side)
Side *side;
{
    int curactor;
    Unit *unit;

    if (!side->ingame)
      return FALSE;
    /* Test current actor first, most likely to be still acting. */
    if (side->curactor_pos < side->actionvector->numunits) {
	unit = (side->actionvector->units)[side->curactor_pos].unit;
	if (unit_still_acting(unit) && side_controls_unit(side, unit))
	  return TRUE;
    }
    for (curactor = 0; curactor < side->actionvector->numunits; ++curactor) {
	unit = (side->actionvector->units)[curactor].unit;
	if (unit_still_acting(unit) && side_controls_unit(side, unit))
	  return TRUE;
    }
    return FALSE;
}

static void
init_movement()
{
    int i;
    Side *side;

    i = 1;
    curpriority = 9999;
    /* In the absence of any idea about how to handle actions by
       independent units, force their turn to be finished
       immediately. */
    indepside->finishedturn = TRUE;
    for_all_sides(side) {
	if (side->ingame) {
	    /* Record that this side was active during at least one turn. */
	    side->everingame = TRUE;
	    /* No units are waiting for orders initially. */
	    side->numwaiting = 0;
	}
	side->turnstarttime = time(0);
	/* Didn't really do input, but useful to pretend so. */
	side->lasttime = time(0);
	/* Calculate side priorities; do here so future versions can
	   set priorities dynamically. */
	if (g_use_side_priority()) {
	    if (side->priority < 0) {
		side->priority = i++;
	    }
	}
	side->busy = FALSE;
	if (side_has_display(side))
	  update_action_display(side, TRUE);
    }
    /* Set independent units to move after units on sides. */
    if (g_use_side_priority()) {
	if (indepside->priority < 0) {
	    indepside->priority = i;
	}
	for_all_sides_plus_indep(side) {
	    if (!side->finishedturn && side->priority < curpriority)
	      curpriority = side->priority;
	}
    }
}

static void
compose_actionvectors()
{
    int priority;
    Unit *unit;
    Side *side, *side2;

    for_all_sides_plus_indep(side) {
	if (side->actionvector == NULL)
	  side->actionvector = make_unit_vector(max(numunits, 100));
	clear_unit_vector(side->actionvector);
	for_all_side_units(side, unit) {
	    if (unit->act && unit->act->initacp > 0) {
		priority = unit_priority(unit);
		if (priority == cur_unit_priority) {
		    side->actionvector = add_unit_to_vector(side->actionvector, unit, 0);
		    /* Clear any delay flags. */
		    if (unit->plan)
		      unit->plan->delayed = FALSE;
		}
	    }
	    if (unit->plan) {
		unit->plan->execs_this_turn = 0;
	    }
	}
	Dprintf("Action vector for %s has %d units, at priority %d\n",
		side_desig(side), side->actionvector->numunits, cur_unit_priority);
    }
    /* Inform sides with displays of how many units are ready to act. */
    for_all_sides(side) {
	if (side_has_display(side)) {
	    for_all_sides(side2) {
		update_side_display(side, side2, TRUE);
	    }
	}
    }
}

int
unit_priority(unit)
Unit *unit;
{
    int pri;
    Side *side;

    pri = unit_extra_priority(unit);
    if (pri >= 0)
      return pri;
    side = (unit->side ? unit->side : indepside);
    if (side->action_priorities != NULL) {
	pri = side->action_priorities[unit->type];
	if (pri >= 0)
	  return pri;
    }
    return u_action_priority(unit->type);
}

/* Do some number of actions. */

static int
move_some_units(lim)
int lim;
{
    int num = 0, sidenum;
    Side *side;

    for_all_sides_plus_indep(side) {
	if ((g_use_side_priority() ?
	     curpriority == side->priority :
	     TRUE)) {
	    sidenum = side_move_some_units(side, lim);
	    num = max(num, sidenum);
	}
    }
    return num;
}

/* Do some number of actions. */

static int
side_move_some_units(side, lim)
Side *side;
int lim;
{
    int num, foundanytomove, curactor0, curactor, numdelayed;
    Unit *unit;

    num = 0;
    curactor0 = 0;
    if (side->curactor_pos < side->actionvector->numunits
        && side->curactor == (side->actionvector->units)[side->curactor_pos].unit
        && side->curactor != NULL
        && side->curactor_id == ((side->actionvector->units)[side->curactor_pos].unit)->id)
      curactor0 = side->curactor_pos;
    if (numremotes > 0)
      curactor0 = 0;
  tryagain:
    foundanytomove = FALSE;
    numdelayed = 0;
    for (curactor = curactor0; curactor < side->actionvector->numunits; ++curactor) {
	unit = (side->actionvector->units)[curactor].unit;
	if (0 /*numremotes > 0*/)
	  Dprintf("Considering moving %s with plan %s\n",
		  unit_desig(unit), plan_desig(unit->plan));
	/* Count and skip over deliberately delayed units. */
	if (unit->plan && unit->plan->delayed) {
	    ++numdelayed;
	    continue;
	}
	current_play_rate = slow_play_rate;
	/* AIs should play as fast as possible if turns are sequential or if the
	   human players are all done. */
	if (g_use_side_priority() || all_human_only_sides_finished()) {
	    current_play_rate = fast_play_rate;
	}
	/* If the unit is keeping formation, then give it a chance to
	   adjust its position, even if it's not "still acting". */
	if (is_active(unit)
	    && (unit->side ?
		(unit->side->ingame && !unit->side->finishedturn) : TRUE)
	    && (unit->act && unit->act->acp > 0)
	    && (unit->plan && unit->plan->formation)) {
	    num += move_one_unit_multiple(unit, lim - num);
	    foundanytomove = TRUE;
	}
	if (unit->side
	    && unit->side->orders
	    && unit->plan
	    && unit->plan->tasks == NULL
	    && execute_standing_order(unit, FALSE)) {
	    /* We're not waiting because standing order execution will
	       shortly be allowed to fill in a task for real. */
	    set_waiting_for_tasks(unit, FALSE);
	    num += move_one_unit_multiple(unit, lim - num);
	    foundanytomove = TRUE;
	}
	if (unit_still_acting(unit)
	    && (unit->plan && !unit->plan->waitingfortasks)) {
	    num += move_one_unit_multiple(unit, lim - num);
	    foundanytomove = TRUE;
	} else if (unit_still_acting_no_plan(unit)) {
	    num += move_one_unit_multiple(unit, lim - num);
	    foundanytomove = TRUE;
	}
	if (unit_still_acting(unit)) {
	    foundanytomove = TRUE;
	}
	if (num >= lim) {
	    if (foundanytomove && unit != NULL) {
		side->curactor_pos = curactor;
		side->curactor = unit;
		side->curactor_id = unit->id;
	    } else {
		side->curactor_pos = 0;
		side->curactor = NULL;
		side->curactor_id = 0;
	    }
	    return num;
	}
    }
    /* If started in middle of list, rescan from beginning. */
    if (!foundanytomove && curactor0 > 0) {
	curactor0 = 0;
	goto tryagain;
    }
    /* Clear all the delay flags and rescan the action vector. */
    if (!foundanytomove && numdelayed > 0) {
	for (curactor = 0; curactor < side->actionvector->numunits; ++curactor) {
	    unit = (side->actionvector->units)[curactor].unit;
	    if (unit->plan)
	      unit->plan->delayed = FALSE;
	}
	curactor0 = 0;
	goto tryagain;
    }
    if (!foundanytomove && 0 /* not at max priority */) {
	/* (should recompose action vector for new priority?) */
    }
    side->curactor_pos = 0;
    side->curactor = NULL;
    side->curactor_id = 0;
    return num;
}

static int
unit_still_acting(unit)
Unit *unit;
{
    return (is_active(unit)
	    && (unit->side
		&& unit->side->ingame
		&& !unit->side->finishedturn)
	    && (unit->act
		&& unit->act->acp > 0)
	    && ((unit->plan
		&& !unit->plan->asleep
		&& !unit->plan->reserve)
		|| has_pending_action(unit)));
}

static int
unit_still_acting_no_plan(unit)
Unit *unit;
{
    return (is_active(unit)
	    && (unit->side
		&& unit->side->ingame
		&& !unit->side->finishedturn)
	    && (unit->act
		&& unit->act->acp > 0)
	    && has_pending_action(unit));
}

/* Do a single unit's actions, up to the given limit or until it runs
   out of things it wants to do (or something happens to it). */

int lastexecution = 0;

static int
move_one_unit_multiple(unit, lim)
Unit *unit;
int lim;
{
    int num = 0, buzz = 0, acp1;
    int rslt;

    /* If unit is incapable of acting right now, get out of here. */
    if (unit->act == NULL || unit->act->initacp < 1)
      return 0;
    acp1 = unit->act->acp;
    while (is_active(unit)
	   && (unit->act
	       && unit->act->acp > u_acp_min(unit->type))
	   && ((unit->plan
		&& !unit->plan->asleep
		&& !unit->plan->reserve
		&& !unit->plan->delayed
		&& !(need_ai_task_reaction
		     && unit->plan->last_task_outcome != TASK_UNKNOWN))
	       || has_pending_action(unit))
	   && num < lim
	   && buzz < lim) {
	if (numremotes > 0)
	  Dprintf("  Moving %s (%d/%d, buzz %d) with plan %s\n",
		  unit_desig(unit), num, lim, buzz, plan_desig(unit->plan));
	if (has_pending_action(unit)) {
	    /* Execute the action directly. */
	    rslt = execute_action(unit, &(unit->act->nextaction));
	    /* Clear the action.  Note that the unit might have changed
	       to a non-acting type, so we have to check for act struct. */
	    if (unit->act)
	      unit->act->nextaction.type = ACTION_NONE;
	    /* In any case, the game state is irrevocably altered. */
	    gamestatesafe = FALSE;
	    ++num;
	} else if (unit->plan != NULL) {
	    /* Even units that are asleep, in reserve, etc must execute
	       any standing orders that apply. */
	    if (unit->side
		&& unit->side->orders
		&& unit->plan->tasks == NULL
		&& execute_standing_order(unit, TRUE)) {
		execute_plan(unit, 1);
		gamestatesafe = FALSE;
		++buzz;
	    }
	    /* Similarly for formations. */
	    if (unit->plan->formation && move_into_formation(unit)) {
		execute_plan(unit, 1);
		gamestatesafe = FALSE;
		++buzz;
	    }
	    /* Flag so that we run AI code in the near future (after
	       run_game exits). */
	    if (side_has_ai(unit->side))
	      need_ai_planning = TRUE;
	    /* Get out of here if unit is set not to do anything on
	       its own. */
	    if (unit->plan->waitingfortasks
		|| unit->plan->asleep
		|| unit->plan->reserve
		|| unit->plan->delayed)
	      break;
	    /* Normal plan execution. */
	    execute_plan(unit, 1);
	    record_ms();
	    gamestatesafe = FALSE;
	    ++buzz;
	} else {
	    run_warning("Planless \"%s\" was asked to act", unit_desig(unit));
	    ++buzz;
	}
	/* If the unit is trying to do actions several times in this
	   loop and and none of them are succeeding, something is
	   wrong; blast the plan and eventually put the unit to
	   sleep, if the problem persists. */
	if (unit->act && unit->act->acp == acp1 && num > 1) {
	    /* Blast the action. */
	    unit->act->nextaction.type = ACTION_NONE;
	    /* Blast the plan. */
	    if (unit->plan)
	      unit->plan->type = PLAN_PASSIVE;
	    if (unit->plan && probability(5))
	      unit->plan->asleep = TRUE;
	    run_warning("\"%s\" trying multiple bad actions, clearing its plan",
			unit_desig(unit));
	}
    }
    return num;
}

/* This explicitly finishes out a side's activity for the turn. */

void
finish_turn(side)
Side *side;
{
    int nextpriority;
    Side *side2, *side3;

    /* Flag the side as being done for this turn. */
    side->finishedturn = TRUE;
    /* Stop counting down our time consumption. */
    side->totaltimeused += (time(0) - side->turnstarttime);
    if (g_use_side_priority()) {
	nextpriority = 9999;
	for_all_sides_plus_indep(side2) {
	    if (!side2->finishedturn
/*		&& side2->priority > curpriority */
		&& side2->priority < nextpriority) {
		nextpriority = side2->priority;
	    }
	    if (!side2->finishedturn && side2->priority < curpriority)
	      run_warning("%s not finished, but priority is %d, less than current %d",
			  side_desig(side2), side2->priority, curpriority);
	}
	if (nextpriority > curpriority)
	  curpriority = nextpriority;
    }
    /* Clue everybody in. */
    if (g_use_side_priority()) {
	/* Several sides may change, if current priority changes. */
	for_all_sides(side2) {
	    for_all_sides(side3) {
		update_side_display(side2, side3, TRUE);
	    }
	}
    } else {
	/* Only the turn-finishing side changes. */
	for_all_sides(side2) {
	    update_side_display(side2, side, TRUE);
	}
    }
    Dprintf("%s finished its turn.\n", side_desig(side));
}

void run_ai_plan_adjust PARAMS ((Side *side));

int
run_local_ai(when, maxplanning)
int when, maxplanning;
{
    Side *side;
    Unit *unit;

    /* Only do this routine at a certain rate. */
    if (current_play_rate > 0
	&& !n_ms_elapsed(60000 / current_play_rate)
	)
      return;
    if (need_ai_for_new_side) {
	for_all_sides(side) {
	    if (side_wants_ai(side) && !side_has_ai(side)) {
		init_ai(side);
	    }
	}
	need_ai_for_new_side = FALSE;
    }
    if (need_ai_init_turn) {
	for_all_sides(side) {
	    if (side_has_ai(side))
	      ai_init_turn(side);
	}
	need_ai_init_turn = FALSE;
    }
    if (need_ai_planning) {
	for_all_sides(side) {
	    if (side_has_ai(side) /* (should) and need planning for this side*/) {
		for_all_side_units(side, unit) {
		    if (is_active(unit)
			&& unit->plan
			&& unit->plan->aicontrol
			&& !unit->plan->asleep) {
			ai_decide_plan(side, unit);
		    }
		}
	    }
	}
	need_ai_planning = FALSE;
    }
    if (need_ai_task_reaction) {
	for_all_sides(side) {
	    if (side_has_ai(side)) {
		for_all_side_units(side, unit) {
		    if (is_active(unit)
			&& unit->plan
			&& unit->plan->aicontrol
			&& unit->plan->last_task_outcome != TASK_UNKNOWN) {
			ai_react_to_task_result(side, unit, &(unit->plan->last_task),
						unit->plan->last_task_outcome);
			unit->plan->last_task_outcome = TASK_UNKNOWN;
		    }
		}
	    }
	}
	need_ai_task_reaction = FALSE;
    }
    if (1) {
	for_all_sides(side) {
	    if (side_has_ai(side) && !side->finishedturn) {
		run_ai_plan_adjust(side);
	    }
	}
    }
    if (need_ai_finish_movement) {
	for_all_sides(side) {
	    if (side_has_ai(side) && !side->finishedturn) {
		ai_finish_movement(side);
	    }
	}
	need_ai_finish_movement = FALSE;
    }
}

void
run_ai_plan_adjust(side)
Side *side;
{
    int curactor, domore;
    Unit *unit;

    if (side->actionvector == NULL)
      return;
    domore = TRUE;
    for (curactor = 0; curactor < side->actionvector->numunits; ++curactor) {
	unit = (side->actionvector->units)[curactor].unit;
	if (unit->plan
	    && unit->plan->aicontrol) {
	    domore = ai_adjust_plan(side, unit);
	    if (!domore)
	      return;
	}
    }
}

void
set_play_rate(slow, fast)
int slow, fast;
{
    if (slow < 0 || fast < 0 || fast < slow) {
	run_warning("Bad play rates slow=%d fast=%d, ignoring", slow, fast);
	return;
    }
    slow_play_rate = slow;
    fast_play_rate = fast;
}

/* Resignation, possibly giving away any remaining units. */

void
resign_game(side, side2)
Side *side, *side2;
{
    /* Nothing to do if we're not in the game. */
    if (!side->ingame)
      return;
    notify_all_of_resignation(side, side2);
    side_loses(side, side2, -1);
}

/* This is true if there is any kind of realtime limit on the game. */

int
realtime_game()
{
    return (g_rt_for_game() > 0
    	    || g_rt_per_side() > 0
    	    || g_rt_per_turn() > 0);
}

/* Pass NULL to see if all sides are now willing to save the game. */

int
all_others_willing_to_save(side)
Side *side;
{
    Side *side2;

    for_all_sides(side2) {
	if (side != side2 && !side2->willingtosave)
	  return FALSE; 
    }
    return TRUE;
}

/* Pass NULL to see if all sides are now willing to declare a draw. */

int
all_others_willing_to_quit(side)
Side *side;
{
    Side *side2;

    for_all_sides(side2) {
	if (side != side2 && !side2->willingtodraw)
	  return FALSE; 
    }
    return TRUE;
}

/* This forces an end to the game directly. */

void
end_the_game()
{
    Side *side;

    Dprintf("The game is over.\n");
    /* Make sure everybody sees this. */
    notify_all("END OF THE GAME!");
    record_event(H_GAME_ENDED, ALLSIDES);
    /* Set the global that indicates the game is over for everybody. */
    endofgame = TRUE;
    end_history();
    /* (should compute final scoring) */
    record_into_scorefile();
    /* Done with internal state change, now echo onto displays. */
    for_all_sides(side) {
    	if (side_has_display(side)) {
    	    update_turn_display(side, TRUE);
    	    /* (should update side's view of all sides?) */
    	    update_side_display(side, side, TRUE);
    	}
    }
    dump_statistics();
    /* We've done everything the kernel needs to have done; it's now up to
       the interface to decide when to exit. */
    ok_to_exit = TRUE;
}
