/* Implements loading/saving setting to a rc file.
 *
 * gsumi version 0.5
 *
 * Copyright 1997 Owen Taylor <owt1@cornell.edu>
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "lispish.h"
#include "gsumi.h"

Node *rc_node = NULL;
int rc_bad = 0;
char *rc_name = NULL;

static char *axis_use_strings[GDK_AXIS_LAST] = 
{
  "",
  "x",
  "y",
  "pressure",
  "x-tilt",
  "y-tilt"
};

/* find a device structure, insert if it doesn't exist */
static Node *
find_device_rc(char *devname)
{
  Node *n = node_list_find_match(rc_node, "device",0, devname,1, NULL);

  if (!n)
    {
      n = node_list_new();
      node_list_append(n,node_string_new("device"));
      node_list_append(n,node_string_new(devname));
      node_list_append(rc_node,n);
    }
  return n;
}

static void
update_tool_device(Node *device_rc, char *name)
{
  int i, j;
  ToolInfo *tool = NULL;
  Node *tmp_node;
  char *val;
    
  for (i=0;i<num_tools;i++)
    if (!strcasecmp(tools[i]->name,name))
      tool = tools[i];
  
  if (!tool) return;

  val = node_list_lookup_pair(device_rc,"tool");
  if (val)
    {
      if (!strcmp(val,"pen"))
	tool->tool = PEN_TOOL;
      else if (!strcmp(val,"eraser"))
	tool->tool = ERASER_TOOL;
      else
	{
	  fprintf(stderr,"Unknown tool type %s, setting to pen.\n",val);
	  tool->tool = PEN_TOOL;
	}
    }

  val = node_list_lookup_pair(device_rc,"size");
  if (val)
    {
      tool->size = atof(val);
    }

  val = node_list_lookup_pair(device_rc,"sens");
  if (val)
    {
      tool->sens = atof(val);
    }

  val = node_list_lookup_pair(device_rc,"aspect");
  if (val)
    {
      tool->aspect = atof(val);
    }

  val = node_list_lookup_pair(device_rc,"angle");
  if (val)
    {
      tool->angle = atof(val);
    }

  tool->use_threshhold = FALSE;
  val = node_list_lookup_pair(device_rc,"threshhold");
  if (val)
    {
      tool->use_threshhold = TRUE;
      tool->threshhold = atof(val);
    }

  tmp_node = node_list_find_match(device_rc, "profile", 0, 0);
  if (tmp_node)
    {
      tmp_node = tmp_node->c.l->next;
      j = 0;
      while (tmp_node && j<32)
	{
	  if (tmp_node->type != NODE_STRING)
	    break;
	  tool->profile[j] = atof(tmp_node->c.s);
	  tmp_node = tmp_node->next;
	  j++;
	}
    }
  
  tool_box_set_values(tool->id);
}

/* Copied from gtk_menu_factory_parse_accelerator() */
static void
gsumirc_parse_accelerator (const char   *accelerator,
			   GdkDeviceKey *key)
{
  int done;

  g_return_if_fail (accelerator != NULL);
  g_return_if_fail (key != NULL);

  key->modifiers = 0;

  done = FALSE;
  while (!done)
    {
      if (strncmp (accelerator, "<shift>", 7) == 0)
        {
          accelerator += 7;
          key->modifiers |= GDK_SHIFT_MASK;
        }
      else if (strncmp (accelerator, "<alt>", 5) == 0)
        {
          accelerator += 5;
          key->modifiers |= GDK_MOD1_MASK;
        }
      else if (strncmp (accelerator, "<control>", 9) == 0)
        {
          accelerator += 9;
          key->modifiers |= GDK_CONTROL_MASK;
        }
      else
        {
          done = TRUE;
	  /* Tricky, but works... ("" => keyval = 0, or no translation) */
          key->keyval = accelerator[0];
        }
    }
}

static void
update_xinput_device(Node *device_rc, char *name)
{
  char *val;
  int j;

  GList *tmp_list;
  GdkDeviceInfo *info;
  GdkAxisUse *new_axes;

  tmp_list = gdk_input_list_devices();

  info = NULL;
  while (tmp_list)
    {
      if (!strcasecmp(((GdkDeviceInfo *)tmp_list->data)->name, name))
	info = (GdkDeviceInfo *)tmp_list->data;
      tmp_list = tmp_list->next;
    }
  if (!info) return;

  val = node_list_lookup_pair(device_rc,"mode");
  if (val)
    {
      GdkInputMode mode;
      if (!strcmp(val,"disabled"))
	mode = GDK_MODE_DISABLED;
      else if (!strcmp(val,"screen"))
	mode = GDK_MODE_SCREEN;
      else if (!strcmp(val,"window"))
	mode = GDK_MODE_DISABLED;
      else
	{
	  fprintf(stderr,"Unknown mode %s, setting to disabled.\n",val);
	  mode = GDK_MODE_DISABLED;
	}
      if (mode != info->mode)
	{
	  if (gdk_input_set_mode (info->deviceid,mode))
	    {
	      if (info->mode == GDK_MODE_DISABLED)
		area_disable_device(NULL, info->deviceid);
	      else
		area_enable_device (NULL, info->deviceid);
	    }
	}
    }

  new_axes = g_new (GdkAxisUse, info->num_axes);
  for (j=0;j<info->num_axes;j++)
    new_axes[j] = info->axes[j];

  for (j=GDK_AXIS_X; j<GDK_AXIS_LAST; j++)
    {
      char *use;
      Node *n;

      use = axis_use_strings[j];
      n = node_list_find_match(device_rc, "axis", 0, use, 0, NULL);

      if (n)
	{
	  int axis = j-1;
	  Node *sn = n->c.l->next->next;
	  if (sn)
	    {
	      if (sn->type == NODE_STRING)
		axis = atoi(sn->c.s);
	    }
	  new_axes[axis] = j;
	}
    }

  gdk_input_set_axes (info->deviceid, new_axes);
  g_free (new_axes);

  for (j=0;j<info->num_keys;j++)
    {
      char buf[16];
      Node *n;
      
      sprintf(buf, "%d", j);
      n = node_list_find_match(device_rc, "key", 0, buf, 0, NULL);
		
      if (n)
	{
	  Node *sn = n->c.l->next->next;
	  
	  if (sn)
	    {
	      if (sn->type == NODE_STRING)
		{
		  GdkDeviceKey key;
		  gsumirc_parse_accelerator (sn->c.s, &key);
		  
		  gdk_input_set_key (info->deviceid, j,
				     key.keyval, key.modifiers);
		}
	    }
	}
    }

}

void
gsumirc_update_devices(void)
{
  Node *n;

  if (rc_bad || !rc_node)
    return;
  
  n = rc_node->c.l;
  while (n)
    {
      if (n->type == NODE_LIST &&
	  n->c.l->type == NODE_STRING &&
	 !strcmp(n->c.l->c.s,"device"))
	{
	  char *name = NULL;
	  if (n->c.l->next &&
	      n->c.l->next->type == NODE_STRING)
	    {
	      name = n->c.l->next->c.s;
	      update_xinput_device(n,name);
	      update_tool_device(n,name);
	    }
	}

      n = n->next;
    }
}

void
gsumirc_update_menus()
{
  Node *n;
  Node *nn;

  if (rc_bad || !rc_node)
    return;
  
  n = rc_node->c.l;
  while (n)
    {
      if (n->type == NODE_LIST &&
	  n->c.l->type == NODE_STRING &&
	  !strcmp(n->c.l->c.s,"menu-path"))
	{
	  gchar *path = NULL;
	  gchar *accel = NULL;
	  
	  nn = n->c.l->next;
	  if (nn && nn->type == NODE_STRING)
	    {
	      path = nn->c.s;
	      nn = nn->next;
	    }
	  if (nn && nn->type == NODE_STRING)
	    {
	      accel = nn->c.s;
	      nn = nn->next;
	    }
	  if (path && accel)
	    menus_add_path(path,accel);
	}
      
      n = n->next;
    }
}

void
load_gsumirc()
{
  FILE *file;
  char *home = getenv("HOME");

  rc_name = malloc(sizeof(char)*(strlen(home)+strlen("/.gsumirc") + 1));
  strcpy(rc_name,home);
  strcat(rc_name,"/.gsumirc");

  file = fopen(rc_name,"r");
  if (!file)
    {
      rc_node = NULL;
      rc_bad = 0;
    }
  else
    {
      rc_node = node_list_from_file(file,1);
      if (!rc_node)
	{
	  fprintf(stderr,"Error reading gsumirc, you wil not be able to\n"
		  "save an updated version.\n");
	  rc_bad = 1;
	}
      else
	rc_bad = 0;

      fclose(file);
    }
}

static void
replace_rc_pair(Node *device_rc, char *key, Node *new_pair)
{
  Node *old_pair;
  old_pair = node_list_remove_pair(device_rc,key);
  if (old_pair)
    node_free(old_pair);
  node_list_append(device_rc,new_pair);
}

static void
update_to_gsumirc()
{
  int i,j;
  Node *device_rc;
  GList *tmp_list;
  GdkDeviceInfo *info;

  if (!rc_node)
    rc_node = node_list_new();
  
  /* loop through active tools */
  for (i=0;i<num_tools;i++)
    {
      Node *tmp_node;
      char buffer[20];
      
      device_rc = find_device_rc(tools[i]->name);

      replace_rc_pair(device_rc,"tool",
          node_pair_string("tool", 
			   tools[i]->tool == PEN_TOOL ? "pen" : "eraser"));
      replace_rc_pair(device_rc,"size",
          node_pair_double("size", tools[i]->size));
      replace_rc_pair(device_rc,"sens",
          node_pair_double("sens", tools[i]->sens));
      replace_rc_pair(device_rc,"aspect",
          node_pair_double("aspect", tools[i]->aspect));
      replace_rc_pair(device_rc,"angle",
          node_pair_double("angle", tools[i]->angle));
      
      tmp_node = node_list_remove_pair(device_rc,"threshhold");
      if (tmp_node)
	node_free(tmp_node);

      if (tools[i]->use_threshhold)
	node_list_append(device_rc,
          node_pair_double("threshhold", tools[i]->threshhold));

      tmp_node = node_list_remove_match(device_rc, "profile", 0, 0);
      if (tmp_node)
	node_free(tmp_node);

      tmp_node = node_list_new();
      node_list_append(tmp_node, node_string_new("profile"));
      for (j=0; j<32; j++)
	{
	  sprintf(buffer,"%g",tools[i]->profile[j]);
	  node_list_append(tmp_node,node_string_new(buffer));
	}

      node_list_append(device_rc, tmp_node);
    }
  /* and through the XInput devices */

  tmp_list = gdk_input_list_devices();
  while (tmp_list) 
    {
      char *mode = NULL;	/* sop to gcc */
      
      info = (GdkDeviceInfo *)(tmp_list->data);
      device_rc = find_device_rc(info->name);
      
      switch (info->mode)
	{
	case GDK_MODE_DISABLED:
	  mode = "disabled";
	  break;
	case GDK_MODE_SCREEN:
	  mode = "screen";
	  break;
	case GDK_MODE_WINDOW:
	  mode = "window";
	  break;
	default:
	  g_error("Unknown mode %d",info->mode);
	}

	replace_rc_pair(device_rc,"mode", node_pair_string("mode", mode));

	for (j=GDK_AXIS_X; j<GDK_AXIS_LAST; j++)
	  {
	    char *use;
	    char buf[16];
	    Node *n;

	    use = axis_use_strings[j];
	    n = node_list_remove_match(device_rc, "axis", 0, use, 0, NULL);

	    if (n)
	      node_free(n);

	    for (i=0; i<info->num_axes;i++)
	      if (info->axes[i] == j)
		{
		  n = node_list_new();
		  node_list_append(n,node_string_new("axis"));
		  node_list_append(n,node_string_new(use));
		  
		  sprintf(buf,"%d",i);
		  node_list_append(n,node_string_new(buf));
		  
		  node_list_append(device_rc,n);

		  break;
		}
	  }

	for (j=0; j<info->num_keys; j++)
	  {
	    if (info->keys[j].keyval)
	      {
		char buf[16];
		char *accel;
		Node *n;
	    
		sprintf(buf, "%d", j);

		n = node_list_remove_match(device_rc, "key", 0, buf, 0, NULL);
		
		if (n)
		  node_free(n);
	    
		n = node_list_new();
		node_list_append(n,node_string_new("key"));
		node_list_append(n,node_string_new(buf));

		accel = menus_stringify_accelerator (info->keys[j].keyval,
						     info->keys[j].modifiers);
		node_list_append(n,node_string_new(accel));
		g_free (accel);

		node_list_append(device_rc,n);
	      }
	  }

	tmp_list = tmp_list->next;
      }

  /* and menu items */
  menus_store();
}

void
gsumirc_store_menu_path(char *path, char *accel)
{
  Node *on = node_list_remove_match(rc_node, "menu-path", 0, path, 0, NULL);
  Node *nn;

  nn = node_list_new();
  node_list_append(nn,node_string_new("menu-path"));
  node_list_append(nn,node_string_new(path));
  node_list_append(nn,node_string_new(accel));
  node_list_append(rc_node,nn);

  /* We wait until afterwards before freeing this because some elements
     of it might have been stored in the menu hash! */

  if (on)
    node_free(on);
}

void
save_gsumirc()
{
  FILE *file;

  if (rc_bad)
    {
      fprintf(stderr,"Error reading gsumirc, not saved\n");
      return;
    }

  update_to_gsumirc();

  /* write new information to disk */

  file = fopen(rc_name,"w");
  if (!file)
    {
      fprintf(stderr,"Can't open '%s' to save\n",rc_name);
    }
  else
    {
      node_list_to_file(rc_node,file,0);
      fclose(file);
    }
}

