/* This file is adapted from menus.c from:
 *
 * The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "gsumi.h"

/* menu callbacks */

static void undo_callback(GtkWidget *w, gpointer data);
static void redo_callback(GtkWidget *w, gpointer data);
static void zoom_in_callback(GtkWidget *w, gpointer data);
static void zoom_out_callback(GtkWidget *w, gpointer data);
static void save_options_callback(GtkWidget *w, gpointer data);
static void clear_image_callback(GtkWidget *w, gpointer data);

/* local functions */

static void menus_init (void);
static void menus_foreach (gpointer key,
			   gpointer value,
			   gpointer user_data);
static gint menus_install_accel (GtkWidget *widget,
				 gchar     *signal_name,
				 gchar      key,
				 gchar      modifiers,
				 gchar     *path);
static void menus_remove_accel (GtkWidget *widget,
				gchar     *signal_name,
				gchar     *path);


static GtkMenuEntry menu_items[] =
{
  { "File/New", "<control>N", do_new, NULL },
  { "File/Open", "<control>O", do_open, NULL },
  { "File/Save", "<control>S", do_save, NULL },
  { "File/Save As...", NULL, do_save_as, NULL },
  { "File/Input Dialog...", NULL, do_input_dialog, NULL },
  { "File/Toolbox...", NULL, do_tool_box, NULL },
  { "File/Save Options", NULL, save_options_callback, NULL },
  { "File/<separator>", NULL, NULL, NULL },
  { "File/Quit", "<control>Q", do_exit, NULL },

  { "Image/Undo", "<control>Z", undo_callback, NULL },
  { "Image/Redo", "<control>R", redo_callback, NULL },
  { "Image/Zoom In", "=", zoom_in_callback, NULL },
  { "Image/Zoom Out", "-", zoom_out_callback, NULL },
  { "Image/<separator>", NULL, NULL, NULL },
  { "Image/Clear", "<control>.", clear_image_callback, NULL },
};

static int nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);

static int initialize = TRUE;
static GtkMenuFactory *factory = NULL;
static GHashTable *entry_ht = NULL;

static void 
undo_callback(GtkWidget *w, gpointer data)
{
  undo();
}

static void
redo_callback(GtkWidget *w, gpointer data)
{
  redo();
}

static void
zoom_in_callback(GtkWidget *w, gpointer data)
{
  decrement_scale();
}
static void 
zoom_out_callback(GtkWidget *w, gpointer data)
{
  increment_scale();
}
static void 
save_options_callback(GtkWidget *w, gpointer data)
{
  save_gsumirc();
}

static void 
clear_image_callback(GtkWidget *w, gpointer data)
{
  rect rec;
  rec.x0 = 0;
  rec.x1 = bitmap->width;
  rec.y0 = 0;
  rec.y1 = bitmap->height;
  
  undo_begin();
  bitmap_update (&rec);
  undo_update(&rec);
  bitmap_clear();
}

void
menus_get_main_menubar (GtkWidget           **menubar,
			GtkAcceleratorTable **table)
{
  if (initialize)
    menus_init ();

  if (menubar)
    *menubar = factory->widget;
  if (table)
    *table = factory->table;
}

void
menus_create (GtkMenuEntry *entries,
	      int           nmenu_entries)
{
  char *accelerator;
  int i;

  if (initialize)
    menus_init ();

  if (entry_ht)
    for (i = 0; i < nmenu_entries; i++)
      {
	accelerator = g_hash_table_lookup (entry_ht, entries[i].path);
	if (accelerator)
	  {
	    if (accelerator[0] == '\0')
	      entries[i].accelerator = NULL;
	    else
	      entries[i].accelerator = accelerator;
	  }
      }

  gtk_menu_factory_add_entries (factory, entries, nmenu_entries);

  for (i = 0; i < nmenu_entries; i++)
    if (entries[i].widget)
      {
	gtk_signal_connect (GTK_OBJECT (entries[i].widget), "install_accelerator",
			    (GtkSignalFunc) menus_install_accel,
			    entries[i].path);
	gtk_signal_connect (GTK_OBJECT (entries[i].widget), "remove_accelerator",
			    (GtkSignalFunc) menus_remove_accel,
			    entries[i].path);
      }
}

void
menus_set_sensitive (char *path,
		     int   sensitive)
{
  GtkMenuPath *menu_path;

  if (initialize)
    menus_init ();

  menu_path = gtk_menu_factory_find (factory, path);
  if (menu_path)
    gtk_widget_set_sensitive (menu_path->widget, sensitive);
  else
    g_warning ("Unable to set sensitivity for menu which doesn't exist: %s", path);
}

void
menus_add_path (char *path,
		char *accelerator)
{
  if (!entry_ht)
    entry_ht = g_hash_table_new (g_str_hash, g_str_equal);

  g_hash_table_insert (entry_ht, path, accelerator);
}

void
menus_store ()
{
  if (!entry_ht)
    return;

  g_hash_table_foreach (entry_ht, menus_foreach, NULL);
}


static void
menus_init ()
{
  if (initialize)
    {
      initialize = FALSE;

      factory = gtk_menu_factory_new (GTK_MENU_FACTORY_MENU_BAR);

      menus_create (menu_items, nmenu_items);
    }
}

static void
menus_foreach (gpointer key,
	       gpointer value,
	       gpointer user_data)
{
  gsumirc_store_menu_path((char *)key, (char *)value);
}

gchar *
menus_stringify_accelerator (guint keyval, 
			     GdkModifierType modifiers)
{
  char accel[64];
  char t2[2];

  accel[0] = '\0';
  if (modifiers & GDK_CONTROL_MASK)
    strcat (accel, "<control>");
  if (modifiers & GDK_SHIFT_MASK)
    strcat (accel, "<shift>");
  if (modifiers & GDK_MOD1_MASK)
    strcat (accel, "<alt>");
  
  t2[0] = keyval;
  t2[1] = '\0';
  strcat (accel, t2);

  return g_strdup (accel);
}

static gint
menus_install_accel (GtkWidget *widget,
		     gchar     *signal_name,
		     gchar      key,
		     gchar      modifiers,
		     gchar     *path)
{
  char *accel, *t1;

  accel = menus_stringify_accelerator (key, modifiers);

  if (entry_ht)
    {
      t1 = g_hash_table_lookup (entry_ht, path);
      g_free (t1);
    }
  else
    entry_ht = g_hash_table_new (g_str_hash, g_str_equal);

  g_hash_table_insert (entry_ht, path, accel);

  return TRUE;
}

static void
menus_remove_accel (GtkWidget *widget,
		    gchar     *signal_name,
		    gchar     *path)
{
  char *t;

  if (entry_ht)
    {
      t = g_hash_table_lookup (entry_ht, path);
      g_free (t);

      g_hash_table_insert (entry_ht, path, g_strdup (""));
    }
}
