/****************************************************************************
 * NCSA HDF                                                                 *
 * National Comptational Science Alliance                                   *
 * University of Illinois at Urbana-Champaign                               *
 * 605 E. Springfield, Champaign IL 61820                                   *
 *                                                                          *
 * For conditions of distribution and use, see the accompanying             *
 * hdf/COPYING file.                                                        *
 *                                                                          *
 ****************************************************************************/

package ncsa.hdf.jhv;

import java.awt.image.*;
import java.lang.Thread;
import java.awt.*;
import java.awt.event.*;
import java.io.*;

import ncsa.hdf.hdflib.*;
import ncsa.hdf.awt.image.*;
import ncsa.hdf.awt.*;
import ncsa.hdf.awt.event.*;
import ncsa.hdf.message.*;


/* This class will create the spreadsheet to display the dataset value. 
 * Port to jdk-1.1 by apu. 
 * @version  1.00 
 * @auther   Xinjian Lu
 */
public class JHVDataFrame extends Frame implements AdjustmentListener, 
  ActionListener,
  ItemListener{
  
    /** the frame of the image */      
    JHVImageFrame   imgFrame;
    
    /** the canvas of the image */
    JHVImageCanvas  imgCanvas;
    
    /** the canvas of the  displayed data */
    JHVDataCanvas	  dataCanvas;
    
    /** the canvas to display the dimension scale value of the dataset */ 
    JHVDataRangeCanvas	rowInfoCanvas,  colnumInfoCanvas;
    
    /** the scrollbar associated with the spreadsheet */
    Scrollbar	hScrollbar, vScrollbar;
    
    /** the flag to indicate if the mouse is in draged status */
    boolean		dragFlag=false;
    
    /** factor value */
    TextField factor = null;
    
    /** image operation */
    Checkbox  imageStyle = null;

    /** Constructor of the class.
     * @param frame the object.
     * @param rect  the rectangle to specify the range of the dataset 
     */ 
    
  public JHVDataFrame(JHVImageFrame frame, Rectangle rect) {
    // set the image canvas to let me know where the image is from
    imgCanvas    = frame.imageCanvas;
    
    // set image frame to let me communicate with that with the event
    // generated by the spreadsheet
    imgFrame     = frame;

    // set frame title
    setTitle();
	
    // create new JHVDataCanvas(actual spreadsheet panel)
    dataCanvas = new JHVDataCanvas(this,rect);

    // set canvas getSize(default, which size is best?)
    dataCanvas.setCanvasSize(2*256,256);
	
    // create the canvas to display the dimension scale information(row)
    rowInfoCanvas = new JHVDataRangeCanvas(this, JHVDataRangeCanvas.VERTICAL);

    // another canvas to hold the dimension scale  information(colnum)
    colnumInfoCanvas=new JHVDataRangeCanvas(this,JHVDataRangeCanvas.HORIZONTAL);

    // create spreadsheet and control panel(Graphical User Interface)
    createSpreadsheetGUI();

    // check the spreadsheet size
    dataCanvas.checkSize();
    this.addWindowListener(new WindowClosedProcess());
  }

  /** setup the frame tittle */
  public void setTitle() {
    String winString = "Image Data from: ";
    String tmpStr    = new String(imgCanvas.app.hdfFile);
    // set title	
    super.setTitle(winString.concat(tmpStr));
  }

  /** create Spreadsheet */
  public void createSpreadsheetGUI() {

    // set Layout Manager
    setLayout(new BorderLayout());

    // spreadsheet panel
    Panel	sPanel = new Panel();
    sPanel.setLayout(new BorderLayout());	
	
    // Horizontal & vertical Scrollbar 
    hScrollbar = new Scrollbar(Scrollbar.HORIZONTAL);
    hScrollbar.addAdjustmentListener(this);
    vScrollbar = new Scrollbar();
    vScrollbar.addAdjustmentListener(this);

    //Panel rPanel = new Panel();
    //rPanel.add("North", colnumInfoCanvas);

    sPanel.add("North", colnumInfoCanvas);	
    sPanel.add("West",  rowInfoCanvas);
    sPanel.add("Center",dataCanvas);
    sPanel.add("East",  vScrollbar);
    sPanel.add("South", hScrollbar);

    // set Control panel
    Panel ctrPanel1 = new Panel();
	
    // set layout
    ctrPanel1.setLayout(new GridLayout(1,0,20,5));
    
    // append control

    ctrPanel1.add(new Label("Expansion factor:", Label.RIGHT));
    ctrPanel1.add(factor = new TextField("2"));
    
    imageStyle = new Checkbox("Interpolate");
    ctrPanel1.add(imageStyle);
    imageStyle.addItemListener(this);
    imageStyle.setState(false);
    
    Button MakeImageButton = new Button("Make Image");
    Button DismissButton = new Button("Dismiss");
    MakeImageButton.addActionListener(this);
    DismissButton.addActionListener(this);
    ctrPanel1.add(MakeImageButton);
    ctrPanel1.add(DismissButton);
    
  
    //ctrPanel1.add(new Button("Double"));
    //ctrPanel1.add(new Button("Half"));

    /***************************************************
    // set Control panel
    Panel ctrPanel2 = new Panel();
    // append control
    ctrPanel2.add(new Label("Expansion factor:", Label.RIGHT));
    ctrPanel2.add(factor = new TextField("8"));
    ctrPanel2.add(new Button("Make Image"));
    ctrPanel2.add(new Button("Dismiss"));
    *****************************************************/
    
    // set control
    Panel ctrPanel = new Panel();
	
    // set Layout Manager
    ctrPanel.setLayout(new BorderLayout());
    ctrPanel.add("North", ctrPanel1);
    //ctrPanel.add("South", ctrPanel2);

    // set Spreadsheet
    add("Center", sPanel);
    add("South",  ctrPanel);	
  }

  /** popup the new frame actually */
  public void popup() {

    // reset the byte data
    HDFObject.object2byte(imgCanvas.hdfData, imgCanvas.hdfDataType,
        null,imgCanvas.imageData);

    // paint dataspread sheet
    dataCanvas.repaint();
    
    // show component of the frame
    //setVisible(true);
    pack();
    setSize(3*256, 2*256);
    show();
    
    // compute the cell position
    dataCanvas.computeCellPosition();
    
  }

  /** 
   * get the font for the spreadsheet 
   *@see #getFont
   */
  public Font getFont() {
    // default
    return( new Font("Fixed", Font.PLAIN, 14));
    
  }

  public void adjustmentValueChanged(AdjustmentEvent e)
  {
    int type = e.getAdjustmentType();
    Scrollbar target = (Scrollbar)e.getAdjustable();
    switch (type) {
	
    case AdjustmentEvent.UNIT_INCREMENT :
    case AdjustmentEvent.UNIT_DECREMENT :
    case AdjustmentEvent.BLOCK_INCREMENT :
    case AdjustmentEvent.BLOCK_DECREMENT :
    case AdjustmentEvent.TRACK :
	
      // detect the vertical scrollbar
      if (target ==  vScrollbar) {
	
        // if select the spreadsheet then redraw it
	// set drawFlag
	dataCanvas.drawFlag = false;
	imgCanvas.drawSubsetFlag = false;

	// get offset
	//dataCanvas.ty = ((Integer) evt.arg).intValue();
	dataCanvas.ty = vScrollbar.getValue();
	
	// set scrollbar offset to display the correct data
	dataCanvas.setVoffset();		
		
	// adjust the horizontal scrollbar
	// dataCanvas.setVScrollbarValue();
	    
	// repaint the graphics
	dataCanvas.repaint();
      }
	    
      // detect the horizontal scrollbar
      if (target ==  hScrollbar) {
	    
	// if select the spreadsheet then redraw it
	// set drawFlag
	dataCanvas.drawFlag = false;
	imgCanvas.drawSubsetFlag = false;

        // get offset
	// dataCanvas.tx = ((Integer) evt.arg).intValue();
	dataCanvas.tx = hScrollbar.getValue();
	
	// set horizontal scrollbar offset
	dataCanvas.setHoffset();
	    
	// adjust the horizontal scrollbar
	// dataCanvas.setHScrollbarValue();
	
	// repaint the graphics
	dataCanvas.repaint();
      }
    } // switch(evt.id)

  }

  /**
   * Called if an action occurs in the Component
   * @param evt the event
   * @param what the action that's occuring
   * @see java.awt.Component#action
   */

  public void actionPerformed(ActionEvent e)
  {
    String arg = e.getActionCommand();

    if ("Dismiss".equals(arg)) {
      // Data Frame will be distroyed
      
      dispose();
    } 
	
    if ("Make Image" .equals(arg)) {
         setCursor(new Cursor(Cursor.WAIT_CURSOR));
	 if (imageStyle.getState()) {
	   // create interpolate image 
	   dataCanvas.displayInterpolateImage();
	 
	 } else {
	   // create the sample image
	    dataCanvas.displaySampleImage();
         }
         setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    }
    

  }

  public void setCursor(Cursor cursor) {

    super.setCursor(cursor);

    // default toolkit, make cursor effective
    Toolkit.getDefaultToolkit().sync();
  }

  public void itemStateChanged(ItemEvent e)
  {
    if (dataCanvas.imageFrame != null) {
      if (imageStyle.getState()) {
	// interpolated image
	dataCanvas.imageFrame.imageCanvas.setImage(dataCanvas.interpolateImage);
      }
      else 
	dataCanvas.imageFrame.imageCanvas.setImage(dataCanvas.defaultImage);
      }
    
  }
} // end of class JHVDataFrame

/** This class is the implementation of the spreadsheet of the HDF object */
class JHVDataCanvas extends Canvas
                   implements MouseListener, MouseMotionListener {

  // the frame of the spreadsheet
  JHVDataFrame	datFrame;

  // who is the object ancestor?
  JHV		app;

  // the image canvas that the spreadsheet will communicate with
  JHVImageCanvas	imgCanvas;
   
  // dataset value of the selected image area on the image
  // take the data  to be as float so it can be easilly processed 
  float[][]       data = null;

  // the generated image from the selected area of the spreadsheet
  Image           image = null;

  /** the image that is the duplicate pixel value of the subset of the orignal image */
  Image  defaultImage = null;
    
  /** the image that has been processed to be smooth */
  Image  interpolateImage = null;
  
  // the image width & width
  int		imgWidth,imgHeight;
	
  // the current canvas size
  int           canvasWidth ;
  int           canvasHeight;
    
  // the cell size of the spreadsheet
  public  static  Rectangle CellRect = null ;

  // how many lines?
  public  static  int cellNumberByRow;

  // how many colnums?
  public  static  int cellNumberByColnum;
    
  // 10 digits display for each cell 
  public  static  int NumberLength = 18;
	
  // create the new font with the specified font size
  public static   int defaultFontSize = 14;

  // Create the new font for displaying the  data correctly 
  // in the specified canvas within the scrollbar
  Font dataFont        = new Font("Times", Font.PLAIN, defaultFontSize);

  // default font width & height
  public static   int fontWidth = 14;
  public static   int fontHeight= 15;

  // Flicker-free update with translation by scrollbars
  Image 		offScreenImage = null;
  Dimension 	offscreensize;		
  Graphics 	offGraphics;

  // the indicator of the mouse that had been draged, pressed.
  boolean		dragFlag=false;

  //  the indicator of the spreadsheet frame
  boolean frameDisplayed = false;
    
  // translated value
  // All subsequent operations on this graphics context will be relative
  // to this origin.
  int tx = 0;
  int ty = 0;
    
  // offset value of the scrollbar
  int hOffset = 0;
  int vOffset = 0;
    
  // dataset range
  Rectangle datasetRange = new Rectangle(1,1,1,1);
    
  // the position and size of each cell in the spreadsheet
  Rectangle [][] cells = new Rectangle[128][20];
    
  // the indicator that the spreadsheet has been selected 
  boolean drawFlag = false;
    
  // draw area
  Rectangle drawRectangle = null;
    
  // select area for subsetting
  Rectangle  selectedDataset = new Rectangle();
    
  /** the seprate image frame */
  ImageFrame  imageFrame = null;
    
  /** the rectangle to draw on the image by responding to the data subselected */ 
  Rectangle  subsetArea = new Rectangle();

  public JHVDataCanvas(JHVDataFrame frame, Rectangle rect) {

    // set spreadsheet frame 
    datFrame = frame;

    // set the applet wich the current object belongs to
    app      = frame.imgCanvas.app;

    //set the image canvas associated with data canvas
    imgCanvas= frame.imgCanvas;

    frameDisplayed = false;
    
    // set font
    setFont(dataFont);
	
    // set cell size based on the couurent selected font.
    CellRect = new Rectangle(NumberLength*(fontWidth+1), fontHeight + 4);

    // set the dataset range
    datasetRange.setBounds(rect.x,rect.y,rect.width,rect.height);

    // prepare the spreadsheet data
    getSpreadsheetData();
    this.addMouseListener(this);
    this.addMouseMotionListener(this);
  }

  /** set size for current canvas 
   * @param w the width
   * @param h the height
   */
  public void setCanvasSize(int w,int h) {
    // set canvas size
    canvasWidth = w;
    canvasHeight= h;
  }

  // Set minimum size at SxS for current canvas  
  public Dimension getMinimumSize(){
    return new Dimension(canvasWidth, canvasHeight);
  }

  //  Set preferred  size at SxS for current canvas  
  public Dimension getPreferredSize() {
    return getMinimumSize();
  }

  /** return integer based on the string */
  public static int getInteger(String str) {
	
    try {
      return Integer.parseInt(str);	
    } catch (Exception e) {
      return 0;
    }
  }

  /** Prepare the spreadsheet data.  */
  public  void getSpreadsheetData() {
	
	getSpreadsheetData((Object)imgCanvas.hdfData,(int)imgCanvas.hdfDataType);   
  } 

  /** Prepare the spreadsheet data.  */
  public  void getSpreadsheetData(byte[] hdfData, int nt) {

    HDFNativeData convert = new HDFNativeData();
 
    // specify the data size
    data = null;
    data = new float[datasetRange.height][datasetRange.width];
    
    // data type of the raw data
    int datatype = nt;

    if ((datatype & HDFConstants.DFNT_LITEND) != 0) {
	datatype -= HDFConstants.DFNT_LITEND;
    }
    // data size for the original data based on the data number type
        int datasize = 0;
	try { datasize = HDFLibrary.DFKNTsize(datatype);
	} catch (HDFException e) {}
    
    // the image dataset range
    int w = imgCanvas.imageWidth;
    int h = imgCanvas.imageHeight;

    // get the first address to get the raw data from hdfData
    int firstPos = 0;
    if ((imgCanvas.node.type == HDFObjectNode.RIS24) ||
        (imgCanvas.node.type == HDFObjectNode.GRDATASET)) 
         // adjust the first data position
         firstPos = w * h * datasize * (imgCanvas.imageNumber - 1);
 
    // extract the dataset to be processed
    for (int i=0; i<datasetRange.height; i++) {
	
      // first extract number position by row
      int pos = ((datasetRange.y  + i) * w + (datasetRange.x )) * datasize;

      // adjust the pos.
      pos += firstPos;
	
      for (int j=0; j<datasetRange.width; j++) {
	
	switch(datatype) {
	// one bit char
	case HDFConstants.DFNT_CHAR:
	case HDFConstants.DFNT_UCHAR8:
	case HDFConstants.DFNT_UINT8:
	  data[i][j] = (float)((byte)hdfData[pos]);
	  // convert to positive if the number is negative 
	  if (data[i][j] < 0)  
	     data[i][j] += 256.0f;	
	  break;
		
	// signed integer (byte)	
	case HDFConstants.DFNT_INT8:
	  
	  data[i][j] = (float)((byte)hdfData[pos]);
	  break;
	  
        // short	
	case HDFConstants.DFNT_INT16:
	case HDFConstants.DFNT_UINT16:
	      
		Short shval = new Short(convert.byteToShort(hdfData,pos));
		data[i][j] = shval.floatValue();
	  break;
	    
	case HDFConstants.DFNT_INT32:
	case HDFConstants.DFNT_UINT32:
		
		Integer ival = new Integer(convert.byteToInt(hdfData,pos));
		data[i][j] = ival.floatValue();
	  break;
		  
	//case HDFConstants.DFNT_FLOAT:
	case HDFConstants.DFNT_FLOAT32:
	
		Float fval = new Float(convert.byteToFloat(hdfData,pos));
		data[i][j] = fval.floatValue();
	  break;
	    
	//case HDFConstants.DFNT_DOUBLE:
	case HDFConstants.DFNT_FLOAT64:
	
		Double dval = new Double(convert.byteToDouble(hdfData,pos));
		data[i][j] = dval.floatValue();
	  break;
	
	default:
	  data[i][j] = 0;
	}
	    
	// position moved
	pos += datasize;
	    
      }
    } 
    
  } 

 /** Prepare the spreadsheet data.  */
  public  void getSpreadsheetData(Object hdfData, int nt) {
 
    // specify the data size
    data = null;
    data = new float[datasetRange.height][datasetRange.width];
    
    // data type of the raw data
    int datatype = imgCanvas.hdfDataType;

    if ((datatype & HDFConstants.DFNT_LITEND) != 0) {
	datatype -= HDFConstants.DFNT_LITEND;
    }

    int datasize = 1;

    int w = imgCanvas.imageWidth;
    int h = imgCanvas.imageHeight;

    // get the first address to get the raw data from hdfData
    int firstPos = 0;
    if ((imgCanvas.node.type == HDFObjectNode.RIS24) ||
        (imgCanvas.node.type == HDFObjectNode.GRDATASET)) 
         // adjust the first data position
         firstPos = w * h * datasize * (imgCanvas.imageNumber - 1);
 
    // extract the dataset to be processed
    for (int i=0; i<datasetRange.height; i++) {
	
      // first extract number position by row
      int pos = ((datasetRange.y  + i) * w + (datasetRange.x )) * datasize;

      // adjust the pos.
      pos += firstPos;
	
      for (int j=0; j<datasetRange.width; j++) {
	
	switch(datatype) {
	// one bit char
	case HDFConstants.DFNT_CHAR:
	case HDFConstants.DFNT_UCHAR8:
	case HDFConstants.DFNT_UINT8:
	  byte bdat[] = (byte[])hdfData;
	  data[i][j] = (float)(bdat[pos]);
	  // convert to positive if the number is negative 
	  if (data[i][j] < 0)  
	     data[i][j] += 256.0f;	
	  break;
		
	// signed integer (byte)	
	case HDFConstants.DFNT_INT8:
	  byte bidat[] = (byte[])hdfData;
	  data[i][j] = (float)(bidat[pos]);
	  break;
	  
        // short	
	case HDFConstants.DFNT_INT16:
	case HDFConstants.DFNT_UINT16:
	        short sdat[] = (short[])hdfData;
		data[i][j] = (float)sdat[pos];
		// Short shval = new Short(convert.byteToShort(hdfData,pos));
		// data[i][j] = shval.floatValue();
	  break;
	    
	case HDFConstants.DFNT_INT32:
	case HDFConstants.DFNT_UINT32:
		int idat[] = (int[])hdfData;
		data[i][j] = (float)idat[pos];
		// Integer ival = new Integer(convert.byteToInt(hdfData,pos));
		// data[i][j] = ival.floatValue();
	  break;
		  
	//case HDFConstants.DFNT_FLOAT:
	case HDFConstants.DFNT_FLOAT32:
		float fdat[] = (float[])hdfData;
		data[i][j] = (float)fdat[pos];
		// Float fval = new Float(convert.byteToFloat(hdfData,pos));
		// data[i][j] = fval.floatValue();
	  break;
	    
	//case HDFConstants.DFNT_DOUBLE:
	case HDFConstants.DFNT_FLOAT64:
		double ddat[] = (double[])hdfData;
		data[i][j] = (float)ddat[pos];	
		// Double dval = new Double(convert.byteToDouble(hdfData,pos));
		// data[i][j] = dval.floatValue();
	  break;
	
	default:
	  data[i][j] = 0;
	}
	    
	// position moved
	pos += datasize;
	    
      }
    } 
    
  } 

  /** draw the spreadsheet */
  public synchronized void paint(Graphics g) {

    // get background color
    Color bColor = getBackground();

    // get current graphics color 
    Color gColor = g.getColor();
    
    // get current canvas size
    int w = getSize().width;
    int h = getSize().height;
    
    // compute the current lines & colnums based on the canvas size
    cellNumberByColnum = w / CellRect.width  ;
    cellNumberByRow    = h / CellRect.height ;
    
    //g.translate(-tx,-ty);

    // repaint the info panel
    datFrame.rowInfoCanvas.init(h-2);
    datFrame.colnumInfoCanvas.init(w-2);
    
    int startx = 1;
    int starty = 1;
	
    // draw the rectangle
    g.drawRect(startx,starty,w-2,h-2);

    // draw the selected area
    if (drawFlag) {
      // set background color
      g.setColor(Color.cyan);

      // draw cell
      if (drawRectangle != null)
	g.fillRect(drawRectangle.x, drawRectangle.y, 
		   drawRectangle.width-1, drawRectangle.height-1);
    }

    // set color
    g.setColor(Color.lightGray);
    
    // draw grid
    startx = 1;
    starty = 1;
   	
    // draw the grid (Horizontal)
    for (int i=1; i<= cellNumberByRow; i++) {
      starty += CellRect.height;
      //g.drawLine(startx+1, starty+1, w-3, starty+1);
      g.draw3DRect(startx+1, starty+1, w-3, 1, true);
    }
	
    startx = 1;
    starty = 1;
    // draw the grid (Vertical)
    for (int i=1; i<= cellNumberByColnum; i++) {
      startx += CellRect.width;
      g.draw3DRect(startx+1, starty+1, 1, h-3, true);
    }  
    
    // reset the color
    g.setColor(gColor);
  	
    for (int i =0; i<cellNumberByRow; i++) {
      for (int j=0; j<=cellNumberByColnum; j++) {
	// display position
	startx = CellRect.width * j + 4;
	starty = CellRect.height * (i+1) ;
	
	// display float number
	float dispNumber = 0;
	if (((vOffset+i) < datasetRange.height) &&
	    ((hOffset+j) < datasetRange.width))    {
		
	  // assign value
	  dispNumber = data[vOffset+i][hOffset+j];
         
	  g.drawString(Float.toString(dispNumber), startx, starty);
		
	}
	else
	  continue; 
      }
    } 

  }

  /**
   * Updates the component. This method is called in
   * response to a call to repaint. You can assume that
   * the background is not cleared.
   * @param g the specified Graphics window
   * @see java.awt.Component#update
   */
  public   void update(Graphics g) {

    Dimension d = getSize();
    setCanvasSize(d.width, d.height);
    
    if (offScreenImage == null) {
      // offScreenImage not created; create it.
      offScreenImage = createImage(d.width*2, d.height*2);	
      
      // get the off-screen graphics context    
      offGraphics    = offScreenImage.getGraphics();
      
      // set the font for offGraphics
      offGraphics.setFont(getFont()); 
    }
    
    // paint the background on the off-screen graphics context
    offGraphics.setColor(getBackground());
    offGraphics.fillRect(1,1,d.width-2,d.height-2);    
    offGraphics.setColor(getForeground());
    
    // draw the current frame to the off-screen 
    paint(offGraphics);

    //then draw the image to the on-screen 
    g.drawImage(offScreenImage, 0, 0, null);
	
  }

  /** return the specified space string */
  public String space(int size) {

    int tabWidth = 8;
    
    String space = "";
    for (; size >= tabWidth; size -= tabWidth) {
      space += "        ";
    }
    for (; size > 0; size--) {
      space += " ";
    }
    return space;
  }

  /** 
   * Sets the font of the component.
   * @param f the font
   */
  public synchronized void setFont(Font f) {
    // set Font for component
    super.setFont(f);
    
    // get new FontMetricsp
    FontMetrics fontMetrics = getFontMetrics(getFont());
    
    // set font width & height
    fontHeight = fontMetrics.getHeight();
    fontWidth  = fontMetrics.charWidth(' ');
    
  }

  /**
   * If the size of the spreadsheet is appropriate 
   * the spreadsheet will be displayed
   */
  public synchronized void checkSize() {
    
    // set canvas size
    setSize(canvasWidth, canvasHeight);

    if (!frameDisplayed) {
      // popup frame
      datFrame.popup();
      frameDisplayed = true;
    }
    
  }

  /** Reshapes the Component to the specified bounding box. 
   * @param x the x coordinate
   * @param y the y coordinate
   * @param width the width of the component
   * @param height the height of the component
   */
  public synchronized  void setBounds(int x, int y, int w, int h) {
  
    super.setBounds(x, y, w, h);
  
    // resize horizontal scrollbar
    setHScrollbarValue();
    
    // resize the vertical scrollbar
    setVScrollbarValue();
    
    // recompute the cell position
    computeCellPosition();
  }
  

  /** Compute the each cell position after the canvas changed */
  public void computeCellPosition() {
  
    // current canvas size
    int w = getSize().width;
    int h = getSize().height;
    
    // spreadsheet lines & colnums
    cellNumberByColnum = w / CellRect.width;
    cellNumberByRow    = h / CellRect.height;

    // specify the dimension for the cells
    //cells = new Rectangle[cellNumberByRow][cellNumberByColnum+1];
    
    int starty = 2;
    for (int i=0; i<cellNumberByRow; i++ ) {

      starty = CellRect.height*i + 3;
      
      int startx = 2;
      for (int j=0; j<=cellNumberByColnum; j++) {
	 
	// x position
	startx = CellRect.width*j + 3;

	// set rectangle
	if (cells[i][j] == null)
	  cells[i][j] = new Rectangle(startx, starty, 
				      CellRect.width, CellRect.height);
	else
	  cells[i][j].setBounds(startx, starty, CellRect.width, CellRect.height);
      }
    }
  }
    

  /** Adjust the Scrollbar value by the specifyed dataset range	 */
  void setHScrollbarValue() {

    // get current canvas size
    int canvasWidth = getSize().width;
    int canvasHeight= getSize().height;

    // canvas is valid?
    if ((canvasWidth <= 0)||(canvasHeight<=0)) { 
      return;
    }
    
    //Shift everything to the right if we're displaying empty space
    //on the right side.
    int width = datasetRange.width * CellRect.width ;
    
    if ((tx + canvasWidth) > width) {
      int newtx = width - canvasWidth;
      if (newtx < 0) {
	newtx = 0;
      }
      tx = newtx;
    }
    
int p = (int) (canvasWidth * 0.9);
int m = (int)(width - (canvasWidth - p));
    datFrame.hScrollbar.setValues(
	     //draw the part of the dataset that starts at this x:
	     tx, 
	     //amount to scroll for a "page":
	     p, //(int)(canvasWidth * 0.9), 
	     //minimum image x to specify:
	     0,
	     //maximum image x to specify:
	     m); //width - canvasWidth);

    //"visible" arg to setValues() has no effect after scrollbar is visible.
    datFrame.hScrollbar.setBlockIncrement(p); //(int)(canvasWidth * 0.9));
    
    return;
  }

  /** Adjust the Scrollbar value by the specifyed dataset range	 */
  void setVScrollbarValue() {

    // get current canvas size
    int canvasWidth = getSize().width;
    int canvasHeight= getSize().height;
    
    // canvas is valid?
    if ((canvasWidth <= 0)||(canvasHeight<=0)) { 
      return;
    }
        
    //Shift everything to the right if we're displaying empty space
    //on the right side.
    
    int height = datasetRange.height * CellRect.height ;
 
    if ((ty + canvasHeight) > height) {
      int newty = height - canvasHeight;
      if (newty < 0) {
	newty = 0;
      }
      ty  = newty;
    }
    
int p = (int) (canvasHeight * 0.9);
int m = (int)(height - (canvasHeight - p));
    datFrame.vScrollbar.setValues(
             //draw the part of the dataset that starts at this x:
             ty, 
	     //amount to scroll for a "page":
	     p, //(int)(canvasHeight ), 
	     //minimum image y to specify:
	     0,
	     //maximum image y to specify:
	     m);//height - canvasHeight);

    //"visible" arg to setValues() has no effect after scrollbar is visible.
    datFrame.hScrollbar.setBlockIncrement(p); //(int)(canvasHeight ));
    
    return;
  }
  

  /** set scrollbar offset(vertical) */
  public void setVoffset() {
    vOffset = ty/CellRect.height;
  }

  /** set horizontal scrollbar offset value */
  public void setHoffset() {
    hOffset = tx/CellRect.width;
  }


  

  public void mouseEntered(MouseEvent e) {}

  public void mouseExited(MouseEvent e) {}

  public void mouseClicked(MouseEvent e) {}
  

  public void mouseReleased(MouseEvent e) {}

  /**
   * Called if the mouse is down.
   * @param evt the event
   * @param x the x coordinate
   * @param y the y coordinate
   * @see java.awt.Component#mouseDown
   */
  public void mousePressed(MouseEvent e)
  {
    int x = e.getX();
    int y = e.getY();
    // set draw flag
    drawFlag = false;
    
    // when mouse down , paint the cell to deep gray(darkGray)
    // first using the mouse position to get the cell position
    int col = x / CellRect.width  ;
    int row = y / CellRect.height ;
    
    Rectangle rect = null;
    drawRectangle  = null;

    if (cells[row][col] != null)
      rect = cells[row][col];
    
    if (rect != null) {
      // confirm this cell will be redraw
      if (rect.contains(x,y)) {  
	// the specified point lies inside a rectangle
	drawFlag = true;
	drawRectangle = new Rectangle(rect.x, rect.y, rect.width, rect.height);
	
        selectedDataset.setBounds(hOffset+col, vOffset+row, 1,1);

	// draw the responding area in the image
        subsetArea.setBounds(imgCanvas.datasetRange.x + selectedDataset.x,
	 		       imgCanvas.datasetRange.y + selectedDataset.y,
                               1,
                               1);
       imgCanvas.drawRectangle(subsetArea);	    
      }
    }
    
    // repaint
    repaint();
  }
    

  public void mouseMoved(MouseEvent e) {}

  /**
   * Called if the mouse is draged.
   * @param evt the event
   * @param x the x coordinate
   * @param y the y coordinate
   * @see java.awt.Component#mouseDrag
   */
  public void mouseDragged(MouseEvent e)
  {
    int x = e.getX();
    int y = e.getY();
    // set draw flag
    drawFlag = false;
    
    // when mouse draged , paint the cell to deep gray(darkGray)
    // firstly, using the mouse position to get the draw area .
    
    int col = x / CellRect.width  ;
    int row = y / CellRect.height ;
    
    Rectangle rect = null;
    
    if (cells[row][col] != null)
      rect = cells[row][col];
    
    if ((drawRectangle != null) && (rect != null)) {
      // confirm this cell will be redraw
      if ((rect.contains(x,y)) && 
	  ((vOffset+row) < datasetRange.height) &&
	  ((hOffset+col) < datasetRange.width)  ) {
	
	// the specified point lies inside a rectangle
	drawFlag = true;
	
	// reset the drawRectangle;
	int w = (rect.x - drawRectangle.x) + CellRect.width;
	int h = (rect.y - drawRectangle.y) + CellRect.height;
	
	// resize the rectangle   	
	drawRectangle.setSize(w,h);
	    
	selectedDataset.setSize(hOffset+col-selectedDataset.x+1,
	                       vOffset+row-selectedDataset.y+1);
	    
	// resize the subsetArea
	subsetArea.setSize(selectedDataset.width,
	                  selectedDataset.height);
	imgCanvas.drawRectangle(subsetArea);
	       
      }
      else {
	// point invalid?
	drawFlag = true;
	
	// keep the old draw area
	
      }
    }
    
    // repaint
    repaint();

  }

    /**
     * make the sample image on the seprate frame
     */
    public void displaySampleImage() {

	int factorVal    = 1;
	String factorStr = datFrame.factor.getText();
	if (Character.isDigit(factorStr.charAt(0)))
	    // get  value
	    factorVal = Integer.parseInt(factorStr);

	    byte[][] tmp;
	    
	    tmp = new byte[datasetRange.height][datasetRange.width];
	    
	    for (int i=0; i<datasetRange.height; i++) {

		// first extract number position by row
		int pos = ((datasetRange.y  + i) * imgCanvas.imageWidth + (datasetRange.x )) ;
		for (int j=0; j<datasetRange.width; j++) 		
		    tmp[i][j] = (byte)imgCanvas.imageData[pos++];
		
	    }
	    Rectangle rect = new Rectangle(0,0,datasetRange.width, datasetRange.height);

	    image = null;
 	    image = createSampleImage(tmp,rect, factorVal, imgCanvas.imagePalette);

	// set default image
	defaultImage = image;

	if (imageFrame == null) {
	    // display the image in the seprate frame
	    imageFrame = new ImageFrame(image);
	    
	    // set dataFrame for imageFrame
	    imageFrame.dataFrame = this.datFrame;
	}
	else {
	    imageFrame.imageCanvas.setImage(defaultImage);
        }
	
	return ;
    }

    /**
     * make the interpolate  image on the seprate frame
     */
    public void displayInterpolateImage() {

	int factorVal    = 1;
	String factorStr = datFrame.factor.getText();
	
	if (Character.isDigit(factorStr.charAt(0))) 
	    // get  value
	    factorVal = Integer.parseInt(factorStr);
 	
	    byte[][] tmp = new byte[datasetRange.height][datasetRange.width];
	    
	    for (int i=0; i<datasetRange.height; i++) {

		// first extract number position by row
		int pos = ((datasetRange.y  + i) * imgCanvas.imageWidth + (datasetRange.x )) ;
		for (int j=0; j<datasetRange.width; j++) 		
		    tmp[i][j] = (byte)imgCanvas.imageData[pos++];
		
	    Rectangle rect = new Rectangle(0,0,datasetRange.width, datasetRange.height);   

	    image = null;
	    image = createInterpolateImage(tmp,rect, factorVal, imgCanvas.imagePalette);

	}
	
	// set interpolate  image
	interpolateImage  = image;

	if (imageFrame == null) {
	    // display the image in the seprate frame
	    imageFrame = new ImageFrame(image);
	    
	    // set dataFrame for imageFrame
	    imageFrame.dataFrame = this.datFrame;
	}
	else
	    imageFrame.imageCanvas.setImage(image);
	 
	
	return ;
    }

/** create sample image by the ratio 
 * @param data the scientific data value
 * @param subset the selected area of the dataset
 * @param ratio  the factor to make the image from dataset
 * @param pal the palette value when making the image
 * return image if succeed, or null
 */
public Image createSampleImage(float[][] data, Rectangle subset, int ratio, byte[] pal) {
  
    // Is the ratio valid? 
    if (ratio <=0)
        return null;
    
    if (subset == null) 
        return null;
   
    // specify the pixel value of the generic image 
    byte[] pixels;
    int w = subset.width  * ratio;
    int h = subset.height * ratio;
    int size = w*h;
    
    pixels = new byte[size];
    
    if (getImageData(data,subset, ratio, pixels))  {
	
	if (pal != null) {
	    // set the RGB
	    byte[] red   = new byte[256];
	    byte[] green = new byte[256];
	    byte[] blue  = new byte[256];
	
	    for (int i=0; i<256; i++) {
		red[i]   = (byte)(pal[i*3]);
		green[i] = (byte)(pal[i*3+1]);
		blue[i]  = (byte)(pal[i*3+2]);
	    }
			
	    // set the color model
	    ImageColorModel colorModel = new ImageColorModel(red,green,blue);

	    // create the image
	    return(app.createImage(new MemoryImageSource(w,h,colorModel.getColorModel(),pixels,0,w)));
	}
	else
	    return(app.createImage(new MemoryImageSource(w,h,ColorModel.getRGBdefault(), pixels,0,w)));
	
    }
    else 
      return null;
  
}

/** create sample image by the ratio 
 * @param data the scientific data value
 * @param subset the selected area of the dataset
 * @param ratio  the factor to make the image from dataset
 * @param pal the palette value when making the image
 * return image if succeed, or null
 */
public Image createSampleImage(byte[][] data, Rectangle subset, int ratio, byte[] pal) {
  
    // Is the ratio valid? 
    if (ratio <=0)
        return null;
    
    if (subset == null) 
        return null;

    // specify the pixel value of the generic image
    byte[] pixels;
    int w = subset.width  * ratio;
    int h = subset.height * ratio;
    int size = w*h;

    pixels = new byte[size];
    
    if (getImageData(data,subset, ratio, pixels))  {
	
	if (pal != null) {
	    // set the RGB
	    byte[] red   = new byte[256];
	    byte[] green = new byte[256];
	    byte[] blue  = new byte[256];
	
	    for (int i=0; i<256; i++) {
		red[i]   = (byte)(pal[i*3]);
		green[i] = (byte)(pal[i*3+1]);
		blue[i]  = (byte)(pal[i*3+2]);
	    }
			
	    // set the color model
	    ImageColorModel colorModel = new ImageColorModel(red,green,blue);

	    // create the image
	    return(app.createImage(new MemoryImageSource(w,h,colorModel.getColorModel(),pixels,0,w)));
	}
	else
	    return(app.createImage(new MemoryImageSource(w,h,ColorModel.getRGBdefault(), pixels,0,w)));
	
    }
    else 
      return null;
  
}

/** create interpolate image by the ratio 
 * @param data the scientific data value
 * @param subset the selected area of the dataset
 * @param ratio  the factor to make the image from dataset
 * @param pal the palette value when making the image
 * return image if succeed, or null
 */
public Image createInterpolateImage(byte[][] data, Rectangle subset, int ratio, byte[] pal) {
  
    // Is the ratio valid? 
    if (ratio <=0)
        return null;
    
    if (subset == null) 
        return null;
   
    // specify the pixel value of the generic image 
    byte[] pixels;
    int w = subset.width  * ratio;
    int h = subset.height * ratio;
    int size = w*h;
    
    pixels = new byte[size];
    
    if (getInterpolateImageData(data,subset, ratio, pixels))  {

     // the following code make me confused, I don't know  why I can't get image 
     // painted under Win95. However if I changed pixel value for "pixel[0]" 
     // to zero(0), then it works.    
     // try to get OS name
     String osName = System.getProperties().getProperty("os.name");
     if (osName.toUpperCase().indexOf("WINDOW") != -1) // window 95 or nt
	pixels[0] = 0;

	
      if (pal != null) {
	// set the RGB
	byte[] red   = new byte[256];
	byte[] green = new byte[256];
	byte[] blue  = new byte[256];
	
	for (int i=0; i<256; i++) {
	  red[i]   = (byte)(pal[i*3]);
	  green[i] = (byte)(pal[i*3+1]);
	  blue[i]  = (byte)(pal[i*3+2]);
	}
	
	// set the color model
	ImageColorModel colorModel = new ImageColorModel(red,green,blue);
	
	// create the image
	return(app.createImage(new MemoryImageSource(w,h,colorModel.getColorModel(),pixels,0,w)));
      }
      else
	return(app.createImage(new MemoryImageSource(w,h,ColorModel.getRGBdefault(), pixels,0,w)));
      
    }
    else 
      return null;
    
}
  

  /** create sample image data by the ratio 
   * @param data the scientific data value
   * @param subset the selected area of the dataset
   * @param ratio  the factor to make the image from dataset
   * @param pixels the converted image data
   * @return true if succeed, or false
   */
  public boolean getImageData(float[][] data,
			      Rectangle subset,
			      int ratio,
			      byte[] pixels) {
  
    // Is the ratio valid? 
    if (ratio <=0)
        return false;
    
    if (subset == null) 
        return false;
   
    // get image data by subset
    int bx = subset.width  * ratio;
    int by = subset.height * ratio;
    
    float[][] buffer = new float[by][bx];
    
    for (int i=0; i<subset.height; i++) {
      // the data position by row
      int pos = (i + subset.y)  ;
      
      for (int j=0; j<subset.width; j++) {
	// get the data value
	float val = data[pos][subset.x+j];
	
	// repeat the data storage
	for (int k=0; k<ratio; k++ ) 
	  for (int l =0; l<ratio; l++) 
	    buffer[i*ratio + k][j*ratio + l] = val;
	
      }
    }
    
    // convert  to image data
    dataToImage(buffer, subset.width*ratio, subset.height*ratio, pixels);
    
    return true;
  }

  /** create sample image data by the ratio 
   * @param data the scientific data value
   * @param subset the selected area of the dataset
   * @param ratio  the factor to make the image from dataset
   * @param pixels the converted image data
   * @return true if succeed, or false
   */
  public boolean getImageData(byte[][] data,
			      Rectangle subset,
			      int ratio, 
			      byte[] pixels) {
  
    // Is the ratio valid? 
    if (ratio <=0)
        return false;
    
    if (subset == null) 
        return false;
   
    // get image data by subset
    int bx = subset.width  * ratio;
    int by = subset.height * ratio;
    
    for (int i=0; i<subset.height; i++) {
	// the data position by row
	int pos = (i + subset.y)  ;

	int xpos = i*ratio;
	
	for (int j=0; j<subset.width; j++) {
	  // get the data value
	  byte val = data[pos][subset.x+j];
	  
	  int y1 = j*ratio;
	  
	  // repeat the data storage
	  for (int k=0; k<ratio; k++ ) {
	    
	    int  x1 = (xpos + k) * bx;
	    
	    for (int l =0; l<ratio; l++) {
	      int index = x1 + y1 + l;	    
	      pixels[index] = val;
	    }
	  }
	}
    }
    
    return true;
  }
  

  /** create interpolate image data by the ratio 
   * @param data the scientific data value
   * @param subset the selected area of the dataset
   * @param ratio  the magnify factor to make the image from dataset
   * @param pixels the converted image data
   * @return true if succeed, or false
   */
  public boolean getInterpolateImageData(byte[][] data,
					 Rectangle subset,
					 int ratio,
					 byte[] pixels) {
    
    // Is the ratio valid? 
    if (ratio <=0)
        return false;
    
    if (subset == null) 
        return false;
   
    // get image data by subset
    // the width or height of the souce image
    int  windWidth  = subset.width  ;
    int  windHeight = subset.height ;
    
    // the array of width or height of the cells
    short xSizes[] = new short[windWidth];
    short ySizes[] = new short[windHeight];
    
    //the width or height of the interpolated image
    int to_width = windWidth * ratio;
    int to_height= windHeight * ratio;
  
    // ulc: upper left byte for cell
    // llc: lower left byte for cell
    short ulc, llc;
    
    // template variables
    float xSize, ySize;
    
    //precalculated float
    float t1,t2,t3, c1,c2,c3;
    
    int u,v,w,x;
    
    int fromOff, bufOff, cellOff, lineOff;
    short  total;
    
    /*  
    calculate heights and widths of cells into xSizes and ySizes...
    these arrays hold scales for each pixel, so that, fer instance,
    interpolating a 10 x 20 pixel wide source image to 60 x 40 pixels
    will produce an xSizes array that has 10 entries (each equal to 6)
    and ySizes array that has 20 entries (each equal to 2)...
    xSizes is as long as the width of the source image,
    ySizes is as long as the height of the source image */

    xSize = (float)(to_width - 1) / (float)(windWidth - 1);
    
    total = 0;
    for (  u = 0; u < (windWidth - 1); u++) {
	xSizes[u] = (short)((short)(((float)(u+1) * xSize) + 0.5) - total);	
	total += xSizes[u];
    }
   
    ySize = (float)(to_height - 1) / (float)(windHeight - 1);
    total = 0;
    for (  u = 0; u < (windHeight - 1); u++) {
	ySizes[u] = (short)( (short)(((float)(u+1) * ySize) + 0.5) - total);	
	total += ySizes[u];
    } 
    
    fromOff = (subset.width * subset.y) + subset.x;
    bufOff = 0;
    
    /* start doing the interpolation...iterate over each
       pixel in the source image */
    for  ( u = 0; u < (windHeight - 1); u++) {
	for (  v = 0; v < (windWidth - 1); v++) {
                 
	    /* the corner values for this cell will be the
	       four pixels in the source image */
	    ulc = (short)data[subset.y+u][subset.x+v];
 	    if (ulc<0)
 	       ulc += (short)256;
	    
	    llc = data[subset.y+u+1][subset.x+v];
	    if (llc<0)
	       llc += 256;
	    
	    ySize = (float)ySizes[u];
	    xSize = (float)xSizes[v];
            
	    short datTmp = data[subset.y+u][subset.x+v+1];
	    if (datTmp<0)
	       datTmp += (short)256;
	    c1 = (float)(datTmp - ulc)/xSize;
	    
	    datTmp = data[subset.y+u+1][subset.x+v+1];
	    if (datTmp<0)
	       datTmp += (short)256;
	    
	    c2 = (((float)(datTmp - llc) / xSize) - c1) / ySize;

	    c3 = (float)((int)llc - (int)ulc) / ySize;
	      
	    /*
	      save the pointer to this cell so we can move to the next
	      one easily, save the pointer to this line so we can move
	      down the cell easily */
              
	    cellOff = bufOff;
	    lineOff = bufOff;
            
	    /*   
	      the following part calculates the Bytes across the first
	      row of this cell...it's here for efficiency because we
	      know that for the first row in the cell our vertical distane
	      down the cell is zero and so we don't need to do bilinear,
	      so this avoids some unnecessary multiplies */
              
	    t1 = ulc;  
	    for (  x = 0; x < (byte)xSize; x++) {
		pixels[bufOff++] = (byte)t1;
		t1 += c1;
	    }
	    
	    /* reset pointer to next row */
	    bufOff = lineOff + to_width;	    
	
	    /* do the interpolation for the remaining part of the cell */
	    for (  w = 1; w < (short)ySize; w++) {
		lineOff = bufOff;
		t1 = (float)ulc + (c3 * (float)w);
		t2 = (c2 * (float)w) + c1;
		t3 = t1;
		for ( x = 0; x < (short)xSize; x++) {
		  pixels[bufOff++] = (byte)t3;				 
		  t3 += t2;
		}
		/* reset pointer to next row */
		bufOff = lineOff + to_width;
	    }
            
	    /* reset the pointer to the next cell */
	    bufOff = cellOff + (int)xSize;
	    
	}
	/* we've reach the end of this row of cells, so move down to the
	   next */
	bufOff += (((int)ySize - 1) * to_width) + 1;
    }	    

    /* fill in the last column along the right hand side */
    bufOff = to_width -1;
    
    for (  u = 0; u < (windHeight - 1); u++) {
	ulc = data[subset.y + u][subset.x + windWidth-1];
	if (ulc<0)
	   ulc += (short)256;
	
	ySize = ySizes[u];
	
	short datTmp  = data[subset.y + 1 + u][subset.x+windWidth -1];
	if (datTmp<0)
	   datTmp += (short)256;
	
	c1 = (float)(datTmp-ulc)/ySize;
	
	t1 = ulc;
	for (  w = 0; w < ySize; w++)  { 
	    pixels[bufOff] = (byte)t1;
	    bufOff += to_width;
	    t1 += c1;
	}
    }
    
    /* fill in the last row along the bottom */
    bufOff -= (to_width - 1);
    
    for ( v = 0; v < (windWidth - 1); v++) {
	ulc = data[subset.y+windHeight-1][subset.x+v];
	if (ulc<0)
	  ulc += (short)256;
	
	xSize = xSizes[v];
	short datTmp = data[subset.y+windHeight-1][subset.x+v+1];
	if (datTmp <0)
	  datTmp += (short)256;
	
	c1 = (float)((int)(datTmp) - (int)ulc) / xSize;
	t1 = ulc;
	for (  x = 0; x < xSize; x++) {
	  pixels[bufOff++] = (byte)t1;
	  t1 += c1;
	}
    }
    /* set the last element in the last row in the destination */
    
    return true;
  }
  

  /** convert dataset to image pixel value
   * @param data  the scientific data value
   * @param width the width of the converted image
   * @param height the height of the converted image
   * @param pixels the converted image data
   */
  public void dataToImage(float[][] data,int w, int h, byte[] pixels) {
    
    // the maxinum & mininum value of the dataset
    float   dataMax = data[0][0];
    float   dataMin = data[0][0];
    
    // find max & min
    
    for (int i=0; i<h; i++) 
        for (int j=0; j<w; j++) {
	
	dataMax = Math.max(data[i][j], dataMax);
	dataMin = Math.min(data[i][j], dataMin);
    }
    
    float spread = (dataMax == dataMin) ? 1.0f : 255.0f/ (dataMax - dataMin);
    int index = 0;
    
    if (spread > 255.0/(2.1E+9f)) {
      // Is max-min too big for an byte ?
      for (int i=0; i<h; i++) 
	for (int j=0; j<w; j++)
	  pixels[index++] = (byte)((data[i][j] - dataMin)*spread);
    }
    else {
      for (int i=0; i<w*h; i++)
	pixels[index++] = 0;
    }
    
  }
}

/** This calss will create the new canvas to paint the dimension scale
 *  value based on the selected dataset range 
 */
class JHVDataRangeCanvas extends Canvas { 
  
  /** the spreadsheet frame */
  JHVDataFrame	datFrame;
  
  /** the indicator of the canvas attributes */
  int             orientation;
   
  /** The horizontal Canvas variable. */
  public static final int	HORIZONTAL = 0;
   
  /** The vertical Canvas variable.  */    
  public static final int	VERTICAL   = 1;

  // default value for data spreadsheet row or colnum number
  int rectLen 	= 30;	
  int xLen	= 256*2;
  int yLen	= 256*1;

  Font		font;
  FontMetrics	fontMetrics;
  
  int		fontWidth, fontHeight;
    
  // double buffering
  // variables for duble-buffer
  Image 	offScreenImage = null;
  Graphics 	offGraphics;

  // dimension scale info.
  int		nt;
  Object	xscale;
  Object	yscale;
  boolean	scaleFlag = false;

  int 		scaleFactor = 1;

  /** new class constructor 
   * @param frame the spreadsheet frame
   * @param direction the variable to specify the canvas associated with spreadsheet
   */
  public JHVDataRangeCanvas(JHVDataFrame frame, int direction) {
    
    // set spreadsheet frame
    datFrame    = frame;

    // set canvas type
    orientation = direction;

    // set the current used font
    font 	    = frame.dataCanvas.dataFont;

    // Sets the font of the component.
    setFont(font);

    // dimension scale info.
    nt  = datFrame.imgCanvas.scaleDataType;
    xscale = datFrame.imgCanvas.xscale;
    yscale = datFrame.imgCanvas.yscale;
    scaleFlag = ((xscale!=null)&&(yscale!=null));

    if (scaleFlag)	scaleFactor = 2;
    // initialize 
    init();
  }

  /** Initialize the canvas 
   * @param len the canvas width or length
   */
  public void init(int len) {
	
    // set the canvas size
    if (orientation == HORIZONTAL) {
      // Assume scrollbar width to 10;
      xLen = len + rectLen*scaleFactor + 20 ;
      setSize(xLen, rectLen);
    }
    else {
      yLen = len;
      setSize(rectLen*scaleFactor, yLen);
    }
    // repaint the canvas
    repaint();
  }

  /** Initialize the canvas by default */
  public void init() {
    // set the canvas size
    if (orientation == HORIZONTAL) 
      	setSize(xLen, rectLen);	
    else 
      	setSize(rectLen*scaleFactor, yLen); 
 }

  /** 
   * Sets the font of the component.
   * @param f the font
   */
  public synchronized void setFont(Font f) {
    // set Font for component
    super.setFont(f);

    // get new FontMetrics
    fontMetrics = getFontMetrics(getFont());

    // set font width & height
    fontHeight = fontMetrics.getHeight();
    fontWidth  = fontMetrics.charWidth(' ');
    
  }

  /** draw the dimension scale responding to the spreadsheet */
  public synchronized void paint(Graphics g) {

    // cell size of the spreadsheet
    Rectangle rect = datFrame.dataCanvas.CellRect;
  
    // the row and colnum of the current spreadsheet
    int rowNumber  = datFrame.dataCanvas.cellNumberByRow;
    int colNumber  = datFrame.dataCanvas.cellNumberByColnum;
    
    if (orientation == HORIZONTAL)  {
      g.setColor(Color.pink);
      
      g.fill3DRect(rectLen*scaleFactor,1, xLen , rectLen, false);
      g.setColor(Color.white);
      
      // drawable position
      int x = rectLen*scaleFactor;
      int y = 1;
      
      // first value
      int kk = datFrame.dataCanvas.hOffset + 
	datFrame.dataCanvas.datasetRange.x +
	datFrame.imgCanvas.subsetRange.x ;
      
      for (int i=0; i<=colNumber; i++)  
	if ((datFrame.dataCanvas.hOffset + i) <
	    datFrame.dataCanvas.datasetRange.width)  {
	  x = rectLen*scaleFactor + rect.width*i + 4;
	  // g.drawString(Integer.toString(kk+i),x, 20);

	  if (scaleFlag) { // draw scale
		float fscale = ImageDataConverter.getData(xscale,nt,kk+i);
		g.drawString(Float.toString(fscale),x, 20);
	  }
	  else
	  	g.drawString(Integer.toString(kk+i),x, 20);
	}
      
    }
    else {
      // set color 
      g.setColor(Color.magenta);

      // draw the graphics
      g.fill3DRect(1,1, rectLen*scaleFactor, yLen, true);
      g.setColor(Color.white);
      
      int y = 1;  
      // first value
      int kk = datFrame.dataCanvas.vOffset + 
	datFrame.dataCanvas.datasetRange.y + 
	datFrame.imgCanvas.subsetRange.y ;
      
      for (int i=1; i<=rowNumber; i++)  
	if ((datFrame.dataCanvas.vOffset + i) <=
	    datFrame.dataCanvas.datasetRange.height)  {
	  y = rect.height*i;

	  if (scaleFlag) {
		float fscale = ImageDataConverter.getData(yscale,nt,kk+i-1);
		g.drawString(Float.toString(fscale),1,y);
	  }
	  else
		g.drawString(Integer.toString(kk+i-1), 1, y);
	}
    } 
  }
  

  /**
   * Updates the component. This method is called in
   * response to a call to repaint. You can assume that
   * the background is not cleared.
   * @param g the specified Graphics window
   * @see java.awt.Component#update
   */
  public void update(Graphics g) {

    Dimension d = getSize();
    
    if (offScreenImage == null) {
      
      // offScreenImage not created; create it.
      offScreenImage = createImage(d.width*2, d.height*2);	
      
      // get the off-screen graphics context    
      offGraphics    = offScreenImage.getGraphics();
      
      // set the font for offGraphics
      offGraphics.setFont(getFont());	 
    }
    
    // paint the background on the off-screen graphics context
    offGraphics.setColor(getBackground());
    offGraphics.fillRect(1,1,d.width-2,d.height-2);    
    offGraphics.setColor(getForeground());
    
    
    // draw the current frame to the off-screen 
    paint(offGraphics);
    
    //then draw the image to the on-screen 
    g.drawImage(offScreenImage, 0, 0, null);
    
  }
}
