/* 
  mxTextTools -- Fast text manipulation routines

  (c) Marc-Andre Lemburg; all rights reserved
*/

/* We want all our symbols to be exported */
#define MX_BUILDING_MXTEXTTOOLS

/* Logging file used by debugging facility */
#ifndef MAL_DEBUG_OUTPUTFILE
# define MAL_DEBUG_OUTPUTFILE "mxTextTools.log"
#endif

#include "mx.h"
#include "mxTextTools.h"
#include <ctype.h>

#define VERSION "1.0.2"

/* Initial list size used by e.g. setsplit(), setsplitx(),... since
   modern OSes only reserve memory instead of actually allocating it
   at malloc() time, this should be set high enough to accomodate most
   average cases reasonable well. */
#define INITIAL_LIST_SIZE 500

/* --- module doc-string -------------------------------------------------- */

static char *Module_docstring = 

 MXTEXTTOOLS_MODULE" -- Tools for fast text processing. Version "VERSION"\n\n"

 "(c) Copyright Marc-Andre Lemburg, 1997, 1998, mal@lemburg.com,\n\n"
 "                 All Rights Reserved\n\n"
 "See the documentation for further information on copyrights,\n"
 "or contact the author."
;

/* --- internal macros ---------------------------------------------------- */

#define _mxBMS_Check(v) \
        (((mxBMSObject *)(v))->ob_type == &mxBMS_Type)

#ifdef MXFASTSEARCH
#define _mxFS_Check(v) \
        (((mxFSObject *)(v))->ob_type == &mxFS_Type)
#endif

/* --- module globals ----------------------------------------------------- */

/* Translation strings */
static PyObject *mxTo_Upper;
static PyObject *mxTo_Lower;

/* --- forward declarations ----------------------------------------------- */

/* --- module helper ------------------------------------------------------ */

static
PyObject *mxTo_Upper_New()
{
    char tr[256];
    int i;
    
    for (i = 0; i < 256; i++)
	tr[i] = toupper((char)i);
    return PyString_FromStringAndSize(tr,sizeof(tr));
}

static
PyObject *mxTo_Lower_New()
{
    char tr[256];
    int i;
    
    for (i = 0; i < 256; i++)
	tr[i] = tolower((char)i);
    return PyString_FromStringAndSize(tr,sizeof(tr));
}

/* --- module interface --------------------------------------------------- */

/* --- Boyer Moore Substring Search Object -----------------------------*/

staticforward PyMethodDef mxBMS_Methods[];

/* allocation */

Py_C_Function( mxBMS_new,
	       "BMS(match[,translate])\n"
	       "Create a Boyer Moore substring search object\n"
	       "for the string match; translate is an optional\n"
	       "translate-string like the one used in the module re.")
{
    PyObject *s = 0;
    PyObject *t = 0;
    mxBMSObject *bm;

    bm = PyObject_NEW(mxBMSObject,&mxBMS_Type);
    if (bm == NULL) return NULL;

    Py_Get2Args("O|O:BMS.__init__",s,t);
    
    Py_Assert(PyString_Check(s),
	      PyExc_TypeError,
	      "argument must be a string");

    if (t) {
	
	Py_Assert(PyString_Check(t),
		  PyExc_TypeError,
		  "translate table must be a string");

	Py_Assert(PyString_GET_SIZE(t) == 256,
		  PyExc_TypeError,
		  "translate string must have exactly 256 chars");

	Py_INCREF(t);
    }
    bm->tr = t;

    Py_INCREF(s);
    bm->match = s;
    
    bm->c = bm_init(PyString_AS_STRING(s),
		    PyString_GET_SIZE(s));

    Py_Assert(bm->c != NULL,
	      PyExc_TypeError,
	      "error initializing the search object");

    return (PyObject *)bm;

 onError:
    Py_XDECREF(t);
    Py_XDECREF(s);
    Py_DECREF(bm);
    return NULL;
}

static void
mxBMS_Free(mxBMSObject *bm)
{
    bm_free(bm->c);
    Py_XDECREF(bm->match);
    Py_XDECREF(bm->tr);
    PyMem_DEL(bm);
}

/* methods */

Py_C_Function( mxBMS_search,
	       "BMS.search(text,start=0,len_text=len(text))\n"
	       "Search for the substring in text, looking only at the\n"
	       "slice [start:len_text] and return the slice (l,r)\n"
	       "where the substring was found, (start,start) otherwise.")
{
    PyObject *s = 0;
    int start = 0;
    int index;
    int len_text = INT_MAX;

    if (!_mxBMS_Check(self)) {
	PyErr_BadInternalCall();
	return NULL;
    }

#define self ((mxBMSObject *)self)

    Py_Get3Args("O|ii:BMS.search",s,start,len_text);

    Py_Assert(PyString_Check(s),
	      PyExc_TypeError,
	      "argument must be a string");

    Py_CheckSlice(s,start,len_text);
    
    if (self->tr) {
	/* search with translate table */
	index = bm_tr_search(self->c,
			     PyString_AS_STRING(s),
			     start,
			     len_text,
			     PyString_AS_STRING(self->tr));
    }
    else {
	/* exact search */
	index = bm_search(self->c,
			  PyString_AS_STRING(s),
			  start,
			  len_text);
    }
    
    if (index != start) /* found */
	start = index - self->c->len_match;

    Py_Assert(index >= 0,
	      PyExc_SystemError,
	      "internal error");

    /* return found slice */
    Py_Return2("ii",start,index);

 onError:
    return NULL;

#undef self
}

Py_C_Function( mxBMS_find,
	       "BMS.find(text,start=0,len_text=len(text))\n"
	       "Search for the substring in text, looking only at the\n"
	       "slice [start:len_text] and return the index\n"
	       "where the substring was found, -1 otherwise.")
{
    PyObject *s = 0;
    int start = 0;
    int index;
    int len_text = INT_MAX;

    if (!_mxBMS_Check(self)) {
	PyErr_BadInternalCall();
	return NULL;
    }

#define self ((mxBMSObject *)self)

    Py_Get3Args("O|ii:BMS.find",s,start,len_text);

    Py_Assert(PyString_Check(s),
	      PyExc_TypeError,
	      "argument must be a string");

    Py_CheckSlice(s,start,len_text);
    
    if (self->tr) {
	/* search with translate table */
	index = bm_tr_search(self->c,
			     PyString_AS_STRING(s),
			     start,
			     len_text,
			     PyString_AS_STRING(self->tr));
    }
    else {
	/* exact search */
	index = bm_search(self->c,
			  PyString_AS_STRING(s),
			  start,
			  len_text);
    }
    
    if (index != start) /* found */
	start = index - self->c->len_match;
    else
	start = -1;

    Py_Assert(index >= 0,
	      PyExc_SystemError,
	      "internal error");

    return PyInt_FromLong(start);

 onError:
    return NULL;

#undef self
}

Py_C_Function( mxBMS_findall,
	       "BMS.findall(text,start=0,len_text=len(text))\n"
	       "Search for the substring in text, looking only at the\n"
	       "slice [start:len_text] and return a list of all\n"
	       "non overlapping slices (l,r) in text where the match\n"
	       "string can be found.")
{
    PyObject *s = 0;
    PyObject *list = 0;
    int start = 0;
    int index = 0;
    int len_text = INT_MAX;

    if (!_mxBMS_Check(self)) {
	PyErr_BadInternalCall();
	return NULL;
    }

#define self ((mxBMSObject *)self)

    Py_Get3Args("O|ii:BMS.findall",s,start,len_text);

    Py_Assert(PyString_Check(s),
	      PyExc_TypeError,
	      "argument must be a string");

    Py_CheckSlice(s,start,len_text);

    list = PyList_New(0);
    if (!list)
	goto onError;
        
    if (!self->tr)
	while (start <= len_text - self->c->len_match) {
	    register PyObject *t,*v;

	    /* exact search */
	    index = bm_search(self->c,
			      PyString_AS_STRING(s),
			      start,
			      len_text);
	    if (index < 0 || index == start) /* Error or Not found */
		break;
	
	    /* Build slice and append to list */
	    start = index - self->c->len_match;
	    t = PyTuple_New(2);
	    if (!t) 
		goto onError;
	    v = PyInt_FromLong(start);
	    if (!v)
		goto onError;
	    PyTuple_SET_ITEM(t,0,v);
	    v = PyInt_FromLong(index);
	    if (!v)
		goto onError;
	    PyTuple_SET_ITEM(t,1,v);
	    PyList_Append(list,t);
	    Py_DECREF(t);
	    
	    start = index;
	}
    else
	while (start <= len_text - self->c->len_match) {
	    register PyObject *t,*v;

	    /* search with translate table */
	    index = bm_tr_search(self->c,
				 PyString_AS_STRING(s),
				 start,
				 len_text,
				 PyString_AS_STRING(self->tr));
	    if (index < 0 || index == start) /* Error or Not found */
		break;
	
	    /* Build slice and append to list */
	    start = index - self->c->len_match;
	    t = PyTuple_New(2);
	    if (!t) 
		goto onError;
	    v = PyInt_FromLong(start);
	    if (!v)
		goto onError;
	    PyTuple_SET_ITEM(t,0,v);
	    v = PyInt_FromLong(index);
	    if (!v)
		goto onError;
	    PyTuple_SET_ITEM(t,1,v);
	    PyList_Append(list,t);
	    Py_DECREF(t);
	    
	    start = index;
	}
    
    Py_Assert(index >= 0,
	      PyExc_SystemError,
	      "internal error");

    /* return list of slices */
    return list;

 onError:
    Py_XDECREF(list);
    return NULL;

#undef self
}

static int
mxBMS_Print(mxBMSObject *self,
	    FILE *fp, 
	    int flags)
{
    if (self == NULL) return -1;
    fprintf(fp,"<Boyer-Moore Search object for ");
    PyObject_Print(self->match,fp,flags);
    fprintf(fp," at %lx>",(long)self);
    return 0;
}

static PyObject
*mxBMS_Getattr(mxBMSObject *self,
	       char *name)
{
    return Py_FindMethod(mxBMS_Methods, (PyObject *)self, (char *)name);
}

/* Python Type Table */

PyTypeObject mxBMS_Type = {
        PyObject_HEAD_INIT(0)		/* init at startup ! */
	0,			  	/*ob_size*/
	"Boyer Moore Search",	  	/*tp_name*/
	sizeof(mxBMSObject),	  	/*tp_basicsize*/
	0,			  	/*tp_itemsize*/
	/* methods */
	(destructor)mxBMS_Free,		/*tp_dealloc*/
	(printfunc)mxBMS_Print,	  	/*tp_print*/
	(getattrfunc)mxBMS_Getattr,  	/*tp_getattr*/
	(setattrfunc)0,		  	/*tp_setattr*/
	(cmpfunc)0,		  	/*tp_compare*/
	(reprfunc)0,		  	/*tp_repr*/
        0,			  	/*tp_as_number*/
	0,				/*tp_as_number*/
	0,				/*tp_as_mapping*/
	(hashfunc)0,			/*tp_hash*/
	(ternaryfunc)0,			/*tp_call*/
	(reprfunc)0,			/*tp_str*/
	(getattrofunc)0, 		/*tp_getattro*/
	(setattrofunc)0, 		/*tp_setattro*/
};

/* Python Method Table */

statichere
PyMethodDef mxBMS_Methods[] =
{   
    Py_MethodListEntry("search",mxBMS_search),
    Py_MethodListEntry("find",mxBMS_find),
    Py_MethodListEntry("findall",mxBMS_findall),
    {NULL,NULL} /* end of list */
};

#ifdef MXFASTSEARCH

/* --- Fast Search Object --------------------------------------*/

staticforward PyMethodDef mxFS_Methods[];

/* allocation */

Py_C_Function( mxFS_new,
	       "FS(match[,translate])\n"
	       "Create a Fast substring Search object\n"
	       "for the string match; translate is an optional\n"
	       "translate-string like the one used in the module re.")
{
    PyObject *s = 0;
    PyObject *t = 0;
    mxFSObject *fs;

    fs = PyObject_NEW(mxFSObject, &mxFS_Type);
    if (fs == NULL) return NULL;

    Py_Get2Args("O|O:FS.__init__",s,t);
    
    Py_Assert(PyString_Check(s),
	      PyExc_TypeError,
	      "argument must be a string");

    if (t) {
	
	Py_Assert(PyString_Check(t),
		  PyExc_TypeError,
		  "translate table must be a string");

	Py_Assert(PyString_GET_SIZE(t) == 256,
		  PyExc_TypeError,
		  "translate string must have exactly 256 chars");

	Py_INCREF(t);
    }
    fs->tr = t;

    Py_INCREF(s);
    fs->match = s;
    
    fs->c = fs_init(PyString_AS_STRING(s),
		    PyString_GET_SIZE(s));

    Py_Assert(fs->c != NULL,
	      PyExc_TypeError,
	      "error initializing the search object");

    return (PyObject *)fs;

 onError:
    Py_XDECREF(t);
    Py_XDECREF(s);
    Py_DECREF(fs);
    return NULL;
}

static void
mxFS_Free(mxFSObject *fs)
{
    fs_free(fs->c);
    Py_XDECREF(fs->match);
    Py_XDECREF(fs->tr);
    PyMem_DEL(fs);
}

/* methods */

Py_C_Function( mxFS_search,
	       "FS.search(text,start=0,len_text=len(text))\n"
	       "Search for the substring in text, looking only at the\n"
	       "slice [start:len_text] and return the slice (l,r)\n"
	       "where the substring was found, (start,start) otherwise.")
{
    PyObject *s = 0;
    int start = 0;
    int index;
    int len_text = INT_MAX;

    if (!_mxFS_Check(self)) {
	PyErr_BadInternalCall();
	return NULL;
    }

#define self ((mxFSObject *)self)

    Py_Get3Args("O|ii:FS.search",s,start,len_text);

    Py_Assert(PyString_Check(s),
	      PyExc_TypeError,
	      "argument must be a string");

    Py_CheckSlice(s,start,len_text);
    
    if (self->tr) {
	/* search with translate table */
	index = fs_tr_search(self->c,
			     PyString_AS_STRING(s),
			     start,
			     len_text,
			     PyString_AS_STRING(self->tr));
    }
    else {
	/* exact search */
	index = fs_search(self->c,
			  PyString_AS_STRING(s),
			  start,
			  len_text);
    }
    
    if (index != start) /* found */
	start = index - self->c->len_match;

    Py_Assert(index >= 0,
	      PyExc_SystemError,
	      "internal error");

    /* return found slice */
    Py_Return2("ii",start,index);

 onError:
    return NULL;

#undef self
}

Py_C_Function( mxFS_find,
	       "FS.find(text,start=0,len_text=len(text))\n"
	       "Search for the substring in text, looking only at the\n"
	       "slice [start:len_text] and return the index\n"
	       "where the substring was found, -1 otherwise.")
{
    PyObject *s = 0;
    int start = 0;
    int index;
    int len_text = INT_MAX;

    if (!_mxFS_Check(self)) {
	PyErr_BadInternalCall();
	return NULL;
    }

#define self ((mxFSObject *)self)

    Py_Get3Args("O|ii:FS.find",s,start,len_text);

    Py_Assert(PyString_Check(s),
	      PyExc_TypeError,
	      "argument must be a string");

    Py_CheckSlice(s,start,len_text);
    
    if (self->tr) {

	/* search with translate table */
	index = fs_tr_search(self->c,
			     PyString_AS_STRING(s),
			     start,
			     len_text,
			     PyString_AS_STRING(self->tr));
    }
    else {

	/* exact search */
	index = fs_search(self->c,
			  PyString_AS_STRING(s),
			  start,
			  len_text);
    }
    
    if (index != start) /* found */
	start = index - self->c->len_match;
    else
	start = -1;

    Py_Assert(index >= 0,
	      PyExc_SystemError,
	      "internal error");

    return PyInt_FromLong(start);

 onError:
    return NULL;

#undef self
}

Py_C_Function( mxFS_findall,
	       "FS.findall(text,start=0,len_text=len(text))\n"
	       "Search for the substring in text, looking only at the\n"
	       "slice [start:len_text] and return a list of all\n"
	       "non overlapping slices (l,r) in text where the match\n"
	       "string can be found.")
{
    PyObject *s = 0;
    PyObject *list = 0;
    int start = 0;
    int index = 0;
    int len_text = INT_MAX;

    if (!_mxFS_Check(self)) {
	PyErr_BadInternalCall();
	return NULL;
    }

#define self ((mxFSObject *)self)

    Py_Get3Args("O|ii:FS.findall",s,start,len_text);

    Py_Assert(PyString_Check(s),
	      PyExc_TypeError,
	      "argument must be a string");

    Py_CheckSlice(s,start,len_text);

    list = PyList_New(0);
    if (!list)
	goto onError;
        
    if (!self->tr)
	while (start <= len_text - self->c->len_match) {
	    register PyObject *t,*v;

	    /* exact search */
	    index = fs_search(self->c,
			      PyString_AS_STRING(s),
			      start,
			      len_text);
	    if (index < 0 || index == start) /* Error or Not found */
		break;
	
	    /* Build slice and append to list */
	    start = index - self->c->len_match;
	    t = PyTuple_New(2);
	    if (!t) 
		goto onError;
	    v = PyInt_FromLong(start);
	    if (!v)
		goto onError;
	    PyTuple_SET_ITEM(t,0,v);
	    v = PyInt_FromLong(index);
	    if (!v)
		goto onError;
	    PyTuple_SET_ITEM(t,1,v);
	    PyList_Append(list,t);
	    Py_DECREF(t);
	    
	    start = index;
	}
    else
	while (start <= len_text - self->c->len_match) {
	    register PyObject *t,*v;

	    /* search with translate table */
	    index = fs_tr_search(self->c,
				 PyString_AS_STRING(s),
				 start,
				 len_text,
				 PyString_AS_STRING(self->tr));
	    if (index < 0 || index == start) /* Error or Not found */
		break;
	
	    /* Build slice and append to list */
	    start = index - self->c->len_match;
	    t = PyTuple_New(2);
	    if (!t) 
		goto onError;
	    v = PyInt_FromLong(start);
	    if (!v)
		goto onError;
	    PyTuple_SET_ITEM(t,0,v);
	    v = PyInt_FromLong(index);
	    if (!v)
		goto onError;
	    PyTuple_SET_ITEM(t,1,v);
	    PyList_Append(list,t);
	    Py_DECREF(t);
	    
	    start = index;
	}
    
    Py_Assert(index >= 0,
	      PyExc_SystemError,
	      "internal error");

    /* return list of slices */
    return list;

 onError:
    Py_XDECREF(list);
    return NULL;

#undef self
}

static int
mxFS_Print(mxFSObject *self,
	    FILE *fp, 
	    int flags)
{
    if (self == NULL) return -1;
    fprintf(fp,"<Fast Search object for ");
    PyObject_Print(self->match,fp,flags);
    fprintf(fp," at %lx>",(long)self);
    return 0;
}

static PyObject
*mxFS_Getattr(mxFSObject *self,
	       char *name)
{
    return Py_FindMethod(mxFS_Methods, (PyObject *)self, (char *)name);
}

/* Python Type Table */

PyTypeObject mxFS_Type = {
        PyObject_HEAD_INIT(0)		/* init at startup ! */
	0,			  	/*ob_size*/
	"Fast Search",		  	/*tp_name*/
	sizeof(mxFSObject),	  	/*tp_basicsize*/
	0,			  	/*tp_itemsize*/
	/* methods */
	(destructor)mxFS_Free,		/*tp_dealloc*/
	(printfunc)mxFS_Print,	  	/*tp_print*/
	(getattrfunc)mxFS_Getattr,  	/*tp_getattr*/
	(setattrfunc)0,		  	/*tp_setattr*/
	(cmpfunc)0,		  	/*tp_compare*/
	(reprfunc)0,		  	/*tp_repr*/
        0,			  	/*tp_as_number*/
	0,				/*tp_as_number*/
	0,				/*tp_as_mapping*/
	(hashfunc)0,			/*tp_hash*/
	(ternaryfunc)0,			/*tp_call*/
	(reprfunc)0,			/*tp_str*/
	(getattrofunc)0, 		/*tp_getattro*/
	(setattrofunc)0, 		/*tp_setattro*/
};

/* Python Method Table */

statichere
PyMethodDef mxFS_Methods[] =
{   
    Py_MethodListEntry("search",mxFS_search),
    Py_MethodListEntry("find",mxFS_find),
    Py_MethodListEntry("findall",mxFS_findall),
    {NULL,NULL} /* end of list */
};

#endif

/* --- Module functions ------------------------------------------------*/

/* Interface to the tagging engine in mxte.c */

Py_C_Function( mxTextTools_tag,
	       "tag(text,tagtable,[startindex=0,len_text=len(text),taglist=[]]) \n"""
	       "Produce a tag list for a string, given a tag-table\n"
	       "- returns a tuple (success, taglist, nextindex)\n"
	       "- if taglist == None, then no taglist is created\n"
	       "- note: this function does not except keywords !")
{
    PyObject *pytext = 0;
    PyObject *table = 0;
    int len_text = INT_MAX;
    int start = 0;
    PyObject *taglist = 0;
    int next,result;
    PyObject *res;
    
    Py_Get5Args("OO|iiO:tag",pytext,table,start,len_text,taglist);

    if (taglist == NULL) { 
	/* not given, so use default: an empty list */
	taglist = PyList_New(0);
	if (taglist == NULL)
	    goto onError;
    }
    else {
	Py_INCREF(taglist);
	Py_Assert(PyList_Check(taglist) || taglist == Py_None,
		  PyExc_TypeError,
		  "optional fourth argument must be a list or None");
    }
    
    Py_Assert(PyString_Check(pytext),
	      PyExc_TypeError,
	      "first argument must be a string");
    Py_Assert(PyTuple_Check(table),
	      PyExc_TypeError,
	      "second argument must be a tuple (the tag table)");

    Py_CheckSlice(pytext,start,len_text);

    /* Call the tagging engine */
    result = fast_tag(pytext,
		      PyString_AS_STRING(pytext),
		      len_text,
		      table,
		      start,
		      taglist,
		      &next);

    if (result == 0)
	goto onError;
    result--;

    /* Build result tuple */
    res = PyTuple_New(3);
    if (!res)
	goto onError;
    PyTuple_SET_ITEM(res,0,PyInt_FromLong(result));
    PyTuple_SET_ITEM(res,1,taglist);
    PyTuple_SET_ITEM(res,2,PyInt_FromLong(next));
    return res;

 onError:
    if (!PyErr_Occurred())
	Py_Error(PyExc_SystemError,
		 "NULL result without error in builtin tag()");
    Py_XDECREF(taglist);
    return NULL;
}

/* An extended version of string.join() for taglists: */

Py_C_Function( mxTextTools_join,
	       "join(joinlist,sep='')\n"
	       "Copy snippets from different strings together producing a\n"
	       "new string\n"
	       "- the first argument must be a list of tuples or strings;\n"
	       "  tuples must be of the form (string,l,r[,...]) and turn out\n"
	       "  as string[l:r]\n"
	       "- NOTE: the syntax used for negative slices is different\n"
	       "  than the Python standard: -1 corresponds to the first\n"
	       "  character *after* the string, e.g. ('Example',0,-1) gives\n"
	       "  'Example' and not 'Exampl', like in Python\n"
	       "- takes an optional argument: a separator string")
{
    PyObject *taglist = 0;
    register int i;
    int len_list;
    char *st;
    int len_newstring;
    int current_len = 0;
    PyObject *newstring = 0;
    char *p;
    char *sep;
    int len_sep = 0;

    Py_Get3Args("O|s#:join",taglist,sep,len_sep);

    Py_Assert(PyList_Check(taglist),
	      PyExc_TypeError,
	      "first argument needs to be a list");

    len_list = PyList_GET_SIZE(taglist);

    /* Short-cut */
    if (len_list == 0)
	return PyString_FromString("");

    /* Create an empty new string */
    len_newstring = (10 + len_sep) * len_list;
    newstring = PyString_FromStringAndSize((char*)NULL,len_newstring);
    if (newstring == NULL) 
	goto onError;
    p = PyString_AS_STRING(newstring);

    if (len_sep == 0)

	/* No seperator */
	for (i = 0; i < len_list; i++) {
	    register PyObject *o;
	    int len_st;

	    o = PyList_GET_ITEM(taglist,i);

	    if (PyTuple_Check(o)) {
		/* Tuple entry: (string,l,r,[...]) */
		PyObject *s,*v;
		register int l,r;

		/* parse tuple */
		Py_Assert(PyTuple_GET_SIZE(o) >= 3,
			  PyExc_TypeError,
			  "tuples must have length >= 3");
		s = PyTuple_GET_ITEM(o,0);
		Py_Assert(PyString_Check(s),
			  PyExc_TypeError,
			  "tuples must be of the format (string,l,r[,...])");
		v = PyTuple_GET_ITEM(o,1);
		Py_Assert(PyInt_Check(v),
			  PyExc_TypeError,
			  "tuples must be of the format (string,l,r[,...])");
		l = PyInt_AS_LONG(v);
		v = PyTuple_GET_ITEM(o,2);
		Py_Assert(PyInt_Check(v),
			  PyExc_TypeError,
			  "tuples must be of the format (string,l,r[,...])");
		r = PyInt_AS_LONG(v);
		len_st = PyString_GET_SIZE(s);

		/* compute slice */
		if (r > len_st) r = len_st;
		else if (r < 0) {
		    r += len_st + 1;
		    if (r < 0)
			r = 0;
		}
		if (l > len_st) l = len_st;
		else if (l < 0) {
		    l += len_st + 1;
		    if (l < 0)
			l = 0;
		}

		/* empty ? */
		if (l > r)
		    continue;
		len_st = r - l;
		if (len_st == 0) 
		    continue;

		/* get pointer right */
		st = PyString_AS_STRING(s) + l;
	    }
	    else if (PyString_Check(o)) {
		/* String entry: take the whole string */
		st = PyString_AS_STRING(o);
		len_st = PyString_GET_SIZE(o);
	    }
	    else
		Py_Error(PyExc_TypeError,
			 "list must contain tuples or strings as entries");

	    /* Resize the new string if needed */
	    while (current_len + len_st >= len_newstring) {
		len_newstring += len_newstring >> 1;
		if (_PyString_Resize(&newstring, len_newstring))
		    goto onError;
		p = PyString_AS_STRING(newstring) + current_len;
	    }

	    /* Copy snippet into new string */
	    memcpy(p,st,len_st);
	    p += len_st;
	    current_len += len_st;
	}

    else

	/* Join with seperator */
	for (i = 0; i < len_list; i++) {
	    register PyObject *o;
	    int len_st;

	    o = PyList_GET_ITEM(taglist,i);

	    if PyTuple_Check(o) {
		/* Tuple entry: (string,l,r,[...]) */
		PyObject *s,*v;
		register int l,r;

		/* parse tuple */
		Py_Assert(PyTuple_GET_SIZE(o) >= 3,
			  PyExc_TypeError,
			  "tuples must have length >= 3");
		s = PyTuple_GET_ITEM(o,0);
		Py_Assert(PyString_Check(s),
			  PyExc_TypeError,
			  "tuples must be of the format (string,l,r[,...])");
		v = PyTuple_GET_ITEM(o,1);
		Py_Assert(PyInt_Check(v),
			  PyExc_TypeError,
			  "tuples must be of the format (string,l,r[,...])");
		l = PyInt_AS_LONG(v);
		v = PyTuple_GET_ITEM(o,2);
		Py_Assert(PyInt_Check(v),
			  PyExc_TypeError,
			  "tuples must be of the format (string,l,r[,...])");
		r = PyInt_AS_LONG(v);
		len_st = PyString_GET_SIZE(s);

		/* compute slice */
		if (r > len_st) r = len_st;
		else if (r < 0) {
		    r += len_st + 1;
		    if (r < 0)
			r = 0;
		}
		if (l > len_st) l = len_st;
		else if (l < 0) {
		    l += len_st + 1;
		    if (l < 0)
			l = 0;
		}

		/* empty ? */
		if (l > r)
		    continue;
		len_st = r - l;
		if (len_st == 0) 
		    continue;

		/* get pointer right */
		st = PyString_AS_STRING(s) + l;
	    }
	    else if (PyString_Check(o)) {
		/* String entry: take the whole string */
		st = PyString_AS_STRING(o);
		len_st = PyString_GET_SIZE(o);
	    }
	    else
		Py_Error(PyExc_TypeError,
			 "list must contain tuples or strings as entries");

	    /* Resize the new string if needed */
	    while (current_len + len_st + len_sep >= len_newstring) {
		len_newstring += len_newstring >> 1;
		if (_PyString_Resize(&newstring, len_newstring))
		    goto onError;
		p = PyString_AS_STRING(newstring) + current_len;
	    }

	    /* Insert separator */
	    if (i > 0) {
		memcpy(p, sep, len_sep);
		p += len_sep;
		current_len += len_sep;
	    }

	    /* Copy snippet into new string */
	    memcpy(p,st,len_st);
	    p += len_st;
	    current_len += len_st;
	}

    /* Resize new string to the actual length */
    if (_PyString_Resize(&newstring,current_len))
	goto onError;

    return newstring;

 onError:
    Py_XDECREF(newstring);
    return NULL;
}

/*
   Special compare function for taglist-tuples, comparing
   the text-slices given:
    - slices starting at a smaller index come first
    - for slices starting at the same index, the longer one
      wins
*/

Py_C_Function( mxTextTools_cmp,
	       "cmp(a,b)\n"
	       "Compare two valid taglist tuples w/r to their slice\n"
	       "position; this is useful for sorting joinlists.")
{
    PyObject *v,*w;
    int cmp;

    Py_Get2Args("OO:cmp",v,w);

    Py_Assert(PyTuple_Check(v) && PyTuple_Check(w) && 
	      PyTuple_GET_SIZE(v) >= 3 && PyTuple_GET_SIZE(w) >= 3,
	      PyExc_TypeError,
	      "invalid taglist-tuple");

    cmp = PyObject_Compare(PyTuple_GET_ITEM(v,1),PyTuple_GET_ITEM(w,1));
    if (cmp != 0) 
	return PyInt_FromLong(cmp);
    cmp = - PyObject_Compare(PyTuple_GET_ITEM(v,2),PyTuple_GET_ITEM(w,2));
    return PyInt_FromLong(cmp);

 onError:
    return NULL;
}

Py_C_Function( mxTextTools_joinlist,
	       "joinlist(text,list,start=0,stop=len(text))\n"
	       "Takes a list of tuples (replacement,l,r,...) and produces\n"
	       "a taglist suitable for join() which creates a copy\n"
	       "of text where every slice [l:r] is replaced by the\n"
	       "given replacement\n"
	       "- the list must be sorted using cmp() as compare function\n"
	       "- it may not contain overlapping slices\n"
	       "- the slices may not contain negative indices\n"
	       "- if the taglist cannot contain overlapping slices, you can\n"
	       "  give this function the taglist produced by tag() directly\n"
	       "  (sorting is not needed, as the list will already be sorted)\n"
	       "- start and stop set the slice to work in, i.e. text[start:stop]"
)
{
    PyObject *list;
    PyObject *text;
    PyObject *joinlist = 0;
    int len_list;
    int len_text = INT_MAX;
    int pos = 0;
    register int i;
    int listitem = 0;
    int listsize = INITIAL_LIST_SIZE;
    
    Py_Get4Args("OO|ii:joinlist",text,list,pos,len_text);

    Py_Assert(PyString_Check(text),
	      PyExc_TypeError,
	      "first argument needs to be a string");
    Py_Assert(PyList_Check(list),
	      PyExc_TypeError,
	      "second argument needs to be a list");

    if (len_text > PyString_GET_SIZE(text))
	len_text = PyString_GET_SIZE(text);
    
    len_list = PyList_GET_SIZE(list);

    joinlist = PyList_New(listsize);
    if (joinlist == NULL)
	goto onError;

    for (i = 0; i < len_list; i++) {
	register PyObject *t;
	register int left,right;
	
	t = PyList_GET_ITEM(list,i);
	Py_Assert(PyTuple_Check(t) && (PyTuple_GET_SIZE(t) >= 3),
		  PyExc_TypeError,
		  "list must contain tuples of length >= 3");
	Py_Assert(PyInt_Check(PyTuple_GET_ITEM(t,1)) &&
		  PyInt_Check(PyTuple_GET_ITEM(t,2)),
		  PyExc_TypeError,
		  "tuples must be of the form (string,int,int,...)");
	left = PyInt_AS_LONG(PyTuple_GET_ITEM(t,1));
	right = PyInt_AS_LONG(PyTuple_GET_ITEM(t,2));

	Py_Assert(left >= pos,
		  PyExc_ValueError,
		  "list is not sorted ascending");

	if (left > pos) { /* joinlist.append((text,pos,left)) */
	    register PyObject *v;
	    register PyObject *w;
	    
	    v = PyTuple_New(3);
	    if (v == NULL)
		goto onError;

	    Py_INCREF(text);
	    PyTuple_SET_ITEM(v,0,text);

	    w = PyInt_FromLong(pos);
	    if (w == NULL)
		goto onError;
	    PyTuple_SET_ITEM(v,1,w);

	    w = PyTuple_GET_ITEM(t,1);
	    Py_INCREF(w);
	    PyTuple_SET_ITEM(v,2,w);

	    if (listitem < listsize)
		PyList_SET_ITEM(joinlist,listitem,v);
	    else {
		PyList_Append(joinlist,v);
		Py_DECREF(v);
	    }
	    listitem++;
	}
	
	/* joinlist.append(string) */
	if (listitem < listsize) {
	    register PyObject *v = PyTuple_GET_ITEM(t,0);
	    Py_INCREF(v);
	    PyList_SET_ITEM(joinlist,listitem,v);
	}
	else
	    PyList_Append(joinlist,PyTuple_GET_ITEM(t,0));
	listitem++;
	
	pos = right;
    }
    
    if (pos < len_text) { /* joinlist.append((text,pos,len_text)) */
	register PyObject *v;
	register PyObject *w;
	    
	v = PyTuple_New(3);
	if (v == NULL)
	    goto onError;

	Py_INCREF(text);
	PyTuple_SET_ITEM(v,0,text);

	w = PyInt_FromLong(pos);
	if (w == NULL)
	    goto onError;
	PyTuple_SET_ITEM(v,1,w);

	w = PyInt_FromLong(len_text);
	if (w == NULL)
	    goto onError;
	PyTuple_SET_ITEM(v,2,w);

	if (listitem < listsize)
	    PyList_SET_ITEM(joinlist,listitem,v);
	else {
	    PyList_Append(joinlist,v);
	    Py_DECREF(v);
	}
	listitem++;
    }

    /* Resize list if necessary */
    if (listitem < listsize)
	PyList_SetSlice(joinlist,listitem,listsize,(PyObject*)NULL);

    return joinlist;

 onError:

    Py_XDECREF(joinlist);
    return NULL;
}

Py_C_Function( mxTextTools_charsplit,
	       "charsplit(text,char,start=0,stop=len(text))\n"
	       "Split text[start:stop] into substrings at char and\n"
	       "return the result as list of strings."
)
{
    PyObject *text;
    PyObject *list = 0;
    char *seperator;
    int sep_len;
    int len_text = INT_MAX;
    int start = 0;
    register int x;
    char *tx;
    int listitem = 0;
    int listsize = INITIAL_LIST_SIZE;

    Py_Get5Args("Os#|ii:charsplit",text,seperator,sep_len,start,len_text);

    Py_Assert(PyString_Check(text),
	      PyExc_TypeError,
	      "first argument needs to be a string");
    Py_Assert(sep_len == 1,
	      PyExc_TypeError,
	      "second argument needs to be a single character");
    Py_CheckSlice(text,start,len_text);

    list = PyList_New(listsize);
    if (!list)
	goto onError;

    x = start;
    tx = PyString_AS_STRING(text);

    while (1) {
	PyObject *s;
	register int z;

	/* Skip to next seperator */
	z = x;
	for (;x < len_text; x++) 
	    if (tx[x] == *seperator)
		break;

	/* Append the slice to list */
	s = PyString_FromStringAndSize(&tx[z], x - z);
	if (!s)
	    goto onError;
	if (listitem < listsize)
	    PyList_SET_ITEM(list,listitem,s);
	else {
	    PyList_Append(list,s);
	    Py_DECREF(s);
	}
	listitem++;

	if (x == len_text)
	    break;

	/* Skip seperator */
	x++;
    }

    /* Resize list if necessary */
    if (listitem < listsize)
	PyList_SetSlice(list,listitem,listsize,(PyObject*)NULL);

    return list;
    
 onError:
    Py_XDECREF(list);
    return NULL;
}

Py_C_Function( mxTextTools_splitat,
	       "splitat(text,char,nth=1,start=0,stop=len(text))\n"
	       "Split text[start:stop] into two substrings at the nth\n"
	       "occurance of char and return the result as 2-tuple. If the\n"
	       "character is not found, the second string is empty. nth may\n"
	       "be negative: the search is then done from the right and the\n"
	       "first string is empty in case the character is not found."
)
{
    PyObject *text;
    PyObject *tuple = 0;
    char *seperator;
    int sep_len;
    int len_text = INT_MAX;
    int start = 0;
    int nth = 1;
    register int x;
    char *tx;
    PyObject *s;

    Py_Get6Args("Os#|iii:splitat",text,seperator,sep_len,nth,start,len_text);

    Py_Assert(PyString_Check(text),
	      PyExc_TypeError,
	      "first argument needs to be a string");
    Py_Assert(sep_len == 1,
	      PyExc_TypeError,
	      "second argument needs to be a single character");
    Py_CheckSlice(text,start,len_text);

    tuple = PyTuple_New(2);
    if (!tuple)
	goto onError;

    tx = PyString_AS_STRING(text);

    if (nth > 0) {
	/* Skip to nth seperator from the left */
	x = start;
	while (1) {
	    for (; x < len_text; x++) 
		if (tx[x] == *seperator)
		    break;
	    if (--nth == 0 || x == len_text)
		break;
	    x++;
	}
    }
    else if (nth < 0) {
	/* Skip to nth seperator from the right */
	x = len_text - 1;
	while (1) {
	    for (; x >= start; x--) 
		if (tx[x] == *seperator)
		    break;
	    if (++nth == 0 || x < start)
		break;
	    x--;
	}
    }
    else
	Py_Error(PyExc_ValueError,
		 "nth must be non-zero");
    
    /* Add to tuple */
    if (x < start)
	s = PyString_FromStringAndSize("",0);
    else
	s = PyString_FromStringAndSize(&tx[start], x - start);
    if (!s)
	goto onError;
    PyTuple_SET_ITEM(tuple,0,s);

    /* Skip seperator */
    x++;

    if (x >= len_text)
	s = PyString_FromStringAndSize("",0);
    else
	s = PyString_FromStringAndSize(&tx[x], len_text - x);
    if (!s)
	goto onError;
    PyTuple_SET_ITEM(tuple,1,s);

    return tuple;
    
 onError:
    Py_XDECREF(tuple);
    return NULL;
}

Py_C_Function( mxTextTools_set,
	       "set(string,logic=1)\n"
	       "Returns a character set for string: a bit encoded version\n"
	       "of the characters occurring in string.\n"
	       "- logic can be set to 0 if all characters *not* in string\n"
	       "  should go into the set")
{
    PyObject *sto;
    char *s,*st;
    int len_s;
    int logic = 1;
    int i;

    Py_Get3Args("s#|i:set",s,len_s,logic);

    sto = PyString_FromStringAndSize(NULL,32);
    if (sto == NULL)
	goto onError;
    
    st = PyString_AS_STRING(sto);

    if (logic) {
	memset(st,0x00,32);
	for (i = 0; i < len_s; i++,s++) {
	    int j = (unsigned char)*s;
	    
	    st[j >> 3] |= 1 << (j & 7);
	}
    }
    else {
	memset(st,0xFF,32);
	for (i = 0; i < len_s; i++,s++) {
	    int j = (unsigned char)*s;
	    
	    st[j >> 3] &= ~(1 << (j & 7));
	}
    }
    return sto;

 onError:
    return NULL;
}

Py_C_Function( mxTextTools_setfind,
	       "setfind(text,set,start=0,stop=len(text))\n"
	       "Find the first occurence of any character from set in\n"
	       "text[start:stop]\n"
	       "- set must be a string obtained with set()"
)
{
    PyObject *text;
    PyObject *set;
    int len_text = INT_MAX;
    int start = 0;
    register int x;
    register char *tx;
    register unsigned char *setstr;
    
    Py_Get4Args("OO|ii:setfind",text,set,start,len_text);

    Py_Assert(PyString_Check(text),
	      PyExc_TypeError,
	      "first argument needs to be a string");
    Py_Assert(PyString_Check(set) && PyString_GET_SIZE(set) == 32,
	      PyExc_TypeError,
	      "second argument needs to be a set");
    Py_CheckSlice(text,start,len_text);

    x = start;
    tx = PyString_AS_STRING(text) + x;
    setstr = (unsigned char *)PyString_AS_STRING(set);

    for (;x < len_text &&
	  (setstr[(unsigned char)*tx >> 3] & (1 << (*tx & 7))) == 0;
	 tx++, x++) ;
    
    if (x == len_text)
	/* Not found */
	return PyInt_FromLong(-1L);
    else
	return PyInt_FromLong(x);

 onError:
    return NULL;
}

Py_C_Function( mxTextTools_setsplit,
	       "setsplit(text,set,start=0,stop=len(text))\n"
	       "Split text[start:stop] into substrings using set,\n"
	       "omitting the splitting parts and empty substrings.\n"
	       "- set must be a string obtained from set()"
)
{
    PyObject *text;
    PyObject *set;
    PyObject *list = 0;
    int len_text = INT_MAX;
    int start = 0;
    register int x;
    unsigned char *tx;
    unsigned char *setstr;
    int listitem = 0;
    int listsize = INITIAL_LIST_SIZE;

    Py_Get4Args("OO|ii:setsplit",text,set,start,len_text);

    Py_Assert(PyString_Check(text),
	      PyExc_TypeError,
	      "first argument needs to be a string");
    Py_Assert(PyString_Check(set) && PyString_GET_SIZE(set) == 32,
	      PyExc_TypeError,
	      "second argument needs to be a set");
    Py_CheckSlice(text,start,len_text);

    list = PyList_New(listsize);
    if (!list)
	goto onError;

    x = start;
    tx = (unsigned char *)PyString_AS_STRING(text);
    setstr = (unsigned char *)PyString_AS_STRING(set);

    while (x < len_text) {
	register int z;

	/* Skip all text not in set */
	z = x;
	for (;x < len_text; x++) {
	    register unsigned int c = tx[x];
	    register unsigned int block = setstr[c >> 3];
	    if (block && ((block & (1 << (c & 7))) != 0))
		break;
	}

	/* Append the slice to list if it is not empty */
	if (x > z) {
	    PyObject *s;
	    s = PyString_FromStringAndSize(&tx[z], x - z);
	    if (!s)
		goto onError;
	    if (listitem < listsize)
		PyList_SET_ITEM(list,listitem,s);
	    else {
		PyList_Append(list,s);
		Py_DECREF(s);
	    }
	    listitem++;
	}

	if (x == len_text)
	    break;

	/* Skip all text in set */
	for (;x < len_text; x++) {
	    register unsigned int c = tx[x];
	    register unsigned int block = setstr[c >> 3];
	    if (!block || ((block & (1 << (c & 7))) == 0))
		break;
	}
    }

    /* Resize list if necessary */
    if (listitem < listsize)
	PyList_SetSlice(list,listitem,listsize,(PyObject*)NULL);

    return list;
    
 onError:
    Py_XDECREF(list);
    return NULL;
}

Py_C_Function( mxTextTools_setsplitx,
	       "setsplitx(text,set,start=0,stop=len(text))\n"
	       "Split text[start:stop] into substrings using set, so\n"
	       "that every second entry consists only of characters in set.\n"
	       "- set must be a string obtained with set()"
)
{
    PyObject *text;
    PyObject *set;
    PyObject *list = 0;
    int len_text = INT_MAX;
    int start = 0;
    register int x;
    unsigned char *tx;
    unsigned char *setstr;
    int listitem = 0;
    int listsize = INITIAL_LIST_SIZE;

    Py_Get4Args("OO|ii:setsplit",text,set,start,len_text);

    Py_Assert(PyString_Check(text),
	      PyExc_TypeError,
	      "first argument needs to be a string");
    Py_Assert(PyString_Check(set) && PyString_GET_SIZE(set) == 32,
	      PyExc_TypeError,
	      "second argument needs to be a set");
    Py_CheckSlice(text,start,len_text);

    list = PyList_New(listsize);
    if (!list)
	goto onError;

    x = start;
    tx = (unsigned char *)PyString_AS_STRING(text);
    setstr = (unsigned char *)PyString_AS_STRING(set);

    while (x < len_text) {
	PyObject *s;
	register int z;

	/* Skip all text not in set */
	z = x;
	for (;x < len_text; x++) {
	    register unsigned int c = tx[x];
	    register unsigned int block = setstr[c >> 3];
	    if (block && ((block & (1 << (c & 7))) != 0))
		break;
	}

	/* Append the slice to list */
	s = PyString_FromStringAndSize(&tx[z], x - z);
	if (!s)
	    goto onError;
	if (listitem < listsize)
	    PyList_SET_ITEM(list,listitem,s);
	else {
	    PyList_Append(list,s);
	    Py_DECREF(s);
	}
	listitem++;

	if (x >= len_text)
	    break;

	/* Skip all text in set */
	z = x;
	for (;x < len_text; x++) {
	    register unsigned int c = tx[x];
	    register unsigned int block = setstr[c >> 3];
	    if (!block || ((block & (1 << (c & 7))) == 0))
		break;
	}

	/* Append the slice to list if it is not empty */
	s = PyString_FromStringAndSize(&tx[z], x - z);
	if (!s)
	    goto onError;
	if (listitem < listsize)
	    PyList_SET_ITEM(list,listitem,s);
	else {
	    PyList_Append(list,s);
	    Py_DECREF(s);
	}
	listitem++;
    }

    /* Resize list if necessary */
    if (listitem < listsize)
	PyList_SetSlice(list,listitem,listsize,(PyObject*)NULL);

    return list;
    
 onError:
    Py_XDECREF(list);
    return NULL;
}

Py_C_Function( mxTextTools_upper,
	       "upper(text)\n"
	       "Return text converted to upper case.")
{
    PyObject *text;
    PyObject *ntext;
    register unsigned char *s;
    register unsigned char *orig;
    register int i;
    unsigned char *tr;
    int	len;
    
    Py_GetArgObject(text);

    Py_Assert(PyString_Check(text),
	      PyExc_TypeError,
	      "expected a Python string");

    len = PyString_GET_SIZE(text);
    ntext = PyString_FromStringAndSize(NULL,len);
    if (!ntext)
	goto onError;
    
    /* Translate */
    tr = (unsigned char *)PyString_AS_STRING(mxTo_Upper);
    orig = (unsigned char *)PyString_AS_STRING(text);
    s = (unsigned char *)PyString_AS_STRING(ntext);
    for (i = 0; i < len; i++, s++, orig++)
	*s = tr[*orig];
    
    return ntext;
    
 onError:
    return NULL;
}

Py_C_Function( mxTextTools_lower,
	       "lower(text)\n"
	       "Return text converted to lower case.")
{
    PyObject *text;
    PyObject *ntext;
    register unsigned char *s;
    register unsigned char *orig;
    register int i;
    unsigned char *tr;
    int len;
    
    Py_GetArgObject(text);

    Py_Assert(PyString_Check(text),
	      PyExc_TypeError,
	      "expected a Python string");

    len = PyString_GET_SIZE(text);
    ntext = PyString_FromStringAndSize(NULL,len);
    if (!ntext)
	goto onError;
    
    /* Translate */
    tr = (unsigned char *)PyString_AS_STRING(mxTo_Lower);
    orig = (unsigned char *)PyString_AS_STRING(text);
    s = (unsigned char *)PyString_AS_STRING(ntext);
    for (i = 0; i < len; i++, s++, orig++)
	*s = tr[*orig];
    
    return ntext;
    
 onError:
    return NULL;
}

/* --- module init --------------------------------------------------------- */

/* Python Method Table */

static PyMethodDef Module_methods[] =
{   
    Py_MethodListEntry("tag",mxTextTools_tag),
    Py_MethodListEntry("join",mxTextTools_join),
    Py_MethodListEntry("cmp",mxTextTools_cmp),
    Py_MethodListEntry("joinlist",mxTextTools_joinlist),
    Py_MethodListEntry("set",mxTextTools_set),
    Py_MethodListEntry("setfind",mxTextTools_setfind),
    Py_MethodListEntry("setsplit",mxTextTools_setsplit),
    Py_MethodListEntry("setsplitx",mxTextTools_setsplitx),
    Py_MethodListEntry("BMS",mxBMS_new),
#ifdef MXFASTSEARCH
    Py_MethodListEntry("FS",mxFS_new),
#endif
    Py_MethodListEntrySingleArg("upper",mxTextTools_upper),
    Py_MethodListEntrySingleArg("lower",mxTextTools_lower),
    Py_MethodListEntry("charsplit",mxTextTools_charsplit),
    Py_MethodListEntry("splitat",mxTextTools_splitat),
    {NULL,NULL} /* end of list */
};

/* Cleanup function */
static 
void mxTextToolsModule_Cleanup(void)
{
}

MX_EXPORT(void) 
     initmxTextTools(void)
{
    PyObject *module, *moddict;
    
    /* Init type objects */
    PyType_Init(mxBMS_Type);
#ifdef MXFASTSEARCH
    PyType_Init(mxFS_Type);
#endif

    /* create module */
    module = Py_InitModule4(MXTEXTTOOLS_MODULE, /* Module name */
			    Module_methods, /* Method list */
			    Module_docstring, /* Module doc-string */
			    (PyObject *)NULL, /* always pass this as *self */
			    PYTHON_API_VERSION); /* API Version */
    if (!module)
	goto onError;

    /* Register cleanup function */
    if (Py_AtExit(mxTextToolsModule_Cleanup))
	/* XXX what to do if we can't register that function ??? */;

    /* Add some symbolic constants to the module */
    moddict = PyModule_GetDict(module);
    PyDict_SetItemString(moddict, 
			 "__version__",
			 PyString_FromString(VERSION));

    mxTo_Upper = mxTo_Upper_New();
    PyDict_SetItemString(moddict, 
			 "to_upper",
			 mxTo_Upper);

    mxTo_Lower = mxTo_Lower_New();
    PyDict_SetItemString(moddict, 
			 "to_lower",
			 mxTo_Lower);
  
    /* Type objects */
    Py_INCREF(&mxBMS_Type);
    PyDict_SetItemString(moddict, "BMSType",
			 (PyObject *)&mxBMS_Type);
#ifdef MXFASTSEARCH
    Py_INCREF(&mxFS_Type);
    PyDict_SetItemString(moddict, "FSType",
			 (PyObject *)&mxFS_Type);
#endif

    /* Check for errors */
    if (PyErr_Occurred())
	Py_Error(PyExc_ImportError,
		 "initialization of module "MXTEXTTOOLS_MODULE" failed");
 onError:
    return;
}
