/*
	test pattern generator
	**********************
	$Id: testpattern.c,v 1.14 1998/10/12 20:15:31 marcus Exp $
	(c) H. Niemann  1997

	This program creates a test pattern like ctmon3 and the
	Nokia montest program created or is/was broadcast on tv
	during program breaks, when there was no 24h program.

	The second part is a convergence test researched from the
	Nokia program, to check whether red, green and blue beams
	are adjusted properly.

	for a first glance: run
		testpattern 4 640 480
	or
		testpattern 8 320 240

	Very much is copied from ./demo.c
	The text output would require about 400x300 dots, but it works
	with less too.
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>

#include <ggi/ggi.h>

#define ASSERT(x)	((x) || fprintf(stderr, "testpattern: ASSERT failed\n"))

ggi_visual_t vis;

ggi_uint white, black, red, green, blue, yellow, magenta, cyan;  

void usage(const char *prog)
{
	fprintf(stderr,"Usage:\n\n"
		       "%s <bpp> <xsize> <ysize> [<virtx> <virty>]\n\n"
		       "Default: %s 8 320 200 320 200\n",prog,prog);
	exit(1);
}

void waitabit(void)
{
	int key;
	key = ggiGetc(vis);
	if ((key == 'q') || (key == 'Q')) /* Q pressed */
		ggiExit();
}

void dotone(int x1,int y1,int x2,int y2,ggi_uint col1, ggi_uint col2)
{
	int x,y;

	ggiSetGCForeground(vis,col1);
	ggiDrawBox(vis,x1,y1,x2-x1+1,y2-y1+1);

	ggiSetGCForeground(vis,col2);
	for (x=x1; x<=x2;x+=2)
		for (y=y1; y<=y2;y+=2)
			ggiPutPixel(vis,x,y,col2);
}
void chessboardone(int x1,int y1,int x2,int y2,ggi_uint col1, ggi_uint col2)
{
	int x,y;

	ggiSetGCForeground(vis,col1);
	ggiDrawBox(vis,x1,y1,x2-x1+1,y2-y1+1);

	for (x=x1; x<x2;x+=2)
		for (y=y1; y<y2;y+=2){
			ggiPutPixel(vis,x,y,col2);
			ggiPutPixel(vis,x+1,y+1,col2);
		}
}
                 
void stripevert(int x1,int y1,int x2,int y2,ggi_uint col1, ggi_uint col2, ggi_uint s)
{
	int x,xi;
	assert(s>0);

	ggiSetGCForeground(vis,col1);
	ggiDrawBox(vis,x1,y1,x2-x1+1,y2-y1+1);

	ggiSetGCForeground(vis,col2);
	for (x=x1; x<=x2;x+=s+s)
		for (xi=0; xi<s;xi++)
			ggiDrawVLine(vis,x+xi,y1,y2-y1+1);
}
void stripehor(int x1,int y1,int x2,int y2,ggi_uint col1, ggi_uint col2,ggi_uint s)
{
	int y,yi;
	assert(s>0);

	ggiSetGCForeground(vis,col1);
	ggiDrawBox(vis,x1,y1,x2-x1+1,y2-y1+1);

	ggiSetGCForeground(vis,col2);
	for (y=y1; y<=y2;y+=s+s)
		for (yi=0;yi<s;yi++)
			ggiDrawHLine(vis,x1,y+yi,x2-x1+1);
}

void starone(int x1,int y1,int x2,int y2,ggi_uint col1, ggi_uint col2)
{
	int x,y;

	ggiSetGCForeground(vis,col1);
	ggiDrawBox(vis,x1,y1,x2-x1+1,y2-y1+1);

	ggiSetGCForeground(vis,col2);
	for (x=x1; x<=x2;x+=2)
		ggiDrawLine(vis,x,y1,x2+x1-x,y2);
	for (y=y1; y<=y2;y+=2)
		ggiDrawLine(vis,x1,y,x2,y2+y1-y);
}

void starfive(int x1,int y1,int x2,int y2,ggi_uint col1, ggi_uint col2)
{
	int x,y;
	ggiSetGCForeground(vis,col1);
	ggiDrawBox(vis,x1,y1,x2-x1+1,y2-y1+1);

        ggiSetGCForeground(vis,col2);
	for (x=x1; x<=x2;x+=5)
		ggiDrawLine(vis,x,y1,x2+x1-x,y2);
	for (y=y1; y<=y2;y+=5)
		ggiDrawLine(vis,x1,y,x2,y2+y1-y);
}
void startwoten(int x1,int y1,int x2,int y2,ggi_uint col1, ggi_uint col2)
{
	int x,y;

	ggiSetGCForeground(vis,col1);
	ggiDrawBox(vis,x1,y1,x2-x1+1,y2-y1+1);

	ggiSetGCForeground(vis,col2);
	x = x1 ;
	while (x < x2){
		ggiDrawLine(vis,x,y1,x2+x1-x,y2);
		ggiDrawLine(vis,x+1,y1,x2+x1-x-1,y2);
		x +=10;
	}
	y = y1 ;
	while (y < y2) {
		ggiDrawLine(vis,x1,y,x2,y2+y1-y);
		ggiDrawLine(vis,x1,y+1,x2,y2+y1-y-1);
		y += 10;
	}
}
void setcolors(void)
{
	/* just set the needed colour names */
	ggi_color col;
	ggi_mode mode;

	ggiGetMode(vis, &mode);

	/* If we're in indexed mode, the initial palette is undefined */
	if(GT_SCHEME(mode.graphtype)==GT_PALETTE) {
		ggi_color colormap[8];
		int i;

		for(i = 0; i<8; i++) {
			colormap[i].r = (i & 4) ? 0xFFFF : 0;
			colormap[i].g = (i & 2) ? 0xFFFF : 0;
			colormap[i].b = (i & 1) ? 0xFFFF : 0;
		}
		
		/* Place colormap anywhere the target wants */
		ggiSetPalette(vis, -1, 8, colormap);
	}

	col.r=0xFFFF; col.g=0xFFFF; col.b=0xFFFF;
	white=ggiMapColor(vis, &col);

	col.r=0xFFFF; col.g=0xFFFF; col.b=0x0000;
	yellow=ggiMapColor(vis, &col);

	col.r=0xFFFF; col.g=0x0000; col.b=0xFFFF;
	magenta=ggiMapColor(vis, &col);

	col.r=0xFFFF; col.g=0x0000; col.b=0x0000;
	red=ggiMapColor(vis, &col);

	col.r=0x0000; col.g=0xFFFF; col.b=0xFFFF;
	cyan=ggiMapColor(vis, &col);

	col.r=0x0000; col.g=0xFFFF; col.b=0x0000;
	green=ggiMapColor(vis, &col);

	col.r=0x0000; col.g=0x0000; col.b=0xFFFF;
	blue=ggiMapColor(vis, &col);

	col.r=0x0000; col.g=0x0000; col.b=0x0000;
	black=ggiMapColor(vis, &col);
}

int circle(ggi_visual_t vis,int xcenter,int ycenter,int radius)
{
        /* easiest implementation, not time-optimised on weird archs. */
        int x,y,od,md,sd;

        /* original clipping code deleted. Leave clipping to ggiDrawPixel */
        /* shouldn't clip in this application anyway. */

        x=radius;y=md=0;
        while(x>=y) {
          ggiDrawPixel(vis,xcenter-y,ycenter-x);
          ggiDrawPixel(vis,xcenter+y,ycenter-x);
          ggiDrawPixel(vis,xcenter+x,ycenter-y);
          ggiDrawPixel(vis,xcenter-x,ycenter-y);
          ggiDrawPixel(vis,xcenter-x,ycenter+y);
          ggiDrawPixel(vis,xcenter+x,ycenter+y);
          ggiDrawPixel(vis,xcenter+y,ycenter+x);
          ggiDrawPixel(vis,xcenter-y,ycenter+x);
          od=md+y+y+1;sd=od-x-x-1;y++;md=od;
          if (abs(sd)<abs(od)) {x--;md=sd;}
        }

        return(0);
}

void testpattern(void)
{
	int xmax, ymax;
	ggi_color col;
	int i;
	ggi_mode currmode;

	int FONTX, FONTY; /* used to be #defines */

	ggiGetMode(vis,&currmode);
	xmax = currmode.visible.x;
	ymax = currmode.visible.y;

	ggiGetCharSize(vis, &FONTX, &FONTY);

	ggiSetGCForeground(vis,black);
	ggiFillscreen(vis);

#if 0
	ggiSetGCForeground(vis,yellow);
	ggiDrawBox(vis,0,0,xmax,ymax);
	ggiSetGCForeground(vis,6);[C
	ggiDrawHLine(vis,5,5,256);
	for (i=0;i<32;i++){

		int j;

		for (j=0;j<16;j++){

			ggiSetGCForeground(vis,i+32*j);
			ggiDrawBox(vis,i*8,j*16,8,16);
		}
		for (j=0;j<8;j++){

			ggiSetGCForeground(vis,j);
			ggiDrawBox(vis,xmax-24,j*16,8,16);

			ggiSetGCForeground(vis,8*j);
			ggiDrawBox(vis,xmax-8,j*16,8,16);

			ggiSetGCForeground(vis,32*j);
			ggiDrawBox(vis,xmax-16,j*16,8,16);
		}
	}
#endif

#define PARTSHOR 16
#define PARTSVERT 12
#define PARTX(x) ((x)*(xmax-1)/PARTSHOR)
#define PARTWIDTH (xmax/PARTSHOR)
#define PARTY(y) ((y)*(ymax-1)/PARTSVERT)
#define PARTHEIGHT (ymax/PARTSVERT)
#define MIDX PARTX(PARTSHOR/2)
#define MIDY PARTY(PARTSVERT/2)

	ggiSetGCForeground(vis,white);

	/* Raster */
	for (i=0;i<=PARTSHOR;i++) 
		ggiDrawVLine(vis,PARTX(i),0,ymax);
	for (i=0;i<=PARTSVERT;i++) {
		ggiDrawHLine(vis,0,PARTY(i),xmax);
	}

#define ggiDrawCircle(v,x,y,r) circle(v,x,y,r)

	ggiDrawCircle(vis,PARTX(1),PARTY(1),PARTHEIGHT);
	ggiDrawCircle(vis,PARTX(PARTSHOR-1),PARTY(1),PARTHEIGHT);
	ggiDrawCircle(vis,PARTX(1),PARTY(PARTSVERT-1),PARTHEIGHT);
	ggiDrawCircle(vis,PARTX(PARTSHOR-1),PARTY(PARTSVERT-1),PARTHEIGHT);
	ggiDrawCircle(vis,PARTX(PARTSHOR/2),PARTY(PARTSVERT/2),/*ymax/2*/
			(PARTY(PARTSVERT-1)-PARTY(1))/2);

	ggiSetGCForeground(vis,white);
	ggiDrawBox(vis,PARTX(PARTSHOR/2-4),PARTY(3)+1,PARTWIDTH,PARTY(5)-PARTY(3)-1);
	ggiSetGCForeground(vis,yellow);
	ggiDrawBox(vis,PARTX(PARTSHOR/2-3),PARTY(3)+1,PARTWIDTH,PARTY(5)-PARTY(3)-1);
	ggiSetGCForeground(vis,red);
	ggiDrawBox(vis,PARTX(PARTSHOR/2-2),PARTY(3)+1,PARTWIDTH,PARTY(5)-PARTY(3)-1);
	ggiSetGCForeground(vis,green);
	ggiDrawBox(vis,PARTX(PARTSHOR/2-1),PARTY(3)+1,PARTWIDTH,PARTY(5)-PARTY(3)-1);
	ggiSetGCForeground(vis,cyan);
	ggiDrawBox(vis,PARTX(PARTSHOR/2+0),PARTY(3)+1,PARTWIDTH,PARTY(5)-PARTY(3)-1);
	ggiSetGCForeground(vis,blue);
	ggiDrawBox(vis,PARTX(PARTSHOR/2+1),PARTY(3)+1,PARTWIDTH,PARTY(5)-PARTY(3)-1);
	ggiSetGCForeground(vis,magenta);
	ggiDrawBox(vis,PARTX(PARTSHOR/2+2),PARTY(3)+1,PARTWIDTH,PARTY(5)-PARTY(3)-1);
	ggiSetGCForeground(vis,black);
	ggiDrawBox(vis,PARTX(PARTSHOR/2+3),PARTY(3)+1,PARTWIDTH,PARTY(5)-PARTY(3)-1);

	{
		int xleft=PARTX(PARTSHOR/2-4)+1;
		int xright=PARTX(PARTSHOR/2+4)-1;
		int reddish=2,bluish=3,greenish=4,grey=5;
		int x,c;

		for (x=xleft;x<=xright;x++){
			c=((xright-x)*0xFFFF)/(xright-xleft);
			col.r=c; col.g=c; col.b=c;
			grey=ggiMapColor(vis, &col);
			col.r=c; col.g=0; col.b=0;
			reddish=ggiMapColor(vis, &col);
			col.r=0; col.g=c; col.b=0;
			greenish=ggiMapColor(vis, &col);
			col.r=0; col.g=0; col.b=c;
			bluish=ggiMapColor(vis, &col);
			ggiSetGCForeground(vis,reddish);
			ggiDrawVLine(vis,x,PARTY(PARTSVERT-5)+1,PARTHEIGHT/2);
			ggiSetGCForeground(vis,bluish);
			ggiDrawVLine(vis,x,PARTY(PARTSVERT-5)+1+PARTHEIGHT/2 ,PARTHEIGHT/2);
			ggiSetGCForeground(vis,greenish);
			ggiDrawVLine(vis,x,PARTY(PARTSVERT-4)+1,PARTHEIGHT/2);
			ggiSetGCForeground(vis,grey);
			ggiDrawVLine(vis,x,PARTY(PARTSVERT-4)+1+PARTHEIGHT/2 ,PARTHEIGHT/2);
		}
	}

	stripevert(PARTX(0)+1,PARTY(0)+1,
		PARTX(1)-1,PARTY(1)-1,black,white,1);
	stripevert(PARTX(PARTSHOR-1)+1,PARTY(0)+1,
		PARTX(PARTSHOR)-1,PARTY(1)-1,black,white,1);
	stripevert(PARTX(0)+1,PARTY(PARTSVERT-1)+1,
		PARTX(1)-1,PARTY(PARTSVERT)-1,black,white,1);
	stripevert(PARTX(PARTSHOR-1)+1,PARTY(PARTSVERT-1)+1,
		PARTX(PARTSHOR)-1,PARTY(PARTSVERT)-1,black,white,1);

	stripevert(PARTX(PARTSHOR/2-4)+1,PARTY(PARTSVERT/2-1)+1,
		PARTX(PARTSHOR/2-3)-1,PARTY(PARTSVERT/2)-1,black,white,1);
	stripevert(PARTX(PARTSHOR/2-4)+1,PARTY(PARTSVERT/2)+1,
		PARTX(PARTSHOR/2-3)-1,PARTY(PARTSVERT/2+1)-1,black,red,1);
	stripevert(PARTX(PARTSHOR/2+3)+1,PARTY(PARTSVERT/2-1)+1,
		PARTX(PARTSHOR/2+4)-1,PARTY(PARTSVERT/2)-1,black,green,1);
	stripevert(PARTX(PARTSHOR/2+3)+1,PARTY(PARTSVERT/2)+1,
		PARTX(PARTSHOR/2+4)-1,PARTY(PARTSVERT/2+1)-1,black,blue,1);

	{
		float horfreq=11007.0,vertfreq=29.789;  /* Hz */ /* FIXME */
		char s[255];
		char mode[30];

		switch (currmode.graphtype){
			case GT_TEXT16:
				sprintf(mode,"text16");break;
			case GT_TEXT32:
				sprintf(mode,"text32");break;
			case GT_1BIT:
				sprintf(mode,"1");break;
			case GT_4BIT:
				sprintf(mode,"4");break;
			case GT_8BIT:
				sprintf(mode,"8");break;
			case GT_15BIT:
				sprintf(mode,"15");break;
			case GT_16BIT:
				sprintf(mode,"16");break;
			case GT_24BIT:
				sprintf(mode,"24");break;
			case GT_32BIT:
				sprintf(mode,"32");break;
			default: 
				sprintf(mode,"unknown");break;
		}

	/* text output: */
		ggiSetGCForeground(vis,white);
		ggiSetGCBackground(vis,black);
		ggiPuts(vis,MIDX-(15*FONTX)/2,MIDY-3*FONTY,"GGI screen test");

		/* -visible resolution */
		sprintf(s,"vis%4dx%3dx%s",currmode.visible.x,currmode.visible.y,mode);
		ggiPuts(vis,MIDX-(14*FONTX)/2,MIDY-2*FONTY,s);
	
		/* - frequencies */
		sprintf(s,"%2.1f kHz %2.1f Hz",horfreq*0.001,vertfreq);
		ggiPuts(vis,MIDX-(16*FONTX)/2,MIDY-1*FONTY,s);
	
		/* -virtual resolution */
		sprintf(s,"virt%4dx%3d f=%2d",currmode.virt.x,currmode.virt.y,currmode.frames);
		ggiPuts(vis,MIDX-(18*FONTX)/2,MIDY+0*FONTY,s);
	
		/* -size */
		sprintf(s,"%4d mm x%3d mm",currmode.size.x,currmode.size.y);
		ggiPuts(vis,MIDX-(15*FONTX)/2,MIDY+1*FONTY,s);

		/* -font size */
		if(currmode.graphtype==GT_TEXT16 || currmode.graphtype==GT_TEXT32)
			sprintf(s, "fonts %2d x%2d ",currmode.dpp.x,currmode.dpp.y);
		else
			sprintf(s, "fonts %2d x%2d ",FONTX,FONTY);
		ggiPuts(vis,MIDX-(15*FONTX)/2,MIDY+2*FONTY,s);

	}
	ggiFlush(vis);
}

void convergence(void)
{
	ggi_uint i,j,k,xmax,ymax;
	ggi_uint f[3];
	ggi_mode currmode;

	ggiGetMode(vis,&currmode);
	xmax = currmode.visible.x;
	ymax = currmode.visible.y;

	f[0]=red; 
	f[1]=green;
	f[2]=blue;

	for (k=0;k<4;k++){
		ggiSetGCForeground(vis,black);
		ggiDrawBox(vis,0,0,xmax,ymax);
		ggiSetGCForeground(vis,2);
		for (i=0;i<=PARTSHOR;i++)
			for (j=0;j<=PARTSVERT;j++){
				ggiSetGCForeground(vis,f[(i+j+k)%3]);
				ggiDrawHLine(vis,PARTX(i),PARTY(j),PARTWIDTH/2);
				ggiDrawVLine(vis,PARTX(i),PARTY(j),PARTHEIGHT/2);
				ggiDrawHLine(vis,PARTX(i)-PARTWIDTH/2,
					PARTY(j),PARTWIDTH/2);
				ggiDrawVLine(vis,PARTX(i),PARTY(j)-PARTHEIGHT/2,
					PARTHEIGHT/2);
		}
		ggiFlush(vis);
		waitabit();
	}
}

void resolution(void)
{
	ggi_uint xmax,ymax;
	ggi_mode currmode;

	ggiGetMode(vis,&currmode);
	xmax = currmode.visible.x;
	ymax = currmode.visible.y;
		ggiSetGCForeground(vis,blue);
		ggiDrawBox(vis,0,0,xmax,ymax);

	stripevert(0,0,xmax-1,ymax-1,black,white,1);
	ggiFlush(vis);
	waitabit();

	stripevert(0,0,xmax-1,ymax-1,black,white,2);
	ggiFlush(vis);
	waitabit();

	stripevert(0,0,xmax-1,ymax-1,black,white,3);
	ggiFlush(vis);
	waitabit();

	stripevert(0,0,xmax-1,ymax-1,black,white,4);
	ggiFlush(vis);
	waitabit();

	stripehor(0,0,xmax-1,ymax-1,black,white,1);
	ggiFlush(vis);
	waitabit();

	stripehor(0,0,xmax-1,ymax-1,black,white,2);
	ggiFlush(vis);
	waitabit();

	starone(0,0,xmax-1,ymax-1,black,white);
	ggiFlush(vis);
	waitabit();

	starfive(0,0,xmax-1,ymax-1,white,black);
	ggiFlush(vis);
	waitabit();

	startwoten(0,0,xmax-1,ymax-1,white,black);
	ggiFlush(vis);
	waitabit();

	ggiSetGCForeground(vis,black);
	ggiDrawBox(vis,0,0,xmax,ymax);

	stripevert( 1*xmax/16,2*ymax/8, 3*xmax/16-1,3*ymax/8-1,black,red,4);
	stripevert( 3*xmax/16,2*ymax/8, 5*xmax/16-1,3*ymax/8-1,black,red,3);
	stripevert( 5*xmax/16,2*ymax/8, 7*xmax/16-1,3*ymax/8-1,black,red,2);
	stripevert( 7*xmax/16,2*ymax/8, 9*xmax/16-1,3*ymax/8-1,black,red,1);
	stripevert( 9*xmax/16,2*ymax/8,11*xmax/16-1,3*ymax/8-1,black,red,2);
	stripevert(11*xmax/16,2*ymax/8,13*xmax/16-1,3*ymax/8-1,black,red,3);
	stripevert(13*xmax/16,2*ymax/8,15*xmax/16-1,3*ymax/8-1,black,red,4);
	stripevert( 1*xmax/16,3*ymax/8, 3*xmax/16-1,4*ymax/8-1,black,green,4);
	stripevert( 3*xmax/16,3*ymax/8, 5*xmax/16-1,4*ymax/8-1,black,green,3);
	stripevert( 5*xmax/16,3*ymax/8, 7*xmax/16-1,4*ymax/8-1,black,green,2);
	stripevert( 7*xmax/16,3*ymax/8, 9*xmax/16-1,4*ymax/8-1,black,green,1);
	stripevert( 9*xmax/16,3*ymax/8,11*xmax/16-1,4*ymax/8-1,black,green,2);
	stripevert(11*xmax/16,3*ymax/8,13*xmax/16-1,4*ymax/8-1,black,green,3);
	stripevert(13*xmax/16,3*ymax/8,15*xmax/16-1,4*ymax/8-1,black,green,4);
	stripevert( 1*xmax/16,4*ymax/8, 3*xmax/16-1,5*ymax/8-1,black,blue,4);
	stripevert( 3*xmax/16,4*ymax/8, 5*xmax/16-1,5*ymax/8-1,black,blue,3);
	stripevert( 5*xmax/16,4*ymax/8, 7*xmax/16-1,5*ymax/8-1,black,blue,2);
	stripevert( 7*xmax/16,4*ymax/8, 9*xmax/16-1,5*ymax/8-1,black,blue,1);
	stripevert( 9*xmax/16,4*ymax/8,11*xmax/16-1,5*ymax/8-1,black,blue,2);
	stripevert(11*xmax/16,4*ymax/8,13*xmax/16-1,5*ymax/8-1,black,blue,3);
	stripevert(13*xmax/16,4*ymax/8,15*xmax/16-1,5*ymax/8-1,black,blue,4);
	stripevert( 1*xmax/16,5*ymax/8, 3*xmax/16-1,6*ymax/8-1,black,white,4);
	stripevert( 3*xmax/16,5*ymax/8, 5*xmax/16-1,6*ymax/8-1,black,white,3);
	stripevert( 5*xmax/16,5*ymax/8, 7*xmax/16-1,6*ymax/8-1,black,white,2);
	stripevert( 7*xmax/16,5*ymax/8, 9*xmax/16-1,6*ymax/8-1,black,white,1);
	stripevert( 9*xmax/16,5*ymax/8,11*xmax/16-1,6*ymax/8-1,black,white,2);
	stripevert(11*xmax/16,5*ymax/8,13*xmax/16-1,6*ymax/8-1,black,white,3);
	stripevert(13*xmax/16,5*ymax/8,15*xmax/16-1,6*ymax/8-1,black,white,4);
	ggiFlush(vis);
}

void moiree(void)
{
	ggi_uint xmax,ymax;
	ggi_mode currmode;

	ggiGetMode(vis,&currmode);
	xmax = currmode.visible.x;
	ymax = currmode.visible.y;
	stripevert(0,0,xmax-1,ymax-1,black,white,1); ggiFlush(vis); waitabit();
	stripevert(0,0,xmax-1,ymax-1,red,white,1); ggiFlush(vis); waitabit();
	stripevert(0,0,xmax-1,ymax-1,green,white,1); ggiFlush(vis); waitabit();
	stripevert(0,0,xmax-1,ymax-1,blue,white,1); ggiFlush(vis); waitabit();

	dotone(0,0,xmax-1,ymax-1,black,white); ggiFlush(vis); waitabit();
	dotone(0,0,xmax-1,ymax-1,red,white); ggiFlush(vis); waitabit();
	dotone(0,0,xmax-1,ymax-1,green,white); ggiFlush(vis); waitabit();
	dotone(0,0,xmax-1,ymax-1,blue,white); ggiFlush(vis); waitabit();

	chessboardone(0,0,xmax-1,ymax-1,black,white); ggiFlush(vis); waitabit();
	chessboardone(0,0,xmax-1,ymax-1,red,white); ggiFlush(vis); waitabit();
	chessboardone(0,0,xmax-1,ymax-1,green,white); ggiFlush(vis); waitabit();
	chessboardone(0,0,xmax-1,ymax-1,blue,white); ggiFlush(vis); waitabit();
}


int main(int argc,char **argv)
{
	const char *prog=argv[0];
	ggi_graphtype type;
	int depth;
	int sx,sy,vx,vy;
	int err;
	int textmode=0;

	if (argc==1)
		depth=-1;
	else {
		if ((argv[1][0] | 0x20)=='t') {
			textmode=1;
			depth=atoi(argv[1]+1);
		} else
			depth=atoi(argv[1]);
		argc--;
		argv++;
	}

	if (argc==1) {
		sx=GGI_AUTO;
		sy=GGI_AUTO;
		vx=sx;
		vy=sy;
	} else if (argc==3) {
		sx=atoi(argv[1]);
		sy=atoi(argv[2]);
		vx=sx;
		vy=sy;
	} else if (argc==5) {
		sx=atoi(argv[1]);
		sy=atoi(argv[2]);
		vx=atoi(argv[3]);
		vy=atoi(argv[4]);
	} else {
		usage(prog);
		return 1;
	}

	printf("Using mode (%dx%d [%dx%d] %c%dbpp)\n",
		sx,sy,vx,vy,textmode ? 'T' : ' ',depth);

	if (textmode==1) {
		if (depth==16) {
			type=GT_TEXT16;
		} else if (depth==32) {
			type=GT_TEXT32;
		} else {
			 fprintf(stderr,"%s: Invalid textmode depth!\n\n",prog);
			 usage(prog);
			 return 1;
		}
	} else {
	   switch (depth) {
		case -1:
			type=GT_AUTO;
			break;
		case 1:
			type=GT_1BIT;
			break;
		case 4: type=GT_4BIT;
			break;
		case 8:	type=GT_8BIT;
			break;
		case 15: type=GT_15BIT;
			 break;
		case 16: type=GT_16BIT;
			 break;
		case 24: type=GT_24BIT;
			 break;
		case 32: type=GT_32BIT;
			 break;
		default:
			 fprintf(stderr,"%s: Invalid depth!\n\n",prog);
			 usage(prog);
			 return 1;
	   }
	}

	if (ggiInit() != 0) {
		fprintf(stderr, "%s: unable to initialize libggi, exiting.\n",
			argv[0]);
		exit(1);
	}
	if ((vis=ggiOpen(NULL)) == NULL) {
		fprintf(stderr,
			"%s: unable to open default visual, exiting.\n",
			argv[0]);
		exit(1);
	}
	ggiSetFlags(vis, GGIFLAG_ASYNC);

	if (textmode) 
		err=ggiSetTextMode(vis,sx,sy,vx,vy,GGI_AUTO,GGI_AUTO,type);
	else
		err=ggiSetGraphMode(vis,sx,sy,vx,vy,type);

	if (err) {
		fprintf(stderr,"Can't set mode\n");
		return 2;
	}
	setcolors();

	testpattern();
	waitabit();

	convergence();
	resolution();
	waitabit();

	moiree();
	waitabit();

	ggiClose(vis);

	ggiExit();	

	return 0;
}
