/* $Id: conf.c,v 1.1 1998/09/29 23:19:19 taylorj Exp $
***************************************************************************
  
   Configuration handling
  
   Copyright (C) 1997 Jason McMullan   [jmcc@ggi-project.org]
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
  
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
  
   MODULE SYNOPSIS conf.c :
   
   This module deal with parsing /etc/ggi/libggi.conf .
   It   does   not   export  user      functions.
   It   does   not   export  extension functions.
   It exports the following  internal  functions:
   * int         _ggiLoadConfig(const char *file)
   * void        _ggiFreeConfig(void)
   * const char *_ggiMatchConfig(const char *name,const char *ver)
   It exports the following global variables:
   * ggi_dlconfig *_ggiMasterConfig

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include <ggi/internal/internal.h>

/* Private structures */
typedef struct ggi_string_list {
	const char *str;
	struct ggi_string_list *next;
} ggi_string_list;

typedef struct ggi_dlconfig {
	const char *name;
	const char *filename;
	ggi_string_list *version;
	struct ggi_dlconfig *next;
} ggi_dlconfig;

/* Static variables */
static ggi_dlconfig   *_ggiMasterConfig=NULL;

int _ggiLoadConfig(const char *file)
{
	FILE *inf;
	char root[2048],line[2048],buff[2048],*cp,*bp;
	char *name,*fname;
	ggi_dlconfig *dltmp,*dlconf;
	static int incl=0;
	root[0]=0;

	inf=fopen(file,"r");
	if (inf==NULL)
		return -1;

	dlconf=_ggiMasterConfig;

	while (!feof(inf)) {
		fgets(line,2048,inf);
		cp=line;

		while (isspace((int)*cp)) cp++;
		if (*cp==0 || *cp=='#')
			continue;
	
		if (strncmp(cp,".root:",6)==0) {
			cp+=6;
			while (isspace((int)*cp)) cp++;
			name=cp;
			while (*cp && !isspace((int)*cp)) cp++;
			strncpy(root,name,cp-name);
			root[cp-name]=0;
			continue;
		}
		if (strncmp(cp,".include",8)==0) {
			cp+=8;
			while (isspace((int)*cp)) cp++;
			name=cp;
			while (*cp && !isspace((int)*cp)) cp++;
			strncpy(buff,name,cp-name);
			buff[cp-name]=0;
			incl++;
			if (incl<20) {
				_ggiMasterConfig=dlconf;
				_ggiLoadConfig(buff);
				dlconf=_ggiMasterConfig;
			}
			else {
				fprintf(stderr,"libggi: .include recursion "
						"too deep\n");
			}
			incl--;

		}	
		bp=buff;
		while (!isspace((int)*cp)) *(bp++)=*(cp++);
		*(bp)=0;
		name=strdup(buff);
			
			
		while (isspace((int)*cp)) cp++;
		if (*cp==0 || *cp=='#') {
			free(name);
			continue;
		}
		bp=buff;
		while (!isspace((int)*cp)) *(bp++)=*(cp++);
		*(bp)=0;
		if (buff[0]!=CHAR_DIRDELIM) {
			fname=(char *)_ggi_malloc(strlen(buff)+strlen(root)+2);
			strcpy(fname,root);
			strcat(fname,STRING_DIRDELIM);
			strcat(fname,buff);
		} else
			fname=strdup(buff);

		/* Add the config entry....
		 */	
		dltmp=(ggi_dlconfig *)_ggi_malloc(sizeof(ggi_dlconfig));
		dltmp->name=name;
		dltmp->filename=fname;
		dltmp->version=NULL;
		dltmp->next=dlconf;
		dlconf=dltmp;

		do {	
			ggi_string_list *vtmp;

			while (isspace((int)*cp)) cp++;
			if (*cp==0 || *cp=='#') 
				break;
			bp=buff;
			while (!isspace((int)*cp)) *(bp++)=*(cp++);
			*(bp)=0;
			vtmp=(ggi_string_list *)_ggi_malloc(sizeof(ggi_string_list));
			vtmp->str=strdup(buff);
			vtmp->next=dltmp->version;
			dltmp->version=vtmp;
		} while (*cp!=0);
			
	}
	fclose(inf);

	_ggiMasterConfig=dlconf;

	return 0;
}


/*
 * _ggiFreeConfig()
 *
 * Free the entire _ggiMasterConfig list
 */

void _ggiFreeConfig(void)
{
	ggi_dlconfig *confptr = _ggiMasterConfig, *conftmp;
	
	while (confptr != NULL) {
		ggi_string_list *strlist = confptr->version, *strtmp;
		free((char*)confptr->name);
		free((char*)confptr->filename);
		while (strlist != NULL) {
			strtmp = strlist->next;
			free((char*)strlist->str);
			free(strlist);
			strlist = strtmp;
		}
		conftmp = confptr->next;
		free(confptr);
		confptr = conftmp;
	}
	_ggiMasterConfig = NULL;
}


/* Takes a ggi_dlconfig, lib name, and a version. Returns the
 * closest match (Ha!)
 */
const char *_ggiMatchConfig(const char *name,const char *ver)
{
	ggi_dlconfig *dlc,*dtmp,*best=NULL;
	ggi_string_list *vtmp;

	dlc=_ggiMasterConfig;

	for (dtmp=dlc;dtmp!=NULL;dtmp=dtmp->next) {
		if (strcasecmp(name,dtmp->name)!=0)
			continue;

		if (dtmp->version==NULL) {
			best=dtmp;
			continue;
		}

		for (vtmp=dtmp->version;vtmp!=NULL;vtmp=vtmp->next) 
			if (strcmp(ver,vtmp->str)==0) {
				return dtmp->filename;
			}
	}

	return (best==NULL) ? NULL : best->filename;
}
