/*
 * Performance test program for Vector Font Library VFlib 2
 *
 *  This program measures the drawing time of outline data.
 * (not including reading time of outline from disk.)
 *
 *
 *  Programmed by Hirotsugu KAKUGAWA, Hiroshima University
 *  E-Mail:  kakugawa@se.hiroshima-u.ac.jp
 *
 * Edition History
 *  22 Jan 1994  
 *  11 Mar 1994  version 2  New performance test
 *
 */


/* This file is part of VFlib
 *
 * Copyright (C) 1993,1994  Hirotsugu KAKUGAWA.   All rights reserved.
 *
 * VFlib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
 * to anyone for the consequences of using it or for whether it serves any
 * particular purpose or works at all, unless he says so in writing.  Refer
 * to the GNU General Public License for full details.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * VFlib, but only under the conditions described in the GNU
 * General Public License.  A copy of this license is supposed to have been
 * given to you along with VFlib so you can know your rights and
 * responsibilities.  It should be in a file named COPYING.  Among other
 * things, the copyright notice and this notice must be preserved on all
 * copies. 
 */



/*** 
 ***   CAUTION!!    THIS FILE CONTAINS 8-BIT EUC KANJI CHARACTER CODES
 ***/

/** KANJI CODE (EUC/SJIS) **/
#define   EUC
#undef    SJIS


#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/times.h>
#include "../src/VF.h"
#include "../src/defs.h"
#include "str.h"


/* Default Pixel Size */
#define W_XDOTS     100
#define W_YDOTS     100

int ToJIS(char *s);



/** THIS IS EUC KANJI TEXT **/
/*
  ͵  (ع), 
  "ʬƥǤλ񸻳ƥꥴꥺ", 
  1994 ǯ ߤ LA ݥͽƤȴ.
*/

static char *TestString = 
"͵(ع)\
ʬƥǤλ񸻳ƥꥴꥺ\
ǯߤΣ̣ݥͽƤȴ\
\
\
ܹƤǤϡʬƥǤζͭ񸻤γͻ롥ץ\
˻Ѳǽʶͭ񸻤ۤʤ롤ȤǥФ񸻳\
ˡɽꥳ꡼Ȥ¤򿷤Ƥ롥ޤɽꥳ\
꡼Ѥ񸻳ƤԤʤʬ르ꥺƤ\
򼨤\
\
ˣ\
塡\
졡\
졡\
䡡\
Ϥ\
ʣΥץŪʬ餬ߤ̿򤷤ʤ򴹤\
ʤʬƥ˱񸻤ζͭͤ롥ͭ񸻤¾Ū\
ʻѡ¨İʾΥץϣĤλ񸻤Ʊ˻Ѥ\
Ȳꤹ롥񸻤ʬƥǶͭϽ褫褯\
ƤƤ롥㤨Сʬǡ١Ǥϡǡܤ򣱤Ĥζ\
ͭ񸻤ȸ魯Ȥ롥ơǡιԤʤ٤˻񸻤\
ݤС¾Υץ񸻤ݤƤʤݾڤΤǡ\
ǡΰ򼺤ȤʤȤԤʤ롥\
\
θǤϡĤλ񸻤ʬƥǶͭȤ񸻶\
ͭΥǥ뤬갷졤񸻤ؤΥԤʤ׳ΰˤ˥\
οƻˤƹ⡹Ĥ¤ʬӽȤƸ\
ƤΣϡΣϡΣ᣸\
ϡΣ飸ϡΣ᣸ϡΣ죸\
Σ䣸ϡ̣Σϡ\
ңΣϤˤäƤ줿ʬӽ르\
ꥺϡӽԤʤ٤ƤΥץ˥åͤФ\
餺åѸξ깥ޤʤǣݣͣ\
Σ᣸Ϥϥ꡼ʣˡȤ\
¤Ƥå٤ץν礬ɬƤΥץ\
ǤϤʤΥץΤߤǤӽݾڤ뤳ȤǤ\
򼨤Σ᣸ϤϡӽԤʤ\
åϡʣˡǺѤॳ꡼ιˡʬ\
ӽ르ꥺ򼨤ǡʬƥ˴ޤޤ\
οǤ롥";
/** END OF KANJI STRING **/




int
main(int argc, char **argv)
{
  int    Repeat, Fd, Ccode, i;
  int    XDots,  YDots, XBytes, DispC, UseFSlib;
  char   *Ent, *Vfcap, *Comment;
  char   *Buff;
  long   *vfdata, *VFTable[2000];
  unsigned char *p;
  struct tms  t_start, t_end;

  Vfcap    = NULL;
  Ent      = "min";
  XDots    = W_XDOTS;
  YDots    = W_YDOTS;
  DispC    = 0;
  UseFSlib = 0;
  Comment  = "";

  for (i = 1; i < argc; i++){
    if (argv[i][0] == '-')
      switch (argv[i][1]){
      case 'c':
	Comment = argv[++i];
	break;
      case 'd':
	DispC = 1;
	break;
      case 'f':
	Ent = argv[++i];
	break;
      case 'F':
	UseFSlib = 1;
	break;
      case 'x':
	XDots = atoi(argv[++i]); 
	break;
      case 'y':
	YDots = atoi(argv[++i]); 
	break;
      case 'v':
	Vfcap = argv[++i];
	break;
      case 'h':
      default:
	printf("vfperf  - performance test program for VFlib 2\n");
	printf("Usage vfperf [options]\n");
	printf("Options:\n");
	printf("-c COMMENT    : comment.\n");	
	printf("-d            : display char codes.\n");	
	printf("-F            : use FS_scan_conv() of FSlib to draw.\n");
	printf("-f FONT_ENTRY : specify font entry name. (`min' is default)\n");
	printf("-x DOTS       : set character width.\n");
	printf("-y DOTS       : set character height.\n");
	printf("-v            : set vfontcap file. (must give full path)\n");	
	exit(0);
      }
  }

  XBytes = (XDots+7)/8;
  if ((Buff = malloc(XBytes*YDots)) == NULL){
    fprintf(stderr, "malloc err\n");
    exit(0);
  } 


  VF_Init(Vfcap);
  if ((Fd = VF_OpenFont(Ent)) < 0){
    fprintf(stderr, "open error; %s\n", Ent);
    exit(-1);
  }

  /*** Read Outline Data ***/
  i = 0;
  p = (unsigned char*) TestString;
  while (*p != '\0'){
    if (*p < 0x80){
      p++;
      continue;     /* ignore 1 byte char */
    }

    Ccode  = ToJIS(p);
    if ((vfdata = VF_GetOutline(Ccode, Fd)) == NULL){
      printf("Failed to get outline : %04x\n", Ccode);
      exit(-1);
    } else {
      VFTable[i++] = vfdata;
    }
    p = p + 2;
  }
  VFTable[i++] = NULL;


  /*** START MEASURING ***/
  times(&t_start);
  
  Repeat = 10;
  while (Repeat > 0){
    for (i = 0; VFTable[i] != NULL; i++){
      bzero(Buff, XBytes*YDots);
#ifndef USE_SONY_FONT
      VF_DrawOutline(vfdata, Fd, XDots, YDots, XBytes, 0, Buff);
#else
      if (UseFSlib == 0){
	VF_DrawOutline(vfdata, Fd, XDots, YDots, XBytes, 0, Buff);
      } else {
	FS_scan_conv(&vfdata[2], XDots, YDots, XBytes, 0, Buff, 0, 0, 0);
      }
#endif
    }
    --Repeat;
  }

  /*** END MEASURING ***/
  times(&t_end);


  /*** Release Outline Data ***/
  for (i = 0; VFTable[i] != NULL; i++){
    VF_FreeOutline(VFTable[i], Fd);
  }

  VF_CloseFont(Fd);
  VF_Deinit();


  /*** Show performance ***/
  printf("FONT ENTRY: %s\n", Ent);
  printf("Comment:  %s  \n", Comment);
  printf("Size:     X=%d  Y=%d\n", XDots, YDots);
  printf("USR TIME: %.3f SEC\n", 
	 (float) (t_end.tms_utime - t_start.tms_utime) / 60.0);
  printf("SYS TIME: %.3f SEC\n\n", 
	 (float) (t_end.tms_stime - t_start.tms_stime) / 60.0);
  
  return 0;
}




#define	etoj(c1, c2)	{(c1) &= 0x7f; (c2) &= 0x7f;}
#define	iseuc1st(c)	((c) >= 0xa1 && (c) <= 0xfe)

#define	stoe(c1, c2)	\
{\
	if (c1 >=0xe0)\
		c1 -=  0x40;\
	if (c2 >= 0x9f) {\
		c1 = (c1 - 0x88) * 2 + 0xb0;\
		c2 += 0x02;\
	} else {\
		if (c2 >= 0x7f)\
			c2 -= 0x01;\
		c1 = (c1 - 0x89) * 2 + 0xb1;\
		c2 = c2 + 0x61;\
	}\
}
#define	issjis1st(c)	(((c) >= 0x81 && (c) <= 0x9f) || \
			 ((c) >= 0xe0 && (c) <= 0xfc))

int ToJIS(char *s)
{
  int j1, j2;

  j1 = s[0];
  j2 = s[1];
#ifdef EUC
  etoj(j1, j2);
#endif
#ifdef SJIS
  stoe(j1, j2);
  etoj(j1, j2);
#endif
  return (j1*0x100 + j2);
}


