/*
 * Copyright (c) 1997, 1998 Akira Yoshiyama <yosshy@debian.or.jp>
 * Copyright (c) 1998 Fumitoshi UKAI <ukai@debian.or.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public Licenses as
 * published by the Free Software Foundation; either version 2 of 
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * mbsnrtowcs_locale ()
 * for conversion from multibyte character to UCS-4
 * programed
 *   by A. Yoshiyama (yosshy@jedi.seg.kobe-u.ac.jp)
 *   on Dec.17,1996
 */

#include <stdio.h>
#include <wchar.h>
#include <errno.h>
#include <wcsmbs_locale.h>
#include "mbs2wcs.h"

#if DEBUG
#include <stdio.h>
#include <stdlib.h>
#endif

#ifndef EILSEQ
#define EILSEQ EINVAL
#endif

extern node_t table_mbs2wcs[];

size_t
dll_mbsnrtowcs_locale(dst, src, srclim, dstlim, ps)
     wchar_t *dst;
     const char **src;
     size_t srclim;
     size_t dstlim;
     mbstate_t *ps;
{
  static mbstate_t internal = {0, (wint_t)0};
  size_t written = 0;
  const char *run;
  node_t *cur, *default_node;

#if DEBUG
  fprintf(stderr,"%s:dll_mbsnrtowcs_locale(dst, src, srclim, dstlim, ps)\n",
	  getenv(MONENV));
  fprintf(stderr,"dst=%p, src=%p, srclim=%d, dstlim=%d, ps=%p)\n",
	  dst, src, srclim, dstlim, ps);
  if (*src != NULL)
    fprintf(stderr,"*src=%p:%s\n", *src, *src);
#endif

  /* Assign PS to INTERNAL if PS is NULL */
  if (ps == NULL)
    ps = &internal;

#if DEBUG
  fprintf(stderr, "ps = (%d, %d)\n", ps->count, ps->value);
#endif
  /* Initialize ps if SRC or *SRC is NULL. */
  if (src == NULL || *src == NULL)
    {
      ps->count = ps->value = 0;
#if DEBUG
      fprintf(stderr,"dll_mbsnrtowcs_locale:result = %d\n", 0);
#endif
      return (size_t)0;
    }
  else
    run = *src;

  if (ps->count < 0 || ps->count >= table_mbs2wcs_maxentry)
    {
      /* The PS->COUNT is out of range. */
#if DEBUG
      fprintf(stderr,"dll_mbsnrtowcs_locale:result = %d (%d <=> %d)\n",
	      -EINVAL, ps->count, table_mbs2wcs_maxentry);
#endif
      __set_errno (EINVAL);
      return (size_t)-1;
    }
  else 
    /* Set the current node to PS->SHIFT. */
    cur = default_node = table_mbs2wcs + ps->count;

  /* if dst == NULL, dstlim is ignored */
  while ((written < dstlim || dst == NULL) && srclim > 0)
    {
      /* Check whether 'run' reachs last of mbs */
      if (*run == '\0')
	{
	  if (dst != NULL)
	    *dst = L'\0';
	  if (cur != default_node)
	    {
	      /* XXX: intermediate multibyte character */
	      __set_errno (EILSEQ);
	      return (size_t)-2;
	    }
	  src = NULL;
#if DEBUG
	  fprintf(stderr,"dll_mbsnrtowcs_locale:result = %d (EOS)\n", 
		  written);
#endif
	  return written;
	}

      while (1)
	{
	  /* Check whether the current node is a default_node return point */
	  if (cur->flag & DEFAULT_RETURN_NODE)
	    default_node = cur;

	  /* Check whether the current node is the wanted node */
	  if (cur->key == *run)
	    {
	      run++;
	      srclim--;
	      if (srclim < 0)
	        {
		  __set_errno (EILSEQ);
		  return (size_t)-2;
		}
	      /* Check whether the current node is a link node */
	      if (cur->flag & LINK_NODE)
		{
		  /* Link node */
		  cur = &table_mbs2wcs[(int)(cur->value)];
		}
	      else
		{
		  /* Code node */
		  if (dst != NULL)
		    *dst++ = (wchar_t)(cur->value);
		  written++;
		  *src = run;
#if DEBUG
		  fputc(*run, stderr);
#endif
		  cur = default_node;
		}
	      break;
	    }

	  /* Check whether the current node is last of this line */
	  if (cur->flag & END_NODE)
	    {
	      __set_errno(EILSEQ);
#if DEBUG
	      fprintf(stderr,"dll_mbsnrtowcs_locale:result = %d (%x & %x)\n", 
		      -1, cur->flag, END_NODE);
#endif
	      return (size_t)-1;
	    }

	  /* The current node isn't the wanted node */
	  /* Seek to the next node */
	  cur++;
	}
    }
  ps->count = (int)(default_node - table_mbs2wcs);

#if DEBUG
  fprintf(stderr,"dll_mbsnrtowcs_locale:result = %d (written)\n", written);
#endif
  return written;
}
