/*
 * =========================================================================
 * llsearch - A program that extracts a subset of the data in a GNIS file.
 * Copyright (c) 1997  Fred M. Erickson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * =========================================================================
 *
 *
 * This is a filter to search a GNIS file for any entries within
 * the specified latitude/longitude box.
 *
 * Each GNIS file is for a separate state, but some of the larger
 * states span many degrees of latitude and longitude.
 * It is convenient to be able to pull out just the data you need
 * for a given map.
 *
 * The program reads from stdin and writes to stdout.
 */
#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>

ssize_t buf_read(int, void *, size_t);
ssize_t buf_write(int, const void *, size_t);
ssize_t get_a_line(int, void *, size_t);
double lat_conv(unsigned char *);
double lon_conv(unsigned char *);



main(int argc, char *argv[])
{
	unsigned char buf[16384];
	double latitude_low;
	double latitude_high;
	double longitude_low;
	double longitude_high;
	double latitude;
	double longitude;
	ssize_t ret_val;

	if (argc != 5)  {
		fprintf(stderr, "Usage:  %s latitude_low longitude_low latitude_high longitude_high\n", argv[0]);
		fprintf(stderr, "        (The latitude/longitude values are in decimal degrees.)\n", argv[0]);
		exit(0);
	}
	latitude_low = strtod(argv[1], (char **)0);	/* We assume that these are in decimal degrees */
	longitude_low = strtod(argv[2], (char **)0);	/* We assume that these are in decimal degrees */
	latitude_high = strtod(argv[3], (char **)0);	/* We assume that these are in decimal degrees */
	longitude_high = strtod(argv[4], (char **)0);	/* We assume that these are in decimal degrees */

	if ((latitude_high < -90.0) || (latitude_high > 90.0) ||
	    (latitude_low < -90.0) || (latitude_low > 90.0) ||
	    (longitude_high < -180.0) || (longitude_high > 180.0) ||
	    (longitude_low < -180.0) || (longitude_low > 180.0))  {
		fprintf(stderr, "Error:  Parameters appear incorrect\n");
		exit(0);
	}
	if ((latitude_high < latitude_low) || (longitude_high < longitude_low))  {
		fprintf(stderr, "Error:  Parameters appear incorrect\n");
		exit(0);
	}

	while ((ret_val = get_a_line(0, buf, 16384)) > 0)  {
		/* These are in degree/min/sec format */
		latitude = lat_conv(&buf[149]);
		longitude = lon_conv(&buf[156]);

		if ((latitude >= latitude_low) &&
		    (latitude <= latitude_high) &&
		    (longitude >= longitude_low) &&
		    (longitude <= longitude_high))  {
			write(1, buf, ret_val);
		}
	}
}
