/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include "common.h"
#include "error.h"
#include "filename.h"
#include "getline.h"
#include "settings.h"
#include "misc.h"
#include "var.h"
#include "version.h"
#include "str.h"
#include "lexer.h"
#include "lexerP.h"

/* Global variables. */
getl_script *getl_head;
getl_script *getl_tail;
int getl_interactive;
int getl_welcomed;
int getl_mode;
char *getl_buf;
int getl_buf_len;
size_t getl_buf_size;
int getl_prompt;
char *getl_history;
char *getl_include_path;

/* Number of levels of DO REPEAT structures we're nested inside.  If
   this is greater than zero then DO REPEAT macro substitutions are
   performed. */
static int DO_REPEAT_level;

/* Initialize getline. */
void
getl_initialize (void)
{
  getl_include_path = xstrdup (getenv_default ("STAT_INCLUDE_PATH",
					       include_path));
}

/* Returns a string that represents the directory that the syntax file
   currently being read resides in.  If there is no syntax file then
   returns the OS current working directory.  Return value must be
   free()'d. */
char *
getl_get_current_directory (void)
{
  if (getl_head)
    return blp_dirname (getl_head->fn);
  else
    return gnu_getcwd ();
}

/* Delete everything from the include path. */
void
getl_clear_include_path (void)
{
  free (getl_include_path);
  getl_include_path = NULL;
}

/* Add to the include path. */
void
getl_add_include_dir (const char *path)
{
  if (getl_include_path)
    {
      int len = strlen (getl_include_path);

      getl_include_path = xrealloc (getl_include_path,
				    len + 1 + strlen (path) + 1);
      getl_include_path[len] = PATH_DELIMITER;
      strcpy (&getl_include_path[len + 1], path);
    }
  else
    getl_include_path = xstrdup (path);
}

/* Adds FN to the tail end of the list of script files to execute.
   OPTIONS is the value to stick in the options field of the
   getl_script struct.  If WHERE is zero then the file is added after
   all other files; otherwise it is added before all other files (this
   can be done only if parsing has not yet begun). */
void
getl_add_file (const char *fn, int separate, int where)
{
  getl_script *n = xmalloc (sizeof (getl_script));

  assert (fn);
  n->next = NULL;
  if (getl_tail == NULL)
    getl_head = getl_tail = n;
  else if (!where)
    getl_tail = getl_tail->next = n;
  else
    {
      assert (getl_head->f == NULL);
      n->next = getl_head;
      getl_head = n;
    }
  n->included_from = n->includes = NULL;
  n->fn = xstrdup (fn);
  n->ln = 0;
  n->f = NULL;
  n->separate = separate;
  n->first_line = NULL;
}

/* Inserts the given file with filename FN into the current file after
   the current line. */
void
getl_include (const char *fn)
{
  getl_script *n;
  char *real_fn;

  assert (getl_include_path);
  {
    char *cur_dir = getl_get_current_directory ();
    real_fn = search_path (fn, getl_include_path, cur_dir);
    free (cur_dir);
  }
  if (!real_fn)
    {
      msg (SE, _("Can't find `%s' in include file search path."), fn);
      return;
    }

  if (!getl_head)
    {
      getl_add_file (real_fn, 0, 0);
      free (real_fn);
    }
  else
    {
      n = xmalloc (sizeof (getl_script));
      n->included_from = getl_head;
      getl_head = getl_head->includes = n;
      n->includes = NULL;
      n->next = NULL;
      n->fn = real_fn;
      n->ln = 0;
      n->f = NULL;
      n->separate = 0;
    }
}

/* Add the virtual file FILE to the list of files to be processed.
   The first_line field in FILE must already have been initialized. */
void 
getl_add_virtual_file (getl_script *file)
{
  if (getl_tail == NULL)
    getl_head = getl_tail = file;
  else
    getl_tail = getl_tail->next = file;
  file->included_from = file->includes = NULL;
  file->next = NULL;
  file->fn = file->first_line->line;
  file->ln = -file->first_line->len - 1;
  file->separate = 0;
  file->f = NULL;
  file->cur_line = NULL;
  file->remaining_loops = 2;
  file->loop_index = -1;
  file->macros = NULL;
}

/* Causes the DO REPEAT virtual file passed in FILE to be included in
   the current file.  The first_line, cur_line, remaining_loops,
   loop_index, and macros fields in FILE must already have been
   initialized. */
void
getl_add_DO_REPEAT_file (getl_script *file)
{
  /* getl_head == NULL can't happen. */
  assert (getl_head);

  DO_REPEAT_level++;
  file->included_from = getl_head;
  getl_head = getl_head->includes = file;
  file->includes = NULL;
  file->next = NULL;
  assert (file->first_line->len < 0);
  file->fn = file->first_line->line;
  file->ln = -file->first_line->len - 1;
  file->separate = 0;
  file->f = NULL;
}

/* Display a welcoming message. */
void
welcome (void)
{
  getl_welcomed = 1;
  puts (_("PSPP is free software and you are welcome to distribute "
	"copies of it"));
  puts (_("under certain conditions; type \"show copying.\" to see "
	"the conditions."));
  puts (_("There is ABSOLUTELY NO WARRANTY for PSPP; type \"show "
	"warranty.\" for details."));
  puts (stat_version);
}

/* Reads a single line from the user's terminal. */
static int read_console (void);

/* From repeat.c. */
extern void perform_DO_REPEAT_substitutions (void);
  
/* Reads a single line from the line buffer associated with getl_head.
   Returns 1 if a line was successfully read or 0 if no more lines are
   available. */
int
handle_line_buffer (void)
{
  getl_script *s = getl_head;

  /* Check that we're not all done. */
  do
    {
      if (s->cur_line == NULL)
	{
	  s->remaining_loops--;
	  s->loop_index++;
	  if (s->remaining_loops == 0)
	    return 0;
	  s->cur_line = s->first_line;
	}

      if (s->cur_line->len < 0)
	{
	  s->ln = -s->cur_line->len - 1;
	  s->fn = s->cur_line->line;
	  s->cur_line = s->cur_line->next;
	  continue;
	}
    }
  while (s->cur_line == NULL);

  /* Check that the line buffer is big enough. */
  if ((size_t) s->cur_line->len + 1 > getl_buf_size)
    {
      getl_buf_size = s->cur_line->len * 2;
      getl_buf = xrealloc (getl_buf, getl_buf_size);
    }
  
  memcpy (getl_buf, s->cur_line->line, s->cur_line->len + 1);
  getl_buf_len = s->cur_line->len;

  /* Advance pointers. */
  s->cur_line = s->cur_line->next;
  s->ln++;

  return 1;
}

/* Reads a single line into getl_buf from the list of files.  Will not
   read from the eof of one file to the beginning of another unless
   the options field on the new file's getl_script is nonzero.  Return
   zero on eof. */
int
getl_read_line (void)
{
  getl_mode = GETL_MODE_BATCH;
  
  while (getl_head)
    {
      getl_script *s = getl_head;

      if (s->separate)
	return 0;

      if (s->first_line)
	{
	  if (!handle_line_buffer ())
	    {
	      getl_close_file ();
	      continue;
	    }
	  perform_DO_REPEAT_substitutions ();
	  return 1;
	}
      
      if (NULL == s->f)
	{
	  verbose_msg (1, _("%s: Opening as syntax file."), s->fn);
	  s->f = open_file (s->fn, "r");

	  if (s->f == NULL)
	    {
	      msg (ME, _("Opening `%s': %s."), s->fn, strerror (errno));
	      getl_close_file ();
	      continue;
	    }
	}

      getl_buf_len = getline (&getl_buf, &getl_buf_size, s->f);
      if (getl_buf_len == -1)
	{
	  if (ferror (s->f))
	    msg (ME, _("Reading `%s': %s."), s->fn, strerror (errno));
	  getl_close_file ();
	  continue;
	}

      /* Macro substitutions from DO REPEAT. */
      if (DO_REPEAT_level)
	perform_DO_REPEAT_substitutions ();

      getl_head->ln++;

      /* Allows shebang invocation: `#! /usr/local/bin/pspp'. */
      if (getl_buf[0] == '#' && getl_buf[1] == '!')
	continue;

      return 1;
    }

  if (getl_interactive == 0)
    return 0;

  getl_mode = GETL_MODE_INTERACTIVE;
  
  if (getl_welcomed == 0)
    welcome ();

  return read_console ();
}

/* PORTME: Adapt to your local system's idea of the terminal. */
#if HAVE_LIBREADLINE

#if HAVE_READLINE_READLINE_H
#include <readline/readline.h>
#else /* no readline/readline.h */
extern char *readline (char *);
#endif /* no readline/readline.h */

#if HAVE_LIBHISTORY
#if HAVE_READLINE_HISTORY_H
#include <readline/history.h>
#else /* no readline/history.h */
extern void add_history (char *);
extern void using_history (void);
extern int read_history (char *);
extern void stifle_history (int);
#endif /* no readline/history.h */
#endif /* -lhistory */

static int
read_console (void)
{
  char *line;
  char *prompt;

  error_count = warning_count = 0;
  error_already_flagged = 0;

#if HAVE_LIBHISTORY
  if (!history_file)
    {
#if unix
      history_file = tilde_expand (HISTORY_FILE);
#endif
      using_history ();
      read_history (history_file);
      stifle_history (MAX_HISTORY);
    }
#endif /* -lhistory */

  switch (getl_prompt)
    {
    case GETL_PRPT_STANDARD:
      prompt = set_prompt;
      break;
    case GETL_PRPT_CONTINUATION:
      prompt = set_cprompt;
      break;
    case GETL_PRPT_DATA:
      prompt = set_dprompt;
      break;
    default:
      assert (0);
    }
  line = readline (prompt);
  if (!line)
    return 0;

#if HAVE_LIBHISTORY
  if (*line)
    add_history (line);
#endif

  free (getl_buf);
  getl_buf = line;
  getl_buf_len = strlen (getl_buf);
  getl_buf_size = getl_buf_len + 1;

  return 1;
}
#else /* no -lreadline */
static int
read_console (void)
{
  error_count = warning_count = 0;
  error_already_flagged = 0;

  fputs (getl_prompt ? set_cprompt : set_prompt, stdout);
  getl_buf_len = getline (&getl_buf, &getl_buf_size, stdin);
  if (getl_buf_len != -1)
    return 1;
  if (ferror (stdin))
    msg (FE, "stdin: fgets(): %s.", strerror (errno));
  return 0;
}
#endif /* no -lreadline */

/* Closes the current file, whether it be a main file or included
   file, then moves getl_head to the next file in the chain. */
void
getl_close_file (void)
{
  getl_script *s = getl_head;

  if (!s)
    return;
  assert (getl_tail);

  if (s->first_line)
    {
      getl_line_list *cur, *next;

      s->fn = NULL; /* It will be freed below. */
      for (cur = s->first_line; cur; cur = next)
	{
	  next = cur->next;
	  free (cur->line);
	  free (cur);
	}

      DO_REPEAT_level--;
    }
  
  if (s->f && EOF == close_file (s->f, s->fn))
    msg (MW, _("Closing `%s': %s."), s->fn, strerror (errno));
  free (s->fn);

  if (s->included_from)
    {
      getl_head = s->included_from;
      getl_head->includes = NULL;
    }
  else
    {
      getl_head = s->next;
      if (NULL == getl_head)
	getl_tail = NULL;
    }
  
  free (s);
}

/* Closes all files. */
void
getl_close_all (void)
{
  while (getl_head)
    getl_close_file ();
}

/* Sets the options flag of the current script to 0, thus allowing it
   to be read in.  Returns nonzero if this action was taken, zero
   otherwise. */
int
getl_perform_delayed_reset (void)
{
  if (getl_head && getl_head->separate)
    {
      getl_head->separate = 0;
      discard_variables ();
      reset_eof ();
      return 1;
    }
  return 0;
}
