### $Id: lme.q,v 1.52 1998/07/02 21:25:19 bates Exp $
###
###            Fit a general linear mixed effects model
###
### Copyright 1997, 1998 Jose C. Pinheiro <jcp@research.bell-labs.com>,
###                      Douglas M. Bates <bates@stat.wisc.edu>
###
### This file is part of the nlme library for S and related languages.
### It is made available under the terms of the GNU General Public
### License, version 2, or at your option, any later version,
### incorporated herein by reference.
### 
### This program is distributed in the hope that it will be
### useful, but WITHOUT ANY WARRANTY; without even the implied
### warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
### PURPOSE.  See the GNU General Public License for more
### details.
### 
### You should have received a copy of the GNU General Public
### License along with this program; if not, write to the Free
### Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
### MA 02111-1307, USA

lme <- 
  ## fits general linear mixed effects model by maximum likelihood, or
  ## residual maximum likelihood using Newton-Raphson algorithm.
  function(fixed,
	   data = sys.parent(),
	   random,
	   correlation = NULL,
	   weights = NULL,
	   subset,
	   REML = FALSE, 
	   na.action = na.fail, 
	   control = list())
  UseMethod("lme")

lme.groupedData <- 
  function(fixed,
	   data = sys.parent(),
	   random,
	   correlation = NULL,
	   weights = NULL,
	   subset,
	   REML = FALSE, 
	   na.action = na.fail, 
	   control = list())
{
  args <- as.list(match.call())[-1]
  names(args)[1] <- "data"
  form <- getResponseFormula(fixed)
  form[[3]] <- getCovariateFormula(fixed)[[2]]
  do.call("lme", c(list(fixed = form), args))
}

lme.lmList <- 
  function(fixed,
	   data = sys.parent(),
	   random,
	   correlation = NULL,
	   weights = NULL,
	   subset,
	   REML = FALSE, 
	   na.action = na.fail, 
	   control = list())
{
  if (length(grpForm <- getGroupsFormula(fixed, asList = TRUE)) > 1) {
    stop("Can only fit lmList objects with single grouping variable")
  }
  this.call <- as.list(match.call())[-1]
  ## warn "data" is passed to this function
  if (!is.na(match("data", names(this.call)))) {
    warning("lme.lmList will redefine \"data\"")
  }
  ## add object, data, and groups from the call that created object
  last.call <- as.list(attr(fixed, "call"))[-1]
  whichLast <- match(c("object", "data", "na.action"), names(last.call))
  whichLast <- whichLast[!is.na(whichLast)]
  last.call <- last.call[whichLast]
  names(last.call)[match(names(last.call), "object")] <- "fixed"
  this.call[names(last.call)] <- last.call
  if (missing(random)) {
    random <- eval(as.call(this.call[["fixed"]][-2]))
  }
  random <- reStruct(random, data = NULL)
  mData <- this.call[["data"]]
  if (is.null(mData)) {			# will try to construct
    allV <- all.vars(formula(random))
    if (length(allV) > 0) {
      alist <- lapply(as.list(allV), as.name)
      names(alist) <- allV
      alist <- c(as.list(as.name("data.frame")), alist)
      mode(alist) <- "call"
      mData <- eval(alist, sys.parent(1))
    }
  } else {
    if (is.name(mData)) {
      mData <- eval(mData)
    } 
  }

  reSt <- reStruct(random, data = mData) # getting random effects names
  names(reSt) <- names(grpForm)
  if (length(reSt) > 1) {
    stop("Can only fit lmList objects with single grouping variable")
  }
  rNames <- Names(reSt[[1]])
  if (all(match(rNames, names(aux1 <- coef(fixed)), 0))) {
    if (isInitialized(reSt)) {
      warning("Initial value for reStruct overwritten in lme.lmList")
    }
    matrix(reSt[[1]]) <- var(na.omit(aux1)[, rNames])/(pooledSD(fixed)^2)
  }
  this.call[["random"]] <- reSt
  do.call("lme", this.call)
}

lme.formula <- 
  function(fixed,
	   data = sys.parent(),
	   random = pdSymm( eval( as.call( fixed[ -2 ] ) ) ),
	   correlation = NULL,
	   weights = NULL,
	   subset,
	   REML = FALSE, 
	   na.action = na.fail, 
	   control = list())
{
  Call <- match.call()

  ## control parameters
  controlvals <- lmeControl()
  controlvals[names(control)] <- control

  ##
  ## checking arguments
  ##
  if (!inherits(fixed, "formula") || length(fixed) != 3) {
    stop("\nFixed-effects model must be a formula of the form \"resp ~ pred\"")
  }
  reSt <- reStruct(random, REML = REML, data = NULL)
  groups <- getGroupsFormula(reSt)
  if (is.null(groups)) {
    if (inherits(data, "groupedData")) {
      groups <- getGroupsFormula(data)
      groupsL <- rev(getGroupsFormula(data, asList = TRUE))
      Q <- length(groupsL)
      if (length(reSt) != Q) { # may need to repeat reSt
	if (length(reSt) != 1) {
	  stop("Incompatible lengths for \"random\" and grouping factors")
	}
	auxForm <- eval(parse(text=paste("~",deparse(formula(random)[[2]]),"|",
				  deparse(groups[[2]]))))
	reSt <- reStruct(auxForm, REML = REML, data = NULL)
      } else {
	names(reSt) <- names(groupsL)
      }
    } else {
      stop (paste("Data must inherit from \"groupedData\" class ",
		  "if random does not define groups"))
    }
  }
  ## check if corStruct is present and assign groups to its formula
  if (!is.null(correlation)) {
    ## will assign innermost group
    aux <- getGroupsFormula(eval(parse(text = paste("~1", 
			   deparse(groups[[2]]), sep ="|"))), asList = TRUE)
    aux <- aux[[length(aux)]]
    attr(correlation, "formula") <- 
      eval(parse(text = paste("~", 
		     deparse(getCovariateFormula(formula(correlation))[[2]]),
		     "|", deparse(aux[[2]]))))
  }
  ## create an lme structure containing the random effects model and plug-ins
  lmeSt <- lmeStruct(reStruct = reSt, corStruct = correlation, 
		     varStruct = varFunc(weights))

  ## extract a data frame with enough information to evaluate
  ## fixed, groups, reStruct, corStruct, and varStruct
  mfArgs <- list(formula = asOneFormula(formula(lmeSt), fixed, groups),
		 data = data, na.action = na.action)
  if (!missing(subset)) {
    mfArgs[["subset"]] <- asOneSidedFormula(Call[["subset"]])[[2]]
  }
  dataMix <- do.call("model.frame", mfArgs)
  origOrder <- row.names(dataMix)	# preserve the original order
  ## sort the model.frame by groups and get the matrices and parameters
  ## used in the estimation procedures
  grps <- getGroups(dataMix, 
	     eval(parse(text = paste("~1", deparse(groups[[2]]), sep = "|"))))

  ## ordering data by groups
  if (inherits(grps, "factor")) {	# single level
    ord <- order(grps)	#"order" treats a single named argument peculiarly
    grps <- data.frame(grps)
    row.names(grps) <- origOrder
    names(grps) <- as.character(deparse((groups[[2]])))
  } else {
    ord <- do.call("order", grps)
    ## making group levels unique
    for(i in 2:ncol(grps)) {
      grps[, i] <- paste(as.character(grps[, i-1]), as.character(grps[,i]),
			 sep = "/")
      NULL
    }
  }
  grps <- grps[ord, , drop = FALSE]
  dataMix <- dataMix[ord, ,drop = FALSE]
  revOrder <- match(origOrder, row.names(dataMix)) # putting in orig. order

  ## obtaining basic model matrices
  N <- nrow(grps)
  Z <- model.matrix(reSt, dataMix)
  ncols <- attr(Z, "ncols")
  Names(lmeSt$reStruct) <- attr(Z, "nams")
  ## keeping the contrasts for later use in predict
  contr <- attr(Z, "contr")
  X <- model.frame(fixed, dataMix)
  auxContr <- lapply(X, function(el) 
		     if (inherits(el, "factor")) contrasts(el))
  contr <- c(contr, auxContr[is.na(match(names(auxContr), names(contr)))])
  contr <- contr[!unlist(lapply(contr, is.null))]
  X <- model.matrix(fixed, X)
  y <- eval(fixed[[2]], dataMix)
  ncols <- c(ncols, dim(X)[2], 1)
  Q <- ncol(grps)
  ## creating the condensed linear model
  attr(lmeSt, "conLin") <-
    list(Xy = array(c(Z, X, y), c(N, sum(ncols)), 
	     list(row.names(dataMix), c(dimnames(Z)[[2]], dimnames(X)[[2]],
					deparse(fixed[[2]])))),
	 dims = MEdims(grps, ncols), logLik = 0)
  ## initialization
  lmeSt <- initialize(lmeSt, dataMix, grps, control = controlvals)
  parMap <- attr(lmeSt, "pmap")
  ## Checking possibility of single decomposition
  if (length(coef(lmeSt)) == length(coef(lmeSt$reStruct)) &&
      is.null(weights))  {	# can do one decomposition
    ## need to save conLin for calculating fitted values and residuals
    oldConLin <- attr(lmeSt, "conLin")
    decomp <- T
    attr(lmeSt, "conLin") <- MEdecomp(attr(lmeSt, "conLin"))
  } else decomp <- F

  if (needUpdate(lmeSt)) {              # initializing varying weights
    attr(lmeSt, "lmeFit") <- MEestimate(lmeSt, grps)
    lmeSt <- update(lmeSt, dataMix)
  }
  ##
  ## getting the linear mixed effects fit object,
  ## possibly iterating for variance functions
  ##
  numIter <- 0
  attach(controlvals)
  repeat {
    oldPars <- coef(lmeSt)
    if( exists( "is.R" ) && is.function( is.R ) && is.R() ) {
      aNlm <- nlm(f = function(lmePars) -logLik(lmeSt, lmePars),
                  p = c(coef(lmeSt)),
                  hessian = TRUE,
                  print = ifelse(msVerbose, 2, 0))
      numIter0 <- NULL
      coef(lmeSt) <- aNlm$estimate
    } else {
      aMs <- ms(~-logLik(lmeSt, lmePars),
                start = list(lmePars = c(coef(lmeSt))),
                control = list(rel.tolerance = msTol, maxiter = msMaxIter,
		  scale = msScale), trace = msVerbose)
      coef(lmeSt) <- aMs$parameters
      numIter0 <- aMs$numIter <- aMs$flags[31]
    }
    attr(lmeSt, "lmeFit") <- MEestimate(lmeSt, grps)
    ## checking if any updating is needed
    if (!needUpdate(lmeSt)) break
    ## updating the fit information
    numIter <- numIter + 1
    lmeSt <- update(lmeSt, dataMix)
    ## calculating the convergence criterion
    aConv <- coef(lmeSt)
    conv <- abs((oldPars - aConv)/ifelse(aConv == 0, 1, aConv))
    aConv <- NULL
    for(i in names(lmeSt)) {
      if (any(parMap[,i])) {
	aConv <- c(aConv, max(conv[parMap[,i]]))
	names(aConv)[length(aConv)] <- i
      }
    }
    if (max(aConv) <= tolerance) {
      break
    }
    if (numIter > maxIter) {
      stop("Maximum number of iterations reached without convergence.")
    }
  }
  detach()

  ## wrapping up
  lmeFit <- attr(lmeSt, "lmeFit")
  names(lmeFit$beta) <- namBeta <- dimnames(X)[[2]]
  varFix <- crossprod(lmeFit$sigma * lmeFit$varFix)
  dimnames(varFix) <- list(namBeta, namBeta)
  ##
  ## fitted.values and residuals (in original order)
  ##
  Fitted <- fitted(lmeSt, level = 0:Q, 
		   conLin = if (decomp) {
		     oldConLin 
		   } else {
		     attr(lmeSt, "conLin")
		   })[revOrder, , drop = FALSE]
  Resid <- y[revOrder] - Fitted
  attr(Resid, "std") <- lmeFit$sigma/(varWeights(lmeSt)[revOrder])
  ## putting groups back in original order
  grps <- grps[revOrder, , drop = FALSE]
  ## making random effects estimates consistently ordered
#  for(i in names(lmeSt$reStruct)) {
#    lmeFit$b[[i]] <- lmeFit$b[[i]][unique(as.character(grps[, i])),, drop = F]
#    NULL
#  }
  ## inverting back reStruct 
  lmeSt$reStruct <- solve(lmeSt$reStruct)
  ## saving part of dims
  dims <- attr(lmeSt, "conLin")$dims[c("N", "Q", "qvec", "ngrps", "ncol")]
  ## getting the approximate var-cov of the parameters 
  if (controlvals$apVar) {
    apVar <- lmeApVar(lmeSt, lmeFit$sigma, 
		      .relStep = controlvals[[".relStep"]],
		      natural = controlvals[["natural"]])
  } else {
    apVar <- "Approximate variance-covariance matrix not available"
  }
  ## getting rid of condensed linear model and fit
  attr(lmeSt, "conLin") <- NULL
  attr(lmeSt, "lmeFit") <- NULL
  ##
  ## creating the  lme object
  ##
  estOut <- list(modelStruct = lmeSt,
		 dims = dims,
		 contrasts = contr,
		 coefficients = list(
		     fixed = lmeFit$beta,
		     random = lmeFit$b),
		 varFix = varFix,
		 sigma = lmeFit$sigma,
		 apVar = apVar,
		 logLik = lmeFit$logLik,
		 numIter = if (needUpdate(lmeSt)) numIter
		   else numIter0,
		 groups = grps,
		 call = Call,
		 estMethod = c("ML", "REML")[REML + 1],
		 fitted = Fitted,
		 residuals = Resid)
  if (inherits(data, "groupedData")) {
    ## saving labels and units for plots
    attr(estOut, "units") <- attr(data, "units")
    attr(estOut, "labels") <- attr(data, "labels")
  }
  class(estOut) <- "lme"
  estOut
}

### Auxiliary functions used internally in lme and its methods

lmeApVar <-
  function(lmeSt, sigma, conLin = attr(lmeSt, "conLin"), natural = TRUE,
           .relStep = (.Machine$double.eps)^(1/3))
{
  ## calculate approximate variance-covariance matrix of all parameters
  ## except the fixed effects. By default, uses natural parametrization for
  ## for pdSymm matrices
  fullLmeLogLik <-
    function(Pars, object, conLin, dims, N, settings) {
      ## logLik as a function of sigma and coef(lmeSt)
      npar <- length(Pars)
      sigma <- exp(Pars[npar])              # within-group std. dev.
      Pars <- Pars[-npar]
      coef(object) <- Pars
      if ((lO <- length(object)) > 1) {
	for(i in lO:2) {
	  conLin <- recalc(object[[i]], conLin)
	  NULL
	}
      }
      val <- .C("mixed_loglik",
		as.double(conLin$Xy),
		as.integer(unlist(dims)),
		as.double(sigma * unlist(pdFactor(solve(object$reStruct)))),
		as.integer(settings),
		logLik = double(1),
		lRSS = double(1))[c("logLik", "lRSS")]
      aux <- (exp(val[["lRSS"]])/sigma)^2
      conLin[["logLik"]] + val[["logLik"]] + (N * log(aux) - aux)/2
    }
  dims <- conLin$dims
  sett <- attr(lmeSt, "settings")
  N <- dims$N - sett[1] * dims$ncol[dims$Q + 1]
  sett[2:3] <- c(1, 0)			# asDelta = T and no grad/Hess
  conLin[["logLik"]] <- 0               # making sure
  sig2 <- sigma * sigma
  reSt <- lmeSt[["reStruct"]]
  for(i in seq(along = reSt)) {
    matrix(reSt[[i]]) <- sig2 * pdMatrix(reSt[[i]])
    if (inherits(reSt[[i]], "pdSymm") && natural) {
      reSt[[i]] <- pdNatural(reSt[[i]])
    }
  }
  lmeSt[["reStruct"]] <- reSt
  Pars <- c(coef(lmeSt), lSigma = log(sigma))
  val <- fdHess(Pars, fullLmeLogLik, lmeSt, conLin, dims, N, sett,
		.relStep = .relStep)[["Hessian"]]
  if (all(eigen(val)$values < 0)) {
    ## negative definite - OK
    val <- solve(-val)
    nP <- names(Pars)
    dimnames(val) <- list(nP, nP)
    attr(val, "Pars") <- Pars
    attr(val, "natural") <- natural
    val
  } else {
    ## problem - solution is not a maximum
    "Non-positive definite approximate variance-covariance"
  }
}

MEdecomp <-
 function(conLin)
  ## decompose a condensed linear model.  Returns another condensed 
  ## linear model 
{
  dims <- conLin$dims
  if (dims[["StrRows"]] >= dims[["ZXrows"]]) {
    ## no pint in doing the decomposition
    return(conLin)
  }
  dc <- array(.C("mixed_decomp", 
		 as.double(conLin$Xy), 
		 as.integer(unlist(dims)))[[1]], 
	      c(dims$StrRows, dims$ZXcols))
  dims$ZXrows <- dims$StrRows
  dims$ZXoff <- dims$DecOff
  dims$ZXlen <- dims$DecLen
  conLin[c("Xy", "dims")] <- list(Xy = dc, dims = dims)
  conLin
}

MEEM <-
  function(object, conLin, niter = 0)
  ## perform niter iterations of the EM algorithm for conLin 
  ## assumes that object is in precision form
{
  if (niter > 0) {
    dd <- conLin$dims
    pdCl <- attr(object, "settings")[-(1:3)]
    pdCl[pdCl == -1] <- 0
    precvec <- unlist(pdFactor(object))
    zz <- .C("mixed_EM",
	     as.double(conLin$Xy),
	     as.integer(unlist(dd)),
	     precvec = as.double(precvec),
	     as.integer(niter),
	     as.integer(pdCl),
	     as.integer(attr(object, "settings")[1]),
	     double(1),
	     double(length(precvec)),
	     double(1))[["precvec"]]
    Prec <- vector("list", length(object))
    names(Prec) <- names(object)
    for (i in seq(along = object)) {
      len <- dd$qvec[i]^2
      matrix(object[[i]]) <-
        crossprod(matrix(zz[1:len + dd$DmOff[i]], ncol = dd$qvec[i]))
    }
  }
  object
}

MEestimate <-
  function(object, groups, conLin = attr(object, "conLin"))
{
  dd <- conLin$dims
  nc <- dd$ncol
  REML <- attr(object$reStruct, "settings")[1]
  Q <- dd$Q
  rConLin <- recalc(object, conLin)
  zz <- .C("mixed_estimate",
	   as.double(rConLin$Xy),
	   as.integer(unlist(dd)),
	   as.double(unlist(pdFactor(object$reStruct))),
	   as.integer(REML),
	   double(1),
	   estimates = double(dd$StrRows * dd$ZXcols),
	   as.logical(F))[["estimates"]]
  estimates <- array(zz, c(dd$StrRows, dd$ZXcols))
  resp <- estimates[ , dd$ZXcols]
  reSt <- object$reStruct
  nam <- names(reSt)
  val <- vector(mode = "list", length = Q)
  names(val) <- nam
  start <- dd$StrRows * c(0, cumsum(nc))
  for (i in seq(along = reSt)) {
    val[[i]] <- 
      matrix(resp[as.vector(outer(1:(nc[i]), dd$SToff[[i]] - start[i], "+"))],
	     ncol = nc[i], byrow = TRUE, 
	     dimnames = list(unique(as.character(groups[, nam[i]])),
		 Names(reSt[[i]])))
    NULL
  }
  p <- nc[Q + 1]
  N <- dd$N - REML * p
  dimE <- dim(estimates)
  list(logLik = N * (log(N) - (1 + log(2 * pi)))/2 + rConLin$logLik, 
       b = rev(val), 
       beta = resp[dimE[1] - (p:1)],
       sigma = abs(resp[dimE[1]])/sqrt(N),
       varFix = t(solve(estimates[dimE[1]-(p:1), dimE[2]-(p:1), drop = FALSE])))
}

MEdims <-
  function(groups, ncols)
{
  ## define constants used in matrix decompositions and log-lik calculations
  ## first need some local functions
  lengths <-
    ## returns the group lengths from a vector of last rows in the group
    function(lstrow) diff(c(0, lstrow))
  offsets <-
    ## converts total number of columns(N), columns per level(ncols), and
    ## a list of group lengths to offsets in C arrays
    function(N, ncols, lstrow, triangle = FALSE)
  {
    pop <- function(x) x[-length(x)]
    cstart <- c(0, cumsum(N * ncols))
    for (i in seq(along = lstrow)) {
      lstrow[[i]] <- cstart[i] +
        if (triangle) {
          lstrow[[i]] - ncols[i]        # storage offsets style
        } else {
          pop(c(0, lstrow[[i]]))        # decomposition style
        }
    }
    lstrow
  }
  Q <- ncol(groups)                     # number of levels
  N <- nrow(groups)                     # number of observations
  ## 'isLast' indicates if the row is the last row in the group at that level.
  ## this version propagates changes from outer groups to inner groups
#  isLast <- (array(unlist(lapply(c(rev(as.list(groups)),
#                                 list(X = rep(0, N), y = rep(0, N))),
#                                function(x) c(0 != diff(codes(x)), TRUE))),
#                  c(N, Q+2), list(NULL, c(rev(names(groups)), "X", "y")))
#             %*% (row(diag(Q+2)) >= col(diag(Q+2)))) != 0
  ## this version does not propagate changes from outer to inner.
  isLast <- array(FALSE, dim(groups) + c(0, 2),
                  list(NULL, c(rev(names(groups)), "X", "y")))
  for(i in 1:Q) {
    isLast[, Q + 1 - i] <- c(0 != diff(codes(groups[[i]])), TRUE)
  }
  isLast[N,  ] <- TRUE
  lastRow <- as.list(apply(isLast, 2, function(x) seq(along = x)[x]))
  isLast <- t(isLast)
  strSizes <- cumsum(ncols * isLast) * isLast # required storage sizes
  lastStr <- as.list(apply(t(strSizes), 2, function(x) x[x != 0]))
  strRows <- max(lastStr[[length(lastStr)]])
  lastBlock <- vector("list", Q)
  names(lastBlock) <- dimnames(strSizes)[[1]][1:Q]
  for(i in 1:Q) lastBlock[[i]] <- c(strSizes[i, -N], strRows)
  maxStr <- do.call("pmax", lastBlock)
  for(i in 1:Q) lastBlock[[i]] <- maxStr[as.logical(lastBlock[[i]])]
  lastBlock <- c(lastBlock, list(X = strRows, y = strRows))
  list(N = N,                   # total number of rows in data
       ZXrows = N,              # no. of rows in array
       ZXcols = sum(ncols),     # no. of columns in array
       Q = Q,                   # no. of levels of random effects
       StrRows = strRows,       # no. of rows required for storage
       qvec = ncols * c(rep(1, Q), 0, 0), # lengths of random effects
                                        # no. of groups at each level
       ngrps = c(unlist(lapply(lastRow, length), N, N)),
                                        # offsets into DmHalf array by level
       DmOff = (c(0, cumsum(ncols^2)))[1:(Q+2)],
       ncol = ncols,            # no. of columns decomposed per level
                                        # no. of columns rotated per level
       nrot = (rev(c(0, cumsum(rev(ncols)))))[-1],
       ZXoff = offsets(N, ncols, lastRow), # offsets into ZXy
       ZXlen = lapply(lastRow, lengths), # lengths of ZXy groups
                                        # storage array offsets
       SToff = offsets(strRows, ncols, lastStr, triangle = TRUE),
                                        # decomposition offsets
       DecOff = offsets(strRows, ncols, lastBlock),
                                        # decomposition lengths
       DecLen = lapply(lastBlock, lengths)
       )
}

### Methods for standard generics

anova.lme <- 
  function(object, ..., test = TRUE, verbose = FALSE)

{
  ## returns the likelihood ratio statistics, the AIC, and the BIC
  dots <- list(...)
  if ((rt <- length(dots) + 1) == 1) {
    if (!inherits(object,"lme")) {
      stop("Object must inherit from class \"lme\" ")
    }
    ##
    ## if just one object returns the t.table for the fixed effects
    ##
    stdFix <- sqrt(diag(object$varFix))
    ##
    ## fixed effects coefficients, std. deviations, z-ratios, and p-values
    ##
    beta <- fixed.effects(object)
    zratio <- beta/stdFix
    aod <- data.frame(beta, stdFix, zratio, 2 * pnorm(-abs(zratio)))
    dimnames(aod) <- 
      list(names(beta),c("Value","Std.Error","z-value", "p-value"))
    attr(aod,"rt") <- rt
  }
  ##
  ## Otherwise construct the likelihood ratio and information table
  ## objects in ... may inherit from lm, lmList, and lme (for now)
  ##
  else {
    ancall <- sys.call()
    ancall$verbose <- ancall$test <- NULL
    object <- list(object, ...)
    termsClass <- unlist(lapply(object, data.class))
    if(!all(match(termsClass, c("gls", "lm", "lmList", "lme"), 0))) {
      stop(paste("Objects must inherit from classes \"gls\", \"lm\",",
		 "\"lmList\", or \"lme\""))
    }
    resp <- unlist(lapply(object, 
		  function(el) deparse(getResponseFormula(el)[[2]])))
    ## checking if responses are the same
    subs <- as.logical(match(resp, resp[1], F))
    if (!all(subs))
      warning(paste("Some fitted objects deleted because", 
		    "response differs from the first model"))
    if (sum(subs) == 1)
      stop("First model has a different response from the rest")
    object <- object[subs]
    rt <- length(object)
    termsModel <- lapply(object, function(el) formula(el)[-2])
    estMeth <- unlist(lapply(object, 
			     function(el) {
			       val <- el[["estMethod"]]
			       if (is.null(val)) val <- NA
			       val
			     }))
    ## checking consistency of estimation methods
    if(length(uEst <- unique(estMeth[estMeth != "NA"])) > 1) {
      stop("All fitted objects must have the same estimation method.")
    }
    estMeth[is.na(estMeth)] <- uEst
    ## checking if all models have same fixed effects when estMeth = "REML"
    if(uEst == "REML") {
      aux <- unlist(lapply(termsModel, 
	   function(el) {
	     aux <- terms(el)
	     val <- paste(sort(attr(aux, "term.labels")), collapse = "&")
	     if (attr(aux, "intercept") == 1) {
	       val <- paste(val, "(Intercept)", sep = "&")
	     }
	   }))
      if(length(unique(aux)) > 1) {
	warning(paste("Fitted objects with different fixed effects.",
		      "REML comparisons are not meaningful."))
      }
    }
    termsCall <-
      lapply(object, function(el) {
        if (is.null(val <- el$call)) {
          if (is.null(val <- attr(el, "call"))) {
            stop("Objects must have a \"call\" component or attribute.")
          }
        }
        val
      })
    termsCall <- unlist(lapply(termsCall,
			       function(el) paste(deparse(el), collapse ="")))
    
    aux <- lapply(object, logLik, uEst == "REML")
    dfModel <- unlist(lapply(aux, function(el) attr(el, "df")))
    logLik <- unlist(lapply(aux, function(el) c(el)))
    AIC <- unlist(lapply(aux, AIC))
    BIC <- unlist(lapply(aux, BIC))
    aod <- data.frame(call = termsCall,
		      Model = (1:rt),
		      df = dfModel, 
		      AIC = AIC,
		      BIC = BIC,
		      logLik = logLik,
		      check.names = FALSE)
    if (test) {
      ddf <-  diff(dfModel)
      if (sum(abs(ddf)) > 0) {
	effects <- rep("", rt)
	for(i in 2:rt) {
	  if (ddf[i-1] != 0) {
	    effects[i] <- paste(i - 1, i, sep = " vs. ")
	  }
	}
	pval <- rep(NA, rt - 1)
	ldf <- as.logical(ddf)
	lratio <- 2 * abs(diff(logLik))
	lratio[!ldf] <- NA
	pval[ldf] <- 1 - pchisq(lratio[ldf],abs(ddf[ldf]))
	aod <- data.frame(aod,
			  Test = effects, 
			  "Lik.Ratio" = c(NA, lratio),  
			  "p-value" = c(NA, pval),
			  check.names = FALSE)
      }
    }
    row.names(aod) <- unlist(lapply(as.list(ancall[-1]), as.character))
    attr(aod, "rt") <- rt
    attr(aod, "verbose") <- verbose
  }

  class(aod) <- c("anova.lme", "data.frame")
  aod
}

augPred.lme <- 
  function(object, primary = NULL, minimum = min(primary), 
	   maximum = max(primary), length.out = 51, level = Q, ...)
{
  data <- eval(object$call$data)
  if (!inherits(data, "data.frame")) {
    stop(paste("Data in", substitute(object),
               "call must evaluate to a data frame"))
  }
  if(is.null(primary)) {
    if (!inherits(data, "groupedData")) {
      stop(paste(sys.call()[[1]],
      "without \"primary\" can only be used with fits of groupedData objects"))
    }
    primary <- getCovariate(data)
    prName <- deparse(getCovariateFormula(data)[[2]])
  } else{
    primary <- asOneSidedFormula(primary)[[2]]
    prName <- deparse(primary)
    primary <- eval(primary, data)
  }
  newprimary <- seq(from = minimum, to = maximum, length.out = length.out)

  Q <- object$dims$Q                    # number of levels
  if (is.null(level)) level <- Q
  nL <- length(level)                   # number of requested levels
  maxLev <- max(c(level, 1))
  groups <- getGroups(object, level = maxLev)
  if (!is.ordered(groups)) {
    groups <- ordered(groups, levels = unique(as.character(groups)))
  }
  grName <- ".groups"
  ugroups <- unique(groups)
  value <- data.frame(rep(rep(newprimary, length(ugroups)), nL),
		      rep(rep(ugroups, rep(length(newprimary),
                                           length(ugroups))), nL))
  names(value) <- c(prName, grName)
  ## recovering other variables in data that may be needed for predictions
  ## varying variables will be replaced by their means
  summData <- gsummary(data, groups = groups)
  if (any(toAdd <- is.na(match(names(summData), names(value))))) {
    summData <- summData[, toAdd, drop = FALSE]
  }
  value[, names(summData)] <- summData[value[, 2], ]
  pred <- predict(object, value[1:(nrow(value)/nL), , drop = FALSE], level = level)

  if (nL > 1) {                         # multiple levels
    pred <- pred[, ncol(pred) - (nL - 1):0] # eliminating groups
    predNames <- rep(names(pred), rep(nrow(pred), nL))
    pred <- c(unlist(pred))
  } else {
    predNames <- rep("predicted", nrow(value))
  }
  newvals <- cbind(value[, 1:2], pred)
  names(newvals)[3] <- respName <-
    deparse(getResponseFormula(object)[[2]])
  orig <- data.frame(primary, groups, getResponse(object))
  names(orig) <- names(newvals)
  value <- rbind(orig, newvals)
  attributes(value[, 2]) <- attributes(groups)
  value[, ".type"] <- ordered(c(rep("original", nrow(data)), predNames),
                              levels = c(unique(predNames), "original"))
  class(value) <- c("augPred", class(value))
  labs <- list(x = prName, y = respName)
  unts <- list(x = "", y = "")
  if(inherits(data, "groupedData")) {
    labs[names(attr(data, "labels"))] <- attr(data, "labels")
    unts[names(attr(data, "units"))] <- attr(data, "units")
    attr(value, "units") <- attr(data, "units")
  } 
  attr(value, "labels") <- labs
  attr(value, "units") <- unts
  attr(value, "formula") <- 
      eval(parse(text = paste(respName, "~", prName, "|", grName)))
  value
}

coef.lme <-
  function(object, augFrame = FALSE, level = Q, data, which = 1:ncol(data),
	   FUN = mean, omitGroupingFactor = TRUE, subset = NULL)
{
  Q <- object$dims$Q
  if (length(level) > 1) {
    stop("Only single level allowed")
  }
  fixed <- fixed.effects(object)
  p <- length(fixed)
  value <- random.effects(object, level = 1:level)
  grps <- object[["groups"]]
  if (Q > 1) {
    grpNames <- t(array(rep(rev(names(grps)), Q), c(Q, Q)))
    grpNames[lower.tri(grpNames)] <- ""
    grpNames <- 
      rev(apply(grpNames, 1,
                function(x) paste(x[x != ""], collapse = " %in% ")))[level]
  } else {
    grpNames <- names(grps)
  }
  grps <- grps[, 1:level, drop = FALSE]
  grps <- gsummary(grps, groups = grps[, level])
  if (level == 1) value <- list(value)
  effNams <- unlist(lapply(value, names))
  grps <- grps[row.names(value[[level]]), , drop = FALSE]
  M <- nrow(grps)
  effNams <- unique(c(names(fixed), effNams))
  effs <- array(0, c(M, length(effNams)), 
		list(row.names(grps), effNams))

  effs[, names(fixed)] <- array(rep(fixed, rep(M, p)),	c(M, p))
  for (i in 1:level) {
    nami <- names(value[[i]])
    effs[, nami] <- effs[, nami] + value[[i]][as.character(grps[, i]), ]
  }

  if (augFrame) {			# can only do that for last level
    if (missing(data)) {
      mCall <- object[["call"]]
      data <- mCall[["data"]]
      if (mode(data) == "name") {
	data <- eval(data)
      }
    } 
    data <- as.data.frame(data)
    data <- data[, which, drop = FALSE]
    value <- random.effects(object, T, level, data, FUN = FUN,
			    omitGroupingFactor = omitGroupingFactor,
                            subset = subset)
    whichKeep <- is.na(match(names(value), effNams))
    if (any(whichKeep)) {
      effs <- cbind(effs, value[, whichKeep, drop = FALSE])
    }
  }
  effs <- as.data.frame(effs)
  attr(effs, "level") <- level
  attr(effs, "label") <- "Coefficients"
  attr(effs, "effectNames") <- effNams
  attr(effs, "standardized") <- F
  attr(effs, "grpNames") <- grpNames
  class(effs) <- unique(c("coef.lme", "random.effects.lme", class(effs)))
  effs
}

fitted.lme <- 
  function(object, level = Q, asList = FALSE)
{
  Q <- object$dims$Q
  val <- object[["fitted"]]
  if (is.character(level)) {		# levels must be given consistently
    nlevel <- match(level, names(val))
    if (any(aux <- is.na(nlevel))) {
      stop(paste("Nonexistent level(s)", level[aux]))
    } 
    level <- nlevel
  } else {				# assuming integers
    level <- 1 + level
  }
  val <- val[, level]
  if (length(level) == 1) {
    grps <- as.character(object[["groups"]][, max(c(1, level - 1))])
    if (asList) {
      val <- split(val, ordered(grps, levels = unique(grps)))
    } else {
      names(val) <- grps
    }
    lab <- "Fitted values"
    if (!is.null(aux <- attr(object, "units")$y)) {
      lab <- paste(lab, aux)
    }
    attr(val, "label") <- lab
  }
  val
}

formula.lme <- function(object) eval(object$call$fixed)

fixed.effects.lme <-
  function(object) object$coefficients$fixed

getGroups.lme <-
  function(object, form, level = Q)
{
  Q <- object$dims$Q
  val <- object[["groups"]][, level]
  if (length(level) == 1) {		# single group
    attr(val, "label") <- names(object[["groups"]])[level]
  }
  val
}

getGroupsFormula.lme <-
  function(object, asList = FALSE)
{
  getGroupsFormula(object$modelStruct$reStruct, asList)
}

getResponse.lme <-
  function(object, form)
{
  val <- resid(object) + fitted(object)
  if (is.null(lab <- attr(object, "labels")$y)) {
    lab <- deparse(getResponseFormula(object)[[2]])
  }
  if (!is.null(aux <- attr(object, "units")$y)) {
    lab <- paste(lab, aux)
  }
  attr(val, "label") <- lab
  val
}

intervals.lme <-
  function(object, level = 0.95, which = c("all", "var-cov", "fixed"))
{
  which <- match.arg(which)
  mult <- -qnorm((1-level)/2)
  val <- list()
  if (which != "var-cov") {		# fixed effects included
    est <- fixed.effects(object)
    std <- sqrt(diag(object$varFix))
    val <- list(fixed = array(c(est - mult * std, est, est + mult * std),
                  c(length(est), 3), list(names(est), c("lower", "est.", "upper"))))
    attr(val[["fixed"]], "label") <- "Fixed effects:"
  }

  if (which != "fixed") {		# variance-covariance included
    if (is.character(aV <- object$apVar)) {
      stop(paste("Cannot get confidence intervals on var-cov components:",
		 aV))
    }
    nat <- attr(aV, "natural")
    est <- attr(aV, "Pars")
    nP <- length(est)
    std <- sqrt(diag(aV))
    lmeSt <- object[["modelStruct"]]
    namL <- names(lmeSt)
    auxVal <- vector("list", length(namL) + 1)
    names(auxVal) <- c(namL, "sigma")
    aux <-
      array(c(est - mult * std, est, est + mult * std),
	    c(nP, 3), list(NULL, c("lower", "est.", "upper")))
    if (nat) {
      for(i in seq(along = lmeSt$reStruct)) {
	if (inherits(lmeSt$reStruct[[i]], "pdSymm")) {
	  lmeSt$reStruct[[i]] <- pdNatural(lmeSt$reStruct[[i]])
	}
      }
    }
    auxVal[["sigma"]] <- exp(aux[nP,])
    attr(auxVal[["sigma"]], "label") <- "Within-group standard error:"
    aux <- aux[-nP,, drop = FALSE]
    dimnames(aux)[[1]] <- namP <- names(coef(lmeSt, F))
    for(i in 1:3) {
      coef(lmeSt) <- aux[,i]
      aux[,i] <- coef(lmeSt, unconstrained = FALSE)
    }
    for(i in namL) {
      if (exists("is.R") && is.function(is.R) && is.R()) {
        ## this is the way the code was originally written but it doesn't
        ## work under S-PLUS for Windows because the Windows version of
        ## grep is broken.  "regexpr" is used in S-PLUS instead but is not
        ## defined in R
        auxVal[[i]] <- aux[grep(i, namP), , drop = FALSE]
      } else {
        auxVal[[i]] <- aux[regexpr(i, namP) != -1, , drop = FALSE]
      }
      dimnames(auxVal[[i]])[[1]] <- 
	substring(dimnames(auxVal[[i]])[[1]], nchar(i) + 2)
      if (i == "reStruct") {
	namR <- names(lmeSt$reStruct)
	auxRe <- vector("list", length(namR))
	names(auxRe) <- namR
	namVR <- dimnames(auxVal[[i]])[[1]]
	for(j in namR) {
          if (exists("is.R") && is.function(is.R) && is.R()) {
            auxRe[[j]] <- auxVal[[i]][grep(j, namVR), , drop = FALSE]
          } else {
            auxRe[[j]] <- auxVal[[i]][regexpr(j, namVR)!=-1, , drop = FALSE]
          }
	  dimnames(auxRe[[j]])[[1]] <-
	    substring(dimnames(auxRe[[j]])[[1]], nchar(j) + 2)
	}
	auxVal[[i]] <- rev(auxRe)
      }
      attr(auxVal[[i]], "label") <-
	switch(i,
	       reStruct = "Random Effects:",
	       corStruct = "Correlation structure:",
	       varStruct = "Variance function:",
	       paste(i,":",sep=""))
    }
    val <- c(val, auxVal)
  }
  attr(val, "level") <- level
  class(val) <- "intervals.lme"
  val
}						  

logLik.lme <-
  function(object, REML)
{
  p <- object$dims$ncol[object$dims$Q + 1]
  N <- object$dims$N
  Np <- N - p
  estM <- object$estMethod
  if (missing(REML)) REML <- estM == "REML"
  val <- object[["logLik"]]
  if (REML && (estM == "ML")) {			# have to correct logLik
    val <- val + (p * (log(2 * pi) + 1) + (N - p) * log(1 - p/N) +
		  sum(log(abs(svd(object$varFix)$d)))) / 2
  }
  if (!REML && (estM == "REML")) {	# have to correct logLik
    val <- val - (p * (log(2*pi) + 1) + N * log(1 - p/N) +
		  sum(log(abs(svd(object$varFix)$d)))) / 2
  }
  attr(val, "nobs") <- object$dims$N - REML * p
  attr(val, "df") <- p + length(coef(object[["modelStruct"]])) + 1
  class(val) <- "logLik"
  val
}

pairs.lme <- 
  function(object, form = ~coef(.), label, id = NULL, idLabels = NULL, 
	   grid = FALSE, ...)
{
  ## scatter plot matrix plots, generally based on coef or random.effects
  if (!inherits(form, "formula")) {
    stop("\"Form\" must be a formula")
  }
  if (length(form) != 2) {
    stop("\"Form\" must be a one-sided formula")
  }
  ## constructing data 
  allV <- all.vars(asOneFormula(form, id, idLabels))
  allV <- allV[is.na(match(allV,c("T","F","TRUE","FALSE")))]
  if (length(allV) > 0) {
    mCall <- object$call
    mData <- mCall[["data"]]
    if (is.null(mData)) {		# try to construct data
      alist <- lapply(as.list(allV), as.name)
      names(alist) <- allV
      alist <- c(as.list(as.name("data.frame")), alist)
      mode(alist) <- "call"
      data <- eval(alist, sys.parent(1))
    } else {
      if (is.name(mData)) {
	data <- eval(mData)
      } else {
	data <- mData
      }
      if (any(naV <- is.na(match(allV, names(data))))) {
	stop(paste(allV[naV], "not found in data"))
      }
    }
  } else data <- NULL

  ## argument list
  dots <- list(...)
  if (length(dots) > 0) args <- dots
  else args <- list()

  ## covariate - must be present as a data.frame
  covF <- getCovariateFormula(form)
  .x <- eval(covF[[2]], list(. = object)) # only function of "."
  if (!inherits(.x, "data.frame")) {
    stop("Covariate must be a data frame")
  }
  level <- attr(.x, "level")
  if (!is.null(effNams <- attr(.x, "effectNames"))) {
    .x <- .x[, effNams, drop = FALSE]
  }
  ## eliminating constant effects
  isFixed <- unlist(lapply(.x, function(el) length(unique(el)) == 1))
  .x <- .x[, !isFixed, drop = FALSE]
  if (ncol(.x) == 1) {
    stop("Cannot do pairs of just one variable")
  }
  if (!missing(label)) {
    names(.x) <- labels
  }
  if (ncol(.x) == 2) {
    ## will use xyplot
    argForm <- .y ~ .x
    argData <- .x
    names(argData) <- c(".x", ".y")
    if (is.null(args$xlab)) {
      args$xlab <- names(.x)[1]
    }
    if (is.null(args$ylab)) {
      args$ylab <- names(.x)[2]
    }
  } else {				# splom
    argForm <- ~ .x
    argData <- list(.x = .x)
  }
  
  auxData <- list()
  ## groups - need not be present
  grpsF <- getGroupsFormula(form)
  if (!is.null(grpsF)) {
    gr <- splitFormula(grpsF, sep = "*")
    for(i in 1:length(gr)) {
      argData[[deparse(gr[[i]][[2]])]] <- eval(gr[[i]][[2]], data)
    }
    if (length(argForm) == 2)
      argForm <- eval(parse(text = paste("~ .x |", deparse(grpsF[[2]]))))
    else argForm <- eval(parse(text = paste(".y ~ .x |", deparse(grpsF[[2]]))))
  }
  
  ## id and idLabels - need not be present
  if (!is.null(id)) {			# identify points in plot
    N <- object$dims$N
    id <- 
      switch(mode(id),
	     numeric = {
	       if ((id <= 0) || (id >= 1)) {
		 stop("Id must be between 0 and 1")
	       }
	       if (is.null(level)) {
	 stop("Covariate must have a level attribute, when groups are present")
       }
	       aux <- t(as.matrix(random.effects(object, level = level)))
	       aux <- as.logical(apply(
	(solve(t(pdMatrix(object$modelStruct$reStruct, fact = TRUE)[[level]]),
		 aux)/object$sigma)^2, 2, sum) > qchisq(1 - id, dim(aux)[1]))
	       aux
	     },
	     call = eval(asOneSidedFormula(id)[[2]], data),
	     stop("\"Id\" can only be a formula or numeric.")
	     )
    if (length(id) == N) {
      ## id as a formula evaluated in data
      if (is.null(level)) {
	stop("Covariate must have a level attribute, when id is a formula")
      }
      auxData[[".id"]] <- id
    }

    if (is.null(idLabels)) {
      idLabels <- row.names(.x)
    } else {
      if (mode(idLabels) == "call") {
	idLabels <-
	  as.character(eval(asOneSidedFormula(idLabels)[[2]], data))
      } else if (is.vector(idLabels)) {
	if (length(idLabels <- unlist(idLabels)) != N) {
	  stop("\"IdLabels\" of incorrect length")
	} 
	idLabels <- as.character(idLabels)
      } else {
	stop("\"IdLabels\" can only be a formula or a vector")
      }
    }
    if (length(idLabels) == N) {
      ## idLabels as a formula evaluated in data
      if (is.null(level)) {
      stop("Covariate must have a level attribute, when idLabels is a formula")
      }
      auxData[[".Lid"]] <- idLabels
    }
  }

  if (length(auxData)) {		# need collapsing
    auxData <- gsummary(as.data.frame(auxData), 
			groups = getGroups(object, level = level))
    auxData <- auxData[row.names(.x), , drop = FALSE]
    if (!is.null(auxData[[".g"]])) {
      argData[[".g"]] <- auxData[[".g"]]
    }

    if (!is.null(auxData[[".id"]])) {
      id <- auxData[[".id"]]
    }

    if (!is.null(auxData[[".Lid"]])) {
      idLabels <- auxData[[".Lid"]]
    }
  }

  assign("id", as.logical(as.character(id)) , frame = 1)
  assign("idLabels", as.character(idLabels), frame = 1)
  assign("grid", grid, frame = 1)

  ## adding to args list
  args <- c(args, formula = list(argForm), data = list(argData))
  if (is.null(args$strip)) {
    args$strip <- function(...) strip.default(..., style = 1)
  }
  if (is.null(args$cex)) args$cex <- par("cex")
  if (is.null(args$adj)) args$adj <- par("adj")

  ## defining the type of plot
  if (length(argForm) == 3) {		# xyplot
    plotFun <- "xyplot"
    args <- c(args, 
	      panel = list(function(x, y, subscripts, ...) 
		  {
                    dots <- list(...)
		    if (grid) panel.grid()
		    panel.xyplot(x, y, ...)
		    if (!is.null(aux <- id[subscripts])) {
		      text(x[aux], y[aux], idLabels[subscripts][aux],
                           cex = dots$cex, adj = dots$adj)
		    }
		  }))
  } else {				# splom
    plotFun <- "splom"
    args <- c(args, 
	      panel = list(function(x, y, subscripts, ...)
		  {
                    dots <- list(...)
		    if (grid) panel.grid()
		    panel.xyplot(x, y, ...)
		    if (!is.null(aux <- id[subscripts])) {
		      text(x[aux], y[aux], idLabels[subscripts][aux],
                           cex = dots$cex, adj = dots$adj)
                    }
		  }))
  }
  do.call(plotFun, args)
}


plot.lme <- 
  function(object, form = resid(., type = "pearson") ~ fitted(.), abline, 
	   id = NULL, idLabels = NULL,  grid, ...)  
  ## Diagnostic plots based on residuals and/or fitted values
{
  if (!inherits(form, "formula")) {
    stop("\"Form\" must be a formula")
  }
  ## constructing data 
  allV <- all.vars(asOneFormula(form, id, idLabels))
  allV <- allV[is.na(match(allV,c("T","F","TRUE","FALSE")))]
  if (length(allV) > 0) {
    mCall <- object$call
    mData <- mCall[["data"]]
    if (is.null(mData)) {		# try to construct data
      alist <- lapply(as.list(allV), as.name)
      names(alist) <- allV
      alist <- c(as.list(as.name("data.frame")), alist)
      mode(alist) <- "call"
      data <- eval(alist, sys.parent(1))
    } else {
      if (is.name(mData)) {
	data <- eval(mData)
      } else {
	data <- mData
      }
      if (any(naV <- is.na(match(allV, names(data))))) {
	stop(paste(allV[naV], "not found in data"))
      }
    }
  } else data <- NULL

  if (inherits(data, "groupedData")) {	# save labels and units, if present
    ff <- formula(data)
    rF <- deparse(getResponseFormula(ff)[[2]])
    cF <- deparse(getCovariateFormula(ff)[[2]])
    lbs <- attr(data, "labels")
    unts <- attr(data, "units")
    if (!is.null(lbs$x)) cL <- paste(lbs$x, unts$x) else cF <- NULL
    if (!is.null(lbs$y)) rL <- paste(lbs$y, unts$y) else rF <- NULL
  } else {
    rF <- cF <- NULL
  }

  ## argument list
  dots <- list(...)
  if (length(dots) > 0) args <- dots
  else args <- list()
  ## appending object to data
  data <- c(as.list(data), . = list(object))

  ## covariate - must always be present
  covF <- getCovariateFormula(form)
  .x <- eval(covF[[2]], data)
  if (!is.numeric(.x)) {
    stop("Covariate must be numeric")
  }
  argForm <- ~ .x
  argData <- data.frame(.x = .x)
  if (is.null(xlab <- attr(.x, "label"))) {
    xlab <- deparse(covF[[2]])
    if (!is.null(cF) && (xlab == cF)) xlab <- cL  #### BUG!!!!
    else if (!is.null(rF) && (xlab == rF)) xlab <- rL
  }
  if (is.null(args$xlab)) args$xlab <- xlab
      
  ## response - need not be present
  respF <- getResponseFormula(form)
  if (!is.null(respF)) {
    .y <- eval(respF[[2]], data)
    if (is.null(ylab <- attr(.y, "label"))) {
      ylab <- deparse(respF[[2]])
      if (!is.null(cF) && (ylab == cF)) ylab <- cL
      else if (!is.null(rF) && (ylab == rF)) ylab <- rL
    }
    argForm <- .y ~ .x
    argData[, ".y"] <- .y
    if (is.null(args$ylab)) args$ylab <- ylab
  }

  ## groups - need not be present
  grpsF <- getGroupsFormula(form)
  if (!is.null(grpsF)) {
    gr <- splitFormula(grpsF, sep = "*")
    for(i in 1:length(gr)) {
      argData[[deparse(gr[[i]][[2]])]] <- eval(gr[[i]][[2]], data)
    }
    if (length(argForm) == 2)
      argForm <- eval(parse(text = paste("~ .x |", deparse(grpsF[[2]]))))
    else argForm <- eval(parse(text = paste(".y ~ .x |", deparse(grpsF[[2]]))))
  }
  ## adding to args list
  args <- c(args, formula = list(argForm), data = list(argData))
  if (is.null(args$strip)) {
    args$strip <- function(...) strip.default(..., style = 1)
  }
  if (is.null(args$cex)) args$cex <- par("cex")
  if (is.null(args$adj)) args$adj <- par("adj")

  if (!is.null(id)) {			# identify points in plot
    id <- 
      switch(mode(id),
	     numeric = {
	       if ((id <= 0) || (id >= 1)) {
		 stop("Id must be between 0 and 1")
	       }
	       as.logical(abs(resid(object, type="pearson")) > -qnorm(id / 2))
	     },
	     call = eval(asOneSidedFormula(id)[[2]], data),
	     stop("\"Id\" can only be a formula or numeric.")
	     )
    if (is.null(idLabels)) {
      idLabels <- getGroups(object)
      if (length(idLabels) == 0) idLabels <- 1:object$dims$N
      idLabels <- as.character(idLabels)
    } else {
      if (mode(idLabels) == "call") {
	idLabels <-
	  as.character(eval(asOneSidedFormula(idLabels)[[2]], data))
      } else if (is.vector(idLabels)) {
	if (length(idLabels <- unlist(idLabels)) != length(id)) {
	  stop("\"IdLabels\" of incorrect length")
	} 
	idLabels <- as.character(idLabels)
      } else {
	stop("\"IdLabels\" can only be a formula or a vector")
      }
    }
  }

  ## defining abline, if needed
  if (missing(abline)) {
    if (missing(form)) {		# r ~ f
      abline <- c(0, 0)
    } else {
      abline <- NULL
    }
  } 

  assign("id", id , frame = 1)
  assign("idLabels", idLabels, frame = 1)
  assign("abl", abline, frame = 1)

  ## defining the type of plot
  if (length(argForm) == 3) {
    if (is.numeric(.y)) {		# xyplot
      plotFun <- "xyplot"
      args <- c(args, 
		panel = list(function(x, y, subscripts, ...) 
		    {
                      dots <- list(...)
		      if (grid) panel.grid()
		      panel.xyplot(x, y, ...)
		      if (!is.null(aux <- id[subscripts])) {
			text(x[aux], y[aux], idLabels[subscripts][aux],
                             cex = dots$cex, adj = dots$adj)
		      }
		      if (!is.null(abl)) {
			panel.abline(abl, ...)
		      }
		    }))
    } else {				# assume factor or character
      plotFun <- "bwplot"
      args <- c(args, 
		panel = list(function(x, y, ...) 
		    {
		      if (grid) panel.grid()
		      panel.bwplot(x, y, ...)
		      if (!is.null(abl)) {
			panel.abline(v = abl[1], ...)
		      }
		    }))
    }
  } else {
    plotFun <- "histogram"
    args <- c(args, 
	      panel = list(function(x, y, ...) 
		  {
		    if (grid) panel.grid()
		    panel.histogram(x, y, ...)
		    if (!is.null(abl)) {
		      panel.abline(v = abl[1], ...)
		    }
		  }))
  }
  ## defining grid
  if (missing(grid)) {
    if (plotFun == "xyplot") grid <- T
    else grid <- F
  }
  assign("grid", grid, frame = 1)
  do.call(plotFun, args)
}

plot.random.effects.lme <-
  function(object, outer = NULL, omitFixed = TRUE, level = Q, ...)
{
  if (!inherits(object, "data.frame")) {
    ## must be a list of data frames
    Q <- length(object)
    if (length(level) > 1) {
      stop("Only single level allowed.")
    }
    oAttr <- attributes(object)[c("label", "standardized", "namsEff")]
    object <- object[[level]]
    oAttr$namsEff <- oAttr$namsEff[level]
    attributes(object)[c("label", "standardized", "namsEff")] <- oAttr
  }
  if (omitFixed) {			# eliminating constant effects
    isFixed <- unlist(lapply(object, function(el) length(unique(el)) == 1))
    if (any(isFixed)) {
      oattr <- attributes(object)
      oattr <- oattr[names(oattr) != "names"]
      object <- object[, !isFixed, drop = FALSE]
      oattr$effectNames <- oattr$effectNames[!is.na(match(oattr$effectNames,
							  names(object)))]
      attributes(object)[names(oattr)] <- oattr
    }
  }
  eNames <- attr(object, "effectNames")
  eLen <- length(eNames)
  argData <- data.frame(.pars = as.vector(unlist(object[, eNames])), 
   	 .enames = ordered(rep(eNames, rep(nrow(object), eLen)),
	     level = eNames))
  for(i in names(object)[is.na(match(names(object), eNames))]) {
    argData[[i]] <- rep(object[[i]], eLen)
  }
  argForm <- .groups ~ .pars | .enames
  argData[[".groups"]] <- rep(row.names(object), eLen)
  if (!is.null(outer)) {
    if (!inherits(outer, "formula") || (length(outer) != 2)) {
      stop("\"Outer\" must be a one-sided formula")
    }
    outer <- asOneSidedFormula(outer)
    onames <- all.vars(outer)
    if (any(whichNA <- is.na(match(onames, names(argData))))) {
      stop(paste(paste(onames[whichNA], collapse = ", "),
		 "not available for plotting"))
    }
    argData[[".groups"]] <- 
      as.character(argData[[as.character(onames[1])]])
    if (length(onames) > 1) {
      for(i in onames[-1]) {
	argData[[".groups"]] <- 
	  paste(as.character(argData[[".groups"]]),
		as.character(argData[[i]]))
      }
    }
  }
  argData[[".groups"]] <- ordered(argData[[".groups"]], 
				  levels = unique(argData[[".groups"]]))
  args <- list(formula = argForm, data = argData, ...)
  if (is.null(args$xlab)) {
    args$xlab <- attr(object, "label")
  }
  if (is.null(args$ylab)) {
    if (is.null(outer)) {
      args$ylab <- attr(object, "grpNames")
    } else {
      args$ylab <- deparse(outer[[2]])
    }
  }
      
  if (is.null(args$scales)) {
    if (!is.null(attr(object, "standardized")) &&
	!attr(object, "standardized")) {
      args$scales <- list(x = list(relation = "free"))
    }
  }
  if (is.null(args$strip)) {
    args$strip <- function(...) strip.default(..., style = 1)
  }

  do.call("dotplot", args)
}

predict.lme <- 
  function(object, newdata, level = Q, asList = FALSE, na.action = na.fail)  
{
  ##
  ## method for predict() designed for objects inheriting from class lme
  ##
  Q <- object$dims$Q
  if (missing(newdata)) {		# will return fitted values
    val <- fitted(object, level, asList)
    if (length(level) == 1) return(val)
    return(data.frame(object[["groups"]][,level[level != 0], drop = FALSE],
		      predict = val))
  }
  maxQ <- max(level)			# maximum level for predictions
  mCall <- object$call
  fixed <- eval(as.call(mCall[["fixed"]][-2]))
  newdata <- as.data.frame(newdata)

  if (maxQ > 0) {			# predictions with random effects
    reSt <- object$modelStruct$reStruct[Q - (maxQ - 1):0]
    lmeSt <- lmeStruct(reStruct = reSt)
    groups <- getGroupsFormula(reSt)
    if (any(is.na(match(all.vars(groups), names(newdata))))) {
      ## groups cannot be evaluated in newdata
      stop("Cannot evaluate groups for desired levels on \"newdata\"")
    }
  } else {
    reSt <- NULL
  }

  mfArgs <- list(formula = asOneFormula(formula(reSt), fixed),
		 data = newdata, na.action = na.action)
  dataMix <- do.call("model.frame", mfArgs)
  origOrder <- row.names(dataMix)	# preserve the original order
  whichRows <- match(origOrder, row.names(newdata))
  
  if (maxQ > 0) {
    ## sort the model.frame by groups and get the matrices and parameters
    ## used in the estimation procedures
    grps <- getGroups(newdata, 
	      eval(parse(text = paste("~1", deparse(groups[[2]]), sep = "|"))))
    ## ordering data by groups
    if (inherits(grps, "factor")) {	# single level
      grps <- pruneLevels(grps[whichRows])
      oGrps <- data.frame(grps)
      ## checking if there are missing groups
      if (any(naGrps <- is.na(grps))) {
	grps[naGrps] <- levels(grps)[1]	# input with existing level
      }
      ord <- order(grps)     #"order" treats a single named argument peculiarly
      grps <- data.frame(grps)
      row.names(grps) <- origOrder
      names(grps) <- names(oGrps) <- as.character(deparse((groups[[2]])))
    } else {
      grps <- oGrps <- 
	do.call("data.frame", lapply(grps[whichRows, ], pruneLevels))
      ## checking for missing groups
      if (any(naGrps <- is.na(grps))) {
	## need to input missing groups
	for(i in names(grps)) {
	  grps[naGrps[, i], i] <- levels(grps[,i])[1]
	}
	naGrps <- t(apply(naGrps, 1, cumsum)) # propagating NAs
      }
      ord <- do.call("order", grps)
      ## making group levels unique
      grps[, 1] <- pruneLevels(grps[, 1])
      for(i in 2:ncol(grps)) {
	grps[, i] <- paste(as.character(grps[, i-1]), as.character(grps[,i]),
			   sep = "/")
	NULL
      }
    }
    naGrps <- cbind(F, naGrps)[ord, , drop = FALSE]
    grps <- grps[ord, , drop = FALSE]
    dataMix <- dataMix[ord, ,drop = FALSE]
    ## making sure factor levels are the same as in contrasts
    contr <- object$contrasts
    for(i in names(dataMix)) {
      if (inherits(dataMix[,i], "factor") && !is.null(contr[[i]])) {
        levs <- levels(dataMix[,i])
        levsC <- dimnames(contr[[i]])[[1]]
        if (any(wch <- is.na(match(levs, levsC)))) {
          stop(paste("Levels", paste(levs[wch], collapse = ","),
                     "not allowed for", i))
        }
        if (length(levs) < length(levsC)) {
          if (inherits(dataMix[,i], "ordered")) {
            dataMix[,i] <- ordered(as.character(dataMix[,i]), levels = levsC)
          } else {
            dataMix[,i] <- factor(as.character(dataMix[,i]), levels = levsC)
          }
        }
      }
    }
    revOrder <- match(origOrder, row.names(dataMix)) # putting in orig. order
    Z <- model.matrix(reSt, dataMix, contr)
    ncols <- attr(Z, "ncols")
    Names(lmeSt$reStruct) <- attr(Z, "nams")
  } 
  N <- nrow(dataMix)
  if (length(all.vars(fixed)) > 0) {
    X <- model.matrix(fixed, model.frame(fixed, dataMix), object$contrasts) 
  } else {
    X <- array(1, c(N, 1), list(row.names(dataMix), "(Intercept)"))
  }
  if (maxQ == 0) {
    ## only population predictions
    return(c(X %*% fixed.effects(object)))
  }
  
  ncols <- c(ncols, dim(X)[2], 1)
  ## creating the condensed linear model
  attr(lmeSt, "conLin") <-
    list(Xy = array(c(Z, X, double(N)), c(N, sum(ncols)), 
	     list(row.names(dataMix), c(dimnames(Z)[[2]], dimnames(X)[[2]],
					"resp"))),
	 dims = MEdims(grps, ncols))
  ## Getting the appropriate BLUPs of the random effects
  re <- object$coefficients$random[1:maxQ]
  for(i in names(re)) {
    ugrps <- unique(as.character(grps[, i]))
    val <- array(NA, c(length(ugrps), ncol(re[[i]])),
		 list(ugrps, dimnames(re[[i]])[[2]]))
    mGrps <- match(ugrps, dimnames(re[[i]])[[1]])
    mGrps <- mGrps[!is.na(mGrps)]
    re[[i]] <- re[[i]][mGrps, , drop = FALSE]
    val[dimnames(re[[i]])[[1]], ] <- re[[i]]
    re[[i]] <- val
  }
  
  attr(lmeSt, "lmeFit") <- list(beta = fixed.effects(object), b = re)
  val <- fitted(lmeSt, level = 0:maxQ)
  val[as.logical(naGrps)] <- NA			# setting missing groups to NA
  ## putting back in original order and extracting levels
  val <- val[revOrder, level + 1]		# predictions
  if (maxQ > 1) {                      # making groups unique
    for(i in 2:maxQ) {
      oGrps[, i] <- paste(as.character(oGrps[,i-1]), as.character(oGrps[,i]),
                          sep = "/")
    }
  }
  if (length(level) == 1) {
    if (level > 1) {
      grps <- do.call("paste", 
		      c(lapply(oGrps[, 1:level, drop = FALSE], 
			       as.character),  sep = "/"))
    } else {
      grps <- as.character(oGrps[,1])
    }
    if (asList) {
      val <- split(val, ordered(grps, levels = unique(grps)))
    } else {
      names(val) <- grps
    }
  } else {
    val <- data.frame(oGrps, predict = data.frame(val))
  }
  val
}

print.anova.lme <-
  function(x, verbose = attr(x, "verbose"))
{
  if ((rt <- attr(x,"rt")) == 1) {
    print(zapsmall(as.matrix(x)))
  } else {
    if (verbose) {
      cat("Call:\n")
      objNams <- row.names(x)
      for(i in 1:rt) {
	cat(" ",objNams[i],":\n", sep ="")
	cat(" ",as.character(x[i,"call"]),"\n")
      }
      cat("\n")
    }
    x <- as.data.frame(x[,-1])
    for(i in names(x)) {
      xx <- x[[i]]
      if ((0 == length(levels(xx))) && is.numeric(xx)) {
	xna <- is.na(xx)
	xx <- format(zapsmall(xx, 5))
	xx[xna] <- ""
	x[[i]] <- xx
      }
    }
    invisible(print(x))
  }
}

print.intervals.lme <-
  function(x, ...)
{
  cat(paste("Approximate ", attr(x, "level") * 100,
	    "% confidence intervals\n", sep = ""))
  for(i in names(x)) {
    aux <- x[[i]]
    cat("\n ",attr(aux, "label"), "\n", sep = "")
    if (i == "reStruct") {
      for(j in names(aux)) {
	cat("  Level:", j, "\n")
	print.matrix(aux[[j]], ...)
      }
    } else {
      if (i == "sigma") print(c(aux), ...)
      else print.matrix(aux, ...)
    }
  }
}

print.lme <- 
  function(x, ...)
{
  dd <- x$dims
  if (inherits(x, "nlme")) {	# nlme object
    cat( "Nonlinear mixed-effects model fit by " )
    cat( ifelse( x$estMethod == "REML", "REML\n", "maximum likelihood\n") )
    cat("  Model:", deparse(as.vector(x$call$object)),"\n")
  } else {				# lme objects
    cat( "Linear mixed-effects model fit by " )
    cat( ifelse( x$estMethod == "REML", "REML\n", "maximum likelihood\n") )
  }    
  cat("  Data:", as.character( x$call$data ), "\n")
  cat("  Log-", ifelse(x$estMethod == "REML", "restricted-", ""),
             "likelihood: ", format(x$logLik), "\n", sep = "")
  fixF <- x$call$fixed
  if (inherits(fixF, "formula") || is.call(fixF)) {
    cat("  Fixed:", deparse(as.vector(x$call$fixed)), "\n")
  } else {
    cat("  Fixed:", deparse(lapply(fixF, function(el)
                                   as.name(deparse(as.vector(el))))), "\n")
  }
  print(fixed.effects(x))
  cat("\n")
  print(summary(x$modelStruct), sigma = x$sigma)
  cat("Number of Observations:", dd[["N"]])
  cat("\nNumber of Groups: ")
  Ngrps <- dd$ngrps[1:dd$Q]
  if ((lNgrps <- length(Ngrps)) == 1) {	# single nesting
    cat(Ngrps,"\n")
  } else {				# multiple nesting
    sNgrps <- 1:lNgrps
    aux <- rep(names(Ngrps), sNgrps)
    aux <- split(aux, array(rep(sNgrps, lNgrps), 
			    c(lNgrps, lNgrps))[!lower.tri(diag(lNgrps))])
    names(Ngrps) <- unlist(lapply(aux, paste, collapse = " %in% "))
    cat("\n")
    print(rev(Ngrps))
  }
}

print.summary.lme <-
  function(x, verbose = FALSE, ...)
{
  dd <- x$dims
  verbose <- verbose || attr(x, "verbose")
  if (inherits(x, "nlme")) {	# nlme object
    cat( "Nonlinear mixed-effects model fit by " )
    cat( ifelse( x$estMethod == "REML", "REML\n", "maximum likelihood\n") )
    cat("  Model:", deparse(as.vector(x$call$object)),"\n")
  } else {				# lme objects
    cat( "Linear mixed-effects model fit by " )
    cat( ifelse( x$estMethod == "REML", "REML\n", "maximum likelihood\n") )
  }    
  estMethod <- x$estMethod
  cat(" Data:", as.character( x$call$data ), "\n")
  print( data.frame( AIC = x$AIC, BIC = x$BIC, logLik = x$logLik, row.names = " ") )
  if (verbose) { cat("Convergence at iteration:",x$numIter,"\n") }
  cat("\n")
  print(summary(x$modelStruct), sigma = x$sigma, 
	reEstimates = x$coef$random, verbose = verbose)
  cat("Fixed effects: ")
  fixF <- x$call$fixed
  if (inherits(fixF, "formula") || is.call(fixF)) {
    cat(deparse(as.vector(x$call$fixed)), "\n")
  } else {
    cat(deparse(lapply(fixF, function(el) as.name(deparse(as.vector(el))))),
        "\n")
  }
  print(zapsmall(x$zTable))
  if (nrow(x$zTable) > 1) {
    corr <- x$corFixed
    class(corr) <- "correlation"
    print(corr,
	  title = " Correlation:",
	  ...)
  }
  cat("\nStandardized Within-Group Residuals:\n")
  print(x$residuals)
  cat("\nNumber of Observations:",x$dims[["N"]])
  cat("\nNumber of Groups: ")
  Ngrps <- dd$ngrps[1:dd$Q]
  if ((lNgrps <- length(Ngrps)) == 1) {	# single nesting
    cat(Ngrps,"\n")
  } else {				# multiple nesting
    sNgrps <- 1:lNgrps
    aux <- rep(names(Ngrps), sNgrps)
    aux <- split(aux, array(rep(sNgrps, lNgrps), 
			    c(lNgrps, lNgrps))[!lower.tri(diag(lNgrps))])
    names(Ngrps) <- unlist(lapply(aux, paste, collapse = " %in% "))
    cat("\n")
    print(rev(Ngrps))
  }
}

qqnorm.lme <-
  function(object, form = ~ resid(., type = "p"), abline = NULL,
           id = NULL, idLabels = NULL, grid = FALSE, ...)
  ## normal probability plots for residuals and random effects 
{
  if (!inherits(form, "formula")) {
    stop("\"Form\" must be a formula")
  }
  ## constructing data 
  allV <- all.vars(asOneFormula(form, id, idLabels))
  allV <- allV[is.na(match(allV,c("T","F","TRUE","FALSE")))]
  if (length(allV) > 0) {
    mCall <- object$call
    mData <- mCall[["data"]]
    if (is.null(mData)) {		# try to construct data
      alist <- lapply(as.list(allV), as.name)
      names(alist) <- allV
      alist <- c(as.list(as.name("data.frame")), alist)
      mode(alist) <- "call"
      data <- eval(alist, sys.parent(1))
    } else {
      if (is.name(mData)) {
	data <- eval(mData)
      } else {
	data <- mData
      }
      if (any(naV <- is.na(match(allV, names(data))))) {
	stop(paste(allV[naV], "not found in data"))
      }
    }
  } else data <- NULL
  ## argument list
  dots <- list(...)
  if (length(dots) > 0) args <- dots
  else args <- list()
  ## appending object to data
  data <- c(as.list(data), . = list(object))

  ## covariate - must always be present
  covF <- getCovariateFormula(form)
  .x <- eval(covF[[2]], data)
  labs <- attr(.x, "label")
  if (is.null(labs) || (regexpr("[Rr]esiduals", labs) == -1 &&
                        regexpr("[Rr]andom effects", labs) == -1)) {
    stop("Only residuals and random effects allowed")
  }
  if (regexpr("[Rr]esiduals", labs) == -1) {
    type <- "reff"
  } else {
    type <- "res"
  }
  if (is.null(args$xlab)) args$xlab <- labs
  if (is.null(args$ylab)) args$ylab <- "Quantiles of standard normal"
  if(type == "res") {			# residuals
    fData <- qqnorm(.x, plot.it = F)
    data[[".y"]] <- fData$x
    data[[".x"]] <- fData$y
    dform <- ".y ~ .x"
    if (!is.null(grp <- getGroupsFormula(form))) {
      dform <- paste(dform, deparse(grp[[2]]), sep = "|")
    }
    if (!is.null(id)) {			# identify points in plot
      id <- 
        switch(mode(id),
               numeric = {
                 if ((id <= 0) || (id >= 1)) {
                   stop("Id must be between 0 and 1")
                 }
                 as.logical(abs(resid(object, type="pearson"))
                            > -qnorm(id / 2))
               },
               call = eval(asOneSidedFormula(id)[[2]], data),
               stop("\"Id\" can only be a formula or numeric.")
               )
      if (is.null(idLabels)) {
        idLabels <- getGroups(object)
        if (length(idLabels) == 0) idLabels <- 1:object$dims$N
        idLabels <- as.character(idLabels)
      } else {
        if (mode(idLabels) == "call") {
          idLabels <-
            as.character(eval(asOneSidedFormula(idLabels)[[2]], data))
        } else if (is.vector(idLabels)) {
          if (length(idLabels <- unlist(idLabels)) != length(id)) {
            stop("\"IdLabels\" of incorrect length")
          } 
          idLabels <- as.character(idLabels)
        } else {
          stop("\"IdLabels\" can only be a formula or a vector")
        }
      }
    }
  } else {				# random.effects
    level <- attr(.x, "level")
    std <- attr(.x, "standardized")
    if (!is.null(effNams <- attr(.x, "effectNames"))) {
      .x <- .x[, effNams, drop = FALSE]
    }
    nc <- ncol(.x)
    nr <- nrow(.x)
    fData <- lapply(as.data.frame(.x), qqnorm, plot.it = F)
    fData <- data.frame(.x = unlist(lapply(fData, function(x) x[["y"]])),
			.y = unlist(lapply(fData, function(x) x[["x"]])),
			.g = ordered(rep(names(fData),rep(nr, nc)),
                          levels = names(fData)))
    dform <- ".y ~ .x | .g"
    if (!is.null(grp <- getGroupsFormula(form))) {
      dform <- paste(dform, deparse(grp[[2]]), sep = "*")
      auxData <- data
    } else {
      auxData <- list()
    }
    ## id and idLabels - need not be present
    if (!is.null(id)) {			# identify points in plot
      N <- object$dims$N
      id <- 
        switch(mode(id),
               numeric = {
                 if ((id <= 0) || (id >= 1)) {
                   stop("Id must be between 0 and 1")
                 }
                 aux <- ranef(object, level = level, standard = TRUE)
                 as.logical(abs(c(unlist(aux))) > -qnorm(id / 2))
               },
               call = eval(asOneSidedFormula(id)[[2]], data),
               stop("\"Id\" can only be a formula or numeric.")
               )
      if (length(id) == N) {
        ## id as a formula evaluated in data
        auxData[[".id"]] <- id
      }
      
      if (is.null(idLabels)) {
        idLabels <- row.names(.x)
      } else {
        if (mode(idLabels) == "call") {
          idLabels <-
            as.character(eval(asOneSidedFormula(idLabels)[[2]], data))
        } else if (is.vector(idLabels)) {
          if (length(idLabels <- unlist(idLabels)) != N) {
            stop("\"IdLabels\" of incorrect length")
          } 
          idLabels <- as.character(idLabels)
        } else {
          stop("\"IdLabels\" can only be a formula or a vector")
        }
      }
      if (length(idLabels) == N) {
        ## idLabels as a formula evaluated in data
        auxData[[".Lid"]] <- idLabels
      }
    }

    if (length(auxData)) {		# need collapsing
      auxData <- gsummary(as.data.frame(auxData), 
                          groups = getGroups(object, level = level))
      auxData <- auxData[row.names(.x), , drop = FALSE]

      if (!is.null(auxData[[".id"]])) {
        id <- auxData[[".id"]]
      }
      
      if (!is.null(auxData[[".Lid"]])) {
        idLabels <- auxData[[".Lid"]]
      }
      data <- cbind(fData, auxData)
    } else {
      data <- fData
    }
  }
  assign("id", if (is.null(id)) NULL else as.logical(as.character(id)),
         frame = 1)
  assign("idLabels", as.character(idLabels), frame = 1)
  assign("grid", grid, frame = 1)
  assign("abl", abline, frame = 1)
  if (is.null(args$strip)) {
    args$strip <- function(...) strip.default(..., style = 1)
  }
  if (is.null(args$cex)) args$cex <- par("cex")
  if (is.null(args$adj)) args$adj <- par("adj")

  args <- c(list(formula = eval(parse(text = dform)),
                 data = substitute(data),
                 panel = function(x, y, subscripts, ...){
                   dots <- list(...)
                   if (grid) panel.grid()
                   panel.xyplot(x, y, ...)
                   if (!is.null(aux <- id[subscripts])) {
                     text(x[aux], y[aux], idLabels[subscripts][aux],
                          cex = dots$cex, adj = dots$adj)
                   }
                   if (!is.null(abl)) panel.abline(abl, ...)
                 }), args)
  if(type == "reff" && !std) {
    args[["scales"]] <- list(x = list(relation = "free"))
  }
  do.call("xyplot", args)
}

random.effects.lme <-
  ##  Extracts the random effects from an lme object.
  ##  If aug.frame is true, the returned data frame is augmented with a
  ##  values from the original data object, if available.  The variables
  ##  in the original data are collapsed over the cluster variable by the
  ##  function fun.
function(object, augFrame = FALSE, level = 1:Q, data, which = 1:ncol(data), 
	 FUN = mean, standard = FALSE , omitGroupingFactor = TRUE,
         subset = NULL)
{
  Q <- object$dims$Q
  effects <- object$coefficients$random
  if (Q > 1) {
    grpNames <- t(array(rep(rev(names(effects)), Q), c(Q, Q)))
    grpNames[lower.tri(grpNames)] <- ""
    grpNames <- 
      rev(apply(grpNames, 1, function(x) paste(x[x != ""], collapse = " %in% ")))
  } else {
    grpNames <- names(effects)
  }
  effects <- effects[level]
  grpNames <- grpNames[level]
  if (standard) {
    for (i in names(effects)) {
      effects[[i]] <- 
	t(t(effects[[i]]) / (object$sigma * 
		     sqrt(diag(as.matrix(object$modelStruct$reStruct[[i]])))))
    }
  }
  effects <- lapply(effects, as.data.frame)
  if (augFrame) {
    if (length(level) > 1) {
      stop("Augmentation of random effects only available for single level")
    }
    effects <- effects[[1]]
    effectNames <- names(effects)
    if (missing(data)) {
      mCall <- object[["call"]]
      data <- mCall[["data"]]
      if (mode(data) == "name") {
	data <- eval(data)
      }
    } 
    data <- as.data.frame(data)
    if (is.null(subset)) {              # nlme case
      subset <- eval(object$call[["naPattern"]])
    } else {
      subset <- asOneSidedFormula(as.list(match.call())[["subset"]])
    }
    if (!is.null(subset)) {
      subset <- eval(subset[[2]], data)
      data <- data[subset,  ,drop=FALSE]
    }
    data <- data[, which, drop = FALSE]
    ## eliminating columns with same names as effects
    data <- data[, is.na(match(names(data), effectNames)), drop = FALSE]
    grps <- as.character(object[["groups"]][, level])
    data <- gsummary(data, FUN = FUN, groups = grps)
    if (omitGroupingFactor) {
      data <- 	
	data[, is.na(match(names(data), names(object$modelStruct$reStruct))), 
	      drop = FALSE]
    }
    if (length(data) > 0) {
      effects <- cbind(effects, data[row.names(effects),, drop = FALSE])
    }
    attr(effects, "effectNames") <- effectNames
  } else {
    effects <- lapply(effects,
                      function(el) {
                        attr(el, "effectNames") <- names(el)
                        el
                      })
    if (length(level) == 1) effects <- effects[[1]]
  }
  attr(effects, "label") <- 
    if (standard) {
      "Standardized random effects"
    } else {
      "Random effects"
    }
  attr(effects, "level") <- max(level)
  attr(effects, "standardized") <- standard
  attr(effects, "grpNames") <- grpNames
  class(effects) <- c("random.effects.lme", class(effects))
  effects
}

residuals.lme <- 
  function(object, level = Q, type = c("response", "pearson"), asList = FALSE)
	  
{
  type <- match.arg(type)
  Q <- object$dims$Q
  val <- object[["residuals"]]
  if (is.character(level)) {		# levels must be given consistently
    nlevel <- match(level, names(val))
    if (any(aux <- is.na(nlevel))) {
      stop(paste("Nonexistent level(s)", level[aux]))
    } 
    level <- nlevel
  } else {				# assuming integers
    level <- 1 + level
  }
  if (type == "pearson") {		# standardize
    ## have to standardize properly for when corStruct neq corIdent
    val <- val[, level]/attr(val, "std")
  } else {
    val <- val[, level]
  }
  if (length(level) == 1) {
    grps <- as.character(object[["groups"]][, max(c(1, level - 1))])
    if (asList) {
      val <- split(val, ordered(grps, levels = unique(grps)))
    } else {
      names(val) <- grps
    }
    if (type == "pearson") {
      attr(val, "label") <- "Standardized residuals"
    } else {
      lab <- "Residuals"
      if (!is.null(aux <- attr(object, "units")$y)) {
	lab <- paste(lab, aux)
      }
      attr(val, "label") <- lab
    }
  }
  val
}

summary.lme <- function(object, verbose = FALSE)
{
  ##  variance-covariance estimates for fixed effects
  fixed <- fixed.effects(object)
  stdFixed <- sqrt(diag(as.matrix(object$varFix)))
  object$corFixed <- array(t(object$varFix/stdFixed)/stdFixed,
                           dim(object$varFix), list(names(fixed),names(fixed)))
  ## fixed effects coefficients, std. deviations and z-ratios
  ##
  zTable <- data.frame(fixed, stdFixed, fixed/stdFixed, fixed)
  dimnames(zTable)<-
    list(names(fixed),c("Value","Std.Error","z-value","p-value"))
  zTable[, "p-value"] <- 2 * pnorm(-abs(zTable[,"z-value"]))
  object$zTable <- as.matrix(zTable)
  ##
  ## residuals
  ##
  resd <- resid(object, type = "pearson")
  if (length(resd) > 5) {
    resd <- quantile(resd)
    names(resd) <- c("Min","Q1","Med","Q3","Max")
  }
  object$residuals <- resd
  ##
  ## generating the final object
  ##
  aux <- logLik(object)
  object$BIC <- BIC(aux)
  object$AIC <- AIC(aux)
  if (!inherits(object, "nlme")) {
    class(object) <- c("summary.lme", class(object))
  } else {
    class(object) <- c("summary.nlme", "summary.lme", class(object))
  }
  attr(object, "verbose") <- verbose
  object
}

update.lme <-
  function(object, 
	   fixed,
	   data,
	   random,
	   correlation,
	   weights,
	   subset,
	   REML,
	   na.action, 
	   control)
{
  thisCall <- as.list(match.call())[-(1:2)]
  nextCall <- as.list(object$call)[-1]
  if( exists("is.R") && is.function(is.R) && is.R() ) {
    argNams <- unique( c(names(nextCall), names(thisCall)) )
    args <- vector("list", length(argNams))
    names(args) <- argNams
    args[ names(nextCall) ] <- nextCall
    nextCall <- args
  }
  nextCall[names(thisCall)] <- thisCall
  do.call("lme", nextCall)
}

###*### lmeStruct - a model structure for lme fits

lmeStruct <-
  ## constructor for lmeStruct objects
  function(reStruct, corStruct = NULL, varStruct = NULL)
{

  val <- list(reStruct = reStruct, corStruct = corStruct,
              varStruct = varStruct)
  val <- val[!sapply(val, is.null)]	# removing NULL components
  attr(val, "settings") <- attr(val$reStruct, "settings")
  class(val) <- c("lmeStruct", "modelStruct")
  val
}

##*## lmeStruct methods for standard generics

fitted.lmeStruct <-
  function(object, level = Q, lmeFit = attr(object, "lmeFit"),
	   conLin = attr(object, "conLin"))
{
  if (is.null(conLin)) {
    stop("No condensed linear model")
  }
  if (is.null(lmeFit)) {
    stop("No fitted lme object")
  } 
  dd <- conLin$dims
  Q <- dd$Q
  Qp1 <- Q + 1
  nc <- dd$ncol
  fit <- array(0, c(dd$N, Qp1), 
       list(dimnames(conLin$Xy)[[1]], c("fixed", rev(names(object$reStruct)))))
  ZXstart <- rev(cumsum(c(1, nc[1:Q])))
  ZXend <- rev(cumsum(nc[1:Qp1]))
  ZXlen <- dd$ZXlen[Q:1]
  ZXngrps <- dd$ngrps[Q:1]
  ZXb <- lmeFit$b
  nc <- nc[Q:1]

  fit[, "fixed"] <-			# population fitted values
    conLin$Xy[, ZXstart[1]: ZXend[1], drop = FALSE] %*% lmeFit$beta

  for(i in 1:Q) {
    j <- i + 1
    fit[, j] <- fit[, i] + 
      (conLin$Xy[, ZXstart[j]:ZXend[j], drop = FALSE] * 
       ZXb[[i]][rep(1:ZXngrps[i], ZXlen[[i]]),,drop = FALSE]) %*% rep(1, nc[i])
  }
  fit[, level + 1]
}

initialize.lmeStruct <-
  function(object, data, groups, conLin = attr(object, "conLin"), 
	   control= list(niterEM = 20, gradHess = TRUE))
{
  object[] <- lapply(object, initialize, data, conLin, control)
  theta <- lapply(object, coef)
  len <- unlist(lapply(theta, length))
  num <- seq(along = len)
  if (sum(len) > 0) {
    pmap <- outer(rep(num, len), num, "==")
  } else {
    pmap <- array(F, c(1, length(len)))
  }
  dimnames(pmap) <- list(NULL, names(object))
  attr(object, "pmap") <- pmap
  if (length(coef(object)) == length(coef(object$reStruct)) && # only reStruct
      !needUpdate(object) &&		# and fixed varFunc 
      (attr(object, "settings")[4] >= 0) && # known pdMat class
      control[["gradHess"]]) { 
    ## can use numerical derivatives
    attr(object, "settings")[2:3] <- c(0, 1)
    class(object) <- c("lmeStructInt", class(object))
  }
  if (needUpdate(object)) {
    attr(object, "lmeFit") <- MEestimate(object, groups)
    update(object, data)
  } else {
    object
  }
}

logLik.lmeStruct <-
  function(object, Pars, conLin = attr(object, "conLin"))
{
  coef(object) <- Pars			# updating parameter values
  recalc(object, conLin)[["logLik"]]	# updating conLin
}

logLik.lmeStructInt <-
  function(object, Pars, conLin = attr(object, "conLin"))
{
  ## logLik for objects with reStruct parameters only, with 
  ## internally defined class
  q <- length(Pars)
  aux <- .C("mixed_loglik",
	    as.double(conLin[["Xy"]]),
	    as.integer(unlist(conLin$dims)),
	    as.double(Pars),
	    as.integer(attr(object, "settings")),
	    val = double(1 + q * (q + 1)),
	    double(1))[["val"]]
  val <- aux[1]
  attr(val, "gradient") <- -aux[1 + (1:q)]
  attr(val, "hessian") <- -array(aux[-(1:(q+1))], c(q, q))
  val
}

residuals.lmeStruct <-
  function(object, level = Q, lmeFit = attr(object, "lmeFit"),
	   conLin = attr(object, "conLin"))
{
  Q <- conLin$dims$Q
  conLin$Xy[, conLin$dims$ZXcols] - fitted(object, level, lmeFit, conLin)
}

varWeights.lmeStruct <-
  function(object)
{
  if (is.null(object$varStruct)) rep(1, attr(object, "conLin")$dims$N)
  else varWeights(object$varStruct)
}

## Auxiliary control functions

lmeScale <- function(start) 
# 
# function used to set the scale inside ms(), for lme() and nlme()
# calls
#
{
  scale <- abs(start)
  nonzero <- scale > 0
  if (any(nonzero)) {
    scale[nonzero] <- 1/scale[nonzero]
    scale[!nonzero] <- median(scale[nonzero])
  }
  else {
    scale <- rep(1, length(scale))
  }
  scale
}

lmeControl <-
  ## Control parameters for lme
  function(maxIter = 50, msMaxIter = 50, tolerance = 1e-6, niterEM = 25,
	   msTol = 1e-7, msScale = lmeScale, msVerbose = FALSE,
           returnObject = FALSE, gradHess = TRUE, apVar = TRUE, 
	   .relStep = (.Machine$double.eps)^(1/3), natural = TRUE)
{
  list(maxIter = maxIter, msMaxIter = msMaxIter, tolerance = tolerance,
       niterEM = niterEM, msTol = msTol, msScale = msScale,
       msVerbose = msVerbose, returnObject = returnObject, gradHess = gradHess,
       apVar = apVar, .relStep = .relStep, natural = natural)
}

## Local Variables:
## mode:S
## End:

