/*  gui.c */

/* Vis5D version 4.3 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <unistd.h>
#include "../lui5/lui.h"
#include "analysis.h"
#include "api.h"
#include "cursor.h"
#include "graphics.h"   /*TODO: remove this!!!! */
#include "gui.h"
#include "labels.h"
#include "slice.h"
#include "script.h"
#include "tclsave.h"

#include "rgbsliders.h"
#include "isocolor.h"
#include "uvwwidget.h"
#include "sounding.h"
#include "soundingGUI.h"

#define MM_NORMAL 1
#define MM_TRAJ   2
#define MM_LABEL  3
#define MM_SLICE  4
#define MM_PROBE  5
#define MM_SOUND  6

static GuiContext gtx_table[VIS5D_MAX_CONTEXTS];

GuiContext get_gui_gtx( int index )
{
  GuiContext gtx;
  if ((gtx = gtx_table[index]) == NULL || index<0 || index>=VIS5D_MAX_CONTEXTS) {
    printf("bad gui_context\n");
    return NULL;
  }
  return gtx;
}


/*
 * Deallocate a context and everything it points to.
 */
void destroy_gui_context( GuiContext gtx )
{
   free( gtx );
}


/*
 * Allocate a new vis5d context, initialize to default values.
 */
GuiContext create_gui_context( int index )
{
  GuiContext gtx;
  static int first_time = 1;

  if (first_time) {
    int i;
    for (i=0;i<VIS5D_MAX_CONTEXTS;i++) {
      gtx_table[i] = NULL;
    }
    first_time = 0;
  }

  if (gtx_table[index]) {
    destroy_gui_context( gtx_table[index] );
    gtx_table[index] = NULL;
  }

  gtx = gtx_table[index] = (GuiContext) calloc(1, sizeof(struct gui_context));
  if (gtx) {
    /* init */
    memset( gtx, 0, sizeof(struct gui_context) );
    gtx->context_index = index;
  }
  return gtx;
}




Display *GuiDpy;          /* The X display for the GUI */
int GuiScr;               /* The X screen number for the GUI */
Visual *GuiVisual;        /* The X visual for the GUI */
Colormap GuiColormap;     /* The X colormap for the GUI */
int GuiDepth;             /* The depth of the GUI visual */


#define CP_WIDTH  384
#define BW 6
#define BUTTONSIZE 93   /* Width of regular buttons */
#define RADIOSIZE 120   /* Width of radio buttons */

/* depends on window border size: */
#ifdef stellar
#  define SPACE 15
#else
#  define SPACE 40
#endif


/*
 * List of external functions and expressions per context.
 */
static int NumFuncs[VIS5D_MAX_CONTEXTS];
static int FuncType[VIS5D_MAX_CONTEXTS][MAX_FUNCS];
static char FuncName[VIS5D_MAX_CONTEXTS][MAX_FUNCS][1000];


static char *Copyright[] = {
   " Vis5D version 4.3  Copyright (C) 1990 - 1997 ",
   "    Bill Hibbard, Johan Kellum, Brian Paul    ",
   "              and Andre Battaiola             "
/*
   "              Vis5D version 4.3               ",
   "           Copyright (C) 1990 - 1996          ",
   "  Bill Hibbard, Brian Paul & Andre Battaiola  "
*/
};

static char *ModeInfo1[] = {
    " Change the Viewing Angle ",
    "                          ",
    "      Mouse Buttons       ",
    "--------------------------",
    " rotate | zoom & | trans- ",
    "  view  |  clip  |  late  "
};

static char *ModeInfo2[] = {
    " Make & View Trajectories ",
    "                          ",
    "       Mouse Buttons      ",
    "--------------------------",
    " rotate |  make  |  move  ",
    "  view  | trajec | cursor "
};

static char *ModeInfo3[] = { 
    " Make & Edit Text Labels  ",
    "                          ",
    "       Mouse Buttons      ",
    "--------------------------",
    "  make  |  move  | delete ",
    "  label |  label | label  "
};

static char *ModeInfo4[] = {
    "  Change Slice Positions  ",
    "                          ",
    "       Mouse Buttons      ",
    "--------------------------",
    " rotate | zoom & | move   ",
    "  view  |  clip  | slice  "
};

static char *ModeInfo5[] = {
    "  Probe Data With Cursor  ",
    "                          ",
    "       Mouse Buttons      ",
    "--------------------------",
    " rotate | zoom & | move   ",
    "  view  |  clip  | cursor "
};



static char v5dfile[1000];


static int new_function_cb( LUI_NEWBUTTON *pb, int state );



static void init_var_colortable( GuiContext gtx, int var )
{
   int index = gtx->context_index;
   unsigned int *table;
   float *p;

   /* Isosurfaces */
   vis5d_get_color_table_params( index, VIS5D_ISOSURF, var, &p);
   vis5d_get_color_table_address( index, VIS5D_ISOSURF, var, &table );
   vis5d_color_table_init_params( p, 1, 1 );
   vis5d_color_table_set_alpha( p, 255 );
   vis5d_color_table_recompute( table, 256, p, 1, 1 );

   /* CHSlices */
   vis5d_get_color_table_params( index, VIS5D_CHSLICE, var, &p);
   vis5d_get_color_table_address( index, VIS5D_CHSLICE, var, &table );
   vis5d_color_table_init_params( p, 1, 1 );
   vis5d_color_table_set_alpha( p, 255 );
   vis5d_color_table_recompute( table, 256, p, 1, 1 );

   /* CVSlices */
   vis5d_get_color_table_params( index, VIS5D_CVSLICE, var, &p);
   vis5d_get_color_table_address( index, VIS5D_CVSLICE, var, &table );
   vis5d_color_table_init_params( p, 1, 1 );
   vis5d_color_table_set_alpha( p, 255 );
   vis5d_color_table_recompute( table, 256, p, 1, 1 );

   /* Volumes */
   vis5d_get_color_table_params( index, VIS5D_VOLUME, var, &p);
   vis5d_get_color_table_address( index, VIS5D_VOLUME, var, &table );
   vis5d_color_table_init_params( p, 1, 1 );
   vis5d_color_table_set_alpha( p, -1 );
   vis5d_color_table_recompute( table, 256, p, 1, 1 );

   /* Trajectories */
   vis5d_get_color_table_params( index, VIS5D_TRAJ, var, &p);
   vis5d_get_color_table_address( index, VIS5D_TRAJ, var, &table );
   vis5d_color_table_init_params( p, 1, 1 );
   vis5d_color_table_set_alpha( p, 255 );
   vis5d_color_table_recompute( table, 256, p, 1, 1 );

   /* Topography */
   vis5d_get_color_table_params( index, VIS5D_TOPO, var, &p);
   vis5d_get_color_table_address( index, VIS5D_TOPO, var, &table );
   vis5d_color_table_init_params( p, 1, 1 );
   vis5d_color_table_set_alpha( p, 255 );
   vis5d_color_table_recompute( table, 256, p, 1, 1 );
}



static void init_colortables( GuiContext gtx )
{
   int index = gtx->context_index;
   int var, numvars;
   float *p;

   vis5d_get_numvars( index, &numvars );
   for (var=0;var<numvars;var++) {
      init_var_colortable( gtx, var );
   }

   /* Topography height color table */
   {
      unsigned int *table;
      vis5d_get_color_table_params( index, VIS5D_TOPO, -1, &p);
      vis5d_get_color_table_address( index, VIS5D_TOPO, -1, &table );
      vis5d_color_table_init_params( p, 1, 1 );
      vis5d_color_table_set_alpha( p, 255 );
      vis5d_color_table_recompute( table, 256, p, 1, 1 );
      vis5d_reset_topo_colors( index );
   }
}



/*******************************************************/
/*** CALL BACK FUNCTIONS FOR VERIFY AND ALERT WINDOW ***/
/*******************************************************/

static int verify_value;


static int ok_cb( LUI_NEWBUTTON *pb )
{
  int index = pb->context_index;
  GuiContext gtx = get_gui_gtx(index);
  verify_value = 1;
  return 0;
}


static int cancel_cb( LUI_NEWBUTTON *pb )
{
  int index = pb->context_index;
  GuiContext gtx = get_gui_gtx(index);
  verify_value = 0;
  return 0;
}


/*** verify ***********************************************************
   Open the verification window with a prompt asking the user to
   specify either an OK or CANCEL.
   Input:  prompt - a string which asks the user what he wants to do.
                    Ex: "Do you want to exit?"
   Returned:  1 = OK selected
              0 = CANCEL selected
**********************************************************************/
int verify( int index, char *prompt )
{
   XEvent pe;
   GuiContext gtx = get_gui_gtx(index);

   LUI_NewLabelChangeText( gtx->VerifyLabel, prompt );

   verify_value = -1;
   XMapWindow( GuiDpy, gtx->VerifyWindow );

   while (verify_value < 0) {
      if (XPending(GuiDpy)) {
         XNextEvent(GuiDpy, &pe);
         LUI_EventDispatch(&pe);
      }
   }
   XUnmapWindow( GuiDpy, gtx->VerifyWindow );
   XSync( GuiDpy, 0 );
   vis5d_signal_redraw(index, 1);
   return(verify_value);
}



/*** alert ***********************************************************
   Open the alert window with a message and wait until the user clicks
   on "OK".
   Input:  prompt - a message string
**********************************************************************/
int alert( int index, char *prompt )
{
   XEvent pe;
   GuiContext gtx = get_gui_gtx(index);

   LUI_NewLabelChangeText( gtx->AlertLabel, prompt );

   verify_value = -1;
   XMapWindow( GuiDpy, gtx->AlertWindow );

   while (verify_value < 0) {
      if (XPending(GuiDpy)) {
         XNextEvent(GuiDpy, &pe);
         LUI_EventDispatch(&pe);
      }
   }
   XUnmapWindow( GuiDpy, gtx->AlertWindow );
   vis5d_signal_redraw(index, 1);
   return 1;
}




/**********************************************************************/
/*****                   Widget callback functions                *****/
/**********************************************************************/


#ifdef LEAVEOUT
static int debug_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
/*   Debug = !Debug;*/
   return 0;
}
#endif


/* toggle trajectory ribbon button */
static int ribbon_cb( LUI_NEWBUTTON *pb )
{
   float UserTrajStep, UserTrajLength;
   int RibbonFlag;
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   vis5d_get_traj(index, &UserTrajStep, &UserTrajLength, &RibbonFlag);
   RibbonFlag = !RibbonFlag;
   vis5d_set_traj(index, UserTrajStep, UserTrajLength, RibbonFlag);
   vis5d_signal_redraw(index, 1);
   return 0;
}


/* delete last trajectory */
static int dellast_cb( LUI_NEWBUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   vis5d_delete_last_traj(index);
   vis5d_invalidate_frames(index);
   return 0;
}


/* delete trajectory set */
static int delset_cb( LUI_NEWBUTTON *pb )
{
   char msg[100];
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   sprintf(msg,"Delete trajectory set %d?", gtx->cur_trajset+1);
   if (verify(index, msg)) {
      vis5d_delete_traj_set(index, gtx->cur_trajset);
   }
   vis5d_invalidate_frames(index);
   return 0;
}



/* called when trajectory step type-in changes */
static int trajstep_cb( LUI_FIELD *field )
{
   float UserTrajStep, UserTrajLength;
   int RibbonFlag;
   int index = field->context_index;
   GuiContext gtx = get_gui_gtx(index);
   vis5d_get_traj(index, &UserTrajStep, &UserTrajLength, &RibbonFlag);
   UserTrajStep = LUI_FieldGetDouble( gtx->TrajStepField );
   if (UserTrajStep<=0.0) {
      UserTrajStep = 1.0;
      LUI_FieldSetDouble( gtx->TrajStepField, 1.0 );
   }
   vis5d_set_traj(index, UserTrajStep, UserTrajLength, RibbonFlag);
   return 0;
}



/* called when trajectory length type-in changes */
static int trajlength_cb( LUI_FIELD *field )
{
   float UserTrajStep, UserTrajLength;
   int RibbonFlag;
   int index = field->context_index;
   GuiContext gtx = get_gui_gtx(index);
   vis5d_get_traj(index, &UserTrajStep, &UserTrajLength, &RibbonFlag);
   UserTrajLength = LUI_FieldGetDouble( gtx->TrajLenField );
   if (UserTrajLength<=0.0) {
      UserTrajLength = 1.0;
      LUI_FieldSetDouble( gtx->TrajLenField, 1.0 );
   }
   vis5d_set_traj(index, UserTrajStep, UserTrajLength, RibbonFlag);
   return 0;
}



/*
 * Display the expression type-in window with the given expression shown.
 */
static void map_expr_window( int index, char *expr )
{
   GuiContext gtx = get_gui_gtx(index);
   strcpy( gtx->Expression, expr );
   LUI_FieldSetText( gtx->expr_field, expr );
   XMapWindow( GuiDpy, gtx->ExprWindow );
}


/*** add_button_row ***************************************************
   Add a new row of variable buttons to the control panel window.
   Input:  name - name of new variable
           newrow - which row of buttons this is to be.
**********************************************************************/
static void add_button_row( int index, char *name, int newrow )
{
   float red[6], green[6], blue[6];
   char *labels[6];
   int j;
   int Nr, Nc, Nl[MAXVARS], LowLev[MAXVARS], MaxNl, MaxNlVar, WindNl, WindLow;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_size(index, &Nr, &Nc, Nl, LowLev, &MaxNl, &MaxNlVar, &WindNl, &WindLow);

   for (j=0; j<gtx->Columns; j++) {
      if (j==5) {
         /* volume button always white */
         red[j] = green[j] = blue[j] = 1.0;
      }
      else {
         float alpha;
         vis5d_get_color( index, j, newrow, &red[j], &green[j], &blue[j],
                          &alpha );
      }
      if (Nl[newrow]>1 || (j==1 || j==3)) {
         labels[j] = name;
      }
      else {
         labels[j] = NULL;
      }
   }
   LUI_ContextIndex(index);
   LUI_ButtonMatrixAddRow( gtx->ButtonMatrix, labels, red, green, blue );
   LUI_ButtonMatrixShowBottom( gtx->ButtonMatrix );
}


/* called when a button in the clone window is selected */
static int do_clone_cb( LUI_NEWBUTTON *b, int state )
{
  int index = b->context_index;
  int newvar;
  int numvars;
  GuiContext gtx = get_gui_gtx(index);

  vis5d_get_numvars( index, &numvars );

  if (b->index==9999) {
     /* Cancel button */
  }
  else if (b->index>=0) {
     /* clone an existing variable */
     int var_to_clone = b->index;
     char newname[100];
     int i;

     /* make name for new variable (original name plus quotes (')) */
     vis5d_get_var_name( index, var_to_clone, newname );
     for (i=0;i<MAXVARS;i++) {
        strcat( newname, "'" );
        if (vis5d_find_var( index, newname )<0) {
           break;
        }
     }
     if (i==MAXVARS ||
         vis5d_make_clone_variable(index, var_to_clone, newname, &newvar)<0) {
        /* error, unable to make new variable */
     }
     else {
        /* add a row of buttons to the control panel */
        add_button_row(index, newname, newvar);
        /* initialize graphics for new variable */
        init_var_colortable( gtx, newvar );
        vis5d_reset_var_graphics(index, newvar);
     }
  }
  else if (b->index<=0) {
    /* compute an external function variable */
    int func = -(b->index+1);
    if (FuncType[index][func]==VIS5D_EXT_FUNC) {
      char funcname[1000];
      if (gtx->funcpath[0]) {
         strcpy( funcname, gtx->funcpath );
      }
      else {
         strcpy( funcname, FUNCTION_PATH );
      }
      strcat( funcname, "/" );
      strcat( funcname, FuncName[index][func] );
      if (vis5d_compute_ext_func( index, funcname, &newvar) < 0 ) {
        /* error */
        char str[1000];
        sprintf(str, "External function \"%s\" failed", funcname );
        alert( index, str );
      }
      else {
        /* success */
        if (newvar >= 0) {
           char newname[10];
           vis5d_get_var_name( index, newvar, newname );
           add_button_row(index, newname, newvar);
           init_var_colortable( gtx, newvar );
           vis5d_reset_var_graphics(index, newvar);
        }
      }
    }
    else if (FuncType[index][func]==VIS5D_EXPRESSION) {
      /* Recompute a variable defined by an expression */
      int var, flag;
      flag = -1;
      for (var=0;var<numvars;var++) {
         char varname[10];
         vis5d_get_var_name( index, var, varname );
         if (strcmp(varname, FuncName[index][func])==0) {
            char expression[1000];
            vis5d_get_var_info( index, var, (void *) expression );
            map_expr_window( index, expression );
            flag = var;
         }
      }
      if (flag == -1) {
        printf("Error in do_clone_cb, var not found\n");
        return 0;
      }
    }
    else {
      printf("Error in do_clone_cb\n");
    }
  }

  XUnmapWindow( GuiDpy, gtx->CloneWindow );

  vis5d_signal_redraw(index, 1);
  return 0;
}




/*
 * Scan the list of variables for the type VIS5D_EXPRESSION.  Put the
 * names of the expressions into the exprname array.
 * Return:  number of expression variables
 */
static int find_expr_variables( int index, char exprname[][1000] )
{
   int var, numvars, vartype, num;

   vis5d_get_numvars( index, &numvars );
   num = 0;

   for (var=0;var<numvars;var++) {
      vis5d_get_var_type( index, var, &vartype );
      if (vartype==VIS5D_EXPRESSION) {
         vis5d_get_var_name( index, var, exprname[num] );
         num++;
      }
   }
   return num;
}



static void unmap_expr_window( int index )
{
   GuiContext gtx = get_gui_gtx(index);
   XUnmapWindow( GuiDpy, gtx->ExprWindow );
}



/* map the horizontal wind slice control window */
static void map_hwind_window( int index, int ws )
{
   float HWindDensity, HWindScale, HWindLevel;
   char str[1000];
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_hwindslice(index, ws, &HWindDensity, &HWindScale, &HWindLevel);

   gtx->cur_hwind = ws;

   XUnmapWindow( GuiDpy, gtx->WindWindow );
   sprintf( str, "H-Wind %d Scale:", ws+1 );
   LUI_NewLabelChangeText( gtx->windscale_label, str );
   LUI_FieldSetDouble( gtx->windscale_field, HWindScale );
   LUI_FieldSetDouble( gtx->winddensity_field, HWindDensity );

   XMapWindow( GuiDpy, gtx->WindWindow );
}


/* map the vertical wind slice control window */
static void map_vwind_window( int index, int ws )
{
   float VWindDensity, VWindScale;
   float row0, col0, row1, col1;
   char str[1000];
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_vwindslice(index, ws, &VWindDensity, &VWindScale,
                        &row0, &col0, &row1, &col1);

   gtx->cur_vwind = ws;

   XUnmapWindow( GuiDpy, gtx->WindWindow );
   sprintf( str, "V-Wind %d Scale:", ws+1 );
   LUI_NewLabelChangeText( gtx->windscale_label, str );
   LUI_FieldSetDouble( gtx->windscale_field, VWindScale );
   LUI_FieldSetDouble( gtx->winddensity_field, VWindDensity );

   XMapWindow( GuiDpy, gtx->WindWindow );
}


/* map the horizontal stream slice control window */
static void map_hstream_window( int index, int ws )
{
   float HStreamDensity, HStreamScale, HStreamLevel;
   char str[1000];
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_hstreamslice(index, ws, &HStreamDensity, &HStreamLevel);
   HStreamScale = 1.0;

   gtx->cur_hstream = ws;

   XUnmapWindow( GuiDpy, gtx->WindWindow );
   sprintf( str, "HStream %d Scale:", ws+1 );
   LUI_NewLabelChangeText( gtx->windscale_label, str );
   LUI_FieldSetDouble( gtx->windscale_field, HStreamScale );
   LUI_FieldSetDouble( gtx->winddensity_field, HStreamDensity );

   XMapWindow( GuiDpy, gtx->WindWindow );
}


/* map the vertical stream slice control window */
static void map_vstream_window( int index, int ws )
{
   float VStreamDensity, VStreamScale;
   float VStreamR1, VStreamC1, VStreamR2, VStreamC2;
   char str[1000];
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_vstreamslice(index, ws, &VStreamDensity,
                          &VStreamR1, &VStreamC1, &VStreamR2, &VStreamC2);
   VStreamScale = 1.0;

   gtx->cur_vstream = ws;

   XUnmapWindow( GuiDpy, gtx->WindWindow );
   sprintf( str, "VStream %d Scale:", ws+1 );
   LUI_NewLabelChangeText( gtx->windscale_label, str );
   LUI_FieldSetDouble( gtx->windscale_field, VStreamScale );
   LUI_FieldSetDouble( gtx->winddensity_field, VStreamDensity );

   XMapWindow( GuiDpy, gtx->WindWindow );
}


/* map the isosurface isolevel window */
static void map_isosurf_window( int index, int var )
{
   float min, max, isolevel;
   char name[10];
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_var_range( index, var, &min, &max );
   if (min > max ) {
      min = max = 0.0;
      isolevel = 0.0;
   }
   else {
      vis5d_get_isosurface(index, var, &isolevel);
   }
   vis5d_get_var_name( index, var, name );
   LUI_NewSliderChange( gtx->IsoSlider, name, min, max, isolevel );
   XMapWindow(GuiDpy, gtx->IsoWindow);
}


/* map the horizontal contour line slice control window */
static void map_hslice_window( int index, int var )
{
   char str[1000];
   float min, max;
   char name[10];
   float HSliceInterval, HSliceLowLimit, HSliceHighLimit, HSliceLevel;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_var_range(index, var, &min, &max);
   vis5d_get_var_name(index, var, name );
   vis5d_get_hslice(index, var, &HSliceInterval,
                    &HSliceLowLimit, &HSliceHighLimit, &HSliceLevel);

   sprintf( str, "%8s Hor. Slice Interval:", name );
   LUI_NewLabelChangeText( gtx->hslice_label, str );
   if (HSliceLowLimit==min && HSliceHighLimit==max)
     sprintf( str, "%g", HSliceInterval );
   else
     sprintf( str, "%g (%g,%g)", HSliceInterval, HSliceLowLimit, HSliceHighLimit );
   LUI_FieldSetText( gtx->hslice_field, str );

   XMapWindow(GuiDpy,gtx->HSliceWindow);
}



/* map the vertical contour line slice control window */
static void map_vslice_window( int index, int var )
{
   char str[1000];
   float min, max;
   char name[10];
   float VSliceInterval, VSliceLowLimit, VSliceHighLimit;
   float row0, col0, row1, col1;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_var_range(index, var, &min, &max);
   vis5d_get_var_name( index, var, name );
   vis5d_get_vslice(index, var, &VSliceInterval, &VSliceLowLimit,
                    &VSliceHighLimit, &row0, &col0, &row1, &col1);

   sprintf( str, "%8s Ver. Slice Interval:", name );
   LUI_NewLabelChangeText( gtx->vslice_label, str );
   if (VSliceLowLimit==min && VSliceHighLimit==max)
     sprintf( str, "%g", VSliceInterval );
   else
     sprintf( str, "%g (%g,%g)", VSliceInterval,
             VSliceLowLimit, VSliceHighLimit );
   LUI_FieldSetText( gtx->vslice_field, str );

   XMapWindow(GuiDpy,gtx->VSliceWindow);
}





static int cb_graphic = -1;
static int cb_var = -1;

/*
 * Display a color_table widget window.
 * Input:  index - context index
 *         graphic - either VIS5D_TOPO, VIS5D_CHSLICE, VIS5D_CVSLICE,
 *                   or VIS5D_VOLUME
 *         var - which variable
 */
static void show_colorbar_window( int index, int graphic, int var )
{
   GuiContext gtx = get_gui_gtx(index);
   float min, max;
   unsigned int *colors;
   char name[20], label[100];
   float *p;

   if (cb_graphic>=0 && cb_var>=0) {
      LUI_ColorBarHide( gtx->Colorbar );
   }

   if (graphic==VIS5D_TOPO) {
      vis5d_get_color_table_address( index, VIS5D_TOPO, 0, &colors );
      vis5d_get_color_table_params(index, VIS5D_TOPO, 0, &p);
      vis5d_get_topo_range( index, &min, &max );
      LUI_ColorBarChange( gtx->Colorbar, "Topography", min, max,
                          colors, 256, p );
   }
   else if (graphic==VIS5D_CHSLICE) {
      vis5d_get_color_table_address( index, VIS5D_CHSLICE, var, &colors );
      vis5d_get_color_table_params(index, VIS5D_CHSLICE, var, &p);
      vis5d_get_var_range( index, var, &min, &max );
      vis5d_get_var_name( index, var, name );
      sprintf( label, "%s horiz slice", name );
      LUI_ColorBarChange( gtx->Colorbar, label, min, max, colors, 255, p);
   }
   else if (graphic==VIS5D_CVSLICE) {
      vis5d_get_color_table_address( index, VIS5D_CVSLICE, var, &colors );
      vis5d_get_color_table_params(index, VIS5D_CVSLICE, var, &p);
      vis5d_get_var_range( index, var, &min, &max );
      vis5d_get_var_name( index, var, name );
      sprintf( label, "%s vert slice", name );
      LUI_ColorBarChange( gtx->Colorbar, label, min, max, colors, 255, p);
   }
   else if (graphic==VIS5D_VOLUME) {
      vis5d_get_color_table_address( index, VIS5D_VOLUME, var, &colors );
      vis5d_get_color_table_params(index, VIS5D_VOLUME, var, &p);
      vis5d_get_var_range( index, var, &min, &max );
      vis5d_get_var_name( index, var, name );
      sprintf( label, "%s volume", name );
      LUI_ColorBarChange( gtx->Colorbar, label, min, max, colors, 255, p);
   }

   cb_graphic = graphic;
   cb_var = var;
   LUI_ColorBarShow( gtx->Colorbar );
}



static void hide_colorbar_window( int index )
{
   GuiContext gtx = get_gui_gtx(index);

   LUI_ColorBarHide( gtx->Colorbar );
   cb_graphic = -1;
   cb_var = -1;
}


/*
 * Set the alpha value for all color table entries for a colored slice.
 */
void set_slice_alpha( int index, int graphic, int var, int alpha )
{
   GuiContext gtx = get_gui_gtx(index);
   unsigned int *colors;
   float *params;

   if (graphic==VIS5D_CHSLICE) {
      vis5d_get_color_table_address( index, VIS5D_CHSLICE, var, &colors );
      vis5d_get_color_table_params(index, VIS5D_CHSLICE, var, &params);
   }
   else if (graphic==VIS5D_CVSLICE) {
      vis5d_get_color_table_address( index, VIS5D_CVSLICE, var, &colors );
      vis5d_get_color_table_params(index, VIS5D_CVSLICE, var, &params);
   }

   vis5d_color_table_set_alpha( params, alpha );
   vis5d_color_table_recompute( colors, 256, params, 0, 1 );

   if (cb_graphic==graphic && cb_var==var) {
      /* Change alpha of displayed colorbar widget */
      char label[100], name[20];
      float min, max;
      if (graphic==VIS5D_CHSLICE) {
         vis5d_get_var_name( index, var, name );
         sprintf( label, "%s horiz slice", name );
      }
      else {
         vis5d_get_var_name( index, var, name );
         sprintf( label, "%s vert slice", name );
      }
      vis5d_get_var_range( index, var, &min, &max );

      LUI_ColorBarChange( gtx->Colorbar, label, min, max, colors, 255, params);
   }
}




/************************************************************/
/* Create the window save requester.                        */
/************************************************************/
static void make_savepic_window( int index )
{
   GuiContext gtx = get_gui_gtx(index);
   LUI_NEWBUTTON *b;
   char **labels;
   int n, formats;

   formats = vis5d_get_image_formats();

   gtx->SavePicWindow = LUI_CreateWindowAt( LUI_RootWindow, 50, 250, 290,222 );

   LUI_NewLabelCreate( gtx->SavePicWindow, LUI_LEFT, LUI_TOP, 280, 30,
                       "Save Window Image" );

   /* allocate array of 6 char pointers */
   labels = (char **) malloc( 6 * sizeof(char *) );

   /* make labels for radio button and update SaveFormats array */
   n = 0;
   if (formats & VIS5D_RGB) {
      labels[n] = strdup("SGI .rgb");
      gtx->SaveFormats[n] = VIS5D_RGB;
      n++;
   }
   if (formats & VIS5D_GIF) {
      labels[n] = strdup("GIF");
      gtx->SaveFormats[n] = VIS5D_GIF;
      n++;
   }
   if (formats & VIS5D_PS) {
      labels[n] = strdup("PostScript");
      gtx->SaveFormats[n] = VIS5D_PS;
      n++;
   }
   if (formats & VIS5D_COLOR_PS) {
      labels[n] = strdup("Color PostScript");
      gtx->SaveFormats[n] = VIS5D_COLOR_PS;
      n++;
   }
   if (formats & VIS5D_XWD) {
      labels[n] = strdup("xwd (X Window Dump)");
      gtx->SaveFormats[n] = VIS5D_XWD;
      n++;
   }

   gtx->SavePicRadio = LUI_RadioCreate( gtx->SavePicWindow,
                                        LUI_LEFT, LUI_NEXT_Y, 280,
                                        n, labels );

   LUI_NewLabelCreate( gtx->SavePicWindow, LUI_LEFT, LUI_NEXT_Y, 280, 30,
                     "File name: " );

   gtx->SavePicField = LUI_FieldCreate( gtx->SavePicWindow,
                                   LUI_LEFT, LUI_NEXT_Y, 280, 26 );

   b = LUI_PushButtonCreate( gtx->SavePicWindow, 30, LUI_NEXT_Y,
                             100, 26, "Save" );
   LUI_ButtonCallback( b, ok_cb );

   b = LUI_PushButtonCreate( gtx->SavePicWindow, 160, LUI_SAME_Y,
                             100, 26, "Cancel" );
   LUI_ButtonCallback( b, cancel_cb );

   XUnmapWindow( GuiDpy, gtx->SavePicWindow );
}




/*
 * This is called after we execute a Tcl script, do a RESTORE, or compute
 * a new type-in expression.  It updates the color and on/off state of
 * all GUI buttons.
 */
static void reset_widgets( int index )
{
   int i, j, time, var, ws, numvars;
   GuiContext gtx = get_gui_gtx(index);
   float r, g, b, a;

   vis5d_get_numvars( index, &numvars );
   if (gtx->cur_isosurf>=0) {
      XUnmapWindow( GuiDpy, gtx->IsoWindow );
      map_isosurf_window( index, gtx->cur_isosurf );
   }

   if (gtx->cur_hslice>=0) {
      XUnmapWindow( GuiDpy, gtx->HSliceWindow );
      map_hslice_window( index, gtx->cur_hslice );
   }

   if (gtx->cur_vslice>=0) {
      XUnmapWindow( GuiDpy, gtx->VSliceWindow );
      map_vslice_window( index, gtx->cur_vslice );
   }

   if (cb_graphic>=0 && cb_var>=0) {
      show_colorbar_window( index, cb_graphic, cb_var );
   }

   if (gtx->cur_hwind>=0 || gtx->cur_vwind>=0 ||
       gtx->cur_hstream>=0 || gtx->cur_vstream>=0) {
      XUnmapWindow( GuiDpy, gtx->WindWindow );
      if (gtx->cur_hwind>=0) {
        map_hwind_window( index, gtx->cur_hwind );
      }
      else if (gtx->cur_vwind>=0) {
        map_vwind_window( index, gtx->cur_vwind );
      }
      else if (gtx->cur_hstream>=0) {
        map_hstream_window( index, gtx->cur_hstream );
      }
      else {
        map_vstream_window( index, gtx->cur_vstream );
      }
   }

   refresh_isocolor_window( gtx );
   refresh_rgb_sliders( gtx );

   /* redraw all buttons with restored colors */
   for (i=0;i<numvars;i++) {
      for (j=0;j<gtx->Columns;j++) {
         float r, g, b, a;
         int what;
         switch (j) {
            case 0:  what = VIS5D_ISOSURF;  break;
            case 1:  what = VIS5D_HSLICE;   break;
            case 2:  what = VIS5D_VSLICE;   break;
            case 3:  what = VIS5D_CHSLICE;  break;
            case 4:  what = VIS5D_CVSLICE;  break;
            case 5:  what = VIS5D_VOLUME;   break;
            default:
               /* better never happen! */
               abort();
         }
         vis5d_get_color( index, what, i, &r, &g, &b, &a );
         LUI_ButtonMatrixSetColor( gtx->ButtonMatrix, i, what, r, g, b );
         if (vis5d_enable_graphics( index, what, i, VIS5D_GET )) {
            LUI_ButtonMatrixSetState( gtx->ButtonMatrix, i, j, 1 );
         }
         else {
            LUI_ButtonMatrixSetState( gtx->ButtonMatrix, i, j, 0 );
         }
      }
   }

   if (gtx->perspec_button != NULL) {
     int PerspecFlag;
     PerspecFlag = vis5d_graphics_mode(index, VIS5D_PERSPECTIVE, VIS5D_GET);
     LUI_ButtonState( gtx->perspec_button, PerspecFlag);
   }

   /* trajectory colors */
   for (i=0;i<VIS5D_TRAJ_SETS;i++) {
      vis5d_get_color( index, VIS5D_TRAJ, i, &r, &g, &b, &a );
      LUI_ButtonSetColor( gtx->TrajButton[i], (int)(r*255.0), (int)(g*255.0),
                         (int)(b*255.0) );
   }

   {
     float UserTrajStep, UserTrajLength;
     int RibbonFlag;
     vis5d_get_traj(index, &UserTrajStep, &UserTrajLength, &RibbonFlag);
     LUI_ButtonSetState( gtx->TrajRibbonButton, RibbonFlag);
     LUI_FieldSetDouble( gtx->TrajStepField, UserTrajStep );
     LUI_FieldSetDouble( gtx->TrajLenField, UserTrajLength );
   }

   /* update wind slice buttons */
   for (ws=0;ws<VIS5D_WIND_SLICES;ws++) {
      /* color */
      vis5d_get_color( index, VIS5D_HWIND, ws, &r, &g, &b, &a );
      LUI_ButtonColor( gtx->hwind_button[ws], r, g, b );
      vis5d_get_color( index, VIS5D_VWIND, ws, &r, &g, &b, &a );
      LUI_ButtonColor( gtx->vwind_button[ws], r, g, b );
      /* on/off status */
      if (vis5d_enable_graphics(index, VIS5D_HWIND, ws, VIS5D_GET)) {
         LUI_ButtonSetState( gtx->hwind_button[ws], 1 );
      }
      else {
         LUI_ButtonSetState( gtx->hwind_button[ws], 0 );
      }
      if (vis5d_enable_graphics(index, VIS5D_VWIND, ws, VIS5D_GET)) {
         LUI_ButtonSetState( gtx->vwind_button[ws], 1 );
         }
      else {
         LUI_ButtonSetState( gtx->vwind_button[ws], 0 );
      }
   }
   /* update stream slice buttons */
   ws = 0;
   vis5d_get_color( index, VIS5D_HSTREAM, ws, &r, &g, &b, &a );
   LUI_ButtonColor( gtx->hstream_button[ws], r, g, b );
   vis5d_get_color( index, VIS5D_VSTREAM, ws, &r, &g, &b, &a );
   LUI_ButtonColor( gtx->vstream_button[ws], r, g, b );
   if (vis5d_enable_graphics(index, VIS5D_HSTREAM, ws, VIS5D_GET)) {
     LUI_ButtonSetState( gtx->hstream_button[ws], 1 );
   }
   else {
     LUI_ButtonSetState( gtx->hstream_button[ws], 0 );
   }
   if (vis5d_enable_graphics(index, VIS5D_VSTREAM, ws, VIS5D_GET)) {
     LUI_ButtonSetState( gtx->vstream_button[ws], 1 );
   }
   else {
     LUI_ButtonSetState( gtx->vstream_button[ws], 0 );
   }
}


/*
 * This is called after we execute a Tcl script, do a RESTORE, or compute
 * a new type-in expression.  It recomputes any graphics whose parameters
 * may have been changed by the script.
 */
static void recompute_graphics( int index )
{
   int numvars, numtimes, curtime, var, ws, time;

   vis5d_get_numvars( index, &numvars );
   vis5d_get_numtimes( index, &numtimes );

   /* update isosurfaces and slices */
   for (var=0;var<numvars;var++) {
      if (vis5d_enable_graphics(index, VIS5D_HSLICE, var, VIS5D_GET)) {
         for (time=0;time<numtimes;time++) {
            vis5d_make_hslice(index, time, var, 0);
         }
      }
      if (vis5d_enable_graphics(index, VIS5D_VSLICE, var, VIS5D_GET)) {
         for (time=0;time<numtimes;time++) {
            vis5d_make_vslice(index, time, var, 0);
         }
      }
      if (vis5d_enable_graphics(index, VIS5D_CHSLICE, var, VIS5D_GET)) {
         for (time=0;time<numtimes;time++) {
            vis5d_make_chslice(index, time, var, 0);
         }
      }
      if (vis5d_enable_graphics(index, VIS5D_CVSLICE, var, VIS5D_GET)) {
         for (time=0;time<numtimes;time++) {
            vis5d_make_cvslice(index, time, var, 0);
         }
      }
   }

   /* update wind and stream slices */
   for (ws=0;ws<VIS5D_WIND_SLICES;ws++) {
      if (vis5d_enable_graphics(index, VIS5D_HWIND, ws, VIS5D_GET)) {
         for (time=0;time<numtimes;time++) {
            vis5d_make_hwindslice(index, time, ws, 0);
         }
      }
      if (vis5d_enable_graphics(index, VIS5D_VWIND, ws, VIS5D_GET)) {
         for (time=0;time<numtimes;time++) {
            vis5d_make_vwindslice(index, time, ws, 0);
         }
      }
      if (vis5d_enable_graphics(index, VIS5D_HSTREAM, ws, VIS5D_GET)) {
         for (time=0;time<numtimes;time++) {
            vis5d_make_hstreamslice(index, time, ws, 0);
         }
      }
   }
}



/* This is called by LUI when it wants to pass us an event */
static int controlpanel_cb( Window window, XEvent *event )
{
  int index;
  GuiContext gtx;

  switch (event->type) {
  case ConfigureNotify:
    for (index=0; index<VIS5D_MAX_CONTEXTS; index++) {
      if ((gtx = gtx_table[index]) != NULL &&
          event->xconfigure.window==gtx->CpWindow){
        int width, height;
        gtx->CpHeight = event->xconfigure.height;
        width = gtx->ButtonMatrixWidth;
        height = gtx->CpHeight - gtx->ButtonMatrixTop;
        if (height<50) {
          height = 50;
        }
        LUI_ButtonMatrixResize( gtx->ButtonMatrix, width, height );
        break;
      }
    }
    break;
  }
  return 0;
}



/****************************************************/
/*** CALL-BACK FUNCTIONS FOR MAIN CONTROL BUTTONS ***/
/****************************************************/


/*** anim_cb **********************************************************
   Animate button call-back function.  Toggle animation on/off.
   Input:  pb - LUI button pointer
**********************************************************************/
static int anim_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   if (gtx->GoTime)
      gtx->GoTime = 0;
   else {
      if (pb->event->xbutton.button == Button3)
         /* reverse animation */
          gtx->GoTime = -1;
      else
         /* forward animation */
          gtx->GoTime = 1;
   }

   vis5d_signal_redraw(index, 1);
   return 0;
}



/*
 * ANIM-REC callback.
 */
static int animrec_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   if (gtx->GoTime) {
      gtx->GoTime = 0;
      vis5d_graphics_mode( index, VIS5D_ANIMRECORD, VIS5D_OFF );
   }
   else {
      if (pb->event->xbutton.button == Button3)
         /* reverse animation */
          gtx->GoTime = -1;
      else
         /* forward animation */
          gtx->GoTime = 1;
      vis5d_graphics_mode(index, VIS5D_ANIMRECORD, VIS5D_ON );
   }

   vis5d_signal_redraw(index, 1);
   return 0;
}




/*** step_cb **********************************************************
   Step button call-back function.  Step forward or backward one time
   step or set time to first time step depending on which mouse
   button was pressed.
   Input:  pb - LUI button pointer
**********************************************************************/
static int step_cb( LUI_BUTTON *pb )
{
   int NumTimes;
   int CurTime;
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   int verylarge;

   vis5d_get_numtimes( index, &NumTimes );
   vis5d_get_timestep( index, &CurTime );
   verylarge = vis5d_verylarge_mode(index, VIS5D_GET);

   /* CurTime must be in [0..NumTimes-1] */

   if (gtx->GoTime == 0) {
      if (pb->event->xbutton.button == Button1) {
         if (CurTime==NumTimes-1)
            CurTime = 0;
         else
            CurTime++;
      }
      else if(pb->event->xbutton.button == Button2) {
         CurTime = 0;
      }
      else if(pb->event->xbutton.button == Button3) {
         if (CurTime==0)
            CurTime = NumTimes-1;
         else
            CurTime--;
      }

      if (verylarge) {
         /* make graphics */
         vis5d_make_timestep_graphics(index, CurTime);
         vis5d_finish_work();
      }


      vis5d_set_timestep(index, CurTime);
   }
   return 0;
}



/* callback for quit button */
static int exit_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   if (verify(index, "Do you really want to exit?")) {
      /* send quit message to server */
      /*request_quit( NULL );*/

      /* Destroy all GUI's */
      for (index=0;index<VIS5D_MAX_CONTEXTS;index++) {
         if (gtx_table[index]) {
            /*gui_cleanup( gtx_table[index] );*/
         }
      }

      vis5d_terminate(1);
      exit(0);
   }
   return 0;
}



static int box_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   /* toggle box rendering */
   vis5d_graphics_mode(index, VIS5D_BOX, VIS5D_TOGGLE);
   vis5d_invalidate_frames(index);
   return 0;
}


static int clock_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   /* toggle clock rendering */
   vis5d_graphics_mode(index, VIS5D_CLOCK, VIS5D_TOGGLE);
   vis5d_invalidate_frames(index);
   return 0;
}



/*
 * Called when user clicks on SAVE button.
 */
static int save_cb( LUI_BUTTON *pb )
{
   static char savefile[1000] = "";
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   if (savefile[0]==0) {
      /* set initial save file name */
      sprintf( savefile, "%s.SAVE", v5dfile );
      LUI_FieldSetText( gtx->SaveNameField, savefile );
   }
   XMapWindow( GuiDpy, gtx->SaveWindow );
   return 1;
}


/*
 * Called when user clicks on SAVE's OK button.
 */
static int save_ok_cb( LUI_NEWBUTTON *pb )
{
   char savefile[1000], str[1000];
   int result;
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   LUI_FieldGetText( gtx->SaveNameField, savefile );

   result = tcl_save( index, savefile );

   XUnmapWindow( GuiDpy, gtx->SaveWindow );

   if (result==0) {
      /* printf("Saved: %s\n", savefile ); */
      vis5d_signal_redraw(index, 1);
   }
   else if (result==1) {
      sprintf( str, "Error: Unable to open\n%s for writing", savefile );
      alert( index, str );
   }
   else if (result==2) {
      sprintf( str, "Error while writing %s\nCheck if disk is full.",
              savefile );
      alert( index, str );
   }
   return 0;
}



/*
 * Called when user clicks on SAVE's Cancel button.
 */
static int save_cancel_cb( LUI_NEWBUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   XUnmapWindow( GuiDpy, gtx->SaveWindow );
   return 0;
}  



/*
 * Called when user clicks on RESTORE button.
 */
static int restore_cb( LUI_BUTTON *pb )
{
   static char savefile[1000] = "";
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   if (savefile[0]==0) {
      /* set initial save file name */
      sprintf( savefile, "%s.SAVE", v5dfile );
      LUI_FieldSetText( gtx->RestoreNameField, savefile );
   }
   XMapWindow( GuiDpy, gtx->RestoreWindow );
   return 1;
}



/*
 * Called when user clicks on RESTORE button.
 */
static int restore_ok_cb( LUI_NEWBUTTON *pb )
{
   char savefile[100];
   int res;
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   int var, oldnumvars, numvars;

   LUI_FieldGetText( gtx->RestoreNameField, savefile );
   XUnmapWindow( GuiDpy, gtx->RestoreWindow );

   vis5d_get_numvars( index, &oldnumvars );

   /* Determine if restoring a binary save file or Tcl script. */
   {
      FILE *f;
      f = fopen(savefile,"r");
      if (!f) {
         res = VIS5D_BAD_VALUE;
      }
      else {
         char str[10];
         fgets( str, 3, f );
         if (str[0]=='#') {
            /* restore tcl file */
            if (execute_script( index, savefile )) {
               res = 0;
            }
            else {
               res = VIS5D_FAIL;
            }
         }
         else {
            /* restore old-style binary .SAVE file */
            res = vis5d_restore( index, savefile );
         }
      }
   }

   if (res==0) {
      /* may have to add more button rows to control panel now */
      vis5d_get_numvars( index, &numvars );
      for (var=oldnumvars;var<numvars;var++) {
         char varname[20];
         vis5d_get_var_name( index, var, varname );
         add_button_row( index, varname, var );
      }
      reset_widgets(index);
      recompute_graphics(index);
      vis5d_signal_redraw(index, 1);
   }
   else if (res==VIS5D_BAD_VALUE) {
      char str[1000];
      sprintf( str, "Error: unable to open\n%s for reading", savefile );
      alert( index, str );
   }
   else if (res==VIS5D_FAIL) {
      char str[1000];
      sprintf( str,
              "Error while reading %s\nFile may be corrupt or an old version",
              savefile );
      alert( index, str );
   }
   vis5d_invalidate_frames(index);
   return 0;
}



/*
 * Called when user clicks on RESTORE's Cancel button.
 */
static int restore_cancel_cb( LUI_NEWBUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   XUnmapWindow( GuiDpy, gtx->RestoreWindow );
   return 0;
}  



/* callback for topography button */
static int topo_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   if (pb->event->xbutton.button==Button1) {
      /* toggle display of topography */
      vis5d_graphics_mode(pb->context_index, VIS5D_TOPO, VIS5D_TOGGLE);
   }
   else if (pb->event->xbutton.button==Button2 ||
            pb->event->xbutton.button==Button3) {
      /* topo color widget */
      if (pb->state == LUI_BUTTON_OUT) {
         pb->state = LUI_BUTTON_IN;
      }
      else {
         pb->state = LUI_BUTTON_OUT;
      }
      show_isocolor_window( gtx, VIS5D_TOPO, 0 );
   }
   vis5d_invalidate_frames(index);
   return 0;
}



/* callback for map display toggle */
static int map_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   if (pb->event->xbutton.button==Button1) {
      /* toggle display */
      vis5d_graphics_mode(pb->context_index, VIS5D_MAP, VIS5D_TOGGLE);
   }
   else if (pb->event->xbutton.button==Button3) {
      /* set map color */
      if (pb->state == LUI_BUTTON_OUT)
         pb->state = LUI_BUTTON_IN;
      else
         pb->state = LUI_BUTTON_OUT;
      show_rgb_sliders( gtx, VIS5D_MAP, 0 );
   }

   return 0;
}




/* toggle perspective/parallel projection */
static int proj_cb( LUI_BUTTON *pb )
{
   vis5d_graphics_mode(pb->context_index, VIS5D_PERSPECTIVE, VIS5D_TOGGLE);
   return 0;
}



/* toggle contour label drawing */
static int contnum_cb( LUI_BUTTON *pb )
{
   vis5d_graphics_mode(pb->context_index, VIS5D_CONTOUR_NUMBERS, VIS5D_TOGGLE);
   return 0;
}



/* toggle between geographic and grid coordinate display */
static int coord_cb( LUI_BUTTON *pb )
{
   vis5d_graphics_mode(pb->context_index, VIS5D_GRID_COORDS, VIS5D_TOGGLE);
   return 0;
}


/* Called when user clicks on HWind button */
static int hwind_cb( LUI_NEWBUTTON *b, int state )
{
   GuiContext gtx = get_gui_gtx( b->context_index );
   int index = b->context_index;
   int slice = b->index;

   if (b->mousebutton==Button1) {
      /* toggle display on/off */
      if (state==1) {
         int time, numtimes, curtime;
         vis5d_enable_graphics(index, VIS5D_HWIND, slice, VIS5D_ON);
         vis5d_get_timestep(index, &curtime);
         vis5d_get_numtimes(index, &numtimes);
         gtx->cur_hwind = slice;
         map_hwind_window( index, gtx->cur_hwind );
         for (time=0;time<numtimes;time++) {
            vis5d_make_hwindslice(index, time, slice, time==curtime);
         }
         gtx->cur_vwind = -1;
         gtx->cur_hstream = -1;
         gtx->cur_vstream = -1;
      }
      else {
         XUnmapWindow(GuiDpy,gtx->WindWindow);
         vis5d_enable_graphics(index, VIS5D_HWIND, slice, VIS5D_OFF);
         gtx->cur_hwind = -1;
      }
   }
   else if (b->mousebutton==Button2) {
      /* show/hide pop-up control window */
      if (gtx->cur_hwind==slice) {
         XUnmapWindow(GuiDpy,gtx->WindWindow);
         gtx->cur_hwind = -1;
      }
      else {
         gtx->cur_hwind = slice;
         map_hwind_window( index, gtx->cur_hwind );
      }
      gtx->cur_vwind = -1;
      gtx->cur_hstream = -1;
      gtx->cur_vstream = -1;
   }
   else if (b->mousebutton==Button3) {
      show_rgb_sliders( gtx, VIS5D_HWIND, slice );
   }

   vis5d_invalidate_frames(index);
   return 0;
}



/* Called when user clicks on VWind button */
static int vwind_cb( LUI_NEWBUTTON *b, int state )
{
   GuiContext gtx = get_gui_gtx( b->context_index );
   int index = b->context_index;
   int slice = b->index;

   if (b->mousebutton==Button1) {
      /* toggle display on/off */
      if (state==1) {
         int time, numtimes, curtime;
         vis5d_enable_graphics(index, VIS5D_VWIND, slice, VIS5D_ON);
         vis5d_get_timestep(index, &curtime);
         vis5d_get_numtimes(index, &numtimes);
         gtx->cur_vwind = slice;
         map_vwind_window( index, gtx->cur_vwind );
         for (time=0;time<numtimes;time++) {
            vis5d_make_vwindslice(index, time, slice, time==curtime);
         }
         gtx->cur_hwind = -1;
         gtx->cur_hstream = -1;
         gtx->cur_vstream = -1;
      }
      else {
         XUnmapWindow(GuiDpy,gtx->WindWindow);
         vis5d_enable_graphics(index, VIS5D_VWIND, slice, VIS5D_OFF);
         gtx->cur_vwind = -1;
      }
   }
   else if (b->mousebutton==Button2) {
      /* show/hide pop-up control window */
      if (gtx->cur_vwind==slice) {
         XUnmapWindow(GuiDpy,gtx->WindWindow);
         gtx->cur_vwind = -1;
      }
      else {
         gtx->cur_vwind = slice;
         map_vwind_window( index, gtx->cur_vwind );
      }
      gtx->cur_hwind = -1;
      gtx->cur_hstream = -1;
      gtx->cur_vstream = -1;
   }
   else if (b->mousebutton==Button3) {
      show_rgb_sliders( gtx, VIS5D_VWIND, slice );
   }

   vis5d_invalidate_frames(index);
   return 0;
}



/* Called when user clicks on HStream button */
static int hstream_cb( LUI_NEWBUTTON *b, int state )
{
   GuiContext gtx = get_gui_gtx( b->context_index );
   int index = b->context_index;
   int slice = b->index;

   if (b->mousebutton==Button1) {
      /* toggle display on/off */
      if (state==1) {
         int time, numtimes, curtime;
         vis5d_enable_graphics(index, VIS5D_HSTREAM, slice, VIS5D_ON);
         vis5d_get_timestep(index, &curtime);
         vis5d_get_numtimes(index, &numtimes);
         gtx->cur_hstream = slice;
         map_hstream_window( index, gtx->cur_hstream );
         for (time=0;time<numtimes;time++) {
            vis5d_make_hstreamslice(index, time, slice, time==curtime);
         }
         gtx->cur_hwind = -1;
         gtx->cur_vwind = -1;
         gtx->cur_vstream = -1;
      }
      else {
         XUnmapWindow(GuiDpy,gtx->WindWindow);
         vis5d_enable_graphics(index, VIS5D_HSTREAM, slice, VIS5D_OFF);
         gtx->cur_hstream = -1;
      }
   }
   else if (b->mousebutton==Button2) {
      /* show/hide pop-up control window */
      if (gtx->cur_hstream==slice) {
         XUnmapWindow(GuiDpy,gtx->WindWindow);
         gtx->cur_hstream = -1;
      }
      else {
         gtx->cur_hstream = slice;
         map_hstream_window( index, gtx->cur_hwind );
      }
      gtx->cur_hwind = -1;
      gtx->cur_vwind = -1;
      gtx->cur_vstream = -1;
   }
   else if (b->mousebutton==Button3) {
      show_rgb_sliders( gtx, VIS5D_HSTREAM, slice );
   }

   vis5d_invalidate_frames(index);
   return 0;
}


/* Called when user clicks on VStream button */
static int vstream_cb( LUI_NEWBUTTON *b, int state )
{
   GuiContext gtx = get_gui_gtx( b->context_index );
   int index = b->context_index;
   int slice = b->index;

   if (b->mousebutton==Button1) {
      /* toggle display on/off */
      if (state==1) {
         int time, numtimes, curtime;
         vis5d_enable_graphics(index, VIS5D_VSTREAM, slice, VIS5D_ON);
         vis5d_get_timestep(index, &curtime);
         vis5d_get_numtimes(index, &numtimes);
         gtx->cur_vstream = slice;
         map_vstream_window( index, gtx->cur_vstream );
         for (time=0;time<numtimes;time++) {
            vis5d_make_vstreamslice(index, time, slice, time==curtime);
         }
         gtx->cur_hwind = -1;
         gtx->cur_vwind = -1;
         gtx->cur_hstream = -1;
      }
      else {
         XUnmapWindow(GuiDpy,gtx->WindWindow);
         vis5d_enable_graphics(index, VIS5D_VSTREAM, slice, VIS5D_OFF);
         gtx->cur_vstream = -1;
      }
   }
   else if (b->mousebutton==Button2) {
      /* show/hide pop-up control window */
      if (gtx->cur_vstream==slice) {
         XUnmapWindow(GuiDpy,gtx->WindWindow);
         gtx->cur_vstream = -1;
      }
      else {
         gtx->cur_vstream = slice;
         map_vstream_window( index, gtx->cur_hwind );
      }
      gtx->cur_hwind = -1;
      gtx->cur_vwind = -1;
      gtx->cur_hstream = -1;
   }
   else if (b->mousebutton==Button3) {
      show_rgb_sliders( gtx, VIS5D_VSTREAM, slice );
   }

   vis5d_invalidate_frames(index);
   return 0;
}


/*** view_cb **********************************************************
   Called when the "TOP", "SOUTH" or "WEST" buttons are selected.
**********************************************************************/
static int view_cb( LUI_BUTTON *b )
{
   int index = b->context_index;
   GuiContext gtx = get_gui_gtx(index);
   if (b->event->xbutton.button==Button3) {
      if (b->index==1) {
         vis5d_set_ortho_view(index, VIS5D_BOTTOM );
      }
      else if (b->index==2) {
         vis5d_set_ortho_view(index, VIS5D_NORTH);
      }
      else {
         vis5d_set_ortho_view(index, VIS5D_EAST);
      }
   }
   else {
      if (b->index==1) {
         vis5d_set_ortho_view(index, VIS5D_TOP);
      }
      else if (b->index==2) {
         vis5d_set_ortho_view(index, VIS5D_SOUTH);
      }
      else {
         vis5d_set_ortho_view(index, VIS5D_WEST);
      }
   }

   return 0;
}




/*** mode_cb ******************************************************
   This function is called when the user clicks on any of the mode
   radio button widgets.
**********************************************************************/
static int mode_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   XWindowAttributes winatts;

   if (gtx->MouseMode==MM_TRAJ)
      XUnmapWindow( GuiDpy, gtx->TrajWindow );

   if (gtx->MouseMode==MM_SOUND){
      vis5d_map_sndwindow( index, 0);
      XUnmapWindow( GuiDpy, gtx->SoundCtrlWindow );
   }
   if (gtx->MouseMode==MM_LABEL)
      vis5d_edit_label( index, '\r' );

   gtx->MouseMode = pb->index;

   switch (gtx->MouseMode) {
      case MM_NORMAL:
         LUI_LabelChangeText( gtx->ModeInfoLabel, 6, ModeInfo1 );
         vis5d_graphics_mode(index, VIS5D_CURSOR, VIS5D_OFF);
         vis5d_graphics_mode(index, VIS5D_PROBE, VIS5D_OFF);
         vis5d_graphics_mode(index, VIS5D_SOUND, VIS5D_OFF);
         break;
      case MM_TRAJ:
         LUI_LabelChangeText( gtx->ModeInfoLabel, 6, ModeInfo2 );
         XMapWindow( GuiDpy, gtx->TrajWindow );
         vis5d_graphics_mode(index, VIS5D_CURSOR, VIS5D_ON);
         vis5d_graphics_mode(index, VIS5D_PROBE, VIS5D_OFF);
         vis5d_graphics_mode(index, VIS5D_SOUND, VIS5D_OFF);
         break;
      case MM_SLICE:
         LUI_LabelChangeText( gtx->ModeInfoLabel, 6, ModeInfo4 );
         vis5d_graphics_mode(index, VIS5D_CURSOR, VIS5D_OFF);
         vis5d_graphics_mode(index, VIS5D_PROBE, VIS5D_OFF);
         vis5d_graphics_mode(index, VIS5D_SOUND, VIS5D_OFF);
         break;
      case MM_LABEL:
         LUI_LabelChangeText( gtx->ModeInfoLabel, 6, ModeInfo3 );
         vis5d_graphics_mode(index, VIS5D_CURSOR, VIS5D_OFF);
         vis5d_graphics_mode(index, VIS5D_PROBE, VIS5D_OFF);
         vis5d_graphics_mode(index, VIS5D_SOUND, VIS5D_OFF);
         break;
      case MM_PROBE:
         LUI_LabelChangeText( gtx->ModeInfoLabel, 6, ModeInfo5 );
         vis5d_graphics_mode(index, VIS5D_CURSOR, VIS5D_ON);
         vis5d_graphics_mode(index, VIS5D_PROBE, VIS5D_ON);
         vis5d_graphics_mode(index, VIS5D_SOUND, VIS5D_OFF);
         break;
      case MM_SOUND:
         LUI_LabelChangeText( gtx->ModeInfoLabel, 6, ModeInfo5 );
         vis5d_graphics_mode(index, VIS5D_CURSOR, VIS5D_ON);
         vis5d_graphics_mode(index, VIS5D_PROBE, VIS5D_OFF);
         vis5d_graphics_mode(index, VIS5D_SOUND, VIS5D_ON);
         XMapWindow( GuiDpy, gtx->SoundCtrlWindow );
         vis5d_map_sndwindow( index, 1);
         vis5d_draw_sounding_only( index, 1);
         sleep(1);  
         vis5d_draw_sounding_only( index, 1);
         XGetWindowAttributes(GuiDpy, gtx->SoundCtrlWindow, &winatts);
         if (gtx->othersnddpy != 0){
            vis5d_resize_sounding_window( index, winatts.width, winatts.height,
                                          winatts.x, winatts.y);
         }
         LUI_BorderWidth(2);
         break; 
      }

   vis5d_invalidate_frames(index);
   return 0;
}



/* trajectory set buttons */
static int traj_cb( LUI_BUTTON *pb )
{
   int set;
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   set = pb->index;

   if (pb->event->xbutton.button==Button1) {
      /*** Left Button -> toggle set on/off ***/
      if (vis5d_enable_graphics(index, VIS5D_TRAJ, set, VIS5D_TOGGLE)) {
         vis5d_set_cursor_color( index, set );
         gtx->cur_trajset = set;
      }
   }
   else if (pb->event->xbutton.button==Button2){
      /* Middle Button -> select set */
      if (pb->state == LUI_BUTTON_OUT)
         pb->state = LUI_BUTTON_IN;
      else
         pb->state = LUI_BUTTON_OUT;
      vis5d_set_cursor_color( index, set );
      gtx->cur_trajset = set;
   }
   else if (pb->event->xbutton.button==Button3) {
      /* Right Button -> color selector */
      if (pb->state == LUI_BUTTON_OUT)
         pb->state = LUI_BUTTON_IN;
      else
         pb->state = LUI_BUTTON_OUT;
      show_isocolor_window( gtx, VIS5D_TRAJ, set );
   }

   vis5d_invalidate_frames(index);
   return 0;
}



/**********************************************************/
/*** CALL-BACK FUNCTIONS FOR MATRIX OF GRAPHICS BUTTONS ***/
/**********************************************************/


/*
 * This function is called whenever a button in the button matrix is
 * clicked on.
 */
static int button_matrix_cb( LUI_BUTTON_MATRIX *bm, int row, int col,
                             int button )
{
   int curtime, numtimes, time, CurrentVolume;
   int index = bm->context_index;
   GuiContext gtx = get_gui_gtx(index);
   int state = LUI_ButtonMatrixGetState( bm, row, col );

   vis5d_get_timestep(index, &curtime);
   vis5d_get_numtimes(index, &numtimes);

   if (button==Button1) {
      /*** Left Button -> toggle graphic on/off ***/
      if (col<5) {
         LUI_ButtonMatrixSetState( bm, row, col, !state );
      }
      vis5d_signal_redraw(index, 1);

      switch (col) {
         case 0:
            /** Iso-Surface **/
            if (gtx->cur_isosurf==row) {
               XUnmapWindow(GuiDpy,gtx->IsoWindow);
               gtx->cur_isosurf = -1;
            }
            if (vis5d_enable_graphics(index, VIS5D_ISOSURF, row, VIS5D_TOGGLE)) {
               if (gtx->cur_isosurf>=0) {
                  XUnmapWindow(GuiDpy,gtx->IsoWindow);
               }
               gtx->cur_isosurf = row;
               map_isosurf_window( index, gtx->cur_isosurf );
            }
            break;
         case 1:
            /** Horizontal Contour Line Slice **/
            if (gtx->cur_hslice==row) {
               XUnmapWindow(GuiDpy,gtx->HSliceWindow);
               gtx->cur_hslice = -1;
            }
            if (vis5d_enable_graphics(index, VIS5D_HSLICE, row, VIS5D_TOGGLE)) {
               if (gtx->cur_hslice>=0)
                  XUnmapWindow(GuiDpy,gtx->HSliceWindow);
               gtx->cur_hslice = row;
               map_hslice_window( index, gtx->cur_hslice );
               for (time=0;time<numtimes;time++) {
                  vis5d_make_hslice(index, time, row, time==curtime);
               }
            }
            break;
         case 2:
            /** Vertical Contour Line Slice **/
            if (gtx->cur_vslice==row) {
               XUnmapWindow(GuiDpy,gtx->VSliceWindow);
               gtx->cur_vslice = -1;
            }
            if (vis5d_enable_graphics(index, VIS5D_VSLICE, row, VIS5D_TOGGLE)) {
               if (gtx->cur_vslice>=0)
                  XUnmapWindow(GuiDpy,gtx->VSliceWindow);
               gtx->cur_vslice = row;
               map_vslice_window( index, gtx->cur_vslice );
               for (time=0;time<numtimes;time++) {
                  vis5d_make_vslice(index, time, row, time==curtime);
               }
            }
            break;
         case 3:
            /** Horizontal Colored Slice **/
            if (vis5d_enable_graphics(index, VIS5D_CHSLICE, row, VIS5D_TOGGLE)) {
               /* Display color widget */
               show_colorbar_window( index, VIS5D_CHSLICE, row );
               for (time=0;time<numtimes;time++) {
                  vis5d_make_chslice(index, time, row, time==curtime);
               }
            }
            else if (cb_graphic==VIS5D_CHSLICE && cb_var==row) {
               /* Hide color widget */
               hide_colorbar_window( index );
            }
            break;
         case 4:
            /** Vertical Colored Slice **/
            if (vis5d_enable_graphics(index, VIS5D_CVSLICE, row, VIS5D_TOGGLE)) {
               show_colorbar_window( index, VIS5D_CVSLICE, row );
               for (time=0;time<numtimes;time++) {
                  vis5d_make_cvslice(index, time, row, time==curtime);
               }
            }
            else {
               if (cb_graphic==VIS5D_CVSLICE && cb_var==row) {
                  hide_colorbar_window( index );
               }
            }
            break;
         case 5:
            /** Volume **/
            vis5d_get_volume(index, &CurrentVolume);
            if (row==CurrentVolume) {
               /* turn off the volume */
               LUI_ButtonMatrixSetState( gtx->ButtonMatrix, CurrentVolume, 5, 0 );
               if (cb_graphic==VIS5D_VOLUME && cb_var==row) {
                  hide_colorbar_window( index );
               }
               CurrentVolume = -1;
            }
            else {
               /* turn on a different volume */
               if (CurrentVolume>=0) {
                  /* turn off previous volume button */
                  LUI_ButtonMatrixSetState( gtx->ButtonMatrix, CurrentVolume, 5, 0 );
               }
               CurrentVolume = row;
               LUI_ButtonMatrixSetState( gtx->ButtonMatrix, row, 5, 1 );
               show_colorbar_window( index, VIS5D_VOLUME, CurrentVolume );
            }
            vis5d_set_volume(index, CurrentVolume);
            break;
      } /*switch*/
      vis5d_invalidate_frames(index);
   }
   else if (button==Button2) {
      /* Middle Button -> toggle slider/selector window on/off ***/
      switch (col) {
         case 0:
            if (gtx->cur_isosurf>=0)
               XUnmapWindow(GuiDpy,gtx->IsoWindow);
            if (gtx->cur_isosurf == row)
              gtx->cur_isosurf = -1;
            else {
               gtx->cur_isosurf = row;
               map_isosurf_window( index, gtx->cur_isosurf );
            }
            break;
         case 1:
            if (gtx->cur_hslice>=0)
               XUnmapWindow(GuiDpy,gtx->HSliceWindow);
            if (gtx->cur_hslice==row)
               gtx->cur_hslice = -1;
            else {
               gtx->cur_hslice = row;
               map_hslice_window( index, gtx->cur_hslice );
            }
            break;
         case 2:
            if (gtx->cur_vslice>=0)
               XUnmapWindow(GuiDpy,gtx->VSliceWindow);
            if (gtx->cur_vslice==row)
               gtx->cur_vslice = -1;
            else {
               gtx->cur_vslice = row;
               map_vslice_window( index, gtx->cur_vslice );
            }
            break;
         case 3:
            if (cb_graphic==VIS5D_CHSLICE && cb_var==row) {
               hide_colorbar_window( index );
            }
            else {
               show_colorbar_window( index, VIS5D_CHSLICE, row );
            }
            break;
         case 4:
            if (cb_graphic==VIS5D_CVSLICE && cb_var==row) {
               hide_colorbar_window( index );
            }
            else {
               show_colorbar_window( index, VIS5D_CVSLICE, row );
            }
            break;
         case 5:
            if (cb_graphic==VIS5D_VOLUME && cb_var==row) {
               hide_colorbar_window( index );
            }
            else {
               show_colorbar_window( index, VIS5D_VOLUME, row );
            }
            break;
      }
   }
   else if (button==Button3) {
      /* Right Button -> color selector */
      if (col<5) {
         gtx->ColorRow = row;
         gtx->ColorCol = col;
         if (col==0) {
            /* Isosurface color */
            show_isocolor_window( gtx, VIS5D_ISOSURF, row );
         }
         else if (col==1) {
            /* HSlice color */
            show_rgb_sliders( gtx, VIS5D_HSLICE, row );
         }
         else if (col==2) {
            /* VSlice color */
            show_rgb_sliders( gtx, VIS5D_VSLICE, row );
         }
         else if (col==3) {
            /* Colored HSlice tickmark color */
            show_rgb_sliders( gtx, VIS5D_CHSLICE, row );
         }
         else if (col==4) {
            /* Colored VSlice tickmark color */
            show_rgb_sliders( gtx, VIS5D_CVSLICE, row );
         }
      }
   }
   return 1;
}




/****************************************************************/
/*** CALL-BACK FUNCTIONS FOR ISOSURFACE, SLICE WINDOW WIDGETS ***/
/****************************************************************/


/*** isosurface_cb ****************************************************
   This is the call-back function for the isosurface level widget.
**********************************************************************/
static isosurface_cb( LUI_NEWSLIDER *s, float value )
{
   float IsoLevel;
   int index = s->context_index;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_isosurface(index, gtx->cur_isosurf, &IsoLevel);
   if (IsoLevel!=value) {
      IsoLevel = value;
   }
   vis5d_set_isosurface(index, gtx->cur_isosurf, IsoLevel);
   return 0;
}


/* called when the "NEWSURF" button is selected */
static newsurf_cb( LUI_NEWBUTTON *pb )
{
   int curtime, numtimes, i;
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_timestep(index, &curtime);
   vis5d_get_numtimes( index, &numtimes );
   if (gtx->cur_isosurf>-1) {
      for (i=0;i<numtimes;i++) {
         vis5d_make_isosurface( index, i, gtx->cur_isosurf, i==curtime );
      }
   }
   return 0;
}


static int texture_cb( LUI_BUTTON *b, int state )
{
   int index = b->context_index;
   vis5d_graphics_mode(index, VIS5D_TEXTURE, VIS5D_TOGGLE);
   vis5d_invalidate_frames(index);
   return 0;
}



/*
 * This function is called when the user has entered a new value into
 * the horizontal contour line slice interval widget.
 */
static int hslice_cb( LUI_FIELD *field )
{
   char valstr[1000];
   int curtime, numtimes, time;
   float minval, maxval;
   float cur_interval, cur_low, cur_high, cur_level;
   int index = field->context_index;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_timestep(index, &curtime);
   vis5d_get_numtimes(index, &numtimes);
   vis5d_get_var_range(index, gtx->cur_hslice, &minval, &maxval);
   vis5d_get_hslice(index, gtx->cur_hslice, &cur_interval, &cur_low,
                    &cur_high, &cur_level);

   LUI_FieldGetText( gtx->hslice_field, valstr );

   if (gtx->cur_hslice>=0) {
      float interval, low, high;
      if ( strchr(valstr,'(') && strchr(valstr,')') && strchr(valstr,',') ) {
         sscanf( valstr,"%f (%f,%f)", &interval, &low, &high );
      }
      else {
         sscanf( valstr,"%f", &interval );
         low = minval;
         high = maxval;
      }
      /* interval = 0 not allowed */
      if (interval==0.0)
        interval = cur_interval;
      /* verify low and high */
      if (low>=high) {
         low = minval;
         high = maxval;
      }
      if (cur_interval!=interval || cur_low!=low || cur_high!=high) {
         vis5d_set_hslice(index, gtx->cur_hslice, interval, low, high,
                          cur_level);
         for (time=0;time<numtimes;time++) {
            vis5d_make_hslice(index, time, gtx->cur_hslice, time==curtime);
         }
      }
   }
   return 0;
}



/*
 * This function is called when the user has entered a new value into
 * the vertical contour line slice interval widget.
 */
static int vslice_cb( LUI_FIELD *field )
{
   char valstr[1000];
   int curtime, numtimes, time;
   float minval, maxval;
   float VSliceInterval, VSliceLowLimit, VSliceHighLimit;
   float row0, col0, row1, col1;
   int index = field->context_index;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_timestep(index, &curtime);
   vis5d_get_numtimes(index, &numtimes);
   vis5d_get_var_range(index, gtx->cur_vslice, &minval, &maxval);
   vis5d_get_vslice(index, gtx->cur_vslice, &VSliceInterval, &VSliceLowLimit,
                    &VSliceHighLimit, &row0, &col0, &row1, &col1);

   LUI_FieldGetText( gtx->vslice_field, valstr );

   if (gtx->cur_vslice>=0) {
      float interval, low, high;
      if ( strchr(valstr,'(') && strchr(valstr,')') && strchr(valstr,',') ) {
         sscanf( valstr,"%f (%f,%f)", &interval, &low, &high );
      }
      else {
         sscanf( valstr,"%f", &interval );
         low = minval;
         high = maxval;
      }
      /* interval = 0 not allowed */
      if (interval==0.0)
        interval = VSliceInterval;
      /* verify low and high */
      if (low>=high) {
         low = minval;
         high = maxval;
      }
      if (VSliceInterval!=interval || VSliceLowLimit!=low || VSliceHighLimit!=high) {
         vis5d_set_vslice(index, gtx->cur_vslice, interval, low, high,
                          row0, col0, row1, col1);
         for (time=0;time<numtimes;time++) {
            vis5d_make_vslice(index, time, gtx->cur_vslice, time==curtime);
         }
      }
   }
   return 0;
}



/*** windscale_cb *****************************************************
   This function is called when the user has entered a new value into
   the wind scale widget.
**********************************************************************/
static int windscale_cb( LUI_FIELD *field, char *text )
{
   float f;
   int curtime, numtimes, time;
   float HWindDensity, HWindScale, HWindLevel;
   float VWindDensity, VWindScale, row0, col0, row1, col1;
   int index = field->context_index;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_timestep(index, &curtime);
   vis5d_get_numtimes(index, &numtimes);

   f = LUI_FieldGetDouble( field );
   if (f==0.0) {
      f = 1.0;
      LUI_FieldSetDouble( field, f );
   }


   if (gtx->cur_hwind>=0) {
      vis5d_get_hwindslice(index, gtx->cur_hwind,
                           &HWindDensity, &HWindScale, &HWindLevel);
      if (HWindScale!=f) {
         HWindScale = f;
         vis5d_set_hwindslice(index, gtx->cur_hwind,
                              HWindDensity, HWindScale, HWindLevel);
         for (time=0;time<numtimes;time++) {
            vis5d_make_hwindslice(index, time, gtx->cur_hwind, time==curtime);
         }
      }
   }
   else if (gtx->cur_vwind>=0) {
      vis5d_get_vwindslice(index, gtx->cur_vwind, &VWindDensity, &VWindScale,
                           &row0, &col0, &row1, &col1);
      if (VWindScale!=f) {
         VWindScale = f;
         vis5d_set_vwindslice(index, gtx->cur_vwind, VWindDensity, VWindScale,
                              row0, col0, row1, col1);
         for (time=0;time<numtimes;time++) {
            vis5d_make_vwindslice(index, time, gtx->cur_vwind, time==curtime);
         }
      }
   }
   else if (gtx->cur_hstream>=0) {
      f = 1.0;
      LUI_FieldSetDouble( field, f );
   }

   return 0;
}




/*** winddensity_cb *****************************************************
   This function is called when the user has entered a new value into
   the wind density widget.
**********************************************************************/
static int winddensity_cb( LUI_FIELD *field )
{
   float d;
   int curtime, numtimes, time;
   float HWindDensity, HWindScale, HWindLevel;
   float VWindDensity, VWindScale, row0, col0, row1, col1;
   float HStreamDensity, HStreamLevel;
   float VStreamDensity;
   int index = field->context_index;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_timestep(index, &curtime);
   vis5d_get_numtimes(index, &numtimes);

   d = LUI_FieldGetDouble( gtx->winddensity_field );

   if (gtx->cur_hwind>=0) {
      if (d<0.0 || d>1.0) {
         d = 1.0;
         LUI_FieldSetDouble( gtx->winddensity_field, d );
      }
      vis5d_get_hwindslice(index, gtx->cur_hwind,
                           &HWindDensity, &HWindScale, &HWindLevel);
      if (HWindDensity!=d) {
         HWindDensity = d;
         vis5d_set_hwindslice(index, gtx->cur_hwind,
                              HWindDensity, HWindScale, HWindLevel);
         for (time=0;time<numtimes;time++) {
            vis5d_make_hwindslice(index, time, gtx->cur_hwind, time==curtime);
         }
      }
   }
   else if (gtx->cur_vwind>=0) {
      if (d<0.0 || d>1.0) {
         d = 1.0;
         LUI_FieldSetDouble( gtx->winddensity_field, d );
      }
      vis5d_get_vwindslice(index, gtx->cur_vwind, &VWindDensity, &VWindScale,
                           &row0, &col0, &row1, &col1);
      if (VWindDensity!=d) {
         VWindDensity = d;
         vis5d_set_vwindslice(index, gtx->cur_vwind, VWindDensity, VWindScale,
                              row0, col0, row1, col1);
         for (time=0;time<numtimes;time++) {
            vis5d_make_vwindslice(index, time, gtx->cur_vwind, time==curtime);
         }
      }
   }
   else if (gtx->cur_hstream>=0) {
      if (d < 0.5 || d > 2.0) {
         if (d < 0.5) d = 0.5;
         if (d > 2.0) d = 2.0;
         LUI_FieldSetDouble( gtx->winddensity_field, d );
      }
      vis5d_get_hstreamslice(index, gtx->cur_hstream, &HStreamDensity, &HStreamLevel);
      if (HStreamDensity!=d) {
         HStreamDensity = d;
         vis5d_set_hstreamslice(index, gtx->cur_hstream, HStreamDensity, HStreamLevel);
         for (time=0;time<numtimes;time++) {
            vis5d_make_hstreamslice(index, time, gtx->cur_hstream, time==curtime);
         }
      }
   }
   else if (gtx->cur_vstream>=0) {
      if (d < 0.5 || d > 2.0) {
         if (d < 0.5) d = 0.5;
         if (d > 2.0) d = 2.0;
         LUI_FieldSetDouble( gtx->winddensity_field, d );
      }
      vis5d_get_vstreamslice(index, gtx->cur_vstream, &VStreamDensity,
                             &row0, &col0, &row1, &col1);
      if (VStreamDensity!=d) {
         VStreamDensity = d;
         vis5d_set_vstreamslice(index, gtx->cur_vstream, VStreamDensity,
                                row0, col0, row1, col1);
         for (time=0;time<numtimes;time++) {
            vis5d_make_vstreamslice(index, time, gtx->cur_vstream, time==curtime);
         }
      }
   }
   return 0;
}



/************************************/
/***  Variable Cloning Functions  ***/
/************************************/


/*
 * Create the "new variable" window.  This function is called when the
 * user clicks on the "NEWVAR" button.  We rebuild the window from scratch
 * each time to get the latest external functions.
 */
static void make_clone_window( int index )
{
   LUI_NEWBUTTON *b;
   int x, y, i, var, num, n;
   char name[1000];
   GuiContext gtx = get_gui_gtx(index);
   int NumVars;
   char VarName[MAXVARS][10];
   int VarType[MAXVARS];

   vis5d_get_numvars( index, &NumVars );
   for (i=0;i<NumVars;i++) {
      vis5d_get_var_name( index, i, VarName[i] );
      vis5d_get_var_type( index, i, &VarType[i] );
   }

   if (gtx->CloneWindow) {
      /* destroy current clone window before remaking it */
      LUI_DestroyWindow( gtx->CloneWindow );  /* all widgets destroyed too */
      gtx->CloneWindow = NULL;
   }

   gtx->CloneWindow = LUI_CreateWindow( LUI_RootWindow, 100, 100 );

   LUI_BorderWidth( 1 );

   LUI_NewLabelCreate( gtx->CloneWindow, LUI_LEFT, LUI_TOP, 300, 26,
                       "Select variable to clone:" );

   /* Make buttons for REGULAR and EXT_FUNC variables */
   num = 0;
   for (var=0;var<NumVars;var++) {
      if (VarType[var]==VIS5D_REGULAR || VarType[var]==VIS5D_EXT_FUNC) {
         if (num%3==0) {
            x = LUI_LEFT;
            y = LUI_NEXT_Y;
         }
         else {
            x = LUI_NEXT_X;
            y = LUI_SAME_Y;
         }
         b = LUI_PushButtonCreate( gtx->CloneWindow, x,y, 80,22, VarName[var]);
         LUI_ButtonCallback( b, do_clone_cb );
         LUI_ButtonIndex( b, var );
         num++;
      }
   }


   /* Scan the userfuncs directory for analysis functions */
   if (gtx->funcpath[0]) {
      /* search directory specified by -funcpath <path> */
      n = find_analysis_functions( gtx->funcpath, FuncName[index] );
   }
   else {
      /* search default directory */
      n = find_analysis_functions( FUNCTION_PATH, FuncName[index] );
   }
   NumFuncs[index] = n;
   for (i=0;i<n;i++) {
      FuncType[index][i] = VIS5D_EXT_FUNC;
   }

   /* Look for variables computed by expressions too. */
   n = find_expr_variables(index, FuncName[index]+NumFuncs[index] );
   for (i=0;i<n;i++) {
      int j = NumFuncs[index]+i;
      FuncType[index][j] = VIS5D_EXPRESSION;
   }
   NumFuncs[index] += n;

   if (NumFuncs[index]>0) {
      /* Make button pad listing functions we can compute */
      LUI_NewLabelCreate( gtx->CloneWindow, LUI_LEFT, LUI_NEXT_Y, 300, 26,
                          "Or function to compute:" );

      for (i=0;i<NumFuncs[index];i++) {
         if (FuncType[index][i]==VIS5D_EXPRESSION) {
            /* Put = ... after the name so user knows it's a type-in expr. */
            sprintf(name, "%s = ...", FuncName[index][i] );
         }
         else {
            strcpy( name, FuncName[index][i] );
         }
         if (i%3==0) {
            x = LUI_LEFT;
            y = LUI_NEXT_Y;
         }
         else {
            x = LUI_NEXT_X;
            y = LUI_SAME_Y;
         }
         b = LUI_PushButtonCreate( gtx->CloneWindow, x, y,  80, 22, name );
         LUI_ButtonCallback( b, do_clone_cb );
         LUI_ButtonIndex( b, -(i+1) );
      }
   }

   LUI_NewLabelCreate( gtx->CloneWindow, LUI_LEFT, LUI_NEXT_Y,
                       250, 26, "Or:" );
   b = LUI_PushButtonCreate( gtx->CloneWindow, LUI_LEFT, LUI_NEXT_Y,
                             246, 22, "Make type-in expression..." );
   LUI_ButtonCallback( b, new_function_cb );
   LUI_NewLabelCreate( gtx->CloneWindow, LUI_LEFT, LUI_NEXT_Y,
                       250, 12, " " );
   b = LUI_PushButtonCreate( gtx->CloneWindow, 80, LUI_NEXT_Y,
                         90, 22, "Cancel" );
   LUI_ButtonCallback( b, do_clone_cb );
   LUI_ButtonIndex( b, 9999 );


   gtx->CloneWidth = 252;
   gtx->CloneHeight = LUI_LayoutGet( LUI_NEXT_Y );
   LUI_MoveResizeWindow( gtx->CloneWindow, 450,400,
                         gtx->CloneWidth, gtx->CloneHeight );
   XMapWindow( GuiDpy, gtx->CloneWindow );
}


/* called when the NEWVAR button is pressed */
static int newvar_cb( LUI_BUTTON *pb )
{
   int numvars;
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_numvars( index, &numvars );

   if (pb->state == LUI_BUTTON_OUT) {
      /* unmap clone window */
      XUnmapWindow( GuiDpy, gtx->CloneWindow );
      unmap_expr_window(index);
   }
   else {
      /* check if there's room for another variable */
      if (numvars>=MAXVARS) {
         alert( index, "Cannot make anymore clones." );
         return 0;
      }
      /* display clone window */
      make_clone_window(index);
   }
   return 0;
}





#ifdef LEAVEOUT
static delete_button_row( row )
int row;
{


}
#endif



/*
 * Update the labels on various widgets when the min and max values for
 * a variable may have changed.
 */
void update_var( int index, int newvar )
{
   char str[1000];
   char varname[10];
   float minval, maxval;
   unsigned int *colors;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_var_name( index, newvar, varname );
   vis5d_get_var_range(index, newvar, &minval, &maxval);
   vis5d_reset_var_graphics(index, newvar);

   if (cb_graphic>=0 && cb_var>=0) {
      show_colorbar_window( index, cb_graphic, cb_var );
   }
}




/*
 * This is called when the "New Function" button is pressed.
 */
static int new_function_cb( LUI_NEWBUTTON *pb, int state )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   XUnmapWindow( GuiDpy, gtx->CloneWindow );
   map_expr_window( index, "" );
   return 0;
}


/*
 * This is called when Enter is pressed in the expression type-in window.
 */
static int expr_cb( LUI_FIELD *field )
{
   int index = field->context_index;
   GuiContext gtx = get_gui_gtx(index);
   LUI_FieldGetText( gtx->expr_field, gtx->Expression );
   return 0;
}




/*
 * This is called when OK is selected on the expression type-in window.
 */
static int expr_ok_cb( LUI_NEWBUTTON *pb )
{
   char expr[1000], newname[100], mess[100];
   int var, recompute;
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   LUI_FieldGetText( gtx->expr_field, expr );

   /* evaluate */
   if (vis5d_make_expr_var(index, expr, newname, mess, &var, &recompute) < 0 &&
       mess[0] != 0) {
      alert(index, mess);
      var = -1;
   }

   if (var>=0) {
      /* success */
      /* Update the GUI with a new row of buttons if we're not recomputing */
      /* an existing variable. */
      if (!recompute) {
        add_button_row(index, newname, var);
        init_var_colortable( gtx, var );
        vis5d_reset_var_graphics(index, var);
      }
      else {
        update_var(index, var);
        reset_widgets(index);
      }

      unmap_expr_window(index);
   }
   else {
      /* error- don't unmap expression window, user probably wants to edit */
   }
   return 0;
}


/*
 * This is called when cancel is selected on the expression type-in window.
 */
static int expr_cancel_cb( LUI_NEWBUTTON *pb )
{
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);
   unmap_expr_window(index);
   return 0;
}



/* toggle pretty rendering on/off */
static int pretty_cb( LUI_BUTTON *pb )
{
   int PrettyFlag;
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   /* only turn on pretty when not animating */
   if (gtx->GoTime==0) {
      PrettyFlag = vis5d_graphics_mode(index, VIS5D_PRETTY, VIS5D_TOGGLE);
      if (PrettyFlag) vis5d_signal_redraw(index, 1);
   }
   else {
      /* turn pretty off */
      vis5d_graphics_mode(index, VIS5D_PRETTY, VIS5D_OFF);
   }
   return 0;
}



/* toggle reverse video option */
static int reverse_cb( LUI_BUTTON *pb )
{
   int index = pb->context_index;
   if (pb->state==LUI_BUTTON_OUT) {
      /* black background, white box, map, labels */
      vis5d_graphics_mode(index, VIS5D_REVERSE, VIS5D_OFF);
   }
   else {
      /* white background, black box, map, labels */
      vis5d_graphics_mode(index, VIS5D_REVERSE, VIS5D_ON);
   }
   vis5d_signal_redraw( index, 1 );
   vis5d_invalidate_frames( index );
   return 0;
}


/* toggle information display */
static void info_cb( int index)
{
   vis5d_graphics_mode(index, VIS5D_INFO, VIS5D_TOGGLE);
   vis5d_invalidate_frames(index);
}


/* called when SAVE PIC button is selected */
static int savepic_cb( LUI_BUTTON *pb )
{
   XEvent pe;
   char filename[1000];
   int index = pb->context_index;
   GuiContext gtx = get_gui_gtx(index);

   verify_value = -1;
   XMapWindow( GuiDpy, gtx->SavePicWindow );

   while (verify_value < 0) {
      if (XPending(GuiDpy)) {
         XNextEvent(GuiDpy, &pe);
         LUI_EventDispatch(&pe);
      }
   }
   XUnmapWindow( GuiDpy, gtx->SavePicWindow );
   if (verify_value==1 && gtx->MouseMode != MM_SOUND) {
      int n = LUI_RadioGetCurrent( gtx->SavePicRadio );
      int format = gtx->SaveFormats[n];

      /* raise 3-D window and redraw */
      XRaiseWindow( GfxDpy, gtx->mainwin );
      XSync( GfxDpy, 0 );
      vis5d_draw_frame(index, gtx->GoTime);
      vis5d_swap_frame(index);
      XSync( GfxDpy, 0 );
      vis5d_draw_frame(index, gtx->GoTime);
      vis5d_swap_frame(index);
      XSync( GfxDpy, 0 );
      LUI_FieldGetText( gtx->SavePicField, filename );
      vis5d_save_window( index, filename, format );
      /* raise control panel window */
      XRaiseWindow( GuiDpy, gtx->CpWindow );
   }
   else if (verify_value==1 && gtx->MouseMode ==MM_SOUND) {
      int n = LUI_RadioGetCurrent( gtx->SavePicRadio );
      int format = gtx->SaveFormats[n];

      /* raise 3-D window and redraw */
      XRaiseWindow( GfxDpy, gtx->SoundCtrlWindow );
      XSync( GfxDpy, 0 );
      vis5d_draw_frame(index, gtx->GoTime);
      vis5d_swap_frame(index);
      XSync( GfxDpy, 0 );
      vis5d_draw_frame(index, gtx->GoTime);
      vis5d_swap_frame(index);
      XSync( GfxDpy, 0 );
      LUI_FieldGetText( gtx->SavePicField, filename );
      vis5d_save_snd_window( index, filename, format );
      /* raise control panel window */
      XRaiseWindow( GuiDpy, gtx->CpWindow );
   }

   return 0;
}




static void iconify( int index )
{
   GuiContext gtx = get_gui_gtx(index);
   printf("Iconify\n");
   XUnmapWindow( GfxDpy, gtx->mainwin );
   XUnmapWindow( GuiDpy, gtx->CpWindow );
   XUnmapWindow( GuiDpy, gtx->IsoWindow );
   XUnmapWindow( GuiDpy, gtx->HSliceWindow );
   XUnmapWindow( GuiDpy, gtx->VSliceWindow );
   if (gtx->TrajWindow) {
      XUnmapWindow( GuiDpy, gtx->TrajWindow );
   }
   vis5d_map_sndwindow( index, 0);
   XUnmapWindow( GuiDpy, gtx->SoundCtrlWindow );
   XUnmapWindow( GuiDpy, gtx->VerifyWindow );
   XUnmapWindow( GuiDpy, gtx->AlertWindow );
   if (gtx->CloneWindow) {
      XUnmapWindow( GuiDpy, gtx->CloneWindow );
   }
   if (gtx->WindWindow) {
      XUnmapWindow( GuiDpy, gtx->WindWindow );
   }
   XUnmapWindow( GuiDpy, gtx->ExprWindow );
}


static void deiconify( int index )
{
   GuiContext gtx = get_gui_gtx(index);
   printf("Deiconify\n");
   XMapWindow( GfxDpy, gtx->mainwin );
   XMapWindow( GuiDpy, gtx->CpWindow );
   if (gtx->cur_isosurf>-1)  XMapWindow( GuiDpy, gtx->IsoWindow );
   if (gtx->cur_hslice>-1)  XMapWindow( GuiDpy, gtx->HSliceWindow );
   if (gtx->cur_vslice>-1)  XMapWindow( GuiDpy, gtx->VSliceWindow );
   if (gtx->MouseMode==MM_TRAJ && gtx->TrajWindow) {
      XMapWindow( GuiDpy, gtx->TrajWindow );
   }
   if (gtx->MouseMode==MM_SOUND && gtx->SoundCtrlWindow) {
      XMapWindow( GuiDpy, gtx->SoundCtrlWindow );
      vis5d_map_sndwindow( index, 1);
      vis5d_draw_sounding_only( index, 1);
   }
   if ((gtx->cur_hwind>-1 || gtx->cur_vwind>-1 || gtx->cur_hstream)
       && gtx->WindWindow) {
      XMapWindow( GuiDpy, gtx->WindWindow );
   }
   XSync( GuiDpy, 0 );
   XSync( GfxDpy, 0 );
   vis5d_signal_redraw(index, 2);
}


/*
 * Print window image to PostScript printer.
 */
static void print_cb( int index )
{
   char *printer, msg[1000];
   GuiContext gtx = get_gui_gtx(index);

   printer = getenv( "PRINTER" );
   if (printer) {
      sprintf( msg, "Print to %s?", printer );
   }
   else {
      sprintf( msg, "Print to default printer?" );
   }

   if (verify(index, msg) && gtx->MouseMode != MM_SOUND) {
      vis5d_draw_frame(index, gtx->GoTime);
      vis5d_swap_frame(index);
      vis5d_print_window(index);
   }
   else if (verify(index, msg) && gtx->MouseMode == MM_SOUND) {
      vis5d_draw_frame(index, gtx->GoTime);
      vis5d_swap_frame(index);
      vis5d_print_snd_window(index);
   }
}


/*
 * Called when "INTERP" button is pressed.
 */
static int interpret_cb( LUI_BUTTON *b )
{
   Display *dpy;
   Window win;
   char *dpy_name;

   /*
    * Try to automatically raise the shell window
    */
   if (getenv("DISPLAY")) {
      dpy_name = getenv("DISPLAY");
      dpy = XOpenDisplay(dpy_name);
      if (dpy) {
         if (getenv("WINDOWID")) {
            win = atoi( getenv("WINDOWID") );
            if (win) {
               XRaiseWindow( dpy, win );
            }
         }
         XCloseDisplay(dpy);
      }
   }

   interpret( b->context_index );
   return 0;
}



/*
 * Called when "OK" button in script file browser is pressed.
 */
static int run_script_cb( LUI_BROWSER *b, char *file )
{
   int index = b->context_index;
   int var, oldnumvars, numvars;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_numvars( index, &oldnumvars );
   if (execute_script( index, file )==0) {
      alert( index, "Script returned error status");
   }

   /* may have to add more button rows to control panel now */
   vis5d_get_numvars( index, &numvars );
   for (var=oldnumvars;var<numvars;var++) {
      char varname[20];
      vis5d_get_var_name( index, var, varname );
      add_button_row( index, varname, var );
      /*init_var_colortable( gtx, var );*/
   }
   reset_widgets( index );
   recompute_graphics( index );
   vis5d_signal_redraw( index, 1 );
   return 0;
}

/* This is called after exiting from the interpreter and if there */
/* are new buttons that need to be added */
int make_new_buttons_after_interp( int index, int number_of_new_buttons )
{
   GuiContext gtx = get_gui_gtx(index);
   char varname[999];
   int varthing;
   int varnum;
   int yo;
   vis5d_get_numvars( index, &varnum);
   for (yo = 0; yo < number_of_new_buttons; yo++){
      varthing = varnum - number_of_new_buttons + yo;
      vis5d_get_var_name( index, varthing, varname ); 
      add_button_row(index, varname, varthing);
      init_var_colortable( gtx, varthing );
      vis5d_reset_var_graphics(index, varthing);
   }
   reset_widgets( index );
   recompute_graphics( index );
   vis5d_signal_redraw( index, 1 );
   return 0;
}    

/*
 * Called when "SCRIPT.." button is pressed.
 */
static int script_cb( void )
{
   static LUI_BROWSER *b = NULL;

   if (!b) {
      /* create the file browser once */
      LUI_BorderWidth( 2 );
      b = LUI_BrowserCreate( 300, 400 );
      LUI_BrowserCallback( b, run_script_cb );
   }

   LUI_BrowserActivate( b, NULL );
   return 0;
}



static int uvw_cb( LUI_BUTTON *b )
{
   int index = b->context_index;
   GuiContext gtx = get_gui_gtx(index);

   show_uvw_widget( gtx );
   return 0;
}


static int legend_cb( LUI_BUTTON *b )
{
   int index = b->context_index;
   vis5d_graphics_mode( index, VIS5D_LEGENDS, VIS5D_TOGGLE );
}



/*
 * Called when color table in a colorbar widget is modified.
 */
static int colorbar_cb( LUI_COLORBAR *cb, int action )
{
   int index = cb->context_index;
   float *p;
   GuiContext gtx = get_gui_gtx(index);

   if (cb_graphic==VIS5D_TOPO) {
      if (action==LUI_RGB_RESET) {
         /* user changed table with mouse */
         vis5d_reset_topo_colors( index );
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_RGB_CHANGE) {
         vis5d_signal_redraw( index, 1 );
      }
      else {
          /* ignore alpha events, not used */
      }
   }
   else if (cb_graphic==VIS5D_CHSLICE) {
      if (action==LUI_RGB_RESET) {
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_CHSLICE, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_CHSLICE, cb_var, &table );
         vis5d_color_table_init_params( p, 1, 0 );
         vis5d_color_table_recompute( table, 256, p, 1, 0 );
         vis5d_signal_redraw( index, 1 );
         LUI_ColorBarRedraw( cb );
      }
      else if (action==LUI_ALPHA_RESET) {
         unsigned int *table;
         int opacity;
         vis5d_get_color_table_params(index, VIS5D_CHSLICE, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_CHSLICE, cb_var, &table );
         vis5d_color_table_init_params( p, 0, 1 );
         vis5d_color_table_recompute( table, 256, p, 0, 1 );
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_RGB_SHAPE) {
         /* change shape of rgb curves */
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_CHSLICE, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_CHSLICE, cb_var, &table );
         vis5d_color_table_recompute( table, 256, p, 1, 0 );
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_ALPHA_SHAPE) {
         /* change shape of alpha curve */
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_CHSLICE, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_CHSLICE, cb_var, &table );
         vis5d_color_table_recompute( table, 256, p, 0, 1 );
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_RGB_CHANGE) {
         /* mouse used to draw curve */
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_ALPHA_CHANGE) {
         vis5d_signal_redraw( index, 1 );
      }
   }
   else if (cb_graphic==VIS5D_CVSLICE) {
      if (action==LUI_RGB_RESET) {
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_CVSLICE, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_CVSLICE, cb_var, &table );
         vis5d_color_table_init_params( p, 1, 0 );
         vis5d_color_table_recompute( table, 256, p, 1, 0 );
         vis5d_signal_redraw( index, 1 );
         LUI_ColorBarRedraw( cb );
      }
      else if (action==LUI_ALPHA_RESET) {
         unsigned int *table;
         int opacity;
         vis5d_get_color_table_params(index, VIS5D_CVSLICE, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_CVSLICE, cb_var, &table );
         vis5d_color_table_init_params( p, 0, 1 );
         vis5d_color_table_recompute( table, 256, p, 0, 1 );
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_RGB_SHAPE) {
         /* change shape of rgb curves */
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_CVSLICE, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_CVSLICE, cb_var, &table );
         vis5d_color_table_recompute( table, 256, p, 1, 0 );
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_ALPHA_SHAPE) {
         /* change shape of alpha curve */
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_CVSLICE, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_CVSLICE, cb_var, &table );
         vis5d_color_table_recompute( table, 256, p, 0, 1 );
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_RGB_CHANGE) {
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_ALPHA_CHANGE) {
         vis5d_signal_redraw( index, 1 );
      }
   }
   else if (cb_graphic==VIS5D_VOLUME) {
      if (action==LUI_RGB_RESET) {
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_VOLUME, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_VOLUME, cb_var, &table );
         vis5d_color_table_init_params( p, 1, 0 );
         vis5d_color_table_recompute( table, 256, p, 1, 0);
         vis5d_signal_redraw( index, 1 );
         LUI_ColorBarRedraw( cb );
      }
      else if (action==LUI_ALPHA_RESET) {
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_VOLUME, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_VOLUME, cb_var, &table );
         vis5d_color_table_set_alpha( p, -1 );
         vis5d_color_table_init_params( p, 0, 1 );
         vis5d_color_table_recompute( table, 256, p, 0, 1);
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_RGB_SHAPE) {
         /* change shape of rgb curves */
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_VOLUME, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_VOLUME, cb_var, &table );
         vis5d_color_table_recompute( table, 256, p, 1, 0);
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_ALPHA_SHAPE) {
         /* change shape of alpha curve */
         unsigned int *table;
         vis5d_get_color_table_params(index, VIS5D_VOLUME, cb_var, &p);
         vis5d_get_color_table_address( index, VIS5D_VOLUME, cb_var, &table );
         vis5d_color_table_recompute( table, 256, p, 0, 1);
         LUI_ColorBarRedraw( cb );
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_RGB_CHANGE) {
         vis5d_signal_redraw( index, 1 );
      }
      else if (action==LUI_ALPHA_CHANGE) {
         /* TODO: update opacity */
         vis5d_signal_redraw( index, 1 );
      }

   }

   return 0;
}



/*** These functions are called when the F1, F2, F3, etc. function ***/
/*** keys are called.  You can put whatever you want in them to    ***/
/*** make macros, shortcuts, etc.                                  ***/

static void func1( int index )
{
   GuiContext gtx = get_gui_gtx(index);

   if (gtx->widget_enable) {
      gtx->widget_enable = 0;
      /* raise 3-D window */
      XRaiseWindow( GfxDpy, gtx->mainwin );
   }
   else {
      gtx->widget_enable = 1;
      /* raise control panel window */
      XRaiseWindow( GuiDpy, gtx->CpWindow );
   }
}

static void func2( int index )
{
   /* toggle info display */
   info_cb(index);
   vis5d_invalidate_frames(index);
}

static void func3( int index )
{
   GuiContext gtx = get_gui_gtx(index);
   gtx->GoTime = (gtx->GoTime == 0) ? 1 : 0; /* toggle animation */
   return;
}

static void func4( int index )
{
}

static void func5( int index )
{
}

static void func6( int index )
{
}

static void func7( int index )
{
}

static void func8( int index )
{
}

static void func9( int index )
{
}



/**********************************************************************/
/*****                   Event Handling                           *****/
/**********************************************************************/



/*
 * Handle an input event in the 3-D graphics window.
 * Input:  index - context index
 *         ev - the X event
 */
static int gfx_window_event( int index, XEvent *ev )
{
   int  current_x, current_y;
   float  anglex, angley, scale, transx, transy;
   MATRIX  temp;
   MATRIX  mat;
   int i, r;
   Window window;
   int WinWidth, WinHeight;
   int CurTime;
   float CursorX, CursorY, CursorZ;
   int Uvar, Vvar, Wvar, Uvar2, Vvar2, Wvar2, TrajU, TrajV, TrajW;
   int NumTimes;

   static int perspec;
   static float frontclip, zoom;


   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_window( index, &window, &WinWidth, &WinHeight );
   vis5d_get_timestep(index, &CurTime);
   vis5d_get_cursor(index, &CursorX, &CursorY, &CursorZ);
   vis5d_get_wind_vars(index, &Uvar, &Vvar, &Wvar,
                       &Uvar2, &Vvar2, &Wvar2, &TrajU, &TrajV, &TrajW);
   vis5d_get_numtimes( index, &NumTimes );


/* Iconification... */
#ifdef LEAVEOUT
   if (ev->type==UnmapNotify
       && (ev->xany.window==gtx->mainwin || ev->xany.window==gtx->CpWindow)) {
      /* Go to sleep if iconified */
      iconify(index);
      do {
         if (XPending(GfxDpy))
           XNextEvent(GfxDpy, ev);
         else
           XNextEvent(GuiDpy, ev);
      } while (ev->type!=MapNotify);
      deiconify(index);
   }
#endif


   if (ev->xany.window==gtx->fakewin) {
      /* fudge the mouse coordinates in the event struct */
      if (ev->type==ButtonPress || ev->type==ButtonRelease) {
         ev->xbutton.x = ev->xbutton.x * WinWidth / gtx->fakewidth;
         ev->xbutton.y = ev->xbutton.y * WinHeight / gtx->fakeheight;
      }
      else if (ev->type==MotionNotify) {
         ev->xmotion.x = ev->xmotion.x * WinWidth / gtx->fakewidth;
         ev->xmotion.y = ev->xmotion.y * WinHeight / gtx->fakeheight;
         vis5d_set_pointer(index, ev->xmotion.x, ev->xmotion.y);
         vis5d_signal_redraw(index, 1);
      }
   }

   /* Check for mouse-mode-specific events first */
   if (gtx->MouseMode==MM_SLICE && slice_event(index, *ev, CurTime)) {
      vis5d_graphics_mode(index, VIS5D_PRETTY, VIS5D_OFF);
      vis5d_invalidate_frames(index);
   }
   else if (gtx->MouseMode==MM_LABEL && labeling_event(index, *ev) ) {
      vis5d_graphics_mode(index, VIS5D_PRETTY, VIS5D_OFF);
      vis5d_invalidate_frames(index);
   }
   else if (gtx->MouseMode==MM_TRAJ && (r=cursor_event(index, *ev))) {
      if (r==2) {
         /* convert cursor position from (x,y,z) to (r,c,l) */
         float cursor_r, cursor_c, cursor_l;
         vis5d_xyz_to_grid( index, CurTime, Uvar, CursorX, CursorY, CursorZ,
                            &cursor_r, &cursor_c, &cursor_l );
         if (gtx->GoTime) {
            for (i=0;i<NumTimes;i++) {
               vis5d_make_traj( index, cursor_r, cursor_c, cursor_l,
                                i, gtx->cur_trajset );
            }
         }
         else {
            /* one traj */
            vis5d_make_traj( index, cursor_r, cursor_c, cursor_l,
                             CurTime, gtx->cur_trajset );
         }
      }
      vis5d_graphics_mode(index, VIS5D_PRETTY, VIS5D_OFF);
      vis5d_invalidate_frames(index);
   }
   else if (gtx->MouseMode==MM_PROBE && cursor_event(index, *ev)==1) {
      vis5d_graphics_mode(index, VIS5D_PRETTY, VIS5D_OFF);
      vis5d_signal_redraw(index, 1);
   } 
   else if (gtx->MouseMode==MM_SOUND && cursor_event(index, *ev)==1) {
      vis5d_graphics_mode(index, VIS5D_PRETTY, VIS5D_OFF);
      vis5d_signal_redraw(index, 1); 
   }
      
   /*** Button press ***/
   else if (ev->type == ButtonPress) {
      gtx->start_x = ev->xbutton.x;
      gtx->start_y = ev->xbutton.y;
      vis5d_get_matrix(index, gtx->ctm);
      vis5d_get_camera( index, &perspec, &frontclip, &zoom );

      if (ev->xbutton.button==Button1 && !gtx->p2 && !gtx->p3) {
         gtx->p1 = 1;
      }
      else if (ev->xbutton.button==Button2 && !gtx->p1 && !gtx->p3) {
         gtx->p2 = 1;
      }
      else if (ev->xbutton.button==Button3 && !gtx->p1 && !gtx->p2) {
         gtx->p3 = 1;
      }
      vis5d_graphics_mode(index, VIS5D_PRETTY, VIS5D_OFF);
      vis5d_invalidate_frames(index);
   } /* if ev->type==ButtonPress */

   /*** Button down and pointer moving ***/
   else if (ev->type == MotionNotify) {
      current_x = ev->xmotion.x;
      current_y = ev->xmotion.y;
      if (gtx->p1) {
         /* rotate */
         angley = (current_x-gtx->start_x) * 200.0 / (float) WinWidth;
         anglex = (current_y-gtx->start_y) * 200.0 / (float) WinHeight;
         make_matrix( anglex,angley,0.0, 1.0, 0.0,0.0,0.0, mat );
         mat_mul( temp, gtx->ctm, mat );
         vis5d_set_matrix(index, temp);
      }
      else if (gtx->p2) {
         /* clip */
         float zdelt = (gtx->start_x - current_x) / (float) WinWidth;
         float newfront = frontclip - zdelt;
         float scale = exp( (gtx->start_y-current_y) / (float) WinHeight );
         float newzoom = zoom * scale;
         vis5d_set_camera( index, perspec, newfront, newzoom );
      }
      else if (gtx->p3) {
         /* translate */
         transx = (gtx->start_x - current_x) * -2.0 / (float) WinWidth;
         transy = (gtx->start_y - current_y) * 2.0 / (float) WinHeight;
         make_matrix( 0.0,0.0,0.0, 1.0, transx,transy,0.0, mat );
         mat_mul( temp, gtx->ctm, mat );
         vis5d_set_matrix(index, temp);
      }
   } /* if ev->type==MotionNotify */
      
   /*** Button release ***/
   else if (ev->type == ButtonRelease) {
      if (ev->xbutton.button == Button1) {
         gtx->p1 = 0;
      }
      else if (ev->xbutton.button == Button2) {
         gtx->p2 = 0;
      }
      else if (ev->xbutton.button == Button3) {
         gtx->p3 = 0;
      }
      vis5d_signal_redraw(index, 1);
   } /* if ev->type==ButtonRelease */
      
   /*** Key press ***/
   else if (ev->type == KeyPress) {
      int insoundmode;
      KeySym keysym;
      XComposeStatus compose;
      char buff[20];
      XLookupString( &ev->xkey, buff, 4, &keysym, &compose );
      if (buff[0]=='p' || buff[0]=='P') {
         print_cb( index );
      }
      else if (buff[0]=='f' || buff[0]=='F') {
         gtx->AnimRate = gtx->AnimRate - 10;
         if (gtx->AnimRate < 1) gtx->AnimRate = 1;
      }
      else if (buff[0]=='s' || buff[0]=='S') {
         gtx->AnimRate = gtx->AnimRate + 10;
      }
      else if (keysym == XK_F1) func1(index);
      else if (keysym == XK_F2) func2(index);
      else if (keysym == XK_F3) func3(index);
      else if (keysym == XK_F4) func4(index);
      else if (keysym == XK_F5) func5(index);
      else if (keysym == XK_F6) func6(index);
      else if (keysym == XK_F7) func7(index);
      else if (keysym == XK_F8) func8(index);
      else if (keysym == XK_F9) func9(index);
   }

   /*** Window size ***/
   else if (ev->type==ConfigureNotify) {
      if (ev->xany.window==gtx->fakewin) {
         /* update fake window size */
         gtx->fakewidth = ev->xconfigure.width;
         gtx->fakeheight = ev->xconfigure.height;
      }
      if (ev->xany.window==gtx->SoundCtrlWindow){
         /* update sounding window */

         if (gtx->othersnddpy != 0){
            vis5d_resize_sounding_window( index, ev->xconfigure.width,
                                          ev->xconfigure.height,
                                          ev->xconfigure.x,
                                          ev->xconfigure.y);
         }
         else {
            vis5d_resize_sounding_window( index, ev->xconfigure.width,
                                          ev->xconfigure.height, 0, 95 );
         }
      }
      else {
         /* 3-D viewing window has changed size */
         vis5d_resize_3d_window( index, ev->xconfigure.width,
                                        ev->xconfigure.height);
         vis5d_invalidate_frames(index);
      }
   }

   /*** Window visibility ***/
   else if (ev->type==Expose || ev->type==VisibilityNotify
            || ev->type==GraphicsExpose) {
      vis5d_signal_redraw(index, 2);
   }

   /*** Flush remaining motion events in event queue ***/
   while (QLength(GfxDpy) > 0) {
      XEvent ne;
      XPeekEvent(GfxDpy, &ne);
      /* Don't flush if its a button press or release */
      if (ne.type == ButtonPress || ne.type == ButtonRelease ||
          ne.xany.window != gtx->mainwin)
        break;

      /* Get the next event */
      XNextEvent(GfxDpy, &ne);
   }
   if (GfxDpy!=GuiDpy) {
      while (QLength(GuiDpy) > 0) {
         XEvent ne;
         XPeekEvent(GuiDpy, &ne);
         /* Don't flush if its a button press or release */
         if (ne.type == ButtonPress || ne.type == ButtonRelease ||
             ne.xany.window != gtx->fakewin)
           break;
         /* Get the next event */
         XNextEvent(GuiDpy, &ne);
      }
   }
   return 1;
}



/*
 * Wait or poll for an X event, process the event.
 * Input:  block - 1 = wait for event, 0 = poll for event
 * Return:  0 = there was no input
 *          1 = there was a an input event
 *          2 = there was a LUI or Color Widget event
 */
int get_user_input( int block )
{
   XEvent ev;
   int index;
   int result;

   if (GuiDpy==GfxDpy && block) {
      XNextEvent( GuiDpy, &ev );      /* blocking wait */
   }
   else if (XPending(GuiDpy)) {       /* poll */
      XNextEvent( GuiDpy, &ev );
   }
   else if (XPending(GfxDpy)) {       /* poll */
      XNextEvent( GfxDpy, &ev );
   }
   else {
      return 0;
   }

   /*
    * Determine which context the event belongs to by searching
    * the context list.
    */
   for (index=0; index<VIS5D_MAX_CONTEXTS; index++) {
      if (gtx_table[index]) {
         if (ev.xany.window == gtx_table[index]->mainwin 
             || ev.xany.window == gtx_table[index]->SoundCtrlWindow
#ifdef DENALI
             || ev.xany.window==gtx_table[index]->NPGLwin
#endif
             || ev.xany.window==gtx_table[index]->fakewin){ 
            return gfx_window_event( index, &ev );
         }
      }
   }

   /* probably a LUI event */
   if (ev.type==ButtonPress) {
      /*vis5d_graphics_mode(index, VIS5D_PRETTY, VIS5D_OFF);*/
   }
   LUI_EventDispatch( &ev );
   return 2;
}




/**********************************************************************/
/*****                  GUI Construction                          *****/
/**********************************************************************/


/*
 * Make the graphics save window.
 */
static void make_save_window( int index )
{
   GuiContext gtx = get_gui_gtx(index);
   LUI_NEWBUTTON *b;
   Window w;

   w = LUI_CreateWindowAt( LUI_RootWindow, 450, 250, 239, 100 );
   gtx->SaveWindow = w;

   LUI_NewLabelCreate( w, LUI_LEFT, LUI_TOP, 280, 30,
                       "Save settings as..." );

   gtx->SaveNameField = LUI_FieldCreate( w, LUI_LEFT, LUI_NEXT_Y, 233, 28 );

   b = LUI_PushButtonCreate( w, LUI_LEFT, LUI_NEXT_Y, 115, 25, "Save" );
   LUI_ButtonCallback( b, save_ok_cb );
   b = LUI_PushButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 115, 25, "Cancel" );
   LUI_ButtonCallback( b, save_cancel_cb );
}



/*
 * Make the graphics restore window.
 */
static void make_restore_window( int index )
{
   GuiContext gtx = get_gui_gtx(index);
   LUI_NEWBUTTON *b;
   Window w;

   w = LUI_CreateWindowAt( LUI_RootWindow, 450, 250, 239, 100 );
   gtx->RestoreWindow = w;

   LUI_NewLabelCreate( w, LUI_LEFT, LUI_TOP, 280, 30,
                       "Restore settings from..." );
   gtx->RestoreNameField = LUI_FieldCreate( w, LUI_LEFT, LUI_NEXT_Y, 233, 28 );

   b = LUI_PushButtonCreate( w, LUI_LEFT, LUI_NEXT_Y, 115, 25, "OK" );
   LUI_ButtonCallback( b, restore_ok_cb );
   b = LUI_PushButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 115, 25, "Cancel" );
   LUI_ButtonCallback( b, restore_cancel_cb );
}



/*
 * Make the pop-up window used for entering/editing expressions.
 */
static void make_expression_window( int index )
{
   LUI_NEWBUTTON *b;
   GuiContext gtx = get_gui_gtx(index);

   gtx->ExprWindow = LUI_CreateWindowAt( LUI_RootWindow, 450, 450,
                                         406, 26*3+3*4 );

   LUI_NewLabelCreate( gtx->ExprWindow, LUI_LEFT, LUI_TOP, 400, 26,
                       "Enter/edit expression:" );
   gtx->expr_field = LUI_FieldCreate( gtx->ExprWindow, LUI_LEFT, LUI_NEXT_Y,
                                      400, 26 );
   LUI_FieldCallback( gtx->expr_field, expr_cb );

   b = LUI_PushButtonCreate( gtx->ExprWindow, LUI_LEFT, LUI_NEXT_Y,
                             100, 26, "OK" );
   LUI_ButtonCallback( b, expr_ok_cb );
   b = LUI_PushButtonCreate( gtx->ExprWindow, LUI_NEXT_X, LUI_SAME_Y,
                             100, 26, "Cancel" );
   LUI_ButtonCallback( b, expr_cancel_cb );
}




/*
 * Make the verify window
 */
static void make_verify_window( int index )
{
   LUI_NEWBUTTON *b;
   GuiContext gtx = get_gui_gtx(index);

   gtx->VerifyWindow = LUI_CreateWindowAt( LUI_RootWindow, 600, 400, 300, 90 );

   gtx->VerifyLabel = LUI_NewLabelCreate( gtx->VerifyWindow, LUI_LEFT, LUI_TOP,
                                     300, 55, "nil" );

   b = LUI_PushButtonCreate( gtx->VerifyWindow, 33, LUI_NEXT_Y, 100,25, "OK");
   LUI_ButtonCallback( b, ok_cb );

   b = LUI_PushButtonCreate( gtx->VerifyWindow, 166, LUI_SAME_Y,
                             100,25, "Cancel");
   LUI_ButtonCallback( b, cancel_cb );
}



/*
 * Make the alert window.
 */
static void make_alert_window( int index )
{
   LUI_NEWBUTTON *b;
   GuiContext gtx = get_gui_gtx(index);

   gtx->AlertWindow = LUI_CreateWindowAt( LUI_RootWindow, 600, 400, 600, 100 );

   gtx->AlertLabel = LUI_NewLabelCreate( gtx->AlertWindow, LUI_LEFT, LUI_TOP,
                                    600, 60, "nil" );

   b = LUI_PushButtonCreate( gtx->AlertWindow, 100, LUI_NEXT_Y, 100, 25, "OK");
   LUI_ButtonCallback( b, ok_cb );
}



/*
 * Make the trajectory window.
 */
static void make_trajectory_window( int index )
{
   int x, y, w, h, i;
   char label[1000];
   LUI_NEWBUTTON *b;
   int height;
   float UserTrajStep, UserTrajLength;
   int RibbonFlag;
   GuiContext gtx = get_gui_gtx(index);

   vis5d_get_traj(index, &UserTrajStep, &UserTrajLength, &RibbonFlag);

   gtx->TrajWindow = LUI_CreateWindowAt(LUI_RootWindow, 405, 830, 300, 150 );

   /* Create the user trajectories title label. */
   LUI_NewLabelCreate( gtx->TrajWindow, LUI_LEFT, LUI_TOP, 300, 36,
     "   Interactive Wind Trajectories\n LFT-Display, CNT-Select, RT-Color");

   /* Trajectory set buttons */
   LUI_ButtonPadOpen ("gtx->TrajButtons",   /* name */
                 gtx->TrajWindow,           /* parent window */
                 0, 40,       /* origin relative to parent */
                 BW,                   /* border size */
                 0x000000              /* background color */        
                 );
   for (i=0;i<VIS5D_TRAJ_SETS;i++) {
      float r, g, b, a;
      vis5d_get_color( index, VIS5D_TRAJ, i, &r, &g, &b, &a );
      sprintf(label, "Set %d", i+1 );
      gtx->TrajButton[i] = LUI_ButtonCreate( label, LUI_BUTTON_TOGGLE,
                                        i%4, i/4, 72, i, (LUI_FNCP) traj_cb );
      LUI_ButtonSetColor( gtx->TrajButton[i], (int)(r*255.0), (int)(g*255.0),
                          (int)(b*255.0) );
      if (vis5d_enable_graphics(index, VIS5D_TRAJ, i, VIS5D_GET)) {
         LUI_ButtonState( gtx->TrajButton[i], 1 );
      }
   }
   LUI_ButtonPadClose("gtx->TrajButtons");
   LUI_ButtonPadVisible("gtx->TrajButtons", 1);
   LUI_ButtonPadQuery("gtx->TrajButtons", &x, &y, &w, &h);
   height = y + h;

   /* Traj Step */
   LUI_NewLabelCreate( gtx->TrajWindow, BW, height, 50, 25, "Step:" );
   gtx->TrajStepField = LUI_FieldCreate( gtx->TrajWindow, LUI_NEXT_X, LUI_SAME_Y, 80,25);
   LUI_FieldSetDouble( gtx->TrajStepField, UserTrajStep );
   LUI_FieldCallback( gtx->TrajStepField, trajstep_cb ); 

   /* Traj Length */
   LUI_NewLabelCreate( gtx->TrajWindow, LUI_NEXT_X, LUI_SAME_Y, 65, 25, "Length:" );
   gtx->TrajLenField = LUI_FieldCreate( gtx->TrajWindow, LUI_NEXT_X, LUI_SAME_Y, 80,25 );
   LUI_FieldSetDouble( gtx->TrajLenField, UserTrajLength );
   LUI_FieldCallback( gtx->TrajLenField, trajlength_cb ); 

   gtx->TrajRibbonButton = LUI_CheckButtonCreate( gtx->TrajWindow, LUI_LEFT, LUI_NEXT_Y,
                               70, 25, "Ribbon");
   LUI_ButtonCallback( gtx->TrajRibbonButton, ribbon_cb );

   b = LUI_PushButtonCreate( gtx->TrajWindow, LUI_NEXT_X, LUI_SAME_Y,
                             110, LUI_SAME_H, "Delete Last" );
   LUI_ButtonCallback( b, dellast_cb );

   b = LUI_PushButtonCreate( gtx->TrajWindow, LUI_NEXT_X, LUI_SAME_Y,
                             105, LUI_SAME_H, "Delete Set" );
   LUI_ButtonCallback( b, delset_cb );

   XUnmapWindow(GuiDpy, gtx->TrajWindow );
}


/*
 * Create all the LUI widgets.
 * Input:  volflag - if non-zero, add a 6th column for volume rendering.
 */
static void create_widgets( int index, int volflag, char *programname )
{
   Window window,sndwindow;
   int WinWidth, WinHeight;
   int NumTimes, NumVars;
   int Nr, Nc, Nl[MAXVARS], LowLev[MAXVARS], MaxNl, MaxNlVar, WindNl, WindLow;
   int Uvar, Vvar, Wvar, Uvar2, Vvar2, Wvar2, TrajU, TrajV, TrajW;
   int topoflag, mapflag, textureflag;
   float MinTopoHgt, MaxTopoHgt;
   char VarName[MAXVARS][10];
   int x, y, w, h, i, j, bottom;
   LUI_BUTTON *lb;
   GuiContext gtx = get_gui_gtx(index);
   LUI_NEWBUTTON *b;

   vis5d_get_numtimes( index, &NumTimes );
   vis5d_get_numvars( index, &NumVars );
   vis5d_get_size(index, &Nr, &Nc, Nl, LowLev, &MaxNl, &MaxNlVar, &WindNl, &WindLow);
   vis5d_get_wind_vars(index, &Uvar, &Vvar, &Wvar,
                       &Uvar2, &Vvar2, &Wvar2, &TrajU, &TrajV, &TrajW);
   vis5d_check_topo( index, &topoflag );
   vis5d_check_map( index, &mapflag );
   vis5d_check_texture( index, &textureflag );
   for (i=0;i<NumVars;i++) {
      vis5d_get_var_name( index, i, VarName[i] );
   }
   vis5d_get_topo_range(index, &MinTopoHgt, &MaxTopoHgt);

   LUI_LayoutGutter( 3 );
   LUI_Initialize( programname, GuiDpy, GuiVisual, GuiDepth, GuiColormap );

   vis5d_get_window( index, &window, &WinWidth, &WinHeight );
   gtx->mainwin = window;


   if (GuiDpy==GfxDpy) {
      LUI_SetMainWindow( gtx->mainwin );
   }

   /* Create control panel window */
   gtx->CpWindow = LUI_CreateWindowAt( LUI_RootWindow, /* parent */
                        gtx->cpx, gtx->cpy, /* position */ 100, 100 /* temporary size */
               );
   gtx->CpHeight = 0;

   /******************************************************************/
   /* Create the copyright label.                                    */
   /******************************************************************/
   LUI_LabelOpen ( "Copyright",      /* name */
             3, Copyright,           /* text strings */
             gtx->CpWindow,               /* parent window */
             0, 0,                   /* origin relative to parent */
             CP_WIDTH, 50,           /* dimensions */
             BW,                     /* border size */
             NULL                    /* callback function */
             );
   LUI_LabelClose("Copyright");
   LUI_LabelVisible("Copyright", 1);
   LUI_LabelHighlight("Copyright", 0);
   LUI_LabelQuery("Copyright", &x, &y, &w, &h);
   gtx->CpHeight += h;


   /************************************************************/
   /* Create the main button pad.                              */
   /************************************************************/
   LUI_ButtonPadOpen ("Buttons", /* name */
                 gtx->CpWindow,       /* parent window */
                 0, gtx->CpHeight,    /* origin relative to parent */
                 BW,             /* border size */
                 0x000000        /* background color */
                 );

   /** Row 1 **/
   if (NumTimes>1) {
      LUI_ButtonCreate("ANIMATE",LUI_BUTTON_TOGGLE, 0,0,BUTTONSIZE,0, (LUI_FNCP) anim_cb);
      LUI_ButtonCreate("STEP",   LUI_BUTTON_ONESHOT,1,0,BUTTONSIZE,0, (LUI_FNCP) step_cb);
   }
   else {
      LUI_ButtonCreate("", LUI_BUTTON_ONESHOT, 0,0,BUTTONSIZE,0, NULL);
      LUI_ButtonCreate("", LUI_BUTTON_ONESHOT, 1,0,BUTTONSIZE,0, NULL);
   }
   LUI_ButtonCreate("NEW VAR..",LUI_BUTTON_ONESHOT,2,0,BUTTONSIZE,0, (LUI_FNCP) newvar_cb);
   LUI_ButtonCreate("EXIT", LUI_BUTTON_ONESHOT,3,0,BUTTONSIZE,0, (LUI_FNCP) exit_cb);

   /** Row 2 **/
   if (textureflag) {
      LUI_ButtonCreate("TEXTURE",LUI_BUTTON_TOGGLE,0,1,BUTTONSIZE,0,(LUI_FNCP) texture_cb);
   }
   else {
      LUI_ButtonCreate("",LUI_BUTTON_ONESHOT,0,1,BUTTONSIZE,0, NULL);
   }
   LUI_ButtonCreate("TOP",    LUI_BUTTON_ONESHOT,1,1,BUTTONSIZE,1, (LUI_FNCP) view_cb);
   LUI_ButtonCreate("SOUTH",  LUI_BUTTON_ONESHOT,2,1,BUTTONSIZE,2, (LUI_FNCP) view_cb);
   LUI_ButtonCreate("WEST",   LUI_BUTTON_ONESHOT,3,1,BUTTONSIZE,3, (LUI_FNCP) view_cb);

   /** Row 3 **/
   if (topoflag) {
      LUI_ButtonCreate("TOPO",    LUI_BUTTON_TOGGLE,0,2,BUTTONSIZE,0,
                       (LUI_FNCP) topo_cb);
   }
   else {
      /* no topo button */
      LUI_ButtonCreate("",    LUI_BUTTON_ONESHOT,0,2,BUTTONSIZE,0, NULL);
   }
   if (mapflag) {
      gtx->map_button = LUI_ButtonCreate("MAP", LUI_BUTTON_TOGGLE,1,2,
                                          BUTTONSIZE,0, (LUI_FNCP) map_cb);
   }
   else {
      /* no map button */
      LUI_ButtonCreate("",    LUI_BUTTON_ONESHOT,1,2,BUTTONSIZE,0, NULL);
   }

   lb=LUI_ButtonCreate("BOX",  LUI_BUTTON_TOGGLE,2,2,BUTTONSIZE,0,
                        (LUI_FNCP) box_cb);
   LUI_ButtonState( lb, 1 );
   lb=LUI_ButtonCreate("CLOCK", LUI_BUTTON_TOGGLE,3,2,BUTTONSIZE,0,
                        (LUI_FNCP) clock_cb);
   LUI_ButtonState( lb, 1 );


   /** Row 4 **/
   LUI_ButtonCreate("SAVE..",   LUI_BUTTON_ONESHOT,0,3,BUTTONSIZE,0, (LUI_FNCP) save_cb);
   LUI_ButtonCreate("RESTORE",LUI_BUTTON_ONESHOT,1,3,BUTTONSIZE,0, (LUI_FNCP) restore_cb);
   LUI_ButtonCreate("GRID #'s",LUI_BUTTON_TOGGLE,2,3,BUTTONSIZE,0, (LUI_FNCP) coord_cb);
   lb = LUI_ButtonCreate("CONT #'s",LUI_BUTTON_TOGGLE,3,3,BUTTONSIZE,0, (LUI_FNCP) contnum_cb);
   LUI_ButtonState( lb, 1 );


   /** Row 5 **/
#if defined(MESA)
   LUI_ButtonCreate("ANIM-REC", LUI_BUTTON_TOGGLE,0,4,BUTTONSIZE,0,(LUI_FNCP) animrec_cb);
#else
   LUI_ButtonCreate("", LUI_BUTTON_ONESHOT,0,4,BUTTONSIZE,0,NULL);
#endif
   LUI_ButtonCreate("REVERSE",LUI_BUTTON_TOGGLE, 1,4,BUTTONSIZE,0,(LUI_FNCP) reverse_cb);
   if (save_formats()>0) {
      LUI_ButtonCreate("SAVE PIC",LUI_BUTTON_ONESHOT,2,4,
                       BUTTONSIZE, 0, (LUI_FNCP) savepic_cb);
   }
   else {
      LUI_ButtonCreate("",LUI_BUTTON_ONESHOT,2,4,BUTTONSIZE,0, NULL);
   }
   if (Perspec_available) {
      gtx->perspec_button = LUI_ButtonCreate("PERSPEC", LUI_BUTTON_TOGGLE,3,4,BUTTONSIZE,0,(LUI_FNCP) proj_cb);
   }
   else {
      gtx->perspec_button = NULL;
      LUI_ButtonCreate("", LUI_BUTTON_ONESHOT, 3,4,BUTTONSIZE,0, NULL );
   }

   /** Row 5 **/
   LUI_ButtonCreate("SCRIPT..", LUI_BUTTON_ONESHOT, 0,5,BUTTONSIZE,0, (LUI_FNCP) script_cb );
   LUI_ButtonCreate("INTERP..", LUI_BUTTON_ONESHOT, 1,5,BUTTONSIZE,0, (LUI_FNCP) interpret_cb );
   LUI_ButtonCreate("UVW VARS..", LUI_BUTTON_ONESHOT, 2,5,BUTTONSIZE,0, (LUI_FNCP) uvw_cb );
   LUI_ButtonCreate("LEGENDS", LUI_BUTTON_TOGGLE, 3,5,BUTTONSIZE,0, (LUI_FNCP) legend_cb );

   LUI_ButtonPadClose("Buttons");
   LUI_ButtonPadVisible("Buttons", 1);
   LUI_ButtonPadQuery("Buttons", &x, &y, &w, &h);
   gtx->CpHeight += h;


   /******************************************************************/
   /* Create the radio buttons.                                    ***/
   /******************************************************************/
   LUI_ButtonPadOpenType( "Modes",  /* name */
         gtx->CpWindow,             /* parent window */
         LUI_BUTTON_RADIO,          /* type */
         0, gtx->CpHeight,          /* position */
         BW,                        /* border size */
         0x000000                   /* background color */
       );
   lb = LUI_ButtonCreateType("Normal", 0,0, RADIOSIZE, MM_NORMAL, (LUI_FNCP) mode_cb);
   LUI_ButtonState( lb, 1 );

   if (NumTimes>1) {
      LUI_ButtonCreateType( "Trajectory", 0,1, RADIOSIZE, MM_TRAJ,
                            (LUI_FNCP) mode_cb );
   }
   else {
      LUI_ButtonCreateType( " ", 0,1, RADIOSIZE,MM_TRAJ,  NULL );
   }
   LUI_ButtonCreateType("Slice", 0,2, RADIOSIZE,MM_SLICE, (LUI_FNCP) mode_cb );
   LUI_ButtonCreateType("Label", 0,3, RADIOSIZE,MM_LABEL, (LUI_FNCP) mode_cb );
   /* Comment out the next statement to remove "Probe" option. */
   LUI_ButtonCreateType("Probe", 0,4, RADIOSIZE,MM_PROBE, (LUI_FNCP) mode_cb );
   LUI_ButtonCreateType("Sounding", 0,5, RADIOSIZE,MM_SOUND, (LUI_FNCP) mode_cb );
   LUI_ButtonPadClose( "Modes" );
   LUI_ButtonPadVisible( "Modes", 1 );
   LUI_ButtonPadQuery( "Modes", &x, &y, &w, &h );

   gtx->ModeInfoLabel = LUI_LabelOpen ( "ModeInfo",   /* name */
             6, ModeInfo1,       /* text strings */
             gtx->CpWindow,           /* parent window */
             w, gtx->CpHeight,        /* origin relative to parent */
             CP_WIDTH-w, h,      /* dimensions */
             BW,                 /* border size */
             NULL                /* callback function */
             );
   LUI_LabelClose("ModeInfo");
   LUI_LabelVisible("ModeInfo", 1);
   LUI_LabelHighlight("ModeInfo", 1);
   LUI_LabelQuery("ModeInfo", &x, &y, &w, &h);
   gtx->CpHeight += h;


   /***********************************************************/
   /* Make wind slice buttons                                 */
   /***********************************************************/
   {
      Window w = gtx->CpWindow;
      LUI_NEWBUTTON *b;
      float red, green, blue, alpha;

      LUI_FrameWidth( 2 );

      b = LUI_ToggleButtonCreate( w, 3, gtx->CpHeight, 58, 22, "Hwind1" );
      LUI_ButtonCallback( b, hwind_cb );
      LUI_ButtonIndex( b, 0 );
      vis5d_get_color( index, VIS5D_HWIND, 0, &red, &green, &blue, &alpha );
      LUI_ButtonColor( b, red, green, blue );
      gtx->hwind_button[0] = b;

      b = LUI_ToggleButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 58, 22, "Vwind1");
      LUI_ButtonCallback( b, vwind_cb );
      LUI_ButtonIndex( b, 0 );
      vis5d_get_color( index, VIS5D_VWIND, 0, &red, &green, &blue, &alpha );
      LUI_ButtonColor( b, red, green, blue );
      gtx->vwind_button[0] = b;

      b = LUI_ToggleButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 65, 22,"HStream");
      LUI_ButtonCallback( b, hstream_cb );
      LUI_ButtonIndex( b, 0 );
      vis5d_get_color( index, VIS5D_HSTREAM, 0, &red, &green, &blue, &alpha );
      LUI_ButtonColor( b, red, green, blue );
      gtx->hstream_button[0] = b;

      b = LUI_ToggleButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 58, 22, "Hwind2");
      LUI_ButtonCallback( b, hwind_cb );
      LUI_ButtonIndex( b, 1 );
      vis5d_get_color( index, VIS5D_HWIND, 1, &red, &green, &blue, &alpha );
      LUI_ButtonColor( b, red, green, blue );
      gtx->hwind_button[1] = b;

      b = LUI_ToggleButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 58, 22, "Vwind2");
      LUI_ButtonCallback( b, vwind_cb );
      LUI_ButtonIndex( b, 1 );
      vis5d_get_color( index, VIS5D_VWIND, 1, &red, &green, &blue, &alpha );
      LUI_ButtonColor( b, red, green, blue );
      gtx->vwind_button[1] = b;

      b = LUI_ToggleButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 65, 22,"VStream");
      LUI_ButtonCallback( b, vstream_cb );
      LUI_ButtonIndex( b, 0 );
      vis5d_get_color( index, VIS5D_VSTREAM, 0, &red, &green, &blue, &alpha );
      LUI_ButtonColor( b, red, green, blue );
      gtx->vstream_button[0] = b;
   }


   /**************************************************/
   /* Column headings                                */
   /**************************************************/

   if (volflag) {
      /* 6 column headings */
      LUI_NewLabelCreate( gtx->CpWindow, 0, LUI_NEXT_Y, 380, 28,
                          "           Contour Slice  Colored Slice\n"
                          "  Isosurf  Horiz.  Vert.  Horiz.  Vert.  Volume" );
   }
   else {
      /* 5 column headings */
      LUI_NewLabelCreate( gtx->CpWindow, 0, LUI_NEXT_Y, 380, 28,
                          "             Contour Slices     Colored Slices\n"
                          "   Isosurf   Horiz.    Vert.    Horiz.   Vert." );
   }

   /**************************************************/
   /* Button Matrix                                  */
   /**************************************************/
   gtx->Columns = volflag ? 6 : 5;
   gtx->ButtonMatrixTop = LUI_LayoutGet( LUI_NEXT_Y );
   gtx->ButtonMatrixWidth = 378;
   gtx->ButtonMatrixHeight = 172;
   gtx->ButtonMatrix = LUI_ButtonMatrixCreate( gtx->CpWindow, 3, LUI_NEXT_Y,
                            gtx->ButtonMatrixWidth, gtx->ButtonMatrixHeight,
                            gtx->Columns );

   for (i=0; i<NumVars; i++) {
      float red[6], green[6], blue[6], alpha;
      char *labels[6];
      int type;
      for (j=0; j<gtx->Columns; j++) {
         if (j==5) {
            /* volume button always white */
            red[j] = green[j] = blue[j] = 1.0;
         }
         else {
            if (j == 0) type = VIS5D_ISOSURF;
            else if (j == 1) type = VIS5D_HSLICE;
            else if (j == 2) type = VIS5D_VSLICE;
            else if (j == 3) type = VIS5D_CHSLICE;
            else if (j == 4) type = VIS5D_CVSLICE;
            else if (j == 5) type = VIS5D_VOLUME;
            vis5d_get_color( index, type, i, &red[j], &green[j],
                             &blue[j], &alpha );
         }
         if (Nl[i]>1 || (j==1 || j==3)) {
            labels[j] = VarName[i];
         }
         else {
            labels[j] = NULL;
         }
      }
      LUI_ButtonMatrixAddRow( gtx->ButtonMatrix, labels, red, green, blue );
   }
   XMapWindow( LUI_Display, gtx->ButtonMatrix->mainwindow );
   LUI_ButtonMatrixCallback( gtx->ButtonMatrix, button_matrix_cb );
   gtx->CpHeight = gtx->ButtonMatrixTop + gtx->ButtonMatrixHeight;

   /* Finish making the control panel. */
   LUI_EventAdd ( gtx->CpWindow, ExposureMask | ButtonPressMask
                 | StructureNotifyMask , (LUI_FNCP) controlpanel_cb);
   LUI_MoveResizeWindow( gtx->CpWindow, gtx->cpx, gtx->cpy, CP_WIDTH, gtx->CpHeight );
   XMapWindow( GuiDpy, gtx->CpWindow );

   bottom = gtx->CpHeight + 2*SPACE;


   /************************************************************/
   /* Make wind slice density & scale window.                  */
   /************************************************************/

   gtx->WindWindow = LUI_CreateWindowAt(LUI_RootWindow,gtx->cpx,bottom,380,32);

   gtx->windscale_label = LUI_NewLabelCreate(gtx->WindWindow,LUI_LEFT,LUI_TOP,
                                            130, 26, "nil" );
   gtx->windscale_field = LUI_FieldCreate(gtx->WindWindow,LUI_NEXT_X,
                                          LUI_SAME_Y, 80, LUI_SAME_H );
   LUI_FieldCallback( gtx->windscale_field, windscale_cb );
   gtx->winddensity_label = LUI_NewLabelCreate( gtx->WindWindow,
                                               LUI_NEXT_X, LUI_SAME_Y,
                                               70, LUI_SAME_H, "Density:");
   gtx->winddensity_field = LUI_FieldCreate( gtx->WindWindow,
                                             LUI_NEXT_X, LUI_SAME_Y,
                                             80, LUI_SAME_H );
   LUI_FieldCallback( gtx->winddensity_field, winddensity_cb );

   XUnmapWindow( GuiDpy, gtx->WindWindow );

   bottom += /*WindHeight*/ 32 + SPACE;


   /******************************************************************/
   /* create 3-D iso-level widget(s)                                 */
   /******************************************************************/

   gtx->IsoWindow = LUI_CreateWindow( LUI_RootWindow, gtx->cpx, gtx->cpy );

   gtx->IsoSlider = LUI_NewSliderCreate( gtx->IsoWindow, 2, 2, CP_WIDTH-37 );
   LUI_NewSliderCallback( gtx->IsoSlider, isosurface_cb );

   gtx->IsosurfWidth = CP_WIDTH;
   gtx->IsosurfHeight = 40;

   b = LUI_PushButtonCreate( gtx->IsoWindow, LUI_NEXT_X, LUI_SAME_Y, 30, 36,
                             "OK");
   LUI_ButtonCallback( b, newsurf_cb );

   LUI_MoveResizeWindow( gtx->IsoWindow, gtx->cpx, bottom,
                         gtx->IsosurfWidth, gtx->IsosurfHeight );

   bottom += gtx->IsosurfHeight + SPACE;



   /******************************************************************/
   /* create horizontal contour line interval widget                 */
   /******************************************************************/
   gtx->HSliceWindow = LUI_CreateWindowAt( LUI_RootWindow, gtx->cpx,
                                           bottom, 380, 32 );

   gtx->hslice_label = LUI_NewLabelCreate( gtx->HSliceWindow, LUI_LEFT,
                                           LUI_TOP, 230, 26, "nil" );
   gtx->hslice_field = LUI_FieldCreate( gtx->HSliceWindow,
                                        LUI_NEXT_X, LUI_SAME_Y,
                                        140, LUI_SAME_H );
   LUI_FieldCallback( gtx->hslice_field, hslice_cb );

   bottom += 32 + SPACE;


   /******************************************************************/
   /* create vertical contour line interval widget                   */
   /******************************************************************/
   gtx->VSliceWindow = LUI_CreateWindowAt( LUI_RootWindow, gtx->cpx,
                                           bottom, 380, 32 );

   gtx->vslice_label = LUI_NewLabelCreate( gtx->VSliceWindow, LUI_LEFT,
                                           LUI_TOP, 230, 26, "nil" );
   gtx->vslice_field = LUI_FieldCreate( gtx->VSliceWindow, LUI_NEXT_X,
                                        LUI_SAME_Y, 140, LUI_SAME_H );
   LUI_FieldCallback( gtx->vslice_field, vslice_cb );

   bottom += /*VSliceHeight*/ 32 + SPACE;


   /***************************************************************/
   /* Make the colorbar widget window                             */
   /***************************************************************/
   if (bottom+gtx->RGBHeight+SPACE>ScrHeight)
      bottom = ScrHeight - gtx->RGBHeight - SPACE;

   gtx->Colorbar = LUI_ColorBarCreate( 0, 10, bottom, CP_WIDTH, 160 );
   LUI_ColorBarCallback( gtx->Colorbar, colorbar_cb );
   LUI_BorderWidth( 2 );
   bottom += gtx->RGBHeight + SPACE;

   make_trajectory_window(index);
   make_verify_window(index); 
   make_alert_window(index);
   make_save_window(index);
   make_restore_window(index);
   make_savepic_window(index);
   make_expression_window(index);
   make_rgb_sliders( gtx );
   make_isocolor_window( gtx );
}



int make_gui( int index, char *dataset, char *wdpy_name, int volflag )
{
   int xpos = 410;
   int ypos = 10;
   int soundy;
   int vertical;
   float xxmin, xxmax, yymin, yymax, zzmin, zzmax, junk1, junk2;
   float hgtlow, hgthigh;
   char programname[128];
   float vertargs[MAXLEVELS];
   GuiContext gtx = get_gui_gtx(index);
   XWindowAttributes winatts;

   strcpy( v5dfile, dataset );
   /* initialize values in gui_context */
   gtx->fakewin = 0;
   gtx->cpx = 10;
   gtx->cpy = 10;
   gtx->RGBWidth=CP_WIDTH;
   gtx->RGBHeight=200;
   gtx->cur_isosurf = -1;
   gtx->cur_hslice = -1;
   gtx->cur_vslice = -1;
   gtx->cur_trajset = 0;
   gtx->cur_hwind = -1;
   gtx->cur_vwind = -1;
   gtx->cur_hstream = -1;
   gtx->cur_vstream = -1;
   gtx->widget_enable = 1;
   gtx->p1 = 0;
   gtx->p2 = 0;
   gtx->p3 = 0;

   vis5d_get_vertical( index, &vertical, vertargs);
   gtx->vertsys = vertical;

   vis5d_get_box_bounds( index, &xxmin, &xxmax, &yymin, &yymax, &zzmin, &zzmax);
   vis5d_xyz_to_geo( index, 0, 0, 0.0, 0.0, zzmin, &junk1, &junk2, &hgtlow );
   vis5d_xyz_to_geo( index, 0, 0, 0.0, 0.0, zzmax, &junk1, &junk2, &hgthigh);
   if ( hgthigh < 1.0 && hgtlow < -1.0 ){
      gtx->oceanonly = 1;
   }
   else {
      gtx->oceanonly = 0;
   }
   sprintf( programname, "Vis5D Control Panel (%s)", dataset );
   /* set Vis5D context for all widget creates */
   LUI_ContextIndex(index);

   if (wdpy_name==NULL) {
      /* graphics and widgets on same display */
      GuiDpy = GfxDpy;
      GuiScr = GfxScr;
      GuiVisual = GfxVisual;
      GuiDepth = GfxDepth;
      GuiColormap = GfxColormap;
      vis5d_set_pointer(index, -1, -1);

   }
   else {
      /* open second display for widgets */
      XSizeHints hints;
      XSetWindowAttributes attr;
      int attr_flags;
      int fakex, fakey;

      GuiDpy = XOpenDisplay( wdpy_name );
     
      if (!GuiDpy) {
         printf("Unable to open display %s\n", wdpy_name );
         exit(1);
      }

      GuiScr = DefaultScreen( GuiDpy );
      find_best_visual( GuiDpy, GuiScr, &GuiDepth, &GuiVisual, &GuiColormap );

      /* create fake 3-D window on widget display so user can rotate */
      /* or interact with the 3-D image using the widget display's mouse */

      attr.colormap = GuiColormap;
      attr.event_mask = ExposureMask | PointerMotionMask | KeyPressMask
                | ButtonPressMask | ButtonReleaseMask | StructureNotifyMask;
      attr.background_pixel = BlackPixel( GfxDpy, GfxScr );
      attr.border_pixel = BlackPixel( GfxDpy, GfxScr );
      attr_flags = CWColormap | CWEventMask | CWBackPixel | CWBorderPixel;
      fakex = 410;
      fakey = 10;
      gtx->fakewidth = DisplayWidth(GuiDpy, DefaultScreen(GuiDpy)) - 420;
      gtx->fakeheight = gtx->fakewidth;

      gtx->fakewin = XCreateWindow( GuiDpy, RootWindow(GuiDpy, GuiScr),
                                     fakex, fakey, gtx->fakewidth, gtx->fakeheight,
                                    0, GuiDepth, InputOutput, GuiVisual,
                                    attr_flags, &attr );
      hints.x = xpos;
      hints.y = ypos;
      hints.flags = USPosition | PPosition;
      XSetStandardProperties( GuiDpy, gtx->fakewin,
                              programname, programname,  /*title, icon*/
                              None,  /*icon_pixmap*/
                              NULL, 0,  /*arguments*/
                              &hints );
      XSelectInput( GuiDpy, gtx->fakewin,
                    ButtonPressMask|ButtonReleaseMask|PointerMotionMask
                    |StructureNotifyMask );
      XMapWindow( GuiDpy, gtx->fakewin );


   }

   /* Initialize colorbar editor */
   LUI_ColorBarPacking( PACK_COLOR('R','G','B','A') );

   /* make the widgets */
   create_widgets( index, volflag, programname );

   /* Initialize colorbars/colortables */
   init_colortables( gtx );

   XRaiseWindow( GuiDpy, gtx->CpWindow );

   XGetWindowAttributes( GuiDpy, gtx->CpWindow, &winatts);

   soundy = DisplayHeight(GuiDpy, DefaultScreen(GuiDpy)) - 620;
   if (soundy < 1) soundy = 1;
   make_sound_window( index, gtx->cpx, soundy, 630, 600);
/*   if (gtx->othersnddpy ){
      vis5d_resize_sounding_window( index, 630, 600,
                                    gtx->cpx, soundy);
   } */
   /* XSynchronize(GuiDpy, True); */
   return 1;
}


