/* rdline.c
 *
 * Copyright (c) 1992-1998 by Mike Gleason.
 * All rights reserved.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 * 
 */

#include "syshdrs.h"

#ifdef HAVE_LIBREADLINE
#	include <readline/readline.h>
#	include <readline/history.h>
#endif	/* HAVE_LIBREADLINE */

#include "shell.h"
#include "util.h"
#include "bookmark.h"
#include "cmds.h"
#include "ls.h"
#include "readln.h"

const char *tcap_normal = "";
const char *tcap_boldface = "";
const char *tcap_underline = "";
const char *tcap_reverse = "";
const char *gTerm;
int gXterm;
int gXtermTitle;	/* Idea by forsberg@lysator.liu.se */
char gCurXtermTitleStr[256];

extern int gEventNumber;
extern int gMaySetXtermTitle;
extern LsCacheItem gLsCache[kLsCacheSize];
extern FTPConnectionInfo gConn;
extern char gRemoteCWD[512];
extern char gOurDirectoryPath[];
extern char gVersion[];
extern int gNumBookmarks;
extern BookmarkPtr gBookmarkTable;

/* For a few selected terminal types, we'll print in boldface, etc.
 * This isn't too important, though.
 */
void
InitTermcap(void)
{
	const char *term;
	
	gXterm = gXtermTitle = 0;
	gCurXtermTitleStr[0] = '\0';

	if ((gTerm = getenv("TERM")) == NULL) {
		tcap_normal = "";
		tcap_boldface = "";
		tcap_underline = "";
		tcap_reverse = "";
		return;
	}

	term = gTerm;
	if (	(strstr(term, "xterm") != NULL) ||
		(strstr(term, "rxvt") != NULL) ||
		(ISTRCMP(term, "scoterm") == 0)
	) {
		gXterm = gXtermTitle = 1;
	}

	if (	(gXterm != 0) ||
		(strcmp(term, "vt100") == 0) ||
		(strcmp(term, "linux") == 0) ||
		(strcmp(term, "vt102") == 0)
	) {
		tcap_normal = "\033[0m";       /* Default ANSI escapes */
		tcap_boldface = "\033[1m";
		tcap_underline = "\033[4m";
		tcap_reverse = "\033[7m";
	} else {
		tcap_normal = "";
		tcap_boldface = "";
		tcap_underline = "";
		tcap_reverse = "";
	}
}	/* InitTermcap */



#ifdef HAVE_LIBREADLINE

static char *
FindStartOfCurrentCommand(void)
{
	char *scp;
	char *start;
	int qc;

	for (scp = rl_line_buffer;;) {
		start = scp;
		for (;;) {
			if (*scp == '\0')
				goto done;
			if (!isspace((int) *scp))
				break;
			scp++;
		}
		start = scp;

		for (;;) {
			if (*scp == '\0') {
				goto done;
			} else if ((*scp == '"') || (*scp == '\'')) {
				qc = *scp++;

				for (;;) {
					if (*scp == '\0') {
						goto done;
					} else if (*scp == '\\') {
						scp++;
						if (*scp == '\0')
							goto done;
						scp++;
					} else if (*scp == qc) {
						scp++;
						break;
					} else {
						scp++;
					}
				}
			} else if (*scp == '\\') {
				scp++;
				if (*scp == '\0')
					goto done;
				scp++;
			} else if ((*scp == ';') || (*scp == '\n')) {
				/* command ended */
				scp++;
				if (*scp == '\0')
					goto done;
				break;
			} else {
				scp++;
			}
		}
	}
done:
	return (start);
}	/* FindStartOfCurrentCommand */



static FileInfoListPtr
GetLsCacheFileList(const char *const item)
{
	int ci;
	int sortBy;
	int sortOrder;
	FileInfoListPtr filp;

	ci = LsCacheLookup(item);
	if (ci < 0) {
		/* This dir was not in the
		 * cache -- go get it.
		 */
		(void) rl_insert_text("...");
		rl_redisplay();
		Ls(item, 'l', "", NULL);
		(void) rl_do_undo();
		ci = LsCacheLookup(item);
		if (ci < 0)
			return NULL;
	}

	sortBy = 'n';		/* Sort by filename. */
	sortOrder = 'a';	/* Sort in ascending order. */
	filp = &gLsCache[ci].fil;
	SortFileInfoList(filp, sortBy, sortOrder);
	return filp;
}	/* GetLsCacheFileList */




static char *
RemoteCompletionFunction(char *text, int state, int fTypeFilter)
{
	char rpath[256];
	char *cp;
	char *cp2;
	char *textbasename;
	int fType;
	FileInfoPtr diritemp;
	FileInfoListPtr filp;
	int textdirlen;
	size_t tbnlen;
	size_t flen, mlen;
	static FileInfoVec diritemv;
	static int i;

	textbasename = strrchr(text, '/');
	if (textbasename == NULL) {
		textbasename = text;
		textdirlen = -1;
	} else {
		textdirlen = (int) (textbasename - text);
		textbasename++;
	}
	tbnlen = strlen(textbasename);

	if (state == 0) {
		if (text[0] == '\0') {
			/* Special case when they do "get <TAB><TAB> " */
			STRNCPY(rpath, gRemoteCWD);
		} else {
			PathCat(rpath, sizeof(rpath), gRemoteCWD, text);
			if (text[strlen(text) - 1] == '/') {
				/* Special case when they do "get /dir1/dir2/<TAB><TAB>" */
				STRNCAT(rpath, "/");
			}
			cp2 = strrchr(rpath, '/');
			if (cp2 == NULL) {
				return NULL;
			} else if (cp2 == rpath) {
				/* Item in root directory. */
				cp2++;
			}
			*cp2 = '\0';
		}

		filp = GetLsCacheFileList(rpath);
		if (filp == NULL)
			return NULL;

		diritemv = filp->vec;
		if (diritemv == NULL)
			return NULL;

		i = 0;
	}

	for ( ; ; ) {
		diritemp = diritemv[i];
		if (diritemp == NULL)
			break;

		i++;
		fType = (int) diritemp->type;
		if ((fTypeFilter == 0) || (fType == fTypeFilter) || (fType == /* symlink */ 'l')) {
			if (strncmp(textbasename, diritemp->relname, tbnlen) == 0) {
				flen = strlen(diritemp->relname);
				if (textdirlen < 0) {
					mlen = flen + 2;
					cp = malloc(mlen);
					if (cp == NULL)
						return (NULL);
					(void) memcpy(cp, diritemp->relname, mlen);
				} else {
					mlen = textdirlen + 1 + flen + 2;
					cp = malloc(mlen);
					if (cp == NULL)
						return (NULL);
					(void) memcpy(cp, text, textdirlen);
					cp[textdirlen] = '/';
					(void) strcpy(cp + textdirlen + 1, diritemp->relname);
				}
				if (fType == 'd') {
#if (HAVE_LIBREADLINE >= 210)
					rl_completion_append_character = '/';
#else
					(void) strcat(cp, "/");
#endif	/* Readline 2.1 or higher */
#if (HAVE_LIBREADLINE >= 210)
				} else {
					rl_completion_append_character = ' ';
#endif	/* Readline 2.1 or higher */
				}
				return cp;
			}
		}
	}
	return NULL;
}	/* RemoteCompletionFunction */




static char *
RemoteFileCompletionFunction(char *text, int state)
{
	char *cp;

	cp = RemoteCompletionFunction(text, state, 0);
	return cp;
}	/* RemoteFileCompletionFunction */




static char *
RemoteDirCompletionFunction(char *text, int state)
{
	char *cp;

	cp = RemoteCompletionFunction(text, state, 'd');
	return cp;
}	/* RemoteDirCompletionFunction */




static char *
BookmarkCompletionFunction(char *text, int state)
{
	char *cp;
	size_t textlen;
	int i, matches;

	if ((gBookmarkTable == NULL) || (state >= gNumBookmarks))
		return (NULL);

#if (HAVE_LIBREADLINE >= 210)
	rl_completion_append_character = ' ';
#endif

	textlen = strlen(text);
	if (textlen == 0) {
		cp = StrDup(gBookmarkTable[state].bookmarkName);
	} else {
		cp = NULL;
		for (i=0, matches=0; i<gNumBookmarks; i++) {
			if (ISTRNCMP(gBookmarkTable[i].bookmarkName, text, textlen) == 0) {
				matches++;
				if (matches >= state) {
					cp = StrDup(gBookmarkTable[i].bookmarkName);
					break;
				}
			}
		}
	}
	return cp;
}	/* BookmarkCompletionFunction */




static char *
CommandCompletionFunction(char *text, int state)
{
	char *cp;
	size_t textlen;
	int i, matches;
	CommandPtr cmdp;

	if ((gBookmarkTable == NULL) || (state >= gNumBookmarks))
		return (NULL);

#if (HAVE_LIBREADLINE >= 210)
	rl_completion_append_character = ' ';
#endif

	textlen = strlen(text);
	if (textlen == 0) {
		cp = NULL;
	} else {
		cp = NULL;
		for (i=0, matches=0; ; i++) {
			cmdp = GetCommandByIndex(i);
			if (cmdp == kNoCommand)
				break;
			if (ISTRNCMP(cmdp->name, text, textlen) == 0) {
				matches++;
				if (matches >= state) {
					cp = StrDup(cmdp->name);
					break;
				}
			}
		}
	}
	return cp;
}	/* CommandCompletionFunction */




void
ReCacheBookmarks(void)
{
	(void) LoadBookmarkTable();
}



static int
HaveCommandNameOnly(char *cmdstart)
{
	char *cp;
	for (cp = cmdstart; *cp != '\0'; cp++) {
		if (isspace((int) *cp))
			return (0);	/* At least one argument in progress. */
	}
	return (1);
}	/* HaveCommandNameOnly */




static char *
CompletionFunction(char *text, int state)
{
	char *cp;
	char *cmdstart;
	ArgvInfo ai;
	int bUsed;
	CommandPtr cmdp;
	static int flags;

	if (state == 0) {
		flags = -1;
		cmdstart = FindStartOfCurrentCommand();
		if (cmdstart == NULL)
			return NULL;
		if (HaveCommandNameOnly(cmdstart)) {
			flags = -2;	/* special case */
			cp = CommandCompletionFunction(text, state);
			return cp;
		}

		(void) memset(&ai, 0, sizeof(ai));
		bUsed = MakeArgv(cmdstart, &ai.cargc, ai.cargv,
			(int) (sizeof(ai.cargv) / sizeof(char *)),
			ai.argbuf, sizeof(ai.argbuf),
			ai.noglobargv);
		if (bUsed <= 0)
			return NULL;
		if (ai.cargc == 0)
			return NULL;

		cmdp = GetCommandByName(ai.cargv[0], 0);
		if (cmdp == kAmbiguousCommand) {
			return NULL;
		} else if (cmdp == kNoCommand) {
			return NULL;
		}
		flags = cmdp->flags;
	}
	if (flags == (-2)) {
		cp = CommandCompletionFunction(text, state);
		return cp;
	}
	if (flags < 0)
		return NULL;
	if ((flags & (kCompleteLocalFile|kCompleteLocalDir)) != 0) {
		cp = filename_completion_function(text, state);
		return cp;
	} else if ((flags & kCompleteRemoteFile) != 0) {
		rl_filename_quoting_desired = 1;
		rl_filename_completion_desired = 1;
		cp = RemoteFileCompletionFunction(text, state);
		return cp;
	} else if ((flags & kCompleteRemoteDir) != 0) {
		rl_filename_quoting_desired = 1;
		rl_filename_completion_desired = 1;
		cp = RemoteDirCompletionFunction(text, state);
		return cp;
	} else if ((flags & kCompleteBookmark) != 0) {
		cp = BookmarkCompletionFunction(text, state);
		return cp;
	}	
	return NULL;
}	/* CompletionFunction */




void
LoadHistory(void)
{
	char pathName[256];

	if (gOurDirectoryPath[0] == '\0')
		return;
	(void) OurDirectoryPath(pathName, sizeof(pathName), kHistoryFileName);

	(void) read_history(pathName);
}	/* LoadHistory */




/* Save the commands they typed in a history file, then they can use
 * readline to go through them again next time.
 */
void
SaveHistory(void)
{
	char pathName[256];

	if (gOurDirectoryPath[0] == '\0')
		return;		/* Don't create in root directory. */
	(void) OurDirectoryPath(pathName, sizeof(pathName), kHistoryFileName);

	stifle_history(100);
	(void) write_history(pathName);
	(void) chmod(pathName, 00600);
}	/* SaveHistory */




void
InitReadline(void)
{
	rl_readline_name = "ncftp";
	rl_terminal_name = getenv ("TERM");
	rl_completion_entry_function = (Function *) CompletionFunction;
#if (HAVE_LIBREADLINE >= 210)
	rl_filename_quote_characters = " \t\n\\\"'@<>=;|&()#$`?*[!:";
#endif
	rl_completer_quote_characters = "\"";
	using_history();
	LoadHistory();
	ReCacheBookmarks();
}	/* InitReadline */




char *
Readline(char *prompt)
{
	return (readline(prompt));
}	/* Readline */



void
AddHistory(char *line)
{
	add_history(line);
}	/* AddHistory */



void
DisposeReadline(void)
{
	SaveHistory();
}	/* DisposeReadline */


#else	/* HAVE_LIBREADLINE */

char *
Readline(char *prompt)
{
	char buf[512];

	fputs(prompt, stdout);
	if (fgets(buf, sizeof(buf) - 1, stdin) == NULL)
		return NULL;
	return (StrDup(buf));
}	/* Readline */



void
ReCacheBookmarks(void)
{
	/* nothing */
}




void
LoadHistory(void)
{
	/* nothing */
}	/* LoadHistory */




void
SaveHistory(void)
{
	/* nothing */
}	/* SaveHistory */



void
AddHistory(char *line)
{
	/* nothing */
}	/* AddHistory */



void
InitReadline(void)
{
	/* nothing */
}	/* InitReadline */



void
DisposeReadline(void)
{
	/* nothing */
}	/* DisposeReadline */


#endif	/* HAVE_LIBREADLINE */



/*VARARGS*/
void
SetXtermTitle(char *fmt, ...)
{
	va_list ap;
	char buf[256];

	if ((gXtermTitle != 0) && (gMaySetXtermTitle != 0)) {
		if ((fmt == NULL) || (ISTRCMP(fmt, "RESTORE") == 0)) {
			strcpy(buf, gTerm);
		} else if (ISTRCMP(fmt, "DEFAULT") == 0) {
			(void) Strncpy(buf, gVersion + 5, 12);
		} else {
			va_start(ap, fmt);
#ifdef HAVE_VSNPRINTF
			(void) vsnprintf(buf, sizeof(buf) - 1, fmt, ap);
			buf[sizeof(buf) - 1] = '\0';
#else
			(void) vsprintf(buf, fmt, ap);
#endif
			va_end(ap);
		}
		if (buf[0] != '\0') {
			if (strcmp(gCurXtermTitleStr, buf) != 0) {
				fprintf(stderr, "\033]0;%s\007", buf);
				STRNCPY(gCurXtermTitleStr, buf);
			}
		}
	}
}	/* SetXtermTitle */




void
PrintStartupBanner(void)
{
	char v[80], *cp;
	char vdate[32];

	/* Print selected information from the version ID. */
	vdate[0] = '\0';
	(void) STRNCPY(v, gVersion + 5);
	cp = strchr(v, ',');
	if (cp != NULL) {
		*cp = '\0';
		cp[-5] = '\0';
		(void) STRNCPY(vdate, " (");
		(void) STRNCAT(vdate, v + 16);
		(void) STRNCAT(vdate, ", ");
		(void) STRNCAT(vdate, cp - 4);
		(void) STRNCAT(vdate, ")");
	}

#if defined(BETA) && (BETA > 0)
	(void) fprintf(stdout, "%s%.11s beta %d%s%s by Mike Gleason.\n",
		tcap_boldface,
		gVersion + 5,
		BETA,
		tcap_normal,
		vdate
	);
#else
	(void) fprintf(stdout, "%s%.11s%s%s by Mike Gleason.\n",
		tcap_boldface,
		gVersion + 5,
		tcap_normal,
		vdate
	);
#endif
	(void) fflush(stdout);
}	/* PrintStartupBanner */




/* Print the command shell's prompt. */
void
MakePrompt(char *dst, size_t dsize)
{
#ifdef HAVE_LIBREADLINE
	char acwd[64];

#	ifdef HAVE_SNPRINTF
	if (gConn.loggedIn != 0) {
		AbbrevStr(acwd, gRemoteCWD, 25, 0);
		snprintf(dst, dsize, "ncftp %s > ", acwd);
	} else {
		snprintf(dst, dsize, "ncftp> ");
	}
#	else	/* HAVE_SNPRINTF */
	(void) Strncpy(dst, "ncftp", dsize);
	if (gConn.loggedIn != 0) {
		AbbrevStr(acwd, gRemoteCWD, 25, 0);
		(void) Strncat(dst, " ", dsize);
		(void) Strncat(dst, acwd, dsize);
		(void) Strncat(dst, " ", dsize);
	}
	(void) Strncat(dst, ">", dsize);
	(void) Strncat(dst, " ", dsize);
#	endif	/* HAVE_SNPRINTF */
#else	/* HAVE_LIBREADLINE */
	char acwd[64];

#	ifdef HAVE_SNPRINTF
	if (gConn.loggedIn != 0) {
		AbbrevStr(acwd, gRemoteCWD, 25, 0);
		snprintf(dst, dsize, "%sncftp%s %s %s>%s ",
			tcap_boldface, tcap_normal, acwd,
			tcap_boldface, tcap_normal);
	} else {
		snprintf(dst, dsize, "%sncftp%s> ",
			tcap_boldface, tcap_normal);
	}
#	else	/* HAVE_SNPRINTF */
	(void) Strncpy(dst, tcap_boldface, dsize);
	(void) Strncat(dst, "ncftp", dsize);
	(void) Strncat(dst, tcap_normal, dsize);
	if (gConn.loggedIn != 0) {
		AbbrevStr(acwd, gRemoteCWD, 25, 0);
		(void) Strncat(dst, " ", dsize);
		(void) Strncat(dst, acwd, dsize);
		(void) Strncat(dst, " ", dsize);
	}
	(void) Strncat(dst, tcap_boldface, dsize);
	(void) Strncat(dst, ">", dsize);
	(void) Strncat(dst, tcap_normal, dsize);
	(void) Strncat(dst, " ", dsize);
#	endif	/* HAVE_SNPRINTF */
#endif	/* HAVE_LIBREADLINE */
}	/* MakePrompt */
