/*
 *  ncplib.c
 *
 *  Copyright (C) 1995, 1996 by Volker Lendecke
 *
 */

#if 0
#define ncp_dprintf(X...)	printf(X)
#else
#define ncp_dprintf(X...)
#endif
#include "ncplib.h"
#include "ncplib_err.h"
#ifdef SIGNATURES
#include "ncpsign.h"
#endif
#ifdef NDS_SUPPORT
extern int bindery_only;
#include "ndslib.h"
#endif

#include <sys/ioctl.h>
extern pid_t wait(int *);
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include "ext/socket.h"
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include "kernel/route.h"
#include <sys/param.h>
#include <stdlib.h>
#include <mntent.h>
#include <pwd.h>
#include <sys/stat.h>
#include <stdarg.h>
#ifdef CONFIG_NATIVE_IP
#include <netdb.h>
#endif

#define NCP_DEFAULT_BUFSIZE	1024
#define NCP_MAX_BUFSIZE		1024
#ifdef SIGNATURES
#define NCP_DEFAULT_OPTIONS	2
int in_options = NCP_DEFAULT_OPTIONS;
#endif

static long
 ncp_negotiate_buffersize(struct ncp_conn *conn,
			  int size, int *target);

#ifdef SIGNATURES
static long
 ncp_negotiate_size_and_options(struct ncp_conn *conn,
			  int size, int options, 
			  int *ret_size, int *ret_options);
#endif

static long
 ncp_login_object(struct ncp_conn *conn,
		  const unsigned char *username,
		  int login_type,
		  const unsigned char *password);

static long
ncp_do_close(struct ncp_conn *conn);

#ifdef CONFIG_NATIVE_IPX
static long
ncp_find_server_ipx(const char **server_name, int type, struct sockaddr_ipx* addr);
#endif

void
str_upper(char *name)
{
	while (*name)
	{
		*name = toupper(*name);
		name = name + 1;
	}
}

#if 0

static int debug_level = 5;
static FILE *logfile = stderr;

static void
dprintf(int level, char *p,...)
{
	va_list ap;

	if (level > debug_level)
	{
		return;
	}
	va_start(ap, p);
	vfprintf(logfile, p, ap);
	va_end(ap);
	fprintf(logfile, "\n");
	fflush(logfile);
}

#endif

/* I know it's terrible to include a .c file here, but I want to keep
   the file nwcrypt.c intact and separate for copyright reasons */
#include "nwcrypt.c"

void
ipx_fprint_node(FILE * file, IPXNode node)
{
	fprintf(file, "%02X%02X%02X%02X%02X%02X",
		(unsigned char) node[0],
		(unsigned char) node[1],
		(unsigned char) node[2],
		(unsigned char) node[3],
		(unsigned char) node[4],
		(unsigned char) node[5]
	    );
}

void
ipx_fprint_network(FILE * file, IPXNet net)
{
	fprintf(file, "%08X", (u_int32_t)ntohl(net));
}

void
ipx_fprint_port(FILE * file, IPXPort port)
{
	fprintf(file, "%04X", ntohs(port));
}

void
ipx_fprint_saddr(FILE * file, struct sockaddr_ipx *sipx)
{
	ipx_fprint_network(file, sipx->sipx_network);
	fprintf(file, ":");
	ipx_fprint_node(file, sipx->sipx_node);
	fprintf(file, ":");
	ipx_fprint_port(file, sipx->sipx_port);
}

void
ipx_print_node(IPXNode node)
{
	ipx_fprint_node(stdout, node);
}

void
ipx_print_network(IPXNet net)
{
	ipx_fprint_network(stdout, net);
}

void
ipx_print_port(IPXPort port)
{
	ipx_fprint_port(stdout, port);
}

void
ipx_print_saddr(struct sockaddr_ipx *sipx)
{
	ipx_fprint_saddr(stdout, sipx);
}

int
ipx_sscanf_node(char *buf, unsigned char node[6])
{
	int i;
	int n[6];

	if ((i = sscanf(buf, "%2x%2x%2x%2x%2x%2x",
			&(n[0]), &(n[1]), &(n[2]),
			&(n[3]), &(n[4]), &(n[5]))) != 6)
	{
		return i;
	}
	for (i = 0; i < 6; i++)
	{
		node[i] = n[i];
	}
	return 6;
}

int
ipx_sscanf_saddr(char *buf, struct sockaddr_ipx *target)
{
	char *p;
	struct sockaddr_ipx addr;
	unsigned long sipx_network;

	addr.sipx_family = AF_IPX;
	addr.sipx_type = NCP_PTYPE;

	if (sscanf(buf, "%lx", &sipx_network) != 1)
	{
		return 1;
	}
	addr.sipx_network = htonl(sipx_network);
	if ((p = strchr(buf, ':')) == NULL)
	{
		return 1;
	}
	p += 1;
	if (ipx_sscanf_node(p, addr.sipx_node) != 6)
	{
		return 1;
	}
	if ((p = strchr(p, ':')) == NULL)
	{
		return 1;
	}
	p += 1;
	if (sscanf(p, "%hx", &addr.sipx_port) != 1)
	{
		return 1;
	}
	addr.sipx_port = htons(addr.sipx_port);
	*target = addr;
	return 0;
}

int
ipx_node_equal(CIPXNode n1, CIPXNode n2)
{
	return memcmp(n1, n2, IPX_NODE_LEN) == 0;
}

static int
x_recvfrom(int sock, void *buf, int len, unsigned int flags,
	   struct sockaddr *sender, socklen_t* addrlen, int timeout,
	   long *err)
{
	fd_set rd, wr, ex;
	struct timeval tv;
	int result;

	FD_ZERO(&rd);
	FD_ZERO(&wr);
	FD_ZERO(&ex);
	FD_SET(sock, &rd);

	tv.tv_sec = timeout;
	tv.tv_usec = 0;

	if ((result = select(sock + 1, &rd, &wr, &ex, &tv)) == -1)
	{
		*err = errno;
		return -1;
	}
	if (FD_ISSET(sock, &rd))
	{
		result = sender?recvfrom(sock, buf, len, flags,
				  sender, addrlen):
				recv(sock, buf, len, flags);
	} else
	{
		result = -1;
		errno = ETIMEDOUT;
	}
	if (result < 0)
	{
		*err = errno;
	}
	return result;
}

static int
x_recv(int sock, void *buf, int len, unsigned int flags, int timeout,
	 long *err)
{
	return x_recvfrom(sock, buf, len, flags, NULL, 0,
			    timeout, err);
}

#ifdef CONFIG_NATIVE_IPX
#ifdef __MAKE_SULIB__
static long
ipx_sap_find_nearest(int server_type, struct sockaddr_ipx *result,
		     char server_name[NCP_BINDERY_NAME_LEN])
{
	struct sockaddr_ipx addr;
	char data[1024];
	int sock;
	int opt;
	int packets;
	int len;

	struct sap_server_ident *ident;

	if ((sock = socket(PF_IPX, SOCK_DGRAM, IPXPROTO_IPX)) < 0)
	{
		if (errno == EINVAL)
		{
			return NCPL_ET_NO_IPX;
		}
		return errno;
	}
	opt = 1;
	/* Permit broadcast output */
	if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &opt, sizeof(opt)) == -1)
	{
		goto finished;
	}
	memzero(addr);
	addr.sipx_family = AF_IPX;
	addr.sipx_network = htonl(0x0);
	addr.sipx_port = htons(0x0);
	addr.sipx_type = IPX_SAP_PTYPE;

	if (bind(sock, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		if (errno == EADDRNOTAVAIL)
		{
			errno = NCPL_ET_NO_INTERFACE;
		}
		goto finished;
	}
	WSET_HL(data, 0, IPX_SAP_NEAREST_QUERY);
	WSET_HL(data, 2, server_type);

	memzero(addr);
	addr.sipx_family = AF_IPX;
	addr.sipx_port = htons(IPX_SAP_PORT);
	addr.sipx_type = IPX_SAP_PTYPE;
	addr.sipx_network = htonl(0x0);
	ipx_assign_node(addr.sipx_node, IPX_BROADCAST_NODE);

	if (sendto(sock, data, 4, 0,
		   (struct sockaddr *) &addr, sizeof(addr)) < 0)
	{
		goto finished;
	}
	packets = 5;
	do
	{
		long err;
		len = x_recv(sock, data, 1024, 0, 1, &err);
		if (len < 66)
		{
			packets = packets - 1;
			continue;
		}
	}
	while ((ntohs(*((__u16 *) data)) != IPX_SAP_NEAREST_RESPONSE)
	       && (packets > 0));

	if (packets == 0)
	{
		close(sock);
		return NCPL_ET_NO_SERVER;
	}
	ident = (struct sap_server_ident *) (data + 2);

	result->sipx_family = AF_IPX;
	result->sipx_network = ident->server_network;
	result->sipx_port = ident->server_port;
	ipx_assign_node(result->sipx_node, ident->server_node);

	memcpy(server_name, ident->server_name, sizeof(ident->server_name));

	errno = 0;

      finished:
	close(sock);
	return errno;
}
#endif	/* __MAKE_SULIB__ */

static int
ipx_make_reachable(const struct sockaddr_ipx* target)
{
#ifdef __MAKE_SULIB__
	IPXNet	network = target->sipx_network;
	struct rtentry rt_def;
	/* Router */
	struct sockaddr_ipx *sr = (struct sockaddr_ipx *) &rt_def.rt_gateway;
	/* Target */
	struct sockaddr_ipx *st = (struct sockaddr_ipx *) &rt_def.rt_dst;

	struct ipx_rip_packet rip;
	struct sockaddr_ipx addr;
	socklen_t addrlen;
	int sock;
	int opt;
	int res = -1;
	int i;
	int packets;

	if (geteuid() != 0)
	{
		errno = EPERM;
		return -1;
	}
	memzero(rip);

	sock = socket(PF_IPX, SOCK_DGRAM, IPXPROTO_IPX);

	if (sock == -1)
	{
		if (errno == EINVAL)
		{
			return NCPL_ET_NO_IPX;
		}
		return errno;
	}
	opt = 1;
	/* Permit broadcast output */
	if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &opt, sizeof(opt)) != 0)
	{
		goto finished;
	}
	memzero(addr);
	addr.sipx_family = AF_IPX;
	addr.sipx_network = htonl(0x0);
	addr.sipx_port = htons(0x0);
	addr.sipx_type = IPX_RIP_PTYPE;

	if (bind(sock, (struct sockaddr *) &addr, sizeof(addr)) != 0)
	{
		goto finished;
	}
	addr.sipx_family = AF_IPX;
	addr.sipx_port = htons(IPX_RIP_PORT);
	addr.sipx_type = IPX_RIP_PTYPE;
	addr.sipx_network = htonl(0x0);
	ipx_assign_node(addr.sipx_node, IPX_BROADCAST_NODE);

	rip.operation = htons(IPX_RIP_REQUEST);
	rip.rt[0].network = network;

	if (sendto(sock, &rip, sizeof(rip), 0,
		   (struct sockaddr *) &addr, sizeof(addr)) < 0)
	{
		goto finished;
	}
	packets = 3;
	do
	{
		long err;
		int len;

		if (packets == 0)
		{
			goto finished;
		}
		addrlen = sizeof(struct sockaddr_ipx);

		len = x_recvfrom(sock, &rip, sizeof(rip), 0, (struct sockaddr*)sr, &addrlen, 1,
				   &err);

		if (len < sizeof(rip))
		{
			packets = packets - 1;
			continue;
		}
	}
	while (ntohs(rip.operation) != IPX_RIP_RESPONSE);

	if (rip.rt[0].network != network)
	{
		goto finished;
	}
	rt_def.rt_flags = RTF_GATEWAY;
	st->sipx_network = network;
	st->sipx_family = AF_IPX;
	sr->sipx_family = AF_IPX;

	i = 0;
	do
	{
		res = ioctl(sock, SIOCADDRT, &rt_def);
		i++;
	}
	while ((i < 5) && (res < 0) && (errno == EAGAIN));

      finished:
	close(sock);

	if (res != 0)
	{
		errno = ENETUNREACH;
	}
	return res;
#else
	FILE *p;
	char buf[40];
    
	sprintf(buf, "nwsfind -a %08x:%02x%02x%02x%02x%02x%02x:%04x",
		(u_int32_t)ntohl(target->sipx_network),
		((const unsigned char*)target->sipx_node)[0],
		((const unsigned char*)target->sipx_node)[1],
		((const unsigned char*)target->sipx_node)[2],
		((const unsigned char*)target->sipx_node)[3],
		((const unsigned char*)target->sipx_node)[4],
		((const unsigned char*)target->sipx_node)[5],
		ntohs(target->sipx_port));

	if (!(p = popen(buf, "r"))) {
		return errno;
	}
	fgets(buf, sizeof(buf), p);
	if (pclose(p)) {
		return ENETUNREACH;
	}
	return 0;
#endif
}

void
ipx_assign_node(IPXNode dest, CIPXNode src)
{
	memcpy(dest, src, IPX_NODE_LEN);
}

static int
install_wdog(struct ncp_conn *conn)
{
	int parent_pid = getpid();
	int pid;
	int sock = conn->wdog_sock;

	char buf[1024];
	struct sockaddr_ipx sender;
	int sizeofaddr = sizeof(struct sockaddr_ipx);
	int pktsize;


	if ((pid = fork()) < 0)
	{
		return -1;
	}
	if (pid != 0)
	{
		/* Parent, should go on as usual */
		conn->wdog_pid = pid;
		return 0;
	}
	while (1)
	{
		long err;
		/* every 120 seconds we look if our parent is
		   still alive */
		pktsize = x_recvfrom(sock, buf, sizeof(buf), 0,
				     (struct sockaddr*)&sender, &sizeofaddr, 120, &err);

		if (getppid() != parent_pid)
		{
			/* our parent has died, so nothing to do
			   anymore */
			exit(0);
		}
		if ((pktsize != 2)
		    || (buf[1] != '?'))
		{
			continue;
		}
		buf[1] = 'Y';
		pktsize = sendto(sock, buf, 2, 0,
				 (struct sockaddr *) &sender,
				 sizeof(sender));
	}
}
#endif	/* CONFIG_NATIVE_IPX */

static void
 assert_conn_locked(struct ncp_conn *conn);

static void
assert_conn_not_locked(struct ncp_conn *conn)
{
	if (conn->lock != 0)
	{
		ncp_printf("ncpfs: conn already locked!\n");
	}
}

static void
ncp_lock_conn(struct ncp_conn *conn)
{
	assert_conn_not_locked(conn);
	conn->lock = 1;
}

void
ncp_unlock_conn(struct ncp_conn *conn)
{
	assert_conn_locked(conn);
	conn->lock = 0;
}

static long
do_ncp_call(struct ncp_conn *conn, int request_size)
{
	struct ncp_request_header request;

	int result;
	int retries = 20;
	int len;
	long err;

	memcpy(&request, conn->packet, sizeof(request));
#ifdef SIGNATURES
	if (conn->sign_active)
	{
		sign_packet(conn, &request_size);
	}
#endif	
	while (retries > 0)
	{
		struct ncp_reply_header reply;

		retries -= 1;

		result = send(conn->ncp_sock, conn->packet,
			      request_size, 0);

		if (result < 0)
		{
			return errno;
		}
	      re_select:
		len = x_recv(conn->ncp_sock,
			     (char *) &reply, sizeof(reply),
			     MSG_PEEK, 3, &err);

		if ((len < 0) && (err == ETIMEDOUT))
		{
			continue;
		}
		if (len < 0)
		{
			return err;
		}
		if (
		/* Did the sender send a positive acknowledge? */
			      ((len == sizeof(reply))
			       && (reply.type == NCP_POSITIVE_ACK))

		/* Did we get a bogus answer? */
			   || ((len < sizeof(reply))
			       || (reply.type != NCP_REPLY)
			       || ((request.type != NCP_ALLOC_SLOT_REQUEST)
				   && ((reply.sequence != request.sequence)
				       || (reply.conn_low != request.conn_low)
				 || (reply.conn_high != request.conn_high)))))
		{
			/* Then throw away the packet */
			x_recv(conn->ncp_sock, (char *) &reply, sizeof(reply),
				 0, 1, &err);
			goto re_select;
		}
		len = x_recv(conn->ncp_sock, conn->packet, NCP_PACKET_SIZE,
			 0, 1, &err);
		conn->reply_size = len;
		return 0;
	}
	return ETIMEDOUT;
}

static int
ncp_mount_request(struct ncp_conn *conn, int function)
{
	struct ncp_ioctl_request request;
	int result;

	assert_conn_locked(conn);

	if (conn->has_subfunction != 0)
	{
		WSET_HL(conn->packet, 7, conn->current_size
			- sizeof(struct ncp_request_header) - 2);
	}
	request.function = function;
	request.size = conn->current_size;
	request.data = conn->packet;

	if ((result = ioctl(conn->mount_fid, NCP_IOC_NCPREQUEST, &request)) < 0)
	{
		return result;
	}
	conn->completion = BVAL(conn->packet, 6);
	conn->conn_status = BVAL(conn->packet, 7);
	conn->ncp_reply_size = result - sizeof(struct ncp_reply_header);

	if ((conn->completion != 0) && (conn->verbose != 0))
	{
		ncp_printf("ncp_request_error: %d\n", conn->completion);
	}
	return conn->completion == 0 ? 0 : NCPL_ET_REQUEST_ERROR;
}

static long
ncp_temp_request(struct ncp_conn *conn, int function)
{
	long err;

	assert_conn_locked(conn);

	conn->sequence += 1;

	WSET_LH(conn->packet, 0, NCP_REQUEST);
	BSET(conn->packet, 2, conn->sequence);
	BSET(conn->packet, 3, (conn->i.connection) & 0xff);
	BSET(conn->packet, 5, (conn->i.connection) >> 8);
	BSET(conn->packet, 4, 1);
	BSET(conn->packet, 6, function);

	if (conn->has_subfunction != 0)
	{
		WSET_HL(conn->packet, 7, conn->current_size
			- sizeof(struct ncp_request_header) - 2);
	}
	if ((err = do_ncp_call(conn, conn->current_size)) != 0)
	{
		return err;
	}
	conn->completion = BVAL(conn->packet, 6);
	conn->conn_status = BVAL(conn->packet, 7);
	conn->ncp_reply_size =
	    conn->reply_size - sizeof(struct ncp_reply_header);

	if ((conn->completion != 0) && (conn->verbose != 0))
	{
		ncp_printf("ncp_completion_code: %d\n", conn->completion);
	}
	return conn->completion == 0 ? 0 : NCPL_ET_REQUEST_ERROR;
}

long
ncp_renegotiate_connparam(struct ncp_conn *conn, int buffsize, int in_options)
{
	int neg_buffsize;
	int err;
#ifdef SIGNATURES
	int options;
	
	/* TODO: Signatures active? I am not sure whether we can unnegotiate them */
	if (conn->sign_active) in_options |= 2;

	if ((err = ncp_negotiate_size_and_options(conn, buffsize, in_options, 
	 &neg_buffsize, &options)) == 0)
	{
		if ((options & 2) != (in_options & 2)) 
		{
			err = ncp_negotiate_size_and_options(conn,
			 buffsize, options & 2, &neg_buffsize, &options);
		}
	}
	else
#endif
	{
#ifdef SIGNATURES
		options = 0;
#endif
		err = ncp_negotiate_buffersize(conn, NCP_DEFAULT_BUFSIZE,
			      &neg_buffsize);
	}			  
	if (err) {
		return err;
	}
	if ((neg_buffsize < 512) || (neg_buffsize > NCP_MAX_BUFSIZE))
		return -1;
	conn->i.buffer_size = neg_buffsize;
#ifdef SIGNATURES	
	conn->sign_wanted = (options & 2) ? 1:0;
	if (conn->is_connected == CONN_PERMANENT) {
		int cursign;
		
		if (ioctl(conn->mount_fid, NCP_IOC_SIGN_WANTED, &cursign)) {
		/* ncpfs does not support SIGN_WANTED -> current sign level = 0 */
			cursign = 0;
		}
		if (cursign) cursign = 1;
		if (cursign != conn->sign_wanted) {
			int newsign = conn->sign_wanted;
			
			err = ioctl(conn->mount_fid, NCP_IOC_SET_SIGN_WANTED, &newsign);
		}
	}
#endif
	return 0;
}

#ifdef CONFIG_NATIVE_IPX
static long
ncp_connect_ipx_addr(struct ncp_conn *conn, const struct sockaddr_ipx *target,
		 int wdog_needed)
{
	struct sockaddr_ipx addr;
	socklen_t addrlen;

	int ncp_sock, wdog_sock;
	long err;

	conn->is_connected = NOT_CONNECTED;
	conn->verbose = 0;

	if ((ncp_sock = socket(PF_IPX, SOCK_DGRAM, IPXPROTO_IPX)) == -1)
	{
		return errno;
	}
	if ((wdog_sock = socket(PF_IPX, SOCK_DGRAM, IPXPROTO_IPX)) == -1)
	{
		close(ncp_sock);
		return errno;
	}
	addr.sipx_family = AF_IPX;
	addr.sipx_port = htons(0x0);
	addr.sipx_type = NCP_PTYPE;
	addr.sipx_network = IPX_THIS_NET;
	ipx_assign_node(addr.sipx_node, IPX_THIS_NODE);

	addrlen = sizeof(addr);

	if ((bind(ncp_sock, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	|| (getsockname(ncp_sock, (struct sockaddr *) &addr, &addrlen) == -1))
	{
		int saved_errno = errno;
		close(ncp_sock);
		close(wdog_sock);
		return saved_errno;
	}
	addr.sipx_port = htons(ntohs(addr.sipx_port) + 1);

	if (bind(wdog_sock, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		int saved_errno = errno;
		close(ncp_sock);
		close(wdog_sock);
		return saved_errno;
	}
	conn->ncp_sock = ncp_sock;
	conn->wdog_sock = wdog_sock;

	conn->sequence = 0;
	conn->i.addr = *target;

	if (connect(ncp_sock, (const struct sockaddr*)target, sizeof(*target)) == -1)
	{
		int saved_errno = errno;

		if ((saved_errno != ENETUNREACH) || ipx_make_reachable(target) 
		  || connect(ncp_sock, (const struct sockaddr*)target, sizeof(*target))) {;
			close(ncp_sock);
			close(wdog_sock);
			return saved_errno;
		}
	}

	WSET_LH(conn->packet, 0, NCP_ALLOC_SLOT_REQUEST);
	BSET(conn->packet, 2, conn->sequence);
	BSET(conn->packet, 3, 0xff);
	BSET(conn->packet, 4, 1);
	BSET(conn->packet, 5, 0xff);
	BSET(conn->packet, 6, 0);

	if ((err = do_ncp_call(conn, sizeof(struct ncp_request_header))) != 0)
	{
		if ((err != ENETUNREACH)
		    || (ipx_make_reachable(target) != 0)
		    || ((err =
			 do_ncp_call(conn,
				     sizeof(struct ncp_request_header))) != 0))
		{
			close(ncp_sock);
			close(wdog_sock);
			return err;
		}
	}
	if (wdog_needed != 0)
	{
		install_wdog(conn);
	} else
	{
		conn->wdog_pid = 0;
	}

	conn->sequence = 0;
	conn->i.connection =
	    BVAL(conn->packet, 3) + (BVAL(conn->packet, 5) << 8);
	conn->is_connected = CONN_TEMPORARY;

#ifdef SIGNATURES
	conn->sign_active = 0;
	conn->sign_wanted = 0;
	err = ncp_renegotiate_connparam(conn, NCP_DEFAULT_BUFSIZE, in_options);
#else
	err = ncp_renegotiate_connparam(conn, NCP_DEFAULT_BUFSIZE, 0);
#endif

	if (err != 0)   
	{
		ncp_do_close(conn);
		return -1;
	}
	return 0;
}
#endif	/* CONFIG_NATIVE_IPX */

#ifdef CONFIG_NATIVE_IP
static long
ncp_connect_in_addr(struct ncp_conn *conn, const struct sockaddr_in *target,
		 int wdog_needed)
{
	struct sockaddr_in addr;
	socklen_t addrlen;

	int ncp_sock;
	long err;

	conn->is_connected = NOT_CONNECTED;
	conn->verbose = 0;

	if ((ncp_sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1)
	{
		return errno;
	}
	addr.sin_family = AF_INET;
	addr.sin_port = htons(0x0);
	addr.sin_addr.s_addr = INADDR_ANY;

	addrlen = sizeof(addr);

	if (bind(ncp_sock, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		int saved_errno = errno;
		close(ncp_sock);
		return saved_errno;
	}

	conn->ncp_sock = ncp_sock;
	conn->wdog_sock = -1;

	conn->sequence = 0;
	conn->i.addr = *(const struct sockaddr_ipx*)target;

	if (connect(ncp_sock, (const struct sockaddr*)target, sizeof(*target)) == -1)
	{
		int saved_errno = errno;
		close(ncp_sock);
		return saved_errno;
	}

	WSET_LH(conn->packet, 0, NCP_ALLOC_SLOT_REQUEST);
	BSET(conn->packet, 2, conn->sequence);
	BSET(conn->packet, 3, 0xff);
	BSET(conn->packet, 4, 1);
	BSET(conn->packet, 5, 0xff);
	BSET(conn->packet, 6, 0);

	if ((err = do_ncp_call(conn, sizeof(struct ncp_request_header))) != 0)
	{
		/* ? request route ? */
		close(ncp_sock);
		return err;
	}

	conn->wdog_pid = 0;

	conn->sequence = 0;
	conn->i.connection =
	    BVAL(conn->packet, 3) + (BVAL(conn->packet, 5) << 8);
	conn->is_connected = CONN_TEMPORARY;

#ifdef SIGNATURES
	conn->sign_active = 0;
	conn->sign_wanted = 0;
	err = ncp_renegotiate_connparam(conn, NCP_DEFAULT_BUFSIZE, in_options);
#else
	err = ncp_renegotiate_connparam(conn, NCP_DEFAULT_BUFSIZE, 0);
#endif

	if (err != 0)   
	{
		ncp_do_close(conn);
		return -1;
	}
	return 0;
}
#endif	/* CONFIG_NATIVE_IP */

static long
ncp_connect_addr(struct ncp_conn *conn, const struct sockaddr *target,
		 int wdog_needed) {
#ifdef CONFIG_NATIVE_IPX
	if (target->sa_family == AF_IPX) {
		return ncp_connect_ipx_addr(conn, (const struct sockaddr_ipx*)target, wdog_needed);
	}
#endif	/* CONFIG_NATIVE_IPX */
#ifdef CONFIG_NATIVE_IP
	if (target->sa_family == AF_INET) {
		return ncp_connect_in_addr(conn, (const struct sockaddr_in*)target, wdog_needed);
	}
#endif	/* CONFIG_NATIVE_IP */
	return EAFNOSUPPORT;
}

static long
ncp_connect_any(struct ncp_conn *conn, int wdog_needed)
{
#ifdef CONFIG_NATIVE_IPX
#ifndef __MAKE_SULIB__
	const char* server = NULL;
#else /* __MAKE_SULIB__ */
	char server[NCP_BINDERY_NAME_LEN];
#endif /* __MAKE_SULIB__ */
	struct sockaddr_ipx addr;
	long result;

#ifndef __MAKE_SULIB__
	if ((result = ncp_find_server_ipx(&server, NCP_BINDERY_FSERVER, &addr)) != 0)
#else /* __MAKE_SULIB__ */
	if ((result = ipx_sap_find_nearest(IPX_SAP_FILE_SERVER,
					   &addr, server)) != 0)
#endif /* __MAKE_SULIB__ */
	{
		return result;
	}
	if ((result = ncp_connect_ipx_addr(conn, &addr, wdog_needed)) != 0)
	{
		return result;
	}
	strcpy(conn->server, server);
	return 0;
#else
	return EAFNOSUPPORT;
#endif	/* CONFIG_NATIVE_IPX */
}

#ifdef CONFIG_NATIVE_IPX
static long
ncp_find_server_ipx(const char **server_name, int type, struct sockaddr_ipx* server_addr)
{
	char server[NCP_BINDERY_NAME_LEN + 1];
#ifndef __MAKE_SULIB__
	char command[256];
	static char buf[128];
	FILE *p;
	int res;
	char *n;
#else
	long err;
	static char nearest[NCP_BINDERY_NAME_LEN + 1];
	struct nw_property prop;
	struct prop_net_address *n_addr = (struct prop_net_address *) &prop;
	struct ncp_conn conn;

	initialize_NCPL_error_table();
#endif /* __MAKE_SULIB__ */

	memset(server, 0, sizeof(server));

#ifndef __MAKE_SULIB__
	if (*server_name != NULL)
	{
		strncpy(server, *server_name, sizeof(server) - 1);
		str_upper(server);
	}
	sprintf(command, "nwsfind -t %d %s", type, server);
	p = popen(command, "r");

	if (p == NULL)
	{
		return errno;
	}
	fgets(buf, sizeof(buf), p);
	res = strlen(buf);
	if (res && (buf[res - 1] == '\n'))
	{
		buf[res - 1] = '\0';
	}
	if (((res = pclose(p)) != 0) || (ipx_sscanf_saddr(buf, server_addr) != 0))
	{
		return (*server_name != NULL)
		    ? NCPL_ET_HOST_UNKNOWN : NCPL_ET_NO_SERVER;
	}
	if (*server_name == NULL)
	{
		if ((n = strchr(buf, ' ')) == NULL)
		{
			return NCPL_ET_HOST_UNKNOWN;
		}
		*server_name = n;
	}
	return 0;
#else /* __MAKE_SULIB__ */
	memset(nearest, 0, sizeof(nearest));

	if (*server_name != NULL)
	{
		if (strlen(*server_name) >= sizeof(server))
		{
			return NCPL_ET_NAMETOOLONG;
		}
		strcpy(server, *server_name);
		str_upper(server);
	}
	if ((err = ipx_sap_find_nearest(type, server_addr, nearest)) != 0)
	{
		return err;
	}
	/* We have to ask the nearest server for our wanted server */
	memzero(conn);
	if ((err = ncp_connect_ipx_addr(&conn, server_addr, 0)) != 0)
	{
		return err;
	}
	if (*server_name == NULL)
	{
		*server_name = nearest;
		ncp_do_close(&conn);
		return 0;
	}
	/* The following optimization should have been done before
	   ncp_connect_addr. This would be convenient if there was a
	   simple way to find out whether there is a route to the
	   server. Parsing /proc/net/ipx_route is not too nice, so we
	   just connect to the server and immediately disconnect
	   again. This way we also find out if the server still has
	   free connection slots. */
	if (strcmp(server, nearest) == 0)
	{
		/* Our wanted server answered the SAP GNS request, so
		   use it */
		ncp_do_close(&conn);
		return 0;
	}
	if (ncp_read_property_value(&conn, type, server, 1,
				    "NET_ADDRESS", &prop) != 0)
	{
		ncp_do_close(&conn);
		return NCPL_ET_HOST_UNKNOWN;
	}
	if ((err = ncp_do_close(&conn)) != 0)
	{
		return err;
	}
	server_addr->sipx_family = AF_IPX;
	server_addr->sipx_network = n_addr->network;
	server_addr->sipx_port = n_addr->port;
	ipx_assign_node(server_addr->sipx_node, n_addr->node);

	/* To make the final server reachable, we connect again. See
	   above. (When can we rely on all users running ipxd??? :-)) */
	memzero(conn);
	if (((err = ncp_connect_ipx_addr(&conn, server_addr, 0)) != 0)
	    || ((err = ncp_do_close(&conn)) != 0))
	{
		return err;
	}
	return 0;
#endif /* __MAKE_SULIB__ */
}
#endif	/* CONFIG_NATIVE_IPX */

long 
ncp_find_server(const char **server_name, int type, struct sockaddr* addr, socklen_t len) {
	long err = NCPL_ET_HOST_UNKNOWN;
#ifdef CONFIG_NATIVE_IPX
	if (len >= sizeof(struct sockaddr_ipx)) {
		err = ncp_find_server_ipx(server_name, type, (struct sockaddr_ipx*)addr);
		if (!err) return 0;
	}
#endif	/* CONFIG_NATIVE_IPX */
	return err;
}

long
ncp_find_fileserver(const char *server_name, struct sockaddr* addr, socklen_t len)
{
	return ncp_find_server(&server_name, NCP_BINDERY_FSERVER, addr, len);
}

static int
ncp_login_conn(struct ncp_conn* conn, const char* object_name, int object_type, const char* password) {
	int err;

#ifdef NDS_SUPPORT
	if (!nds_get_tree_name(conn, NULL, 0))
	{
		err = nds_login_auth(conn, object_name, password);
		if (!err) return 0;
		if ((err == NCPL_ET_REQUEST_ERROR) && (conn->completion == NDS_GRACE_PERIOD)) {
			fprintf(stderr, "Your password has expired\n");
			return 0;
		}
		fprintf(stderr, "Unable to NDS log-in (error %d), trying bindery...\n", conn->completion);
	}
#endif    	
	if (ncp_login_object(conn, object_name, object_type, password) != 0)
	{
		return EACCES;
	}
	return 0;
}

static long
ncp_open_temporary(struct ncp_conn *conn,
		   const char* server)
{
	struct sockaddr addr;
	long err;

	if (server == NULL)
	{
		return ncp_connect_any(conn, 1);
	}
	if ((err = ncp_find_fileserver(server, &addr, sizeof(addr))) != 0)
	{
		return err;
	}
	if ((err = ncp_connect_addr(conn, &addr, 1)) != 0)
	{
		return err;
	}
	strcpy(conn->server, server);

	return 0;
}

static long
ncp_open_temporary2(struct ncp_conn *conn,
		     const char* address,
		     const char* server)
{
#ifdef CONFIG_NATIVE_IP
	struct sockaddr_in addr;
	long err;
	struct hostent* h;

	if (!address) return EFAULT;

	h = gethostbyname(address);
	if (!h) return NCPL_ET_HOST_UNKNOWN;
	if (h->h_addrtype != AF_INET) return EAFNOSUPPORT;
	if (h->h_length != 4) return EAFNOSUPPORT;
	addr.sin_family = AF_INET;
	addr.sin_port = htons(0x020C);
	memcpy(&addr.sin_addr.s_addr, h->h_addr, 4);

	if ((err = ncp_connect_addr(conn, (struct sockaddr*)&addr, 1)) != 0)
	{
		return err;
	}
	strcpy(conn->server, server);

	return 0;
#else	/* CONFIG_NATIVE_IP */
	(void)conn;
	(void)address;
	(void)server;
	return EAFNOSUPPORT;
#endif	/* CONFIG_NATIVE_IP */
}

char *
ncp_find_permanent(const struct ncp_conn_spec *spec)
{
	FILE *mtab;
	struct ncp_conn_ent *conn_ent;
	char *result = NULL;
	int mount_fid;
	struct ncp_fs_info i;

	initialize_NCPL_error_table();

	if ((mtab = fopen(MOUNTED, "r")) == NULL)
	{
		return NULL;
	}
	while ((conn_ent = ncp_get_conn_ent(mtab)) != NULL)
	{
		if (spec != NULL)
		{
			if ((conn_ent->uid != spec->uid)
			    || ((strlen(spec->server) != 0)
				&& (strcasecmp(conn_ent->server,
					       spec->server) != 0))
			    || ((strlen(spec->user) != 0)
				&& (strcasecmp(conn_ent->user,
					       spec->user) != 0)))
			{
				continue;
			}
		}
		mount_fid = open(conn_ent->mount_point, O_RDONLY, 0);
		if (mount_fid < 0)
		{
			continue;
		}
		i.version = NCP_GET_FS_INFO_VERSION;

		if (ioctl(mount_fid, NCP_IOC_GET_FS_INFO, &i) < 0)
		{
			close(mount_fid);
			continue;
		}
		close(mount_fid);
		result = conn_ent->mount_point;
		break;
	}

	fclose(mtab);
	errno = (result == NULL) ? ENOENT : 0;
	return result;
}

#ifdef SIGNATURES
static void
ncp_sign_init_perm(struct ncp_conn *conn)
{
	if (ioctl(conn->mount_fid, NCP_IOC_SIGN_WANTED, 
	          &conn->sign_wanted) != 0)
		conn->sign_wanted = 0;
	conn->sign_active = 0;
}
#endif

static int
ncp_open_permanent(const struct ncp_conn_spec *spec, struct ncp_conn** conn)
{
	char *mount_point;
	int err;

	if ((mount_point = ncp_find_permanent(spec)) == NULL)
	{
		return -1;
	}
	err = ncp_open_mount(mount_point, conn);
	if (!err) {
		if (spec != NULL) {
			strncpy((*conn)->server, spec->server, sizeof((*conn)->server));
			(*conn)->user = strdup(spec->user);
		} else {
			memset((*conn)->server, '\0', sizeof((*conn)->server));
			(*conn)->user = NULL;
		}
	}
	return err;
}

static struct ncp_conn *
ncp_open_2(const struct ncp_conn_spec *spec, long *err, const char* address)
{
	struct ncp_conn *result;

	initialize_NCPL_error_table();

	if (ncp_open_permanent(spec, &result) == 0) {
		return result;
	}

	result = malloc(sizeof(struct ncp_conn));

	if (result == NULL)
	{
		*err = ENOMEM;
		return NULL;
	}
	memzero(*result);

	if (spec) {
		*err = ncp_open_temporary2(result, address, spec->server);
		if (*err) *err = ncp_open_temporary(result, spec->server);
	} else {
		*err = ncp_connect_any(result, 1);
	}
	if (*err) {
		free(result);
		return NULL;
	}
	if (spec && (strlen(spec->user) != 0)) {
		*err = ncp_login_conn(result, spec->user, spec->login_type, spec->password);
		if (*err) {
			ncp_close(result);
			return NULL;
		}
		result->user = strdup(spec->user);
	}
	return result;
}

struct ncp_conn *
ncp_open(const struct ncp_conn_spec *spec, long *err) {
	return ncp_open_2(spec, err, NULL);
}

int
ncp_open_mount(const char *mount_point, struct ncp_conn** conn)
{
	struct ncp_conn *result;

	initialize_NCPL_error_table();

	*conn = NULL;

	if (strlen(mount_point) >= sizeof(result->mount_point)) {
		return ENAMETOOLONG;
	}
	result = malloc(sizeof(struct ncp_conn));

	if (result == NULL) {
		return ENOMEM;
	}
	memzero(*result);

	result->is_connected = NOT_CONNECTED;

	result->mount_fid = open(mount_point, O_RDONLY, 0);
	if (result->mount_fid < 0)
	{
		free(result);
		return ENODEV;
	}
	strcpy(result->mount_point, mount_point);
	result->is_connected = CONN_PERMANENT;

	result->i.version = NCP_GET_FS_INFO_VERSION;

	if (ioctl(result->mount_fid, NCP_IOC_GET_FS_INFO, &(result->i)) != 0)
	{
		free(result);
		return NCPL_ET_NO_NCPFS_FILE;
	}
#ifdef SIGNATURES
	ncp_sign_init_perm(result);
#endif	
	*conn = result;
	return 0;
}

static long
ncp_user_disconnect(struct ncp_conn *conn)
{
	long result;

	conn->sequence += 1;

	WSET_LH(conn->packet, 0, NCP_DEALLOC_SLOT_REQUEST);
	BSET(conn->packet, 2, conn->sequence);
	BSET(conn->packet, 3, (conn->i.connection) & 0xff);
	BSET(conn->packet, 4, 1);
	BSET(conn->packet, 5, (conn->i.connection) >> 8);
	BSET(conn->packet, 6, 0);

	if ((result = do_ncp_call(conn, sizeof(struct ncp_request_header))) != 0)
	{
		return result;
	}
	close(conn->ncp_sock);
	if (conn->wdog_sock != -1) {
		close(conn->wdog_sock);
	}

	if (conn->wdog_pid != 0)
	{
		kill(conn->wdog_pid, SIGTERM);
		wait(NULL);
	}
	return 0;
}

static long
ncp_do_close(struct ncp_conn *conn)
{
	long result = -1;

	switch (conn->is_connected)
	{
	case CONN_PERMANENT:
		result = close(conn->mount_fid);
		break;

	case CONN_TEMPORARY:
		result = ncp_user_disconnect(conn);
		break;

	default:
		break;
	}

	conn->is_connected = NOT_CONNECTED;

	if (conn->user) {
		free(conn->user);
		conn->user = NULL;	/* To be safe */
	}

	return result;
}

long
ncp_close(struct ncp_conn *conn)
{
	long result;
	if (conn == NULL)
	{
		return 0;
	}
	if ((result = ncp_do_close(conn)) != 0)
	{
		return result;
	}
	free(conn);
	return 0;
}

int 
ncp_get_mount_uid(int fid, uid_t* uid)
{
	__kernel_uid_t k_uid;
	int err;

	err = ioctl(fid, NCP_IOC_GETMOUNTUID, &k_uid);
	if (err) return err;
	*uid = k_uid;
	return 0;
}

struct ncp_conn_ent *
ncp_get_conn_ent(FILE * filep)
{
	static struct ncp_conn_ent entry;
	static char server[2 * NCPFS_MAX_CFG_USERNAME];
	char *user;
	struct mntent *mnt_ent;
	int fid;

	memzero(server);
	memzero(entry);

	while ((mnt_ent = getmntent(filep)) != NULL)
	{
		if (strcmp(mnt_ent->mnt_type, "ncpfs") != 0)
		{
			continue;
		}
		if (strlen(mnt_ent->mnt_fsname) >= sizeof(server))
		{
			continue;
		}
		strcpy(server, mnt_ent->mnt_fsname);
		user = strchr(server, '/');
		if (user == NULL)
		{
			continue;
		}
		*user = '\0';
		user += 1;
		entry.user = user;
		if ((strlen(server) >= sizeof(entry.server))
		    || (strlen(mnt_ent->mnt_dir) >= sizeof(entry.mount_point)))
		{
			continue;
		}
		strcpy(entry.server, server);
		strcpy(entry.mount_point, mnt_ent->mnt_dir);

		fid = open(entry.mount_point, O_RDONLY, 0);

		if (fid == -1)
		{
			continue;
		}
		if (ncp_get_mount_uid(fid, &entry.uid) != 0)
		{
			close(fid);
			continue;
		}
		close(fid);
		return &entry;
	}

	return NULL;
}

static struct ncp_conn_spec *
ncp_get_nwc_ent(FILE * nwc)
{
	static struct ncp_conn_spec spec;
	char line[512];
	int line_len;
	char *user;
	char *password;

	memzero(spec);
	spec.uid = getuid();

	while (fgets(line, sizeof(line), nwc) != NULL)
	{
		if ((line[0] == '\n')
		    || (line[0] == '#'))
		{
			continue;
		}
		line_len = strlen(line);
		if (line[line_len - 1] == '\n')
		{
			line[line_len - 1] = '\0';
		}
		user = strchr(line, '/');
		password = strchr(user != NULL ? user : line, ' ');

		if (password != NULL)
		{
			*password = '\0';
			password += 1;
		}
		if (user != NULL)
		{
			*user = '\0';
			user += 1;
			if (strlen(user) >= sizeof(spec.user))
			{
				continue;
			}
			strcpy(spec.user, user);
		}
		if (strlen(line) >= sizeof(spec.server))
		{
			continue;
		}
		strcpy(spec.server, line);

		if (password != NULL)
		{
			while (*password == ' ')
			{
				password += 1;
			}

			if (strlen(password) >= sizeof(spec.password))
			{
				continue;
			}
			strcpy(spec.password, password);
		}
		return &spec;
	}
	return NULL;
}

FILE *
ncp_fopen_nwc(const char *user, const char *mode, long *err)
{
	char path[MAXPATHLEN];
	char *home = NULL;
	struct stat st;

	if (mode == NULL)
	{
		mode = "r";
	}
	if (user == NULL)
	{
		home = getenv("HOME");
	} else
	{
		struct passwd *pwd;

		if ((pwd = getpwnam(user)) != NULL)
		{
			home = pwd->pw_dir;
		}
	}

	if ((home == NULL)
	    || (strlen(home) + sizeof(NWCLIENT) + 2 > sizeof(path)))
	{
		*err = ENAMETOOLONG;
		return NULL;
	}
	strcpy(path, home);
	strcat(path, "/");
	strcat(path, NWCLIENT);

	if (stat(path, &st) != 0)
	{
		*err = errno;
		return NULL;
	}
	if ((st.st_mode & (S_IRWXO | S_IRWXG)) != 0)
	{
		*err = NCPL_ET_INVALID_MODE;
		return NULL;
	}
	return fopen(path, mode);
}

struct ncp_conn_spec *
ncp_find_conn_spec2(const char *server, const char *user, const char *password,
		   int login_necessary, uid_t uid, int allow_multiple_conns, long *err)
{
	static struct ncp_conn_spec spec;

	FILE *nwc;
	struct ncp_conn_spec *nwc_ent;

	initialize_NCPL_error_table();

	*err = 0;
	memzero(spec);
	spec.uid = getuid();

	if (server != NULL)
	{
		if (strlen(server) >= sizeof(spec.server))
		{
			*err = NCPL_ET_NAMETOOLONG;
			return NULL;
		}
		strcpy(spec.server, server);
	} else
	{
		if ((nwc = ncp_fopen_nwc(NULL, NULL, err)) == NULL)
		{
			*err = NCPL_ET_NO_SERVER;
			return NULL;
		}
		nwc_ent = ncp_get_nwc_ent(nwc);
		fclose(nwc);

		if (nwc_ent == NULL)
		{
			*err = NCPL_ET_NO_SPEC;
			return NULL;
		}
		strcpy(spec.server, nwc_ent->server);
		strcpy(spec.user, nwc_ent->user);
	}

	str_upper(spec.server);

	if (login_necessary == 0)
	{
		memset(spec.user, 0, sizeof(spec.user));
		memset(spec.password, 0, sizeof(spec.password));
		return &spec;
	}
	if (user != NULL)
	{
		if (strlen(user) >= sizeof(spec.user))
		{
			*err = NCPL_ET_NAMETOOLONG;
			return NULL;
		}
		strcpy(spec.user, user);
	}
	str_upper(spec.user);
	spec.login_type = NCP_BINDERY_USER;

	if (!allow_multiple_conns) {
		struct ncp_conn* conn;

		if (ncp_open_permanent(&spec, &conn) == 0)
		{
			ncp_close(conn);
			return &spec;
		}
	}
	if (password != NULL)
	{
		if (strlen(password) >= sizeof(spec.password))
		{
			*err = NCPL_ET_NAMETOOLONG;
			return NULL;
		}
		strcpy(spec.password, password);
	} else
	{
		if ((nwc = ncp_fopen_nwc(NULL, NULL, err)) != NULL)
		{
			while ((nwc_ent = ncp_get_nwc_ent(nwc)) != NULL)
			{
				if ((strcasecmp(spec.server,
						nwc_ent->server) != 0)
				    || ((*spec.user != '\0')
					&& (strcasecmp(spec.user,
						       nwc_ent->user) != 0)))
				{
					continue;
				}
				strcpy(spec.user, nwc_ent->user);
				strcpy(spec.password, nwc_ent->password);
				break;
			}
			fclose(nwc);
		}
	}

	if (strlen(spec.user) == 0)
	{
		*err = NCPL_ET_NO_USER;
		return NULL;
	}
	if ((strlen(spec.password) == 0) && (password == NULL))
	{
		char *password;
		if (!(isatty(0) && isatty(1)))
		{
			return NULL;
		}
		printf("Logging into %s as %s\n",
		       spec.server, spec.user);

		password = getpass("Password: ");
		if (strlen(password) > sizeof(spec.password))
		{
			return NULL;
		}
		strcpy(spec.password, password);
	} else
	{
		if (strcmp(spec.password, NWC_NOPASSWORD) == 0)
		{
			*spec.password = '\0';
		}
	}

	str_upper(spec.server);
	str_upper(spec.user);
	str_upper(spec.password);
	return &spec;
}

struct ncp_conn_spec *
ncp_find_conn_spec(const char *server, const char *user, const char *password,
		   int login_necessary, uid_t uid, long *err) {
	return ncp_find_conn_spec2(server, user, password, login_necessary,
				  uid, 0, err);
}

struct ncp_conn *
ncp_initialize_2(int *argc, char **argv, int login_necessary, 
		 int login_type, long *err, int required)
{
	const char *server = NULL;
	const char *user = NULL;
	const char *password = NULL;
	const char *address = NULL;
	struct ncp_conn_spec *spec;
	int i = 1;

	int get_argument(int arg_no, const char **target)
	{
		int count = 1;

		if (target != NULL)
		{
			if (arg_no + 1 >= *argc)
			{
				/* No argument to switch */
				errno = EINVAL;
				return -1;
			}
			*target = argv[arg_no + 1];
			count = 2;
		}
		/* Delete the consumed switch from the argument list
		   and decrement the argument count */
		while (count + arg_no < *argc)
		{
			argv[arg_no] = argv[arg_no + count];
			arg_no += 1;
		}
		*argc -= count;
		return 0;
	}

	initialize_NCPL_error_table();

	*err = EINVAL;

	while (i < *argc)
	{
		if ((argv[i][0] != '-')
		    || (strlen(argv[i]) != 2))
		{
			i += 1;
			continue;
		}
		switch (argv[i][1])
		{
		case 'S':
			if (get_argument(i, &server) != 0)
			{
				return NULL;
			}
			continue;
		case 'U':
			if (get_argument(i, &user) != 0)
			{
				return NULL;
			}
			continue;
		case 'P':
			if (get_argument(i, &password) != 0)
			{
				return NULL;
			}
			continue;
		case 'n':
			if (get_argument(i, NULL) != 0)
			{
				return NULL;
			}
			password = NWC_NOPASSWORD;
			continue;
#ifdef NDS_SUPPORT
		case 'b':
			if (get_argument(i, NULL) != 0)
			{
				return NULL;
			}
			bindery_only = 1;
			continue;
#endif
#ifdef CONFIG_NATIVE_IP
		case 'A':
			if (get_argument(i, &address) != 0)
			{
				return NULL;
			}
			continue;
#endif	/* CONFIG_NATIVE_IP */
		}
		i += 1;
	}

	if ((!required) && !(server || user || password || address))
		return NULL;
	spec = ncp_find_conn_spec(server, user, password, login_necessary,
				  getuid(), err);

	if (spec == NULL)
	{
		if (login_necessary != 0)
		{
			return NULL;
		} else
		{
			return ncp_open(NULL, err);
		}
	}
	spec->login_type = login_type;

	if (login_necessary == 0)
	{
		spec->user[0] = '\0';
	}
	return ncp_open_2(spec, err, address);
}

struct ncp_conn *
ncp_initialize_as(int *argc, char **argv, int login_necessary,
		  int login_type, long *err)
{
	return ncp_initialize_2(argc, argv, login_necessary,
		  login_type, err, 1);
}

struct ncp_conn *
ncp_initialize(int *argc, char **argv,
	       int login_necessary, long *err)
{
	return ncp_initialize_as(argc, argv, login_necessary,
				 NCP_BINDERY_USER, err);
}

long
ncp_request(struct ncp_conn *conn, int function)
{
	switch (conn->is_connected)
	{
	case CONN_PERMANENT:
		return ncp_mount_request(conn, function);
	case CONN_TEMPORARY:
		return ncp_temp_request(conn, function);
	default:
	}
	return ENOTCONN;
}

/****************************************************************************/
/*                                                                          */
/* Helper functions                                                         */
/*                                                                          */
/****************************************************************************/

static inline int
min(int a, int b)
{
	return (a < b) ? a : b;
}

#ifndef __MAKE_SULIB__
struct nw_time_buffer
{
	__u8 year __attribute__((packed));
	__u8 month __attribute__((packed));
	__u8 day __attribute__((packed));
	__u8 hour __attribute__((packed));
	__u8 minute __attribute__((packed));
	__u8 second __attribute__((packed));
	__u8 wday __attribute__((packed));
};

static time_t
nw_to_ctime(struct nw_time_buffer *source)
{
	struct tm u_time;

	memzero(u_time);
	u_time.tm_sec = source->second;
	u_time.tm_min = source->minute;
	u_time.tm_hour = source->hour;
	u_time.tm_mday = source->day;
	u_time.tm_mon = source->month - 1;
	u_time.tm_year = source->year;

	if (u_time.tm_year < 80)
	{
		u_time.tm_year += 100;
	}
	return mktime(&u_time);
}
#endif /* not __MAKE_SULIB__ */

#ifndef __MAKE_SULIB__
static void
ncp_add_dword_hl(struct ncp_conn *conn, dword x)
{
	assert_conn_locked(conn);
	DSET_HL(conn->packet, conn->current_size, x);
	conn->current_size += 4;
	return;
}
#endif	/* __MAKE_SULIB__ */

void
ncp_add_mem(struct ncp_conn *conn, const void *source, int size)
{
	assert_conn_locked(conn);
	memcpy(&(conn->packet[conn->current_size]), source, size);
	conn->current_size += size;
	return;
}

static void
ncp_add_pstring(struct ncp_conn *conn, const char *s)
{
	int len = strlen(s);
	assert_conn_locked(conn);
	if (len > 255)
	{
		ncp_printf("ncpfs: string too long: %s\n", s);
		len = 255;
	}
	ncp_add_byte(conn, len);
	ncp_add_mem(conn, s, len);
	return;
}

void
ncp_init_request(struct ncp_conn *conn)
{
	ncp_lock_conn(conn);

	conn->current_size = sizeof(struct ncp_request_header);
	conn->has_subfunction = 0;
}

static void
ncp_init_request_s(struct ncp_conn *conn, int subfunction)
{
	ncp_init_request(conn);
	ncp_add_word_lh(conn, 0);	/* preliminary size */

	ncp_add_byte(conn, subfunction);

	conn->has_subfunction = 1;
}

char *
ncp_reply_data(struct ncp_conn *conn, int offset)
{
	return &(conn->packet[sizeof(struct ncp_reply_header) + offset]);
}

static byte
ncp_reply_byte(struct ncp_conn *conn, int offset)
{
	return *(byte *) (ncp_reply_data(conn, offset));
}

static word
ncp_reply_word_hl(struct ncp_conn *conn, int offset)
{
	return WVAL_HL(ncp_reply_data(conn, offset), 0);
}

#ifndef __MAKE_SULIB__
static word
ncp_reply_word_lh(struct ncp_conn *conn, int offset)
{
	return WVAL_LH(ncp_reply_data(conn, offset), 0);
}
#endif	/* __MAKE_SULIB__ */

static dword
ncp_reply_dword_hl(struct ncp_conn *conn, int offset)
{
	return DVAL_HL(ncp_reply_data(conn, offset), 0);
}

static dword
ncp_reply_dword_lh(struct ncp_conn *conn, int offset)
{
	return DVAL_LH(ncp_reply_data(conn, offset), 0);
}

/* Here the ncp calls begin
 */

static long
ncp_negotiate_buffersize(struct ncp_conn *conn,
			 int size, int *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_word_hl(conn, size);

	if ((result = ncp_request(conn, 33)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*target = min(ncp_reply_word_hl(conn, 0), size);

	ncp_unlock_conn(conn);
	return 0;
}

#ifdef SIGNATURES
static long
 ncp_negotiate_size_and_options(struct ncp_conn *conn,
			  int size, int options, 
			  int *ret_size, int *ret_options)
{
	long result;

	ncp_init_request(conn);
	ncp_add_word_hl(conn, size);
	ncp_add_byte(conn, options);

	if ((result = ncp_request(conn, 0x61)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	if (ncp_reply_word_hl(conn, 0) == 0) 
		*ret_size = size;
	else 
		*ret_size = min(ncp_reply_word_hl(conn, 0), size);
	*ret_options = ncp_reply_byte(conn, 4);

	ncp_unlock_conn(conn);
	return 0;
}
#endif

#ifndef __MAKE_SULIB__
long
ncp_get_file_server_description_strings(struct ncp_conn *conn,
					char target[512])
{
	long result;

	ncp_init_request_s(conn, 201);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(target, ncp_reply_data(conn, 0), 512);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_file_server_time(struct ncp_conn *conn, time_t * target)
{
	long result;

	ncp_init_request(conn);

	if ((result = ncp_request(conn, 20)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*target = nw_to_ctime((struct nw_time_buffer *) ncp_reply_data(conn, 0));
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_set_file_server_time(struct ncp_conn *conn, time_t * source)
{
	long result;
	int year;
	struct tm *utime = localtime(source);

	year = utime->tm_year;
	if (year > 99)
	{
		year -= 100;
	}
	ncp_init_request_s(conn, 202);
	ncp_add_byte(conn, year);
	ncp_add_byte(conn, utime->tm_mon + 1);
	ncp_add_byte(conn, utime->tm_mday);
	ncp_add_byte(conn, utime->tm_hour);
	ncp_add_byte(conn, utime->tm_min);
	ncp_add_byte(conn, utime->tm_sec);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_get_file_server_information(struct ncp_conn *conn,
				struct ncp_file_server_info *target)
{
	long result;
	ncp_init_request_s(conn, 17);
	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(target, ncp_reply_data(conn, 0), sizeof(*target));
	target->MaximumServiceConnections
	    = htons(target->MaximumServiceConnections);
	target->ConnectionsInUse
	    = htons(target->ConnectionsInUse);
	target->MaxConnectionsEverUsed
	    = htons(target->MaxConnectionsEverUsed);
	target->NumberMountedVolumes
	    = htons(target->NumberMountedVolumes);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_connlist(struct ncp_conn *conn,
		 __u16 object_type, const char *object_name,
		 int *returned_no, __u8 conn_numbers[256])
{
	long result;

	ncp_init_request_s(conn, 21);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*returned_no = ncp_reply_byte(conn, 0);
	memcpy(conn_numbers, ncp_reply_data(conn, 1), (*returned_no));
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_stations_logged_info(struct ncp_conn *conn,
			     __u32 connection,
			     struct ncp_bindery_object *target,
			     time_t * login_time)
{
	long result;
	ncp_init_request_s(conn, 28);
	ncp_add_dword_lh(conn, connection);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memset(target, 0, sizeof(*target));
	target->object_id = ncp_reply_dword_hl(conn, 0);
	target->object_type = ncp_reply_word_hl(conn, 4);
	memcpy(target->object_name, ncp_reply_data(conn, 6),
	       sizeof(target->object_name));
	*login_time = nw_to_ctime((struct nw_time_buffer *)
				  ncp_reply_data(conn, 54));
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_internet_address(struct ncp_conn *conn,
			 __u32 connection,
			 struct sockaddr_ipx *target,
			 __u8 * conn_type)
{
	long result;
	ncp_init_request_s(conn, 26);
	ncp_add_dword_lh(conn, connection);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memset(target, 0, sizeof(*target));
	target->sipx_network = ncp_reply_dword_lh(conn, 0);
	memcpy(&(target->sipx_node), ncp_reply_data(conn, 4), 6);
	target->sipx_port = ncp_reply_word_lh(conn, 10);
	*conn_type = ncp_reply_byte(conn, 12);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_send_broadcast(struct ncp_conn *conn,
		   __u8 no_conn, const __u8 * connections,
		   const char *message)
{
	long result;

	if (strlen(message) > 58)
	{
		return NCPL_ET_MSG_TOO_LONG;
	}
	ncp_init_request_s(conn, 0);
	ncp_add_byte(conn, no_conn);
	ncp_add_mem(conn, connections, no_conn);
	ncp_add_pstring(conn, message);

	result = ncp_request(conn, 21);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_send_broadcast2(struct ncp_conn *conn,
		    unsigned int conns, const unsigned int* connlist, 
		    const char* message)
{
	int i;
	long result;

	i = strlen(message);
	if (i > 255)
	{
		return NCPL_ET_MSG_TOO_LONG;
	}
	if (conns > 350)	/* max pkt len ~ 1KB */
				/* maybe do it by handshaked length ? */
		return NCPL_ET_MSG_TOO_LONG;
		
	ncp_init_request_s(conn, 0x0A);
	ncp_add_word_lh(conn, conns);
	for (;conns; --conns)
		ncp_add_dword_lh(conn, *connlist++);
	ncp_add_byte(conn, i);
	ncp_add_mem(conn, message, i);
	result = ncp_request(conn, 0x15);
	ncp_unlock_conn(conn);
	return result;
}
#endif /* not __MAKE_SULIB__ */

/*
 * target is a 8-byte buffer
 */
long
ncp_get_encryption_key(struct ncp_conn *conn,
		       char *target)
{
	long result;

	ncp_init_request_s(conn, 23);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 8)
	{
		ncp_unlock_conn(conn);
		return NCPL_ET_REPLY_FORMAT;
	}
	memcpy(target, ncp_reply_data(conn, 0), 8);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_bindery_object_id(struct ncp_conn *conn,
			  __u16 object_type,
			  const char *object_name,
			  struct ncp_bindery_object *target)
{
	long result;
	ncp_init_request_s(conn, 53);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 54)
	{
		ncp_unlock_conn(conn);
		return NCPL_ET_REPLY_FORMAT;
	}
	target->object_id = ncp_reply_dword_hl(conn, 0);
	target->object_type = ncp_reply_word_hl(conn, 4);
	memcpy(target->object_name, ncp_reply_data(conn, 6), 48);
	ncp_unlock_conn(conn);
	return 0;
}

#ifndef __MAKE_SULIB__
long
ncp_get_bindery_object_name(struct ncp_conn *conn,
			    __u32 object_id,
			    struct ncp_bindery_object *target)
{
	long result;
	ncp_init_request_s(conn, 54);
	ncp_add_dword_hl(conn, object_id);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->object_id = ncp_reply_dword_hl(conn, 0);
	target->object_type = ncp_reply_word_hl(conn, 4);
	memcpy(target->object_name, ncp_reply_data(conn, 6), 48);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_scan_bindery_object(struct ncp_conn *conn,
			__u32 last_id, __u16 object_type, char *search_string,
			struct ncp_bindery_object *target)
{
	long result;
	ncp_init_request_s(conn, 55);
	ncp_add_dword_hl(conn, last_id);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, search_string);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->object_id = ncp_reply_dword_hl(conn, 0);
	target->object_type = ncp_reply_word_hl(conn, 4);
	memcpy(target->object_name, ncp_reply_data(conn, 6),
	       NCP_BINDERY_NAME_LEN);
	target->object_flags = ncp_reply_byte(conn, 54);
	target->object_security = ncp_reply_byte(conn, 55);
	target->object_has_prop = ncp_reply_byte(conn, 56);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_create_bindery_object(struct ncp_conn *conn,
			  __u16 object_type,
			  const char *object_name,
			  __u8 object_security,
			  __u8 object_status)
{
	long result;
	ncp_init_request_s(conn, 50);
	ncp_add_byte(conn, object_status);
	ncp_add_byte(conn, object_security);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}


long
ncp_delete_bindery_object(struct ncp_conn *conn,
			  __u16 object_type,
			  const char *object_name)
{
	long result;
	ncp_init_request_s(conn, 51);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}
#endif /* not __MAKE_SULIB__ */

long
ncp_read_property_value(struct ncp_conn *conn,
			int object_type, const char *object_name,
			int segment, const char *prop_name,
			struct nw_property *target)
{
	long result;
	ncp_init_request_s(conn, 61);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_byte(conn, segment);
	ncp_add_pstring(conn, prop_name);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(target->value), ncp_reply_data(conn, 0), 128);
	target->more_flag = ncp_reply_byte(conn, 128);
	target->property_flag = ncp_reply_byte(conn, 129);
	ncp_unlock_conn(conn);
	return 0;
}

#ifndef __MAKE_SULIB__
long
ncp_scan_property(struct ncp_conn *conn,
		  __u16 object_type, const char *object_name,
		  __u32 last_id, const char *search_string,
		  struct ncp_property_info *property_info)
{
	long result;
	ncp_init_request_s(conn, 60);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_dword_hl(conn, last_id);
	ncp_add_pstring(conn, search_string);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(property_info->property_name, ncp_reply_data(conn, 0), 16);
	property_info->property_flags = ncp_reply_byte(conn, 16);
	property_info->property_security = ncp_reply_byte(conn, 17);
	property_info->search_instance = ncp_reply_dword_hl(conn, 18);
	property_info->value_available_flag = ncp_reply_byte(conn, 22);
	property_info->more_properties_flag = ncp_reply_byte(conn, 23);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_add_object_to_set(struct ncp_conn *conn,
		      __u16 object_type, const char *object_name,
		      const char *property_name,
		      __u16 member_type,
		      const char *member_name)
{
	long result;
	ncp_init_request_s(conn, 65);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_pstring(conn, property_name);
	ncp_add_word_hl(conn, member_type);
	ncp_add_pstring(conn, member_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_change_property_security(struct ncp_conn *conn,
			     __u16 object_type, const char *object_name,
			     const char *property_name,
			     __u8 property_security)
{
	long result;
	ncp_init_request_s(conn, 59);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_byte(conn, property_security);
	ncp_add_pstring(conn, property_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_create_property(struct ncp_conn *conn,
		    __u16 object_type, const char *object_name,
		    const char *property_name,
		    __u8 property_flags, __u8 property_security)
{
	long result;
	ncp_init_request_s(conn, 57);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_byte(conn, property_flags);
	ncp_add_byte(conn, property_security);
	ncp_add_pstring(conn, property_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_delete_object_from_set(struct ncp_conn *conn,
			   __u16 object_type, const char *object_name,
			   const char *property_name,
			   __u16 member_type,
			   const char *member_name)
{
	long result;
	ncp_init_request_s(conn, 66);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_pstring(conn, property_name);
	ncp_add_word_hl(conn, member_type);
	ncp_add_pstring(conn, member_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_delete_property(struct ncp_conn *conn,
		    __u16 object_type, const char *object_name,
		    const char *property_name)
{
	long result;
	ncp_init_request_s(conn, 58);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_pstring(conn, property_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_write_property_value(struct ncp_conn *conn,
			 __u16 object_type, const char *object_name,
			 const char *property_name,
			 __u8 segment,
			 const struct nw_property *property_value)
{
	long result;
	ncp_init_request_s(conn, 62);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_byte(conn, segment);
	ncp_add_byte(conn, property_value->more_flag);
	ncp_add_pstring(conn, property_name);
	ncp_add_mem(conn, property_value->value, 128);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_get_big_ncp_max_packet_size(struct ncp_conn *conn,
				__u16 proposed_max_size,
				__u8 proposed_security_flag,
				__u16 * accepted_max_size,
				__u16 * echo_socket,
				__u8 * accepted_security_flag)
{
	long result;
	ncp_init_request(conn);
	ncp_add_word_hl(conn, proposed_max_size);
	ncp_add_byte(conn, proposed_security_flag);

	if ((result = ncp_request(conn, 97)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*accepted_max_size = ncp_reply_word_hl(conn, 0);
	*echo_socket = ncp_reply_word_hl(conn, 2);
	*accepted_security_flag = ncp_reply_byte(conn, 4);
	ncp_unlock_conn(conn);
	return 0;
}
#endif /* not __MAKE_SULIB__ */

long
ncp_login_encrypted(struct ncp_conn *conn,
		    const struct ncp_bindery_object *object,
		    const unsigned char *key,
		    const unsigned char *passwd)
{
	dword tmpID = htonl(object->object_id);
	unsigned char buf[128];
	unsigned char encrypted[8];
	long result;

	shuffle((byte *) & tmpID, passwd, strlen(passwd), buf);
	nw_encrypt(key, buf, encrypted);

	ncp_init_request_s(conn, 24);
	ncp_add_mem(conn, encrypted, 8);
	ncp_add_word_hl(conn, object->object_type);
	ncp_add_pstring(conn, object->object_name);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
#ifdef SIGNATURES	
	if ((result == 0) || ((result == NCPL_ET_REQUEST_ERROR) && 
	    (conn->completion == NCP_GRACE_PERIOD)))
	{
		memcpy(buf + 16, key, 8);
		sign_init(buf, buf);
		result = ncp_sign_start(conn, buf);
	}
#endif	
	return result;
}

long
ncp_login_unencrypted(struct ncp_conn *conn,
		      __u16 object_type, const char *object_name,
		      const unsigned char *passwd)
{
	long result;
	ncp_init_request_s(conn, 20);
	ncp_add_word_hl(conn, object_type);
	ncp_add_pstring(conn, object_name);
	ncp_add_pstring(conn, passwd);
	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

#ifndef __MAKE_SULIB__
long
ncp_change_login_passwd(struct ncp_conn *conn,
			const struct ncp_bindery_object *object,
			const unsigned char *key,
			const unsigned char *oldpasswd,
			const unsigned char *newpasswd)
{
	long id = htonl(object->object_id);
	unsigned char cryptkey[8];
	unsigned char newpwd[16];	/* new passwd as stored by server */
	unsigned char oldpwd[16];	/* old passwd as stored by server */
	unsigned char len;
	long result;

	memcpy(cryptkey, key, 8);
	shuffle((byte *) & id, oldpasswd, strlen(oldpasswd), oldpwd);
	shuffle((byte *) & id, newpasswd, strlen(newpasswd), newpwd);
	nw_encrypt(cryptkey, oldpwd, cryptkey);
	newpassencrypt(oldpwd, newpwd, newpwd);
	newpassencrypt(oldpwd + 8, newpwd + 8, newpwd + 8);
	if ((len = strlen(newpasswd)) > 63)
	{
		len = 63;
	}
	len = ((len ^ oldpwd[0] ^ oldpwd[1]) & 0x7f) | 0x40;

	ncp_init_request_s(conn, 75);
	ncp_add_mem(conn, cryptkey, 8);
	ncp_add_word_hl(conn, object->object_type);
	ncp_add_pstring(conn, object->object_name);
	ncp_add_byte(conn, len);
	ncp_add_mem(conn, newpwd, 16);
	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}
#endif /* not __MAKE_SULIB__ */

long
ncp_login_user(struct ncp_conn *conn,
	       const unsigned char *username,
	       const unsigned char *password)
{
	return ncp_login_object(conn, username, NCP_BINDERY_USER, password);
}

static long
ncp_login_object(struct ncp_conn *conn,
		 const unsigned char *username,
		 int login_type,
		 const unsigned char *password)
{
	long result;
	unsigned char ncp_key[8];
	struct ncp_bindery_object user;

	if ((result = ncp_get_encryption_key(conn, ncp_key)) != 0)
	{
		return ncp_login_unencrypted(conn, login_type, username,
					     password);
	}
	if ((result = ncp_get_bindery_object_id(conn, login_type,
						username, &user)) != 0)
	{
		return result;
	}
	if ((result = ncp_login_encrypted(conn, &user,
					  ncp_key, password)) != 0)
	{
		struct nw_property p;
		struct ncp_prop_login_control *l
		= (struct ncp_prop_login_control *) &p;

		if (conn->completion != NCP_GRACE_PERIOD)
		{
			return result;
		}
		fprintf(stderr, "Your password has expired\n");

		if ((result = ncp_read_property_value(conn, NCP_BINDERY_USER,
						      username, 1,
						      "LOGIN_CONTROL",
						      &p)) == 0)
		{
			fprintf(stderr, "You have %d login attempts left\n",
				l->GraceLogins);
		}
	}
	return 0;
}

#ifndef __MAKE_SULIB__
long
ncp_get_volume_info_with_number(struct ncp_conn *conn, int n,
				struct ncp_volume_info *target)
{
	long result;
	int len;

	ncp_init_request_s(conn, 44);
	ncp_add_byte(conn, n);

	if ((result = ncp_request(conn, 22)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->total_blocks = ncp_reply_dword_lh(conn, 0);
	target->free_blocks = ncp_reply_dword_lh(conn, 4);
	target->purgeable_blocks = ncp_reply_dword_lh(conn, 8);
	target->not_yet_purgeable_blocks = ncp_reply_dword_lh(conn, 12);
	target->total_dir_entries = ncp_reply_dword_lh(conn, 16);
	target->available_dir_entries = ncp_reply_dword_lh(conn, 20);
	target->sectors_per_block = ncp_reply_byte(conn, 28);

	memzero(target->volume_name);

	len = ncp_reply_byte(conn, 29);
	if (len > NCP_VOLNAME_LEN)
	{
		ncp_printf("ncpfs: volume name too long: %d\n", len);
		ncp_unlock_conn(conn);
		return EIO;
	}
	memcpy(&(target->volume_name), ncp_reply_data(conn, 30), len);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_volume_number(struct ncp_conn *conn, const char *name, int *target)
{
	long result;

	ncp_init_request_s(conn, 5);
	ncp_add_pstring(conn, name);

	if ((result = ncp_request(conn, 22)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*target = ncp_reply_byte(conn, 0);
	ncp_unlock_conn(conn);
	return 0;
}


long
ncp_file_search_init(struct ncp_conn *conn,
		     int dir_handle, const char *path,
		     struct ncp_filesearch_info *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, dir_handle);
	ncp_add_pstring(conn, path);

	if ((result = ncp_request(conn, 62)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->volume_number = ncp_reply_byte(conn, 0);
	target->directory_id = ncp_reply_word_hl(conn, 1);
	target->sequence_no = ncp_reply_word_hl(conn, 3);
	target->access_rights = ncp_reply_byte(conn, 5);
	ncp_unlock_conn(conn);
	return 0;
}


long
ncp_file_search_continue(struct ncp_conn *conn,
			 struct ncp_filesearch_info *fsinfo,
			 int attributes, const char *name,
			 struct ncp_file_info *target)
{
	long result;

	ncp_init_request(conn);

	ncp_add_byte(conn, fsinfo->volume_number);
	ncp_add_word_hl(conn, fsinfo->directory_id);
	ncp_add_word_hl(conn, fsinfo->sequence_no);

	ncp_add_byte(conn, attributes);
	ncp_add_pstring(conn, name);

	if ((result = ncp_request(conn, 63)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	fsinfo->sequence_no = ncp_reply_word_hl(conn, 0);

	memzero(target->file_name);
	memcpy(&(target->file_name), ncp_reply_data(conn, 4),
	       NCP_MAX_FILENAME);

	target->file_attributes = ncp_reply_byte(conn, 18);
	target->file_mode = ncp_reply_byte(conn, 19);
	target->file_length = ncp_reply_dword_hl(conn, 20);
	target->creation_date = ncp_reply_word_hl(conn, 24);
	target->access_date = ncp_reply_word_hl(conn, 26);
	target->update_date = ncp_reply_word_hl(conn, 28);
	target->update_time = ncp_reply_word_hl(conn, 30);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_finfo(struct ncp_conn *conn,
	      int dir_handle, const char *path, const char *name,
	      struct ncp_file_info *target)
{
	long result;

	struct ncp_filesearch_info fsinfo;

	if ((result = ncp_file_search_init(conn, dir_handle, path,
					   &fsinfo)) != 0)
	{
		return result;
	}
	if ((result = ncp_file_search_continue(conn, &fsinfo, 0, name,
					       target)) == 0)
	{
		return result;
	}
	if ((result = ncp_file_search_init(conn, dir_handle, path,
					   &fsinfo)) != 0)
	{
		return result;
	}
	return ncp_file_search_continue(conn, &fsinfo, aDIR, name, target);
}

long
ncp_open_file(struct ncp_conn *conn,
	      int dir_handle, const char *path,
	      int attr, int access,
	      struct ncp_file_info *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, attr);
	ncp_add_byte(conn, access);
	ncp_add_pstring(conn, path);

	if ((result = ncp_request(conn, 76)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(target->file_id), ncp_reply_data(conn, 0),
	       NCP_FILE_ID_LEN);

	memzero(target->file_name);
	memcpy(&(target->file_name), ncp_reply_data(conn, 8),
	       NCP_MAX_FILENAME);

	target->file_attributes = ncp_reply_byte(conn, 22);
	target->file_mode = ncp_reply_byte(conn, 23);
	target->file_length = ncp_reply_dword_hl(conn, 24);
	target->creation_date = ncp_reply_word_hl(conn, 28);
	target->access_date = ncp_reply_word_hl(conn, 30);
	target->update_date = ncp_reply_word_hl(conn, 32);
	target->update_time = ncp_reply_word_hl(conn, 34);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_close_file(struct ncp_conn *conn, const char *file_id)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0);
	ncp_add_mem(conn, file_id, 6);

	result = ncp_request(conn, 66);
	ncp_unlock_conn(conn);
	return result;
}

static int
ncp_do_create(struct ncp_conn *conn,
	      int dir_handle, const char *path,
	      int attr,
	      struct ncp_file_info *target,
	      int function)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, attr);
	ncp_add_pstring(conn, path);

	if ((result = ncp_request(conn, function)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(target->file_id), ncp_reply_data(conn, 0),
	       NCP_FILE_ID_LEN);

	memzero(target->file_name);
	memcpy(&(target->file_name), ncp_reply_data(conn, 8),
	       NCP_MAX_FILENAME);

	target->file_attributes = ncp_reply_byte(conn, 22);
	target->file_mode = ncp_reply_byte(conn, 23);
	target->file_length = ncp_reply_dword_hl(conn, 24);
	target->creation_date = ncp_reply_word_hl(conn, 28);
	target->access_date = ncp_reply_word_hl(conn, 30);
	target->update_date = ncp_reply_word_hl(conn, 32);
	target->update_time = ncp_reply_word_hl(conn, 34);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_create_newfile(struct ncp_conn *conn,
		   int dir_handle, const char *path,
		   int attr,
		   struct ncp_file_info *target)
{
	return ncp_do_create(conn, dir_handle, path, attr, target, 77);
}

long
ncp_create_file(struct ncp_conn *conn,
		int dir_handle, const char *path,
		int attr,
		struct ncp_file_info *target)
{
	return ncp_do_create(conn, dir_handle, path, attr, target, 67);
}

long
ncp_erase_file(struct ncp_conn *conn,
	       int dir_handle, const char *path,
	       int attr)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, attr);
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 68);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_rename_file(struct ncp_conn *conn,
		int old_handle, const char *old_path,
		int attr,
		int new_handle, const char *new_path)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, old_handle);
	ncp_add_byte(conn, attr);
	ncp_add_pstring(conn, old_path);
	ncp_add_byte(conn, new_handle);
	ncp_add_pstring(conn, new_path);

	if ((result = ncp_request(conn, 69)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_create_directory(struct ncp_conn *conn,
		     int dir_handle, const char *path,
		     int inherit_mask)
{
	long result;

	ncp_init_request_s(conn, 10);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, inherit_mask);
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_delete_directory(struct ncp_conn *conn,
		     int dir_handle, const char *path)
{
	long result;

	ncp_init_request_s(conn, 11);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_add_trustee(struct ncp_conn *conn,
		int dir_handle, const char *path,
		__u32 object_id, __u8 rights)
{
	long result;

	ncp_init_request_s(conn, 13);
	ncp_add_byte(conn, dir_handle);
	ncp_add_dword_hl(conn, object_id);
	ncp_add_byte(conn, rights);
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_delete_trustee(struct ncp_conn *conn,
		   int dir_handle, const char *path, __u32 object_id)
{
	long result;

	ncp_init_request_s(conn, 14);
	ncp_add_byte(conn, dir_handle);
	ncp_add_dword_hl(conn, object_id);
	ncp_add_byte(conn, 0);
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_get_trustee(struct ncp_conn *conn, __u32 object_id,
		__u8 vol, char *path,
		__u16 * trustee, __u16 * contin)
{
	long result;

	ncp_init_request_s(conn, 71);
	ncp_add_byte(conn, vol);
	ncp_add_word_hl(conn, *contin);
	ncp_add_dword_hl(conn, object_id);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*contin = ncp_reply_word_hl(conn, 0);
	*trustee = ncp_reply_byte(conn, 6);
	strncpy(path, ncp_reply_data(conn, 8), ncp_reply_byte(conn, 7));
	path[ncp_reply_byte(conn, 7)] = 0;
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_rename_directory(struct ncp_conn *conn,
		     int dir_handle,
		     const char *old_path, const char *new_path)
{
	long result;

	ncp_init_request_s(conn, 15);
	ncp_add_byte(conn, dir_handle);
	ncp_add_pstring(conn, old_path);
	ncp_add_pstring(conn, new_path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

static void
ncp_add_handle_path(struct ncp_conn *conn,
		    __u8 vol_num,
		    __u32 dir_base, int have_dir_base,
		    const char *path)
{
	ncp_add_byte(conn, vol_num);
	ncp_add_dword_lh(conn, dir_base);
	if (have_dir_base != 0)
	{
		ncp_add_byte(conn, 1);	/* dir_base */
	} else
	{
		ncp_add_byte(conn, 0xff);	/* no handle */
	}
	if (path != NULL)
	{
		ncp_add_byte(conn, 1);	/* 1 component */
		ncp_add_pstring(conn, path);
	} else
	{
		ncp_add_byte(conn, 0);
	}
}
#endif	/* __MAKE_SULIB */

static void
ncp_extract_file_info(void *structure, struct nw_info_struct *target)
{
	__u8 *name_len;
	const int info_struct_size = sizeof(struct nw_info_struct) - 257;

	memcpy(target, structure, info_struct_size);
	name_len = (u_int8_t*)structure + info_struct_size;
	target->nameLen = *name_len;
	strncpy(target->entryName, name_len + 1, *name_len);
	target->entryName[*name_len] = '\0';
	return;
}

#ifndef __MAKE_SULIB__
long
ncp_obtain_file_or_subdir_info(struct ncp_conn *conn,
			       __u8 source_ns, __u8 target_ns,
			       __u16 search_attribs, __u32 rim,
			       __u8 vol, __u32 dirent, const char *path,
			       struct nw_info_struct *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 6);
	ncp_add_byte(conn, source_ns);
	ncp_add_byte(conn, target_ns);
	ncp_add_word_lh(conn, search_attribs);
	ncp_add_dword_lh(conn, rim);
	ncp_add_handle_path(conn, vol, dirent, 1, path);

	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	ncp_extract_file_info(ncp_reply_data(conn, 0), target);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_eff_directory_rights(struct ncp_conn *conn,
			     __u8 source_ns, __u8 target_ns,
			     __u16 search_attribs,
			     __u8 vol, __u32 dirent, const char *path,
			     __u16 * my_effective_rights)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 29);
	ncp_add_byte(conn, source_ns);
	ncp_add_byte(conn, target_ns);
	ncp_add_word_lh(conn, search_attribs);
	ncp_add_dword_lh(conn, 0);
	ncp_add_handle_path(conn, vol, dirent, 1, path);

	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*my_effective_rights = ncp_reply_word_lh(conn, 0);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_do_lookup(struct ncp_conn *conn,
	      struct nw_info_struct *dir,
	      const char *path,	/* may only be one component */
	      struct nw_info_struct *target)
{
	__u8 vol_num;
	__u32 dir_base;
	long result;
	const char *volname = NULL;

	if (target == NULL)
	{
		return EINVAL;
	}
	if (dir == NULL)
	{

		/* Access a volume's root directory */
		ncp_init_request(conn);
		ncp_add_byte(conn, 22);		/* subfunction */
		ncp_add_byte(conn, 0);	/* dos name space */
		ncp_add_byte(conn, 0);	/* reserved */
		ncp_add_byte(conn, 0);	/* reserved */
		ncp_add_byte(conn, 0);	/* reserved */
		ncp_add_handle_path(conn, 0, 0, 0,	/* no handle */
				    path);

		if ((result = ncp_request(conn, 87)) != 0)
		{
			ncp_unlock_conn(conn);
			return result;
		}
		dir_base = ncp_reply_dword_lh(conn, 4);
		vol_num = ncp_reply_byte(conn, 8);
		ncp_unlock_conn(conn);
		volname = path;
		path = NULL;
	} else
	{
		vol_num = dir->volNumber;
		dir_base = dir->DosDirNum;
	}

	ncp_init_request(conn);
	ncp_add_byte(conn, 6);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, 0);	/* dos name space as dest */
	ncp_add_word_lh(conn, 0xff);	/* get all */
	ncp_add_dword_lh(conn, RIM_ALL);
	ncp_add_handle_path(conn, vol_num, dir_base, 1,
			    path);

	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	ncp_extract_file_info(ncp_reply_data(conn, 0), target);

	if (volname != NULL)
	{
		target->nameLen = strlen(volname);
		strcpy(target->entryName, volname);
	}
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_modify_file_or_subdir_dos_info(struct ncp_conn *conn,
				   struct nw_info_struct *file,
				   __u32 info_mask,
				   struct nw_modify_dos_info *info)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 7);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, 0x8006);	/* search attribs: all */

	ncp_add_dword_lh(conn, info_mask);
	ncp_add_mem(conn, info, sizeof(*info));
	ncp_add_handle_path(conn, file->volNumber,
			    file->DosDirNum, 1, NULL);

	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_del_file_or_subdir(struct ncp_conn *conn,
		       struct nw_info_struct *dir, char *name)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 8);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, 0x8006);	/* search attribs: all */
	ncp_add_handle_path(conn, dir->volNumber,
			    dir->DosDirNum, 1, name);

	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

static inline void
ConvertToNWfromDWORD(__u32 sfd, __u8 ret[6])
{
	__u16 *dest = (__u16 *) ret;
	memcpy(&(dest[1]), &sfd, 4);
	dest[0] = dest[1] + 1;
	return;
}

long
ncp_open_create_file_or_subdir(struct ncp_conn *conn,
			       struct nw_info_struct *dir, char *name,
			       int open_create_mode,
			       __u32 create_attributes,
			       int desired_acc_rights,
			       struct nw_file_info *target)
{
	long result;

	target->opened = 0;

	ncp_init_request(conn);
	ncp_add_byte(conn, 1);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, open_create_mode);
	ncp_add_word_lh(conn, 0x8006);
	ncp_add_dword_lh(conn, RIM_ALL);
	ncp_add_dword_lh(conn, create_attributes);
	/* The desired acc rights seem to be the inherited rights mask
	   for directories */
	ncp_add_word_lh(conn, desired_acc_rights);
	ncp_add_handle_path(conn, dir->volNumber,
			    dir->DosDirNum, 1, name);

	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->opened = 1;
	target->server_file_handle = ncp_reply_dword_lh(conn, 0);
	target->open_create_action = ncp_reply_byte(conn, 4);
	ncp_extract_file_info(ncp_reply_data(conn, 5), &(target->i));
	ConvertToNWfromDWORD(target->server_file_handle, target->file_handle);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_initialize_search(struct ncp_conn *conn,
		      const struct nw_info_struct *dir,
		      int namespace,
		      struct ncp_search_seq *target)
{
	long result;

	if ((namespace < 0) || (namespace > 255))
	{
		return EINVAL;
	}
	memzero(*target);

	ncp_init_request(conn);
	ncp_add_byte(conn, 2);	/* subfunction */
	ncp_add_byte(conn, namespace);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_handle_path(conn, dir->volNumber,
			    dir->DosDirNum, 1, NULL);

	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(target->s), ncp_reply_data(conn, 0), 9);
	target->namespace = namespace;

	ncp_unlock_conn(conn);
	return 0;
}

/* Search for everything */
long
ncp_search_for_file_or_subdir2(struct ncp_conn *conn,
			       int search_attributes,
			       u_int32_t RIM,
			       struct ncp_search_seq *seq,
			       struct nw_info_struct *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 3);	/* subfunction */
	ncp_add_byte(conn, seq->namespace);
	ncp_add_byte(conn, 0);	/* data stream (???) */
	ncp_add_word_lh(conn, search_attributes);	/* Search attribs */
	ncp_add_dword_lh(conn, RIM);	/* return info mask */
	ncp_add_mem(conn, &(seq->s), 9);
	ncp_add_byte(conn, 2);	/* 2 byte pattern */
	ncp_add_byte(conn, 0xff);	/* following is a wildcard */
	ncp_add_byte(conn, '*');

	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(seq->s), ncp_reply_data(conn, 0), sizeof(seq->s));
	ncp_extract_file_info(ncp_reply_data(conn, 10), target);

	ncp_unlock_conn(conn);
	return 0;
}

/* Search for everything */
long
ncp_search_for_file_or_subdir(struct ncp_conn *conn,
			      struct ncp_search_seq *seq,
			      struct nw_info_struct *target)
{
	return ncp_search_for_file_or_subdir2(conn, 0x8006, RIM_ALL, seq, target);
}

long
ncp_ren_or_mov_file_or_subdir(struct ncp_conn *conn,
			      struct nw_info_struct *old_dir, char *old_name,
			      struct nw_info_struct *new_dir, char *new_name)
{
	long result;

	if ((old_dir == NULL) || (old_name == NULL)
	    || (new_dir == NULL) || (new_name == NULL))
		return EINVAL;

	ncp_init_request(conn);
	ncp_add_byte(conn, 4);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, 1);	/* rename flag */
	ncp_add_word_lh(conn, 0x8006);	/* search attributes */

	/* source Handle Path */
	ncp_add_byte(conn, old_dir->volNumber);
	ncp_add_dword_lh(conn, old_dir->DosDirNum);
	ncp_add_byte(conn, 1);
	ncp_add_byte(conn, 1);	/* 1 source component */

	/* dest Handle Path */
	ncp_add_byte(conn, new_dir->volNumber);
	ncp_add_dword_lh(conn, new_dir->DosDirNum);
	ncp_add_byte(conn, 1);
	ncp_add_byte(conn, 1);	/* 1 destination component */

	/* source path string */
	ncp_add_pstring(conn, old_name);
	/* dest path string */
	ncp_add_pstring(conn, new_name);

	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}


/* Create a new job entry */
long
ncp_create_queue_job_and_file(struct ncp_conn *conn,
			      __u32 queue_id,
			      struct queue_job *job)
{
	long result;

	ncp_init_request_s(conn, 121);
	ncp_add_dword_hl(conn, queue_id);
	ncp_add_mem(conn, &(job->j), sizeof(job->j));

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(job->j), ncp_reply_data(conn, 0), 78);
	ConvertToNWfromDWORD(job->j.JobFileHandle, job->file_handle);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_close_file_and_start_job(struct ncp_conn *conn,
			     __u32 queue_id,
			     struct queue_job *job)
{
	long result;

	ncp_init_request_s(conn, 127);
	ncp_add_dword_hl(conn, queue_id);
	ncp_add_dword_lh(conn, job->j.JobNumber);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_attach_to_queue(struct ncp_conn *conn,
		    __u32 queue_id)
{
	long result;

	ncp_init_request_s(conn, 111);
	ncp_add_dword_hl(conn, queue_id);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_detach_from_queue(struct ncp_conn *conn,
		      __u32 queue_id)
{
	long result;

	ncp_init_request_s(conn, 112);
	ncp_add_dword_hl(conn, queue_id);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_service_queue_job(struct ncp_conn *conn, __u32 queue_id, __u16 job_type,
		      struct queue_job *job)
{
	long result;

	ncp_init_request_s(conn, 124);
	ncp_add_dword_hl(conn, queue_id);
	ncp_add_word_hl(conn, job_type);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(job->j), ncp_reply_data(conn, 0), 78);
	ConvertToNWfromDWORD(job->j.JobFileHandle, job->file_handle);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_finish_servicing_job(struct ncp_conn *conn, __u32 queue_id,
			 __u32 job_number, __u32 charge_info)
{
	long result;

	ncp_init_request_s(conn, 131);
	ncp_add_dword_hl(conn, queue_id);
	ncp_add_dword_lh(conn, job_number);
	ncp_add_dword_hl(conn, charge_info);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_abort_servicing_job(struct ncp_conn *conn, __u32 queue_id,
			__u32 job_number)
{
	long result;

	ncp_init_request_s(conn, 132);
	ncp_add_dword_hl(conn, queue_id);
	ncp_add_dword_lh(conn, job_number);

	result = ncp_request(conn, 23);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_get_queue_length(struct ncp_conn *conn,
                      __u32 queue_id,
                      __u32 *queue_length)
{
        long result=-EINVAL;

        ncp_init_request_s(conn, 125);
        ncp_add_dword_hl(conn, queue_id);
        
        if ((result = ncp_request(conn, 23)) != 0)
                goto out;
        
        if (conn->ncp_reply_size < 12)
        {
                result=NCPL_ET_REPLY_FORMAT;
                goto out;
        }
                
        if (ncp_reply_dword_hl(conn,0) != queue_id)
        {
                printf("Ouch! Server didn't reply with same queue id in ncp_get_queue_length!\n");
                result=-EINVAL;
        }
        else
                *queue_length = ncp_reply_dword_lh(conn,8);
        
 out:
        ncp_unlock_conn(conn);
        return result;
}

long 
ncp_get_queue_job_ids(struct ncp_conn *conn,
                       __u32 queue_id,
                       __u32 queue_section,
                       __u32 *length1,
                       __u32 *length2,
                       __u32 ids[])
{
        long result;

        ncp_init_request_s(conn,129);
        ncp_add_dword_hl(conn, queue_id);
        ncp_add_dword_lh(conn, queue_section);
        
        if ((result = ncp_request(conn, 23)) != 0)
                goto out;

        if (conn->ncp_reply_size < 8)
        {
                result=NCPL_ET_REPLY_FORMAT;
                goto out;
        }
         
        *length2 = ncp_reply_dword_lh(conn,4);
        if (conn->ncp_reply_size < 8 + 4*(*length2))
        {
                result=NCPL_ET_REPLY_FORMAT;
                goto out;
        }
        if (ids) {
		int count = min(*length1, *length2)*sizeof(__u32);
		int pos;

		for (pos=0; pos<count; pos+=sizeof(__u32)) {
			*ids++ = ncp_reply_dword_lh(conn, 8+pos);
		}
	}
        *length1 = ncp_reply_dword_lh(conn,0);
        
 out:
        ncp_unlock_conn(conn);
        return result;
}

long 
ncp_get_queue_job_info(struct ncp_conn *conn,
                        __u32 queue_id,
                        __u32 job_id,
                        struct nw_queue_job_entry *jobdata)
{
        long result;

        ncp_init_request_s(conn,122);
        ncp_add_dword_hl(conn, queue_id);
        ncp_add_dword_lh(conn, job_id);
        
        if ((result = ncp_request(conn, 23)) != 0)
                goto out;
        
        if (conn->ncp_reply_size < sizeof(struct nw_queue_job_entry))
        {
                result=NCPL_ET_REPLY_FORMAT;
        }    
        else
                memcpy(jobdata,ncp_reply_data(conn,0), sizeof(struct nw_queue_job_entry));
        
out: 
       ncp_unlock_conn(conn);
        return result;
}

long
NWRemoveJobFromQueue2
(
	struct ncp_conn*	conn,
	__u32			queueID,
	__u32			jobNumber
) {
	long result;

	ncp_init_request_s(conn, 0x80);
	ncp_add_dword_hl(conn, queueID);
	ncp_add_dword_lh(conn, jobNumber);
	result = ncp_request(conn, 0x17);
	if (result) {
		if (result == NCPL_ET_REQUEST_ERROR) {
			result = 0x8900 | conn->completion;
		} else {
			result = 0x88FF;
		}
	} else {
		/* no output */
	}
	ncp_unlock_conn(conn);
	return result;
}

static int
ncp_do_read(struct ncp_conn *conn, const char *file_id,
	    __u32 offset, __u16 to_read,
	    char *target, int *bytes_read)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0);
	ncp_add_mem(conn, file_id, 6);
	ncp_add_dword_hl(conn, offset);
	ncp_add_word_hl(conn, to_read);

	if ((result = ncp_request(conn, 72)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*bytes_read = ncp_reply_word_hl(conn, 0);

	memcpy(target, ncp_reply_data(conn, 2), *bytes_read);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_read(struct ncp_conn *conn, const char *file_id,
	 off_t offset, size_t count, char *target)
{
	const int bufsize = conn->i.buffer_size;
	size_t already_read = 0;

	while (already_read < count)
	{
		int read_this_time;
		int to_read = min(bufsize - (offset % bufsize),
				  count - already_read);

		if (ncp_do_read(conn, file_id, offset, to_read,
				target, &read_this_time) != 0)
		{
			return -1;
		}
		offset += read_this_time;
		target += read_this_time;
		already_read += read_this_time;

		if (read_this_time < to_read)
		{
			break;
		}
	}
	return already_read;
}

static int
ncp_do_write(struct ncp_conn *conn, const char *file_id,
	     __u32 offset, __u16 to_write,
	     const char *source, int *bytes_written)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0);
	ncp_add_mem(conn, file_id, 6);
	ncp_add_dword_hl(conn, offset);
	ncp_add_word_hl(conn, to_write);
	ncp_add_mem(conn, source, to_write);

	if ((result = ncp_request(conn, 73)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*bytes_written = to_write;

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_write(struct ncp_conn *conn, const char *file_id,
	  off_t offset, size_t count, const char *source)
{
	const int bufsize = conn->i.buffer_size;
	size_t already_written = 0;

	while (already_written < count)
	{
		int written_this_time;
		int to_write = min(bufsize - (offset % bufsize),
				   count - already_written);

		if (ncp_do_write(conn, file_id, offset, to_write,
				 source, &written_this_time) != 0)
		{
			return -1;
		}
		offset += written_this_time;
		source += written_this_time;
		already_written += written_this_time;

		if (written_this_time < to_write)
		{
			break;
		}
	}
	return already_written;
}

long
ncp_copy_file(struct ncp_conn *conn,
	      const char source_file[6],
	      const char target_file[6],
	      __u32 source_offset,
	      __u32 target_offset,
	      __u32 count,
	      __u32 * copied_count)
{
	long result;

	ncp_init_request(conn);

	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_mem(conn, source_file, 6);
	ncp_add_mem(conn, target_file, 6);
	ncp_add_dword_hl(conn, source_offset);
	ncp_add_dword_hl(conn, target_offset);
	ncp_add_dword_hl(conn, count);

	if ((result = ncp_request(conn, 74)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*copied_count = ncp_reply_dword_hl(conn, 0);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_dealloc_dir_handle(struct ncp_conn *conn, __u8 dir_handle)
{
	long result;

	ncp_init_request_s(conn, 20);
	ncp_add_byte(conn, dir_handle);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_alloc_short_dir_handle(struct ncp_conn *conn,
			   struct nw_info_struct *dir,
			   word alloc_mode,
			   byte * target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 12);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_hl(conn, alloc_mode);
	ncp_add_handle_path(conn, dir->volNumber, dir->DosDirNum,
			    1, NULL);

	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*target = ncp_reply_byte(conn, 0);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_add_trustee_set(struct ncp_conn *conn,
		    __u8 volume_number, __u32 dir_entry,
		    __u16 rights_mask,
		    int object_count, struct ncp_trustee_struct *rights)
{
	long result = 0;

	ncp_init_request(conn);
	ncp_add_byte(conn, 10);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, 0x8006);
	ncp_add_word_lh(conn, rights_mask);
	ncp_add_word_lh(conn, object_count);
	ncp_add_handle_path(conn, volume_number, dir_entry, 1, NULL);

	while (object_count > 0)
	{
		ncp_add_dword_hl(conn, rights->object_id);
		ncp_add_word_lh(conn, rights->rights);
		object_count -= 1;
		rights += 1;
	}

	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}
#endif /* not __MAKE_SULIB__ */

#ifdef SIGNATURES
long
ncp_sign_start(struct ncp_conn *conn, const char *sign_root)
{
	char init_last[16]={0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
	                    0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10};
	struct ncp_sign_init sign_init;

	if (conn->sign_wanted)
	{
		memcpy(sign_init.sign_root, sign_root, 8);
		memcpy(sign_init.sign_last, init_last, 16);
		conn->sign_active = 1;
		if (conn->is_connected == CONN_PERMANENT)
		{
			if (ioctl(conn->mount_fid, NCP_IOC_SIGN_INIT, 
			          &sign_init))
			          return NCPL_ET_SIGNATURE_FAILED;
		}
		else
		{
			memcpy(conn->sign_root, sign_init.sign_root, 8);
			memcpy(conn->sign_last, sign_init.sign_last, 16);
		}
		    
	}
	return 0;
}
#endif

#ifdef NDS_SUPPORT
long
ncp_send_nds_frag(struct ncp_conn *conn,
		  int ndsverb,
		  const char *inbuf, size_t inbuflen,
		  char *outbuf, size_t outbufsize, size_t *outbuflen)
{
	long result;
	size_t  sizeleft, i;
	size_t	maxdatasize = 514;
	int first = 1;
	int firstReply = 1;
	int fraghnd = -1;
	int32_t	ndsCode = -399;
	size_t	replyLen = 0;
	size_t	fragLen;

	if (outbuflen) *outbuflen = 0;
	do
	{
		sizeleft = maxdatasize;
		ncp_init_request(conn);
		ncp_add_byte(conn, 2);
		ncp_add_dword_lh(conn, fraghnd);
		if (first)
		{
			ncp_add_dword_lh(conn, maxdatasize - 8);
			ncp_add_dword_lh(conn, inbuflen + 12);
			ncp_add_dword_lh(conn, 0);
			ncp_add_dword_lh(conn, ndsverb);
			ncp_add_dword_lh(conn, outbufsize);
			sizeleft -= 25;
			first = 0;
		}
		else
			sizeleft -= 5;
		i = (sizeleft > inbuflen) ? inbuflen : sizeleft;
		if (i) ncp_add_mem(conn, inbuf, i);
		inbuflen -= i;
		inbuf += i;
		if ((result = ncp_request(conn, 0x68)) != 0)
		{
			ncp_unlock_conn(conn);
			ncp_dprintf("Error in ncp_request\n");
			return result;
		}
		fragLen = ncp_reply_dword_lh(conn, 0);
		if (fragLen < 4) {
			ncp_unlock_conn(conn);
			ncp_dprintf("Fragment too short\n");
			return NCPL_ET_REPLY_FORMAT;
		}
		fraghnd = ncp_reply_dword_lh(conn, 4);
		fragLen -= 4;
		if (fragLen) {
			int hdr;

			if (firstReply) {
				ndsCode = ncp_reply_dword_lh(conn, 8);
				hdr = 12;
				fragLen -= 4;
				firstReply = 0;
			} else {
				hdr = 8;
			}
			if (fragLen > outbufsize) {
				ncp_unlock_conn(conn);
				ncp_dprintf("Fragment too large, len=%d, max=%d\n", fragLen, outbufsize);
				return NCPL_ET_REPLY_TOO_LARGE;
			}
			if (outbuf) {
				memcpy(outbuf, ncp_reply_data(conn, hdr), fragLen);
				outbuf += fragLen;
			}
			replyLen += fragLen;
		} else {
			/* if reply len == 0 then we must have something to transmit */
			/* otherwise it can cause endless loop */
			if ((fraghnd != -1) && (inbuflen == 0)) {
				ncp_unlock_conn(conn);
				ncp_dprintf("Why next fragment?\n");
				return NCPL_ET_REPLY_FORMAT;
			}
		}
		ncp_unlock_conn(conn);
		if (fraghnd != -1) {
			ncp_dprintf("Fragmented\n");
		}
	} while (fraghnd != -1);
	if (inbuflen || firstReply) {
		ncp_dprintf("InBufLen after request=%d, FirstReply=%d\n", inbuflen, firstReply);
		return NCPL_ET_REPLY_FORMAT;
	}
	if (outbuflen) *outbuflen = replyLen;
	if ((conn->completion = ndsCode) != 0) {
		ncp_dprintf("NDS error %d\n", ndsCode);
		return NCPL_ET_REQUEST_ERROR;
	}
	return 0;
}

long
ncp_send_nds(struct ncp_conn *conn, int fn,
	     const char *data_in, size_t data_in_len, 
	     char *data_out, size_t data_out_max, size_t *data_out_len)
{
	size_t i;
	long err;

	ncp_init_request(conn);
	ncp_add_byte(conn, fn);
	if (data_in) ncp_add_mem(conn, data_in, data_in_len);
	if (!(err = ncp_request(conn, 0x68)))
	{
		i = conn->ncp_reply_size;
		if (i > data_out_max) i = data_out_max;
		if (data_out)
			memcpy(data_out, ncp_reply_data(conn, 0), i);
		if (data_out_len) *data_out_len = i;
	}
	else
		if (data_out_len) *data_out_len = 0;
	ncp_unlock_conn(conn);
	return err;
}

long
ncp_change_conn_state(struct ncp_conn *conn, int new_state)
{
	long err;

	ncp_init_request_s(conn, 0x1d);
	ncp_add_dword_lh(conn, new_state);
	err = ncp_request(conn, 0x17);
	ncp_unlock_conn(conn);
	return err;
}
#endif	/* NDS_SUPPORT */

struct ncp_conn *
ncp_open_addr(struct sockaddr *target, long *err) {
	struct ncp_conn *conn;

	if (!(conn = malloc(sizeof(struct ncp_conn)))) {
		*err = ENOMEM;
		return NULL;
	}
	memzero(*conn);

	*err = ncp_connect_addr(conn, target, 1);
	if (*err) {
		free(conn);
		return NULL;
	}
	return conn;
}

int
ncp_attach_by_addr(struct sockaddr *target, struct ncp_conn** conn) 
{
	long err;

	*conn = ncp_open_addr(target, &err);
	return err;
}

long
ncp_get_broadcast_message(struct ncp_conn *conn, char message[256])
{
	long result;
	int length;

	ncp_init_request_s(conn, 0x0B);
	if ((result = ncp_request(conn, 0x15)) != 0)
	{
		ncp_unlock_conn(conn);
		ncp_init_request_s(conn, 0x01);
		if ((result = ncp_request(conn, 0x15)) != 0)
		{
			ncp_unlock_conn(conn);
			return result;
		}
	}
	length = ncp_reply_byte(conn, 0);
	message[length] = 0;
	memcpy(message, ncp_reply_data(conn, 1), length);
	ncp_unlock_conn(conn);
	return 0;
}

static void
ncp_add_handle_path2(struct ncp_conn *conn,
		     __u8 vol_num,
		     __u32 dir_base, int dir_style,
		     const unsigned char *encpath, int pathlen)
{
	ncp_add_byte(conn, vol_num);
	ncp_add_dword_lh(conn, dir_base);
	ncp_add_byte(conn, dir_style);	/* 1 = dir_base, 0xFF = no handle, 0 = handle */
	if (encpath) {
		ncp_add_mem(conn, encpath, pathlen);
	} else {
		ncp_add_byte(conn, 0);	/* empty path */
	}
}

#ifndef __MAKE_SULIB__
long
ncp_ns_scan_salvageable_file(struct ncp_conn* conn, __u8 src_ns,
			     int dirstyle, 
			     __u8 vol_num, __u32 dir_base,
			     const unsigned char *encpath, int pathlen,
			     struct ncp_deleted_file* finfo,
			     char* name, int maxnamelen)
{
	long result;
	__u8 namelen;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0x10);
	ncp_add_byte(conn, src_ns);
	ncp_add_byte(conn, 0);
	ncp_add_dword_lh(conn, RIM_NAME);
	ncp_add_dword_lh(conn, finfo->seq);
	ncp_add_handle_path2(conn, vol_num, dir_base, dirstyle, encpath, pathlen);
	result = ncp_request(conn, 0x57);
	if (result) {
		ncp_unlock_conn(conn);
		return result;
	}
	/* reply format:                        * == returned by RIM_NAME
		+00 __u32 lh	next sequence   *
		+04 __u16	deletion time   *
		+06 __u16       deletion date   *
                +08 __u32 lh	deletor ID      *
                +0C __u32 lh	volume          *
		+10 __u32 lh	directory base  *
		+14 __u32	?
		+18 __u32 lh	attributes
		+1C __u16 hl	flags ?
		+1E __u32 lh	size
		+22 __u32 	?
		+26 __u16       ?
		+28 __u16       creation time
		+2A __u16	creation date
		+2C __u32 lh	creator ID
		+30 __u16	modify time
		+32 __u16	modify date
		+34 __u32 lh	modifier ID
		+38 __u16	last access date
		+3A __u16	last archive time
		+3C __u16	last archive date
		+3E __u32 lh	last archiver ID
		+42 __u16	inherited right mask
		+44 __u8[0x18]	?
		+5C __u32 lh	owning namespace
		+60 __u8[var]	name, length preceeded  *
	*/

	if (conn->ncp_reply_size < 0x61) {
		ncp_unlock_conn(conn);
		return NCPL_ET_REPLY_FORMAT;
	}
	finfo->seq = ncp_reply_dword_lh(conn, 0x00);
	finfo->vol = ncp_reply_dword_lh(conn, 0x0C);
	finfo->base = ncp_reply_dword_lh(conn, 0x10);
	if (name) {
		namelen = ncp_reply_byte(conn, 0x60);
		if (namelen >= maxnamelen) {
			result = NCPL_ET_NAMETOOLONG;
			namelen = maxnamelen-1;
		}
		memcpy(name, ncp_reply_data(conn, 0x61), namelen);
		name[namelen] = 0;
	}
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_ns_purge_file(struct ncp_conn* conn,
	          const struct ncp_deleted_file* finfo)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0x12);
	ncp_add_byte(conn, NW_NS_DOS);
	ncp_add_byte(conn, 0);		/* reserved? */
	ncp_add_dword_lh(conn, finfo->seq);
	ncp_add_dword_lh(conn, finfo->vol);
	ncp_add_dword_lh(conn, finfo->base);
	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

struct ncpi_gfn_cookies {
		int	flag;
		int32_t	cookie1;
		int32_t	cookie2;
			};
static long
ncp_ns_get_full_name_int(struct ncp_conn* conn, __u8 src_ns, __u8 dst_ns,
		int dirstyle, __u8 vol_num, __u32 dir_base,
		const unsigned char* encpath, size_t pathlen,
		struct ncpi_gfn_cookies* cookies,
		unsigned char* name, size_t maxnamelen, unsigned char** begin)
{
	long result;
	size_t comps;
	unsigned char* putpos;
	unsigned char* getpos;
	unsigned char* getend;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0x1C);
	ncp_add_byte(conn, src_ns);
	ncp_add_byte(conn, dst_ns);
	ncp_add_word_lh(conn, cookies->flag);
	ncp_add_dword_lh(conn, cookies->cookie1);
	ncp_add_dword_lh(conn, cookies->cookie2);
	ncp_add_handle_path2(conn, vol_num, dir_base, dirstyle, encpath, pathlen);
	result = ncp_request(conn, 0x57);
	if (result) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 14) {
		ncp_unlock_conn(conn);
		return NCPL_ET_REPLY_FORMAT;
	}
	cookies->flag = ncp_reply_word_lh(conn, 0);
	cookies->cookie1 = ncp_reply_dword_lh(conn, 2);
	cookies->cookie2 = ncp_reply_dword_lh(conn, 6);
	comps = ncp_reply_word_lh(conn, 12);
	getpos = ncp_reply_data(conn, 14);
	getend = getpos + ncp_reply_word_lh(conn, 10);
	putpos = name + maxnamelen;
	while (comps--) {
		size_t partl;

		if (getpos >= getend) {
			ncp_unlock_conn(conn);
			return NCPL_ET_REPLY_FORMAT;
		}
		partl = *getpos++;
		if (getpos + partl > getend) {
			ncp_unlock_conn(conn);
			return NCPL_ET_REPLY_FORMAT;
		}
		putpos -= partl+1;
		if (putpos < name) {
			ncp_unlock_conn(conn);
			return NCPL_ET_NAMETOOLONG;
		}
		memcpy(putpos + 1, getpos, partl);
		*putpos = partl;
		getpos += partl;
	}
	ncp_unlock_conn(conn);
	*begin = putpos;
	return 0;
}

static long
ncp_ns_NW_to_path(char* name, size_t maxnamelen,
		  const unsigned char* encpath, const unsigned char* encend)
{
	char* nameend = name + maxnamelen;
	int pos = 0;

	while (encpath < encend) {
		int namel;

		if (pos >= 2) {
			if (name >= nameend) return NCPL_ET_NAMETOOLONG;
			*name++ = '/';
		}	
		namel = *encpath++;
		if (encpath + namel > encend) {
			return NCPL_ET_REPLY_FORMAT;
		}
		if (name + namel >= nameend) {
			return NCPL_ET_NAMETOOLONG;
		}
		memcpy(name, encpath, namel);
		encpath += namel;
		name += namel;
		if (pos == 0) {
			if (name >= nameend) return NCPL_ET_NAMETOOLONG;
			*name++ = ':';
		}
		pos++;
	}
	if (name >= nameend) return NCPL_ET_NAMETOOLONG;
	*name = 0;
	return 0;
}

long
ncp_ns_get_full_name(struct ncp_conn* conn, __u8 src_ns, __u8 dst_ns, 
		     int dirstyle, __u8 vol_num, __u32 dir_base, 
		     const unsigned char* encpath, size_t pathlen,
		     char* name, size_t maxnamelen)
{
	unsigned char space[1024];
	struct ncpi_gfn_cookies cookie = { 0, -1, -1};
	size_t len = sizeof(space);
	long results;

	do {
		unsigned char* npos;

		results = ncp_ns_get_full_name_int(conn, src_ns, dst_ns,
				dirstyle, vol_num, dir_base, encpath, pathlen,
				&cookie, space, len, &npos);
		if (results) return results;
		len = npos-space;
	} while (cookie.cookie2 != -1);
	return ncp_ns_NW_to_path(name, maxnamelen, space+len, space+sizeof(space));
}

int
ncp_get_conn_type(struct ncp_conn* conn) {
	if (conn) {
		switch (conn->is_connected) {
			case CONN_PERMANENT:
					return NCP_CONN_PERMANENT;
			case CONN_TEMPORARY:
					return NCP_CONN_TEMPORARY;
			default:;
		}
	}
	return NCP_CONN_INVALID;
}

#endif	/* not __MAKE_SULIB__ */

int
ncp_path_to_NW_format(const char* path, unsigned char* buff, int buffsize) 
{
	int components = 0;
	unsigned char* pos = buff+1;
	buffsize--;

	if (path != NULL) {
		if (*path == '/') path++;	/* skip optional leading / */
		while (*path) {
			const char *c;
			const char *d;
			int   l;

			c = strchr(path, '/');
			if (!c) c=path+strlen(path);
			l = c-path;
			if (components == 0) {			/* volume */
				d = strchr(path, ':');	/* can be separated by :, / or :/ */
				if (!d) d=path+strlen(path);
				if (d < c) {
					c=d;
					if (c[1]=='/') c++;	/* skip optional / after : */
					l = d-path;
				}
			}
			if (l == 0) 
				return -EINVAL;
			if (l > 255) 
				return -ENAMETOOLONG;
			if ((l != 1)||(*path!='.')) {
				if (buffsize <= l) return -ENOBUFS;
				buffsize -= l+1;
				*pos++ = l;
				memcpy(pos, path, l);
				pos+=l;
				components++;
			}
			path = c;
			if (!*c) break;
			path++;
		}
	}
	*buff = components;
	return pos-buff;
}

long
ncp_obtain_file_or_subdir_info2(struct ncp_conn *conn,
			        __u8 source_ns, __u8 target_ns,
			        __u16 search_attribs, __u32 rim,
				int dir_style,
			        __u8 vol, __u32 dirent, const unsigned char *path, 
				int pathlen, struct nw_info_struct *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 6);
	ncp_add_byte(conn, source_ns);
	ncp_add_byte(conn, target_ns);
	ncp_add_word_lh(conn, search_attribs);
	ncp_add_dword_lh(conn, rim);
	ncp_add_handle_path2(conn, vol, dirent, dir_style, path, pathlen);

	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	ncp_extract_file_info(ncp_reply_data(conn, 0), target);
	ncp_unlock_conn(conn);
	return 0;
}

