/*
 * mib_example.c --
 *
 * This file contains a sample skeletion for adding mib modules to
 * the snmp agent.
 *
 * Copyright (c) 1997
 *
 * Erik Schoenfelder		TU Braunschweig, Germany
 *
 * 
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 */

#include "mib_module.h"

/*
 * The HAVE_EXAMPLE define is per default commented in the Makefile[.in] 
 * files.
 * it is used in this file andin mib_module.h.
 *
 * note: HAVE_EXAMPLE must be defined to make this example work.
 */

#ifdef HAVE_EXAMPLE


#include <stdio.h>
#include <string.h>
#include <malloc.h>


/*
 * Definitions of tags that are used internally to read/write the
 * selected object type. These tags should be unique.
 */

#define EXAMP_VERSION		1
#define EXAMP_DESCR		2


/*
 * our mib objects and the handler:
 */

static u_char *var_examp();

static struct variable examp_variables[] = {

  /* 
   *   +-- our unique tag (see above)
   *   |
   *   |             +--- type
   *   |             |
   *   |             |       +--- access
   *   |             |       |
   *   |             |       |      +--- the handler (see below)
   *   |             |       |      |
   *   |             |       |      |        +--- length of the oids following
   *   |             |       |      |        |
   *   |             |       |      |        |   +--- the oid relative to the
   *   |             |       |      |        |   |    base (see below)
   *   v             v       v      v        v   v
   */
  {EXAMP_VERSION, INTEGER, RONLY, var_examp, 2, {1, 1}},
  {EXAMP_DESCR, STRING, RONLY, var_examp, 2, {1, 2}}
};


/*
 * our mib base:
 *
 * to walk along the example, you may run:  
 * % snmpwalk localhost public .1.3.6.1.3.4711
 *
 */
#define INTERNET_EXPERIMENTAL	1, 3, 6, 1, 3

static oid id_base [] = { INTERNET_EXPERIMENTAL, 4711 };


/*
 * initialize and register example-mib:
 * (this function is called from mib_module.c
 * and declared to be external in mib_module.h)
 */

void
examp_init()
{
  /* cmu snmp agent example mib addon: */
  mib_register (id_base, sizeof(id_base) / sizeof(oid),
		examp_variables,
		sizeof(examp_variables)/sizeof(*examp_variables),
		sizeof(*examp_variables));
}


/*
 * entry for the example mib:
 */

static u_char *
var_examp(vp, name, length, exact, var_len, write_method)
     struct variable *vp;    /* IN - ptr to variable entry that points here */
     oid     *name;	     /* IN/OUT - input name req, output name found */
     int     *length;	     /* IN/OUT - length of input and output oid's */
     int     exact;	     /* IN - TRUE if an exact match was requested. */
     int     *var_len;	     /* OUT - length of var or 0 if function ret. */
     int     (**write_method)();   /* OUT - ptr to func to set var, else 0 */
{
    oid newname[MAX_NAME_LEN];
    int result;

    /*
     * check given oid to be exact or valid; for the 
     * example:  1.3.6.1.3.4711.1.x  with 1 <= x <= 2  (and instance .0)
     */
    bcopy((char *)vp->name, (char *)newname, (int)vp->namelen * sizeof(oid));
    newname[8] = 0;
    result = compare(name, *length, newname, (int)vp->namelen + 1);
    if ((exact && (result != 0)) || (!exact && (result >= 0)))
	return NULL;
    bcopy((char *)newname, (char *)name, ((int)vp->namelen + 1) * sizeof(oid));
    *length = vp->namelen + 1;
    *write_method = 0;

    switch (vp->magic){

      case EXAMP_VERSION:
	{ static long long_return = 33;
	  *var_len = sizeof(long);
	  return (u_char *) &long_return;
	}
      case EXAMP_DESCR:
	{ static char *descr = "cmu-agent mib adding example";
	  *var_len = strlen (descr);
	  return (u_char *) descr;
	}

      default:
        ERROR("");
    }
    return NULL;
}

#endif /* HAVE_EXAMPLE */
