/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: CdrDriver.cc,v $
 * Revision 1.10  1998/10/24 14:28:59  mueller
 * Changed prototype of 'readDiskToc()'. It now accepts the name of the
 * audio file that should be placed into the generated toc-file.
 * Added virtual function 'readDisk()' that for reading disk toc and
 * ripping audio data at the same time.
 *
 * Revision 1.9  1998/10/03 15:10:38  mueller
 * Added function 'writeZeros()'.
 * Updated function 'writeData()'.
 *
 * Revision 1.8  1998/09/27 19:19:18  mueller
 * Added retrieval of control nibbles for track with 'analyzeTrack()'.
 * Added multi session mode.
 *
 * Revision 1.7  1998/09/22 19:15:13  mueller
 * Removed memory allocations during write process.
 *
 * Revision 1.6  1998/09/07 15:20:20  mueller
 * Reorganized read-toc related code.
 *
 * Revision 1.5  1998/09/06 13:34:22  mueller
 * Use 'message()' for printing messages.
 *
 * Revision 1.4  1998/08/30 19:12:19  mueller
 * Added supressing of error messages for some commands.
 * Added structure 'DriveInfo'.
 *
 * Revision 1.3  1998/08/25 19:24:07  mueller
 * Moved basic index extraction algorithm for read-toc to this class.
 * Added vendor codes.
 *
 */

static char rcsid[] = "$Id: CdrDriver.cc,v 1.10 1998/10/24 14:28:59 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <string.h>

#include "CdrDriver.h"
#include "Toc.h"
#include "util.h"

CdrDriver::StringTable CdrDriver::senseKeys[] = {
  { 0x00, "NO SENSE" },
  { 0x01, "RECOVERED ERROR" },
  { 0x02, "NOT READY" },
  { 0x03, "MEDIUM ERROR" },
  { 0x04, "HARDWARE ERROR" },
  { 0x05, "ILLEGAL REQUEST" },
  { 0x06, "UNIT ATTENTION" },
  { 0x08, "BLANK CHECK" },
  { 0x09, "VENDOR SPECIFIC" },
  { 0x0b, "ABORTED COMMAND" },
  { 0x0d, "VOLUME OVERFLOW" },
  { 0x0e, "MISCOMPARE" },
  { 0x00, (char *)0 }};

CdrDriver::CDRVendorTable CdrDriver::vendorTable[] = {
  // permanent codes
  { 97,25,60,  97,45,60, "Xcitek Inc." },
  { 97,26,50,  97,48,60, "Lead Data Inc." },
  { 97,46,40,  97,46,40, "FUJI Photo Film Co., Ltd." },
  { 97,25,20,  97,47,10, "Hitachi Maxell, Ltd." },
  { 97,27,40,  97,48,10, "Kodak Japan Limited" },
  { 97,27,50,  97,48,50, "Mitsui Chemicals, Inc." },
  { 97,27,30,  97,48,30, "Pioneer Video Corporation" },
  { 97,27,10,  97,48,20, "Plasmon Data systems Ltd." },
  { 97,27,20,  97,47,20, "Princo Corporation" },
  { 97,27,60,  97,48,00, "Ricoh Company Limited" },
  { 97,26,20,  0,0,0,    "SKC Co., Ltd." },
  { 97,24,00,  97,46,00, "Taiyo Yuden Company Limited" },
  { 97,32,00,  97,49,00, "TDK Corporation" },
  { 97,34,20,  97,50,20, "Mitsubishi Chemical Corporation" },
  { 97,28,30,  97,46,50, "Auvistar Industry Co.,Ltd." },
  { 97,28,10,  97,49,10, "GIGASTORAGE CORPORATION" },
  { 97,26,00,  97,45,00, "FORNET INTERNATIONAL PTE LTD." },
  { 97,26,40,  0,0,0,    "FUJI Photo Film Co., Ltd." },
  { 97,26,60,  97,46,60, "CMC Magnetics Corporation" },
  { 97,21,40,  0,0,0,    "Optical Disc Manufacturing Equipment" },
  { 97,31,00,  97,47,50, "Ritek Co." },
  { 97,28,20,  97,46,20, "Multi Media Masters & Machinary SA" },
  { 97,32,10,  0,0,0,    "Prodisc Technology Inc." },

  // tentative codes
  { 97,23,00,  97,49,60, "Matsushita Electric Industrial Co., Ltd." },
  { 97,30,10,  97,50,30, "CDA Datentraeger Albrechts GmbH" },
  { 97,47,60,  0,0,0,    "Prodisc Technology Inc." },
  { 97,26,10,  97,47,40, "POSTECH Corporation" },
  { 97,26,30,  0,0,0,    "OPTICAL DISC CORPRATION" },
  { 97,24,10,  97,46,10, "SONY Corporation" },
  { 97,22,40,  97,45,40, "CIS Technology Inc." },
  { 97,24,20,  97,46,30, "Computer Support Italy s.r.l." },

  { 0, 0, 0,  0, 0, 0,  NULL}
};

CdrDriver::CdrDriver(ScsiIf *scsiIf, Toc *toc) 
{
  scsiIf_ = scsiIf;
  toc_ = toc;
  
  multiSession_ = 0;

  blockLength_ = 0;
  blocksPerWrite_ = 0;
  zeroBuffer_ = NULL;
}

CdrDriver::~CdrDriver()
{
  delete[] zeroBuffer_, zeroBuffer_ = NULL;
}

// Sets multi session mode. 0: close session, 1: open next session
// Return: 0: OK
//         1: multi session not supported by driver
int CdrDriver::multiSession(int m)
{
  multiSession_ = m != 0 ? 1 : 0;

  return 0;
}

int CdrDriver::cdrVendor(Msf &code, const char **vendorId, 
			 const char **mediumType)
{
  CDRVendorTable *run = vendorTable;

  *vendorId = NULL;

  char m = code.min();
  char s = code.sec();
  char f = code.frac();

  char type = f % 10;
  f -= type;

  while (run->id != NULL) {
    if ((run->m1 == m && run->s1 == s && run->f1 == f) ||
	(run->m2 == m && run->s2 == s && run->f2 == f)) {
      *vendorId = run->id;
      break;
    }
    run++;
  }

  if (*vendorId != NULL) {
    if (type < 5) {
      *mediumType = "Long Strategy Type, e.g. Cyanine";
    }
    else {
      *mediumType = "Short Strategy Type, e.g. Phthalocyanine";
    }

    return 1;
  }

  return 0;
}

const char *CdrDriver::getFromStringTable(const StringTable *t, int code) const
{
  while (t->message != NULL) {
    if (t->code == code) {
      return t->message;
    }

    t += 1;
  }

  return NULL;
}

// Prints decoded sense message, if 'ignoreUnitAttention' is != 0 and sense
// code indicates unit attention nothing will be printed and 0 will be
// returned.
// return: 0: OK, no error
//         1: sense key indicates error
int CdrDriver::decodeSense(const unsigned char *buf, int len,
			   int ignoreUnitAttention, int showErrorMsg) const
{
  int code = buf[2] & 0x0f;
  const char *msg;

  if (code == 0 || (ignoreUnitAttention != 0 && code == 0x06)) {
    return 0;
  }

  if (showErrorMsg) {
    msg = getFromStringTable(senseKeys, code);

    message(-2, "SCSI command failed:");
    message(-2, "  sense key 0x%x: %s.", code, 
	    msg != NULL ? msg : "unknown code");
    
    if (len > 0x0c && buf[7] != 0) {
      message(-2, "  additional sense code: 0x%x", buf[0x0c]);
    }
    if (len > 0x0d && buf[7] != 0) {
      message(-2, "  additional sense code qualifier: 0x%x", buf[0x0d]);
    }
  }

  return 1;
}

// Sends SCSI command via 'scsiIf_'.
// return: see 'ScsiIf::sendCmd()'
int CdrDriver::sendCmd(const unsigned char *cmd, int cmdLen,
		       const unsigned char *dataOut, int dataOutLen,
		       unsigned char *dataIn, int dataInLen,
		       int showErrorMsg) const
{
  const unsigned char *sense;
  int senseLen;

  switch (scsiIf_->sendCmd(cmd, cmdLen, dataOut, dataOutLen, dataIn,
			   dataInLen)) {
  case 1:
    return 1;
    
  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0, showErrorMsg) != 0) {
      return 2;
    }
    break;
  }
  
  return 0;
}

// checks if unit is ready
// return: 0: OK
//         1: scsi command failed
//         2: not ready
//         3: not ready, no disk in drive
//         4: not ready, tray out

int CdrDriver::testUnitReady(int ignoreUnitAttention) const
{
  unsigned char cmd[6];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 6);

  switch (scsiIf_->sendCmd(cmd, 6, NULL, 0, NULL, 0)) {
  case 1:
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if ((sense[2] & 0x0f) == 0x02) {
      // not ready
      return 2;
    }
    else if (decodeSense(sense, senseLen, ignoreUnitAttention) != 0) {
      return 1;
    }
    else {
      return 0;
    }
  }

  return 0;
}

int CdrDriver::speed2Mult(int speed)
{
  return speed / 176;
}

int CdrDriver::mult2Speed(int mult)
{
  return mult * 177;
}

// start unit ('startStop' == 1) or stop unit ('startStop' == 0)
// return: 0: OK
//         1: scsi command failed

int CdrDriver::startStopUnit(int startStop) const
{
  unsigned char cmd[6];

  memset(cmd, 0, 6);

  cmd[0] = 0x1b;
  
  if (startStop != 0) {
    cmd[4] |= 0x01;
  }

  if (sendCmd(cmd, 6, NULL, 0, NULL, 0) != 0) {
    message(-2, "Cannot start/stop unit.");
    return 1;
  }

  return 0;
}


// blocks or unblocks tray
// return: 0: OK
//         1: scsi command failed

int CdrDriver::preventMediumRemoval(int block) const
{
  unsigned char cmd[6];

  memset(cmd, 0, 6);

  cmd[0] = 0x1e;
  
  if (block != 0) {
    cmd[4] |= 0x01;
  }

  if (sendCmd(cmd, 6, NULL, 0, NULL, 0) != 0) {
    message(-2, "Cannot prevent/allow medium removal.");
    return 1;
  }

  return 0;
}

// reset device to initial state
// return: 0: OK
//         1: scsi command failed

int CdrDriver::rezeroUnit(int showMessage) const
{
  unsigned char cmd[6];

  memset(cmd, 0, 6);

  cmd[0] = 0x01;
  
  if (sendCmd(cmd, 6, NULL, 0, NULL, 0, showMessage) != 0) {
    if (showMessage)
      message(-2, "Cannot rezero unit.");
    return 1;
  }

  return 0;
}

// Flushs cache of drive which inidcates end of write action. Errors resulting
// from this command are ignored because everything is already done and
// at most the last part of the lead-out track may be affected.
// return: 0: OK
int CdrDriver::flushCache() const
{
  unsigned char cmd[10];

  memset(cmd, 0, 10);

  cmd[0] = 0x35; // FLUSH CACHE
  
  if (sendCmd(cmd, 10, NULL, 0, NULL, 0) != 0) {
    message(-1, "Flush cache failed - ignored. Your disk will be OK.");
  }

  return 0;
}

// Reads the cd-rom capacity and stores the total number of available blocks
// in 'length'.
// return: 0: OK
//         1: SCSI command failed
int CdrDriver::readCapacity(long *length, int showMessage)
{
  unsigned char cmd[10];
  unsigned char data[8];

  memset(cmd, 0, 10);
  memset(data, 0, 8);

  cmd[0] = 0x25; // READ CD-ROM CAPACITY

  if (sendCmd(cmd, 10, NULL, 0, data, 8, showMessage) != 0) {
    if (showMessage)
      message(-2, "Cannot read capacity.");
    return 1;
  }
  
  *length = (data[0] << 24) | (data[1] << 16) | (data[2] << 8) | data[3];
  *length += 1;

  return 0;
}

// Writes data to target, the block length depends on the actual writing mode
// and is stored internally. 'len' is number of blocks to write.
// 'lba' specifies the next logical block address for writing and is updated
// by this function.
// return: 0: OK
//         1: scsi command failed
int CdrDriver::writeData(long &lba, const char *buf, long len)
{
  assert(blockLength_ > 0);
  assert(blocksPerWrite_ > 0);
  int nwritten = 0;
  int writeLen = 0;
  unsigned char cmd[10];

  memset(cmd, 0, 10);
  cmd[0] = 0x2a; // WRITE1
  
  while (len > 0) {
    writeLen = (len > blocksPerWrite_ ? blocksPerWrite_ : len);

    cmd[2] = lba >> 24;
    cmd[3] = lba >> 16;
    cmd[4] = lba >> 8;
    cmd[5] = lba;

    cmd[7] = writeLen >> 8;
    cmd[8] = writeLen & 0xff;

    if (sendCmd(cmd, 10, (unsigned char *)(buf + (nwritten * blockLength_)),
		writeLen * blockLength_, NULL, 0) != 0) {
      message(-2, "Write data failed.");
      return 1;
    }

    lba += writeLen;

    len -= writeLen;
    nwritten += writeLen;
  }
      
  return 0;
}

// Writes 'count' blocks with zero data with 'writeData'. 'lba' will be 
// updated.
// Return: 0: OK
//         1: SCSI error occured
int CdrDriver::writeZeros(long &lba, long count)
{
  assert(blockLength_ > 0);
  assert(blocksPerWrite_ > 0);
  assert(zeroBuffer_ != NULL);

  int n;
  long cnt = 0;
  long total = count * blockLength_;
  long cntMb;
  long lastMb = 0;

  while (count > 0) {
    n = (count > blocksPerWrite_ ? blocksPerWrite_ : count);

    if (writeData(lba, zeroBuffer_, n) != 0) {
      return 1;
    }
    
    cnt += n * blockLength_;

    cntMb = cnt >> 20;

    if (cntMb > lastMb) {
      message(0, "Wrote %ld of %ld MB.\r", cntMb, total >> 20);
      fflush(stdout);
      lastMb = cntMb;
    }

    count -= n;
  }

  return 0;
}

// Requests mode page 'pageCode' from device and places it into given
// buffer of maximum length 'bufLen'.
// return: 0: OK
//         1: scsi command failed
//         2: buffer too small for requested mode page
int CdrDriver::getModePage(int pageCode, unsigned char *buf, long bufLen,
			   int showErrorMsg)
{
  unsigned char cmd[10];
  long dataLen = bufLen + 8/*mode parameter header*/ + 
                          100/*spare for block descriptors*/;
  unsigned char *data = new (unsigned char)[dataLen];

  memset(cmd, 0, 10);
  memset(data, 0, dataLen);
  memset(buf, 0, bufLen);

  cmd[0] = 0x5a; // MODE SENSE
  cmd[2] = pageCode & 0x3f;
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen,  showErrorMsg) != 0) {
    delete[] data;
    return 1;
  }

  long modeDataLen = (data[0] << 8) | data[1];
  long blockDescLen = (data[6] << 8) | data[7];

  if (modeDataLen > blockDescLen + 6) {
    unsigned char *modePage = data + blockDescLen + 8;
    long modePageLen = modePage[1] + 2;

    if (modePageLen > bufLen) {
      message(-2, "Requested mode page exceeds provided buffer size.");
      delete[] data;
      return 2;
    }
    else {
      memcpy(buf, modePage, modePageLen);
      delete[] data;
      return 0;
    }
  }
  else {
    message(-2, "No mode page data received.");
    delete[] data;
    return 1;
  }
}

// Sets mode page in device specified in buffer 'modePage'
int CdrDriver::setModePage(unsigned char *modePage, int showErrorMsg)
{
  long pageLen = modePage[1] + 2;
  unsigned char cmd[10];
  long dataLen = pageLen + 8/*mode parameter header*/;
  unsigned char *data = new (unsigned char)[dataLen];

  memset(cmd, 0, 10);
  memset(data, 0, dataLen);
  
  memcpy(data + 8, modePage, pageLen);

  cmd[0] = 0x55; // MODE SELECT
  cmd[1] = 1 << 4;

  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, data, dataLen, NULL, 0, showErrorMsg) != 0) {
    delete[] data;
    return 1;
  }

  delete[] data;
  return 0;
}



// Retrieves TOC data of inserted CD.
// Return: 'NULL' on error, else array of 'CdToc' structures with 'nofTrack'
// entries 
CdToc *CdrDriver::getToc(int *cdTocLen)
{
  unsigned char cmd[10];
  unsigned short dataLen = 100 * 8 + 4;
  unsigned char data[100 * 8 + 4];
  unsigned char *p = NULL;
  int i;
  CdToc *toc;
  int nTracks;

  // read disk toc
  memset(cmd, 0, 10);
  cmd[0] = 0x43; // READ TOC
  cmd[6] = 0; // return info for all tracks
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;
  cmd[9] = 0; // return tack information
  
  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-2, "Cannot read disk toc.");
    return NULL;
  }

  nTracks = data[3] - data[2] + 1;
  if (nTracks > 99) {
    message(-2, "Got illegal toc data.");
    return NULL;
  }

  toc = new CdToc[nTracks + 1];
  
  for (i = 0, p = data + 4; i <= nTracks; i++, p += 8) {
    toc[i].track = p[2];
    toc[i].adrCtl = p[1];
    toc[i].start = (p[4] << 24) | (p[5] << 16) | (p[6] << 8) | p[7];
  }

  *cdTocLen = nTracks + 1;
  return toc;
}


// Creates 'Toc' object for inserted CD.
// audioFilename: name of audio file that is placed into TOC
// Return: newly allocated 'Toc' object or 'NULL' on error
Toc *CdrDriver::readDiskToc(const char *audioFilename)
{
  Toc *toc = NULL;
  int nofTracks = 0;
  int i, j;
  long diskStart = 0;
  CdToc *cdToc = getToc(&nofTracks);
  Msf indexIncrements[98];
  int indexIncrementCnt = 0;
  char isrcCode[13];
  unsigned char trackCtl; // control nibbles of track
  int ctlCheckOk;

  if (cdToc == NULL) {
    return NULL;
  }

  if (nofTracks <= 1) {
    message(-1, "No tracks on disk.");
    delete[] cdToc;
    return NULL;
  }

  nofTracks -= 1; // do not count lead-out

  for (i = 0; i < nofTracks; i++) {
    if ((cdToc[i].adrCtl & 0x04) != 0) {
      message(-1, "Disk contains data track - aborting.");
      delete[] cdToc;
      return NULL;
    }
  }

  toc = new Toc;

  long lastpregap = 0;
  long pregap = 0;

  for (i = 0; i < nofTracks; i++) {
    Track t(Track::AUDIO);

    Msf trackLength(cdToc[i+1].start - cdToc[i].start);

    message(0, "Analyzing track %d: start %s, ", i + 1,
	    Msf(cdToc[i].start).str());
    message(0, "length %s...", trackLength.str());

    t.preEmphasis(cdToc[i].adrCtl & 0x01);
    t.copyPermitted(cdToc[i].adrCtl & 0x02);
    t.audioType(cdToc[i].adrCtl & 0x08);

    if (i == 0 && cdToc[i].start != 0) {
      diskStart = cdToc[i].start;
      t.append(SubTrack(SubTrack::DATA,
			AudioData(AudioData::SILENCE, 
				  Msf(diskStart).samples())));
    }

    lastpregap = pregap;
    pregap = 0;

    if (lastpregap > 0) {
      message(0, "Found pre-gap: %s", Msf(lastpregap).str());
    }

    // Find index increments and pre-gap of next track
    analyzeTrack(i + 1, cdToc[i].start, cdToc[i+1].start,
		 indexIncrements, &indexIncrementCnt, 
		 i < nofTracks - 1 ? &pregap : 0, 
		 isrcCode, &trackCtl);

    trackLength = Msf(cdToc[i+1].start - cdToc[i].start - pregap + lastpregap);

    t.append(SubTrack(SubTrack::DATA,
		      AudioData(audioFilename, 
				Msf(cdToc[i].start - diskStart
				    - lastpregap).samples(), 
				trackLength.samples())));
    if (i == 0) {
      t.start(Msf(diskStart));
    }
    else {
      t.start(Msf(lastpregap));
    }

    for (j = 0; j < indexIncrementCnt; j++)
      t.appendIndex(indexIncrements[j]);

    if (isrcCode[0] != 0 && t.isrc(isrcCode) == 0) {
      message(0, "Found ISRC code.");
    }

    if ((trackCtl & 0x80) != 0) {
      // Check track against TOC control nibbles
      ctlCheckOk = 1;
      if ((trackCtl & 0x01) !=  (cdToc[i].adrCtl & 0x01)) {
	message(-1, "Pre-emphasis flag of track differs from TOC - toc file contains TOC setting.");
	ctlCheckOk = 0;
      }
      if ((trackCtl & 0x08) != (cdToc[i].adrCtl & 0x08)) {
	message(-1, "2-/4-channel-audio  flag of track differs from TOC - toc file contains TOC setting.");
	ctlCheckOk = 0;
      }

      if (ctlCheckOk) {
	message(0, "Control nibbles of track match CD-TOC settings.");
      }
    }

    toc->append(&t);
  }

  if (readCatalog(toc)) {
    message(0, "Found disk catalogue number.");
  }

  delete[] cdToc;

  return toc;
}

// Implementation is based on binary search over all sectors of actual
// track. ISRC codes are not extracted here.
int CdrDriver::analyzeTrack(int trackNr, long startLba, long endLba,
			    Msf *index, int *indexCnt, long *pregap,
			    char *isrcCode, unsigned char *ctl)
{
  isrcCode[0] = 0;
  *ctl = 0;

  if (pregap != NULL) {
    *pregap = findIndex(trackNr + 1, 0, startLba, endLba - 1);
    if (*pregap >= endLba) {
	*pregap = 0;
    }
    else if (*pregap > 0) {
      *pregap = endLba - *pregap;
    }
  }

  // check for index increments
  int ind = 2;
  long indexLba = startLba;
  *indexCnt = 0;

  do {
    if ((indexLba = findIndex(trackNr, ind, indexLba, endLba - 1)) > 0) {
      message(0, "Found index %d at %s", ind, Msf(indexLba).str());
      if (*indexCnt < 98) {
	index[*indexCnt] =  Msf(indexLba);
	*indexCnt += 1;
      }
      ind++;
    }
  } while (indexLba > 0 && indexLba < endLba);


  // Retrieve control nibbles of track, add 75 to track start so we 
  // surely get a block of current track.
  int dummy, track;
  if (getTrackIndex(startLba + 75, &track, &dummy, ctl) == 0 &&
      track == trackNr) {
    *ctl |= 0x80;
  }

  return 0;
}

int CdrDriver::getTrackIndex(long lba, int *trackNr, int *indexNr, 
			     unsigned char *ctl)
{
  return 1;
}

// Scan from lba 'trackStart' to 'trackEnd' for the position at which the
// index switchs to 'index' of track number 'track'.
// return: lba of index start postion or 0 if index was not found
long CdrDriver::findIndex(int track, int index, long trackStart,
			  long trackEnd)
{
  int actTrack;
  int actIndex;
  long start = trackStart;
  long end = trackEnd;
  long mid;

  //message(0, "findIndex: %ld - %ld", trackStart, trackEnd);

  while (start < end) {
    mid = start + ((end - start) / 2);
    
    //message(0, "Checking block %ld...", mid);
    if (getTrackIndex(mid, &actTrack, &actIndex, NULL) != 0) {
      return 0;
    }
    //message(0, "Found track %d, index %d", actTrack, actIndex);
    if ((actTrack < track || actIndex < index) && mid + 1 < trackEnd) {
      //message(0, "  Checking block %ld...", mid + 1);
      if (getTrackIndex(mid + 1, &actTrack, &actIndex, NULL) != 0) {
	return 0;
      }
      //message(0, "  Found track %d, index %d", actTrack, actIndex);
      if (actTrack == track && actIndex == index) {
	//message(0, "Found pregap at %ld", mid + 1);
	return mid;
      }
      else {
	start = mid + 1;
      }
      
    }
    else {
      end = mid;
    }
  }

  return 0;
}

// Reads toc and audio data from CD. Must be overloaded by derived class.
Toc *CdrDriver::readDisk(const char *)
{
  message(-2, 
	  "Sorry, this driver does not support reading audio data from CD.");

  return NULL;
}
