/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: Toc.cc,v $
 * Revision 1.6  1998/10/24 14:28:16  mueller
 * Fixed bug in 'readSamples()'.
 *
 * Revision 1.5  1998/10/03 14:32:31  mueller
 * Applied patch from Bjoern Fischer <bfischer@Techfak.Uni-Bielefeld.DE>.
 *
 * Revision 1.4  1998/09/22 19:17:19  mueller
 * Added seeking to and reading of samples for GUI.
 *
 * Revision 1.3  1998/09/06 12:00:26  mueller
 * Used 'message' function for messages.
 *
 */

static char rcsid[] = "$Id: Toc.cc,v 1.6 1998/10/24 14:28:16 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>

#include "Toc.h"
#include "util.h"

extern Toc *parseToc(FILE *fp, const char *filename);

Toc::Toc() : length_(0)
{
  nofTracks_ = 0;
  tracks_ = lastTrack_ = NULL;

  readTrack_ = NULL;
  readPos_ = 0;
  readPosSample_ = 0;
  iterator_ = NULL;
  open_ = 0;
  catalogValid_ = 0;
}

Toc::~Toc()
{
  TrackEntry *run = tracks_;
  TrackEntry *next;

  while (run != NULL) {
    next = run->next;
    delete run->track;
    delete run;
    run = next;
  }
}

const Track *Toc::first(Msf &start, Msf &end) const
{
  ((Toc *)this)->iterator_ = tracks_;

  return next(start, end);
}

const Track *Toc::next(Msf &start, Msf &end) const
{
  Track *t;

  if (iterator_ != NULL) {
    start = iterator_->start;
    end = iterator_->end;
    t = iterator_->track;
    ((Toc *)this)->iterator_ = iterator_->next;
    return t;
  }
  else {
    return NULL;
  }
}
      
// appends track
// return: 0: OK
//         1: first track contains pregap
int Toc::append(const Track *t)
{
  if (tracks_ == NULL) {
    tracks_ = lastTrack_ = new TrackEntry;
  }
  else {
    lastTrack_->next = new TrackEntry;
    lastTrack_ = lastTrack_->next;
  }

  lastTrack_->track = new Track(*t);
  nofTracks_ += 1;

  update();
  return 0;
}

void Toc::update()
{
  TrackEntry *run;
  long length = 0; // length of disc in blocks
  long tlength; // length of single track in blocks

  for (run = tracks_; run != NULL; run = run->next) {
    tlength = run->track->length().lba();

    run->absStart = Msf(length);
    run->start = Msf(length + run->track->start().lba());
    run->end = Msf(length + tlength);

    length += tlength;
  }

  length_ = Msf(length);
}


int Toc::openData() const
{
  int ret = 0;

  assert(open_ == 0);
  ((Toc *)this)->readTrack_ = tracks_;
  ((Toc *)this)->readPos_ = 0;
  ((Toc *)this)->readPosSample_ = 0;

  if (readTrack_ != NULL) {
    ret = readTrack_->track->openData();
  }

  ((Toc *)this)->open_ = 1;

  return ret;
}

void Toc::closeData() const
{
  if (open_ != 0) {
    if (readTrack_ != NULL) {
      readTrack_->track->closeData();
    }

    ((Toc *)this)->readTrack_ = NULL;
    ((Toc *)this)->readPos_ = 0;
    ((Toc *)this)->open_ = 0;
    ((Toc *)this)->readPosSample_ = 0;
  }
}

long Toc::readData(char *buf, long len) const
{
  long n;
  long nread = 0;

  assert(open_ != 0);
  
  if (readPos_ + len > length_.lba()) {
    if ((len = length_.lba() - readPos_) <= 0) {
      return 0;
    }
  }

  do {
    n = readTrack_->track->readData(buf + (nread * AUDIO_BLOCK_LEN), len);

    if (n < 0) {
      return -1;
    }

    if (n != len) {
      // skip to next track
      readTrack_->track->closeData();
      ((Toc *)this)->readTrack_ = readTrack_->next;

      assert(readTrack_ != NULL);

      if (readTrack_->track->openData() != 0) {
	return -1;
      }
    }
    
    nread += n;
    len -= n;
  } while (len > 0);
  
  ((Toc *)this)->readPos_ += nread;
 
  return nread;
}

Toc *Toc::read(const char *filename)
{
  FILE *fp;
  Toc *ret;

  if ((fp = fopen(filename, "r")) == NULL) {
    message(-2, "Cannot open toc file '%s' for reading: %s",
	    filename, strerror(errno));
    return NULL;
  }

  ret = parseToc(fp, filename);

  fclose(fp);

  return ret;
}

int Toc::check() const
{
  TrackEntry *t;
  int ret;

  for (t = tracks_; t != NULL; t = t->next) {
    if ((ret = t->track->check()) != 0) {
      return ret;
    }
  }

  return 0;
}

// Sets catalog number. 's' must be a string of 13 digits.
// return: 0: OK
//         1: illegal catalog string
int Toc::catalog(const char *s)
{
  int i;

  if (strlen(s) != 13) {
    catalogValid_ = 0;
    return 1;
  }

  for (i = 0; i < 13; i++) {
    if (!isdigit(s[i])) {
      catalogValid_ = 0;
      return 1;
    }
    catalog_[i] = s[i] - '0';
  }

  catalogValid_ = 1;

  return 0;
}

// writes contents in TOC file syntax
void Toc::print(ostream &out) const
{
  int i;
  TrackEntry *t;

  if (catalogValid()) {
    out << "CATALOG \"";
    for (i = 0; i < 13; i++) {
      out << (char)(catalog(i) + '0');
    }
    out << "\"" << endl;
  }

  for (t = tracks_; t != NULL; t = t->next) {
    t->track->print(out);
    out << endl;
  }
}

// find track entry that contains given sample number
// return: found track entry or 'NULL' if sample is out of range
Toc::TrackEntry *Toc::findTrack(unsigned long sample) const
{
  TrackEntry *run;

  for (run = tracks_; run != NULL; run = run->next) {
    if (sample < run->end.samples())
      return run;
  }

  return NULL;
}

// seeks to specified sample (absolute position)
// return: 0: OK
//         10: sample position out of range
//         return codes from 'Track::openData()'
int Toc::seekSample(unsigned long sample) const
{
  int ret;

  assert(open_ != 0);

  // find track that contains 'sample'
  TrackEntry *tr = findTrack(sample);

  if (tr == NULL)
    return 10;

  // open track if necessary
  if (tr != readTrack_) {
    readTrack_->track->closeData();
    ((Toc *)this)->readTrack_ = tr;

    if ((ret = readTrack_->track->openData() != 0))
      return ret;
  }

  assert(sample >= readTrack_->absStart.samples());

  unsigned long offset = sample - readTrack_->absStart.samples();

  // seek in track
  if ((ret = readTrack_->track->seekSample(offset)) != 0)
    return ret;

  ((Toc *)this)->readPosSample_ = sample;

  return 0;
}

long Toc::readSamples(Sample *buf, long len) const
{
  long n;
  long nread = 0;

  assert(open_ != 0);
  
  if (readPosSample_ + (unsigned long)len > length_.samples()) {
    if ((len = length_.samples() - readPosSample_) <= 0)
      return 0;
  }

  do {
    n = readTrack_->track->readSamples(buf + nread , len);

    if (n < 0)
      return -1;

    if (n != len) {
      // skip to next track
      readTrack_->track->closeData();
      ((Toc *)this)->readTrack_ = readTrack_->next;

      assert(readTrack_ != NULL);

      if (readTrack_->track->openData() != 0) {
	return -1;
      }
    }
    
    nread += n;
    len -= n;
  } while (len > 0);
  
  ((Toc *)this)->readPosSample_ += nread;
 
  return nread;
}
