/*
 Does a binary search on the 1st column of an rdbtable.

 Author: Carlo Strozzi <carlos@linux.it>
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define MAX_LOOK_ARGS           16
#define MAX_HAYSTACK_LENGTH     512
#define MAX_NEEDLE_LENGTH       1024
#define MAX_HEADER_LENGTH       1024
#define EMPTY                   ""

void show_help( char *my_name)
{
    printf("
        NoSQL operator: %s

Usage:  %s [options] string header body

Options:
    -help   Print this help info.
    -n      Strip header from output.
    -x      Debug option.
    -l \"xy\" List of any additional options to the Unix \"look\" program. 
            Acceptable look options are \"df\". See look(1) for details.
            The list MUST be one single string, or it must be quoted as
            shown if there are intervening blanks. For instance: '-l f'.
            Any other characters in the string will be silently ignored.
            Note: To use the \"f\" option, the input table must be sorted
            with the \"nsq-sort -f\" command.

This operator does a fast search of an rdbtable using a binary search on
the first (leftmost) column of an rdbtable, displaying any lines which
contain \"string\" as a prefix. The rdbtable being searched MUST be split
into two files, one with the table header and the other with the table
body. The latter MUST be sorted in normal (ascending) order on the first
column, as this operator does a binary search on it.
Chars that are special to the UNIX shell must be quoted. If the search word
is made up of multiple tokens, then it must be quoted to the shell to make
one single string.

This operator reads the header and body files and writes an rdbtable
via STDOUT. The table body must be seekable, so it cannot be a pipe.
If the table body is not sorted on the first column then the results
may be unpredictable.

'$Id: nsq-look.c,v 1.1 1998/05/29 20:43:01 carlos Exp $'

            ----------------------
NoSQL RDBMS, Copyright (C) 1998 Carlo Strozzi.
This program comes with ABSOLUTELY NO WARRANTY; for details
refer to the GNU General Public License.

You should have received a copy of the GNU General Public License
along with this program;  if not, write to the Free Software
Foundation, Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307
USA.
            ----------------------\n", my_name, my_name);
}

int main( int  argc, char *argv[] ) {

  /* For getopt() */
  extern char* optarg;
  extern int optind;

  /* for the rest of the program. */
  register int
    a_loop;
  char
    *my_name=argv[0],
    cmd_buf[MAX_COMMAND_LENGTH],
    look_args[MAX_LOOK_ARGS] = EMPTY,
    haystack[MAX_HAYSTACK_LENGTH] = EMPTY,
    header[MAX_HEADER_LENGTH] = EMPTY,
    needle[MAX_NEEDLE_LENGTH] = EMPTY;

  int no_hdr=0, debug=0;

  while ((a_loop = getopt(argc, argv, "xnhl:")) != EOF) {
    switch (a_loop) {
      case 'h':
        show_help(my_name);
        exit(0);
        break;
      case 'n': 
        no_hdr=1;
        break;
      case 'x': 
        debug=1;
        break;
      case 'l':
        snprintf(look_args,MAX_LOOK_ARGS,"%s", optarg);
        break;
      default:
        show_help(my_name);
        exit(1);
    }
  }

  if( optind < argc )
    snprintf(needle,MAX_NEEDLE_LENGTH,"%s", argv[optind++]);

  if( optind < argc )
    snprintf(header,MAX_HEADER_LENGTH,"%s", argv[optind++]);

  if( optind < argc )
    snprintf(haystack,MAX_HAYSTACK_LENGTH,"%s", argv[optind++]);

  /* Check for mandatory arguments. */
  if( ! strcmp( needle, EMPTY ) ||
      ! strcmp( haystack, EMPTY ) ||
      ! strcmp( header, EMPTY ) ) {
    show_help(my_name);
    exit(1);
  }

  snprintf(cmd_buf,MAX_COMMAND_LENGTH,"#
#
BEGIN {
  NULL = \"\"
  FS = \"\\t\"; OFS = FS;

  # Honour the 'debug' switch.
  if( %d ) {
    arg_vec = \"# ARGC = \" ARGC

    for( arg in ARGV )
      arg_vec = arg_vec \"\\n# ARGV[\" arg \"] = \" ARGV[arg]

    print arg_vec > \"/dev/stderr\"
  }
}
# Print header file.
{ if( ! %d )  print }

END { # Run look(1) against table body.
  look_args = \"%s\"

  # Remove forbidden look options.
  gsub( /[^df]/, NULL, look_args ) 

  # Add leading '-'.
  if( look_args != NULL ) look_args = \"-\" look_args

  # Add terminator char to look options.
  # look_args = look_args \"-t \\\"\\011\\\" \"

  look_cmd = \"look \" look_args \" %s %s \"
  if( %d )  print \"# look_cmd = \" look_cmd > \"/dev/stderr\"
  system( look_cmd )
}", debug, no_hdr, look_args, needle, haystack, debug);

  if( debug )
    fprintf (stderr, "Generated AWK program:
      ----------\n%s\n----------\n",cmd_buf);

  execlp(AWK,"awk",cmd_buf,header,NULL);
  exit(0);
}
