#include <math.h>

#include "arena.h"

#include "XGetHClrs.h"
#include "colour_types.h"
#include "colour.h"
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
#  include "debug.h"
# else
#  include "error.h"
#endif
#include "main.h"
#include "util.h"


int ColourStyle;
InstallOwnColourMap OwnColourMap = IFNECESSARY;

Colormap colourmap;

double Gamma = 2.2;
unsigned char *gamma_table = NULL;

 /* From Scott Nelson <snelson@canopus.llnl.gov> 24 bit colour  1/12/94 */
 /* these are here for speed reasons */
int RPixelShift = 16,   GPixelShift = 8,    BPixelShift = 0;
int RPixelMask  = 0xFF, GPixelMask  = 0xFF, BPixelMask  = 0xFF;

unsigned long textColour, labelColour, flybyColour,
              windowColour, strikeColour, statusColour,
              windowTopShadow, windowBottomShadow, windowShadow,
              transparent;

ArenaImageingCapability imaging; /* COLOUR888, COLOUR232, GREY4 or MONO */

unsigned long greymap[16];   /* for mixing with unsaturated colours */
unsigned long stdcmap[128];  /* 2/3/2 colour maps for gifs etc */
unsigned long  mycmap[256];

/*
 * mycmap is an array containing X11 pixel values.
 * The layout of the array is as follows:
 *
 * mycmap[0]   =    BlackPixel(display, screen);
 * mycmap[1-126]:   magic colours
 * mycmap[127] =    WhitePixel(display, screen);
 *
 * mycmap[128] =    BlackPixel(display, screen);
 * mycmap[129-142]: magic gray
 * mycmap[143] =    WhitePixel(display, screen);
 *
 * mycmap[150-159]: read only colours for application
 * mycmap[160-199]: read only colours for current document
 *
 * mycmap[200-209]: read/write colours for application
 * mycmap[210-249]: read/write colours for current document
 */

#define XX_COL_APP_START   0
#define XX_COL_APP_END   143

#define RO_COL_APP_START 150
#define RO_COL_APP_END   159

#define RO_COL_DOC_START 160
#define RO_COL_DOC_END   199

#define RW_COL_APP_START 200
#define RW_COL_APP_END   209

#define RW_COL_DOC_START 210
#define RW_COL_DOC_END   249

Byte colour_ix_start = 128, colour_ix_end = 128;

unsigned short v2b_table[256];


/*
 * This needs to be called after the visual has changed.
 */
void CalcPixelShift(void)
{
 extern Visual* visual;
 long maskval;
 int  i = 0;


 maskval = (*visual).red_mask;

 while ((maskval & 1) == 0) { i++; maskval = maskval >> 1; }

 RPixelShift = i;

 i = 0;

 maskval = (*visual).green_mask;

 while ((maskval & 1) == 0) { i++; maskval = maskval >> 1; }

 GPixelShift = i;

 i = 0;

 maskval = (*visual).blue_mask;

 while ((maskval & 1) == 0) { i++; maskval = maskval >> 1; }

 BPixelShift = i;
}


unsigned short Brightness2Voltage(unsigned short brightness)
{
 double voltage;
 static double log_a = 0; /* howcome added double */

 if (brightness == 0) return 0;

 /* cast added by howcome 21/9/94 */
 if (log_a == 0) log_a = (Gamma - 1.0) * log((double)65535);

 /* cast added by howcome 21/9/94 */
 voltage = (log_a + log((double)brightness))/Gamma;

 return (unsigned short)(0.5 + exp(voltage));
}


unsigned short V2B(unsigned short voltage)
{
 double brightness;
 static double log_a = 0;

 if (voltage == 0) return 0;

 /* cast added by howcome 21/9/94 */
 if (log_a == 0) log_a = (Gamma - 1.0) * log((double)255);
 brightness = Gamma * log((double)voltage) - log_a;

 return (unsigned short)(0.5 + exp(brightness));
}


void build_v2b_table()
{
 unsigned short i;

 for (i = 256; i--; ) v2b_table[i] = V2B(i);
}


unsigned short Voltage2Brightness(unsigned short voltage)
{
 return v2b_table[(voltage < 256 ? voltage : 255)];
}


Bool VoltageVector2BrightnessVector(unsigned short   red,
				    unsigned short green,
				    unsigned short  blue,
				    unsigned int depth,
				    unsigned short* colour_red,
				    unsigned short* colour_green,
				    unsigned short* colour_blue,
				    unsigned short* colour_grey)
{
 if (depth == 48)
   {
    *colour_red   =   red;
    *colour_green = green;
    *colour_blue  =  blue;
   }
  else
   {
    *colour_red   =   red >> 8;
    *colour_green = green >> 8;
    *colour_blue  =  blue >> 8;
   }

 if (!(imaging == COLOUR888 || imaging == TrueCOLOUR))
   {
    *colour_red   = Voltage2Brightness(*colour_red);
    *colour_green = Voltage2Brightness(*colour_green);
    *colour_blue  = Voltage2Brightness(*colour_blue);
   }

 /* Suspicious normalization */
 *colour_grey  = (3 * (*colour_red) + 6 * (*colour_green) + (*colour_blue))/10;

 return True;
}


unsigned long magic2colour(Byte ix)
{
 int r, g, b;
 unsigned long colour;

 r = (ix << 6) & 0xC0;
 g = (ix << 3) & 0xE0;
 b = (ix << 1) & 0xC0;

 GetColour(r, g, b, &colour);

 return colour;
}


Byte rgb2magic(int r, int g, int b)
{
 if (r == g && g == b)
   {
/*
    cr = Voltage2Brightness(r);
    return(128 | (cr >> 4));
*/
    return(128 | r >> 4);
   }

/*
 cr = Voltage2Brightness(r);
 cg = Voltage2Brightness(g);
 cb = Voltage2Brightness(b);

 r = cr & 0xC0;
 g = cg & 0xE0;
 b = cb & 0xC0;
*/

 return ((r >> 6) | (g >> 3) | (b >> 1));
}


long ix2colour(int ix)
{
 return mycmap[ix];
}


Byte rgb2ix(int status, int r, int g, int b, Bool blink)
{
 XColor xc;
 unsigned long pixels[1];
 int i;
 unsigned long px;
#ifdef ARENA_DEBUG
 char Iam[] = "rgb2ix";
#endif


 if (status) return (255);

 xc.red   = r << 8;
 xc.green = g << 8;
 xc.blue  = b << 8;
 xc.flags = DoRed | DoGreen | DoBlue;

 if (blink)
   {
    if (XAllocColorCells(display, colourmap, False, NULL, 0, pixels, 1))
      {
       xc.pixel = pixels[0];
       XStoreColor(display, colourmap, &xc);

       for (i = RW_COL_DOC_START;
	    (i <= RW_COL_DOC_END) && (mycmap[i] != 0);
	    i++);

       mycmap[i] = xc.pixel;
       return(i);
      }
   }

 if (!XAllocColor(display, colourmap, &xc))
   {
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
    if (VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " Can't allocate exact colour for #%2.2x%2.2x%2.2x,",
		       r, g, b);
#endif

    r = min(r, 255) & 0xC0;
    g = min(g, 255) & 0xE0;
    b = min(b, 255) & 0xC0;

#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
    if (VERBOSE_TRACE)
      Arena_DebugPrint(" using substitute #%2.2x%2.2x%2.2x\n", r, g, b);
#endif

#if 0
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
    if (VERBOSE_TRACE)
      Arena_TracePrint(Iam, " %d\n", ((r >> 6) | (g >> 3) | (b >> 1)));
#endif
#endif

    return ((r >> 6) | (g >> 3) | (b >> 1));
   }

 px = xc.pixel;

 for (i = 0; i < 144; i++)
   {
    if (mycmap[i] == px) return(i);
   }

 for (i = RO_COL_DOC_START;
      (i <= RO_COL_DOC_END) && (mycmap[i] != 0);
      i++)
   {
    if (mycmap[i] == px)
      {
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
       if (COLOUR_TRACE)
	 Arena_TracePrint(Iam,
			  " old: (%d %d %d) %d, %ld\n", r, g, b, i, px);
#endif
       return(i);
      }
   }

 if (i <= RO_COL_DOC_END)
   {
    mycmap[i] = px;
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
    if (COLOUR_TRACE)
      Arena_TracePrint(Iam,
		       " new: (%d %d %d) %d, %ld\n", r, g, b, i, px);
#endif
    return (i);
   }

#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
 Arena_TracePrint(Iam, " ran out of colours...\n");
# else
 Arena_PrintError(_("Ran out of colours...\n"));
#endif

 /* QingLong.01-02-97:
  * if ran out of colours, then let's return 0
  */
 return 0;
}


long rgb2colour(int status, int r, int g, int b, Bool rw)
{
 return (ix2colour(rgb2ix(status, r, g, b, rw)));
}


void rgbClear(void)
{
 int i;
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
 char Iam[] = "rgbClear";
#endif


#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
 if (COLOUR_TRACE) Arena_TracePrint(Iam, " ------.\n");
#endif

 for (i = RO_COL_DOC_START;
      (i <= RO_COL_DOC_END) && (mycmap[i] != 0);
      i++)
   {
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
    if (COLOUR_TRACE) Arena_TracePrint(Iam, " %d %ld\n", i, mycmap[i]);
#endif
    XFreeColors(display, colourmap, &mycmap[i], 1, 0L);
    mycmap[i] = 0;
   }

#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
 if (COLOUR_TRACE) Arena_TracePrint(Iam, " ------'\n");
#endif
}


static unsigned int range (unsigned int val, unsigned int vmax)
{
 return (val > vmax) ? vmax : val;
}


void build_gamma_table()
{
 if (!gamma_table)
   {
    int i;
    double file_gamma = 1.0 / 2.2;
    double g = 1.0 / (file_gamma * Gamma);

    gamma_table = (unsigned char *) Arena_MAlloc(256 * sizeof(char), False);
    for (i = 0; i < 256; i++)
      gamma_table[i] = (unsigned char)range((pow((double)i/255.0, g) * 255.0),
					    255);
  }
}


void free_gamma_table()
{
 if (gamma_table)
   {
    Free(gamma_table);
   }
}


int AllocGreyScale(void)
{
 unsigned long g;
 XColor colour;
#ifdef ARENA_DEBUG
 char Iam [] = "AllocGreyScale";
#endif


 greymap[0]  = BlackPixel(display, screen);
 greymap[15] = WhitePixel(display, screen);

 /*
 mycmap[128 &  0] = BlackPixel(display, screen);
 mycmap[128 & 15] = WhitePixel(display, screen);
  */

 for (g = (OwnColourMap != INSTALL); g < 16; ++g)
   {
    colour.red = colour.green = colour.blue = (g * 65535)/15;

   /* map brightness values into voltages for Gamma correction */

    colour.red   = Brightness2Voltage(colour.red);
    colour.green = Brightness2Voltage(colour.green);
    colour.blue  = Brightness2Voltage(colour.blue);

    if (XAllocColor(display, colourmap, &colour) == 0)
      {
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
       Arena_TracePrint(Iam,
			" Can't allocate standard grey palette\n");
# else
       Arena_PrintError(_("Can't allocate standard grey palette\n"));
#endif
       while (g > 1)
	 XFreeColors(display, colourmap, &(greymap[--g]), 1, 0);

       return 0;
      }
    greymap[g] = colour.pixel;
    /* mycmap[128 & g] = colour.pixel;*/
   }

 return 1;
}


int AllocStandardColours(void)
{
 unsigned long i;
 int colour_ok = 1;
 int status[128];  /* howcome 5/10/94 */
 XColor colours[128];  /* howcome 5/10/94 */
 XColor colour;
#ifdef ARENA_DEBUG
 char Iam[] = "AllocStandardColours";
#endif


 stdcmap[0]   = BlackPixel(display, screen);
 stdcmap[127] = WhitePixel(display, screen);

 mycmap[0]   = BlackPixel(display, screen);
 mycmap[127] = WhitePixel(display, screen);

 for (i = 0; i < 128; ++i)
   {
#if 1
    colour.red   = ( i       & 0x3) * 65535/3;
    colour.green = ((i >> 2) & 0x7) * 65535/7;
    colour.blue  = ((i >> 5) & 0x3) * 65535/3;
# else
    double gammaC = 0.7;
    colour.red   = pow((((double)( i       & 0x3))/3.0), gammaC) * 65535;
    colour.green = pow((((double)((i >> 2) & 0x7))/7.0), gammaC) * 65535;
    colour.blue  = pow((((double)((i >> 5) & 0x3))/3.0), gammaC) * 65535;
#endif

#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
    if (COLOUR_TRACE)
      Arena_TracePrint(Iam,
		       " exact %d %d %d\n",
		       (colour.red   >> 8),
		       (colour.green >> 8),
		       (colour.blue  >> 8));
#endif

    /* howcome 5/10/94: added support for AllocXColours which will
       speed things up. */

#if 1
    /* map brightness values into voltages for Gamma correction */

    colours[i].red   = Brightness2Voltage(colour.red);
    colours[i].green = Brightness2Voltage(colour.green);
    colours[i].blue  = Brightness2Voltage(colour.blue);
# else
    colours[i].red   = colour.red;
    colours[i].green = colour.green;
    colours[i].blue  = colour.blue; 
#endif

#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
    if (COLOUR_TRACE)
      Arena_TracePrint(Iam,
		       " %d %d %d\n",
		       (colours[i].red   >> 8),
		       (colours[i].green >> 8),
		       (colours[i].blue  >> 8));
#endif
   }

 /* howcome 5/10/94: here comes the one and only call to AllocXColours */

 if (OwnColourMap != INSTALL)
   {
    if (AllocXColours(display, colourmap, colours, i, status))
      {
       for (i = 1; i < 127; i++ )
	 {
	  stdcmap[i] = colours[i].pixel;
	   mycmap[i] = colours[i].pixel;
	 }
      }
     else
      {
       int j;
       for (j = 1; j < i ; j++)
	 if (status[j])
	   XFreeColors( display, colourmap, &colours[j].pixel, 1, 0L);

       if (OwnColourMap != INHIBIT) OwnColourMap = INSTALL;
       colour_ok = 0;
      }
   };

 /* howcome 10/8/05: added support for own colourmap */

 if (OwnColourMap == INSTALL)
   {
    int j;


    if (!colour_ok)
      {
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
       Arena_TracePrint(Iam,
			" Can't alloc colours, creating new colourmap.\n");
# else
       Arena_PrintError(_("Can't alloc colours, creating new colourmap.\n"));
#endif
       for (i = 1; i < 15 ; i++)
	 XFreeColors(display, colourmap, &(greymap[i]), 1, 0);
      };

    colourmap = XCreateColormap(display, RootWindow(display, screen),
				visual, AllocNone);
    AllocGreyScale();
    for (j = 0; j < 128; j++)
      {
       if (XAllocColor(display, colourmap, &colours[j]) == 0)
	 {
	  int g;
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
	  Arena_TracePrint(Iam,
			   "\n\tFATAL ERROR:"
			   "Can't allocate my own palette!!\n");
# else
	  Arena_PrintError(_("FATAL ERROR:"
			   " Can't allocate my own palette!!\n"));
#endif
	  g = j;
	  while (g > 1)
	    XFreeColors(display, colourmap, &(stdcmap[--g]), 1, 0);
	  return 0;
	 };
       stdcmap[j] = colours[j].pixel;
        mycmap[j] = colours[j].pixel;
      }
    return 1; /* assume success, what could go wrong? */
   }

 return 1;
}


int SupportTrueColour(void)
{
 long visual_info_mask;
 int number_visuals, i, flag;
 XVisualInfo *visual_array, visual_info_template;


 visual_info_template.screen = DefaultScreen(display);
 visual_info_mask = VisualClassMask | VisualScreenMask;

 visual_info_template.class = TrueColor;
 visual_array = XGetVisualInfo(display, visual_info_mask,
			       &visual_info_template,
			       &number_visuals);

 for (i = flag = 0; i < number_visuals; ++i)
   {
    if (visual_array[i].depth == 16)
      {
       long int maskval;
       int j = 0, k;


       maskval = (*visual).blue_mask;
       while ((maskval & 1) == 0)
	 {
	  j++;
	  maskval = maskval >> 1;
	 }

       maskval = (*visual).blue_mask;
       BPixelMask = 0;
       for (k = 0; k < 16; k++)
	 {
	  if (maskval & 1)
	    {
	     if (!BPixelMask)
	       BPixelMask = 1;
	      else
	       BPixelMask += BPixelMask + 1;
	    }

	  maskval = maskval >> 1;
	 }

       j = 0;
       maskval = (*visual).green_mask;
       while ((maskval & 1) == 0)
	 {
	  j++;
	  maskval = maskval >> 1;
	 }

       GPixelShift = j;
       GPixelMask = 0;
       for (k = 0; k < 16; k++)
	 {
	  if (maskval & 1)
	    {
	     if (!GPixelMask)
	       GPixelMask = 1;
	      else
	       GPixelMask += GPixelMask + 1;
	    }

	  maskval = maskval >> 1;
	 }

       j = 0;
       maskval = (*visual).red_mask;
       while ((maskval & 1) == 0)
	 {
	  j++;
	  maskval = maskval >> 1;
	 }

       RPixelShift = j;
       RPixelMask = 0;
       for (k = 0; k < 16; k++)
	 {
	  if (maskval & 1)
	    {
	     if (!RPixelMask)
	       RPixelMask = 1;
	      else
	       RPixelMask += RPixelMask + 1;
	    }

	  maskval = maskval >> 1;
	 }

       flag = 1;
      }

    if (visual_array[i].depth == 24)
      {
       long int maskval;
       int j = 0;


       maskval = (*visual).blue_mask;
       while ((maskval & 1) == 0)
	 {
	  j++;
	  maskval = maskval >> 1;
	 }

       BPixelShift = j;
       j = 0;
       maskval = (*visual).green_mask;
       while ((maskval & 1) == 0)
	 {
	  j++;
	  maskval = maskval >> 1;
	 }

       GPixelShift = j;
       j = 0;
       maskval = (*visual).red_mask;
       while ((maskval & 1) == 0)
	 {
	  j++;
	  maskval = maskval >> 1;
	 }

       RPixelShift = j;
       flag = 1;
       break;
      }
   }

 XFree((void*)visual_array);
 return flag;
}


/*
 * Convert colour to char sequence represenatation.
 */
Bool ArenaPackColour_to_chars(unsigned long theColour,
			      unsigned int depth,
			      ArenaImageingCapability imageing,
			      unsigned char* theColourL)
{
#ifdef ARENA_DEBUG
 char Iam[] = "ArenaPackColour_to_chars";
#endif


 if (theColourL)
   {
    switch (depth)
      {
       case 1:
	 theColourL[0] = theColour & 0x01;
	 break;

       case 2:
	 theColourL[0] = theColour & 0x03;
	 break;

       case 4:
	 theColourL[0] = theColour & 0x0F;
	 break;

       case 8:
	 theColourL[0] = theColour & 0xFF;
	 break;

       case 16:
	 theColourL[0] = (theColour & 0x00FF);   /* LSB first! */
	 theColourL[1] = (theColour & 0xFF00) >> 8;
	 break;

       case 24:
	 theColourL[0] = '\0';
	 theColourL[1] = (theColour >> RPixelShift) & RPixelMask;
	 theColourL[2] = (theColour >> GPixelShift) & GPixelMask;
	 theColourL[3] = (theColour >> BPixelShift) & BPixelMask;
	 break;

       default:
#ifdef ARENA_DEBUG
	 if (IMAGE_TRACE)
	   Arena_TracePrint(Iam,
			    " ERROR! Cannot pack colour for depth %d.\n",
			    depth);
#endif
	 return False;
	 break;
      }
    /* End ``switch (depth)'' */

    return True;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (COLOUR_TRACE)
      Arena_TracePrint(Iam, " ERROR! The colour pack array is NULL.\n");
#endif
    return False;
   }
}


/*
 * Look for the colour in current colourmap.
 */
Bool ArenaExactColour(unsigned long theColour)
{
 Bool theExactColour_found = False;
 int i, colour_i = 0;


 for (i = XX_COL_APP_START; i <= XX_COL_APP_END; i++)
   if (mycmap[i] == theColour)
     {
      theExactColour_found = True;
      colour_i = i;
      break;
     }

 if (!theExactColour_found)
   for (i = RO_COL_APP_START; i <= RW_COL_DOC_END; i++)
     if (mycmap[i] == theColour)
       {
	theExactColour_found = True;
	colour_i = i;
	break;
       }

 return theExactColour_found;
}


#if 0
/*
 * Look for the RGB in current colourmap. Pack to chars array if found.
 */
Bool ArenaExactRGB(unsigned short red,
		   unsigned short green,
		   unsigned short blue,
		   unsigned int depth, ArenaImageingCapability imageing,
		   unsigned char* theColourL)
{
}
#endif


int Pixel4Colour(XColor* colour, unsigned long *pixel)
{
 unsigned long r, g, b;

 r = gamma_table[colour->red   >> 8];
 g = gamma_table[colour->green >> 8];
 b = gamma_table[colour->blue  >> 8];

 *pixel = (b << BPixelShift | g << GPixelShift | r << RPixelShift ); 

 return 1;
}


int Pixel4PngColour(XColor* colour, unsigned long *pixel)
{
 unsigned long r, g, b;

 r = colour->red   >> 8;
 g = colour->green >> 8;
 b = colour->blue  >> 8;

 *pixel = (b << BPixelShift | g << GPixelShift | r << RPixelShift); 

 return 1;
}


int Pixel2Colour(XColor* colour, unsigned long *pixel)
{
 unsigned long r, g, b;

 r = (colour->red   * (long)(RPixelMask + 1)) >> 16;
 g = (colour->green * (long)(GPixelMask + 1)) >> 16;
 b = (colour->blue  * (long)(BPixelMask + 1)) >> 16;

 *pixel = (b << BPixelShift | g << GPixelShift | r << RPixelShift);
 return 1;
}


int Pixel2PngColour(XColor* colour, unsigned long *pixel)
{
 unsigned long r, g, b;

 r = (colour->red   * (long)(RPixelMask + 1)) >> 16;
 g = (colour->green * (long)(GPixelMask + 1)) >> 16;
 b = (colour->blue  * (long)(BPixelMask + 1)) >> 16;

 *pixel = (b << BPixelShift | g << GPixelShift | r << RPixelShift);
 return 1;
}


int GetNamedColour(char *name, unsigned long *pix)
{
 XColor colour;
#ifdef ARENA_DEBUG
 char Iam[] = "GetNamedColour";
#endif


 switch (depth)
   {
    case 16:
      if (XParseColor(display, colourmap, name, &colour))
	return Pixel2Colour(&colour, pix);
      break;

    case 24:
      if (XParseColor(display, colourmap, name, &colour))
	return Pixel4Colour(&colour, pix);
      break;

    default:
      if (XParseColor(display, colourmap, name, &colour) &&
	  XAllocColor(display, colourmap, &colour))
	{
	 *pix = colour.pixel;
	 return 1;
	}
      break;
   }
 /* End ``switch (depth)'' */

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (X_TRACE)
   Arena_TracePrint(Iam, " can't allocate named colour \"%s\".\n", name);
#endif
 return 0;
}


int GetColour(int red, int green, int blue, unsigned long *pix)
{
 XColor colour;
#ifdef ARENA_DEBUG
 char Iam[] = "GetColour";
#endif

 colour.red   = red   << 8;
 colour.green = green << 8;
 colour.blue  = blue  << 8;

 switch (depth)
   {
    case 16:
      return Pixel2Colour(&colour, pix);
      break;

    case 24:
      return Pixel4Colour(&colour, pix);
      break;

    default:
      if (XAllocColor(display, colourmap, &colour))
	{
	 *pix = colour.pixel;
	 return 1;
	}
      break;
   }
 /* End ``switch (depth)'' */

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (X_TRACE)
   Arena_TracePrint(Iam,
		    " can't allocate colour %d:%d:%d\n",
		    red, green, blue);
# else
 Arena_PrintError(_("Can't allocate colour %d:%d:%d\n"),
		  red, green, blue);
#endif
 *pix = BlackPixel(display, screen);
 return 0;
}


int GetPNGcolour(int red, int green, int blue, unsigned long *pix)
{
 XColor colour;
#ifdef ARENA_DEBUG
 char Iam[] = "GetPNGcolour";
#endif


 colour.red   = red   << 8;
 colour.green = green << 8;
 colour.blue  = blue  << 8;

 switch (depth)
   {
    case 16:
      return Pixel2PngColour(&colour, pix);
      break;

    case 24:
      return Pixel4PngColour(&colour, pix);
      break;

    default:
      if (XAllocColor(display, colourmap, &colour))
	{
	 *pix = colour.pixel;
	 return 1;
	}
      break;
   }
 /* End ``switch (depth)'' */

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (X_TRACE)
   Arena_TracePrint(Iam,
		    " can't allocate colour %d:%d:%d\n",
		    red, green, blue);
# else
 Arena_PrintError(_("Can't allocate colour %d:%d:%d\n"),
		  red, green, blue);
#endif
 *pix = BlackPixel(display, screen);
 return 0;
}


int InitImaging(int ColourStyle)
{
 imaging = MONO;

 build_gamma_table();
 build_v2b_table();

 greymap[0]  = BlackPixel(display, screen);
 greymap[15] = WhitePixel(display, screen);

 if (ColourStyle == MONO)
   {
    imaging = ColourStyle;
    return imaging;
   }

 if (ColourStyle == COLOUR888 && SupportTrueColour())
   {
    imaging = ColourStyle;
    return imaging;
   }

 if (OwnColourMap == INSTALL)
   if (AllocStandardColours())
     {
      imaging = COLOUR232;
      return imaging;
     }

 if (AllocGreyScale())
   {
    imaging = GREY4;

    if (ColourStyle == GREY4) return imaging;

    if (AllocStandardColours()) imaging = COLOUR232;
   }
  else
   if(OwnColourMap == INSTALL)
     if (AllocStandardColours())
       imaging = COLOUR232;

 return imaging;
}


void ReportVisuals(void)
{
 long visual_info_mask;
 int number_visuals, i;
 XVisualInfo *visual_array, visual_info_template;

 visual_info_template.screen = DefaultScreen(display);
 visual_info_mask = VisualClassMask | VisualScreenMask;

 printf("TrueColor:\n");

 visual_info_template.class = TrueColor;
 visual_array = XGetVisualInfo(display, visual_info_mask,
			       &visual_info_template,
			       &number_visuals);

 for (i = 0; i < number_visuals; ++i)
   {
    printf("  visual Id 0x%x\n", (unsigned int)(visual_array[i].visualid));
    printf("  depth = %d, bits per rgb = %d, size = %d\n",
	   visual_array[i].depth,
	   visual_array[i].bits_per_rgb,
	   visual_array[i].colormap_size);
    printf("   rgb masks %lx, %lx, %lx\n",
	   visual_array[i].red_mask,
	   visual_array[i].green_mask,
	   visual_array[i].blue_mask);
   }

 XFree((void *)visual_array);

 printf("DirectColor:\n");

 visual_info_template.class = DirectColor;
 visual_array = XGetVisualInfo(display, visual_info_mask,
			       &visual_info_template,
			       &number_visuals);

 for (i = 0; i < number_visuals; ++i)
   {
    printf("  visual Id 0x%x\n", (unsigned int)(visual_array[i].visualid));
    printf("  depth = %d, bits per rgb = %d, size = %d\n",
	   visual_array[i].depth,
	   visual_array[i].bits_per_rgb, visual_array[i].colormap_size);
    printf("   rgb masks %lx, %lx, %lx\n",
	   visual_array[i].red_mask,
	   visual_array[i].green_mask,
	   visual_array[i].blue_mask);
   }

 XFree((void *)visual_array);

 printf("PseudoColor:\n");

 visual_info_template.class = PseudoColor;
 visual_array = XGetVisualInfo(display, visual_info_mask,
			       &visual_info_template,
			       &number_visuals);

 for (i = 0; i < number_visuals; ++i)
   {
    printf("  visual Id 0x%x\n", (unsigned int)(visual_array[i].visualid));
    printf("  depth = %d, bits per rgb = %d, size = %d\n",
	   visual_array[i].depth,
	   visual_array[i].bits_per_rgb,
	   visual_array[i].colormap_size);
    printf("   rgb masks %lx, %lx, %lx\n",
	   visual_array[i].red_mask,
	   visual_array[i].green_mask,
	   visual_array[i].blue_mask);
   }

 XFree((void *)visual_array);
}


void ReportStandardColourMaps(Atom which_map)
{
 XStandardColormap *std_colourmaps;
 int i, number_colourmaps;
 char *atom_name;


 if (XGetRGBColormaps(display, RootWindow(display, screen),
		      &std_colourmaps, &number_colourmaps, which_map) != 0)
   {
    atom_name = XGetAtomName(display, which_map);
    printf("\nPrinting %d standard colourmaps for %s\n",
	   number_colourmaps, atom_name);
    XFree(atom_name);

    for  (i = 0; i < number_colourmaps; ++i)
      {
       printf("\tColourmap: 0x%x\n",
	      (unsigned int)(std_colourmaps[i].colormap));
       printf("\tMax cells (rgb): %ld, %ld, %ld\n",
	      std_colourmaps[i].red_max,
	      std_colourmaps[i].green_max,
	      std_colourmaps[i].blue_max);
       printf("\tMultipliers: %ld, %ld, %ld\n",
	      std_colourmaps[i].red_mult,
	      std_colourmaps[i].green_mult,
	      std_colourmaps[i].blue_mult);
       printf("\tBase pixel: %ld\n",
	      std_colourmaps[i].base_pixel);
       printf("\tVisual Id 0x%x, Kill Id 0x%x\n",
	      (unsigned int)(std_colourmaps[i].visualid),
	      (unsigned int)(std_colourmaps[i].killid));
      }

    XFree((void *)std_colourmaps);
   }
}
