;; ---------------------------------------------------------------------- ;;
;! @file     scmdoc.scm                                                   !;
;! @created  Fri Jun 13 10:22:30 1997                                     !;
;! @modified Thu Mar  5 15:29:37 1998                                     !;
;! @version  0.1                                                          !;
;; ---------------------------------------------------------------------- ;;
;! @copyright Dominique Boucher                                           !;
;; ---------------------------------------------------------------------- ;;
;; Documentation generator                                                ;;
;; ---------------------------------------------------------------------- ;;

;! @text                                                                  !;
;! <H2 ALIGN=CENTER>A Scheme documentation generator</H2>                 !;
;!                                                                        !;
;! <H3> Overview </H3>                                                    !;
;!                                                                        !;
;! <P> <CODE>scmdoc</CODE> is a documentation generation program for      !;
;! Scheme. It takes as input a standard Scheme file containing            !;
;! documentation declarations enclosed in comments, and generates HTML    !;
;! documentation based on these declarations. These comments are of the   !;
;! form: <PRE>                                                            !;
;!    ;! @directive [ident] text ... [!;]                                 !;
;! </PRE>                                                                 !;
;! Text that span more than a single line can continue on the next        !;
;! lines, given that these lines begin with <CODE>;!</CODE>.              !;
;!                                                                        !;
;! <H3>The directives</H3>                                                !;
;!                                                                        !;
;! <P> The valid directives that can appear at the very beginning of the file !;
;! are:                                                                   !;
;! <PRE>                                                                  !;
;!    ;! @file      name of the file                                      !;
;!    ;! @created   date of creation                                      !;
;!    ;! @modified  date of last modification                             !;
;!    ;! @version   current version number                                !;
;!    ;! @copyright the owner of the file                                 !;
;!    ;! @include   path of file to include                               !;
;! </PRE>                                                                 !;
;! The last directive, <CODE>@include</CODE>, is useful to include some   !;
;! HTML text right before the definitions produced by the program.        !;
;!                                                                        !;
;! <P> The directives that can appear just before a Scheme definition are: !;
;! <PRE>                                                                  !;
;!    ;! @description Description of the variable/function/macro/structure !;
;!    ;! @param arg   Description of the formal argument arg of a function/macro !;
;!    ;! @field fld   Description of the field fld of a structure         !;
;!    ;! @return      Description of the return value                     !;
;! </PRE>                                                                 !;
;!                                                                        !;
;! <P> The program can output documentation for definitions of the        !;
;! form:                                                                  !;
;! <PRE>                                                                  !;
;!   (define variable expression)                                         !;
;!   (define (function args ...) ...)                                     !;
;!   (define-macro (macro args ...) ...)                                  !;
;!   (define-structure name fields ...)                                   !;
;! </PRE>                                                                 !;
;!                                                                        !;
;! For example, the documentation for the function <A                     !;
;! HREF="#SCM->HTML"><CODE>SCM->HTML</CODE></A> has been generated by     !;
;! the following documentation comments:                                  !;
;! <PRE>                                                                  !;
;!   ;! @description                                                      !;
;!   ;!   This is the documentation generation function.                  !;
;!   ;! @param iport The input port (where to read the source program).   !;
;!   ;! @param oport The output port (where to put the documentation text). !;
;!   ;! @return No return value, i.e. returns &lt;CODE&gt;#f&lt;/CODE&gt;. !;
;!   (define (scm->html iport oport)                                      !;
;!    ...                                                                 !;
;! </PRE>                                                                 !;
;!                                                                        !;
;! <P> In the case of a function definition which looks like a variable   !;
;! definition (like <CODE>(define foo (let (..) (lambda (...)             !;
;! ...)))</CODE>, for example), one can explicitly inform                 !;
;! <CODE>scmdoc</CODE> that it IS a function definition by using the      !;
;! <CODE>@function</CODE> directive:                                      !;
;! <PRE>                                                                  !;
;!   ;! @function (foo arg1 arg2)                                         !;
;! </PRE>                                                                 !;
;! The prototype must be valid. The program can abruptly terminate if     !;
;! it's not the case.                                                     !;
;!                                                                        !;
;! <P> Finally, the special directive <CODE>@ignore</CODE> inhibits the   !;
;! generation of documentation for the definition that follows the        !;
;! directive. Note also that <CODE>scmdoc</CODE> does not generate documentation !;
;! for definition which are not preceded by documentation comments.       !;
;!                                                                        !;
;! <H3>Program invocation</H3>                                            !;
;!                                                                        !;
;! <P>To invoke <CODE>scmdoc</CODE>, type the following at the            !;
;! command-line:                                                          !;
;! <PRE>                                                                  !;
;!   % scmdoc -if foo.scm -of foo.html                                    !;
;! </PRE>                                                                 !;
;! where <CODE>foo.scm</CODE> is the Scheme input file and                !;
;! <CODE>foo.html</CODE> is the HTML output file. If the input file is    !;
;! not specified on the command-line, the input is taken from the         !;
;! standard input. Conversely, the HTML file will be sent to the standard !;
;! output if the output file is not specified.                            !;
;!                                                                        !;
;! <H3>Miscellaneous</H3>                                                 !;
;!                                                                        !;
;! All the Scheme symbols appearing in the generated HTML file can be     !;
;! referred to as HTML hyperlinks. I.e., one can add an HTML              !;
;! <CODE>HREF</CODE> tag in a documentation comment which refers to one   !;
;! of the defined variables, functions, etc. Just use its name as it      !;
;! appears in the HTML file. For example,                                 !;
;! <A HREF="#SCM->HTML"><CODE>SCM->HTML</CODE></A>                        !;
;! is the function which reads a Scheme file from a port and formats the  !;
;! documentation to another port.                                         !;
;!                                                                        !;
;! <P>                                                                    !;
;! <A HREF="mailto:boucherd@iro.umontreal.ca">Dominique Boucher</A>,      !;
;! June 20th, 1997.                                                       !;
;! @end                                                                   !;

(module scmdoc
	(main main)
	(import 
	 (globals    "globals.scm")
	 (read       "read.scm")
	 (format     "format.scm")
	 (output     "output.scm")
	 (utils      "utils.scm")
	 (predicates "preds.scm")))

;; ---------------------------------------------------------------------- ;;
;; The main procedure                                                     ;;
;; ---------------------------------------------------------------------- ;;
;! @description This is the main function.                                !;
;! @param args The command-line arguments (a list of strings).            !;
(define (main args)
  (parse-args! (cdr args))
  
  (if (and infile (not (file-exists? infile)))
      (signal-error "unable to open file: ~a" infile))

  ;; --- open the files ------------------------------------------------- ;;
  (let ((iport (if infile
		   (open-input-file infile)
		   (current-input-port)))
	(oport (if outfile
		   (open-output-file outfile)
		   (current-output-port))))

    ;; --- call the documentation generation procedure ------------------ ;;
    (scm->html iport oport)

    ;; --- close the files ---------------------------------------------- ;;
    (if infile (close-input-port iport))
    (if outfile (close-output-port oport))))


;; ---------------------------------------------------------------------- ;;
;; The argument parsing function                                          ;;
;; ---------------------------------------------------------------------- ;;
;! @description This is the command-line arguments parsing function.      !;
;! @param args The command-line arguments (minus the program name).       !;
(define (parse-args! args)
  (let loop ((args args))
    (if (null? args)
	#f
	(let ((arg  (car args))
	      (rest (cdr args)))
	  (cond

	   ;; --- input file specification ------------------------------ ;;
	   ((string=? arg "-if")
	    (if (null? rest)
		(signal-error "bad input file")
		(begin
		  (set! infile (car rest))
		  (loop (cdr rest)))))

	   ;; --- output file specification ----------------------------- ;;
	   ((string=? arg "-of")
	    (if (null? rest)
		(signal-error "bad output file")
		(begin
		  (set! outfile (car rest))
		  (loop (cdr rest)))))
	   
	   (else
	    (signal-error "unrecognized option: ~a" arg)))))))

;; ---------------------------------------------------------------------- ;;
;; error processing                                                       ;;
;; ---------------------------------------------------------------------- ;;
;! @section Error processing                                              !;

;! @description Writes a message to the terminal and exits.               !;
(define (signal-error msg . args)
  (format #t "Error: ")
  (apply format (append (list #t msg) args))
  (format #t "~%")
  (exit 1))

;; ---------------------------------------------------------------------- ;;
;! @section The conversion function                                       !;
;; ---------------------------------------------------------------------- ;;
;! @description This is the documentation generation function.            !;
;! @param iport The input port (where to read the source program).        !;
;! @param oport The output port (where to put the documentation text).    !;
;! @return No return value, i.e. returns <CODE>#f</CODE>.                 !;
(define (scm->html iport oport)
  (let loop ()
    (let ((e (scm-read iport)))
      (if (and (pair? e) (is-program-header? (car e)))
	  (begin
	    (cond
	     ((eq? (car e) 'text)
	      (let ((txt (read-text iport (cdr e))))
		(process-header-comment 'overview txt)))
	     ((eq? (car e) 'doc)
	      #f)
	     (else
	      (process-header-comment (car e) (cadr e))))
	    (loop))
	  
	  (begin
	    (output-file-header oport)
	    
	    (let loop ((e e) (docs '()))
	      (if (eof-object? e)
		
		  (output-file-footer oport)
		  
		  (cond
		   ((eq? (car e) 'expression)
		    (let ((docs (pack-doc-strings (reverse docs))))
			(if (null? (filter ignore-comment? docs))
			    (process-expression oport (cadr e) docs))
			(loop (scm-read iport) '())))
		   ((eq? (car e) 'text) ;***
		    (output-text oport (read-text iport (cdr e)))
		    (loop (scm-read iport) '()))
		   
		   ((eq? (car e) 'section)
		    (output-section-header oport (cadr e))
		    (loop (scm-read iport) '()))

		   (else
		    (loop (scm-read iport) (cons e docs)))))))))))
			 

;; ---------------------------------------------------------------------- ;;
;; Process a file header documentation comment                            ;;
;; ---------------------------------------------------------------------- ;;
(define (process-header-comment type val)
  (case type
    ((file)
     (if (not file) (set! file val)))
    ((created)
     (if (not created) (set! created val)))
    ((modified)
     (if (not modified) (set! modified val)))
    ((copyright)
     (if (not copyright) (set! copyright val)))
    ((version)
     (if (not version) (set! version val)))
    ((include)
     (if (not included) (set! included val)))
    ((overview)
     (if (not overview) (set! overview val)))))


;; ---------------------------------------------------------------------- ;;
;; Process a top-level expression with doc comments                       ;;
;; ---------------------------------------------------------------------- ;;
(define (process-expression oport expr docs)
  (if (pair? docs)
      (cond
       ((defvar-form? expr)
	(output-variable oport (cadr expr) docs))
       ((function-form? expr)
	(output-function oport (caadr expr) (cdadr expr) docs))
       ((macro-form? expr)
	(output-macro oport (caadr expr) (cdadr expr) docs))
       ((structure-form? expr)
	(output-structure oport (cadr expr) (cddr expr) docs)))))


  
