/* --------------------------------------------------------------------------
 * Defines storage datatypes: Text, Name, Module, Tycon, Cell, List, Pair,
 * Triple, ...
 *
 * Copyright (c) The University of Nottingham and Yale University, 1994-1997.
 * All rights reserved. See NOTICE for details and conditions of use etc...
 * Hugs version 1.4, December 1997
 *
 * $RCSfile: storage.h,v $
 * $Revision: 1.40 $
 * $Date: 1997/12/12 15:09:07 $
 * ------------------------------------------------------------------------*/

/* --------------------------------------------------------------------------
 * Typedefs for main data types:
 * Many of these type names are used to indicate the intended us of a data
 * item, rather than for type checking purposes.  Sadly (although sometimes,
 * fortunately), the C compiler cannot distinguish between the use of two
 * different names defined to be synonyms for the same types.
 * ------------------------------------------------------------------------*/

typedef Int	     Text;			 /* text string 	   */
typedef Unsigned     Syntax;			 /* syntax (assoc,preced)  */
typedef Int	     Addr;			 /* address of code	   */
typedef Int	     Cell;			 /* general cell value	   */
typedef Cell far     *Heap;			 /* storage of heap	   */
typedef Cell	     Pair;			 /* pair cell		   */
typedef Int	     StackPtr;			 /* stack pointer	   */
typedef Cell	     Offset;			 /* offset/generic variable*/
typedef Int	     Script;			 /* script file number	   */
typedef Int	     Module;			 /* module 	           */
typedef Cell	     Tycon;			 /* type constructor	   */
typedef Cell	     Type;			 /* type expression	   */
typedef Cell	     Kind;			 /* kind expression	   */
typedef Cell	     Constr;			 /* constructor expression */
typedef Cell	     Name;			 /* named value 	   */
typedef Void	     (*Prim) Args((StackPtr));	 /* primitive function	   */
typedef Cell	     Class;			 /* type class		   */
typedef Cell	     Inst;			 /* instance of type class */
typedef Cell	     Triple;			 /* triple of cell values  */
typedef Cell	     List;			 /* list of cells	   */
#if BIGNUMS
typedef Cell	     Bignum;			 /* bignum integer	   */
#endif
typedef FloatImpType Float;			 /* implementation of Float*/
#if TREX
typedef Cell	     Ext;			 /* extension label	   */
#endif

/* --------------------------------------------------------------------------
 * Text storage:
 * provides storage for the characters making up identifier and symbol
 * names, string literals, character constants etc...
 * ------------------------------------------------------------------------*/

extern	String	     textToStr		Args((Text));
extern	Text	     findText		Args((String));
extern	Text         inventText		Args((Void));
extern  Text	     inventDictText	Args((Void));
extern  Bool	     inventedText	Args((Text));

/* Variants of textToStr and syntaxOf which work for idents, ops whether
 * qualified or unqualified.
 */
extern	String	     identToStr		Args((Cell));
extern	Syntax	     identSyntax	Args((Cell));

/* --------------------------------------------------------------------------
 * Specification of syntax (i.e. default written form of application)
 * ------------------------------------------------------------------------*/

#define MIN_PREC  0		       /* weakest binding operator	   */
#define MAX_PREC  9		       /* strongest binding operator	   */
#define FUN_PREC  (MAX_PREC+2)	       /* binding of function symbols	   */
#define DEF_PREC  MAX_PREC
#define APPLIC	  00000 	       /* written applicatively 	   */
#define LEFT_ASS  02000 	       /* left associative infix	   */
#define RIGHT_ASS 04000 	       /* right associative infix	   */
#define NON_ASS   06000 	       /* non associative infix 	   */
#define DEF_ASS   NON_ASS

#define assocOf(x)	((x)&NON_ASS)
#define precOf(x)	((x)&(~NON_ASS))
#define mkSyntax(a,p)	((a)|(p))
#define DEF_OPSYNTAX	mkSyntax(DEF_ASS,DEF_PREC)

extern	Void   addSyntax  Args((Int,Text,Syntax));
extern	Syntax syntaxOf   Args((Text));

/* --------------------------------------------------------------------------
 * Primitive functions:
 * ------------------------------------------------------------------------*/

extern struct primitive {		/* table of primitives		   */
    String ref;				/* primitive reference string	   */
    Int	   arity;			/* primitive function arity	   */
    Prim   imp;				/* primitive implementation	   */
} primitives[];

/* Modules that define new primitive functions must register a control
 * function (defining INSTALL, RESET, etc code) and a (null-terminated)
 * table of primitive functions.
 *
 * They are stored as a linked list - so there's no wired in limits. 
 * Control functions are called in the order they are registered
 * after all other control functions have been called.
 * (At the moment) there's no way of unregistering a module.
 */

struct primInfo {
    Void              (*controlFun) Args((Int));
    struct primitive  *primFuns;
    struct primInfo   *nextPrimInfo;
};

extern  Void registerPrims Args((struct primInfo*)); 
extern  Void controlFuns   Args((Int)); /* Call all control functions in   */
                                        /* prim list.                      */

/* --------------------------------------------------------------------------
 * Program code storage: for holding compiled function defns etc...
 * ------------------------------------------------------------------------*/

extern	Addr	     getMem	Args((Int));
extern  Void	     nextInstr	Args((Addr));

/* --------------------------------------------------------------------------
 * Heap storage:
 * Provides a garbage collectable heap for storage of expressions etc.
 * ------------------------------------------------------------------------*/

#define heapAlloc(s) (Heap)(farCalloc(s,sizeof(Cell)))
#define heapBuilt()  (heapFst)
extern  Int          heapSize;
extern  Heap	     heapFst, heapSnd;
#ifdef  GLOBALfst
register Heap	     heapTopFst GLOBALfst;
#else
extern   Heap        heapTopFst;
#endif
#ifdef  GLOBALsnd
register Heap	     heapTopSnd GLOBALsnd;
#else
extern   Heap        heapTopSnd;
#endif
extern   Bool        consGC;		/* Set to FALSE to turn off gc from*/
					/* C stack; use with extreme care! */
#define fst(c)	     heapTopFst[c]
#define snd(c)	     heapTopSnd[c]
#if PROFILING
extern   Heap	     heapThd, heapTopThd;
#define thd(c)	     heapTopThd[c]
extern   Name	     producer;
extern   Bool        profiling;
extern   Int 	     profInterval;
extern   Void	     profilerLog     Args((String));
#endif

extern	Pair	     pair	     Args((Cell,Cell));
extern  Void         garbageCollect  Args((Void));

extern  Void	     overwrite	     Args((Pair,Pair));
extern  Cell	     markExpr	     Args((Cell));
extern  Void	     markWithoutMove Args((Cell));

#define mark(v)      v=markExpr(v)

#define isPair(c)    ((c)<0)
#define isGenPair(c) ((c)<0 && -heapSize<=(c))

extern	Cell	     whatIs    Args((Cell));

/* flatAlloc(n) generates an array of (n+1) Cells.
 *   flatLen(flatAlloc(n)) = n.
 *   For any flat array f, flatGet(f,0) = f.
 */
extern  Heap	     flatspace;
extern  Cell	     flatAlloc	     Args((Int,Int));
#define flatOf(f)    snd(f)
#define flatLen(f)   ((Int)(flatspace[flatOf(f)]))
#if CHECK_TAGS
extern Cell* flatGetChecked Args(( Cell, Int ));
#define flatGet(f,i) *flatGetChecked(f,i)
#else
#define flatGet(f,i) flatspace[flatOf(f)+(i)+1]
#endif
#define flatSet(f,i,e) { Cell _tmp = e; flatGet(f,i) = _tmp; }

/* --------------------------------------------------------------------------
 * Box cell tags are used as the fst element of a pair to indicate that
 * the snd element of the pair is to be treated in some special way, other
 * than as a Cell.  Examples include holding integer values, variable name
 * and string text etc.
 * ------------------------------------------------------------------------*/

#define TAGMIN	     1		  /* Box and constructor cell tag values   */
#define BCSTAG	     20 	  /* Box=TAGMIN..BCSTAG-1		   */
#define isTag(c)     (TAGMIN<=(c) && (c)<SPECMIN) /* Tag cell values	   */
#define isBoxTag(c)  (TAGMIN<=(c) && (c)<BCSTAG)  /* Box cell tag values   */
#define isConTag(c)  (BCSTAG<=(c) && (c)<SPECMIN) /* Constr cell tag values*/

#define INDIRECT     1		  /* Indirection node:	      snd :: Cell  */
#define INDIRECT1    2		  /* Temporary indirection:   snd :: Cell  */
#define FREECELL     3		  /* Free list cell:	      snd :: Cell  */
#define VARIDCELL    4		  /* Identifier variable:     snd :: Text  */
#define VAROPCELL    5		  /* Operator variable:       snd :: Text  */
#define DICTVAR	     6		  /* Dictionary variable:     snd :: Text  */
#define CONIDCELL    7		  /* Identifier constructor:  snd :: Text  */
#define CONOPCELL    8		  /* Operator constructor:    snd :: Text  */
#define STRCELL      9		  /* String literal:	      snd :: Text  */
#define INTCELL	     10		  /* Integer literal:	      snd :: Int   */
#if NPLUSK
#define ADDPAT	     11		  /* (_+k) pattern discr:     snd :: Int   */
#endif
#if IO_HANDLES
#define HANDCELL     12		  /* IO monad Handle:	      snd :: Int   */
#endif
#define DICTCELL     13		  /* Dictionary value:        snd :: Flat  */
#define FLATCELL     14		  /* Marked flat resource:    snd :: Tag   */
#if !BREAK_FLOATS
#define FLOATCELL    15		  /* Floating pt number:      snd :: Float */
#endif
#if TREX
#define EXTCOPY	     16		  /* Copy of an Ext:	      snd :: Text  */
#endif
#if GC_MALLOCPTRS
#define MPCELL       17		  /* Malloc Ptr Cell:         snd :: Int   */
#endif

#define textOf(c)	((Text)(snd(c)))         /* c ::  (VAR|CON)(ID|OP) */
#define qmodOf(c)       (textOf(fst(snd(c))))    /* c ::  QUALIDENT        */
#define qtextOf(c)      (textOf(snd(snd(c))))    /* c ::  QUALIDENT        */
#define mkVar(t)	ap(VARIDCELL,t)
#define mkVarop(t)	ap(VAROPCELL,t)
#define mkCon(t)	ap(CONIDCELL,t)
#define mkConop(t)	ap(CONOPCELL,t)
#define mkQVar(m,t)	ap(QUALIDENT,pair(mkCon(m),mkVar(t)))
#define mkQCon(m,t)	ap(QUALIDENT,pair(mkCon(m),mkCon(t)))
#define mkQVarOp(m,t)	ap(QUALIDENT,pair(mkCon(m),mkVarop(t)))
#define mkQConOp(m,t)	ap(QUALIDENT,pair(mkCon(m),mkConop(t)))
#define intValOf(c)	(snd(c))
#define inventVar()	mkVar(inventText())
#define mkDictVar(t)	ap(DICTVAR,t)
#define inventDictVar() mkDictVar(inventDictText())
#define mkStr(t)	ap(STRCELL,t)
extern	Bool		isVar       Args((Cell));
extern	Bool		isCon       Args((Cell));
extern	Bool		isQVar      Args((Cell));
extern	Bool		isQCon      Args((Cell));
extern	Bool		isQualIdent Args((Cell));
extern	Bool		isIdent     Args((Cell));

#define isFloat(c)      (isPair(c) && fst(c)==FLOATCELL)
extern	Cell	        mkFloat		Args((FloatPro));
extern  FloatPro	floatOf		Args((Cell));
extern  String		floatToString   Args((FloatPro));
extern  FloatPro	stringToFloat   Args((String));
#if BREAK_FLOATS
extern  Cell		part1Float	Args((FloatPro));
extern  Cell		part2Float	Args((FloatPro));
extern  FloatPro	floatFromParts	Args((Cell,Cell));
#endif

/* --------------------------------------------------------------------------
 * Constructor cell tags are used as the fst element of a pair to indicate
 * a particular syntactic construct described by the snd element of the
 * pair.
 * Note that a cell c will not be treated as an application (AP/isAp) node
 * if its first element is a constructor cell tag, whereas a cell whose fst
 * element is a special cell will be treated as an application node.
 * ------------------------------------------------------------------------*/

#define LETREC	     20 	  /* LETREC	snd :: ([Decl],Exp)	   */
#define COND	     21 	  /* COND	snd :: (Exp,Exp,Exp)	   */
#define LAMBDA	     22 	  /* LAMBDA	snd :: Alt		   */
#define FINLIST      23 	  /* FINLIST	snd :: [Exp]		   */
#define DOCOMP       24 	  /* DOCOMP	snd :: (Exp,[Qual])	   */
#define BANG	     25		  /* BANG	snd :: Type		   */
#define COMP         26 	  /* COMP	snd :: (Exp,[Qual])	   */
#define ASPAT	     27 	  /* ASPAT	snd :: (Var,Exp)	   */
#define ESIGN	     28 	  /* ESIGN	snd :: (Exp,Type)	   */
#define CASE	     29 	  /* CASE	snd :: (Exp,[Alt])	   */
#define NUMCASE	     30		  /* NUMCASE	snd :: (Exp,Disc,Rhs)	   */
#define FATBAR	     31 	  /* FATBAR	snd :: (Exp,Exp)	   */
#define LAZYPAT      32 	  /* LAZYPAT	snd :: Exp		   */
#define QUAL	     33 	  /* QUAL       snd :: ([Classes],Type)    */
#define RUNST	     34		  /* RUNST	snd :: Exp		   */
#define DERIVE	     35		  /* DERIVE	snd :: Cell		   */
#if BREAK_FLOATS
#define FLOATCELL    36		  /* FLOATCELL  snd :: (Int,Int)	   */
#endif

#if BIGNUMS
#define POSNUM	     37		  /* POSNUM	snd :: [Int]		   */
#define NEGNUM	     38		  /* NEGNUM	snd :: [Int]		   */
#endif

#define BOOLQUAL     39 	  /* BOOLQUAL	snd :: Exp		   */
#define QWHERE	     40 	  /* QWHERE	snd :: [Decl]		   */
#define FROMQUAL     41 	  /* FROMQUAL	snd :: (Exp,Exp)	   */
#define DOQUAL	     42		  /* DOQUAL	snd :: Exp		   */
#define MONADCOMP    43 	  /* MONADCOMP	snd :: ((m,m0),(Exp,[Qual])*/

#define GUARDED      44 	  /* GUARDED	snd :: [guarded exprs]	   */

#define ARRAY        45		  /* Array:     snd :: (Bounds,[Values])   */
#define MUTVAR	     46		  /* Mutvar:	snd :: Cell		   */
#if INTERNAL_PRIMS
#define HUGSOBJECT   47		  /* HUGSOBJECT	snd :: Cell	           */
#endif

#define POLYTYPE     50		  /* POLYTYPE	snd :: (Kind,Type)	   */
#define POLYREC	     54		  /* POLYREC:	snd :: (Int,Type)	   */

#define LABC	     60		  /* LABC:	snd :: (con,[(Vars,Type)]) */
#define CONFLDS	     61		  /* CONFLDS:   snd :: (con,[Field])	   */
#define UPDFLDS	     62		  /* UPDFLDS:   snd :: (Exp,[con],[Field]) */
#if TREX
#define RECORD	     63		  /* RECORD:    snd :: [Val]		   */
#define EXTCASE	     64		  /* EXTCASE:	snd :: (Exp,Disc,Rhs)	   */
#define RECSEL	     65		  /* RECSEL:	snd :: Ext		   */
#define EVSUCC	     66		  /* EVSUCC:	snd :: Evidence		   */
#endif

#define QUALIDENT    70           /* Qualified identifier  snd :: (Id,Id)  */
#define HIDDEN       71           /* hiding import list    snd :: [Entity] */
#define MODULEENT    72           /* module in export list snd :: con      */

#define ONLY	     80		  /* ONLY:      snd :: Exp (used in parser)*/
#define NEG	     81		  /* NEG:       snd :: Exp (used in parser)*/

#if GENERATE_STG
#define STGVAR       90           /* STGVAR     snd :: (StgRhs,info)       */
#define STGAPP       91           /* STGAPP     snd :: (StgVar,[Arg])      */
#define STGPRIM      92           /* STGPRIM    snd :: (PrimOp,[Arg])      */
#define STGCON       93           /* STGCON     snd :: (StgCon,[Arg])      */
#define PRIMCASE     94           /* PRIMCASE   snd :: (Expr,([Var],Expr)) */
#define LETALT       95           /* LETALT     snd :: (AltVar,Expr)       */
#define STGALTVAR    96           /* STGALTVAR  snd :: (Expr,info)         */
#endif

/* Last constructor tag must be less than SPECMIN */

/* --------------------------------------------------------------------------
 * Special cell values:
 * ------------------------------------------------------------------------*/

#define SPECMIN      101
#define isSpec(c)    (SPECMIN<=(c) && (c)<TUPMIN)/* Special cell values    */

#define NONE	     101	  /* Dummy stub				   */
#define STAR	     102	  /* Representing the kind of types	   */
#if TREX
#define ROW	     103	  /* Representing the kind of rows	   */
#endif
#define WILDCARD     104	  /* Wildcard pattern			   */

#define DOTDOT       105          /* ".." in import/export list            */

#if BIGNUMS
#define ZERONUM      108	  /* The zero bignum (see POSNUM, NEGNUM)  */
#endif

#define NAME	     110	  /* whatIs code for isName		   */
#define TYCON	     111	  /* whatIs code for isTycon		   */
#define CLASS	     112	  /* whatIs code for isClass		   */
#define SELECT       113          /* whatIs code for isSelect              */
#define INSTANCE     114          /* whatIs code for isInst                */
#define TUPLE	     115	  /* whatIs code for tuple constructor	   */
#define OFFSET	     116	  /* whatis code for offset		   */
#define AP	     117	  /* whatIs code for application node	   */
#define CHARCELL     118	  /* whatIs code for isChar		   */
#if TREX
#define EXT	     119	  /* whatIs code for isExt		   */
#endif

#define SIGDECL      120	  /* Signature declaration		   */
#define PREDEFINED   121	  /* predefined name, not yet filled       */

#define DATATYPE     130	  /* datatype type constructor		   */
#define NEWTYPE	     131	  /* newtype type constructor		   */
#define SYNONYM	     132	  /* synonym type constructor		   */
#define RESTRICTSYN  133	  /* synonym with restricted scope	   */

#define MODULE       134          /* whatIs code for isModule              */

#define NODEPENDS    135	  /* stop calculation of deps in type check*/

/* --------------------------------------------------------------------------
 * Tuple data/type constructors:
 * ------------------------------------------------------------------------*/

#define TUPMIN	     201
#if TREX
#define isTuple(c)   (TUPMIN<=(c) && (c)<EXTMIN)
#else
#define isTuple(c)   (TUPMIN<=(c) && (c)<OFFMIN)
#endif
#define mkTuple(n)   (TUPMIN+(n))
#define tupleOf(n)   ((Int)((n)-TUPMIN))

#if TREX
#define EXTMIN	     (TUPMIN+NUM_TUPLES)
#define isExt(c)     (EXTMIN<=(c) && (c)<OFFMIN)
#define extText(e)   tabExt[(e)-EXTMIN]
#define extField(c)  arg(fun(c))
#define extRow(c)    arg(c)

extern Text	     DECTABLE(tabExt);
extern Ext           mkExt Args((Text));
#else
#define mkExt(t) NIL
#endif

/* --------------------------------------------------------------------------
 * Offsets: (generic types/stack offsets)
 * ------------------------------------------------------------------------*/

#if TREX
#define OFFMIN	     (EXTMIN+NUM_EXT)
#else
#define OFFMIN	     (TUPMIN+NUM_TUPLES)
#endif
#define isOffset(c)  (OFFMIN<=(c) && (c)<MODMIN)
#define offsetOf(c)  ((c)-OFFMIN)
#define mkOffset(o)  (OFFMIN+(o))

/* --------------------------------------------------------------------------
 * Modules:
 * ------------------------------------------------------------------------*/

#define MODMIN	      (OFFMIN+NUM_OFFSETS)

#if IGNORE_MODULES
#define setCurrModule(m) doNothing()
#else /* !IGNORE_MODULES */
#define isModule(c)   (MODMIN<=(c) && (c)<TYCMIN)
#define mkModule(n)   (MODMIN+(n))
#define module(n)     tabModule[(n)-MODMIN]

/* Under Haskell 1.3, the list of qualified imports is always a subset
 * of the list of unqualified imports.  For simplicity and flexibility,
 * we do not attempt to exploit this fact - when a module is imported
 * unqualified, it is added to both the qualified and unqualified
 * import lists.
 * Similarily, Haskell 1.3 does not allow a constructor to be imported
 * or exported without exporting the type it belongs to but the export
 * list is just a flat list of Texts (before static analysis) or
 * Tycons, Names and Classes (after static analysis).
 */
struct Module {
    Text  text;
    /* Lists of top level objects (local defns + imports)                  */
    List  tycons;
    List  names;
    List  classes;
    List  exports; /* [ Entity | (Entity, NIL|DOTDOT) ] */
    /* List of qualified imports.  Used both during compilation and when
     * evaluating an expression in the context of the current module.
     */
    List  qualImports;
};

extern Module currentModule;           /* Module currently being processed */
extern struct Module DECTABLE(tabModule);

extern Bool   isValidModule Args((Module));
extern Module newModule	    Args((Text));
extern Module findModule    Args((Text));
extern Module findModid     Args((Cell));
extern Void   setCurrModule Args((Module));

#define isPrelude(m) (m==modulePrelude)
#endif /* !IGNORE_MODULES */

/* --------------------------------------------------------------------------
 * Type constructor names:
 * ------------------------------------------------------------------------*/

#define TYCMIN	     (MODMIN+NUM_MODULE)
#define isTycon(c)   (TYCMIN<=(c) && (c)<NAMEMIN)
#define mkTycon(n)   (TCMIN+(n))
#define tycon(n)     tabTycon[(n)-TYCMIN]

struct strTycon {
    Text  text;
    Int   line;
#if !IGNORE_MODULES
    Module mod;                         /* module that defines it          */
#endif
    Int   arity;
    Kind  kind;				/* kind (includes arity) of Tycon  */
    Cell  what;				/* DATATYPE/SYNONYM/RESTRICTSYN... */
    Cell  defn;
    Tycon nextTyconHash;
};

extern struct strTycon DECTABLE(tabTycon);

extern Tycon newTycon	  Args((Text));
extern Tycon findTycon	  Args((Text));
extern Tycon addTycon	  Args((Tycon));
extern Tycon findQualTycon Args((Cell));
extern Tycon addPrimTycon Args((String,Kind,Int,Cell,Cell));

#define isSynonym(h)	(isTycon(h) && tycon(h).what==SYNONYM)
#define mkPolyType(n,t)	pair(POLYTYPE,pair(n,t))
#define isPolyType(t)	(isPair(t) && fst(t)==POLYTYPE)
#define polySigOf(t)	fst(snd(t))
#define monoTypeOf(t)	snd(snd(t))

/* --------------------------------------------------------------------------
 * Globally defined name values:
 * ------------------------------------------------------------------------*/

#define NAMEMIN      (TYCMIN+NUM_TYCON)
#define isName(c)    (NAMEMIN<=(c) && (c)<SELMIN)
#define mkName(n)    (NAMEMIN+(n))
#define name(n)      tabName[(n)-NAMEMIN]

struct strName {
    Text text;
    Int  line;
#if !IGNORE_MODULES
    Module mod;                         /* module that defines it          */
#endif
    Cell parent;                        /* enclosing object                */
                                        /*   :: (Class|Instance,Name)      */
                                        /*      | Name                     */
                                        /*      | NIL                      */
    Int  arity;
    Int  number;
    Cell type;
    Cell defn;
#if GENERATE_STG 
    Cell stgVar;
#endif
    Addr code;
    Prim primDef;
    Name nextNameHash;
#if PROFILING
    Int  count;
#endif
};

extern struct strName DECTABLE(tabName);

/* The number field in a name is used to distinguish various kinds of name:
 *   mfunNo(i) = code for member function, offset i
 *   EXECNAME  = code for executable name (bytecodes or primitive)
 *   SELNAME   = code for selector function
 *   cfunNo(i) = code for data constructor
 *		 datatypes with only one constructor uses cfunNo(0)
 *		 datatypes with multiple constructors use cfunNo(n), n>=1
 */

#define EXECNAME	0
#define SELNAME		1
#define MIN_CFUN        2

#define isSfun(n)	(name(n).number==SELNAME)

#define isCfun(n)	(name(n).number>=MIN_CFUN)
#define cfunOf(n)	(name(n).number-MIN_CFUN)
#define cfunNo(i)	((i)+MIN_CFUN)
#define hasCfun(cs)	(nonNull(cs) && isCfun(hd(cs)))

#define isMfun(n)	(name(n).number<0)
#define mfunOf(n)	(-name(n).number)
#define mfunNo(i)	(-(i))

extern Name   newName	   Args((Text,Cell));
extern Name   findName	   Args((Text));
extern Name   addName	   Args((Name));
extern Name   findQualName Args((Int,Cell));
extern Void   addPrim	   Args((Int,Name,String,Type));
extern Name   addPrimCfun  Args((Text,Int,Int,Cell));
extern Int    sfunPos	   Args((Name,Name));

/* --------------------------------------------------------------------------
 * Type class values:
 * ------------------------------------------------------------------------*/

#define SELMIN       (NAMEMIN+NUM_NAME)          /* dictionary selectors   */
#define isSelect(c)  (SELMIN<=(c) && (c)<INSTMIN)
#define mkSelect(n)  (SELMIN+(n))
#define selectOf(c)  ((Int)((c)-SELMIN))

#define INSTMIN      (SELMIN+NUM_SELECTS)        /* instances              */
#define isInst(c)    (INSTMIN<=(c) && (c)<CLASSMIN)
#define mkInst(n)    (INSTMIN+(n))
#define instOf(c)    ((Int)((c)-INSTMIN))
#define inst(in)     tabInst[(in)-INSTMIN]

struct strInst {
    Class c;				/* class C			   */
    Cell  t;				/* type  T			   */
    Int   arity;			/* number of args		   */
    Int   line;
    List  specifics;			/* :: [Pred]			   */
    Int   numSpecifics;			/* length(specifics)		   */
    List  implements;
    List  dicts;			/* :: [Dict]			   */
    List  superBuild;			/* instructions for superclasses   */
};

/* a predicate (an element :: Pred) is an application of a Class to one or
 * more type expressions
 */

#define CLASSMIN     (INSTMIN+NUM_INSTS)
#define isClass(c)   (CLASSMIN<=(c) && (c)<CHARMIN)
#define mkClass(n)   (CLASSMIN+(n))
#define cclass(n)    tabClass[(n)-CLASSMIN]

struct strClass {
    Text text;				/* Name of class		   */
    Int  line;				/* Line where declaration begins   */
#if !IGNORE_MODULES
    Module mod;                         /* module that declares it         */
#endif
    Int  level;				/* Level in class hierarchy	   */
    Kind sig;				/* Kind of constructors in class   */
    List supers;			/* :: [Class] (immed superclasses) */
    Int  numSupers;			/* length(supers)		   */
    List members;			/* :: [Name]			   */
    Int  numMembers;			/* length(members)		   */
    List defaults;			/* :: [Name]			   */
    List instances;			/* :: [Inst]			   */
};

extern struct strClass    DECTABLE(tabClass);
extern struct strInst far *tabInst;

#if CHECK_TAGS
#define checkDict(c) \
  if (fst(c) != DICTCELL) internal("checkDict1"); \
  if (flatGet(c,0) != c)  internal("checkDict2"); 
extern Int  dictOf  Args((Cell));
extern Cell dictGet Args((Cell,Int));
extern Void dictSet Args((Cell,Int,Cell));
#else
#define dictOf(c)		flatOf(c)
#define dictGet(d,i)		flatGet(d,i)
#define dictSet(d,i,e)		flatSet(d,i,e)
#endif /* CHECK_TAGS */
extern Cell newDict             Args((Int));

#define dictMembersStart(cl)	(1)
#define dictSupersStart(cl)	(dictMembersStart(cl) + cclass(cl).numMembers)
#define dictSpecificsStart(cl)	(dictSupersStart(cl) + cclass(cl).numSupers)
#define dictLength(in)		(dictSpecificsStart(inst(in).c) + \
				 inst(in).numSpecifics)

#if GENERATE_STG
extern Inst  instHw;            /* next unused instance record      */
#endif

extern Class newClass	   Args((Text));
extern Class classMax	   Args((Void));
extern Class findClass	   Args((Text));
extern Class addClass	   Args((Class));
extern Class findQualClass Args((Cell));
extern Inst  newInst	   Args((Void));
extern Inst  findInst	   Args((Class,Type));
extern Inst  findFirstInst Args((Tycon));
extern Inst  findNextInst  Args((Tycon,Inst));
extern Cell  makeInstPred  Args((Inst));

/* --------------------------------------------------------------------------
 * Character values:
 * ------------------------------------------------------------------------*/

#define CHARMIN      (CLASSMIN+NUM_CLASSES)
#define MAXCHARVAL   (NUM_CHARS-1)
#define isChar(c)    (CHARMIN<=(c) && (c)<INTMIN)
#define charOf(c)    ((Char)(c-CHARMIN))
#define mkChar(c)    ((Cell)(CHARMIN+((unsigned)((c)%NUM_CHARS))))

/* --------------------------------------------------------------------------
 * Small Integer values:
 * ------------------------------------------------------------------------*/

#define INTMIN	     (CHARMIN+NUM_CHARS)
#define INTMAX	     MAXPOSINT
#define isSmall(c)   (INTMIN<=(c))
#define INTZERO      (INTMIN/2 + INTMAX/2)
#define mkDigit(c)   ((Cell)((c)+INTMIN))
#define digitOf(c)   ((Int)((c)-INTMIN))

extern	Bool isInt    Args((Cell));
extern	Int  intOf    Args((Cell));
extern	Cell mkInt    Args((Int));
#if BIGNUMS
extern  Bool isBignum Args((Cell));
#endif

/* --------------------------------------------------------------------------
 * Implementation of triples:
 * ------------------------------------------------------------------------*/

#define triple(x,y,z) pair(x,pair(y,z))
#define fst3(c)      fst(c)
#define snd3(c)      fst(snd(c))
#define thd3(c)      snd(snd(c))

/* --------------------------------------------------------------------------
 * Implementation of lists:
 * ------------------------------------------------------------------------*/

#define NIL	     0
#define isNull(c)    ((c)==NIL)
#define nonNull(c)   (c)
#define cons(x,xs)   pair(x,xs)
#define singleton(x) cons(x,NIL)
#define doubleton(x,y) cons(x,cons(y,NIL))
#define hd(c)	     fst(c)
#define tl(c)	     snd(c)

extern	Int	     length	  Args((List));
extern	List	     appendOnto   Args((List,List));
extern	List	     dupOnto      Args((List,List));
extern  List	     dupList	  Args((List));
extern	List	     revOnto	  Args((List, List));
#define rev(xs)      revOnto((xs),NIL)
#if 0
extern  List	     minus	  Args((List,List));
#endif
extern	Cell	     cellIsMember Args((Cell,List));
extern  Cell         cellAssoc    Args((Cell,List));
extern  Cell         cellRevAssoc Args((Cell,List));
extern	Cell	     varIsMember  Args((Text,List));
extern  Cell	     intIsMember  Args((Int,List));
extern	List	     copy	  Args((Int,Cell));
extern	List	     diffList	  Args((List,List));
extern  List         deleteCell   Args((List,Cell));
extern  List	     take	  Args((Int,List));
extern  List	     splitAt	  Args((Int,List));
#if 0 /* Not used in this version of Hugs */
extern  Bool         eqList       Args((List,List));
#endif
extern  Cell	     nth	  Args((Int,List));
extern  List	     initSeg	  Args((List));
extern  List	     removeCell	  Args((Cell,List));

/* The following macros provide `inline expansion' of some common ways of
 * traversing, using and modifying lists:
 *
 * N.B. We use the names _f, _a, _xs, Zs, in an attempt to avoid clashes
 *	with identifiers used elsewhere.
 */

#define mapBasic(_init,_step)	  {List Zs=(_init);\
				   for(;nonNull(Zs);Zs=tl(Zs))  \
				   _step;}
#define mapModify(_init,_step)	  mapBasic(_init,hd(Zs)=_step)

#define mapProc(_f,_xs) 	  mapBasic(_xs,_f(hd(Zs)))
#define map1Proc(_f,_a,_xs)	  mapBasic(_xs,_f(_a,hd(Zs)))
#define map2Proc(_f,_a,_b,_xs)	  mapBasic(_xs,_f(_a,_b,hd(Zs)))
#define map3Proc(_f,_a,_b,_c,_xs) mapBasic(_xs,_f(_a,_b,_c,hd(Zs)))

#define mapOver(_f,_xs) 	  mapModify(_xs,_f(hd(Zs)))
#define map1Over(_f,_a,_xs)	  mapModify(_xs,_f(_a,hd(Zs)))
#define map2Over(_f,_a,_b,_xs)	  mapModify(_xs,_f(_a,_b,hd(Zs)))
#define map3Over(_f,_a,_b,_c,_xs) mapModify(_xs,_f(_a,_b,_c,hd(Zs)))

/* This is just what you want for functions with accumulating parameters */
#define mapAccum(_f,_acc,_xs) 	        mapBasic(_xs,_acc=_f(_acc,hd(Zs)))
#define map1Accum(_f,_acc,_a,_xs)	mapBasic(_xs,_acc=_f(_acc,_a,hd(Zs)))
#define map2Accum(_f,_acc,_a,_b,_xs)	mapBasic(_xs,_acc=_f(_acc,_a,_b,hd(Zs)))
#define map3Accum(_f,_acc,_a,_b,_c,_xs) mapBasic(_xs,_acc=_f(_acc,_a,_b,_c,hd(Zs)))

/* --------------------------------------------------------------------------
 * Implementation of function application nodes:
 * ------------------------------------------------------------------------*/

#define ap(f,x)      pair(f,x)
#define fun(c)	     fst(c)
#define arg(c)	     snd(c)
#define isAp(c)      (isPair(c) && !isTag(fst(c)))
extern	Cell	     getHead	 Args((Cell));
extern	List	     getArgs	 Args((Cell));
extern	Int	     argCount;
extern  Cell         nthArg	 Args((Int,Cell));
extern  Int	     numArgs	 Args((Cell));
extern  Cell	     applyToArgs Args((Cell,List));

/* --------------------------------------------------------------------------
 * Stack implementation:
 *
 * NB: Use of macros makes order of evaluation hard to predict.
 *     For example, "push(1+pop());" doesn't increment TOS.
 * ------------------------------------------------------------------------*/

extern  Cell DECTABLE(cellStack);
#ifdef  GLOBALsp
register StackPtr    sp GLOBALsp;
#else
extern	StackPtr sp;
#endif

#define clearStack() sp=(-1)
#define stackEmpty() (sp==(-1))
#define stack(p)     cellStack[p]
#define chkStack(n)  if (sp>=NUM_STACK-(n)) stackOverflow()
#define push(c)      \
  do {               \
    chkStack(1);     \
    onto(c);         \
  } while (0)
#define onto(c)	     stack(++sp)=(c)
#define pop()	     stack(sp--)
#define drop()	     sp--
#define top()	     stack(sp)
#define pushed(n)    stack(sp-(n))
#define topfun(f)    top()=ap((f),top())
#define toparg(x)    top()=ap(top(),(x))

extern  Void stackOverflow Args((Void));

/* --------------------------------------------------------------------------
 * Script file control:
 * The implementation of script file storage is hidden.
 * ------------------------------------------------------------------------*/

extern Script	   startNewScript   Args((String));
extern Bool        moduleThisScript Args((Module));
extern Module      moduleOfScript   Args((Script));
extern Bool        isPreludeScript  Args((Void));
extern Module      lastModule       Args((Void));
extern Script	   scriptThisFile   Args((Text));
extern Script	   scriptThisName   Args((Name));
extern Script	   scriptThisTycon  Args((Tycon));
extern Script	   scriptThisInst   Args((Inst));
extern Script	   scriptThisClass  Args((Class));
extern String      fileOfModule     Args((Module));
extern Void	   dropScriptsFrom  Args((Script));

/* --------------------------------------------------------------------------
 * I/O Handles:
 * ------------------------------------------------------------------------*/

#if IO_HANDLES
#define HSTDIN		0	/* Numbers for standard handles		   */
#define HSTDOUT		1
#define HSTDERR		2

struct strHandle {		/* Handle description and status flags	   */
    Cell hcell;			/* Heap representation of handle (or NIL)  */
    FILE *hfp;			/* Corresponding file pointer		   */
    Int  hmode;			/* Current mode: see below		   */
};

#define HCLOSED		0000	/* no I/O permitted			   */
#define HSEMICLOSED     0001	/* semiclosed reads only		   */
#define HREAD		0002	/* set to enable reads from handle 	   */
#define HWRITE		0004	/* set to enable writes to handle	   */
#define HAPPEND		0010	/* opened in append mode		   */

extern Cell   openHandle Args((String,Int,Bool));
extern struct strHandle  DECTABLE(handles);
#endif

#if GC_MALLOCPTRS
/* --------------------------------------------------------------------------
 * Malloc Pointers
 * ------------------------------------------------------------------------*/

struct strMallocPtr {		/* Malloc Ptr description                  */
    Cell mpcell;		/* Back pointer to MPCELL                  */
    Void *ptr;			/* Pointer into C world                    */
    Int  refCount;              /* Reference count                         */
    Void (*cleanup) Args((Void *)); /* Code to free the C pointer          */
};

extern struct strMallocPtr       mallocPtrs[];
extern Cell   mkMallocPtr        Args((Void *, Void (*)(Void *)));
extern Void   freeMallocPtr      Args((Cell));
extern Void   incMallocPtrRefCnt Args((Int, Int));

#define mpOf(c)	   snd(c)
#define derefMP(c) (mallocPtrs[(Int)mpOf(c)].ptr)
#define sizeofMallocPtr 2
#endif /* GC_MALLOCPTRS */

/* --------------------------------------------------------------------------
 * Stable pointers
 * ------------------------------------------------------------------------*/

#if GC_STABLEPTRS
extern  Int  mkStablePtr     Args((Cell));
extern  Cell derefStablePtr  Args((Int));
extern  Void freeStablePtr   Args((Int));
#endif

/* --------------------------------------------------------------------------
 * Plugins
 * ------------------------------------------------------------------------*/

/* This is an exact copy of the declaration found in GreenCard.h */

typedef int   HugsStackPtr;
typedef int   HugsStablePtr;
typedef void* HugsForeign;

typedef struct {

  /* evaluate next argument */
  int            (*getInt   )     Args(());  
  unsigned int   (*getWord  )     Args(());
  void*     	 (*getAddr  )     Args(());
  float     	 (*getFloat )     Args(());
  double    	 (*getDouble)     Args(());
  char      	 (*getChar  )     Args(());
  HugsForeign    (*getForeign)    Args(());
  HugsStablePtr  (*getStablePtr)  Args(());

  /* push part of result   */
  void      	 (*putInt   )     Args((int));           
  void      	 (*putWord  )     Args((unsigned int));
  void      	 (*putAddr  )     Args((void *));
  void      	 (*putFloat )     Args((double));
  void      	 (*putDouble)     Args((double));
  void      	 (*putChar  )     Args((char));
  void      	 (*putForeign)    Args((HugsForeign, void (*)(HugsForeign)));
  void      	 (*putStablePtr)  Args((HugsStablePtr));

  /* return n values in IO monad or Id monad */
  void      	 (*returnIO)      Args((HugsStackPtr, int));
  void      	 (*returnId)      Args((HugsStackPtr, int));
  int      	 (*runIO)         Args((int));

  /* free a stable pointer */	    			 
  void      	 (*freeStablePtr) Args((HugsStablePtr));

  /* register the prim table */	    			 
  void      	 (*registerPrims) Args((struct primInfo*));
			   
} HugsAPI2;

extern  HugsAPI2* hugsAPI2     Args((Void));
typedef Void (*InitModuleFun2) Args((HugsAPI2*));

typedef struct {
  Name  nameTrue, nameFalse;
  Name  nameNil,  nameCons;
  Name  nameJust, nameNothing;
  Name  nameLeft, nameRight;
  Name  nameUnit;
  Name  nameIORun;

  Cell  (*makeInt)         Args((Int));
			   
  Cell  (*makeChar)        Args((Char));
  Char  (*CharOf)          Args((Cell));
			   
  Cell  (*makeFloat)       Args((FloatPro));
  Cell  (*makeTuple)       Args((Int));
  Pair  (*pair)            Args((Cell,Cell));
			   
  Cell  (*mkMallocPtr)     Args((Void *, Void (*)(Void *)));
  Void *(*derefMallocPtr)  Args((Cell));
			   
  Int   (*mkStablePtr)     Args((Cell));
  Cell  (*derefStablePtr)  Args((Int));
  Void  (*freeStablePtr)   Args((Int));
			   
  Void  (*eval)            Args((Cell));
  Cell  (*evalWithNoError) Args((Cell));
  Void  (*evalFails)       Args((StackPtr));
  Int   *whnfArgs;	   
  Cell  *whnfHead;	   
  Int   *whnfInt;	   
  Float *whnfFloat;	   
			   
  Void  (*garbageCollect)  Args(());
  Void  (*stackOverflow)   Args(());
  Void  (*internal)        Args((String)) HUGS_noreturn;

  Void  (*registerPrims)   Args((struct primInfo*));
  Name  (*addPrimCfun)     Args((Text,Int,Int,Cell));
  Text  (*inventText)      Args(());

  Cell *(*Fst)             Args((Cell));
  Cell *(*Snd)             Args((Cell));

  Cell  *cellStack;
  StackPtr *sp;
} HugsAPI1;

extern  HugsAPI1* hugsAPI1     Args((Void));
typedef Void (*InitModuleFun1) Args((HugsAPI1*));

/*-------------------------------------------------------------------------*/
