/********************************************************************************
 * Copyright (c) Erik Kunze 1997 - 1998
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Authors: Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#ifndef lint
static char rcsid[] = "$Id: tzx.c,v 4.14 1998/01/31 18:56:41 erik Rel $";
#endif

#include <X11/keysym.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "config.h"
#include "z80.h"
#include "debug.h"
#include "resource.h"
#include "mem.h"
#include "io.h"
#include "util.h"
#ifdef AUDIO
#include "audio.h"
#endif
#include "dialog.h"
#include "emul.h"
#include "loadsave.h"
#ifdef REGISTERED
#include "vtp.h"
#endif
#include "tzx.h"

#define MAJREV			1
#define MINREV			11
#define MS				3500
#define INVALID			65535
#define PAGE_SIZE		17
enum { BS_PILOT, BS_SYNC1, BS_SYNC2, BS_BIT_1, BS_BIT_2, BS_PAUSE };
struct tzxHeader {
	char ident[7];
	char marker;
	unsigned char major;
	unsigned char minor;
};
typedef struct _tzxRecord {
	uns8 *pos;
	char *str;
	uns8 type;
} tzxRecord;
#define LOW_PULSE		{ lastValue = 0; }
#define HIGH_PULSE		{ lastValue = B_EAR; }
#define TOGGLE_PULSE	{ lastValue ^= B_EAR; }
#define SET_IN_LOOP
#define RESET_IN_LOOP	{ loopStart = INVALID; }
#define IN_LOOP			( loopStart != INVALID )
#define SET_IN_CALL
#define RESET_IN_CALL	{ callStart = INVALID; }
#define IN_CALL			( callStart != INVALID )
#ifdef DEBUG
#define DEB(x)			{ if (GETCFG(debug) & D_TAPE) { x } }
#else
#define DEB(x)
#endif

static int tzxGetNextBlock(void);
static unsigned int tzxSelection(unsigned int, uns8 *);
static void tzxScanFile(void);


static uns16 blockType;
static int blockState;
static unsigned long lastTstate;
static uns16 loopStart, loopRepeat;
static uns16 callStart, callCount, callCurrent;
static uns16 pilotLen, pilot, sync1, sync2, bit0, bit1, curBit;
static uns32 blockLen;
static uns16 pauseLen;
static uns8 *blockPtr, lastByte, bitMask, bitsLeft, lastValue;
#define pulseLen		pilotLen
#define pulse			pilot
#define pulseCount		bitsLeft
#define sample			pilot

int
TzxNewFile(void)
{
	struct tzxHeader *header = (struct tzxHeader *)TpInBegin;
	if (memcmp(header->ident, "ZXTape!", sizeof(header->ident)))
	{
		Msg(M_ERR, "input file is not in TZX format");
		return -1;
	}
	DEB(Msg(M_DEBUG, "TZX: version %d.%d", header->major, header->minor););
	if (header->major > MAJREV
		|| (header->major == MAJREV && header->minor > MINREV))
	{
		Msg(M_WARN, "XZX does not support TXZ format newer than %d.%d",
			MAJREV, MINREV);
		return -1;
	}
	RESET_IN_LOOP;
	RESET_IN_CALL;
	tzxScanFile();
#ifdef REGISTERED
	VtpNewFile();
#endif
	return 0;
}

void
TzxStartPlaying(void)
{
	tzxRecord *tr;
	tr = (tzxRecord *)RetrieveElemList(TpInBlockList, TpInBlockCurrent);
	assert(tr != NULL);

	if (tr->type == 0x20 || tr->type == 0x2a)
	{
		TpInBlockCurrent++;
	}
	blockType = 0x00;
	lastTstate = Vlines * Machine->tstatesPerLine + Tstates;

	LOW_PULSE;
}

uns8
TzxGetNextBit(void)
{
	unsigned int passed;
	passed = Vlines * Machine->tstatesPerLine + Tstates - lastTstate;
  again:
	switch (blockType)
	{
		case 0x00:
			TpInBlockCurrent--;
			goto next;
		case 0x10:
		case 0x11:
		case 0x14:
			if (blockState == BS_PILOT)
			{
				while (passed >= pilot)
				{
					passed -= pilot;
					lastTstate += pilot;
					TOGGLE_PULSE;
					if (!--pilotLen)
					{
						blockState = BS_SYNC1;
						goto sync1;
					}
				}
			}
			else if (blockState == BS_SYNC1)
			{
			  sync1:
				if (passed >= sync1)
				{
					passed -= sync1;
					lastTstate += sync1;
					TOGGLE_PULSE;
					blockState = BS_SYNC2;
					goto sync2;
				}
			}
			else if (blockState == BS_SYNC2)
			{
			  sync2:
				if (passed >= sync2)
				{
					passed -= sync2;
					lastTstate += sync2;
					TOGGLE_PULSE;
					if (blockLen)
					{
						bitMask = 0x80;
						bitsLeft = blockLen == 1 ? lastByte : 8;
						curBit = *blockPtr & bitMask ? bit1 : bit0;
						blockState = BS_BIT_1;
						goto bit_1;
					}
					else
					{

						blockState = BS_PAUSE;
						goto pause;
					}
				}
			}
			else if (blockState == BS_BIT_1)
			{
			  bit_1:
				if (passed >= curBit)
				{
					passed -= curBit;
					lastTstate += curBit;
					TOGGLE_PULSE;
					blockState = BS_BIT_2;
					goto bit_2;
				}
			}
			else if (blockState == BS_BIT_2)
			{
			  bit_2:
				if (passed >= curBit)
				{
					passed -= curBit;
					lastTstate += curBit;
					TOGGLE_PULSE;
					if (!--bitsLeft)
					{
						if (!--blockLen)
						{
							if (pauseLen)
							{
								blockState = BS_PAUSE;
								goto pause;
							}
							else
							{
								goto next;
							}
						}
						bitMask = 0x80;
						bitsLeft = blockLen == 1 ? lastByte : 8;
						blockPtr++;
					}
					else
					{
						bitMask >>= 1;
					}
					curBit = *blockPtr & bitMask ? bit1 : bit0;
					blockState = BS_BIT_1;
					goto bit_1;
				}
			}
			else if (blockState == BS_PAUSE)
			{
			  pause:
				while (passed >= 1*MS)
				{
					passed -= 1*MS;
					lastTstate += 1*MS;

					LOW_PULSE;
					if (!--pauseLen)
					{
					  next:
						TpInBlockCurrent++;
						if (!tzxGetNextBlock())
						{
							goto again;
						}
					}
				}
			}
			break;
		case 0x12:
			while (passed >= pulse)
			{
				passed -= pulse;
				lastTstate += pulse;
				TOGGLE_PULSE;
				if (!--pulseLen)
				{
					goto next;
				}
			}
			break;
		case 0x13:
			while (passed >= pulse)
			{
				passed -= pulse;
				lastTstate += pulse;
				TOGGLE_PULSE;
				if (!--pulseCount)
				{
					goto next;
				}
				else
				{
					pulse = (uns16)GET2(blockPtr, 0x00);
					blockPtr += 2;
				}
			}
			break;
		case 0x15:
			if (passed >= sample)
			{
				passed -= sample;
				lastTstate += sample;
				if (*blockPtr & bitMask)
				{
					HIGH_PULSE;
				}
				else
				{
					LOW_PULSE;
				}
				if (!--bitsLeft)
				{
					if (!--blockLen)
					{
						if (pauseLen)
						{
							blockType = 0x10;
							blockState = BS_PAUSE;
							goto pause;
						}
						else
						{
							goto next;
						}
					}
					bitMask = 0x80;
					bitsLeft = blockLen == 1 ? lastByte : 8;
					blockPtr++;
				}
				else
				{
					bitMask >>= 1;
				}
			}
			break;
		case 0x20:
			while (passed >= 1*MS)
			{
				passed -= 1*MS;
				lastTstate += 1*MS;
				if (!--pauseLen)
				{
					goto next;
				}
			}
			break;
	}
#if defined(SUN_AUDIO) || defined(PCSPKR_AUDIO)
	if (GETCFG(noise))
	{
		SpeakerOutByte(lastValue >> 2);
	}
#endif
	return lastValue;
}

static int
tzxGetNextBlock(void)
{
	tzxRecord *tr, *tr2;
	int i;
	for (; TpInBlockCurrent < TpInBlockCount;)
	{
#ifdef REGISTERED
		VtpUpdate();
#endif
		tr = (tzxRecord *)RetrieveElemList(TpInBlockList, TpInBlockCurrent);
		assert(tr != NULL);
		DEB(Msg(M_DEBUG, "TZX: curBlock = %3u, blockType = %02x",
				TpInBlockCurrent, tr->type););
		switch ((blockType = tr->type))
		{
			case 0x00:
				DEB(Msg(M_DEBUG, "TZX: end of tape reached"););
				goto quit;
			case 0x10:
				pauseLen = (uns16)GET2(tr->pos, 0x01);
				blockLen = (uns16)GET2(tr->pos, 0x03);
				blockPtr = tr->pos + 0x05;
				pilotLen = *blockPtr ? 3220 : 8064;
				pilot = 2168;
				sync1 = 667;
				sync2 = 735;
				bit0 = 855;
				bit1 = 1710;
				lastByte = 8;
				blockState = BS_PILOT;
				return 0;
			case 0x11:
				pauseLen = (uns16)GET2(tr->pos, 0x0e);
				blockLen = (uns32)GET3(tr->pos, 0x10);
				blockPtr = tr->pos + 0x13;
				pilotLen = (uns16)GET2(tr->pos, 0x0b);
				pilot = (uns16)GET2(tr->pos, 0x01);
				sync1 = (uns16)GET2(tr->pos, 0x03);
				sync2 = (uns16)GET2(tr->pos, 0x05);
				bit0 = (uns16)GET2(tr->pos, 0x07);
				bit1 = (uns16)GET2(tr->pos, 0x09);
				lastByte = GET1(tr->pos, 0x0d);
				blockState = BS_PILOT;
				return 0;
			case 0x12:
				pulseLen = (uns16)GET2(tr->pos, 0x03);
				pulse = (uns16)GET2(tr->pos, 0x01);
				return 0;
			case 0x13:
				pulseCount = GET1(tr->pos, 0x01);
				pulse = (uns16)GET2(tr->pos, 0x02);
				blockPtr = tr->pos + 0x04;
				return 0;
			case 0x14:
				pauseLen = (uns16)GET2(tr->pos, 0x06);
				blockLen = (uns32)GET3(tr->pos, 0x08);
				blockPtr = tr->pos + 0x0b;
				bit0 = (uns16)GET2(tr->pos, 0x01);
				bit1 = (uns16)GET2(tr->pos, 0x03);
				lastByte = GET1(tr->pos, 0x05);
				bitMask = 0x80;
				bitsLeft = blockLen == 1 ? lastByte : 8;
				curBit = *blockPtr & bitMask ? bit1 : bit0;
				blockState = BS_BIT_1;
				return 0;
			case 0x15:
				pauseLen = (uns16)GET2(tr->pos, 0x03);
				blockLen = (uns32)GET3(tr->pos, 0x06);
				blockPtr = tr->pos + 0x09;
				sample = (uns16)GET2(tr->pos, 0x01);

				sample = sample ? sample : 1;
				lastByte = GET1(tr->pos, 0x05);
				bitMask = 0x80;
				return 0;
			case 0x20:
				pauseLen = (uns16)GET2(tr->pos, 0x01);
				if (!pauseLen)
				{
					TpStopPlayback();
				}
				return 0;
			case 0x21:
			case 0x22:
				TpInBlockCurrent++;
				break;
			case 0x23:
				TpInBlockCurrent += (sgn16)GET2(tr->pos, 1);
				break;
			case 0x24:
				SET_IN_LOOP;
				loopStart = TpInBlockCurrent;
				loopRepeat = (uns16)GET2(tr->pos, 1);
				TpInBlockCurrent++;
				break;
			case 0x25:
				if (!IN_LOOP)
				{
					Msg(M_ERR, "Loop End without Loop in TZX file");
					TpState |= TS_ERRIN;
					TpStopPlayback();
					return -1;
				}
				if (--loopRepeat)
				{
					TpInBlockCurrent = loopStart + 1;
				}
				else
				{
					TpInBlockCurrent++;
					RESET_IN_LOOP;
				}
				break;
			case 0x26:
				SET_IN_CALL;
				callStart = TpInBlockCurrent;
				callCount = (uns16)GET2(tr->pos, 1);
				callCurrent = 0;
				TpInBlockCurrent += (sgn16)GET2(tr->pos, 3);
				break;
			case 0x27:
				if (!IN_CALL)
				{
					Msg(M_ERR, "Return without Call in TZX file");
					TpState |= TS_ERRIN;
					TpStopPlayback();
					return -1;
				}
				if (++callCurrent < callCount)
				{
					tr2 = (tzxRecord *)RetrieveElemList(TpInBlockList,
														(int)callStart);
					assert(tr2 != NULL);
					i = 1 + 2 * callCurrent;
					TpInBlockCurrent = callStart + (sgn16)GET2(tr2->pos, i);
				}
				else
				{
					TpInBlockCurrent = callStart + 1;
					RESET_IN_CALL;
				}
				break;
			case 0x28:
				TpInBlockCurrent = tzxSelection(TpInBlockCurrent, tr->pos);
				break;
			case 0x2a:
				if (GETCFG(machine) <= SP_48_3)
				{
					TpStopPlayback();
					return 0;
				}
				TpInBlockCurrent++;
				break;
			case 0x40:
				Msg(M_WARN, "unsupported TZX block %02x", blockType);
				TpInBlockCurrent++;
				break;
			case 0x30:
			case 0x31:
			case 0x32:
			case 0x33:
			case 0x34:
			case 0x35:
			case 0x5a:
				TpInBlockCurrent++;
				break;
			default:
				Msg(M_WARN, "unknown TZX block %02x", blockType);
				TpInBlockCurrent++;
				break;
		}
	}
  quit:
	TpState |= TS_ENDIN;
	TpStopPlayback();
	return -1;
}

static unsigned int
tzxSelection(unsigned int block, uns8 *bp)
{
#ifndef REGISTERED
	int oldInDialog;
#endif
	int cnt = GET1(bp, 0x03), choice = 0, i;
	char buf[31], **choices;
	char *s;
	if (!cnt)
	{
		DEB(Msg(M_DEBUG, "TZX: selection with zero choices"););
		return ++block;
	}
	choices = Malloc(cnt * sizeof(char  *), "tzxSelection");
	for (i = 0, s = &bp[0x04]; i < cnt; i++)
	{
		(void)sprintf(buf, "%.*s", s[2] > 30 ? 30 : (int)s[2], s + 3);
		choices[i] = Strdup(buf, "tzxSelection");
		s += 3 + s[2];
	}
#ifdef REGISTERED
	if ((choice = BrowseList("Choose an item", choices, cnt, choice)) == -1)
	{
		choice = 0;
	}
#else

	oldInDialog = EnterOSD();

	if (cnt > PAGE_SIZE)
	{
		Msg(M_WARN, "too much choices, reducing to %d", PAGE_SIZE);
		cnt = PAGE_SIZE;
	}

	{
		char *selectorText[PAGE_SIZE + 2];
		int j;
		selectorText[0] = "Choose an item";
		for (i = 1, j = 0; j < cnt; i++, j++)
		{
			selectorText[i] = choices[j];
		}
		selectorText[i] = NULL;
		DisplayMenu(selectorText);
	}
	for (;;)
	{
		switch (BrowseList(&choice, cnt, cnt, NULL))
		{

			case XK_Escape:
				choice = 0;

			case XK_Return:
				goto quit;
		}
	}
  quit:
	LeaveOSD(oldInDialog);
#endif

	while (--cnt >= 0)
	{
		free(choices[cnt]);
	}
	free(choices);

	for (i = 0, s = &bp[0x04]; i < choice; i++)
	{
		s += 3 + s[2];
	}

	block += (sgn16)GET2(s, 0x00);
	return block;
}

static void
tzxScanFile(void)
{
	char buf[31];
	uns8 *scanPtr = TpInBegin + sizeof(struct tzxHeader);
	tzxRecord *tr;
	int i;
	for (;;)
	{
		tr = Malloc(sizeof(tzxRecord), "tzxScanFile");
		TpInBlockList = AppendElemList(TpInBlockList, (void *)tr);
		TpInBlockCount++;
		tr->pos = scanPtr;
		if (scanPtr >= TpInEnd)
		{
			tr->type = 0x00;
			tr->str = Strdup("     *** End Of Tape ***", "tzxScanFile");
			break;
		}
		tr->type = *scanPtr++;
		DEB(Msg(M_DEBUG, "TZX: scaning, id = %02x, pos = %6d",
				tr->type, tr->pos - TpInBegin););
		switch (tr->type)
		{
			case 0x10:
				tr->str = Strdup("Standard", "tzxScanFile");
				scanPtr += 0x04 + GET2(scanPtr, 0x02);
				break;
			case 0x11:
				tr->str = Strdup("Turbo", "tzxScanFile");
				scanPtr += 0x12 + GET3(scanPtr, 0x0f);
				break;
			case 0x12:
				tr->str = Strdup("Pure tone", "tzxScanFile");
				scanPtr += 0x04;
				break;
			case 0x13:
				tr->str = Strdup("Sequence of pulses", "tzxScanFile");
				scanPtr += 0x01 + (GET1(scanPtr, 0) * 2);
				break;
			case 0x14:
				tr->str = Strdup("Pure data", "tzxScanFile");
				scanPtr += 0x0a + GET3(scanPtr, 0x07);
				break;
			case 0x15:
				tr->str = Strdup("Direct", "tzxScanFile");
				scanPtr += 0x08 + GET3(scanPtr, 0x05);
				break;
			case 0x20:
				tr->str = Strdup(GET2(scanPtr, 0x00) ?
								 "Pause" :
								 "Stop the tape", "tzxScanFile");
				scanPtr += 0x02;
				break;
			case 0x21:
				i = GET1(scanPtr, 0x00) > 23 ? 23 : GET1(scanPtr, 0x00);
				(void)sprintf(buf, "Group: %.*s", i, scanPtr + 0x01);
				tr->str = Strdup(buf, "tzxScanFile");
				scanPtr += 0x01 + GET1(scanPtr, 0);
				break;
			case 0x22:
				tr->str = Strdup("Group end", "tzxScanFile");
				scanPtr += 0x00;
				break;
			case 0x23:
				(void)sprintf(buf, "Jump %d",
							  TpInBlockCount + (sgn16)GET2(tr->pos, 1));
				tr->str = Strdup(buf, "tzxScanFile");
				scanPtr += 0x02;
				break;
			case 0x24:
				tr->str = Strdup("Loop start", "tzxScanFile");
				scanPtr += 0x02;
				break;
			case 0x25:
				tr->str = Strdup("Loop end", "tzxScanFile");
				scanPtr += 0x00;
				break;
			case 0x26:
				tr->str = Strdup("Call", "tzxScanFile");
				scanPtr += 0x02 + (GET2(scanPtr, 0x00) * 2);
				break;
			case 0x27:
				tr->str = Strdup("Return", "tzxScanFile");
				scanPtr += 0x00;
				break;
			case 0x28:
				tr->str = Strdup("Select", "tzxScanFile");
				scanPtr += 0x02 + GET2(scanPtr, 0x00);
				break;
			case 0x2a:
				tr->str = Strdup("Stop the tape if in 48k mode", "tzxScanFile");
				scanPtr += 0x04 + GET4(scanPtr, 0x00);
				break;
			case 0x30:
				i = GET1(scanPtr, 0x00) > 30 ? 30 : GET1(scanPtr, 0x00);
				(void)sprintf(buf, "%.*s", i, scanPtr + 0x01);
				tr->str = Strdup(buf, "tzxScanFile");
				scanPtr += 0x01 + GET1(scanPtr, 0);
				break;
			case 0x31:
				tr->str = Strdup("Message", "tzxScanFile");
				scanPtr += 0x02 + GET1(scanPtr, 1);
				break;
			case 0x32:
				tr->str = Strdup("Archive info", "tzxScanFile");
				scanPtr += 0x02 + GET2(scanPtr, 0x00);
				break;
			case 0x33:
				tr->str = Strdup("Hardware", "tzxScanFile");
				scanPtr += 0x01 + *scanPtr * 3;
				break;
			case 0x34:
				tr->str = Strdup("Emulation info", "tzxScanFile");
				scanPtr += 0x08;
				break;
			case 0x35:
				tr->str = Strdup("Custom info", "tzxScanFile");
				scanPtr += 0x14 + GET4(scanPtr, 0x10);
				break;
			case 0x40:
				tr->str = Strdup("Snapshot", "tzxScanFile");
				scanPtr += 0x04 + GET3(scanPtr, 0x01);
				break;
			case 0x5a:
				tr->str = Strdup("Merged files", "tzxScanFile");
				scanPtr += 0x09;
				break;
			default:
				Msg(M_WARN, "unknown TZX block %x", tr->type);
				tr->str = Strdup("Unknown", "tzxScanFile");
				scanPtr += 0x04 + (uns32)GET4(scanPtr, 0x00);
				break;
		}
	}
}

