/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                  Authors:  Paul Taylor                                */
/*                  Date   :  Oct 95                                     */
/*-----------------------------------------------------------------------*/
/*                 Event RFC and Tilt labelling                          */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include "EST_tilt.h"
#include "sigpr/EST_sigpr_utt.h"
#include "EST_cmd_line_options.h"
#include "ling_class/EST_relation_aux.h"
#include "EST_string_aux.h"

#define SIL_NAMES "sil !ENTER !EXIT"
#define EVENT_NAMES "a rb arb m mrb"

void rfc_segment_features_only(EST_Relation &ev);
void target_rfc_to_local_rfc(EST_Relation &ev);
void event_rfc_to_segment_rfc(EST_Track &fz, EST_Relation &ev, 
			      EST_Option &op);
void tilt_segment_features_only(EST_Relation &ev);

void change_label(EST_Relation &seg, const EST_StrList &oname, 
		  const EST_String &nname);

void set_starts(EST_Relation &ev);
void set_fn_start(EST_Relation &ev);

void remove_rfc_features(EST_Relation &ev);
void remove_tilt_features(EST_Relation &ev);

int main(int argc, char *argv[])
{
    EST_Track fz, nfz;
    EST_Relation ev;
    EST_Option al, op;
    EST_StrList files, event_list, sil_list;
    EST_String out_file, pstring;

    parse_command_line(argc, argv, 
       EST_String("Usage:   ")+
       "tilt_analysis <input f0 file> -e <input event label file> -o " +
		"<output file> <options>\n"+
       "Summary: produce rfc file from events and f0 contour\n"+
       "use \"-\" to make input and output files stdin/out\n"+
       "-h               Options help\n"+
       options_track_input()+ "\n"+
       "-event_names   <string>  List of labels to be classed as accents\n"+
       "-sil_names     <string>  List of labels to be classed as silence\n"+
       "-e             <ifile>   Input event label file\n"+
       "-o             <ofile>   Output Tilt or RFC label fil\n"+
       "-otype         <string>  File type of output file\n"+
       "-range         <float>   RFC search region range\n"+
       "-smooth                  Smooth and Interpolate input F0 contour\n"+
       "-w1 <float>      size of smoothing window prior to interpolation\n" +
       "-w2 <float>      size of smoothing window after to interpolation\n" + 
       "-timing_style  <string>  Stream format, \"segment\" or \"event\"\n"+
       "-int_style     <string>  Intonation format, \"tilt\" or \"rfc\"\n", 
			files, al);

    default_rfc_params(op);
    init_lib_ops(al, op);
    
    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    if (read_track(nfz, files.first(), al) == -1)
	    exit(-1);
    // REORG - extract proper f0 channel here
    nfz.copy_sub_track(fz, 0, EST_ALL, 0, 1);

    if (ev.load(al.val("-e")) != format_ok)
	exit(-1);

    cout << ev;

    pstring = (al.present("-event_names") ? al.val("-event_names"): 
	       EST_String("a b ab pos"));
    StringtoStrList(pstring, event_list);
    convert_to_broad(ev, event_list, "int_event", 1);

    // ensure all sil_names are re-written as sil
    pstring = (al.present("-sil_names") ? al.val("-sil_names"): 
	       EST_String(SIL_NAMES));
    StringtoStrList(pstring, sil_list);
    change_label(ev, sil_list, "sil");

    if (al.present("-smooth"))
    {
	EST_Track speech, raw_fz;
	EST_Relation sil_lab;
	sil_lab = ev;	
	StringtoStrList("sil", sil_list);
	convert_to_broad(sil_lab, sil_list, "pos", 0);
	label_to_track(sil_lab, speech, fz.shift());
	raw_fz = fz;
	smooth_phrase(raw_fz, speech, op, fz);
    }

    ev.f.set("name", "intevents");
    ev.f.set("timing_style", "segment");

    // temporary fix until status of start and end is finalised
/*    float prev_end = 0.0;


    EST_Item *e;
    for (e = ev.head(); e; e = next(e))
    {
	cout << *e << endl;
	e->fset("start", prev_end);
	prev_end = e->fF("end");
    }
*/

    set_fn_start(ev);
    cout << "START\n";
    cout << ev;

    rfc_analysis(fz, ev, op);

    if (al.val("-int_style", 0) == "tilt")
    {
	rfc_to_tilt(ev);
	remove_rfc_features(ev);
    }

/*    if (al.val("-timing_style", 0) == "segment")
	event_rfc_to_segment_rfc(fz, ev, op);

    cout << ev << endl;
    sleep(2);

    if (al.val("-int_style", 0) == "rfc_local")
    {
	target_rfc_to_local_rfc(ev);
	ev.f.set("name", "rfc_local");
    }
    }
    else // target tilt style
    {
	rfc_segment_features_only(ev);
	ev.f.set("name", "rfc_target");
    }
*/

    ev.save(out_file);
    return 0;
}

void override_lib_ops(EST_Option &a_list, EST_Option &al)
{
    // general options
    a_list.override_val("sample_rate", al.val("-f", 0));
    a_list.override_val("rfc_start_range", al.val("-f", 0));
    a_list.override_val("rfc_stop_range", al.val("-f", 0));
    a_list.override_val("rfc_search_range", al.val("-range", 0));
    a_list.override_val("window_length", al.val("-w1", 0));
    a_list.override_val("second_length", al.val("-w2", 0));
}    
