/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.i18n;

/** JDK classes **/
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.excalibur.xml.xpath.XPathProcessor;
import org.apache.avalon.excalibur.xml.Parser;
import org.apache.avalon.excalibur.source.Source;
import org.apache.avalon.excalibur.source.SourceResolver;
import org.apache.avalon.excalibur.source.SourceUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.NamedNodeMap;
import org.xml.sax.InputSource;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;

/**
 * @author <a href="mailto:mengelhart@earthtrip.com">Mike Engelhart</a>
 * @author <a href="mailto:neeme@apache.org">Neeme Praks</a>
 * @author <a href="mailto:oleg@one.lv">Oleg Podolsky</a>
 * @version $Id: XmlBundle.java,v 1.8 2002/01/04 13:21:31 cziegeler Exp $
 */
public class XmlBundle extends AbstractBundle implements Configurable, Initializable, Disposable, Composable {

    /** Constants for configuration keys */
    public static class ConfigurationKeys {
        public static final String LOAD_ON_INIT = "load-on-init";
        public static final String USE_ROOT = "use-root-element";
    }

    /** Cache for storing string values for existing XPaths */
    private Map cache = new HashMap();

    /** Cache for storing non-existing XPaths */
    private Map cacheNotFound = new HashMap();

    /** DOM-tree containing the bundle content */
    private Document doc;

    /** Component Manager */
    protected ComponentManager manager = null;

    /** XPath Processor */
    private XPathProcessor processor = null;

    /** bundle info mapper */
    private BundleInfoMapper mapper;

    /** bundle name prefix */
    private String prefix;

    /** bundle name suffix */
    private String suffix;

    /** Load all the keys when initializing? */
    private boolean loadOnInit = true;

    /** Use the XML root element in key names when pre-loading? */
    private boolean useRootElement = false;

    public void compose(ComponentManager manager) {
        this.manager = manager;
        try {
            this.processor = (XPathProcessor)this.manager.lookup(XPathProcessor.ROLE);
            this.mapper = (BundleInfoMapper)this.manager.lookup(BundleInfoMapper.ROLE);
        } catch (Exception e) {
            getLogger().error("cannot obtain XPathProcessor", e);
        }
    }

    public void configure(Configuration configuration) throws ConfigurationException {
        this.loadOnInit = configuration.getAttributeAsBoolean(ConfigurationKeys.LOAD_ON_INIT, true);
        this.useRootElement = configuration.getAttributeAsBoolean(ConfigurationKeys.USE_ROOT, false);
    }

    /**
     * Initalize the bundle
     *
     * @param document          XML source document (DOM)
     */
    public void initialize() throws Exception {
        initialize(getMapper().map(getBundleInfo()));
    }

    /**
     * Initalize the bundle
     *
     * @param uri               URI of the XML source
     * @exception IOException   if an IO error occurs while reading the file
     * @exception ParserConfigurationException if no parser is configured
     * @exception SAXException  if an error occurs while parsing the file
     */
    public void initialize(String uri) throws Exception {
        SourceResolver resolver = (SourceResolver) this.manager.lookup(SourceResolver.ROLE);
        initialize(resolver.resolve(uri));
        this.manager.release(resolver);
    }

    /**
     * Initalize the bundle
     *
     * @param file              XML source file
     * @exception IOException   if an IO error occurs while reading the file
     * @exception ParserConfigurationException if no parser is configured
     * @exception SAXException  if an error occurs while parsing the file
     */
    public void initialize(Source source) throws Exception {
        initialize( SourceUtil.getInputSource( source ) );
    }

    /**
     * Initalize the bundle
     *
     * @param inStream          XML source stream
     * @exception IOException   if an IO error occurs while reading the file
     * @exception ParserConfigurationException if no parser is configured
     * @exception SAXException  if an error occurs while parsing the file
     */
    public void initialize(InputSource inSource) throws Exception {
        Parser parser = (Parser) this.manager.lookup(Parser.ROLE);
        this.doc = parser.parseDocument(inSource);
        load();
    }

    /**
     * Load (pre-cache) the contents
     *
     * @param document          XML source document (DOM)
     */
    private void load() throws Exception {
        if (this.loadOnInit) {
            cacheAll(doc.getDocumentElement(), useRootElement ? '/' + doc.getDocumentElement().getTagName() : "");
        }
    }

    /**
     * Convert the &quot;user view&quot; of the lookup key to the
     * &quot;system view&quot;.
     * Current implementation coverts dots into slashes, so common Java property
     * names become XPath paths.
     * E.g: this.is.java.property.key --> /this/is/java/property/key
     *
     * @param key           user key
     * @return              system key
     */
    public String convertKey(String userKey) {
        return '/' + userKey.replace('.', '/');
    }

    /**
     * Gets the source DOM tree of the bundle.
     *
     * @return the DOM tree
     */
    public Document getDocument() {
        return this.doc;
    }

    /**
     * Does the &quot;key-cache&quot; contain the value with such key?
     *
     * @param   key     the key to the value to be returned
     * @return  true if contains, false otherwise
     */
    protected boolean cacheContains(String key) {
        boolean result = cache.containsKey(key);
        if (getLogger().isDebugEnabled()) getLogger().debug(getBundleInfo() + ": cache contains key '" + key + "': " + result);
        return result;
    }

    /**
     * Does the &quot;key-not-found-cache&quot; contain such key?
     *
     * @param   key     the key to the value to be returned
     * @return  true if contains, false otherwise
     */
    protected boolean cacheNotFoundContains(String key) {
        boolean result = cacheNotFound.containsKey(key);
        if (getLogger().isDebugEnabled()) getLogger().debug(getBundleInfo() + ": cache_not_found contains key '" + key + "': " + result);
        return result;
    }

    /**
     * Cache the key and value in &quot;key-cache&quot;.
     *
     * @param   key     the key
     * @param   value   the value
     */
    protected void cacheKey(String key, String value) {
        if (getLogger().isDebugEnabled()) getLogger().debug(getBundleInfo() + ": caching: " + key + " = " + value);
        cache.put(key, value);
    }

    /**
     * Cache the key in &quot;key-not-found-cache&quot;.
     *
     * @param   key     the key
     */
    protected void cacheNotFoundKey(String key) {
        if (getLogger().isDebugEnabled()) getLogger().debug(getBundleInfo() + ": caching not_found: " + key);
        cacheNotFound.put(key, "");
    }

    /**
     * Gets the value by the key from the &quot;key-cache&quot;.
     *
     * @param   key     the key
     * @return          the value
     */
    protected String getFromCache(String key) {
        if (getLogger().isDebugEnabled()) getLogger().debug(getBundleInfo() + ": returning from cache: " + key);
        return (String) cache.get(key);
    }

    /**
     * Steps through the bundle tree and stores all text element values
     * in bundle's cache. Also stores attributes for all element nodes.
     *
     * @param   parent          parent node, must be an element
     * @param   pathToParent    XPath to the parent node
     */
    protected void cacheAll(Node parent, String pathToParent) {
        NodeList children = parent.getChildNodes();
        int childnum = children.getLength();

        for(int i = 0; i < childnum; i++) {
            Node child = children.item(i);

            if(child.getNodeType() == Node.ELEMENT_NODE) {
                StringBuffer pathToChild = new StringBuffer(pathToParent).append('/').append(child.getNodeName());

                NamedNodeMap attrs = child.getAttributes();
                if(attrs != null) {
                    Node temp = null;
                    String pathToAttr = null;
                    int attrnum = attrs.getLength();
                    for(int j = 0; j < attrnum; j++) {
                        temp = attrs.item(j);
                        if (!temp.getNodeName().equalsIgnoreCase("xml:lang"))
                            pathToChild.append("[@").append(temp.getNodeName())
                                    .append("='").append(temp.getNodeValue())
                                    .append("']");
                    }
                }

                String childValue = getTextValue(child);
                if(childValue != null)
                    cacheKey(pathToChild.toString(), childValue);
                else
                    cacheAll(child, pathToChild.toString());
            }
        }
    }

    /**
     * Get value by key.
     *
     * @param key           key
     * @return              value
     * @exception MissingResourceException if value was not found
     */
    public String getString(String key) throws MissingResourceException {
        String value = _getString(convertKey(key));
        if (value == null)
            throw new MissingResourceException(
                    "Unable to locate resource: " + key,
                    XmlBundle.class.getName(),
                    key);
        else
            return value;
    }

    /**
     * Get value by key.
     *
     * @param   key     the key
     * @return          the value
     */
    protected String _getString(String key) {
        if (key == null) return null;
        String value = getFromCache(key);

        if (value == null && !cacheNotFoundContains(key))
        {
        if (doc != null)
                value = _getString(this.doc.getDocumentElement(), key);

            if (value == null)
            {
                if (getParent() != null)
                    value = getParent().getString(key);
            }

            if (value != null)
                cacheKey(key, value);
            else
                cacheNotFoundKey(key);
        }
        return value;
    }

    /**
     * Get value by key from a concrete node.
     *
     * @param   node    the node
     * @param   key     the key
     * @return          the value
     */
    protected String _getString(Node node, String key) {
        String value = null;
        try {
            value = getTextValue(_getNode(node, key));
        }
        catch (Exception e) {
            getLogger().error(getBundleInfo() + ": error while locating resource: " + key, e);
        }
        return value;
    }

    /**
     * Get the text value of the node.
     *
     * @param   node    the node
     * @return          the value
     */
    protected static String getTextValue(Node element) {
        if (element == null) return null;
        NodeList list = element.getChildNodes();
        int listsize = list.getLength();

        Node item = null;
        String itemValue = null;

        for(int i = 0; i < listsize; i++) {
            item = list.item(i);
        if(item.getNodeType() != Node.TEXT_NODE)
            return null;

            itemValue = item.getNodeValue();
        if(itemValue == null)
            return null;

            itemValue = itemValue.trim();
        if(itemValue.length() == 0)
            return null;

            return itemValue;
        }
        return null;
    }

    /**
     * Get the node with the supplied XPath key.
     *
     * @param   key     the key
     * @return          the node
     */
    protected Node _getNode(String key) {
        return _getNode(this.doc.getDocumentElement(), key);
    }

    /**
     * Get the node with the supplied XPath key, starting from concrete
     * root node.
     *
     * @param   rootNode    the root node
     * @param   key         the key
     * @return              the node
     */
    protected Node _getNode(Node rootNode, String key) {
        Node node = null;
        try {
            node = this.processor.selectSingleNode(rootNode, key);
        }
        catch (Exception e) {
            getLogger().error("Error while locating resource with key: " + key, e);
        }
        return node;
    }

    public void dispose() {
        this.manager.release((Component)this.processor);
    }
}
