/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.lang;

import java.util.HashMap;
import java.util.Map;
import java.util.Iterator;

/**
 * Default <code>ThreadContextPolicy</code> that just maintains the 
 * ContextClassLoader <code>ThreadLocal</code> variable. This is a useful 
 * class to extend for those wanting to write their own Policy.
 *
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 */
public class DefaultThreadContextPolicy
    implements ThreadContextPolicy
{
    /**
     * The activate method is called when the ThreadContext 
     * is associated with a thread. This method sets the ContextClassLoader
     * if CLASSLOADER key is present in context.
     *
     * @param accessor the accessor to retrieve values from ThreadContext
     */
    public void activate( final ThreadContextAccessor accessor )
    {
        final ClassLoader classLoader = (ClassLoader)get( accessor, CLASSLOADER, null, ClassLoader.class );
        if( null != classLoader ) Thread.currentThread().setContextClassLoader( classLoader );
    }

    /**
     * The deactivate method is called when the ThreadContext is 
     * dis-associated with a thread.
     *
     * @param accessor the accessor to retrieve values from ThreadContext
     */
    public void deactivate( final ThreadContextAccessor accessor )
    {
    }

    /**
     * Verify that the key/value pair is valid.
     *
     * @param key The key
     * @param value the value
     * @exception IllegalArgumentException if pair is not valid
     */
    public void verifyKeyValue( final String key, final Object value )
        throws IllegalArgumentException
    {
        if( key.equals( CLASSLOADER ) && !(value instanceof ClassLoader) )
        {
            throw new IllegalArgumentException( "Key " + key + " must be of type " + 
                                                ClassLoader.class.getName() );
        }
    }

    /**
     * Get a value for specified key, using specified default if none present 
     * and making sure value is of specified type.
     *
     * @param key the key used to lookup value
     * @param defaultValue the default value if the key does not specify value
     * @param type the expected type of value
     * @return the value
     */
    protected Object get( final ThreadContextAccessor accessor,
                          final String key, 
                          final Object defaultValue, 
                          final Class type )
    {
        Object result = defaultValue;

        if( accessor.containsKey( key ) )
        {
            result = accessor.get( key );
        }

        if( null != result && !type.isInstance( result ) )
        {
            throw new IllegalArgumentException( "Key " + key + " expected to access " + 
                                                type.getName() + " but got " + 
                                                result.getClass().getName() );
        }

        return result;
    }
}
