/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.source;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * Base interface for resolving a source by system identifiers.
 * Instead of using the java.net.URL classes which prevent you
 * from adding your own custom protocols in a server environment,
 * you should use this resolver for all URLs.
 *
 * The resolver creates for each source a <code>Source</code>
 * object, which could then be asked for an <code>InputStream</code>
 * etc.
 *
 * When the <code>Source</code> object is no longer needed
 * it must be released using the resolver. This is very similar like
 * looking up components from a <code>ComponentManager</code>.
 * In fact a source object can implement most lifecycle interfaces
 * like Composable, Initializable, Disposable etc.
 *
 * @author <a href="mailto:cziegeler@apache.org">Carsten Ziegeler</a>
 * @version CVS $Revision: 1.8 $ $Date: 2002/01/25 21:12:56 $
 */

public interface SourceResolver
extends Component {

    String ROLE = "org.apache.avalon.excalibur.source.SourceResolver";

    /**
     * Get a <code>Source</code> object.
     */
    Source resolve(String location)
    throws MalformedURLException, IOException, ComponentException;

    /**
     * Get a <code>Source</code> object.
     */
    Source resolve(URL base, String location)
    throws MalformedURLException, IOException, ComponentException;

    /**
     * Get a <code>Source</code> object.
     */
    Source resolve(String location,
                   SourceParameters parameters)
    throws MalformedURLException, IOException, ComponentException;

    /**
     * Get a <code>Source</code> object.
     */
    Source resolve(URL base,
                   String location,
                   SourceParameters parameters)
    throws MalformedURLException, IOException, ComponentException;

    /**
     * Releases a resolved resource
     */
    void release( Source source );
}

