/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.source;

import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.excalibur.monitor.FileResource;
import org.apache.avalon.excalibur.monitor.Monitorable;
import org.apache.avalon.excalibur.monitor.Resource;
import org.apache.avalon.excalibur.monitor.SourceResource;
import org.apache.avalon.excalibur.source.validity.TimeStampValidity;
import org.apache.avalon.excalibur.xml.Parser;
import org.apache.avalon.excalibur.xml.XMLConsumer;
import org.apache.avalon.excalibur.xml.XMLizable;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.ext.LexicalHandler;

import java.io.*;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLConnection;

/**
 * Description of a source which is described by an URL.
 *
 * @author <a href="mailto:cziegeler@apache.org">Carsten Ziegeler</a>
 * @version CVS $Revision: 1.11 $ $Date: 2002/01/08 13:43:48 $
 */

public final class URLSource
implements Composable, ModifiableSource, XMLizable, Monitorable {

    /** Identifier for file urls */
    private final String FILE = "file:";

    /** The last modification date or 0 */
    private long lastModificationDate;

    /** The content length */
    private long contentLength;

    /** The system id */
    private String systemId;

    /** The URL of the source */
    private URL url;

    /** The connection for a real URL */
    private URLConnection connection;

    /** Is this a file or a "real" URL */
    private boolean isFile;

    /** Are we initialized? */
    private boolean gotInfos;

    /** The ComponentManager needed for streaming */
    private ComponentManager manager;

    /**
     * Construct a new object
     * @param parameters This is optional
     */
    public URLSource(URL url,
                     SourceParameters parameters)
    throws IOException {
        this.systemId = url.toExternalForm();
        this.isFile = systemId.startsWith(FILE);
        this.url = url;
        this.gotInfos = false;
    }

    public void compose(ComponentManager manager )
    {
        this.manager = manager;
    }

    /**
     * Get the last modification date and content length of the source.
     * Any exceptions are ignored.
     */
    private void getInfos() {
        if (this.gotInfos == false) {
            if (this.isFile == true) {
                File file = new File(this.systemId.substring(FILE.length()));
                this.lastModificationDate = file.lastModified();
                this.contentLength = file.length();
            } else {
                try {
                    if (this.connection == null) {
                        this.connection = this.url.openConnection();
                        String userInfo = this.getUserInfo();
                        if (this.url.getProtocol().startsWith("http") == true && userInfo != null) {
                            this.connection.setRequestProperty("Authorization","Basic "+SourceUtil.encodeBASE64(userInfo));
                        }
                    }
                    this.lastModificationDate = this.connection.getLastModified();
                    this.contentLength = this.connection.getContentLength();
                } catch (IOException ignore) {
                    this.lastModificationDate = 0;
                    this.contentLength = -1;
                }
            }
            this.gotInfos = true;
        }
    }

    /**
     * Get the last modification date of the source or 0 if it
     * is not possible to determine the date.
     */
    public long getLastModified() {
        this.getInfos();
        return this.lastModificationDate;
    }

    /**
     *  Get the corresponding Resource object for monitoring.
     */
    public Resource getResource()
    throws Exception {
        this.getInfos();
        if (this.isFile == true) {
            return new FileResource(this.systemId.substring(FILE.length()));
        } else {
            return new SourceResource(this);
        }
    }

    /**
     * Get the content length of the source or -1 if it
     * is not possible to determine the length.
     */
    public long getContentLength() {
        this.getInfos();
        return this.contentLength;
    }

    /**
     * Return an <code>InputStream</code> object to read from the source.
     *
     * @throws ResourceNotFoundException if file not found or
     *         HTTP location does not exist.
     * @throws IOException if I/O error occured.
     */
    public InputStream getInputStream()
    throws IOException {
        this.getInfos();
        InputStream input = null;
        if (this.isFile == true) {
            input = new FileInputStream(this.systemId.substring(FILE.length()));
        } else {
            if (this.connection == null) {
                this.connection = this.url.openConnection();
                /* The following requires a jdk 1.3 */
                String userInfo = this.getUserInfo();
                if (this.url.getProtocol().startsWith("http") == true && userInfo != null) {
                    this.connection.setRequestProperty("Authorization","Basic "+SourceUtil.encodeBASE64(userInfo));
                }
            }

            input = this.connection.getInputStream();
            this.connection = null; // make sure a new connection is created next time
        }
        return input;
    }

    private static boolean checkedURLClass = false;
    private static boolean urlSupportsGetUserInfo = false;
    private static Method  urlGetUserInfo = null;
    private static Object[] emptyParams = new Object[0];

    /**
     * Check if the <code>URL</code> class supports the getUserInfo()
     * method which is introduced in jdk 1.3
     */
    private String getUserInfo() {
        if (URLSource.checkedURLClass == true) {
            if (URLSource.urlSupportsGetUserInfo == true) {
                try {
                    return (String) URLSource.urlGetUserInfo.invoke(this.url, URLSource.emptyParams);
                } catch (Exception e){
                    // ignore this anyway
                }
            }
            return null;
        } else {
            // test if the url class supports the getUserInfo method
            try {
                URLSource.urlGetUserInfo = URL.class.getMethod("getUserInfo", null);
                String ui = (String)URLSource.urlGetUserInfo.invoke(this.url, URLSource.emptyParams);
                URLSource.checkedURLClass = true;
                URLSource.urlSupportsGetUserInfo = true;
                return ui;
            } catch (Exception e){
            }
            URLSource.checkedURLClass = true;
            URLSource.urlSupportsGetUserInfo = false;
            URLSource.urlGetUserInfo = null;
            return null;
        }
    }

    /**
     * Return the unique identifer for this source
     */
    public String getSystemId() {
        return this.systemId;
    }

    /**
     *  Get the Validity object. This can either wrap the last modification
     *  date or the expires information or...
     *  If it is currently not possible to calculate such an information
     *  <code>null</code> is returned.
     */
    public SourceValidity getValidity() {
        final long lm = this.getLastModified();
        if (lm == -1) {
            return null;
        } else {
            return new TimeStampValidity(lm);
        }
    }

    /**
     * Refresh this object and update the last modified date
     * and content length.
     */
    public void discardValidity() {
        // reset connection
        this.connection = null;
        this.gotInfos = false;
    }

    /**
     * Return a new <code>InputSource</code> object
     *
     * @throws IOException if I/O error occured.
     */
    protected InputSource getInputSource()
    throws IOException
    {
        InputSource newObject = new InputSource(this.getInputStream());
        newObject.setSystemId(this.systemId);
        return newObject;
    }

    /**
     * Stream content to a content handler or to an XMLConsumer.
     *
     * @throws ResourceNotFoundException if file not found or
     *         HTTP location does not exist.
     * @throws SAXException if failed to parse source document.
     */
    public void toSAX(ContentHandler handler)
    throws SAXException
    {
        Parser parser = null;
        try {
            parser = (Parser)this.manager.lookup(Parser.ROLE);

            parser.parse(this.getInputSource(), handler);
        } catch (SAXException e) {
            // Preserve original exception
            throw e;
        } catch (Exception e){
            throw new SAXException("Exception during processing of "
                                          + this.systemId, e);
        } finally {
            if (parser != null) this.manager.release(parser);
        }
    }

}
