//
// File:        LanguageWriterForFortran.java
// Package:     gov.llnl.babel.backend.writers
// Revision:    @(#) $Id: LanguageWriterForFortran.java 4434 2005-03-17 17:05:29Z epperly $
// Description: Fortran language writer for backend code generation
//
// Copyright (c) 2000-2002, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
//
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
//
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.writers;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.writers.LanguageWriter;
import java.io.PrintWriter;

/**
 * Class <code>LanguageWriterForFortran</code> extends the generic language
 * writer to provide additional capabilities for writing Fortran code,
 * such as Fortran comment blocks.
 */
public class LanguageWriterForFortran extends LanguageWriter
                                      implements CodeConstants
{
   /**
    * Create a <code>LanguageWriterForFortran</code> instance that will send
    * output to the specified print writer object.  Set up the output
    * properties of the language writer for Fortran. For F77, it sets the
    * first tab at eight, the tab spacing to three, and a line break at
    * column 72.
    */
   public LanguageWriterForFortran(PrintWriter writer) {
      super(writer);
      if ("f90".equals(BabelConfiguration.getInstance().getTargetLanguage())) {
        enableLineBreak(78, null, " &");
        defineBlockComment("", C_COMMENT_F90);
      } else {	/* F77 assumed */
        setFirstTabStop(8);
        setTabSpacing(3);
        enableLineBreak(72, "     &", null);
        defineBlockComment(C_COMMENT_F77, null);
        setStrictBreaking();
      }
   }

   /**
    * Begin a block comment.  The comment block is enabled and a blank line
    * is printed.  There is no difference between documentation comments and
    * regular comments in Fortran.
    */
   public void beginBlockComment(boolean is_doc) {
      enableBlockComment();
      println();
   }

   /**
    * End a block comment.  A line is printed to close the comment.
    * An empty line is printed after the comment.  There is no difference
    * between documentation commends and regular comments in Fortran.
    */
   public void endBlockComment(boolean is_doc) {
      println();
      disableBlockComment();
      println();
   }

   /**
    * Output a single-line comment to the pretty writer stream.  The comment
    * should not contain any new line characters.  If the comment is null,
    * then nothing is output.
    */
   public void writeCommentLine(String comment) {
      if (comment != null) {
         enableBlockComment();
         println(comment);
         disableBlockComment();
      }
   }
}
