/*
 * $Id: libgnuvd.c,v 1.7 2002/01/10 00:02:55 binnema Exp $
 */

/*
** Copyright (C) 2001 Dirk-Jan C. Binnema <>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/

#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include <libgnuvd.h>


static const char* VANDALE_SERVER = "www.vandale.nl";
static const char* VANDALE_URI    = "http://www.vandale.nl";
static const int   VANDALE_PORT   = 80;

static char* HTTP_PROXY_HOST = NULL;
static int   HTTP_PROXY_PORT = 0;

/*
 * 'private' functions
 */
static u_int32_t   vd_resolve              (const char* dnsname, VDException *ex);
static int         vd_socket_to_site       (int ip, int port, VDException *ex);
static char*       vd_construct_http_query (char *search_term, VDException *ex);
static int	   vd_perform_http_query   (int sock, char *query, VDException *ex);
static char*	   vd_get_http_results     (int sock, size_t *bytes_read, VDException *ex);
static VDErrorType vd_filter_http_buffer   (Buffer *buffer);
static int         vd_parse_http_proxy     (char* http_proxy, char **host, int *port);

#if HAVE_HSTRERROR
/*OK*/
#else
const char* hstrerror (int err) { return (const char*)NULL; }
#endif /*HAVE_HSTRERROR*/

#ifndef VERSION
#define VERSION "?.?"
#endif /*VERSION*/


VDQuery*
vd_query_new (const char *word, VDException *ex)
{
	VDQuery *vd_query;

	assert (word);
	assert (ex);
	
	if (!ex)
		return NULL;
	
	if (!word) {
		ex->_err   = VD_ERR_PARAM;
		ex->_msg   = "word == NULL";
		ex->_owner = 0;
		return NULL;
	}
	
	
	if (getenv ("http_proxy")) {
		if (vd_parse_http_proxy (getenv ("http_proxy"), 
					  &HTTP_PROXY_HOST, 
					  &HTTP_PROXY_PORT) != 0) {
			ex->_err = VD_ERR_PROXY;
			ex->_msg = "error parsing http_proxy";
			ex->_owner = 0;
			return NULL;
		}
	}

	vd_query = (VDQuery*) malloc (sizeof(VDQuery));
	if (!vd_query) {
		ex->_err   = VD_ERR_MEM;
		ex->_msg   = "insufficient memory";
		ex->_owner = 0;
		return NULL;
	}
	
	vd_query->_search_term = strdup (word);
	if (!vd_query->_search_term) {
		free (vd_query);
		return NULL;
	}

	vd_query->_result_buffer = buffer_new ();
	if (!vd_query->_result_buffer) {
		free (vd_query);
		return NULL;
	}


	return vd_query;
}


void
vd_query_destroy (VDQuery* vd_query)
{
	assert (vd_query);
	
	free (vd_query->_search_term);
	buffer_destroy (vd_query->_result_buffer);

	free (HTTP_PROXY_HOST);
	HTTP_PROXY_HOST = NULL;

	free (vd_query);
}



const Buffer*
vd_query_results (const VDQuery *vd_query, VDException *ex)
{
	assert (vd_query);
	assert (ex);
	
	return vd_query->_result_buffer;
}



const char *
vd_query_search_term (const VDQuery *vd_query)
{
	assert (vd_query);
	return vd_query->_search_term;
}



VDErrorType
vd_query_perform (const VDQuery *vd_query, VDException *ex)
{
	int sock, ip, port;
	int err, bytes_read, retval = VD_ERR_OK;
	char *query, *result;
	char *server, *port_str;
	
	assert (vd_query);
	assert (ex);

	/* use a proxy? */
	if (HTTP_PROXY_HOST) {
		server = HTTP_PROXY_HOST;
		port   = HTTP_PROXY_PORT;
	} else {
		server = (char*)VANDALE_SERVER;
		port   = VANDALE_PORT;
	}

	ip = vd_resolve (server, ex);
	if (!ip) 
		return VD_ERR_CONNECT;
	
	sock = vd_socket_to_site (ip, port, ex);
	if (sock==-1)
		return VD_ERR_CONNECT;

	query = vd_construct_http_query (vd_query->_search_term, ex);
	if (!query) {
		retval = VD_ERR_MEM;
		goto cleanup;
	}
	
	err = vd_perform_http_query (sock, query, ex);
	if (err==-1) {
		retval= VD_ERR_HTTP;
		goto cleanup;
	}

	result = vd_get_http_results (sock, &bytes_read, ex);
	if (!result) 
		retval = VD_ERR_HTTP;
	else {
		
		buffer_manage_data (vd_query->_result_buffer, result, bytes_read);
		retval = vd_filter_http_buffer (vd_query->_result_buffer);
	}
	
 cleanup:
	if (query)
		free (query);
	
	shutdown (sock, 2);
	
	return retval;
}


static u_int32_t 
vd_resolve (const char* dnsname, VDException *ex)
{
	struct hostent *host;
	u_int32_t iaddr;
	
	host = gethostbyname (dnsname);
	if (!host) {
		ex->_err   = VD_ERR_DNS;
		ex->_msg   = (char*)hstrerror(h_errno);
		ex->_owner = 0;

		return 0;
	}

	iaddr = 0;
	iaddr += (u_char)host->h_addr_list[0][0] << 24;
	iaddr += (u_char)host->h_addr_list[0][1] << 16;
	iaddr += (u_char)host->h_addr_list[0][2] << 8;
	iaddr += (u_char)host->h_addr_list[0][3];

	return ntohl(iaddr);
}


static int
vd_socket_to_site (int ip, int port, VDException *ex)
{
	
	int sock;
	int conn;

	struct sockaddr_in dest_sa;

	sock = socket (AF_INET, SOCK_STREAM, 0);
	if (sock == -1) {
		ex->_err   = VD_ERR_SOCKET;
		ex->_msg   = strerror(errno);
		ex->_owner = 0;
		return -1;
	}

	bzero ((void*)&dest_sa, sizeof(struct sockaddr_in));
	       
	dest_sa.sin_family      = AF_INET;
	dest_sa.sin_port        = htons (port);
	dest_sa.sin_addr.s_addr = ip;
	
	conn = connect (sock,
			(struct sockaddr*)&dest_sa,
			sizeof(struct sockaddr));
	if (conn == -1) {
		ex->_err   = VD_ERR_CONNECT;
		ex->_msg   = strerror (errno);
		ex->_owner = 0;
		return -1;
	}
	
	return sock;
}




static char*
vd_construct_http_query (char *search_term, VDException *ex)
{
	static const char *request_template =
		"POST %s/cgi-bin/gx.cgi/AppLogic+FTContentServer?pagename="
		"VanDale/ZoekResultaat HTTP/1.1\r\n"
		"Referer: http://www.vandale.nl/cgi-bin/gx.cgi/"
		"AppLogic+FTContentServer?pagename=VanDale/Static/sessionwerk\r\n"
		"Cookie: gx_session_id_FutureTenseContentServer=7736441630508e96\r\n"
		"Host: www.vandale.nl\r\n"
		"User-Agent:  GNUVD/" VERSION "\r\n"
		"Accept: */*\r\nAccept-Language: nl,en\r\n"
		"Content-type: application/x-www-form-urlencoded\r\n"
        	"Content-Length: %d\r\n"
        	"\r\n"
        	"zoekwoord=%s\r\n";

	char *query, *server;
	int  query_size, result;

	/* use proxy? */
	if (HTTP_PROXY_HOST)
		server = (char*)VANDALE_URI;
	else
		server = "";


	query_size = strlen (request_template) + 
	       	     strlen (search_term) + 
	             strlen (server),
	             + 3 + 1;
	       
	query = (char*) malloc (query_size);
	if (!query) {
		ex->_err   = VD_ERR_MEM;
		ex->_msg   = "insufficient memory";
		ex->_owner = 0;
		return NULL;
	}
	
	result = snprintf (query, query_size, request_template,
			   server,
			   strlen ("zoekwoord=") + strlen(search_term), 
			   search_term);
	if (result == -1) {
		free (query);
		ex->_err   = VD_ERR_MEM;
		ex->_msg   = "string too long";
		ex->_owner = 0;
		return NULL;
	} else 
		return query;
}




static int
vd_perform_http_query (int sock, char *query, VDException *ex)
{
	return write (sock, query, strlen (query));
}



static char*
vd_get_http_results (int sock, size_t *bytes_read, VDException *ex)
{
	const int max_buf    = 100000;
	const int chunk_size = 1000;

	char *buf;
	int result, offset;

	buf = (char*) malloc (max_buf);
	if (!buf) {
		ex->_err   = VD_ERR_MEM;
		ex->_msg   = "insufficient memory";
		ex->_owner = 0; 
		return NULL;
	}

	*bytes_read = 0;
	while (*bytes_read < max_buf) {
		
		result = read (sock, buf + *bytes_read, chunk_size);
		if (result == -1 || result == 0)
			break;
		
		*bytes_read += result;
	}
			
	if (-1 == result) {
		
		free (buf);
		ex->_err   = VD_ERR_READ;
		ex->_msg   = strerror(errno);
		ex->_owner = 0;
		
		return NULL;
	}

	return buf;
}


static VDErrorType
vd_filter_http_buffer (Buffer *http_buffer)
{
	const size_t min_length = 100;
	
	static const char* resultaat = "RESULTAAT";
	static const char* opnieuw   = "Opnieuw/verfijnd zoeken";
	int pos;
	char *data;
	BufferIterator iter;
	
	pos = buffer_find (http_buffer, resultaat, strlen(resultaat));
	if (pos != buffer_end (http_buffer)) {
		
		while (pos < buffer_length (http_buffer) && http_buffer->_data[pos] != '\n')
			++pos;
		
		if (pos == buffer_length (http_buffer))
			return;
		else	
			buffer_erase (http_buffer, 0, pos);
	} else 
		/* nothing was found */
		return VD_ERR_NOT_FOUND;
		
	pos = buffer_find (http_buffer, opnieuw, strlen(opnieuw));
	if (pos != buffer_end(http_buffer))
		buffer_erase (http_buffer, pos, buffer_length(http_buffer) - pos - 1);

	/* strip extra space */
	pos = buffer_length (http_buffer) - 1;
	if (pos <= 0)
		return VD_ERR_NOT_FOUND;
	data = (char*) buffer_data (http_buffer);
	
	for (--pos; pos; --pos) 
		if (!isspace(data[pos]))
			break;

	buffer_erase (http_buffer, pos + 1, 
		      buffer_length (http_buffer) - pos - 2);
	
	if (buffer_length (http_buffer) < min_length)
		return VD_ERR_NOT_FOUND;
	else
		return VD_ERR_OK;
} 



VDException*
vd_exception_new (void)
{
	VDException *vd_exception;

	vd_exception = (VDException*) malloc (sizeof(VDException));
	if (!vd_exception)
		return NULL;
	
	vd_exception->_err    = VD_ERR_OK;
	vd_exception->_msg    = "no error";
	vd_exception->_owner  = 0;

	return vd_exception;
}


void
vd_exception_destroy (VDException *vd_exception)
{
	assert (vd_exception);
	
	if (vd_exception->_owner)
		free (vd_exception->_msg);

	free (vd_exception);
}





static int 
vd_parse_http_proxy (char* http_proxy, char **host, int *port)
{
	int i;
	int len = strlen (http_proxy);

	
	/* string too short */
	if (!len)
		return 1;

	/* ignore the closing '/', if any */
	if (http_proxy[len - 1] == '/')
		--len;

	for (i = len - 1; i ; --i) 
		if (http_proxy[i] == ':') {
			*port = strtol (&http_proxy[i+1], NULL, 10);
			break;
		}

	/* port not found, or no space left for host */
	if (i == 0)
		return 1;
	
	/* rest of string is host */
	if (i >= 7 && strncmp (http_proxy, "http://", 7) == 0) 
		*host = (char*) strndup (http_proxy + 7, i - 7);
	else 
		*host = (char*) strndup (http_proxy, i);
	
	if (!*host)
		return 1; /* mem alloc failure */			 
	else
		return 0;
}
