/*************************************************************************
* Bulgarian-English Dictionary
* Copyright (C) 2000, 2001  Radostin Radnev <radnev@yahoo.com>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <qwidget.h>
#include <qmainwindow.h>
#include <qdialog.h>
#include <qfont.h>
#include <qclipboard.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qkeycode.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qtextview.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qstring.h>
#include <qmotifstyle.h>
#include <qcdestyle.h>
#include <qwindowsstyle.h>
#include <qplatinumstyle.h>

#ifdef WITH_KDE
#include <kapp.h>
#include <kmenubar.h>
//#include <kfontdialog.h>
// To the moment KFontDialog doesn't provide way to select CharSet
#include <qfontdialog.h>
#else
#include <qapplication.h>
#include <qfontdialog.h>
#include <qmenubar.h>
#endif

#include "config.h"

#include "encodechar.h"

#include "registry.h"
#include "history.h"
#include "translator.h"
#include "win.h"
#include "test.h"

// Include Icon and Image of dic
#include "kbedicicon.xpm"
#include "kbediciconmini.xpm"

// Include config.h autoconfiguration generated file
#include "config.h"



//=== Class Win ==========================================================
// Written by Radostin Radnev - radnev@yahoo.com
// $Id: win.cpp,v 1.12 2001/03/12 05:18:32 radnev Exp $
//
// This is main window class - GUI
//========================================================================



// Some Constants
static const int  MAX_WORD_LEN           = 70;
static const int  MAX_WORDS_ALLOWED      = 7;
#ifdef ENG_BUL_DAT_CONFIG
static const char ENG_BUL_DAT[]          = ENG_BUL_DAT_CONFIG;
#else
static const char ENG_BUL_DAT[]          = "/usr/local/share/engbul.dat";
#endif
#ifdef BUL_ENG_DAT_CONFIG
static const char BUL_ENG_DAT[]          = BUL_ENG_DAT_CONFIG;
#else
static const char BUL_ENG_DAT[]          = "/usr/local/share/buleng.dat";
#endif
static const char REGISTRY_FILE[]        = ".kbedic";
static const char REGISTRY_FILE_SUFFIX[] = "~";
static const char ENV_HOME[]             = "HOME";
static const char CAPTION[]              = "kbe Dictionary";
static const char FOUND[]                = "Found";
static const char NOT_FOUND[]            = "";
static const char ADV_SRCH_LABEL[]       = "%d words   ";
static const int  ADDITIONAL_WIDTH       = 25;
static const int  ADDITIONAL_HEIGHT      = 10;


// Messages
static const char HELP_MESSAGE[] =
	"To perform <b>Advanced Search</b> click button <b>'?'</b>. "
	"It is a very powerful function that searchs entered word(s) in data.<br><br>"
	"<b>Advanced Search</b> accepts as control chars '<b>+</b>', '<b>-</b>' and '<b>\"</b>'.<br>"
	"<b>+go +in</b> means <b>go</b> AND <b>in</b><br>"
	"<b>\"go in\"</b> means exact phrase <b>go in</b><br>"
	"<b>+go +in -out</b> means <b>go</b> AND <b>in</b> AND NO <b>out</b><br><br>"
	"You can omit control chars. In this case options <b>Search for exact phrase</b> "
	" in <b>Main Menu/Tools/Advanced Search</b> is important.<br>"
	"<b>go in</b> means <b>\"go in\"</b> when option is checked.<br>"
	"<b>go in</b> means <b>+go +in</b> when option is unchecked.<br>"
	"But be careful when you make complicated queries because.<br>"
	"<b>go in +go out</b> means <b>+\"go in\" +\"go out\"</b> in first case.<br>"
	"<b>go in +go out</b> means <b>+go +in +go +out</b> in second case.<br><br>"
	"Another important options is <b>Search for whole word(s)</b> in "
	"<b>Main Menu/Tools/Advanced Search</b>.<br>"
	"<b>go</b> means whole/exact word <b>go</b> when option is checked.<br>"
	"<b>go</b> means also <b>go</b>al, car<b>go</b> ... when option is unchecked.<br><br>"
	"For more information and help please look at <b>README</b> and <b>FAQ</b> files "
	"that come with the distribution package.<br>";

static const char ABOUT_MESSAGE[] =
	"kbe Dictionary version 2.1\n"
	"Copyright (C) 2000 Radostin Radnev\n"
	"\n"
	"Home page: http://kbedic.sourceforge.net\n"
	"\n"
	"kbe Dictionary comes with ABSOLUTELY\n"
	"NO WARRANTY. This is free software, and\n"
	"you are welcome to redistribute it under\n"
	"certain conditions. For details read file\n"
	"COPYING that comes with this package.";


// Default options
static const int  DEFAULT_INPUT_FIELD_WIDTH        = 200;
static const int  DEFAULT_LEFT_PANE_WIDTH          = 150;
static const int  DEFAULT_WORDS_IN_LIST            = 250;
static const char DEFAULT_WORDS_IN_HISTORY         = 50;
static const int  DEFAULT_BORDER_WIDTH             = 5;
static const int  DEFAULT_SYSTEM_STYLE             = 1;
static const bool DEFAULT_TRANSLATE_ON_EVERY_KEY   = true;
static const bool DEFAULT_SEPARATE_MEANINGS        = true;
static const bool DEFAULT_LATIN_OUTPUT             = false;
static const bool DEFAULT_LATIN_INPUT              = false;
static const char DEFAULT_FONT_NAME[]              = "*";
static const int  DEFAULT_FONT_SIZE                = 14;
static const int  DEFAULT_FONT_WEIGHT              = 50;
static const bool DEFAULT_FONT_ITALIC              = false;
static const int  DEFAULT_FONT_CHARSET             = QFont::Latin1;
static const char DEFAULT_GEOMETRY[]               = "50, 50, 500, 300";
static const bool DEFAULT_ADV_SRCH_HIGHLIGHT       = true;
static const bool DEFAULT_ADV_SRCH_WHOLE_WORD      = true;
static const bool DEFAULT_ADV_SRCH_EXACT_PHRASE    = false;
static const bool DEFAULT_ADV_SRCH_ASD_IGNORE_KEY  = true;
static const bool DEFAULT_ADV_SRCH_ASD_IGNORE_HIS  = true;
static const bool DEFAULT_ADV_SRCH_ASD_ENTER       = true;
static const bool DEFAULT_ADV_SRCH_ASD_DO_NOT_REPL = true;


// Min & Max options
static const int  MAX_INPUT_FIELD_WIDTH = 500;
static const int  MIN_INPUT_FIELD_WIDTH = 25;
static const int  MAX_LEFT_PANE_WIDTH   = 500;
static const int  MIN_LEFT_PANE_WIDTH   = 0;
static const int  MAX_WORDS_IN_LIST     = 500;
static const int  MIN_WORDS_IN_LIST     = 0;
static const int  MAX_WORDS_IN_HISTORY  = 200;
static const int  MIN_WORDS_IN_HISTORY  = 0;
static const int  MAX_BORDER_WIDTH      = 25;
static const int  MIN_BORDER_WIDTH      = 0;
static const int  MAX_SYSTEM_STYLE      = 4;
static const int  MIN_SYSTEM_STYLE      = 1;


// Strings in registry
static const char OPTION_ENG_BUL_DAT[]              = "EngBulDat";
static const char OPTION_BUL_ENG_DAT[]              = "BulEngDat";
static const char OPTION_INPUT_FIELD_WIDTH[]        = "InputFieldWidth";
static const char OPTION_LEFT_PANE_WIDTH[]          = "LeftPaneWidth";
static const char OPTION_WORDS_IN_LIST[]            = "WordsInList";
static const char OPTION_WORDS_IN_HISTORY[]         = "WordsInHistory";
static const char OPTION_BORDER_WIDTH[]             = "BorderWidth";
static const char OPTION_SYSTEM_STYLE[]             = "SystemStyle";
static const char OPTION_TRANSLATE_ON_EVERY_KEY[]   = "TraslateOnEveryKey";
static const char OPTION_SEPARATE_MEANINGS[]        = "SeparateMeanings";
static const char OPTION_LATIN_OUTPUT[]             = "LatinOutput";
static const char OPTION_LATIN_INPUT[]              = "LatinInput";
static const char OPTION_FONT_NAME[]                = "FontName";
static const char OPTION_FONT_SIZE[]                = "FontSize";
static const char OPTION_FONT_WEIGHT[]              = "FontWeight";
static const char OPTION_FONT_ITALIC[]              = "FontItalic";
static const char OPTION_FONT_CHARSET[]             = "FontCharset";
static const char OPTION_GEOMETRY[]                 = "Geometry";
static const char OPTION_ADV_SRCH_HIGHLIGHT[]       = "AdvSrch.Highlight";
static const char OPTION_ADV_SRCH_WHOLE_WORD[]      = "AdvSrch.WholeWord";
static const char OPTION_ADV_SRCH_EXACT_PHRASE[]    = "AdvSrch.EaxctPhrase";
static const char OPTION_ADV_SRCH_ASD_IGNORE_KEY[]  = "AdvSrch.IgnoreOnEveryKey";
static const char OPTION_ADV_SRCH_ASD_IGNORE_HIS[]  = "AdvSrch.IgnoreFromHistory";
static const char OPTION_ADV_SRCH_ASD_ENTER[]       = "AdvSrch.EnterSearch";
static const char OPTION_ADV_SRCH_ASD_DO_NOT_REPL[] = "AdvSrch.DoNotReplacePhrase";



//=== Constructor ========================================================
// Create GUI
// Init objects and set default variables
//========================================================================
Win::Win(QWidget *parent = 0, const char *name = 0):QMainWindow(parent, name) {

	// Init some variables
	fromSystemHighlighted = false;
	fromListHighlighted = false;
	fromHistoryBuffer = false;
	fromListSelected = false;
	fromClipboard = false;
	fromSystem = false;
	fromTextChanged = false;
	fromAdvancedSearch = false;

	// Create registry object
	reg = new Registry(getenv(ENV_HOME), REGISTRY_FILE, REGISTRY_FILE_SUFFIX);

	// Create Dictionary object
	dictionary = new Translator(MAX_WORD_LEN, MAX_WORDS_ALLOWED);

	// Get English Bulgarian Dictionary
	dataOk = true;
	dataOk = ((dataOk) && (dictionary->createDictionary(reg->getString(OPTION_ENG_BUL_DAT, ENG_BUL_DAT), dictionary->ENG_BUL)));
	dataOk = ((dataOk) && (dictionary->createDictionary(reg->getString(OPTION_BUL_ENG_DAT, BUL_ENG_DAT), dictionary->BUL_ENG)));

	// Read data from ini file and set it
	inputFieldWidth = reg->getInt(OPTION_INPUT_FIELD_WIDTH, DEFAULT_INPUT_FIELD_WIDTH, MIN_INPUT_FIELD_WIDTH, MAX_INPUT_FIELD_WIDTH);
	leftPaneWidth = reg->getInt(OPTION_LEFT_PANE_WIDTH, DEFAULT_LEFT_PANE_WIDTH, MIN_LEFT_PANE_WIDTH, MAX_LEFT_PANE_WIDTH);
	wordsInList = reg->getInt(OPTION_WORDS_IN_LIST, DEFAULT_WORDS_IN_LIST, MIN_WORDS_IN_LIST, MAX_WORDS_IN_LIST);
	borderWidth = reg->getInt(OPTION_BORDER_WIDTH, DEFAULT_BORDER_WIDTH, MIN_BORDER_WIDTH, MAX_BORDER_WIDTH);
	systemStyle = reg->getInt(OPTION_SYSTEM_STYLE, DEFAULT_SYSTEM_STYLE, MIN_SYSTEM_STYLE, MAX_SYSTEM_STYLE);
	wordsInHistory = reg->getInt(OPTION_WORDS_IN_HISTORY, DEFAULT_WORDS_IN_HISTORY, MIN_WORDS_IN_HISTORY, MAX_WORDS_IN_HISTORY);
	translateOnEveryKey = reg->getBool(OPTION_TRANSLATE_ON_EVERY_KEY, DEFAULT_TRANSLATE_ON_EVERY_KEY);
	advSrchIgnoreKey = reg->getBool(OPTION_ADV_SRCH_ASD_IGNORE_KEY, DEFAULT_ADV_SRCH_ASD_IGNORE_KEY);
	advSrchIgnoreHistory = reg->getBool(OPTION_ADV_SRCH_ASD_IGNORE_HIS, DEFAULT_ADV_SRCH_ASD_IGNORE_HIS);
	advSrchEnterSearch = reg->getBool(OPTION_ADV_SRCH_ASD_ENTER, DEFAULT_ADV_SRCH_ASD_ENTER);
	advSrchDoNotReplacePhrase = reg->getBool(OPTION_ADV_SRCH_ASD_DO_NOT_REPL, DEFAULT_ADV_SRCH_ASD_DO_NOT_REPL);
	dictionary->separateMeanings = reg->getBool(OPTION_SEPARATE_MEANINGS, DEFAULT_SEPARATE_MEANINGS);
	dictionary->latinOutput = reg->getBool(OPTION_LATIN_OUTPUT, DEFAULT_LATIN_OUTPUT);
	dictionary->latinInput = reg->getBool(OPTION_LATIN_INPUT, DEFAULT_LATIN_INPUT);
	dictionary->advancedSearchWholeWord = reg->getBool(OPTION_ADV_SRCH_WHOLE_WORD, DEFAULT_ADV_SRCH_WHOLE_WORD);
	dictionary->advancedSearchExactPhrase = reg->getBool(OPTION_ADV_SRCH_EXACT_PHRASE, DEFAULT_ADV_SRCH_EXACT_PHRASE);
	dictionary->advancedSearchHighlight = reg->getBool(OPTION_ADV_SRCH_HIGHLIGHT, DEFAULT_ADV_SRCH_HIGHLIGHT);
	dictionary->boldDecoration = true;
	dictionary->htmlOutput = true;

	// Create history object
	history = new History(wordsInHistory, MAX_WORD_LEN);

	// Set font, read from file settings and set it
	font = QFont(
		W2U(reg->getString(OPTION_FONT_NAME, DEFAULT_FONT_NAME)),
		reg->getInt(OPTION_FONT_SIZE, DEFAULT_FONT_SIZE),
		reg->getInt(OPTION_FONT_WEIGHT, DEFAULT_FONT_WEIGHT),
		reg->getBool(OPTION_FONT_ITALIC, DEFAULT_FONT_ITALIC),
		(QFont::CharSet)reg->getInt(OPTION_FONT_CHARSET, DEFAULT_FONT_CHARSET));

	// Set caption of window
	setCaption(W2U(CAPTION));
	// Set icon of window
	QPixmap p1(diciconmini);
	setIcon(p1);

	// Set positions and size of window
	int X = 0, Y = 0, W = 0, H = 0;
	sscanf(DEFAULT_GEOMETRY, "%d, %d, %d, %d", &X, &Y, &W, &H);
	sscanf(reg->getString(OPTION_GEOMETRY, DEFAULT_GEOMETRY), "%d, %d, %d, %d", &X, &Y, &W, &H);
	setGeometry(X, Y, W, H);

	// Create and set menu
	QPopupMenu *file = new QPopupMenu(this);
	CHECK_PTR(file);
	file->insertItem(W2U("E&xit"), qApp, SLOT(quit()), CTRL + Key_Q);

	QPopupMenu *edit = new QPopupMenu(this);
	CHECK_PTR(edit);
	edit->insertItem(W2U("&Copy"), this, SLOT(menu_Copy()), CTRL + Key_C);
	edit->insertItem(W2U("&Paste"), this, SLOT(menu_Paste()), CTRL + Key_V);
	edit->insertSeparator();
	edit->insertItem(W2U("&Select All"), this, SLOT(menu_SelectAll()), CTRL + Key_A);
	edit->insertItem(W2U("C&lear"), this, SLOT(menu_Clear()), CTRL + Key_D);
	edit->insertSeparator();
	edit->insertItem(W2U("&Advanced Search"), this, SLOT(button_Search()), CTRL + Key_S);
	edit->insertSeparator();
	edit->insertItem(W2U("History &Back"), this, SLOT(button_Prev()), CTRL + Key_Left);
	edit->insertItem(W2U("History &Forward"), this, SLOT(button_Next()), CTRL + Key_Right);

	QPopupMenu *options = new QPopupMenu(this);
	CHECK_PTR(options);
	options->insertItem(W2U("&Font..."), this, SLOT(menu_Font()));
	options->insertItem(W2U("&Options..."), this, SLOT(menu_Options()));
	options->insertItem(W2U("&Advanced Search..."), this, SLOT(menu_AdvancedSearch()));
	options->insertSeparator();
	options->insertItem(W2U("&Test..."), this, SLOT(menu_Test()));

	QPopupMenu *help = new QPopupMenu(this);
	CHECK_PTR(help);
	help->insertItem(W2U("&Help"), this, SLOT(menu_Help()), Key_F1);
	help->insertSeparator();
	help->insertItem(W2U("&About..."), this, SLOT(menu_About()));

#ifdef WITH_KDE
	KMenuBar *menu = new KMenuBar(this);
#else
	QMenuBar *menu = new QMenuBar(this);
#endif

	menu->insertItem(W2U("&File"), file);
	menu->insertItem(W2U("&Edit"), edit);
	menu->insertItem(W2U("&Tools"), options);
	menu->insertItem(W2U("&Help"), help);

	// Create and set Word (text field)
	word = new QLineEdit(W2U(""), this);
	CHECK_PTR(word);
	word->setMaxLength(MAX_WORD_LEN);
	connect(word, SIGNAL(returnPressed()), this, SLOT(word_EnterPressed()));
	connect(word, SIGNAL(textChanged(const QString&)), this, SLOT(word_TextChanged(const QString&)));

	// Create and set button advanced search
	bsearch = new QPushButton(W2U("?"), this);
	CHECK_PTR(bsearch);
	bsearch->setFocusPolicy(NoFocus);
	connect(bsearch, SIGNAL(clicked()), this, SLOT(button_Search()));

	// Create and set button previous
	bprev = new QPushButton(W2U("<"), this);
	CHECK_PTR(bprev);
	bprev->setFocusPolicy(NoFocus);
	bprev->setEnabled(false);
	connect(bprev, SIGNAL(clicked()), this, SLOT(button_Prev()));

	// Create and set button next
	bnext = new QPushButton(W2U(">"), this);
	CHECK_PTR(bnext);
	bnext->setFocusPolicy(NoFocus);
	bnext->setEnabled(false);
	connect(bnext, SIGNAL(clicked()), this, SLOT(button_Next()));

	// Create and set label found
	found = new QLabel(this);
	CHECK_PTR(found);
	found->setAlignment(QLabel::AlignLeft | QLabel::AlignVCenter);

	// Create and set picture
	pic = new QLabel(this);
	CHECK_PTR(pic);
	QPixmap p2(dicicon);
	pic->setPixmap(p2);
	pic->setFixedSize(p2.width(), p2.height());

	// Create and set list (of words)
	list = new QListBox(this);
	CHECK_PTR(list);
	list->setHScrollBarMode(QScrollView::AlwaysOff );
	connect(list, SIGNAL(highlighted(const QString&)), this, SLOT(list_Highlighted(const QString&)));
	connect(list, SIGNAL(selected(const QString&)), this, SLOT(list_Selected(const QString&)));

	// Create and set translation result
	result = new QTextView(this);
	CHECK_PTR(result);

	// Get application clipboard
#ifdef WITH_KDE
	systemClipboard = KApplication::clipboard();
#else
	systemClipboard = QApplication::clipboard();
#endif

	// Set size, positions and font of widgets
	setView(systemStyle);

	// Set focus?
	word->setFocus();

} // End of Constructor



//=== Destructor =========================================================
// Save geometry in registry file and delete registry, dictionary
// and other objects
//========================================================================
Win::~Win() {
	if (dataOk) {
		if ((dataOk) && (!isMinimized())) {
			char val[32];
			sprintf(val, "%d, %d, %d, %d", x(), y(), width(), height());
			reg->setString(OPTION_GEOMETRY, val);
		}
		delete dictionary;
		delete history;
		delete word;
		delete bprev;
		delete bnext;
		delete bsearch;
		delete found;
		delete pic;
		delete list;
		delete result;
	}
	delete reg;
} // End of Destructor



//=== Get Data Ok ========================================================
// Return true if found dictionary files
// If failed to found them return false
//========================================================================
bool Win::getDataOk() {
	return dataOk;
} // End of getDataOk



//=== Button Prev ========================================================
// Invoked from click at previous button
//========================================================================
void Win::button_Prev() {
	if (history->isNew()) {
		history->prev();
		if (strcmp(history->getCurrentItem(), U2W(word->text())) == 0) {
			history->prev();
		}
	}
	else {
		if ((history->isFirst()) && (!bnext->isEnabled())) {
			if (strcmp(history->getCurrentItem(), U2W(word->text())) == 0) {
				history->prev();
			}
		}
		else {
			history->prev();
		}
	}
	if ((!bnext->isEnabled()) && (!history->isFirst())) {
		bnext->setEnabled(true);
	}
	if ((history->isLast()) && (history->getCurrentSize() > 1)) {
		bprev->setEnabled(false);
	}
	fromHistoryBuffer = true;
	word->setFocus();
	word->setText(W2U(history->getCurrentItem()));
	word->selectAll();
} // End of button_Prev



//=== Button Next ========================================================
// Invoked from click at next button
//========================================================================
void Win::button_Next() {
	if (!bprev->isEnabled()) {
		bprev->setEnabled(true);
	}
	history->next();
	if (history->isFirst()) {
		bnext->setEnabled(false);
	}
	fromHistoryBuffer = true;
	word->setFocus();
	word->setText(W2U(history->getCurrentItem()));
	word->selectAll();
} // End of button_Next



//=== Word Text Changed ==================================================
// Invoked from text changed event in input field
//========================================================================
void Win::word_TextChanged(const QString &newText) {
	if (fromListHighlighted) {
		fromListHighlighted = false;
		return;
	}
	if ((newText[0] == '+') || (newText[0] == '"') || (newText[0] == '-')) {
		if ((!fromListSelected) && (!fromClipboard) && (!fromHistoryBuffer) && (advSrchIgnoreKey)) {
			return;
		}
		if ((fromHistoryBuffer) && (advSrchIgnoreHistory)) {
			fromHistoryBuffer = false;
			return;
		}
	}
	if ((translateOnEveryKey) || (fromListSelected) || (fromHistoryBuffer) || (fromClipboard)) {
		fromTextChanged = true;
		word_EnterPressed();
	}
} // End of word_TextChanged



//=== Word Enter Pressed =================================================
// Invoked from enter pressed event in input field
//========================================================================
void Win::word_EnterPressed() {
	char *Translation = NULL;
	if ((!fromTextChanged) && (advSrchEnterSearch) && ((word->text()[0] == '+') || (word->text()[0] == '"') || (word->text()[0] == '-'))) {
		button_Search();
		return;
	}
	if (dictionary->advancedSearchState) {
		dictionary->advancedSearchState = false;
	}
	if (dictionary->findWord(U2W(word->text()), &Translation)) {
		found->setText(W2U(FOUND));
	}
	else {
		found->setText(W2U(NOT_FOUND));
	}
	if (Translation[0] != '\0') {
		if ((strcmp(Translation, U2W(result->text())) != 0) || (fromListSelected) || (fromSystem)) {
			result->setText(W2U(Translation));
			if (list->isEnabled()) {
				int i = 0;
				list->clear();
				do {
					i++;
					list->insertItem(W2U(dictionary->getWord()));
				}
				while ((dictionary->goToNextWord()) && (i < wordsInList));
				fromSystemHighlighted = true;
				fromAdvancedSearch = false;
				list->setTopItem(0);
				list->setCurrentItem(0);
			}
		}
		if ((fromListSelected) || (fromHistoryBuffer) || (fromClipboard) || (fromSystem) || (!fromTextChanged)) {
			word->selectAll();
			fromListSelected = false;
			fromHistoryBuffer = false;
			if (fromSystem) {
				fromSystem = false;
				fromTextChanged = true;
			}
		}
		if ((!fromTextChanged) || (fromClipboard)) {
			addWordToHistory(U2W(word->text()));
			fromClipboard = false;
		}
	}
	else {
		fromSystem = false;
	}
	fromTextChanged = false;
} // End of word_EnterPressed



//=== Advanced Search ====================================================
// Invoked from Advanced Search Button "?"
//========================================================================
void Win::button_Search() {
	word->setFocus();
	word->selectAll();
	if (dictionary->setAdvancedSearchText(U2W(word->text()))) {
		addWordToHistory(U2W(word->text()));
		int i = 0;
		list->clear();
		while ((dictionary->searchNextWord()) && (i < wordsInList)) {
			i++;
			list->insertItem(W2U(dictionary->getWord()));
		}
		if (i > 0) {
			fromSystemHighlighted = false;
			fromAdvancedSearch = true;
			list->setTopItem(0);
			list->setCurrentItem(0);
		}
		else {
			result->setText(W2U(""));
		}
		char b[strlen(ADV_SRCH_LABEL) + 5];
		sprintf(b, ADV_SRCH_LABEL, i);
		found->setText(W2U(b));
	}
} // End of button_Search



//=== List Selected ======================================================
// Invoked from list selected event
// When press enter in list box
//========================================================================
void Win::list_Selected(const QString &newText) {
	fromListSelected = true;
	if (word->text() != newText) {
		word->setText(newText);
	}
	else {
		word_EnterPressed();
	}
} // End of list_Selected



//=== List Highlighted ===================================================
// Invoked from list highlight event
// When select new element
//========================================================================
void Win::list_Highlighted(const QString &newText) {
	char *Translation = NULL;
	if (fromSystemHighlighted) {
		fromSystemHighlighted = false;
	}
	else {
		dictionary->findWord(U2W(newText), &Translation);
		result->setText(W2U(Translation));
		if (fromAdvancedSearch) {
			fromAdvancedSearch = false;
		}
		else if ((!dictionary->advancedSearchState) || (!advSrchDoNotReplacePhrase)) {
			fromListHighlighted = true;
			word->setText(newText);
			word->selectAll();
		}
	}
} // End of list_Highlighted



//=== Add Word To History ================================================
// Adds word to history object
//========================================================================
void Win::addWordToHistory(const char *word) {
	if (wordsInHistory > 0) {
		history->addItem(word);
		bnext->setEnabled(false);
		bprev->setEnabled(true);
	}
} // End of addWordToHistory



//=== Menu Copy ==========================================================
// Perform copy operation on input field or translation result
//========================================================================
void Win::menu_Copy() {
	if (result->hasSelectedText()) {
		result->copy();
	}
	else if (word->hasMarkedText()) {
		word->copy();
	}
} // End of menu_Copy



//=== Menu Paste =========================================================
// Perform paste operation on input field
//========================================================================
void Win::menu_Paste() {
	fromClipboard = true;
	word->setFocus();
	word->clear();
	word->paste();
} // End of menu_Paste



//=== Menu Clear =========================================================
// Clear input field
//========================================================================
void Win::menu_Clear() {
	word->setFocus();
	word->clear();
} // End of menu_Clear



//=== Menu SelectAll =====================================================
// Select All word field
//========================================================================
void Win::menu_SelectAll() {
	if (word->hasFocus()) {
		word->selectAll();
	}
	else if (result->hasFocus()) {
		word->deselect();
		result->selectAll();
	}
} // End of menu_SelectAll



//=== Menu Options =======================================================
//Del by KDevelop: // Show options dialog box
//Del by KDevelop: // Save data to the registry and update main window view
//Del by KDevelop: //========================================================================
void Win::menu_Options() {

	QDialog options(this, "options", true);
	options.setCaption(W2U("Options"));

	QFontMetrics fm = options.fontMetrics();
	int h = fm.height();    // Height of element
	int y = (h*2)/3;        // Space between elements
	int x = h;              // Space between left border and elements
	int w = h/3;            // Space between two elements at one line
	int s = h;              // Additional space between groups
	int m = 0;              // Max width of element (Used for alignment)


	QCheckBox toek(W2U("Translate on every keystroke"), &options);
	toek.setChecked(translateOnEveryKey);
	toek.adjustSize();
	toek.move(x, y);

	QCheckBox sm(W2U("Separate different meanings"), &options);
	sm.setChecked(dictionary->separateMeanings);
	sm.adjustSize();
	sm.move(x, toek.y() + toek.height() + y);

	QCheckBox lo(W2U("Output Bulgarian as latin"), &options);
	lo.setChecked(dictionary->latinOutput);
	lo.adjustSize();
	lo.move(x, sm.y() + sm.height() + y);

	QCheckBox li(W2U("Input Bulgarian as BIG latin"), &options);
	li.setChecked(dictionary->latinInput);
	li.adjustSize();
	li.move(x, lo.y() + lo.height() + y);

	QLabel lwil(W2U("Words in list:"), &options);
	lwil.setAlignment(QLabel::AlignRight | QLabel::AlignVCenter);
	lwil.adjustSize();
	lwil.move(x, li.y() + li.height() + y + s);

	QSpinBox wil(MIN_WORDS_IN_LIST, MAX_WORDS_IN_LIST, 5, &options);
	wil.setValue(wordsInList);
	wil.adjustSize();
	wil.move(x + lwil.width() + w, lwil.y());

	QLabel lwih(W2U("Words in history:"), &options);
	lwih.setAlignment(QLabel::AlignRight | QLabel::AlignVCenter);
	lwih.adjustSize();
	lwih.move(x, wil.y() + wil.height() + y);

	QSpinBox wih(MIN_WORDS_IN_HISTORY, MAX_WORDS_IN_HISTORY, 5, &options);
	wih.setValue(wordsInHistory);
	wih.adjustSize();
	wih.move(x + lwih.width() + w, lwih.y());

	QLabel llpw(W2U("Left pane width:"), &options);
	llpw.setAlignment(QLabel::AlignRight | QLabel::AlignVCenter);
	llpw.adjustSize();
	llpw.move(x, wih.y() + wih.height() + y);

	QSpinBox lpw(MIN_LEFT_PANE_WIDTH, MAX_LEFT_PANE_WIDTH, 5, &options);
	lpw.setValue(leftPaneWidth);
	lpw.adjustSize();
	lpw.move(x + llpw.width() + w, llpw.y());

	QLabel lifw(W2U("Input field width:"), &options);
	lifw.setAlignment(QLabel::AlignRight | QLabel::AlignVCenter);
	lifw.adjustSize();
	lifw.move(x, lpw.y() + lpw.height() + y);

	QSpinBox ifw(MIN_INPUT_FIELD_WIDTH, MAX_INPUT_FIELD_WIDTH, 5, &options);
	ifw.setValue(inputFieldWidth);
	ifw.adjustSize();
	ifw.move(x + lifw.width() + w, lifw.y());

	QLabel lbw(W2U("Border width:"), &options);
	lbw.setAlignment(QLabel::AlignRight | QLabel::AlignVCenter);
	lbw.adjustSize();
	lbw.move(x, ifw.y() + ifw.height() + y);

	QSpinBox bw(MIN_BORDER_WIDTH, MAX_BORDER_WIDTH, 1, &options);
	bw.setValue(borderWidth);
	bw.adjustSize();
	bw.move(x + lbw.width() + w, lbw.y());

	// Right Justify last elements
	m = (wil.width() > wih.width() ? wil.width() : wih.width());
	m = (lpw.width() > m ? lpw.width() : m);
	m = (ifw.width() > m ? ifw.width() : m);
	m = (bw.width() > m ? bw.width() : m);
	wil.resize(m, wil.height());
	wih.resize(m, wih.height());
	lpw.resize(m, lpw.height());
	ifw.resize(m, ifw.height());
	bw.resize(m, bw.height());
	m = (lwil.width() > lwih.width() ? lwil.width() : lwih.width());
	m = (llpw.width() > m ? llpw.width() : m);
	m = (lifw.width() > m ? lifw.width() : m);
	m = (lbw.width() > m ? lbw.width() : m);
	lwil.resize(m, wil.height());
	lwih.resize(m, wih.height());
	llpw.resize(m, lpw.height());
	lifw.resize(m, ifw.height());
	lbw.resize(m, bw.height());
	wil.move(x + m + w, wil.y());
	wih.move(x + m + w, wih.y());
	lpw.move(x + m + w, lpw.y());
	ifw.move(x + m + w, ifw.y());
	bw.move(x + m + w, bw.y());

	QLabel alpw(W2U("pixels"), &options);
	alpw.setAlignment(QLabel::AlignLeft | QLabel::AlignVCenter);
	alpw.adjustSize();
	alpw.setGeometry(lpw.x() + lpw.width() + w, lpw.y(), alpw.width(), lpw.height());

	QLabel aifw(W2U("pixels"), &options);
	aifw.setAlignment(QLabel::AlignLeft | QLabel::AlignVCenter);
	aifw.adjustSize();
	aifw.setGeometry(ifw.x() + ifw.width() + w, ifw.y(), aifw.width(), ifw.height());

	QLabel abw(W2U("pixels"), &options);
	abw.setAlignment(QLabel::AlignLeft | QLabel::AlignVCenter);
	abw.adjustSize();
	abw.setGeometry(bw.x() + bw.width() + w, bw.y(), abw.width(), bw.height());

#ifndef WITH_KDE
	QLabel lss(W2U("Look&Feel style:"), &options);
	lss.setAlignment(QLabel::AlignLeft | QLabel::AlignVCenter);
	lss.adjustSize();
	lss.move(x, bw.y() + bw.height() + y + s);

	QComboBox ss(false, &options);
	ss.insertItem(W2U("Windows"));
	ss.insertItem(W2U("CDE"));
	ss.insertItem(W2U("Motif"));
	ss.insertItem(W2U("Platinium"));
	ss.setCurrentItem(systemStyle - 1);
	ss.adjustSize();
	ss.move(x + lss.width() + w, lss.y());

	lss.resize(lss.width(), ss.height());
#endif

	QPushButton ok(W2U("OK"), &options);
	ok.setDefault(true);
	connect(&ok, SIGNAL(clicked()), &options, SLOT(accept()));

	QPushButton cancel(W2U("Cancel"), &options);
	connect(&cancel, SIGNAL(clicked()), &options, SLOT(reject()));
	cancel.adjustSize();

	m = (toek.width() > sm.width() ? toek.width() : sm.width());
	m = (lo.width() > m ? lo.width() : m);
	m = (li.width() > m ? li.width() : m);
	ok.setGeometry(x + m + x, y, cancel.width(), cancel.height());
	cancel.move(ok.x(), ok.y() + ok.height() + y);

#ifndef WITH_KDE
	options.setFixedSize(ok.x() + ok.width() + x, ss.y() + ss.height() + y);
#else
	options.setFixedSize(ok.x() + ok.width() + x, bw.y() + bw.height() + y);
#endif

	// Execute dialog and check for OK button
	if (options.exec() == QDialog::Accepted) {
		// Set and save new values
		if (inputFieldWidth != ifw.value()) {
			inputFieldWidth = ifw.value();
			reg->setInt(OPTION_INPUT_FIELD_WIDTH, inputFieldWidth);
		}
		if (borderWidth != bw.value()) {
			borderWidth = bw.value();
			reg->setInt(OPTION_BORDER_WIDTH, borderWidth);
		}
		if (leftPaneWidth != lpw.value()) {
			leftPaneWidth = lpw.value();
			reg->setInt(OPTION_LEFT_PANE_WIDTH, leftPaneWidth);
		}
		if (wordsInList != wil.value()) {
			wordsInList = wil.value();
			reg->setInt(OPTION_WORDS_IN_LIST, wordsInList);
		}
		if (wordsInHistory != wih.value()) {
			wordsInHistory = wih.value();
			reg->setInt(OPTION_WORDS_IN_HISTORY, wordsInHistory);
			history->setMaxSize(wordsInHistory);
			if (wordsInHistory == 0) {
				bnext->setEnabled(false);
				bprev->setEnabled(false);
			}
		}
#ifndef WITH_KDE
		if (systemStyle != ss.currentItem() + 1) {
			systemStyle = ss.currentItem() + 1;
			reg->setInt(OPTION_SYSTEM_STYLE, systemStyle);
		}
#endif
		if (translateOnEveryKey != toek.isChecked()) {
			translateOnEveryKey = toek.isChecked();
			reg->setBool(OPTION_TRANSLATE_ON_EVERY_KEY, translateOnEveryKey);
		}
		if (dictionary->separateMeanings != sm.isChecked()) {
			dictionary->separateMeanings = sm.isChecked();
			reg->setBool(OPTION_SEPARATE_MEANINGS, dictionary->separateMeanings);
		}
		if (dictionary->latinOutput != lo.isChecked()) {
			dictionary->latinOutput = lo.isChecked();
			reg->setBool(OPTION_LATIN_OUTPUT, dictionary->latinOutput);
		}
		if (dictionary->latinInput != li.isChecked()) {
			dictionary->latinInput = li.isChecked();
			reg->setBool(OPTION_LATIN_INPUT, dictionary->latinInput);
		}
		// Update view of main window
		setView(systemStyle);
		// Update word and translation
		fromSystem = true;
		word_EnterPressed();
	}
} // End of menu_Options



//=== Menu Advanced Search ===============================================
// Show advanced search options dialog box
// Save data to the registry and update main window view
//========================================================================
void Win::menu_AdvancedSearch() {

	QDialog options(this, "options", true);
	options.setCaption(W2U("Advanced Search"));

	QFontMetrics fm = options.fontMetrics();
	int h = fm.height();    // Height of element
	int y = (h*2)/3;        // Space between elements
	int x = h;              // Space between left border and elements
	int s = h;              // Additional space between groups
	int m = 0;              // Max width of element (Used for alignment)

	QCheckBox sww(W2U("Search for whole word(s)"), &options);
	sww.setChecked(dictionary->advancedSearchWholeWord);
	sww.adjustSize();
	sww.move(x, y);

	QCheckBox sep(W2U("Search for exact phrase"), &options);
	sep.setChecked(dictionary->advancedSearchExactPhrase);
	sep.adjustSize();
	sep.move(x, sww.y() + sww.height() + y);

	QCheckBox chr(W2U("Color highlight result"), &options);
	chr.setChecked(dictionary->advancedSearchHighlight);
	chr.adjustSize();
	chr.move(x, sep.y() + sep.height() + y);

	QLabel lwasd(W2U("When advanced search syntax detected:"), &options);
	lwasd.setAlignment(QLabel::AlignRight | QLabel::AlignVCenter);
	lwasd.adjustSize();
	lwasd.move(x, chr.y() + chr.height() + y + s);

	QCheckBox itoek(W2U("Ignore translate on every keystroke"), &options);
	itoek.setChecked(advSrchIgnoreKey);
	itoek.adjustSize();
	itoek.move(x, lwasd.y() + lwasd.height() + y);

	QCheckBox iht(W2U("Ignore automatic translation from history"), &options);
	iht.setChecked(advSrchIgnoreHistory);
	iht.adjustSize();
	iht.move(x, itoek.y() + itoek.height() + y);

	QCheckBox ewas(W2U("Enter perform advanced search"), &options);
	ewas.setChecked(advSrchEnterSearch);
	ewas.adjustSize();
	ewas.move(x, iht.y() + iht.height() + y);

	QCheckBox dnrp(W2U("Do not replace phrase in input field"), &options);
	dnrp.setChecked(advSrchDoNotReplacePhrase);
	dnrp.adjustSize();
	dnrp.move(x, ewas.y() + ewas.height() + y);

	QPushButton ok(W2U("OK"), &options);
	ok.setDefault(true);
	connect(&ok, SIGNAL(clicked()), &options, SLOT(accept()));

	QPushButton cancel(W2U("Cancel"), &options);
	connect(&cancel, SIGNAL(clicked()), &options, SLOT(reject()));
	cancel.adjustSize();

	m = (sww.width() > sep.width() ? sww.width() : sep.width());
	m = (chr.width() > m ? chr.width() : m);

	m = x + m + 3*x + cancel.width();

	m = (lwasd.width() > m ? lwasd.width() : m);
	m = (itoek.width() > m ? itoek.width() : m);
	m = (iht.width() > m ? iht.width() : m);
	m = (ewas.width() > m ? ewas.width() : m);
	m = (dnrp.width() > m ? dnrp.width() : m);

	ok.setGeometry(m - cancel.width(), y, cancel.width(), cancel.height());
	cancel.move(ok.x(), ok.y() + ok.height() + y);

	options.setFixedSize(ok.x() + ok.width() + x, dnrp.y() + dnrp.height() + y);

	// Execute dialog and check for OK button
	if (options.exec() == QDialog::Accepted) {
		// Set and save new values
		if (dictionary->advancedSearchWholeWord != sww.isChecked()) {
			dictionary->advancedSearchWholeWord = sww.isChecked();
			reg->setBool(OPTION_ADV_SRCH_WHOLE_WORD, dictionary->advancedSearchWholeWord);
		}
		if (dictionary->advancedSearchExactPhrase != sep.isChecked()) {
			dictionary->advancedSearchExactPhrase = sep.isChecked();
			reg->setBool(OPTION_ADV_SRCH_EXACT_PHRASE, dictionary->advancedSearchExactPhrase);
		}
		if (dictionary->advancedSearchHighlight != chr.isChecked()) {
			dictionary->advancedSearchHighlight = chr.isChecked();
			reg->setBool(OPTION_ADV_SRCH_HIGHLIGHT, dictionary->advancedSearchHighlight);
		}
		if (advSrchIgnoreKey != itoek.isChecked()) {
			advSrchIgnoreKey = itoek.isChecked();
			reg->setBool(OPTION_ADV_SRCH_ASD_IGNORE_KEY, advSrchIgnoreKey);
		}
		if (advSrchIgnoreHistory != iht.isChecked()) {
			advSrchIgnoreHistory = iht.isChecked();
			reg->setBool(OPTION_ADV_SRCH_ASD_IGNORE_HIS, advSrchIgnoreHistory);
		}
		if (advSrchEnterSearch != ewas.isChecked()) {
			advSrchEnterSearch = ewas.isChecked();
			reg->setBool(OPTION_ADV_SRCH_ASD_ENTER, advSrchEnterSearch);
		}
		if (advSrchDoNotReplacePhrase != dnrp.isChecked()) {
			advSrchDoNotReplacePhrase = dnrp.isChecked();
			reg->setBool(OPTION_ADV_SRCH_ASD_DO_NOT_REPL, advSrchDoNotReplacePhrase);
		}
	}
} // End of menu_AdvancedSearch



//=== Menu Font ==========================================================
// Show select font dialog
// Save data to the registry and update main window view
//========================================================================
void Win::menu_Font() {
	bool okSelected;
	QFont f = QFont(font);
#ifdef WITH_KDE
	//okSelected = (KFontDialog::getFont(f, false, this) == QDialog::Accepted);
	// To the moment KFontDialog doesn't provide way to select CharSet
	f = QFontDialog::getFont(&okSelected, f, this);
#else
	f = QFontDialog::getFont(&okSelected, f, this);
#endif
	if (okSelected) {
		font = QFont(f);
		// Ignore some options which user may select
		font.setUnderline(false);
		font.setStrikeOut(false);
		// Save user selections
		reg->setString(OPTION_FONT_NAME, U2W(font.family()));
		reg->setInt(OPTION_FONT_SIZE, font.pointSize());
		reg->setInt(OPTION_FONT_WEIGHT, font.weight());
		reg->setBool(OPTION_FONT_ITALIC, font.italic());
		reg->setInt(OPTION_FONT_CHARSET, font.charSet());
		// Update view of main window
		setView(0);
		// Update word and translation
		fromSystem = true;
		word_EnterPressed();
	}
} // End of menu_Font



//=== Menu Test ==========================================================
// Show test/exam window
//========================================================================
void Win::menu_Test() {
	Test t(this, "test", dictionary, reg, &font, borderWidth);
	t.show();
} // End of menu_Test



//=== Menu Help ==========================================================
// Show help message in translation result text area
//========================================================================
void Win::menu_Help() {
	result->setText(W2U(HELP_MESSAGE));
} // End of menu_Help



//=== Menu About =========================================================
// Show about dialog box
//========================================================================
void Win::menu_About() {

	// Define about dialog
	QDialog about(this, "about", true);
	about.setCaption(W2U("About"));

	// Define space between elements
	QFontMetrics fm = about.fontMetrics();
	int h = fm.height();    // Height of element
	int y = (h*2)/3;        // Space between elements
	int x = h;              // Space between left border and elements

	// Define and set picture
	QLabel picture(&about);
	QPixmap picPixmap(dicicon);
	picture.setPixmap(picPixmap);
	picture.adjustSize();
	picture.move(x, y);

	// Define and set label
	QLabel message(W2U(ABOUT_MESSAGE), &about);
	message.adjustSize();
	message.move(picture.x() + picture.width() + x, y);
	
	// Define and set OK button
	QPushButton ok(W2U("OK"), &about);
	ok.setDefault(true);
	connect(&ok, SIGNAL(clicked()), &about, SLOT(accept()));
	ok.adjustSize();
	ok.move(message.x() + message.width() + x, y);

	// Set size of dialog
	about.setFixedSize(ok.x() + ok.width() + x, message.y() + message.height() + y);

	// Execute dialog
	about.exec();
} // End of menu_About



//=== Set View ===========================================================
// Sets font, style, positions and size of widgets
// called from constructor, menu_Font and menu_Options
//========================================================================
void Win::setView(int style) {

#ifndef WITH_KDE
	// Set Style View
	if (style == 1) {
		qApp->setStyle(new QWindowsStyle);
	}
	else if (style == 2) {
		qApp->setStyle(new QCDEStyle);
	}
	else if (style == 3) {
		qApp->setStyle(new QMotifStyle);
	}
	else if (style == 4) {
		qApp->setStyle(new QPlatinumStyle);
	}
#endif

	// Create FontMetrics Object for height of Word field and for other objects width
	QFontMetrics fm = QFontMetrics(font);

	int m = menuBar()->height();

	// Set Word (text field)
	word->setFont(font);
	word->setGeometry(borderWidth, borderWidth + m, inputFieldWidth, (fm.height() < pic->height() - ADDITIONAL_HEIGHT ? pic->height() : fm.height() + ADDITIONAL_HEIGHT));

	// Set Advanced Search Button
	bsearch->setFont(font);
	bsearch->setGeometry(borderWidth + 1 + word->x() + word->width(), borderWidth + m, fm.width(bsearch->text()) + ADDITIONAL_WIDTH, word->height());

	// Set button previous
	bprev->setFont(font);
	bprev->setGeometry(borderWidth + 1 + bsearch->x() + bsearch->width(), borderWidth + m, bsearch->width(), word->height());

	// Set button next
	bnext->setFont(font);
	bnext->setGeometry(1 + bprev->x() + bprev->width(), borderWidth + m, bprev->width(), word->height());

	// Set label found
	found->setFont(font);
	int fl;
	fl = (fm.width(W2U(FOUND)) > fm.width(W2U(NOT_FOUND)) ? fm.width(W2U(FOUND)) : fm.width(W2U(NOT_FOUND)));
	fl = (fm.width(W2U(ADV_SRCH_LABEL)) > m ? fm.width(W2U(ADV_SRCH_LABEL)) : m);
	found->setGeometry(borderWidth + 1 + bnext->x() + bnext->width(), borderWidth + m, fl + 2, word->height());

	// Set picture
	pic->setGeometry((width() - borderWidth - pic->width() < found->x() + found->width() + borderWidth + 1 ? found->x() + found->width() + borderWidth + 1 : width() - borderWidth - pic->width()), borderWidth + m, pic->width(), pic->height());

	// Set list (of words)
	list->setFont(font);
	list->setGeometry(borderWidth, borderWidth*2 + 1 + word->height() +  m, leftPaneWidth, height() - borderWidth*3 - word->height() - m - 1);
	if ((leftPaneWidth > MIN_LEFT_PANE_WIDTH) && (wordsInList > MIN_WORDS_IN_LIST)) {
		list->setEnabled(true);
		list->show();
	}
	else {
		list->setEnabled(false);
		list->hide();
	}

	// Create new font for translation result (right pane) with Normal Style only
	QFont f2 = QFont(font);
	f2.setWeight(QFont::Normal);

	// Set translation result
	result->setFont(f2);
	if (list->isEnabled()) {
		result->setGeometry(borderWidth*2 + 1 + list->width(), borderWidth*2 + 1 + word->height() + m, width() - borderWidth*3 - list->width() - 1, height() - borderWidth*3 - word->height() - m - 1);
	}
	else {
		result->setGeometry(borderWidth, borderWidth*2 + 1 + word->height() +  m, width() - borderWidth*2, height() - borderWidth*3 - word->height() - m - 1);
	}
	result->setText(result->text());

} // End of setView



//=== Resize Event =======================================================
// Invoked from resize ivent
// Set new size and positions for a components
//========================================================================
void Win::resizeEvent(QResizeEvent *resizeEvent) {
	static bool firstTime = true;
	if (firstTime) {
		setView(0);
		firstTime = false;
	}
	else {
		int m = menuBar()->height();
		word->move(word->x(), borderWidth + m);
		bsearch->move(bsearch->x(), borderWidth + m);
		bprev->move(bprev->x(), borderWidth + m);
		bnext->move(bnext->x(), borderWidth + m);
		found->move(found->x(), borderWidth + m);
		pic->move((width() - borderWidth - pic->width() < found->x() + found->width() + borderWidth + 1 ? found->x() + found->width() + borderWidth + 1 : width() - borderWidth - pic->width()), borderWidth + m);
		list->setGeometry(borderWidth, borderWidth*2 + 1 + word->height() +  m, leftPaneWidth, height() - borderWidth*3 - word->height() - m - 1);
		if (list->isEnabled()) {
			result->setGeometry(borderWidth*2 + 1 + list->width(), borderWidth*2 + 1 + word->height() + m, width() - borderWidth*3 - list->width() - 1, height() - borderWidth*3 - word->height() - m - 1);
		}
		else {
			result->setGeometry(borderWidth, borderWidth*2 + 1 + word->height() +  m, width() - borderWidth*2, height() - borderWidth*3 - word->height() - m - 1);
		}
	}
} // End of resizeEvent
