/* PseudoVertex.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard.matching;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

/**
 * This class represents a so called 'pseudovertex'.
 * When a blossom is contracted into a new vertex, this
 * new vertex is called a pseudovertex. A trivial pseudovertex
 * here is nothing else then a simple vertex(thus not a blossom!).
 */
public class PseudoVertex {
    /** The unique identifier of the vertex. */
    protected int id;
    /** The neighbours of the vertex. */
    protected HashSet<PseudoVertex> neighbours;
    /** List with vertices of possible blossom. */
    protected LinkedList<PseudoVertex> blossomList;
    /** Parent in alternating tree. */
    protected PseudoVertex parent;
    /** Is this vertex saturated? */
    protected boolean saturated;
    
    /** Creates a new instance of PseudoVertex */
    public PseudoVertex(int id) {
        this.id = id;
        this.neighbours = new HashSet<PseudoVertex>();
        this.blossomList = null;
        this.parent = null;
        this.saturated = false;
    }
    
    /** Creates a new instance of PseudoVertex. This
     * constructor creates a blossom with the vertices
     * specified in the list.
     * @param id The identifier of the blossom.
     * @param blossomList List with vertices of blossom.
     */
    public PseudoVertex(int id, LinkedList<PseudoVertex> blossomList) {
        this.id = id;
        this.neighbours = new HashSet<PseudoVertex>();
        this.blossomList = blossomList;
        initializeBlossom();
        this.parent = null;
        this.saturated = false;
    }
    
    /**
     * This method adds a neighbour to the list of neighbours.
     * @param vertex The new neighbour.
     */
    public void addNeighbour(PseudoVertex vertex) {
        this.neighbours.add(vertex);
    }
    
    /**
     * This method removes a neighbour from the list of neighbours.
     * @param vertex The neighbour to remove.
     */
    public void removeNeighbour(PseudoVertex vertex) {
        this.neighbours.remove(vertex);
    }
    
    /**
     * This method returns the unique identifier of the vertex
     * as represented in the orginal graph(an adjacencylist).
     * @return The unique identifier(integer) of the vertex.
     */
    public int getId() {
        return id;
    }
    
    /**
     * This method returns the base of the blossom if
     * this pseudovertex is a blossom(null otherwise).
     * @return The base of the blossom(or null)
     */
    public PseudoVertex getBase() {
        if(isBlossom())
            return blossomList.getLast();
        else
            return null;
    }
    
    /**
     * This method returns a List-object with the neighbours of the vertex.
     * @return The adjacencylist with the neighbours of the vertex.
     */
    public HashSet<PseudoVertex> getNeighbours() {
        return neighbours;
    }
    
    /**
     * This method returns a CircularArrayList-object with the vertices which belong
     * to the blossom. The vertices are PseudoVertex-objects. It is
     * important to notice that a PseudoVertex-object not necessarily
     * represents a blossom. Therefore, first isBlossom should be used.
     * The "base" of the blossom should always be located at the end of the list.
     * @return The list of vertices belonging to this blossom.
     */
    public List<PseudoVertex> getBlossomList() {
        return blossomList;
    }
    
    /**
     * This method returns the parent of this vertex in the
     * alternating tree. The parent is a PseudoVertex-object.
     * @return The parent of this vertex in the alternating tree.
     */
    public PseudoVertex getParent() {
        return parent;
    }
    
    /**
     * This method tells whether this vertex is saturated or not.
     * @return true if vertex is saturated
     */
    public boolean isSaturated() {
        return saturated;
    }
    
    /**
     * Is this pseudovertex a blossom?
     * @return true if this pseudovertex is a blossom
     */
    public boolean isBlossom() {
        return this.blossomList != null;
    }
    
    /**
     * Sets the parent of the vertex in the alternating tree.
     * @param parent The parent of the vertex in the alternating tree.
     */
    public void setParent(PseudoVertex parent) {
        this.parent = parent;
    }
    
    /**
     * Set this vertex saturated or unsaturated
     * depending on the argument.
     * @param saturated Saturated or not?
     */
    public void setSaturated(boolean saturated) {
        this.saturated = saturated;
    }
    
    /**
     * Expand this PseudoVertex if it is a blossom.
     * @return LinkedList with vertices in blossom.
     */
    public LinkedList<PseudoVertex> expandBlossom() {
        if(isBlossom()) {
            HashSet<PseudoVertex> set = new HashSet<PseudoVertex>(blossomList);
            for(PseudoVertex vertex : blossomList) {
                for(PseudoVertex n : vertex.neighbours) {
                    if(!set.contains(n)) {
                        n.neighbours.remove(this);
                        n.neighbours.add(vertex);
                    }
                }
            }
            return blossomList;
        } else
            return null;
    }
    
    /**
     * Removes this vertex from the graph by removing
     * all incident edges.
     */
    public void removeFromGraph() {
        for(PseudoVertex v : neighbours) {
            v.removeNeighbour(this);
        }
    }
    
    /**
     * Initializes the adjacencylist for this blossom and updates
     * the adjacencylists for these neigbours.
     */
    private void initializeBlossom() {
        HashSet<PseudoVertex> set = new HashSet<PseudoVertex>(blossomList);
        for(PseudoVertex vertex : blossomList) { // for all vertices in blossom
            for(PseudoVertex n : vertex.neighbours) { // for all their neighbours
                if(!set.contains(n)) { // if n is not part of the blossom
                    neighbours.add(n);
                    // Update adjacencylist of neighbours of vertex
                    n.neighbours.remove(vertex);
                    n.neighbours.add(this);
                    // Update parent of neighbour if necessary
                    if(set.contains(n.getParent()))
                        n.setParent(this);
                }
            }
        }
    }
    
}
