/* 
 *   Creation Date: <2000/12/28 22:01:59 samuel>
 *   Time-stamp: <2001/01/31 23:13:14 samuel>
 *   
 *	<oldworld.c>
 *	
 *	Oldworld booter
 *   
 *   Copyright (C) 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#define WANTED_ROM_BASE	((char*)0x4fc00000)	/* simplifies debugging */

#define ROM_FIX
#define MAP_PHYSICAL_ROM

#include "mol_config.h"
#include <sys/mman.h>

#include "memory.h"
#include "res_manager.h"
#include "wrapper.h"
#include "debugger.h"
#include "promif.h"
#include "verbose.h"
#include "booter.h"

SET_VERBOSE_NAME("oldworld");

static void map_rom( char *romimage );
static void oldworld_startup( void );
static void oldworld_cleanup( void );

static int romim_fd = -1;

/************************************************************************/
/*	F U N C T I O N S						*/
/************************************************************************/

void
oldworld_booter_init( void )
{
	gPE.booter_startup = oldworld_startup;
	gPE.booter_cleanup = oldworld_cleanup;
}

static void
oldworld_startup( void )
{
	char	*rom_filename = get_str_res("rom_image");
	ulong 	checksum, i;

#ifdef MAP_PHYSICAL_ROM
	struct mmu_mapping rom2;
	map_rom( rom_filename );
	
	rom2 = rom;
	if( !rom_filename ) {
		printm("<Physical ROM mapping>\n");
		rom2.flags |= MAPPING_PHYSICAL;
		rom2.lvbase = (char*)rom2.mbase;
	}
	/* the debugger still accesses ROM using the info in rom */
	_add_mmu_mapping( &rom2 );
#else	
	map_rom( rom_filename );
	_add_mmu_mapping( &rom );
#endif	
	/* Calculate a simple ROM checksum (without mapping too many pages) */
	for(checksum = 0, i=0; i<rom.size; i+=65536*16 )
		checksum += rom.lvbase[i];
	printm("ROM checksum = 0x%lx\n", checksum );
}


static void
oldworld_cleanup( void )
{
	if( rom.lvbase ){
		_remove_mmu_mapping( &rom );
		munmap( rom.lvbase, rom.size );
		rom.lvbase = 0;
	}

	if( romim_fd != -1 )
		close( romim_fd );
}


static void 
map_rom( char *rom_image ) 
{
	mol_device_node_t *dn;
	int 	len, has_physical_rom=0;
	
#ifdef ROM_FIX
	int rom_workaround = 0;
#endif	
	/* find rom using Open Firmware */
	if( (dn = prom_find_type("rom")) != NULL ) {
		if( dn->n_addrs != 1 ) {
			LOG("Expected 1 ROM address entry , got %d\n", dn->n_addrs);
			exit(1);
		}
		rom.size = dn->addrs[0].size;
		rom.mbase = dn->addrs[0].address;

		/* see if a corresponding ROM is physically availble */
		dn = prom_find_type_hw( "rom" );
		has_physical_rom = ( dn != NULL && dn->n_addrs == 1 
				     && dn->addrs[0].size == rom.size 
				     && dn->addrs[0].address == rom.mbase );
	} else if( (dn = prom_find_devices("boot-rom")) != NULL ) {
		ulong *prop;
		/* New-world boot-rom (G3, iMac) */
		printm("WARNING: The build in ROM won't work currently\n");
		prop = (ulong*)prom_get_property( dn, "reg", &len );
		if( prop == NULL || len != sizeof(long[2]) ){
			LOG("Expected a double word property\n");
			exit(1);
		}
		rom.mbase = prop[0];
		rom.size = prop[1];

		/* is a boot-rom physically available? */
		if( (dn = prom_find_devices_hw("boot-rom")) != NULL )
			prop = (ulong*)prom_get_property( dn, "reg", &len );
		has_physical_rom = (dn && prop && len == sizeof(long[2]) 
				    && rom.size == prop[1] 
				    && rom.mbase == prop[0] );
	} else {
		printm("Could not find a ROM entry in the OF device tree\n");
		exit(1);
	}

	rom.flags = 0;		/* implies read only */

	if( rom_image ) {
		if( (romim_fd = open( rom_image, O_RDONLY )) == -1 ) {
			printm("\n----> Could not open ROM-image '%s'\n\n", rom_image );
			printm("Read the file mol/doc/Status for information about ROM-images\n\n");
			exit(1);
		}
		rom.lvbase = mmap( WANTED_ROM_BASE, rom.size, PROT_READ, MAP_SHARED,
				   romim_fd, 0 );
		if( (int)rom.lvbase == -1 ) {
			rom.lvbase = 0;
			perrorm("mmap");
			exit(1);
		}
		printm("%dK ROM mapped at %p from '%s'\n", rom.size/1024, 
		       rom.lvbase, rom_image );
		return;
	}

	/* Use physical ROM if no image is given */
	if( !has_physical_rom ){
		LOG("Could not find a suitable physical ROM\n");
		exit(1);
	}

#ifdef ROM_FIX
	/* there is a bug in the kernel which prevent us from mapping 
	 * the very last page of ROM (0xfffff000), probably due to an owerflow. 
	 */
	if( rom.size + rom.mbase == 0 ) {
		rom.size-=0x1000;
		rom_workaround = 1;
	}
#endif
	/* We'll probably get cache-inhibited access to ROM. 
	 * It really doesn't mater since it is only the debugger which 
	 * uses this mapping anyway.
	 */
	rom.lvbase = map_phys_mem( WANTED_ROM_BASE, rom.mbase, rom.size, PROT_READ );
	if( !rom.lvbase )
		exit(1);

#ifdef ROM_FIX
	if( rom_workaround ) {
		char *last_page;
		
		last_page = map_zero( rom.lvbase+rom.size, 0x1000 );
		if( !last_page ) {
			perrorm("ROM_workaround:");
			exit(1);
		}

		if( last_page != rom.lvbase + rom.size ) {
			printm("ROM workaround failed!\n");
			/* we could fix this by having two ROM-segments */
			/* but we usually don't come here */
			exit(1);
		}
		rom.size += 0x1000;
		
		/* Unfortunately it doesn't work with read() either -
		 * we use this nasty hack to fill in the data.
		 */
		_kernel_memmove( last_page, (char*)0xfffff000, 0x1000 );

		/* write protect page */
		if( mprotect( last_page, 0x1000, PROT_READ ) ) {
			perrorm("init_rom: mprotect:");
		}
		printm("<ROM_FIX> ");
	}
#endif
	printm("%dK ROM mapped at %p\n", rom.size/1024, rom.lvbase );
}
