/* 
 *   Creation Date: <2000/02/06 22:56:55 samuel>
 *   Time-stamp: <2001/01/07 00:39:24 samuel>
 *   
 *	<osi_driver.c>
 *	
 *	OSI driver 
 *	Handles the Mac-side driver interface.
 *   
 *   Copyright (C) 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

/* #define VERBOSE */

#include "mol_config.h"

#include "promif.h"
#include "booter.h"
#include "res_manager.h"
#include "debugger.h"
#include "pci.h"
#include "verbose.h"
#include "pic.h"
#include "os_interface.h"
#include "osi_driver.h"
#include "session.h"
#include <pthread.h>

// SET_VERBOSE_NAME("osi_driver")

static int	prom_add_nwdriver( mol_device_node_t *dn, char *filename );
static int    	allocate_pci_slot( int *pci_bus, int *dev_fn, int need_irq );
static int 	osip_poll_irq( int sel, int *params );
static int 	osip_register_irq( int sel, int *params );
static int 	osip_deregister_irq( int sel, int *params );
static int 	osip_enable_irq( int sel, int *params );
static int 	osip_ack_irq( int sel, int *params );
static int	save_osi_irq_info( void );
static void	load_osi_irq_info( void );

typedef struct osi_driver
{
	int	pci_bus;
	int	pci_devfn;
	int	irq;			/* PCI-irq, might be shared */
	int	irq_cookie;	      	/* unique IRQ cookie (dynamic variable) */

	struct 	osi_driver *next;
} osi_driver_t;

typedef struct 
{
	ulong 	used;
	ulong 	line;
	ulong	enabled;
} mol_irq_t;


#define NUM_IRQS	64
#define IRQ_MASK	(NUM_IRQS-1)

#define	COOKIE_TO_IRQ(irq_cookie)	((irq_cookie) & IRQ_MASK)
#define	COOKIE_TO_BIT(irq_cookie)	((irq_cookie) & ~IRQ_MASK)

static mol_irq_t 	irq_table[NUM_IRQS];		/* mutex guarded */
static osi_driver_t 	*osi_driver_root = NULL;

static pthread_mutex_t irq_mutex = PTHREAD_MUTEX_INITIALIZER;

/* irq_table lock */
#define LOCK		pthread_mutex_lock( &irq_mutex );
#define UNLOCK		pthread_mutex_unlock( &irq_mutex );

/* session_save_data */
typedef struct
{
	int	bus;
	int	devfn;
	int	cookie;
	int	valid;
} cookie_save_info_t;

static cookie_save_info_t 	*rest_cookies=NULL;
static int 			num_rest_cookies=0;


/************************************************************************/
/*	FUNCTIONS							*/
/************************************************************************/

void 
osi_driver_services_init( void )
{
	memset(irq_table, 0, sizeof(irq_table) );
	os_interface_add_proc( OSI_POLL_IRQ, osip_poll_irq );
	os_interface_add_proc( OSI_REGISTER_IRQ, osip_register_irq );
	os_interface_add_proc( OSI_DEREGISTER_IRQ, osip_deregister_irq );
	os_interface_add_proc( OSI_ENABLE_IRQ, osip_enable_irq );
	os_interface_add_proc( OSI_ACK_IRQ, osip_ack_irq );

	/* Saved session support */
	session_save_proc( save_osi_irq_info, NULL, kDynamicChunk );
	if( loading_session() )
		load_osi_irq_info();
}

void
osi_driver_services_cleanup( void )
{
	os_interface_remove_proc( OSI_POLL_IRQ );
	os_interface_remove_proc( OSI_REGISTER_IRQ );
	os_interface_remove_proc( OSI_DEREGISTER_IRQ );
	os_interface_remove_proc( OSI_ENABLE_IRQ );
	os_interface_remove_proc( OSI_ACK_IRQ );

	if( rest_cookies )
		free( rest_cookies );
}

static struct osi_driver *
register_osi_driver_( char *drv_name, char *pnode_name, pci_dev_info_t *pci_config, int add_rom )
{
	mol_device_node_t *dn;
	osi_driver_t dt, *ret_dt;
	char *name=NULL, buf[100];

	memset( &dt, 0, sizeof(dt) );

	/*
	 * Determine PCI slot and interrupt number.
	 */
	if( is_oldworld_boot() ) {
		dt.irq = 0;	/* not known yet */
		if( allocate_pci_slot( &dt.pci_bus, &dt.pci_devfn, 1 ) )
			return NULL;
	}
	if( is_newworld_boot() ) {
		int *p;

		if( add_rom ) {
			strcpy( buf, "osi_" );
			strcat( buf, drv_name );
			strcat( buf, "_pcirom_nw" );

			if( !(name = get_str_res(buf)) ) {
				printm("---> Resource '%s' is missing!\n", buf );
				return NULL;
			}
		}

		if( !(dn = prom_find_devices( pnode_name )) ) {
			printm("---> No '%s' node found in the devicetree\n", pnode_name );
			return NULL;
		}
		if( add_rom )
			prom_add_nwdriver( dn, name );
		
		pci_device_loc( dn, &dt.pci_bus, &dt.pci_devfn );
		if( (p=(int*)prom_get_property( dn, "mol-irq", NULL )) == NULL ) {
			printm("ERROR: mol_irq property missing from node '%s'!\n", pnode_name );
			return NULL;
		}
		dt.irq = *p;
		VPRINT("Using IRQ 0x%x\n", dt.irq );
	}
	if( is_bootx_boot() ) {
		static int next_irq=1;
		printm("FIXME: IRQ assignement and device identification \n");
		dt.irq = next_irq++;
	}

	/* Add PCI card to bus */
	if( is_newworld_boot() || is_oldworld_boot() ) {
		VPRINT("Adding pci card at (%d:%02X)\n", dt.pci_bus, dt.pci_devfn );
		if( add_pci_device( dt.pci_bus, dt.pci_devfn, pci_config, NULL )) {
			printm("Unexpected error: Failed adding pci card!\n");
			return NULL;
		}
	}
	
	/* Add oldworld ROM */
	if( is_oldworld_boot() && add_rom ) {
		strcpy( buf, "osi_" );
		strcat( buf, drv_name );
		strcat( buf, "_pcirom" );
		use_pci_rom( buf, NULL, dt.pci_bus, dt.pci_devfn );
	}

	/* Restore any irq cookie information (applicable only for a saved session) */
	if( rest_cookies ) {
		cookie_save_info_t *cp = rest_cookies;
		int i;
		for( i=0; i<num_rest_cookies; i++, cp++) {
			if( !cp->valid || cp->bus != dt.pci_bus || cp->devfn != dt.pci_devfn )
				continue;
			dt.irq_cookie = cp->cookie;
			cp->valid = 0;
			break;
		}
	}

	/* Add to linked list */
	ret_dt = (osi_driver_t*)malloc( sizeof(dt) );
	*ret_dt = dt;
	ret_dt->next = osi_driver_root;
	osi_driver_root = ret_dt;

	return ret_dt;
}

struct osi_driver *
register_osi_driver( char *drv_name, char *pnode_name, pci_dev_info_t *pci_config )
{
	return register_osi_driver_( drv_name, pnode_name, pci_config, 1 /*add_rom */ );
}

struct osi_driver *
register_osi_driver_norom( char *drv_name, char *pnode_name, pci_dev_info_t *pci_config )
{
	return register_osi_driver_( drv_name, pnode_name, pci_config, 0 /*no_rom */ );
}

void
free_osi_driver( struct osi_driver *dr )
{
	osi_driver_t *d, **ptr;
	
	ptr = &osi_driver_root;
	for( d = osi_driver_root; d; d=d->next ) {
		if( d == dr ) {
			*ptr = d->next;
			break;
		}
		ptr = &d->next;
	}

	if( d )
		free( d );
	else
		printm("free_osi_driver called but no driver found!\n");
}

int
get_driver_pcislot( struct osi_driver *dr, int *r_bus, int *r_devfn )
{
	if( !dr )
		return 1;
	if( r_bus )
		*r_bus = dr->pci_bus;
	if( r_devfn )
		*r_devfn = dr->pci_devfn;
	return 0;
}


/************************************************************************/
/*	IRQ utils							*/
/************************************************************************/

static int
save_osi_irq_info( void )
{
	osi_driver_t *d;
	cookie_save_info_t *si;
	int n, ret;

	if( write_session_data( "IRQT", 0, (char*)irq_table, sizeof(irq_table)) )
		return -1;

	for( n=0, d=osi_driver_root; d; d=d->next, n++ )
		;
	si = calloc( n, sizeof(cookie_save_info_t) );
	for( n=0, d=osi_driver_root; d; d=d->next, n++ ){
		si[n].bus = d->pci_bus;
		si[n].devfn = d->pci_devfn;
		si[n].cookie = d->irq_cookie;
		si[n].valid = 1;
	}
	ret = write_session_data( "Icok", 0, (char*)si, n*sizeof(cookie_save_info_t));
	free( si );
	return ret;
}

static void
load_osi_irq_info( void ) 
{
	int s;
	if( read_session_data( "IRQT", 0, (char*)irq_table, sizeof(irq_table) ) )
		session_failure("Could not read osi-driver information\n");
	if( (s=get_session_data_size( "Icok", 0 )) < 0 )
		session_failure("Could not get size of osi-irq cookies\n");
	rest_cookies = malloc( s );
	if( read_session_data( "Icok", 0, (char*)rest_cookies, s) )
		session_failure("Could not read osi-irq cookies\n");
	num_rest_cookies = s/sizeof(cookie_save_info_t);
}


static inline 
void recalc_irq( int irq )
{
	mol_irq_t *p = &irq_table[irq];
	if( (p->enabled & p->line) )
		irq_line_hi(irq);
	else
		irq_line_low(irq);
}


void
osi_irq_hi( osi_driver_t *d )
{
	// printm("osi_irq_hi %08lX\n", cookie);
	
	LOCK;
	irq_table[d->irq].line |= COOKIE_TO_BIT(d->irq_cookie);
	recalc_irq( d->irq );
	UNLOCK;
}

void
osi_irq_low_( ulong cookie )
{
	int irq = COOKIE_TO_IRQ(cookie );
	mol_irq_t *p = &irq_table[ irq ];
	ulong b = COOKIE_TO_BIT(cookie);

	// printm("osi_irq_low\n");

	LOCK;
	p->line &= ~b;
	recalc_irq( irq );
	UNLOCK;
}

void
osi_irq_low( osi_driver_t *d )
{
	osi_irq_low_(d->irq_cookie );
}

static int
osip_ack_irq( int sel, int *params )
{
	osi_irq_low_( params[0] );
	return 0;
}


osi_driver_t *
get_osi_driver_from_id( ulong osi_id )
{
	osi_driver_t *dr;

	if( !is_bootx_boot() ) {
		int bus = (osi_id>>16) & 0xff;	/* osi_id == first word of reg_property */
		int devfn = (osi_id>>8) & 0xff;

		for( dr = osi_driver_root; dr; dr=dr->next )
			if( dr->pci_bus == bus && dr->pci_devfn == devfn )
				break;
		return dr;
	} else {
		printm("XXX: Fix get_osi_driver_from_id\n");
		return NULL;
	}
}


/* params[0] = first word of reg_property (oldworld/newworld) */
static int
osip_register_irq( int sel, int *params )
{
	osi_driver_t *d = get_osi_driver_from_id( params[0] );
	ulong v, bit;

	// printm("osip_register_irq %08X\n", params[0]);
	if( !d ){
		printm("osip_register_irq: Device not found %08X\n", params[0] );
		return 0;
	}

	if( is_oldworld_boot() ) {
		// printm("Fixing IRQ: (was %d, now %d)\n", d->irq, params[1] );
		d->irq = params[1];
	}
	if( d->irq > NUM_IRQS ) {
		printm("BAD irq number!\n");
		return 0;
	}

	LOCK;
	v = irq_table[d->irq].used;
	for( bit=(IRQ_MASK+1)*2; (v & bit); bit=bit*2 )
		;
	if( !bit ){
		printm("register_irq: Too many shared IRQs!\n");
		UNLOCK;
		return 0;
	}
	irq_table[d->irq].used |= bit;
	irq_table[d->irq].enabled &= ~bit;

	d->irq_cookie = (d->irq | bit);
	recalc_irq(d->irq);
	UNLOCK;
	
	return d->irq_cookie;
}

static int
osip_deregister_irq( int sel, int *params )
{
	osi_driver_t *d;
	ulong	irq_cookie = params[0];
	int 	irq = COOKIE_TO_IRQ(irq_cookie);
	int 	bit = COOKIE_TO_BIT(irq_cookie);

	if( is_oldworld_boot() )
		for( d=osi_driver_root; d; d=d->next )
			if( d->irq_cookie == irq_cookie )
				d->irq_cookie = 0;
	
	LOCK;
	irq_table[irq].used &= ~bit;
	irq_table[irq].enabled &= ~bit;
	recalc_irq(irq);
	UNLOCK;
	return 0;
}

static int
osip_poll_irq( int sel, int *params )
{
	int 	irq = COOKIE_TO_IRQ(params[0]);
	ulong 	b = COOKIE_TO_BIT(params[0]);
	int	ret;

	// printm("osip_poll_irq %08X\n", params[0]);
	LOCK;
	ret = (irq_table[irq].line & irq_table[irq].enabled & b) ? 1:0;
	UNLOCK;

	return ret;
}

static int
osip_enable_irq( int sel, int *params )
{
	int	irq = COOKIE_TO_IRQ(params[0] );
	ulong	bit = COOKIE_TO_BIT(params[0] );
	int	old, enable = params[1];
	
	// printm("osip_enable_irq %d\n", enable);

	LOCK;
	old = irq_table[irq].enabled & bit;
	if( enable )
		irq_table[irq].enabled |= bit;
	else
		irq_table[irq].enabled &= ~bit;
	recalc_irq(irq);
	UNLOCK;
	
	return old ? 1:0;
}


/************************************************************************/
/*	Add NewWorld driver to device tree				*/
/************************************************************************/

static int
prom_add_nwdriver( mol_device_node_t *dn, char *filename )
{
	int fd, size;
	char *ptr;
	
	if( !dn )
		return 1;
	fd = open( filename, O_RDONLY );
	if( fd < 0 ) {
		perrorm("Could not open driver '%s'", filename );
		return 1;
	}

	size = lseek( fd, 0, SEEK_END);
	ptr = malloc( size );
	if( !ptr ){
		printm("Out of memrory\n");
		close(fd );
		return 1;
	}
	lseek( fd, 0, SEEK_SET );
	read( fd, ptr, size );
	prom_add_property( dn, "driver,AAPL,MacOS,PowerPC", ptr, size );
	free( ptr );
	close( fd );

	return 0;
}


/************************************************************************/
/*	Oldworld PCI allocation						*/
/************************************************************************/

typedef struct 
{
	int	allocated;
	int	dev_fn;
} pci_desc_t;

static pci_desc_t slots[] = {
	{0, (13<<3) | 0},		/* (0:68) slot A1, fn 0 */
	{0, (14<<3) | 0},		/* (0:70) slot B1, fn 0 */
	{0, (15<<3) | 0},		/* (0:78) slot C1, fn 0 */

	{0, (13<<3) | 1},		/* (0:69) slot A1, fn 1 */
	{0, (14<<3) | 1},		/* (0:71) slot B1, fn 1 */
	{0, (15<<3) | 1},		/* (0:79) slot C1, fn 1 */

	{0, (13<<3) | 2},		/* (0:6a) slot A1, fn 2 */
	{0, (14<<3) | 2},		/* (0:72) slot B1, fn 2 */
	{0, (15<<3) | 2},		/* (0:7a) slot C1, fn 2 */

	{0, (13<<3) | 3},		/* (0:6b) slot A1, fn 3 */
	{0, (14<<3) | 3},		/* (0:73) slot B1, fn 3 */
	{0, (15<<3) | 3},		/* (0:7b) slot C1, fn 3 */
	{0, 0}
};

static int
allocate_pci_slot( int *pci_bus, int *dev_fn, int need_irq )
{
	int i;

	if( !is_oldworld_boot() ){
		printm("allocate_pci_slot is oldworld only!\n");
		exit(1);
	}

	for( i=0; slots[i].dev_fn; i++ ){
		if( slots[i].allocated )
			continue;
		slots[i].allocated = 1;
		*pci_bus = 0;
		*dev_fn = slots[i].dev_fn;

		/* printm("pci_bus %d, dev_fn %d\n", *pci_bus, *dev_fn );*/
		return 0;
	}
	printm("----> Failed adding PCI-card - no free slots availalble!\n");
	return 1;
}
