/* 
 *   Creation Date: <1999/03/29 07:35:25 samuel>
 *   Time-stamp: <2001/09/29 18:30:55 samuel>
 *   
 *	<options.c>
 *	
 *	Resource manager & option parsing
 * 	Note: This file is used both stand alone and in MOL
 *
 *   Copyright (C) 1999, 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

#include <obstack.h>
#include <getopt.h>
#include "debugger.h"
#include "res_manager.h"

#include "cmdline_opts.h"
#include "version.h"

#define obstack_chunk_alloc malloc
#define obstack_chunk_free free

typedef struct res_head {
	struct obstack 	stack;

	struct res	*first_res;
} res_head_t;

typedef struct res {
	char 		*key;

	struct darg 	*first_darg;
	struct res	*next;		/* next resource */
} res_t;

typedef struct darg {
	char		*data;
	struct darg	*next;		/* next data struct */
} darg_t;

static res_head_t head;

static char *push_str( char *str );
static void do_add_res( char *res_str, int is_default );
static void parse_options( int is_mol, int argc, char **argv );

/* Globals */
int	g_session_id = 0;


void 
res_manager_init( int is_mol, int argc, char **argv )
{
	FILE *file = NULL;
	char *libdir;
	char buf[4096];
	char buf2[128];
	char postfix[16];
	int i;
	

	/* determine the session number */
	g_session_id = 0;
	for( i=0; i<argc; i++ )
		if( sscanf( argv[i], "--session=%d", &g_session_id )==1 )
			break;

	sprintf( postfix, "-%d", g_session_id );
	if( g_session_id != 0 && is_mol )
		printm("Starting MOL session %d\n", g_session_id );

	/* Find a the molrc file */
	while( !file ) {
		/* try /etc/molrc-hostname */
		strcpy( buf, "/etc/molrc-" );
		gethostname( buf2, sizeof(buf2) );
		strcat( buf, buf2 );
		strcat( buf, postfix );
		file = fopen( buf, "r" );

		/* and /etc/molrc */
		if( !file ) {
			strcpy( buf, "/etc/molrc" );
			strcat( buf, postfix );
			file = fopen(buf, "r");
		}

		/* Look for /etc/molrc-0 first and then /etc/molrc */
		if( g_session_id )
			break;
		if( !g_session_id && postfix[0] ) {
			postfix[0]=0;
			continue;
		}
		break;
	}
	if( file ) {
		if( is_mol )
			printm("Configuration file: %s\n", buf );
	} else {
		printm("ERROR: No resource file '%s' found!\n", buf);
		exit(1);
	}

	memset( &head, 0, sizeof(res_head_t ));
	obstack_init( &head.stack );

	/* Process user resource file... */
	for( ;; ){
		if( fgets( buf, sizeof(buf), file ) == NULL )
			break;
		do_add_res( buf, 0 );
	}
	fclose( file );

	/* change working directory to 'rootdir' */
	if( !(libdir=get_str_res("rootdir")) )
		libdir = MOL_LIB_DIR;

	if( chdir( libdir ) < 0 ){
		printm("Could not change working directory to %s\n", libdir );
		exit(1);
	}

	/* Process MOL system resource file */
	if( !(file=fopen( "config/molrc.sys","r")) ) {
		printm("Fatal error: Could not open the config/molrc.sys file!\n");
		exit(1);
	}
	for( ;; ){
		if( fgets( buf, 4096, file ) == NULL )
			break;
		do_add_res( buf, 1 );
	}
	fclose(file);
	
	/* command line options last */
	parse_options( is_mol, argc, argv );

	if( is_mol )
		printm("Library directory: %s\n", getcwd(buf, sizeof(buf)) );
}

void
res_manager_cleanup( void )
{
	obstack_free( &head.stack, NULL );
}

static char *
push_str( char *str )
{
	if( !str )
		return NULL;
	return obstack_copy( &head.stack, str, strlen(str)+1 );
}

static void 
do_add_res( char *const_res_str, int is_default ){

	darg_t	*darg, **next_darg;
	char 	*field;
	res_t	*res;
	char	*res_str;

	/* we must not change the original string */
	res_str = alloca( strlen(const_res_str)+1 );
	strcpy( res_str, const_res_str );

	do {
		field = strsep( &res_str, "\t\n\r, " );
	} while( field && !strlen(field) );

	if( !field || field[0] == '#' )
		return;

	if( field && strlen(field) && field[ strlen(field)-1 ] != ':' ){
		printm("Bad resource line: %s\n", res_str );
		return;
	}

	/* remove ':' */
	field[ strlen(field)-1 ] = 0;

	if( is_default && res_present(field) )
		return;

	/* head */
	res = obstack_alloc( &head.stack, sizeof(res_t) );
       	memset( res,0,sizeof(res_t) );
	
	res->key = push_str( field );

	/* arguments */
	next_darg = &res->first_darg;

	while( res_str && *res_str ) {
		res_str += strspn( res_str, "\t\n\r, ");
		field = res_str;

		if( *field == '#' || !*field )
			break;
		if( *field == '\'' || *field == '"' )
			res_str = strchr( ++res_str, *field++ );
		else
			res_str += strcspn( res_str, "\t\n\r, " );
		if( res_str && *res_str )
			*res_str++ = 0;

		darg = obstack_alloc( &head.stack, sizeof(darg_t) );
		memset( darg, 0, sizeof(darg_t) );
		darg->data = push_str( field );
		*next_darg = darg;
		next_darg = &darg->next;
	}

	/* insert res in table */
	res->next = head.first_res;
	head.first_res = res;
}

static res_t *
find_res( char *key, int index )
{
	res_t *res;
	
	for( res=head.first_res; res; res=res->next ){
		if( !strcmp(res->key, key ) )
			index--;
		if( index<0 )
			break;
	}
	return res;
}

/************************************************************************/
/*	Interface functions						*/
/************************************************************************/

char *
get_lockfile( void )
{
	static char lockfile[28];
	sprintf( lockfile, "/var/lock/mol-%d", g_session_id );
	return lockfile;
}


void
add_res( char *key, char *value )
{
	char *buf;
	int len = strlen(key)+strlen(value)+10;
	buf = alloca( len );

	snprintf( buf, len, "%s: %s", key, value );
	do_add_res( buf, 0 );
}

void
default_res( char *key, char *value )
{
	char *buf;
	int len = strlen(key)+strlen(value)+10;
	buf = alloca( len );

	snprintf( buf, len, "%s: %s", key, value );
	do_add_res( buf, 1 /*is default*/ );
}

int
res_present( char *key )
{
	return (find_res( key, 0))? 1: 0;
}


char *
get_str_res( char *key )
{
	return get_str_res_ind( key, 0, 0 );
}

char *
get_str_res_ind( char *key, int index, int argnum )
{
	res_t *res;
	darg_t *darg;
	
	if( (res = find_res( key, index )) == NULL )
		return NULL;

	for( darg=res->first_darg; darg && argnum>0; argnum--, darg=darg->next )
		;
	return darg? darg->data : NULL;
}

/* returns -1 if not found (otherwise 0 or 1) */
int 
get_bool_res( char *key )
{
	char *str;
	
	str = get_str_res( key );
	if( !str )
		return -1;
	if( strpbrk( str, "yY1tT" ))
		return 1;
	if( strpbrk( str,"nN0fF" ))
		return 0;

	printm("Bad boolean option: '%s'\n",str );
	return -1;
}

/* returns -1 if not found */
long
get_numeric_res( char *key )
{
	long val;
	char *str;

	if( !(str=get_str_res(key)) )
		return -1;
	if( !sscanf(str, "0x%lx", &val) && !sscanf(str, "%ld",&val ) )
		return -1;
	return val;
}

/* match arguments against an option table and return the resulting flags 
 */
ulong
parse_res_options( char *key, int index, int first_arg, opt_entry_t *opts, 
		   char *err_str )
{
	char 	*opt;
	ulong 	flags=0;
	int 	i;
	
	for(;;){
		if( (opt = get_str_res_ind( key, index, first_arg++)) == NULL )
			break;
		for( i=0; opts[i].name; i++ ){
			if( !strcmp(opt, opts[i].name )) {
				flags |= opts[i].flag;
				break;
			}
		}
		if( !opts[i].name )
			printm("%s '%s'\n", err_str ? err_str : "WARNING: Unknown option", opt );
	}
	return flags;
}


/************************************************************************/
/*	option parsing							*/
/************************************************************************/

static void 
parse_options( int is_mol, int argc, char **argv )
{
	int c, opt_index=0;
	
	for( ;; ){
		struct opt_res_entry *or_ptr;
		int match=0;
		
		opterr=0;	/* suppress errors */
		if( (c=getopt_long( argc, argv, "dxhVv::", opt, &opt_index )) == -1 )
			break;

		/* convert to resource */
		for( or_ptr=or_table; or_ptr->opt_code; or_ptr++ ){
			if( or_ptr->opt_code != c )
				continue;
			match=1;
			if( or_ptr->root_only && in_security_mode ) {
				printm("WARNING: A root-only flag was ignored\n");
				break;
			}
			if( !or_ptr->res_name )
				break;
			if( strchr(or_ptr->res_name, ':') )
				do_add_res( or_ptr->res_name, 0 );
			else {
				if( !optarg ){
					printm("res_manager: Internal error!\n");
					exit(1);
				}
				add_res( or_ptr->res_name, optarg );
			}
			/* several entries may match the same opt_code */
		}
		
		/* options which require special attention */
		switch(c) {
		case 0:
			break;
		case 'V':
			if( is_mol ) {
				/* version info is printed in main() */
				exit(0);
			}
			break;
		case 'h':
			if( is_mol ){
				fprintf(stderr, "%s", helpstr );
				exit(0);
			}
			break;
		case 'x': /* debugger */
			add_res( "no_moldeb", "1");
			/* fallthrough */
		case 'd': /* debugger */
			add_res( "debug", "1" );
			add_res( "debug_stop", "1" );
			break;
		case '?':
			fprintf( stderr, "option '%s' not recognized\n", argv[optind-1] );
			if( is_mol )
				exit(1);
			break;
		default:
			if( !match )
				fprintf(stderr, "?? getopt returned unexpected value %d\n", c );
			break;
		case optSessNum:
			break;
		}
	}

	if( optind < argc ) {
		fprintf(stderr, "Options ignored: ");
		while( optind < argc )
			fprintf(stderr, "%s ",argv[optind++]);
		fprintf(stderr, "\n");
	}
}

