package expectj;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.BufferedReader;
import java.io.BufferedWriter;

/**
 * This class handles I/O for the streams we are expecting
 * input on.  This code was factored from the class SpawnedProcess.
 *
 * @author	Sachin Shekar Shetty 
 * @author	Greg Anderson 
 * @version 1.0
 */
public abstract class ExpectHandler implements TimerEventListener {



    // Default time out for expect commands
    protected long m_lDefaultTimeOut = -1;

    // Streams to the process
    protected BufferedReader in = null;
    protected InputStream inStream = null;
    protected BufferedReader err = null;
    protected InputStream errStream = null;
    protected BufferedWriter out = null;
    protected OutputStream outStream = null;

    // Debugger object
    Debugger debug = new Debugger("ExpectHandler", true);

    

    protected volatile boolean continueReading = true;



    /**
     * Constructor
     *
     * @param sCommand Process commmand to be executed
     * @param lDefaultTimeOut Default timeout for expect commands
     *
     */
    ExpectHandler(long lDefaultTimeOut) 
        throws ExpectJException {

           
            m_lDefaultTimeOut = lDefaultTimeOut;
            

        }

    /**
     * Timer callback method
     * This method is invoked when the time-out occurr
     */
    public synchronized void timerTimedOut() {

        continueReading = false;

    }

    /**
     * Timer callback method
     * This method is invoked by the Timer, when the timer thread
     * receives an interrupted exception
     */
    public void timerInterrupted(InterruptedException ioe) {

        continueReading = false;

    } 

    /**
     * This method returns true if the last expect() or expectErr() method 
     * returned because of a time out rather then a match against 
     * the output of the process. 
     */
    public boolean isLastExpectTimeOut() {

        return (!continueReading);

    }
    
    /**
     * This method functions exactly like the Unix expect command. 
     * It waits untill a string is read from the standard output stream 
     * of the spawned process that matches the string pattern. 
     * SpawnedProcess does a cases insensitive substring match for pattern 
     * against the output of the spawned process. 
     * lDefaultTimeOut is the timeout in seconds that the expect command 
     * should wait for the pattern to match. This function returns 
     * when a match is found or after lTimOut seconds. 
     * You can use the SpawnedProcess.isLastExpectTimeOut() to identify 
     * the return path of the method. A timeout of -1 will make the expect 
     * method wait indefinitely untill the supplied pattern matches 
     * with the Standard Out. 
     * This method throws an ExpectJException when some error occurrs.
     */
    public void expect(String pattern, long lDefaultTimeOut) throws 
    ExpectJException {
    	doexpect(pattern, null, lDefaultTimeOut);
    }
    
    public String capture(String pattern, long lDefaultTimeOut) throws 
        ExpectJException {
        	StringBuffer sb = new StringBuffer();
        	
        	doexpect(pattern, sb, m_lDefaultTimeOut);
        	
        	return sb.toString();
    }
    
    /**
     * This method functions exactly like expect described above, 
     * but uses the default timeout specified in the ExpectJ constructor. 
     */
    public String capture(String pattern) throws ExpectJException {

       return capture(pattern, m_lDefaultTimeOut);

    }
    
    private void doexpect(String pattern, StringBuffer sb, long lDefaultTimeOut) throws 
            ExpectJException {
            	

        try {
            debug.print("ExpectHandler.expect(" + pattern + ")");               
            Timer tm = null;
            if (lDefaultTimeOut != -1 ) {
                tm = new Timer(lDefaultTimeOut, this);
                tm.startTimer();
                continueReading = true;
            }
            boolean found = false;
            int i = 0;
            StringBuffer line = new StringBuffer();
            here: while(continueReading) {
                // Sleeping if bytes are not available rather then
                // blocking
                while (inStream.available() == 0 ) {
                    Thread.sleep(500);
                    // Go back and check the condition
                    continue here;
                  }
                  i = inStream.read();
                  if (i== -1)
                      break;
                  char ch = (char)i;
                  line.append(ch);
                  if (line.toString().trim().toUpperCase().indexOf(
                          pattern.toUpperCase()) != -1) {
                      debug.print("Matched for " + pattern + ":" 
                              + line);
                      found = true;
                      if(sb != null)
                    	  sb.append(line);
                      break;
                  }
                  if (i == '\n') {
                      debug.print("Line read: " + line);   
                      if(sb != null)
                      	sb.append(line);
                      line.setLength(0);
                  }
            }
            debug.print("expect Over");
            debug.print("Found: " + found);
            debug.print("Continue Reading:" + continueReading );
            if (tm != null)
                debug.print("Timer Status:" + tm.getStatus());

        }
        catch (Exception exp) {
            throw new ExpectJException("Error in ExpectHandler.expect", exp);
        }

    }

    /**
     * This method functions exactly like the corresponding expect 
     * function except for it tries to match the pattern with the 
     * output  of standard error stream of the spawned process.
     */
    public void expectErr(String pattern, long lDefaultTimeOut)  
        throws ExpectJException {
    	
        if(errStream == null)
        {
        	throw new ExpectJException("Error stream not open in expectErr");
        }
    	try {
            debug.print("ExpectHandler.expectErr(" + pattern + ")");               
            Timer tm = null;
            if (lDefaultTimeOut != -1 ) {
                tm = new Timer(lDefaultTimeOut, this);
                tm.startTimer();
                continueReading = true;
            }
            boolean found = false;
            int i = 0;
            StringBuffer line = new StringBuffer();
            here: while(continueReading) {
                // Sleeping if bytes are not available rather then
                // blocking
                while (errStream.available() == 0 ) {
                    Thread.sleep(500);
                    // Go back and check the condition
                    continue here;
                  }
                  i = errStream.read();
                  if (i== -1)
                      break;
                  char ch = (char)i;
                  line.append(ch);
                  if (line.toString().trim().toUpperCase().indexOf(
                          pattern.toUpperCase()) != -1) {
                      debug.print("Matched for " + pattern + ":" 
                              + line);
                      found = true;
                      break;
                  }
                  if (i == '\n') {
                      debug.print("Line read: " + line);               
                      line.setLength(0);
                  }
            }
            debug.print("expectErr Over");
            debug.print("Found: " + found);
            debug.print("Continue Reading:" + continueReading );
            if (tm != null)
                debug.print("Timer Status:" + tm.getStatus());

        }
        catch (Exception exp) {
            throw new ExpectJException("Error in ExpectHandler.expectErr", exp);
        }

    }

    /**
     * This method functions exactly like expect described above, 
     * but uses the default timeout specified in the ExpectJ constructor. 
     */

    public void expect(String pattern) throws ExpectJException {

        expect(pattern, m_lDefaultTimeOut);

    }
    
    
    /**  
     * This method functions exactly like the corresponding expect 
     * function except for it tries to match the pattern with the output  
     * of standard error stream of the spawned process.
     */
    public void expectErr(String pattern) throws ExpectJException {

        expectErr(pattern, m_lDefaultTimeOut);

    }

    /**
     * This method returns true if the process has already exited. 
     * This method shoud be use use to check the process status 
     * before invoking send()
     */
    public abstract boolean isClosed();

    /**
     * This method returns the exit code of the process if the process has 
     * already exited.
     */
    public abstract int getExitValue();

    /**
     * This method writes the string line to the standard input of the spawned 
     * process. It throws a JExpectException if the spawned process 
     * has already exited.
     */
    public void send(String line)  throws ExpectJException{

        try {
            debug.print("Sending " + line);
//            if (this.isClosed())
//                throw new ExpectJException(
//                        "Process already exited with exit code: " 
//                        + this.getExitValue());
            out.write(line);
            out.flush();
        }
        catch (Exception exp) {
            throw new ExpectJException("Error in ExpectHandler.send", exp);
        }

    }

    public abstract void interact() throws ExpectJException;
    public abstract void stop();


}
