package expectj;

import java.util.Date;
import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.OutputStream;
import java.io.PipedOutputStream;

/**
 * This class runs the process and streams the input and output of the
 * process so as to allow execution of the process in a different thread. 
 *
 * @author	Sachin Shekar Shetty  
 * @version 1.0
 */
public class ProcessRunner  implements TimerEventListener {

    // The commandline of the process to be executed
    private String commandLine = null;

    // The time interval up to which the process should be allowed to run.
    // When time-out = -1, means no time-out, run indefinately.
    private long timeOut = 0;

    // Process object
    private ProcessThread process = null;

    // Timer object to monitor ProcessThread
    private Timer tm = null;

    // Piped Streams to copy the output to standard streams
    private PipedInputStream readSystemOut = null;
    private PipedInputStream readSystemErr = null;

    private PipedOutputStream writeSystemOut = null;
    private PipedOutputStream writeSystemErr = null;

    // StreamPiper objects to pipe the output of one stream to other
    private StreamPiper processOutToSystemOut = null;
    private StreamPiper processErrToSystemErr = null;


    private Debugger debug = new Debugger("ProcessRunner", true);
    
    /**
     * Private constructor
     *
     * @param commandLine process command to be executed
     * @param timeOut time interval in seconds to be allowed for process
     * execution; not used as of now.
     */
    private void initialize(String commandLine, long timeOut) 
        throws ExpectJException {

            if (timeOut < -1) {
                throw new IllegalArgumentException("Time-out is invalid");
            }
            if (commandLine == null || commandLine.trim().equals("")) {
                throw new IllegalArgumentException("Command: " + commandLine
                        + " is null/empty");
            }
            this.timeOut = timeOut;
            this.commandLine = commandLine;

            // Initialise the process thread.
            process = new ProcessThread(commandLine);
            if (timeOut != -1)
                tm = new Timer(timeOut, this);

        }

    /**
     * Constructor
     *
     * @param commandLine process command to be executed 
     * @param timeOut time interval in seconds to be allowed for process
     * execution; not used as of now.
     */
    ProcessRunner(String commandLine, long timeOut) throws ExpectJException {

    		try{
    			initialize(commandLine, timeOut);
    		}
    		catch(ExpectJException e){
    			e.printStackTrace();
    			throw e;
    		}
    }

    /**
     * This constructor allows to run a process with indefinate time-out
     * @param commandLine process command to be executed 
     */
    ProcessRunner (String commandLine) throws ExpectJException {
    	try{
    		initialize(commandLine, -1);
    	}
    	catch(ExpectJException e){
    		e.printStackTrace();
    		throw e;
    	}
    }

    /**
     * Time callback method
     * This method is invoked when the time-out occurr
     */
    public void timerTimedOut() {

        process.stop();

    }

    /**
     * This method stops the spawned process.
     */
    public void stop() {

        process.stop();

    }


    /**
     * Timer callback method
     * This method is invoked by the Timer, when the timer thread
     * receives an interrupted exception
     */
    public void timerInterrupted(InterruptedException ioe) {

    }

    /**
     * This method is used to Stop all the piper object from copying the
     * content to standard out. This is used after interact command.
     */
    synchronized void stopPipingToStandardOut() {

        processOutToSystemOut.stopPipingToStandardOut();          
        processErrToSystemErr.stopPipingToStandardOut();          

    }

    synchronized void startPipingToStandardOut() {

        processOutToSystemOut.startPipingToStandardOut();          
        processErrToSystemErr.startPipingToStandardOut();          

    }


    /**
     * This method executes the given command within the specified time
     * limit. It starts the process thread and also the timer when
     * enabled. It starts the piped streams to enable copying of process
     * stream contents to standard streams.
     */ 
    void start() throws ExpectJException {

        try {
        	System.out.println("ProcessRunner.start(): Start the process and timer if needed");
            // Start the process and timer if needed
            process.start();
            if (tm != null)
                tm.startTimer();

            System.out.println("ProcessRunner.start(): Starting the piped streams and StreamPiper objects");
            // Starting the piped streams and StreamPiper objects
            readSystemOut = new PipedInputStream();
            writeSystemOut = new PipedOutputStream(readSystemOut);
            processOutToSystemOut = new StreamPiper(System.out, 
                    process.process.getInputStream(), writeSystemOut);
            processOutToSystemOut.start();

            System.out.println("ProcessRunner.start(): opening the error streams");
            readSystemErr = new PipedInputStream();
            writeSystemErr = new PipedOutputStream(readSystemErr);
            processErrToSystemErr = new StreamPiper(System.err, 
                    process.process.getErrorStream(), writeSystemErr);
            processErrToSystemErr.start();
            System.out.println("ProcessRunner.start(): complete");
        }
        catch (Exception exp) {
        
        	exp.printStackTrace();
        	throw new ExpectJException("Error in ProcessRunner.start()", exp);
        }



    }

    /**
     * This method returns the input stream of the process.
     */
    InputStream getInputStream() {

        return readSystemOut;

    }


    /**
     * This method returns the output stream of the process.
     */
    OutputStream getOutputStream() {

        return process.process.getOutputStream();

    }

    /**
     * This method returns the error stream of the process.
     */
    InputStream getErrorStream() {

        return readSystemErr;

    }

    /**
     * This method returns true if the process has already exited.
     */
    boolean isClosed() {

        return process.isClosed;

    }

    /**
     * If the process representes by this object has already exited, it
     * returns the exit code. isClosed() should be used in conjunction
     * with this method.
     */
    int getExitValue() {

        return process.exitValue;

    }


    /**
     * This class is responsible for executing the process in a seperate
     * thread.
     */
    class ProcessThread implements Runnable {

        // This represents the command to be executed
        String commandLine = null;

        // Process object for execution of the commandLine
        Process process = null;

        // Thread object to run this file
        Thread processThread = null;

        // is Process closed
        volatile boolean isClosed = false;

        // exitvalue when closed
        int exitValue;




        /**
         * Constructor
         */
        ProcessThread(String commandLine) {

            this.commandLine = commandLine;

        }

        /**
         * This method spawns the thread and runs the process within the
         * thread
         */
        public void start() throws ExpectJException {

            System.out.println("Process Started at:" + new Date());
            processThread = new Thread(this); 
            try {
                process = Runtime.getRuntime().exec(commandLine);
            }
            catch (Exception exp) {
            	exp.printStackTrace();
                throw new ExpectJException("Error in ProcessThread.start", exp);
            }
            processThread.start();

        }

        /**
         * This method executes the process, Thread Run Method
         *
         */
        public void run() {

            try {
                process.waitFor();
                exitValue= process.exitValue();
                isClosed = true;
            } 
            catch (InterruptedException ie) {
            }
            catch (Exception e) {
                e.printStackTrace(); 
            } 

        }

        /**
         * This method interrupts and stops the thread.
         */
        public void stop() {

            debug.print("Process '" + commandLine + "' Killed at:" 
                    + new Date());
            processThread.interrupt();
            process.destroy();

        }

    }


}
